﻿------------------------------------------------------------------------------------------------------
-- Initialise the variable we will be working with
------------------------------------------------------------------------------------------------------
local L = LibStub("AceLocale-3.0"):GetLocale("InstanceInfoFu");
local Tablet = AceLibrary("Tablet-2.0");
local AceConfig = LibStub("AceConfigDialog-3.0");

InstanceInfoFu = LibStub("AceAddon-3.0"):NewAddon("InstanceInfoFu", "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0", "LibFuBarPlugin-3.0");


------------------------------------------------------------------------------------------------------
-- Find the hex colour for each class
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:GetHexColor(class)
	return string.format("%02x%02x%02x", 
		RAID_CLASS_COLORS[class].r * 255,
		RAID_CLASS_COLORS[class].g * 255,
		RAID_CLASS_COLORS[class].b * 255
	);
end

------------------------------------------------------------------------------------------------------
-- What happens when the addon is first loaded
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnInitialize()
	-- Register the database
	InstanceInfoFu.db = LibStub("AceDB-3.0"):New("InstanceInfoFuDB", InstanceInfoFu.defaults);

	for k,v in pairs(InstanceInfoFu.fubarOptions) do
		self:SetFuBarOption(k, v);
	end
	self:SetFuBarIcon(InstanceInfoFu.fubarOptions["iconPath"]);
	InstanceInfoFu.db:RegisterNamespace("FuBarOptions", InstanceInfoFu.fubarDefaults);
	InstanceInfoFu.options.args["FubarOptions"] = ReturnFubarOptions(InstanceInfoFu, InstanceInfoFu.db.profile);

	-- Add info to the Options array for the Profile Options
	InstanceInfoFu.options.args.profile.args.options = LibStub("AceDBOptions-3.0"):GetOptionsTable(InstanceInfoFu.db);
	InstanceInfoFu.options.args.profile.args.options.name = L["Options profile"];
	InstanceInfoFu.options.args.profile.args.options.desc = L["Saved profile for InstanceInfoFu options"];
	InstanceInfoFu.options.args.profile.args.options.order = 2;

	-- Create options table and stuff
	LibStub("AceConfig-3.0"):RegisterOptionsTable("InstanceInfoFu", InstanceInfoFu.options, "instanceinfo")
	InstanceInfoFu.optFrame = AceConfig:AddToBlizOptions("InstanceInfoFu", "InstanceInfoFu")
end

------------------------------------------------------------------------------------------------------
-- What happens when the addon is enabled
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnEnable()
	-- Register for important events
	InstanceInfoFu:RegisterEvent("PLAYER_LEAVING_WORLD");
	InstanceInfoFu:RegisterEvent("UPDATE_INSTANCE_INFO");
	InstanceInfoFu:RegisterEvent("INSTANCE_BOOT_START");
	InstanceInfoFu:RegisterEvent("INSTANCE_BOOT_STOP");

	-- Ask for instance info
	RequestRaidInfo();
	
	-- FuBar Text Update
	self:ScheduleRepeatingTimer(self.OnUpdate, 60, self);

	-- Update FuBar Plugin text and tooltip
	self:UpdateFuBarPlugin();
end

------------------------------------------------------------------------------------------------------
-- What happens when the addon is disabled
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnDisable()
	-- Remove registration for our events
	InstanceInfoFu:UnregisterEvent("PLAYER_LEAVING_WORLD")
	InstanceInfoFu:UnregisterEvent("UPDATE_INSTANCE_INFO")
	InstanceInfoFu:UnregisterEvent("INSTANCE_BOOT_START")
	InstanceInfoFu:UnregisterEvent("INSTANCE_BOOT_STOP")
end

------------------------------------------------------------------------------------------------------
-- FuBar Menu Open code
------------------------------------------------------------------------------------------------------
InstanceInfoFu.OpenMenu = function(frame)
	if LibStub("AceConfigDialog-3.0").OpenFrames["InstanceInfoFu"] then
		LibStub("AceConfigDialog-3.0"):Close("InstanceInfoFu")
	else
		LibStub("AceConfigDialog-3.0"):Open("InstanceInfoFu")
	end
end

------------------------------------------------------------------------------------------------------
-- FuBar space click
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnUpdate()	
	-- Update FuBar Plugin text and tooltip
	self:UpdateFuBarText();
end

------------------------------------------------------------------------------------------------------
-- FuBar space click
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnFuBarClick(button)
	-- Open the Raid frame
	ToggleFriendsFrame(5);
end

------------------------------------------------------------------------------------------------------
-- FuBar text update code
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnUpdateFuBarText()
	local numInstances = 0;
	
	if (InstanceInfoFu.db.profile.showSelf) then
		 numInstances = GetNumSavedInstances();
	else
		for player, playerinfo in pairs(InstanceInfoFu.db.realm.info) do
			if (playerinfo ~= nil) then
				local nMax = #(playerinfo.instances);
				for i = 1, nMax do
					local instance = playerinfo.instances[i];
					if (instance ~= nil) then
						numInstances = numInstances + 1;
					end
				end
			end
		end
	end
	
	if (numInstances == 0) then
		-- We're not saved to anything
		self:SetFuBarText("|cff00ff00"..L["No Saved Instances"].."|r");
	else
		self:SetFuBarText("|cff00ff00"..L["Saved Instances"]..": |r"..numInstances);
	end
end

------------------------------------------------------------------------------------------------------
-- FuBar tooltip update code
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:OnUpdateFuBarTooltip()
	local numInstances = 0;
	local cat;

	if (InstanceInfoFu.db.profile.showSelf) then
		 numInstances = GetNumSavedInstances();
	else
		for player, playerinfo in pairs(InstanceInfoFu.db.realm.info) do
			if (playerinfo ~= nil) then
				local nMax = #(playerinfo.instances);
				for i = 1, nMax do
					local instance = playerinfo.instances[i];
					if (instance ~= nil) then
						numInstances = numInstances + 1;
					end
				end
			end
		end
	end

	if (numInstances == 0) then
		-- We're not saved to anything
		cat = Tablet:AddCategory(
			'columns',	1,
			'child_textR',	1,
			'child_textG',	1,
			'child_textB',	0
		);
		cat:AddLine(
			'text', L["No Saved Instances"]
		);
	else
		-- Display all our saved instances
		self:ShowSavedInstances();
	end
	
	Tablet:SetHint(L["Right-click|r to open the options menu"])
end

------------------------------------------------------------------------------------------------------
-- Function to get and display all saved instances according to our configuration
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:ShowSavedInstances()
	if InstanceInfoFu.db.realm.info == nil then
		-- We have no info from this realm
		return;
	end

	if (not InstanceInfoFu.db.profile.showSelf) then
		for player, playerinfo in pairs(InstanceInfoFu.db.realm.info) do
			if (playerinfo ~= nil and playerinfo ~= {}) then
				self:ShowSavedInstanceInfo(player);
			end
		end
	else
		-- We're only showing ourselves
		self:ShowSavedInstanceInfo(UnitName("player"), cat);
	end
end

------------------------------------------------------------------------------------------------------
-- Function to get and display saved instances for the specified player
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:ShowSavedInstanceInfo(player)
	if InstanceInfoFu.db.realm.info[player] == nil then
		-- We have no info for this character
		return;
	end
	local info = InstanceInfoFu.db.realm.info[player];
	local cat = Tablet:AddCategory(
		'columns', 4,
		'text', "[|cff"..InstanceInfoFu.classColours[info.engClass]..info.player.."|r]",
		'text2', L["Zone"],
		'text3', L["Raid ID"],
		'text4', L["Reset Time"]
	);
	local nMax = #(info.instances);
	for i = 1, nMax do
		local instance = info.instances[i];
		if (instance ~= nil) then
			if (time() <= (info.curtime + instance.remaining)) then
				cat:AddLine(
					'text', "",
					'text2', instance.name,
					'text3', instance.ID,
					'text4', SecondsToTime((info.curtime + instance.remaining) - time())
				);
			else
				-- This timer expired
				InstanceInfoFu.db.realm.info[player].instances[i] = nil;
				self:OnUpdateFuBarText();
			end
		end
	end
end

------------------------------------------------------------------------------------------------------
-- Instance information saving
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:SaveInstanceInfo()
	if (UnitLevel("player") < tonumber(InstanceInfoFu.db.profile.minLevel)) then
		-- We don't want to store instance info for low-level players
		-- Won't really be useful till WotLK, but meh =p
		return;
	end

	local player = UnitName("player");
	if (InstanceInfoFu.db.realm.info[player] == nil) then
		InstanceInfoFu.db.realm.info[player] = {};
	end
	
	-- Store a bunch of fun stuff
	InstanceInfoFu.db.realm.info[player].player	= player;
	InstanceInfoFu.db.realm.info[player].engClass	= select(2, UnitClass("player"));
	InstanceInfoFu.db.realm.info[player].curtime	= time();
	
	-- Store name of instances we are saved to
	InstanceInfoFu.db.realm.info[player].instances	= {};

	local numInstances = GetNumSavedInstances();
	for i = 1, numInstances do
		local instance = {};
		instance.name, instance.ID, instance.remaining = GetSavedInstanceInfo(i);
		InstanceInfoFu.db.realm.info[player].instances[i] = instance;
	end

	table.sort(InstanceInfoFu.db.realm.info[player].instances, function(a, b) return a.name < b.name end);
	table.sort(InstanceInfoFu.db.realm.info, function(a,b) return a.player < b.player; end);

	-- Update FuBar Plugin text and tooltip
	self:UpdateFuBarPlugin();

	return true;
end

------------------------------------------------------------------------------------------------------
-- Event that fires when we leave the world
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:PLAYER_LEAVING_WORLD()
	self:SaveInstanceInfo();
end

------------------------------------------------------------------------------------------------------
-- Event that fires when the instance we are in gets updated
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:UPDATE_INSTANCE_INFO()
	self:SaveInstanceInfo();
end

------------------------------------------------------------------------------------------------------
-- Event that fires when the instance we are in starts up
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:INSTANCE_BOOT_START()
	self:SaveInstanceInfo();
end

------------------------------------------------------------------------------------------------------
-- Event that fires when the instance we are in is done starting
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:INSTANCE_BOOT_STOP()
	self:SaveInstanceInfo();
end

------------------------------------------------------------------------------------------------------
-- Having a function with msg being passed to it allows us to create /commands later if desired
-- If no msg is passed, open the GUI, else accept a chat command.
------------------------------------------------------------------------------------------------------
function InstanceInfoFu:ChatCommand(msg)
	if not input or input:trim() == "" then
		-- We're opening config dialogue
		LibStub("AceConfigDialog-3.0"):Open("InstanceInfoFu");
	else
		-- We have a chat command
		LibStub("AceConfigCmd-3.0").HandleCommand(InstanceInfoFu, "instanceinfo", msg);
	end
	
end
