local Tablet = AceLibrary("Tablet-2.0")
local L = AceLibrary("AceLocale-2.2"):new("FuBar_FuzzyClockFu")

FuzzyClockFu = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceDB-2.0", "FuBarPlugin-2.0")

FuzzyClockFu:RegisterDB("FuzzyClockFuDB")
FuzzyClockFu:RegisterDefaults('profile', {
	twentyFour = false,
	showSeconds = false,
	localTime = false,
	bothTimes = false,
	bubble = false,
	fuzziness = 0,
	r = 1,
	g = 1,
	b = 1,
})

FuzzyClockFu.version = "1.0." .. string.sub("$Revision: 66212 $", 12, -3)
FuzzyClockFu.date = string.sub("$Date: 2008-03-26 19:26:02 -0400 (Wed, 26 Mar 2008) $", 8, 17)
FuzzyClockFu.hasIcon = false
FuzzyClockFu.defaultPosition = 'RIGHT'
FuzzyClockFu.hideWithoutStandby = true

function FuzzyClockFu:OnInitialize()
	if GroupCalendar_OnLoad then
		self.db.profile.bubble = true
	end
end

function FuzzyClockFu:OnEnable()
	self.timeSinceLastUpdate = 0
	self.secondsDifference = 0
	local _
	_,self.lastMinute = GetGameTime()
	if not self:IsShowingBubble() then
		self.db.profile.bubble = true
		self:ToggleShowingBubble()
	end
	self:ScheduleRepeatingEvent(self.Update, 1, self)
end

function FuzzyClockFu:OnDisable()
	if not self:IsShowingBubble() then
		self:ToggleShowingBubble()
		self.db.profile.bubble = not self.db.profile.bubble
	end
end

function FuzzyClockFu:GetColor()
	return self.db.profile.r, self.db.profile.g, self.db.profile.b
end

function FuzzyClockFu:GetHexColor()
	return string.format("%02x%02x%02x", self.db.profile.r * 255, self.db.profile.g * 255, self.db.profile.b * 255)
end

function FuzzyClockFu:SetColor(r, g, b)
	self.db.profile.r = r
	self.db.profile.g = g
	self.db.profile.b = b
	self:UpdateText()
end

function FuzzyClockFu:IsTwentyFour()
	return self.db.profile.twentyFour
end

function FuzzyClockFu:ToggleTwentyFour()
	self.db.profile.twentyFour = not self.db.profile.twentyFour
	self:Update()
	return self.db.profile.twentyFour
end

function FuzzyClockFu:IsShowingSeconds()
	return self.db.profile.showSeconds
end

function FuzzyClockFu:ToggleShowingSeconds()
	self.db.profile.showSeconds = not self.db.profile.showSeconds
	self:Update()
	return self.db.profile.showSeconds
end

function FuzzyClockFu:IsLocalTime()
	return self.db.profile.localTime
end

function FuzzyClockFu:ToggleLocalTime()
	self.db.profile.localTime = not self.db.profile.localTime
	self:Update()
	return self.db.profile.localTime
end

function FuzzyClockFu:IsBothTimes()
	return self.db.profile.bothTimes
end

function FuzzyClockFu:ToggleBothTimes()
	self.db.profile.bothTimes = not self.db.profile.bothTimes
	self:Update()
	return self.db.profile.bothTimes
end

function FuzzyClockFu:IsShowingBubble()
	return self.db.profile.bubble
end

function FuzzyClockFu:ToggleShowingBubble()
	self.db.profile.bubble = not self.db.profile.bubble
	if not self.db.profile.bubble then
		GameTimeFrame:Hide()
	else
		GameTimeFrame:Show()
	end
	return self.db.profile.bubble
end

function FuzzyClockFu:GetLocalOffset()
	if self.localOffset ~= nil then
		return self.localOffset
	end
	local localHour = tonumber(date("%H"))
	local localMinute = tonumber(date("%M"))
	local utcHour = tonumber(date("!%H"))
	local utcMinute = tonumber(date("!%M"))
	local loc = localHour + localMinute / 60
	local utc = utcHour + utcMinute / 60
	self.localOffset = floor((loc - utc) * 2 + 0.5) / 2
	if self.localOffset >= 12 then
		self.localOffset = self.localOffset - 24
	end
	return self.localOffset
end

function FuzzyClockFu:GetServerOffset()
	if self.serverOffset ~= nil then
		return self.serverOffset
	end
	local serverHour, serverMinute = GetGameTime()
	local utcHour = tonumber(date("!%H"))
	local utcMinute = tonumber(date("!%M"))
	local ser = serverHour + serverMinute / 60
	local utc = utcHour + utcMinute / 60
	self.serverOffset = floor((ser - utc) * 2 + 0.5) / 2
	if self.serverOffset >= 12 then
		self.serverOffset = self.serverOffset - 24
	elseif self.serverOffset < -12 then
		self.serverOffset = self.serverOffset + 24
	end
	return self.serverOffset
end

function FuzzyClockFu:GetFuzziness()
	return self.db.profile.fuzziness
end

function FuzzyClockFu:SetFuzziness(f)
	self.db.profile.fuzziness = f
	self:Update()
	return self.db.profile.fuzziness
end

local optionsTable = {
	handler = FuzzyClockFu,
	type = 'group',
	args = {
		twentyFour = {
			type = 'toggle',
			name = L["24-hour format"],
			desc = L["Toggle between 12-hour and 24-hour format"],
			get = "IsTwentyFour",
			set = "ToggleTwentyFour",
		},
		["local"] = {
			type = 'toggle',
			name = L["Local time"],
			desc = L["Toggle between local time and server time"],
			get = "IsLocalTime",
			set = "ToggleLocalTime",
		},
		fuzziness = {
			type = 'range',
			name = L["Fuzziness"],
			desc = L["The amount of fuzziness to apply to clock"],
			min = 0,
			max = 3,
			step =1,
			get = "GetFuzziness",
			set = "SetFuzziness",
		},
		bubble = {
			type = 'toggle',
			name = L["Show day/night bubble"],
			desc = L["Show the day/night bubble on the upper-right corner of the minimap"],
			get = "IsShowingBubble",
			set = "ToggleShowingBubble",
		},
		color = {
			type = 'color',
			name = COLOR or "Color",
			desc = L["Set the color of the text"],
			get = "GetColor",
			set = "SetColor",
			disabled = function()
				return not FuzzyClockFu:IsTextColored()
			end
		}
	}
}
FuzzyClockFu:RegisterChatCommand(L["AceConsole-commands"], optionsTable)
FuzzyClockFu.OnMenuRequest = optionsTable

function FuzzyClockFu:FormatTime(who,hour, minute, second, colorize)
	hournames = { "One", "Two", "Three", "Four", "Five", "Six", "Seven", "Eight", "Nine", "Ten", "Eleven", "Twelve", "One" }
	normalFuzzy = { "0 o\'clock", "Five past 0", "Ten past 0","Quarter past 0", "Twenty past 0","Twenty five past 0","Half past 0","Twenty five to 1","Twenty to 1","Quarter to 1","Ten to 1","Five to 1","1 o\'clock" }
	dayTime = { "Night","Early morning","Morning", "Almost noon","Noon","Afternoon","Evening","Late evening" }
	
	if who =="ttu" or self:GetFuzziness() == 0 then
		if self:IsTwentyFour() then
			if not colorize then
				return string.format("%d:%02d", hour, minute)
			else
				local color = self:GetHexColor()
				return string.format("|cff%s%d|r:|cff%s%02d|r", color, hour, color, minute)
			end
		else
			local pm = floor(hour / 12) == 1
			hour = mod(hour, 12)
			if hour == 0 then
				hour = 12
			end
			if not colorize then
				return string.format("%d:%02d %s", hour, minute, pm and " PM" or " AM")
			else
				local color = self:GetHexColor()
				return string.format("|cff%s%d|r:|cff%s%02d|r %s", color, hour, color, minute, pm and " PM" or " AM")
			end
		end
		
	elseif self:GetFuzziness() == 1 or self:GetFuzziness() == 2 then
		hour = mod(hour, 12)
		if hour == 0 then
			hour = 12
		end
	
		sector = 1
		if self:GetFuzziness() == 1 then
			if minute > 2 then
				sector = math.floor((minute - 3) / 5) + 2
			end
		elseif minute > 6 then
			sector = (math.floor((minute -7) / 15) + 1) * 3 + 1
		end
				
		newTimeStr = normalFuzzy[sector]
		
		if sector > 7 then
			deltaHour = 1
		else
			deltaHour = 0
		end
		
		fuzzyStr = string.gsub(newTimeStr,string.format("%d",deltaHour),hournames[hour+deltaHour])
	elseif self:GetFuzziness() == 3 then
		fuzzyStr = dayTime[math.floor(hour / 3) + 1]
	end
	if not colorize then
		return string.format("%s",fuzzyStr)
	else
		local color = self:GetHexColor()
		return string.format("|cff%s %s", color, fuzzyStr)
	end
end

function FuzzyClockFu:GetServerTime(who,colorize)
	local hour, minute = GetGameTime()
	if self.lastMinute ~= minute then
		local _
		_,self.lastMinute = GetGameTime()
		self.secondsDifference = mod(time(), 60)
	end
	local second = mod(time() - self.secondsDifference, 60)
	return self:FormatTime(who,hour, minute, second, colorize)
end

function FuzzyClockFu:GetLocalTime(who,colorize)
	local hour = tonumber(date("%H"))
	local minute = tonumber(date("%M"))
	local second = tonumber(date("%S"))
	return self:FormatTime(who,hour, minute, second, colorize)
end

function FuzzyClockFu:GetUtcTime(who)
	local hour = tonumber(date("!%H"))
	local minute = tonumber(date("!%M"))
	local second = tonumber(date("!%S"))
	return self:FormatTime(who,hour, minute, second)
end

function FuzzyClockFu:OnTextUpdate()
	who = "otu"
	local hour, minute = GetGameTime()
	if self.lastMinute ~= minute then
		local _
		_,self.lastMinute = GetGameTime()
		self.secondsDifference = mod(time(), 60)
	end
	if self:IsBothTimes() then
		if self:IsLocalTime() then
			self:SetText(string.format("%s || %s", self:GetLocalTime(who,true), self:GetServerTime(who,true)))
		else
			self:SetText(string.format("%s || %s", self:GetServerTime(who,true), self:GetLocalTime(who,true)))
		end
	elseif self:IsLocalTime() then
		self:SetText(self:GetLocalTime(who,true))
	else
		self:SetText(self:GetServerTime(who,true))
	end
end

function FuzzyClockFu:OnTooltipUpdate()
	who = "ttu"
	local s = self:GetServerOffset()
	local l = self:GetLocalOffset()
	
	local cat = Tablet:AddCategory(
		'columns', 2,
		'child_textR', 1,
		'child_textG', 1,
		'child_textB', 0,
		'child_text2R', 1,
		'child_text2G', 1,
		'child_text2B', 1
	)
	cat:AddLine(
		'text', string.format(L["Local time"] .. " (%+03d:%02d)", l, mod(l * 60, 60)),
		'text2', self:GetLocalTime(who)
	)
	cat:AddLine(
		'text', string.format(L["Server time"] .. " (%+03d:%02d)", s, mod(s * 60, 60)),
		'text2', self:GetServerTime(who)
	)
	cat:AddLine(
		'text', L["UTC"],
		'text2', self:GetUtcTime(who)
	)
	
	local DATE = L["DATE"]
	if type(DATE) == "string" then
		DATE = date(DATE)
	else
		DATE = DATE()
	end
	Tablet:AddCategory():AddLine(
		'text', DATE,
		'textR', 1,
		'textG', 1,
		'textB', 1,
		'justify', "CENTER"
	)
end
