local L = AceLibrary("AceLocale-2.2"):new("FarmerFu")
local Crayon = AceLibrary("Crayon-2.0")
local Tablet = AceLibrary("Tablet-2.0")
local PT = AceLibrary("LibPeriodicTable-3.1")

local options = {
	handler = FarmerFu,
	type = "group",
	args = {
		add = {
			type = "text",
			order = 1.0,
			name = L["Add Item"],
			desc = L["Add an item"],
			get = false,
			set = "AddItem",
			usage = L["AddItem-usage"],
		},
		clear = {
			type = "execute",
			order = 1.1,
			name = L["Clear Items"],
			desc = L["Clear all items from the list"],
			func = "ClearItems",
		},
		sep1 = {
			type = "header",
			order = 1.2,
		},
		showtext = {
			type = "group",
			order = 1.3,
			name = L["Text"],
			desc = L["Text Settings"],
			args = {
				bags = {
					type = "toggle",
					order = 1.31,
					name = L["Show Bags"],
					desc = L["Show number of items in bags"],
					get = "IsShowTextBags",
					set = "ToggleShowTextBags",
				},
				bank = {
					type = "toggle",
					order = 1.32,
					name = L["Show Bank"],
					desc = L["Show number of items in bank"],
					get = "IsShowTextBank",
					set = "ToggleShowTextBank",
				},
				total = {
					type = "toggle",
					order = 1.33,
					name = L["Show Total"],
					desc = L["Show number of items total"],
					get = "IsShowTextTotal",
					set = "ToggleShowTextTotal",
				},
			},
		},
		showtooltip = {
			type = "group",
			order = 1.4,
			name = L["Tooltip"],
			desc = L["Tooltip Settings"],
			args = {
				headers = {
					type = "toggle",
					order = 1.41,
					name = L["Show Headers"],
					desc = L["Show column headers"],
					get = "IsShowTooltipHeaders",
					set = "ToggleShowTooltipHeaders",
				},
				bags = {
					type = "toggle",
					order = 1.42,
					name = L["Show Bags"],
					desc = L["Show number of items in bags"],
					get = "IsShowTooltipBags",
					set = "ToggleShowTooltipBags",
				},
				bank = {
					type = "toggle",
					order = 1.43,
					name = L["Show Bank"],
					desc = L["Show number of items in bank"],
					get = "IsShowTooltipBank",
					set = "ToggleShowTooltipBank",
				},
				total = {
					type = "toggle",
					order = 1.44,
					name = L["Show Total"],
					desc = L["Show number of items total"],
					get = "IsShowTooltipTotal",
					set = "ToggleShowTooltipTotal",
				},
				sep1 = {
					type = "header",
					order = 1.45,
				},
				nonzeroonly = {
					type = "toggle",
					order = 1.46,
					name = L["Hide Zero Amounts"],
					desc = L["Hide items that are not in the bags or bank"],
					get = "IsShowTooltipNonZeroOnly",
					set = "ToggleShowTooltipNonZeroOnly",
				},
				hideitems = {
					type = "group",
					order = 1.47,
					name = L["Hide Items"],
					desc = L["Hide specific items on the tooltip.  They will still be tracked."],
					args = {},
				},
			},
		},
	},
}

FarmerFu = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceDB-2.0", "FuBarPlugin-2.0", "AceDebug-2.0")

FarmerFu.version = "2.0."..string.sub("$Revision: 63418 $", 12, -3)
FarmerFu.date = string.sub("$Date: 2008-03-02 20:45:53 -0500 (Sun, 02 Mar 2008) $", 8, 17)
FarmerFu.hasIcon = "Interface\\Icons\\Spell_Nature_Polymorph"
FarmerFu.clickableTooltip = true
FarmerFu.independentProfile = true
FarmerFu.tooltipHiddenWhenEmpty = true
FarmerFu.debugFrame = ChatFrame5

FarmerFu:RegisterChatCommand({"/farmerfu", "/farmfu", "/ffu"}, options)
FarmerFu.OnMenuRequest = options

FarmerFu:RegisterDB("FarmerFu2DB")
FarmerFu:RegisterDefaults("profile", {
	items = {},
	watch = nil,
	hidden = {},
	showtext = {
		bags = true,
		bank = true,
		total = true,
	},
	showtooltip = {
		headers = true,
		bags = true,
		bank = true,
		total = true,
		nonzeroonly = false,
	},
})

function FarmerFu:OnInitialize()
	self.items = {}
end

function FarmerFu:OnEnable()
	self:SetDebugging(true)
	self:RegisterEvent("BAG_UPDATE", 1)
	self:Update()
end

function FarmerFu:OnProfileEnable()
	self:EraseTable(self.items)
	self:Update()
end

function FarmerFu:BAG_UPDATE(bag)
	-- Ammo bags for hunters throw a bazillion events.  Let's try
	-- to cut down on the processing we do when they pew pew.  Ignore 
	-- the ammo bag entirely unless we are farming ammo for some reason.
	if self:IsAmmoBag(bag) and not self:IsFarmingAmmo() then
		return
	end
	self:Update()
end

function FarmerFu:IsAmmoBag(bag)
	-- Bag 0 is the backpack and -2 is the keyring
	if bag ~= 0 and bag ~= -2 then
		local invid = ContainerIDToInventoryID(bag)
		local baglink = GetInventoryItemLink("player", invid)
		if baglink then
			local ammo = PT:ItemInSet(baglink, "Bag.Special.Ammo")
			local quiver = PT:ItemInSet(baglink, "Bag.Special.Quiver")
			return ammo or quiver
		end
	end
	return false
end

function FarmerFu:IsFarmingAmmo()
	for i = 1, #self.db.profile.items do
		local itemid = self.db.profile.items[i].id
		local arrows = PT:ItemInSet(itemid, "Reagent.Ammo.Arrow")
		local bullets = PT:ItemInSet(itemid, "Reagent.Ammo.Bullet")
		if arrows or bullets then
			return true
		end
	end
	return false
end

-- Recycle tables, thanks FactionsFu
local tablecache = {}
local function recycleitemtable(t)
	while #t > 0 do
		table.insert(tablecache, table.remove(t))
	end
	return t
end
local function getitemtable(...)
	local t
	if #tablecache > 0 then
		t = table.remove(tablecache)
	else
		t = {}
	end
	for i = 1, select('#', ...), 2 do
		t[select(i, ...)] = select(i + 1, ...)
	end
	return t
end

function FarmerFu:OnDataUpdate()
	self:GetItems()
	self:UpdateHiddenItems()
end

local tmptext = {}
function FarmerFu:OnTextUpdate()
	local text = self:GetTitle()
	if #self.items > 0 then
		local item = self:GetWatchedItem()
		if item then
			text = Crayon:Green(item.name)
			if self:IsShowTextBags() then
				table.insert(tmptext, item.bags)
			end
			if self:IsShowTextBank() then
				table.insert(tmptext, item.bank)
			end
			if self:IsShowTextTotal() then
				table.insert(tmptext, item.total)
			end
			if #tmptext > 0 then
				text = text.." ("..table.concat(tmptext, "/")..")"
			end
			self:EraseTable(tmptext)
		end
	end
	self:SetText(text)
end

function FarmerFu:GetWatchedItem()
	for i = 1, #self.items do
		if self.items[i].id == self.db.profile.watch then
			return self.items[i]
		end
	end
end

local tmpcols = {}
local tmpcat = {}
local tmpline = {}
function FarmerFu:OnTooltipUpdate()
	if #self.items > 0 then
		table.insert(tmpcols, L["Item"])
		if self:IsShowTooltipBags() then
			table.insert(tmpcols, L["Bag"])
		end
		if self:IsShowTooltipBank() then
			table.insert(tmpcols, L["Bank"])
		end
		if self:IsShowTooltipTotal() then
			table.insert(tmpcols, L["Total"])
		end
		
		local category = Tablet:AddCategory(
			"columns", #tmpcols
		)

		if self:IsShowTooltipHeaders() then
			for i = 1, #tmpcols do
				if i == 1 then
					tmpcat["text"] = tmpcols[i]
				else
					tmpcat["text"..i] = tmpcols[i]
					tmpcat["justify"..i] = "CENTER"
				end
			end
		end

		category:AddLine(tmpcat)
		
		for i = 1, #self.items do
			if not self.items[i].hidden then
				tmpline["text"] = self.items[i].name
				tmpline["textR"] = 1
				tmpline["textG"] = 1
				tmpline["textB"] = 0

				local col = 2
				if self:IsShowTooltipBags() then
					tmpline["text"..col] = self.items[i].bags
					tmpline["justify"..col] = "CENTER"
					tmpline["text"..col.."R"] = 1
					tmpline["text"..col.."G"] = 1
					tmpline["text"..col.."B"] = 1
					col = col + 1
				end
				if self:IsShowTooltipBank() then
					tmpline["text"..col] = self.items[i].bank
					tmpline["justify"..col] = "CENTER"
					tmpline["text"..col.."R"] = 1
					tmpline["text"..col.."G"] = 1
					tmpline["text"..col.."B"] = 1
					col = col + 1
				end
				if self:IsShowTooltipTotal() then
					tmpline["text"..col] = self.items[i].total
					tmpline["justify"..col] = "CENTER"
					tmpline["text"..col.."R"] = 1
					tmpline["text"..col.."G"] = 1
					tmpline["text"..col.."B"] = 1
					col = col + 1
				end
				
				tmpline["hasCheck"] = true
				tmpline["checked"] = self.items[i].id == self.db.profile.watch
				tmpline["func"] = "OnClickItem"
				tmpline["arg1"] = self
				tmpline["arg2"] = self.items[i].id
				
				category:AddLine(tmpline)
			end
		end
		
		Tablet:SetHint(L["Tooltip-hint"])

		self:EraseTable(tmpcols)
		self:EraseTable(tmpcat)
		self:EraseTable(tmpline)
	end
end

function FarmerFu:OnClickItem(itemid)
	if IsShiftKeyDown() then
	elseif IsAltKeyDown() then
		self:RemoveItem(itemid)
	else
		if self.db.profile.watch == itemid then
			PlaySound("igMainMenuOptionCheckBoxOff")
			self.db.profile.watch = nil
		else
			PlaySound("igMainMenuOptionCheckBoxOn")
			self.db.profile.watch = itemid
		end
	end
	self:Update()
end

function FarmerFu:GetItems()
	local items = recycleitemtable(self.items)
	for i = 1, #self.db.profile.items do
		local itemid = self.db.profile.items[i].id
		local name = GetItemInfo(itemid)
		if name then
			local bags = GetItemCount(itemid)
			local bank = GetItemCount(itemid, true) - bags
			local total = bags + bank
			local hidden = self.db.profile.items[i].hidden
			if not hidden and self:IsShowTooltipNonZeroOnly() and total == 0 then
				hidden = true
			end
			table.insert(items, getitemtable("id", itemid, "hidden", hidden, "name", name, "bags", bags, "bank", bank, "total", total))
		end
	end
	table.sort(items, function(a, b) return a.name < b.name end)
	self.items = items
end

function FarmerFu:AddItem(item)
	local _, link = GetItemInfo(item)
	if not link then
		self:Print("Item '"..item.."' not found in local cache.  Please use the item id or link.")
		return
	end

	local itemid = self:GetItemId(link)
	if itemid then
		for i = 1, #self.db.profile.items do
			if self.db.profile.items[i].id == itemid then
				return
			end
		end
		table.insert(self.db.profile.items, {id = itemid, hidden = false})
		self:Update()
	end
end

function FarmerFu:RemoveItem(itemid)
	if itemid then
		for i = 1, #self.db.profile.items do
			if self.db.profile.items[i].id == itemid then
				table.remove(self.db.profile.items, i)
				break
			end
		end
		if self.db.profile.watch == itemid then
			self.db.profile.watch = nil
		end
	end
end

function FarmerFu:ClearItems()
	self.db.profile.watch = nil
	self.db.profile.items = {}
	self.items = {}
	self:Update()
end

function FarmerFu:UpdateHiddenItems()
	self:EraseTable(options.args.showtooltip.args.hideitems.args)
	if #self.items == 0 then
		options.args.showtooltip.args.hideitems.disabled = true
	else
		options.args.showtooltip.args.hideitems.disabled = false
		for i = 1, #self.items do
			options.args.showtooltip.args.hideitems.args[i] = {}
			options.args.showtooltip.args.hideitems.args[i].type = "toggle"
			options.args.showtooltip.args.hideitems.args[i].name = self.items[i].name
			options.args.showtooltip.args.hideitems.args[i].desc = L["Hide item on tooltip"]
			options.args.showtooltip.args.hideitems.args[i].get = function() return self:IsItemHidden(self.items[i].id) end
			options.args.showtooltip.args.hideitems.args[i].set = function() return self:ToggleItemHidden(self.items[i].id) end
		end
	end
end

function FarmerFu:GetItemId(link)
	local foundstring, _, itemstring = string.find(link, "^|%x+|H(.+)|h%[.+%]")
	if foundstring then
		local foundid, _, itemid = string.find(itemstring, "^item:(%d+):%d+:%d+:%d+:%d+:%d+:%-?%d+:%-?%d+")
		if foundid then
			return itemid
		end
	end
end

function FarmerFu:EraseTable(table)
	for k in pairs(table) do
		table[k] = nil
	end
end

function FarmerFu:IsShowTextBags()
	return self.db.profile.showtext.bags
end

function FarmerFu:ToggleShowTextBags()
	self.db.profile.showtext.bags = not self.db.profile.showtext.bags
	self:Update()
	return self.db.profile.showtext.bags
end

function FarmerFu:IsShowTextBank()
	return self.db.profile.showtext.bank
end

function FarmerFu:ToggleShowTextBank()
	self.db.profile.showtext.bank = not self.db.profile.showtext.bank
	self:Update()
	return self.db.profile.showtext.bank
end

function FarmerFu:IsShowTextTotal()
	return self.db.profile.showtext.total
end

function FarmerFu:ToggleShowTextTotal()
	self.db.profile.showtext.total = not self.db.profile.showtext.total
	self:Update()
	return self.db.profile.showtext.total
end

function FarmerFu:IsShowTooltipHeaders()
	return self.db.profile.showtooltip.headers
end

function FarmerFu:ToggleShowTooltipHeaders()
	self.db.profile.showtooltip.headers = not self.db.profile.showtooltip.headers
	self:Update()
	return self.db.profile.showtooltip.headers
end

function FarmerFu:IsShowTooltipBags()
	return self.db.profile.showtooltip.bags
end

function FarmerFu:ToggleShowTooltipBags()
	self.db.profile.showtooltip.bags = not self.db.profile.showtooltip.bags
	self:Update()
	return self.db.profile.showtooltip.bags
end

function FarmerFu:IsShowTooltipBank()
	return self.db.profile.showtooltip.bank
end

function FarmerFu:ToggleShowTooltipBank()
	self.db.profile.showtooltip.bank = not self.db.profile.showtooltip.bank
	self:Update()
	return self.db.profile.showtooltip.bank
end

function FarmerFu:IsShowTooltipTotal()
	return self.db.profile.showtooltip.total
end

function FarmerFu:ToggleShowTooltipTotal()
	self.db.profile.showtooltip.total = not self.db.profile.showtooltip.total
	self:Update()
	return self.db.profile.showtooltip.total
end

function FarmerFu:IsShowTooltipNonZeroOnly()
	return self.db.profile.showtooltip.nonzeroonly
end

function FarmerFu:ToggleShowTooltipNonZeroOnly()
	self.db.profile.showtooltip.nonzeroonly = not self.db.profile.showtooltip.nonzeroonly
	self:Update()
	return self.db.profile.showtooltip.nonzeroonly
end

function FarmerFu:IsItemHidden(itemid)
	for i = 1, #self.db.profile.items do
		if itemid == self.db.profile.items[i].id then
			return self.db.profile.items[i].hidden
		end
	end
end

function FarmerFu:ToggleItemHidden(itemid)
	local hidden = false
	for i = 1, #self.db.profile.items do
		if itemid == self.db.profile.items[i].id then
			self.db.profile.items[i].hidden = not self.db.profile.items[i].hidden
			hidden = self.db.profile.items[i].hidden
		end
	end
	self:Update()
	return hidden
end
