local dewdrop = AceLibrary("Dewdrop-2.0")
local tablet = AceLibrary("Tablet-2.0")
local crayon = AceLibrary("Crayon-2.0")
local L = AceLibrary("AceLocale-2.2"):new("DebuggerFu")

BINDING_HEADER_DBFU_TITLE = "DebuggerFu Bindings";
BINDING_NAME_DBFU_TOOLTIP = "Toggle Attached Tooltip";

local MaxHardLineProtection = 10000;

local fontslist = {}
local optionsTable = {
	type = 'group',
	args = {
		enterVar = {
			type = "text",
			name = "Debug Variable",
			desc = "Enter the name of the variable to debug here",
			get = function()
				return DebuggerFu.db.profile.newRootVariable
			end,
			set = function(value)
				DebuggerFu.db.profile.newRootVariable = value;
				--DEFAULT_CHAT_FRAME:AddMessage("Set : "..DebuggerFu.db.profile.newRootVariable)				
				dewdrop:Close()
				local found=false;
				for _, i in ipairs(DebuggerFu.db.profile.browseHistory) do
					if i == DebuggerFu.db.profile.newRootVariable then
						found = true;
						break;
					end
				end
				if (found == false) then
					table.insert(DebuggerFu.db.profile.browseHistory,1,DebuggerFu.db.profile.newRootVariable);
				end
				DebuggerFu.db.profile.browsePos = 1;
				DebuggerFu:BuildTree(DebuggerFu.db.profile.newRootVariable);
				--DebuggerFu:CalcTooltipVarDisplaySize();
			end,
			validate = function(v) 
				return 1 
			end,
			usage = "name",			
			order = 15
		},					
		["-blank-"] = {
			order = 20,
			type = 'header',
		},		
		variableOptions = {
			order = 25,
			name = "Variable display", type = "group",
			desc = "Options for variable display.",
			args = {
				showVarName = {
					type = "toggle",
					name = "Variable name",
					desc = "Show the name of the selected variable",
					get = function()
						return DebuggerFu.db.profile.showVarName
					end,
					set = function(value)
						DebuggerFu.db.profile.showVarName = value;
						DebuggerFu:CalcTooltipVarDisplaySize();
					end,
					order = 16
				},						
				showFullVarName = {
					type = "toggle",
					name = "Full Variable name",
					desc = "Show the full name of the selected variable (only valid in tooltip)",
					get = function()
						return DebuggerFu.db.profile.showFullVarName
					end,
					set = function(value)
						DebuggerFu.db.profile.showFullVarName = value;				
						DebuggerFu:Update();
					end,
					disabled = function()
						if DebuggerFu.db.profile.showVarName then
							return false;
						else
							return true;
						end
					end,						
					order = 17
				},								
				showType = {
					type = "toggle",
					name = "Show Type",
					desc = "Show the type of the selected variable",
					get = function()
						return DebuggerFu.db.profile.showType
					end,
					set = function(value)
						DebuggerFu.db.profile.showType = value;
						DebuggerFu:CalcTooltipVarDisplaySize();
					end,
					order = 18
				},				
				showValue = {
					type = "toggle",
					name = "Show Value",
					desc = "Show the value of the selected variable",
					get = function()
						return DebuggerFu.db.profile.showValue
					end,
					set = function(value)
						DebuggerFu.db.profile.showValue = value;
						DebuggerFu:CalcTooltipVarDisplaySize();
					end,
					order = 19
				},						
				["-blank202-"] = {
					order = 20,
					type = 'header',
				},													
				showDepth = {
					type = "toggle",
					name = "Show Depth",
					desc = "Shows the depths of the table nodes in brackets at the end",
					get = function()
						return DebuggerFu.db.profile.showDepth
					end,
					set = function(value)
						DebuggerFu.db.profile.showDepth = value;
						DebuggerFu:Update();
					end,
					order = 21
				},														
				showValueInTree = {
					type = "toggle",
					name = "Show value in tree",
					desc = "Shows the value of the variable in the tree",
					get = function()
						return DebuggerFu.db.profile.showValueInTree
					end,
					set = function(value)
						DebuggerFu.db.profile.showValueInTree = value;
						DebuggerFu:Update();
					end,
					order = 22
				},														
				showTypeInTree = {
					type = "toggle",
					name = "Show type in tree",
					desc = "Shows the variable type in the tree in {} brackets",
					get = function()
						return DebuggerFu.db.profile.showTypeInTree
					end,
					set = function(value)
						DebuggerFu.db.profile.showTypeInTree = value;
						DebuggerFu:Update();
					end,
					order = 22
				},																		
			}
		},									
		maxTreeDepth = {
			type = 'range',
			name = 'Tree Depth',
			desc = 'How many depth levels should be displayed in tree (default 6)',
			get = function()
				return DebuggerFu.db.profile.maxDepth;
			end,
			set = function(v)
				DebuggerFu.db.profile.maxDepth = v
			end,
			min = 1,
			max = 10,
			step = 1,
			order = 30
		},		
		maxNodeProtection = {
			type = 'range',
			name = 'Max Line Protection',
			desc = 'Limit the analysation to this amount of lines. (default 1000)',
			get = function()
				if (DebuggerFu.db.profile.maxLineProtection == MaxHardLineProtection) then
					DebuggerFu.db.profile.maxLineProtection = 0;
				end
				return DebuggerFu.db.profile.maxLineProtection;
			end,
			set = function(v)
				DebuggerFu.db.profile.maxLineProtection = v
				if (maxLineProtection == 0) then
					maxLineProtection = MaxHardLineProtection;
				end
			end,
			min = 0,
			max = 2000,
			step = 100,
			order = 33
		},				
		["-blank20-"] = {
			order = 35,
			type = 'header',
		},									
		barDataDisplay = {
			type = "toggle",
			name = "Data display in bar",
			desc = "Display the data info in bar instead of in tooltip (see next setting as well)",
			get = function()
				return DebuggerFu.db.profile.barDataDisplay
			end,
			set = function(value)
				DebuggerFu.db.profile.barDataDisplay = value;
				DebuggerFu:Update();
			end,
			order = 40
		},		
		maxTooltipLines = {
			type = 'range',
			name = 'Max Tooltip Lines',
			desc = 'If lines displayed in tooltip exceeds this number display data in bar',
			get = function()
				return DebuggerFu.db.profile.lineThreshhold;
			end,
			set = function(v)
				DebuggerFu.db.profile.lineThreshhold = v
				DebuggerFu:UpdateText();
			end,
			disabled = function()
				if DebuggerFu.db.profile.barDataDisplay then
					return false;
				else
					return true;
				end
			end,			
			min = 5,
			max = 70,
			step = 1,
			order = 45
		},
		autoDetach = {
			type = "toggle",
			name = "Auto Detach Tooltip",
			desc = "Automatically detaches tooltip when the status bar is used for dislaying variables to prevent tooltip from jumping around",
			get = function()
				return DebuggerFu.db.profile.autoDetach
			end,
			set = function(value)
				DebuggerFu.db.profile.autoDetach = value;
				DebuggerFu:UpdateText();
			end,
			disabled = function()
				if DebuggerFu.db.profile.barDataDisplay then
					return false;
				else
					return true;
				end
			end,						
			order = 50
		},				
		autoAttach = {
			type = "toggle",
			name = "Auto Attach Tooltip",
			desc = "Automatically attaches tooltip when the status bar is no longer used for dislaying variables (displayed in tooltip)",
			get = function()
				return DebuggerFu.db.profile.autoAttach
			end,
			set = function(value)
				DebuggerFu.db.profile.autoAttach = value;
				DebuggerFu:UpdateText();
			end,
			disabled = function()
				if DebuggerFu.db.profile.barDataDisplay then
					return false;
				else
					return true;
				end
			end,						
			order = 55
		},		
		["-blank25-"] = {
			order = 60,
			type = 'header',
		},									
		wipeLastTooltip = {
			type = "toggle",
			name = "Wipe Tooltip on reload",
			desc = "Wipe the tooltip when UI reloaded / restarted (otherwise data will be remembered)",
			get = function()
				return DebuggerFu.db.profile.wipeLastTooltip
			end,
			set = function(value)
				DebuggerFu.db.profile.wipeLastTooltip = value;
			end,
			order = 65
		},				
		debugBoxOptions = {
			order = 70,
			name = "Debug Box Options", type = "group",
			desc = "Set options for the debug box.",
			args = {
				editBoxWidth = {
					type = 'range',
					name = 'EditBox Width',
					desc = 'Sets the width of your editbox',
					get = function()
						return DebuggerFu.db.profile.editBoxWidth;
					end,
					set = function(v)
						DebuggerFu.db.profile.editBoxWidth = v
						DebuggerFu:CheckEditBoxParameters();
					end,
					min = 120,
					max = 500,
					step = 10,
					order = 1
				},
				editBoxAlpha = {
					type = 'range',
					name = 'EditBox Alpha',
					desc = 'Sets the alpha value of your editbox',
					get = function()
						return DebuggerFu.db.profile.editBoxAlpha;
					end,
					set = function(v)
						DebuggerFu.db.profile.editBoxAlpha = v
						DebuggerFu:CheckEditBoxParameters();
					end,
					min = 0.4,
					max = 1,
					step = 0.1,
					order = 2
				},				
				closeOnEnter = {
					type = 'toggle',
					name = 'Close on Enter',
					desc = 'If on it closes the editbox with just the normal enter',
					get = function()
						return DebuggerFu.db.profile.closeOnEnter;
					end,
					set = function(v)
						DebuggerFu.db.profile.closeOnEnter = v
					end,
					order = 3
				},				
				clearHistory = {
					order = 4,
					type = "execute",
					name = "Clear Browse History",
					desc = "Clears the browse history of all entries",
					func = function(v)
						DebuggerFu.db.profile.browseHistory = {};
						DebuggerFu:Debug("Browse history erased !",0);
					end,
				},				
			}
		},		
		showBlank = {
			type = "toggle",
			name = "Show Blank Text",
			desc = "Show a blank text instead of Debugger while showing nothing",
			get = function()
				return DebuggerFu.db.profile.showBlank
			end,
			set = function(value)
				DebuggerFu.db.profile.showBlank = value;
				DebuggerFu:UpdateText();
			end,
			order = 75
		},						
		["-blank29-"] = {
			order = 80,
			type = 'header',
		},			
		debuggingLevel = {
			type = 'range',
			name = 'Debug Level',
			desc = 'Choose level for debugging this tool (-1 = no messages at all) (0 = standard info messages) (>0 for debugging purpose only) ',
			get = function()
				return DebuggerFu.db.profile.debugLevel
			end,
			set = function(v)
				DebuggerFu.db.profile.debugLevel = v
			end,
			min = -1,
			max = 5,
			step = 1,
			order = 85
		},				
		showLogWindow = {
			type = "toggle",
			name = "Toggle Log Window",
			desc = "Toggle The Log Window If Debug-1.0 is installed",
			get = function()
				return getglobal("Debug10LogFrame"):IsShown()
			end,
			set = function(value)
			    if value then 
			        getglobal("Debug10LogFrame"):Show()
			    else
			        getglobal("Debug10LogFrame"):Hide()
			    end
			end,
			hidden = function() 
			    return not AceLibrary:HasInstance("DebugStub-1.0", false) end,
			order = 90
		},		
    logWindowFont = {
    	name = "Set Log Font",
      desc = "Set the log font",
      type = "text",
      order = 95,
      get = "GetLogFont",
      set = "SetLogFont",
      validate = fontslist,
			hidden = function() 
				return not AceLibrary:HasInstance("DebugStub-1.0", false) 
			end,
  	},		
	}	
}

DebuggerFu = AceLibrary("AceAddon-2.0"):new("FuBarPlugin-2.0", "AceEvent-2.0", "AceConsole-2.0", "AceDB-2.0")
DebuggerFu.hasIcon = true
DebuggerFu.clickableTooltip = true
DebuggerFu.tooltipHiddenWhenEmpty = true

DebuggerFu.OnMenuRequest = optionsTable
DebuggerFu:RegisterChatCommand( { "/DebuggerFu" }, optionsTable )

DebuggerFu:RegisterDB("DebuggerFuDB")
DebuggerFu:RegisterDefaults('profile', {
	varTree = {},
	browseHistory = {}, -- saves all inputs in debug text frame
	browsePos = 1,	-- the present position while browsing the history
	closeOnEnter = nil, -- if set , edit box closes on normal enter
	editBoxWidth = 200,
	editBoxAlpha = 0.8,
	showMiniMap = false,
	displayIndex = nil,
	showVarName = nil,
	showFullVarName = nil,
	showValue = 1,
	showType = 1,
	autoDetach =1,
	autoAttach = 1,
	lineThreshhold = 49,
	showDepth = 1, -- set to 1 if the depth of the table nodes is shown behind em
	newTreeBuild = 0,
	maxDepth = 6,
	varDisplaySize =4,  --internal value to figure out how many lines the var display takes in tooltip
	wipeLastTooltip =1, 
	displayedVariable = nil, -- the last querried variable for build tree (might be invalid)
	rootVariable = nil, -- the root variable of the search
	newRootVariable = nil, -- new variable to use will be rootVariable if it is confirmed working
	editMode = nil, -- if set to 1 indicates that editing is taking place
	editIndex = nil, -- set to the index that is edited, if editmode is on
	lastEditBoxText = nil, -- if this is set, the last editboxtext that was erased by editing is stored here
	editBoxVisible = nil, -- indicates if the editbox was visible b4 editing or not
	showValueInTree = 1, -- 1 means the var value will be shown in tooltiptree
	showTypeInTree = 1, -- 1 means the var type will be shown in tooltiptree
	debugLevel= 0,
	maxLineProtection= 1000,
})

local media 

function DebuggerFu:SetLogFont(font)
    local f = getglobal("Debug10LogFrame")
    local fontfile = media:Fetch(media.MediaType.FONT, font)
    local _, s, m = f.Box:GetFont()    
        
    f.Box:SetFont(fontfile, s, m)
    
    for k,v in pairs(f.Lines) do
         _, s, m = v.text:GetFont()  
        v.text:SetFont(fontfile, s, m)
    end
    
   -- Message.Box:SetFont
   -- Message.LevelFilt
end

function DebuggerFu:GetLogFont()
    local f = getglobal("Debug10LogFrame")
    
    return f and f.Box:GetFont()
end


function DebuggerFu:BuildFontList()
    for i,v in ipairs(fontslist) do
        fontslist[i] = nil
    end
    
    for k,v in pairs(media:List(media.MediaType.FONT)) do
        table.insert(fontslist, v)
    end
end
function DebuggerFu:SharedMedia_Registered(mediatype, name)
    if mediatype == media.MediaType.FONT then
        self:BuildFontList()
    end
end

-- Methods

function DebuggerFu:OnInitialize()
	--self:DetachTooltip();
end
	
function DebuggerFu:OnEnable()
    media = media or AceLibrary("SharedMedia-1.0")
	self:ReattachTooltip();
	self:CalcTooltipVarDisplaySize();
	if (self.db.profile.wipeLastTooltip) then
		self.db.profile.varTree = {};
	end
	
	self:CheckEditBoxParameters();
	self.db.profile.browsePos = 1;
	if self.db.profile.browseHistory and (self.db.profile.browseHistory[self.db.profile.browsePos]) then
			self.editBox:SetText(self.db.profile.browseHistory[self.db.profile.browsePos]);
	end
		
    self:RegisterEvent("SharedMedia_Registered")	
    self:BuildFontList()	
	--self:ScheduleRepeatingEvent("DebuggerFu_Update",self.ToggleTooltipDetached, 2, self);
	--self:ScheduleEvent("DebuggerFu_Update",self.CheckTooltipAttachment, 1.5, self);		
end

function DebuggerFu:CheckTooltipAttachment(onlyAttach)
	if (self:UseBarDisplay()) then		
		if (not onlyAttach) and (self.db.profile.autoDetach) and (self:IsTooltipDetached() == false) then			
			self:DetachTooltip();
			--self:UpdateTooltip();			
		end		
	elseif (self:IsTooltipDetached() == true) then		
		if (self.db.profile.autoAttach) then
			self:ReattachTooltip();
			--self:UpdateTooltip();
		end				
	end		
end

function DebuggerFu:OnDisable()
	if (self.db.profile.wipeLastTooltip) then
		self.db.profile.varTree = {};
	end
end

function DebuggerFu:CalcTooltipVarDisplaySize()
	self.db.profile.varDisplaySize = 1;
	if self.db.profile.showVarName then
		self.db.profile.varDisplaySize = self.db.profile.varDisplaySize +1;
	end
	if self.db.profile.showValue then
		self.db.profile.varDisplaySize = self.db.profile.varDisplaySize +1;
	end
	if self.db.profile.showType then
		self.db.profile.varDisplaySize = self.db.profile.varDisplaySize +1;
	end						
	
	if (self.db.profile.varDisplaySize == 1) then -- nothing to display
		self.db.profile.varDisplaySize = nil;
	end
end

	
function DebuggerFu:OnTextUpdate()	
	local txt;
	if (self.db.profile.showBlank) then
		txt = "";
	else
		txt = "Debugger";
	end
	if (self.db.profile.varTree[1]) and (self.db.profile.varDisplaySize) and (self:UseBarDisplay()) then
		
		if (self.db.profile.displayIndex) and (self.db.profile.varTree[self.db.profile.displayIndex]) then			
			
			local pv = self.db.profile.varTree[self.db.profile.displayIndex];

			if (self.db.profile.editMode) and (self.db.profile.editIndex) then -- in edit mode
				txt = ""..crayon:Red("Edit: ");
			else
				txt = "";
			end		
			
			if (self.db.profile.showVarName) then
				if (self.db.profile.showFullVarName) then
					txt = txt..crayon:Orange(self:GetFullVarName(self.db.profile.displayIndex)).." ";
				else				
					txt = txt..crayon:Orange(pv.name).." ";
				end
			end
			
			if type(pv.value) ~= "nil" then
				--DEFAULT_CHAT_FRAME:AddMessage("Adding : "..type(pv.value))				
				if (self.db.profile.showType) then
					txt = txt..crayon:Colorize("00ff00",type(pv.value)).." ";
				end
				
				if (self.db.profile.showValue) then
					if (type(pv.value) == "string") then
						txt = txt..crayon:Colorize("ffffff","\'"..pv.value.."\'");
					else
						txt = txt..crayon:Colorize("ffffff",pv.value);
					end				
				end
			else
				if (self.db.profile.showType) then				
					txt = txt..crayon:Colorize("00ff00","table").." ";			
				end
				if (self.db.profile.showValue) then
					if (self.db.profile.showDepth) then
						txt = txt..crayon:Colorize("ffffff","["..pv.depth.."]");
					else
						txt = txt..crayon:Colorize("ffffff","-");
					end
				end
			end				
		else
			txt = "";			
			if (self.db.profile.showVarName) then
				txt = txt..crayon:Orange("-").." ";
			end			
			if (self.db.profile.showType) then				
				txt = txt..crayon:Colorize("00ff00","-").." ";			
			end
			if (self.db.profile.showValue) then
				txt = txt..crayon:Colorize("ffffff","-");
			end
		end
	end
	self:SetText(txt);
end

function DebuggerFu:TreeUp()	
	local found = true;
	local firstStep = true;
	local name = "";

	local variableName = self.db.profile.displayedVariable;
	local variable;
	
	found,_,part_one,part_two = string.find(variableName,"(.+)%.(.+)")		
	if (found) then
		self:BuildTree(tostring(part_one));			
	end	
	
	--DEFAULT_CHAT_FRAME:AddMessage("Tree up command "..self.db.profile.displayedVariable);	
end

function DebuggerFu:EndEditing()	
	if (self.db.profile.editMode) then
		self.db.profile.editMode = nil;
		self.db.profile.editIndex = nil;			
		if (self.db.profile.lastEditBoxText) then
			self.editBox:SetText(self.db.profile.lastEditBoxText);
		end
		self.db.profile.lastEditBoxText = nil;
		self.editBox:SetTextColor(1,1,1);
		if (self.db.profile.editBoxVisible) then
			self.editFrame:Show();
		else
			self.editFrame:Hide();
		end
		self.editBox:SetNumeric(false);
		self:Update();					
	end
end

function DebuggerFu:StartEditing(index)	
	if (index) then
		self.db.profile.editMode = 1;
		self.db.profile.editIndex = index;
		self.db.profile.editBoxVisible = self.editBox:IsVisible(); 
			
		local pv = self.db.profile.varTree[self.db.profile.editIndex];
			
		self.editBox:SetTextColor(1,0.5,0);
		self.db.profile.lastEditBoxText = self.editBox:GetText();					
		self.editBox:SetText(pv.value);		
		if (type(pv.value) == "number") then
			self.editBox:SetNumeric(true);
		end
		self.editFrame:Show();
	end
end

function DebuggerFu:ToggleAllNodes(depth,expand)	
	if (depth) then
		--DEFAULT_CHAT_FRAME:AddMessage("ToggleAll "..depth.." - "..expand);
		for _,i in ipairs(self.db.profile.varTree) do			
			if (i.expanded) and (i.depth == depth) then -- is a table
				i.expanded = expand;
			end
		end		
	end
end



function DebuggerFu:OnItemClick(index)	
		
	self:EndEditing();
	
	if (index <= 0) or (not self.db.profile.varTree[index]) then
		return;
	end
	
	local selectNewTree = 1;	
	local presentVar;
	
	if (IsShiftKeyDown()) then
		if (self.db.profile.varTree[index].expanded) then -- is a table
			node = self.db.profile.varTree[index];
			DebuggerFu:Debug("clicked "..node.name.." pindex : "..node.parentIndex,1);
			presentVar = self.db.profile.displayedVariable;			
			if (node.parentIndex ~= 0) then			
				subnode =	self.db.profile.varTree[node.parentIndex];
				if (subnode.parentIndex == 0) then -- this is a node 1 above parent										
					presentVar = presentVar.."."..node.name;
					DebuggerFu:Debug("1st Child clicked "..node.name.." "..presentVar,1);
				else
					subnode_two =	self.db.profile.varTree[subnode.parentIndex];
					if (subnode_two.parentIndex == 0) then -- this is a node 2 above parent
						presentVar = presentVar.."."..subnode.name.."."..node.name;
						DebuggerFu:Debug("2nd Child clicked "..node.name.." "..subnode.name.." "..presentVar,1);
					else
						subnode_three =	self.db.profile.varTree[subnode_two.parentIndex];
						if (subnode_three.parentIndex == 0) then -- this is a node 3 above parent
							presentVar = presentVar.."."..subnode_two.name.."."..subnode.name.."."..node.name;
							DebuggerFu:Debug("3rd Child clicked "..node.name.." "..subnode.name.." "..subnode_two.name.." "..presentVar,1);
						else
							subnode_four =	self.db.profile.varTree[subnode_three.parentIndex];
							if (subnode_four.parentIndex == 0) then -- this is a node 4 above parent
								presentVar = presentVar.."."..subnode_three.name.."."..subnode_two.name.."."..subnode.name.."."..node.name;
								DebuggerFu:Debug("4th Child clicked "..node.name.." "..subnode.name.." "..subnode_two.name.." "..subnode_three.name.." "..presentVar,1);							
							else
								selectNewTree = nil;
								DebuggerFu:Debug("Limited to 4 steps deeper jumping - use 2 steps to get there !",0);
							end
						end
					end					
				end
			end
		else --shift click on normal node
			--DEFAULT_CHAT_FRAME:AddMessage("Start Editing");
			selectNewTree = nil;
			self:StartEditing(index);
		end
	elseif (IsControlKeyDown()) then -- collapse/expand all nodes
		selectNewTree = nil;
		if (self.db.profile.varTree[index].expanded) then -- is a table
			local depth = self.db.profile.varTree[index].depth;
			if self.db.profile.varTree[index].expanded == 1 then
				self:ToggleAllNodes(depth,0);
			else
				self:ToggleAllNodes(depth,1);
			end				
		end		
	else	
		selectNewTree = nil;
		if (self.db.profile.varTree[index].expanded) then -- is a table
			if self.db.profile.varTree[index].expanded == 1 then
				self.db.profile.varTree[index].expanded =0;
			else
				self.db.profile.varTree[index].expanded =1;
			end	
		end
	end
	
	if (presentVar) and (selectNewTree) then		
		self:BuildTree(presentVar);	
	else
		self.db.profile.displayIndex = index;
		self:UpdateText();		
	end
end

function DebuggerFu:UseBarDisplay()	-- determins whether the bar is used for display or the tooltip
	local aSize = #self.db.profile.varTree;	
	
	if (self.db.profile.barDataDisplay) and ( (aSize+self.db.profile.varDisplaySize) > self.db.profile.lineThreshhold) then		
		return 1;
	elseif (self:IsTooltipDetached() == true) then		
		return nil;
	end
end

function DebuggerFu:GetFullVarName(index)
	--DEFAULT_CHAT_FRAME:AddMessage("Showing full var name "..self.db.profile.displayedVariable);
	local idx = index;
	local txt = self.db.profile.varTree[idx].name;
	while self.db.profile.varTree[idx].parentIndex > 0 do
		idx = self.db.profile.varTree[idx].parentIndex;
		if (self.db.profile.varTree[idx].parentIndex > 0) then
			txt = self.db.profile.varTree[idx].name.."."..txt;
		end
	end
	txt = self.db.profile.displayedVariable.."."..txt;
	return txt;
end


function DebuggerFu:OnTooltipUpdate()
	
	local cat = tablet:AddCategory(
		'columns', 2,
		'child_textR', 1,
		'child_textG', 1,
		'child_textB', 1
	)
	
	if (not self.db.profile.varTree[1]) then
		return;
	end
	
	if (self.db.profile.varDisplaySize) and (not self:UseBarDisplay()) then
		if (self.db.profile.displayIndex) and (self.db.profile.varTree[self.db.profile.displayIndex]) then					
			local pv = self.db.profile.varTree[self.db.profile.displayIndex];
			
			if (self.db.profile.editMode) and (self.db.profile.editIndex) then -- in edit mode
				cat:AddLine(
					'text', "Editing",
					'textR', 1,
					'textG', 0,
					'textB', 0				
				)						
			end		
						
			if (self.db.profile.showVarName) then
				if (self.db.profile.showFullVarName) then
					cat:AddLine(
						'text', self:GetFullVarName(self.db.profile.displayIndex),
						'textR', 1,
						'textG', 0.496,
						'textB', 0				
					)						
				else				
					cat:AddLine(
						'text', pv.name,
						'textR', 1,
						'textG', 0.496,
						'textB', 0				
					)	
				end				
			end
		
			if type(pv.value) ~= "nil" then
				--DEFAULT_CHAT_FRAME:AddMessage("Adding : "..type(pv.value))			
				if (self.db.profile.showType) then								
					cat:AddLine(
						'text', type(pv.value),
						'textR', 0,
						'textG', 1,
						'textB', 0				
					)		
				end
				
				if (self.db.profile.showValue) then				
					if (type(pv.value) == "string") then
						cat:AddLine(
							'text', "\'"..pv.value.."\'"
						)					
					else					
						cat:AddLine(
							'text', pv.value
						)					
					end
				end				
			else
				if (self.db.profile.showType) then								
					cat:AddLine(
						'text', "table",
						'textR', 0,
						'textG', 1,
						'textB', 0				
					)		
				end

				if (self.db.profile.showValue) then		
					if (self.db.profile.showDepth) then
						cat:AddLine(
							'text', "["..pv.depth.."]"
						)					
					else
						cat:AddLine(
							'text', "-"
						)					
					end					
				end
			end			
		else
			if (self.db.profile.showVarName) then
				cat:AddLine(
					'text', "-",
					'textR', 1,
					'textG', 0.496,
					'textB', 0				
				)					
			end
			if (self.db.profile.showType) then								
				cat:AddLine(
					'text', "-",
					'textR', 0,
					'textG', 1,
					'textB', 0				
				)		
			end
			if (self.db.profile.showValue) then		
				cat:AddLine(
					'text', "-"
				)					
			end
		end		
		cat:AddLine(
				'text', " - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - "
		)						
	end
	
	
	local line = {};	
			
	line['text'] = self.db.profile.varTree[1].name
	line['func'] = 'TreeUp'
	line['arg1'] = self			
	--line['arg2'] = i
	line['hasCheck'] = false							
	
	cat:AddLine(line);
	
	local collapseNodeIndex = 0; 
	
	for i,_ in ipairs(self.db.profile.varTree) do
		if (i > 1) then			
			local line = {};	
			
			if (collapseNodeIndex > 0) and (self.db.profile.varTree[i]) and (self.db.profile.varTree[i].parentIndex > collapseNodeIndex) then
			else
				collapseNodeIndex = 0;
			
				self:Debug("Adding : "..i.." "..self.db.profile.varTree[i].name,5)
				local spacing = "";
				for i=1,self.db.profile.varTree[i].depth-1 do
					spacing = spacing.."   ";
				end
				line['text'] = spacing..self.db.profile.varTree[i].name;
				--line['justify'] = "LEFT"
				line['func'] = 'OnItemClick'
				line['arg1'] = self			
				line['arg2'] = i
				line['hasCheck'] = true							
									
				if (self.db.profile.varTree[i].expanded) then -- is a table
					if (self.db.profile.showDepth) then
						line['text'] = line['text'].." ["..self.db.profile.varTree[i].depth.."]";
					end
					line['checked'] = true				
					if (self.db.profile.varTree[i].expanded == 1) then
						line['checkIcon'] = "Interface\\Buttons\\UI-MinusButton-Up";
					else
						collapseNodeIndex = self.db.profile.varTree[i].parentIndex;
	--					DEFAULT_CHAT_FRAME:AddMessage("Collapsing all > : "..collapseNodeIndex.." "..self.db.profile.varTree[i].name)
						line['checkIcon'] = "Interface\\Buttons\\UI-PlusButton-Up";
					end					
				else -- show the value				
					if (self.db.profile.showValueInTree) then				
						if (type(self.db.profile.varTree[i].value) == "string") then
							line['text'] = line['text'].." = '"..self.db.profile.varTree[i].value.."'";
						elseif (type(self.db.profile.varTree[i].value) == "boolean") then
							if (self.db.profile.varTree[i].value == true) then
								line['text'] = line['text'].." = true";
							else
								line['text'] = line['text'].." = false";
							end
						else
							line['text'] = line['text'].." = "..self.db.profile.varTree[i].value;
						end
					end
				
					if (self.db.profile.showTypeInTree) then
						line['text'] = line['text'].." {"..type(self.db.profile.varTree[i].value).."}";
					end
												
					line['checked'] = false				
					line['textR'] = 1;
					line['textG'] = 1;
					line['textB'] = 0;				
				end
			
				cat:AddLine(line)					
			end
		end
	end	
	
	if (self.db.profile.newTreeBuild == 1) then
		self:ScheduleEvent("DebuggerFu_Update",self.CheckTooltipAttachment, 0.5, self);		
		self.db.profile.newTreeBuild = 0;		
	end	
end

function DebuggerFu:EditBoxEnterPressed()	
	--DEFAULT_CHAT_FRAME:AddMessage("Press");
	if (self.db.profile.editMode) and (self.db.profile.editIndex) then -- box was used to edit something
		--DEFAULT_CHAT_FRAME:AddMessage("Editing");
		local value = self.editBox:GetText();
		local var = nil;
		
		local idx = self.db.profile.editIndex;
		local varParts = {}
		table.insert(varParts,1,self.db.profile.varTree[idx].name);
		--DEFAULT_CHAT_FRAME:AddMessage("Ins : "..self.db.profile.varTree[idx].name);
		while self.db.profile.varTree[idx].parentIndex > 0 do
			idx = self.db.profile.varTree[idx].parentIndex;
			if (self.db.profile.varTree[idx].parentIndex > 0) then
				table.insert(varParts,1,self.db.profile.varTree[idx].name);
				--DEFAULT_CHAT_FRAME:AddMessage("Ins : "..self.db.profile.varTree[idx].name);
			end
		end
		table.insert(varParts,1,self.db.profile.varTree[idx].name);
		--DEFAULT_CHAT_FRAME:AddMessage("Ins : "..self.db.profile.varTree[idx].name);
		
		--DEFAULT_CHAT_FRAME:AddMessage("varParts : "..#varParts);
		
		for i=1,(#varParts)-1 do
			if (i == 1) then
				var = getglobal(varParts[i]);
			else
				var = var[varParts[i]];
			end
			if (not var) or (type(var) ~= "table") then
				self:Debug("Error while reconstruction table variable - Non table or nil value found !",1)
				return;
			end
		end
		local test = var[varParts[#varParts]];
		if (not test) or (type(test) == "table") then -- dont want to edit nil vals or tables
			self:Debug("Error while reconstruction table variable - Table or nil value found at the end !",1)
			return;
		end		
		--DEFAULT_CHAT_FRAME:AddMessage("Found : tp "..type(test).." val "..test);
		
		if ( type(var[varParts[#varParts]]) == "number") then		
			--DEFAULT_CHAT_FRAME:AddMessage("Setting number");
			value = tonumber(value);
		end
		
		var[varParts[#varParts]] = value;		
		
		self.db.profile.varTree[self.db.profile.editIndex].value = value;

		--DEFAULT_CHAT_FRAME:AddMessage("Found : tp "..type(var).." val "..var);
		
		--self.db.profile.varTree[idx].name
		
	
		self:EndEditing();		
		return;
	end
	self.db.profile.newRootVariable = self.editBox:GetText();
	if (IsShiftKeyDown()) or self.db.profile.closeOnEnter then
		self.editFrame:Hide();
	end		
	
	local found=false;
	for _,i in ipairs(self.db.profile.browseHistory) do
		if i == self.db.profile.newRootVariable then
			found = true;
			break;
		end
	end
	if (found == false) then
		table.insert(self.db.profile.browseHistory,1,self.db.profile.newRootVariable);
	end
	self.db.profile.browsePos = 1;
	self:BuildTree(self.db.profile.newRootVariable);		
end

function DebuggerFu:EditBoxTabPressed()	
	if (self.db.profile.editMode) and (self.db.profile.editIndex) then -- box is used to edit something = no tabbing
		return;
	end
	
	local historySize = #self.db.profile.browseHistory;
	--DEFAULT_CHAT_FRAME:AddMessage("TPress"..historySize);
	if (historySize > 1) then
		if (IsShiftKeyDown()) then -- browse backwards
			if (self.db.profile.browsePos < historySize) then
				self.db.profile.browsePos = self.db.profile.browsePos +1;
			else
				self.db.profile.browsePos = 1;
			end
						
			if (self.db.profile.browseHistory[self.db.profile.browsePos]) then
				self.editBox:SetText(self.db.profile.browseHistory[self.db.profile.browsePos]);
			end
			--DEFAULT_CHAT_FRAME:AddMessage("Shift");
		elseif (IsControlKeyDown()) then -- wipe present history pos
			table.remove(self.db.profile.browseHistory,self.db.profile.browsePos);						
			if (self.db.profile.browsePos == historySize) then -- erased the last item
				self.db.profile.browsePos = self.db.profile.browsePos -1;
			end
			if (self.db.profile.browseHistory[self.db.profile.browsePos]) then
				self.editBox:SetText(self.db.profile.browseHistory[self.db.profile.browsePos]);
			end			
		else -- browse forward
			if (self.db.profile.browsePos > 1) then
				self.db.profile.browsePos = self.db.profile.browsePos -1;
			else
				self.db.profile.browsePos = historySize;
			end
			if (self.db.profile.browseHistory[self.db.profile.browsePos]) then
				self.editBox:SetText(self.db.profile.browseHistory[self.db.profile.browsePos]);
			end			
		end
	end
end

function DebuggerFu:CheckEditBoxParameters()	
	if (not self.editFrame) then
		self.editFrame = CreateFrame("Frame", "DebugFu_EditFrame", self.frame);		
		self.editFrame:SetFrameStrata("DIALOG");
		self.editFrame:SetHeight(32)	
			
		self.editFrame:SetMovable(true);	
		self.editFrame:EnableMouse(true);			
		self.editFrame:RegisterForDrag("LeftButton");
		self.editFrame:SetScript("OnDragStart", function() this:StartMoving(); end)
		self.editFrame:SetScript("OnDragStop", function() this:StopMovingOrSizing(); end)
		--self.editFrame:SetScript("OnMouseUp", function() DebuggerFu:EditBoxMousePressed()	end)		
			
		self.editFrame:ClearAllPoints()
		self.editFrame:SetPoint("TOPLEFT", self.frame, "BOTTOMRIGHT", 0, 0)				
	
		self.editFrame:SetBackdrop( { 
			bgFile = "Interface\\DialogFrame\\UI-DialogBox-Background",
 			edgeFile = "Interface\\DialogFrame\\UI-DialogBox-Border", tile = false, tileSize = 8, edgeSize = 8,  	 
  		insets = { left = 1, right = 1, top = 1, bottom = 1}
		});		
	
		self.editBox = CreateFrame("EditBox", "DebugFu_EditBox", self.editFrame,"InputBoxTemplate");
		self.editBox:SetFrameStrata("HIGH");
		self.editBox:SetHeight(20)	
		
		self.editBox:ClearAllPoints()
		self.editBox:SetPoint("TOPLEFT", self.editFrame, "TOPLEFT", 12, -6)							
		self.editBox:SetScript("OnEnterPressed", function() DebuggerFu:EditBoxEnterPressed() end)			
		self.editBox:SetScript("OnTabPressed", function() DebuggerFu:EditBoxTabPressed() end)				
		self.editBox:SetTextColor(1,1,1);
		
		self.editBox:Show();
		self.editFrame:Hide();
	end	
	
	self.editFrame:SetWidth(self.db.profile.editBoxWidth);
	self.editBox:SetWidth(self.db.profile.editBoxWidth-20);	
	self.editFrame:SetAlpha(self.db.profile.editBoxAlpha);		
end

	
function DebuggerFu:OnClick()
	if (IsShiftKeyDown()) then				
		self:CheckEditBoxParameters();
		if (self.editFrame:IsVisible()) then
			self.editFrame:Hide();
		else
			self.editFrame:Show();
		end		
	elseif (IsControlKeyDown()) then
		self.db.profile.varTree = {};
		self:ReattachTooltip();
		self:Update();
	elseif (IsAltKeyDown()) then
		ReloadUI();		
	else		
		self:BuildTree(DebuggerFu.db.profile.rootVariable);	
	end
end

-- Build the variable's tree
function DebuggerFu:BuildTree(variableName)	
	if not variableName then
		DebuggerFu:Debug("No variable set !",1);
		return;
	end
	
	self.db.profile.displayIndex = nil;
	self.db.profile.displayedVariable = variableName; 

	-- Empty the tree
	self.db.profile.varTree = {};
	
	self:Debug("name "..variableName,1);
		
	local part_one,part_two
	local found = true;
	
	local varName = variableName;
	local varParts = {};				
	local aVar;
				
	while (found) do				
		found,_,part_one,part_two = string.find(varName,"(.+)%.(.+)")		
		if (found) then
			table.insert(varParts,1,part_two);
			self:Debug("adding "..part_two,3);
			varName = part_one;
		else
			table.insert(varParts,1,varName);
			self:Debug("adding "..varName,3);
		end
	end
	
	--DEFAULT_CHAT_FRAME:AddMessage("varparts "..#varParts.." - "..varParts[#varParts]);
	
	aVar = getglobal(varParts[1]);					
	if (not aVar) then
		DebuggerFu:Debug(varParts[1].." not found !",1);
		return;
	end
	for i=2,(#varParts)-1 do
		aVar = aVar[varParts[i]];
		if (not aVar) then
			DebuggerFu:Debug(varParts[i].." not found !",1);
			return;
		end		
	end
	
	if #varParts == 1 then
		DebuggerFu.db.profile.rootVariable = variableName;
		self:BuildNode(nil, 0, varParts[1],0);
	else
		if (not aVar[varParts[#varParts]]) then
			DebuggerFu:Debug(varParts[#varParts].." not found !",1);
			return;			
		end
		DebuggerFu.db.profile.rootVariable = variableName;
		self:BuildNode(aVar, 0, varParts[#varParts],0);
	end
	
--	self.db.profile.newTreeBuild = 1;
	--self:CheckTooltipAttachment(1)	
	local amountNodes = #self.db.profile.varTree;		 
	self:Debug("Nodes in tree : "..amountNodes,0);
	self:Update();	
end

local currentIndex 
-- Build a node of the variable's tree
function DebuggerFu:BuildNode(parent, parentIndex, nodeName, depth)

  if nodeName == nil then return nil end
  if depth > self.db.profile.maxDepth then return end
	-- Get the current node
	local node = self:GetNode(nodeName, parent)
	
	if node == nil then 
		--self:Debug("node nil",1);
		return nil 
	end
	
	-- Get the type of the node
	local nodeType = type(node);
	
	self:Debug("name "..nodeName,4);
	self:Debug("type "..nodeType,4);
	
	-- Check the type of the node
	if (nodeType == "number" or nodeType == "string" or nodeType == "boolean") then
		--DEFAULT_CHAT_FRAME:AddMessage("Insert Bool "..nodeName.." "..nodeType);		
		table.insert(self.db.profile.varTree, {name = nodeName, parentIndex = parentIndex,depth = depth, value = node});
	elseif (nodeType == "table") then
		table.insert(self.db.profile.varTree, {name = nodeName, parentIndex = parentIndex,depth = depth, expanded = 1});
		
		local currentIndex = #self.db.profile.varTree;		 
		if (currentIndex >= self.db.profile.maxLineProtection) then
			self:Debug("Exceeding max line protection : "..currentIndex,0);
			return nil;
		end
		--DEFAULT_CHAT_FRAME:AddMessage("Insert "..nodeName.." "..currentIndex);		

		for childNode,_ in pairs(node) do
		  if childNode ~= nil then				
				if (type(childNode) ~= "boolean") then
					--self:Debug("new call"..childNode,5);		
					self:BuildNode(node, currentIndex, childNode, depth+1);
				end
			end
		end
	end
end

function DebuggerFu:GetNode(nodeName, parent)      
	if type(nodeName) ==  "table" or type(nodeName) ==  "function" then return end
	local node
	if (parent) then		
		node = parent[nodeName];
	else
	  node = getglobal(nodeName);		
	end;	
	return node
end

function DebuggerFu:Debug(msg,level)
	if (level <= self.db.profile.debugLevel) then
		if (level <= 0) then
			DEFAULT_CHAT_FRAME:AddMessage("DebugFu : "..msg);
		else
			DEFAULT_CHAT_FRAME:AddMessage(msg);
		end
	end
end