--
-- $Id: AssistFu.lua 55610 2007-11-21 14:36:52Z fritti $
--
-- FuBar_AssistFu -- a FuBar button to set/target your assist
--

local Tablet = AceLibrary("Tablet-2.0")

AssistFu = Rock:NewAddon("AssistFu", "LibRockConfig-1.0", "LibRockDB-1.0", "LibRockEvent-1.0", "LibFuBarPlugin-3.0")

AssistFu:SetDatabase("AssistFuDB")
AssistFu:SetDatabaseDefaults("profile", {
	showmsg = nil,
	msg = ASSISTFU_DEFAULT_MSG,
	channel = "party",
	ma = nil,
	doicon = nil,
	pullicon = 1
})

AssistFu.version = "2.0." .. string.sub("$Revision: 55610 $", 12, -3)
AssistFu.date = string.sub("$Date: 2007-11-21 09:36:52 -0500 (Wed, 21 Nov 2007) $", 8, 17)
AssistFu:SetFuBarOption('hasIcon', true)
AssistFu:SetFuBarOption('defaultPosition', "RIGHT")
AssistFu:SetFuBarOption('tooltipType', "Tablet-2.0")
AssistFu:SetFuBarOption('hideWithoutStandby', true)

local ASSISTFU_PET_MA = "__pet__"
local ASSISTFU_DEFAULT_MSG = "Pulling: %t"


local icons = {
	["star"] = 1,
	["circle"] = 2,
	["diamond"] = 3,
	["triangle"] = 4,
	["moon"] = 5,
	["square"] = 6,
	["cross"] = 7,
	["skull"] = 8
}

function AssistFu:OnInitialize()
	local optionsTable = {
		name = "FuBar_AssistFu",
		desc = self.notes,
		handler = AssistFu,
		type = 'group',
		args = {
			target = {
				type = "execute",
				name = "Set assist to target",
				desc = "Set the main assist to your current target",
				func = "SetAssist",
				passValue = "target",
				order = 10
			},
			player = {
				type = "text",
				name = "Set assist to player",
				desc = "Set the player to assist.",
				usage = "<playername>",
				get = "GetAssist",
				set = "SetAssist",
				order = 20
			},
			pet = {
				type = "execute",
				name = "Set assist to pet",
				desc = "Set the main assist to your pet",
				func = "SetAssist",
				passValue = "pet",
				order = 30
			},
			pull = {
				type = "execute",
				name = "Set pull mode",
				desc = "Reset the main assist, indicating that you are pulling",
				func = "SetAssist",
				passValue = "player",
				order = 40
			},
			clear = {
				type = "execute",
				name = "Clear assist",
				desc = "Reset the saved unit to assist.",
				func = "ClearAssist",
				order = 50
			},
			showpull = {
				type = "toggle",
				name = "Show pull messages",
				desc = "Toggles whether your pull message is shown if you are assisting yourself.",
				get = "GetShowMsg",
				set = "SetShowMsg",
				order = 60
			},
			message = {
				type = "text",
				name = "Pull message",
				desc = "Set your pull message.",
				usage = "<message>",
				get = "GetMsg",
				set = "SetMsg",
				order = 70
			},
			channel = {
				type = "text",
				name = "Pull channel",
				desc = "Set the channel to which we should announce our pulls.",
				usage = "auto|say|party|raid",
				validate = { "auto", "say", "party", "raid" },
				get = "GetChannel",
				set = "SetChannel",
				order = 80
			},
			seticon = {
				type = "boolean",
				name = "Raid icon on pull",
				desc = "Toggles whether your target gets an icon when you are pulling.",
				get = "IsDoIcon",
				set = "ToggleDoIcon",
				order = 90
			},
			pullicon = {
				type = "text",
				name = "Select raid icon",
				desc = "Select the raid icon to set when pulling.",
				get = "GetPullIcon",
				set = "SetPullIcon",
				validate = { "star", "circle", "diamond", "triangle", "moon", "square", "cross", "skull" },
				order = 100
			}
		}
	}
	AssistFu:SetConfigTable(optionsTable)
	AssistFu.OnMenuRequest = optionsTable
end

function AssistFu:OnEnable()
	-- Note: the two following frames cannot be members of the addon table
	-- because the override bindings won't work

	-- Internal secure actionbutton that does the assist
	AssistFu_assistFrame = CreateFrame("Button", nil, nil, "SecureActionButtonTemplate"); 
	AssistFu_assistFrame:SetAttribute("type", "macro")
	AssistFu_assistFrame:SetAttribute("macrotext", "/assist")
	AssistFu_assistFrame:Hide()

	-- Internal secure actionbutton that does the pet assist
	AssistFu_assistPetFrame = CreateFrame("Button", nil, nil, "SecureActionButtonTemplate"); 
	AssistFu_assistPetFrame:SetAttribute("type", "macro")
	AssistFu_assistPetFrame:SetAttribute("macrotext", "/assist pet")
	AssistFu_assistPetFrame:Hide()

	-- Handle the keybindings
	self.keyframe = CreateFrame("Frame", nil, UIParent)
	self:AddEventListener("UPDATE_BINDINGS")
	self:UPDATE_BINDINGS()

	-- Set the initial MA macro text
	self:SetAssist(self.db.profile.ma)

	-- Update FuBar
	self:UpdateFuBarPlugin()

end

function AssistFu:OnDisable()
	-- Transparently fall back to the usual /assist macro
	AssistFu_assistFrame:SetAttribute("macrotext", "/assist")
end

function AssistFu:UPDATE_BINDINGS()
	-- Technique copied from Bartender3, thanks nevcairiel!
	ClearOverrideBindings(self.keyframe)
	local key1, key2 = GetBindingKey("AFUASSIST")
	if key1 then SetOverrideBindingClick(self.keyframe, false, key1, "AssistFu_assistFrame") end
	if key2 then SetOverrideBindingClick(self.keyframe, false, key2, "AssistFu_assistFrame") end
	key1, key2 = GetBindingKey("AFUASSISTPET")
	if key1 then SetOverrideBindingClick(self.keyframe, false, key1, "AssistFu_assistPetFrame") end
	if key2 then SetOverrideBindingClick(self.keyframe, false, key2, "AssistFu_assistPetFrame") end
end

function AssistFu:OnUpdateFuBarText()
	local text
	if self.db.profile.ma == UnitName("player") then
		text = "Pull"
		if self.db.profile.showmsg then
			text = text .. " (" .. self.db.profile.channel .. ")"
		else
			text = text .. " (no msg)"
		end
	elseif self.db.profile.ma == ASSISTFU_PET_MA then
		text = "Your Pet"
	else
		text = self.db.profile.ma
	end
	self:SetFuBarText(text or "No assist")
end

function AssistFu:OnTooltipUpdate()
	local cat = Tablet:AddCategory(
		"columns", 2,
		"child_textR", 1,
		"child_textG", 1,
		"child_textB", 1
	)
	cat:AddLine(
		"text", "Current pull channel:",
		"text2", self.db.profile.channel or "party"
	)
	cat:AddLine(
		"text", "Current pull message:",
		"text2", self.db.profile.msg or ASSISTFU_DEFAULT_MSG
	)
end

function AssistFu:GetAssist()
	local ma = self.db.profile.ma
	if ma == ASSISTFU_PET_MA then return UnitName("pet") or "pet" end
	return ma
end

function AssistFu:SetAssist(name)
	if not self:IsActive() or InCombatLockdown() then
		return
	end

	if not name then
		self:ClearAssist()
	elseif name == "player" then
		self:SetPulling()
	elseif name == "pet" or name == ASSISTFU_PET_MA then
			self:SetAssistPet();
	elseif name ~= "target" and string.len(name) > 0 then
		if string.upper(name) == string.upper(UnitName("player")) then
			self:SetPulling()
		else
			self:SetAssistPlayer(name)
		end
	elseif UnitName("target") == nil then
		self:ClearAssist()
	elseif UnitIsUnit("pet", "target") then
		self:SetAssistPet()
	elseif UnitIsPlayer("target") and UnitIsUnit("target", "player") then
		self:SetPulling()
	elseif UnitIsPlayer("target") and UnitIsFriend("target", "player") then
		self:SetAssistPlayer(UnitName("target"))
	else
		RaidWarningFrame:AddMessage("Cannot set assist to enemies or NPCs.", 1, 0, 0, 1)
		PlaySound("RaidWarning")
	end
	self:UpdateFuBarPlugin()
end

function AssistFu:ClearAssist()
	self.db.profile.ma = nil
	AssistFu_assistFrame:SetAttribute("macrotext", "/assist")
	self:UpdateFuBarPlugin()
end

function AssistFu:SetAssistPet()
	self.db.profile.ma = ASSISTFU_PET_MA
	AssistFu_assistFrame:SetAttribute("macrotext", "/assist pet")
	self:UpdateFuBarPlugin()
end

function AssistFu:SetAssistPlayer(name)
	self.db.profile.ma = name
	AssistFu_assistFrame:SetAttribute("macrotext", "/assist " .. name)
	self:UpdateFuBarPlugin()
end

function AssistFu:SetPulling()
	self.db.profile.ma = UnitName("player")
	AssistFu_assistFrame:SetAttribute("macrotext",
		"/script AssistFu:ShowPull()\n" ..
		"/assist")
	self:UpdateFuBarPlugin()
end

function AssistFu:MsgChannel()
	local chan = self.db.profile.channel

	if not chan or chan == "auto" then
		if UnitInRaid("player") then
			return "RAID"
		elseif UnitInParty("party1") then
			return "PARTY"
		else
			return "SAY"
		end
	end

	return string.upper(chan)
end

function AssistFu:ShowPull()
	local name = UnitName("target")
	if not name then
		return
	end

	if self.db.profile.doicon then
		SetRaidTarget("target", self.db.profile.icon)
	end

	local msg = self.db.profile.msg or ASSISTFU_DEFAULT_MSG
	if not self.db.profile.showmsg or not msg or msg == "" then
		return
	end

	msg = string.gsub(msg, "%%t", name)

	name = UnitName("focus")
	if name then msg = string.gsub(msg, "%%f", name) end

	SendChatMessage(msg, self:MsgChannel())
end

function AssistFu:GetShowMsg()
	return self.db.profile.showmsg
end

function AssistFu:SetShowMsg()
	self.db.profile.showmsg = not self.db.profile.showmsg or nil
	self:UpdateFuBarPlugin()
end

function AssistFu:GetMsg()
	return self.db.profile.msg
end

function AssistFu:SetMsg(msg)
	self.db.profile.msg = msg
	self:UpdateFuBarPlugin()
end

function AssistFu:GetChannel()
	return self.db.profile.channel
end

function AssistFu:SetChannel(chan)
	self.db.profile.channel = chan
	self:UpdateFuBarPlugin()
end

function AssistFu:IsDoIcon()
	return self.db.profile.doicon
end

function AssistFu:ToggleDoIcon()
	self.db.profile.doicon = not self.db.profile.doicon or nil
	self:UpdateFuBarPlugin()
end

function AssistFu:GetPullIcon()
	local i,v
	for i,v in pairs(icons) do
		if v == self.db.profile.pullicon then
			return i
		end
	end
	return "star"
end

function AssistFu:SetPullIcon(icon)
	self.db.profile.pullicon = icons[icon]
end

BINDING_HEADER_ASSISTFU = "AssistFu"
BINDING_NAME_AFUSETASSIST = "Set Assist"
BINDING_NAME_AFUASSIST = "Assist saved"
BINDING_NAME_AFUASSISTPET = "Assist your pet"

-- vim:set ts=4:
