-- Author      : Anthony White
-- Create Date : 1/15/2008 3:19:37 PM
-- FuBar_AlarmClockFu 1.1.0
-- A simple alarm clock module for FuBar users

--[[
TODO: Figure out my logic for setting the alarm hour (why do I need that initial if v > 12 then .. end?)
	  This can probably be refactored
]]

-- Setup libs and Rock
local Tablet = AceLibrary("Tablet-2.0")

AlarmClockFu = Rock:NewAddon("AlarmClockFu", "LibRockTimer-1.0", "LibRockDB-1.0", "LibRockConfig-1.0", "LibFuBarPlugin-3.0")

-- FuBar Plugin Info
AlarmClockFu.version = "1.1.0"
AlarmClockFu:SetFuBarOption("hasIcon", false)
AlarmClockFu:SetFuBarOption("defaultPosition", "RIGHT")
AlarmClockFu:SetFuBarOption("tooltipType", "Tablet-2.0")

-- SavedVariables Table (Set default options)
AlarmClockFu:SetDatabase("AlarmClockFuDB")

AlarmClockFu:SetDatabaseDefaults('profile', {
	twentyFour = false, -- Twenty four / Twelve hour time
	bothTimes = false, -- Display Local Time | Server Time
	localTime = true, -- Display Local Time OR Server Time
	alarm1 = { hour = 0, minute = 0, message = "Alarm 1 Activated!", enabled = false, am = true }, -- Set up the alarms
	alarm2 = { hour = 0, minute = 0, message = "Alarm 2 Activated!", enabled = false, am = true }, -- ...
	alarm3 = { hour = 0, minute = 0, message = "Alarm 3 Activated!", enabled = false, am = true }, -- ...
	countdown = { hours = 0, minutes = 0, hour = 0, minute = 0, enabled = false, message = "Coutdown Triggered!" }, -- Default countdown settings
	snooze = 5, -- amount of minutes to snooze
})
	
local ALARM_ONE = 1
local ALARM_TWO = 2
local ALARM_THREE = 3
local ALARM_COUNTDOWN = 4

local lastAlarmMinute = nil
local activeAlarm = nil

-- OnEnable
function AlarmClockFu:OnEnable()
	-- make sure we update the plugin once a second (for hopefully accurate times)
	self:AddRepeatingTimer(1, "UpdateFuBarPlugin")
end

-- OnDisable
function AlarmClockFu:OnDisable()
	-- empty
	-- having the function defined prevents errors when users try to disable the plugin
end

-- Get / Set Methods

--[[ 12 / 24 Hour Format ]]--

function AlarmClockFu:IsTwentyFour()
	return self.db.profile.twentyFour
end

function AlarmClockFu:ToggleTwentyFour()
	self.db.profile.twentyFour = not self.db.profile.twentyFour
	self:UpdateFuBarPlugin()
	return self.db.profile.twentyFour
end

--[[ Local / Server / Both Time Display ]]--

function AlarmClockFu:IsLocalTime()
	return self.db.profile.localTime
end

function AlarmClockFu:ToggleLocalTime()
	self.db.profile.localTime = not self.db.profile.localTime
	self:UpdateFuBarPlugin()
end

function AlarmClockFu:IsBothTimes()
	return self.db.profile.bothTimes
end

function AlarmClockFu:ToggleBothTimes()
	self.db.profile.bothTimes = not self.db.profile.bothTimes
	self:UpdateFuBarPlugin()
end

-- Add 5 minutes to the given alarm's hour
function AlarmClockFu:Snooze ()
	if activeAlarm == ALARM_ONE then
		AlarmClockFu.db.profile.alarm1.minute = AlarmClockFu.db.profile.alarm1.minute + AlarmClockFu.db.profile.snooze
	elseif activeAlarm == ALARM_TWO then
		AlarmClockFu.db.profile.alarm2.minute = AlarmClockFu.db.profile.alarm2.minute + AlarmClockFu.db.profile.snooze
	else
		AlarmClockFu.db.profile.alarm3.minute = AlarmClockFu.db.profile.alarm3.minute + AlarmClockFu.db.profile.snooze
	end
end

-- END GET / SET METHODS --

function AlarmClockFu:OnInitialize()
	-- Setup config table
	local config = {
		name = "FuBar_AlarmClockFu",
		desc = self.notes,
		type = 'group',
		args = {
			display = {
				type = 'group',
				name = "Display Settings",
				desc = "Choose how the time will be displayed",
				args = {
					disp_local = {
						type = 'toggle',
						name = "Local Time",
						desc = "Toggle between displaying local or server time",
						get = "IsLocalTime",
						set = "ToggleLocalTime"
					},
					disp_both = {
						type = 'toggle',
						name = "Show Both",
						desc = "Display both times",
						get = "IsBothTimes",
						set = "ToggleBothTimes"
					},
					twentyFour = {
						type = 'toggle',
						name = "24 Hour Format",
						desc = "Toggle between 12 and 24 hour format",
						get = "IsTwentyFour",
						set = "ToggleTwentyFour"
					},
					snooze = {
						type = 'text',
						name = "Snooze Time",
						desc = "Set the amount of time to snooze",
						get = function()
							return AlarmClockFu.db.profile.snooze
						end,
						set = function(v)
							AlarmClockFu.db.profile.snooze = tonumber(v)
						end,
					}
				}
			},
			alarm1 = {
				type = "group",
				name = "Alarm 1 Settings",
				desc = "Configure Alarm 1",
				args = {
					hour = {
						order = 1,
						type = "text",
						name = "Hour",
						desc = "Set the alarm hour (1 to 12)",
						usage = "<hour>",
						get = function()
							return AlarmClockFu.db.profile.alarm1.hour
						end,
						set = function(v)
							v = tonumber(v)
							if v > 12 then
								v = v - 12
							end
							
							if AlarmClockFu:IsTwentyFour() then
                                if v > 12 then
                                    AlarmClockFu.db.profile.alarm1.hour = v
                                else
                                	if v == 12 then
										v = 0
									else
										v = v + 12
									end
								end
							else
								if v == 12 then
									v = 0
								else
									v = v + 12
								end
							end
							
							AlarmClockFu.db.profile.alarm1.hour = v
						end,
					},
					minute = {
						order = 2,
						type = "text",
						name = "Minute",
						desc = "Set the alarm minute (0 to 59)",
						usage = "<minute>",
						get = function()
							return AlarmClockFu.db.profile.alarm1.minute
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm1.minute = v
						end,
					},
					enabled = {
						order = 0,
						type = "toggle",
						name = "Enable Alarm",
						desc = "Toggle the alarm on or off",
						get = function()
							return AlarmClockFu.db.profile.alarm1.enabled
						end,
						set = function()
							AlarmClockFu.db.profile.alarm1.enabled = not AlarmClockFu.db.profile.alarm1.enabled
						end
					},
					am = {
						type = "toggle",
						name = "AM",
						desc = "Given time is in the AM",
						get = function()
							return AlarmClockFu.db.profile.alarm1.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm1.am = true
							if AlarmClockFu.db.profile.alarm1.hour > 12 then
								AlarmClockFu.db.profile.alarm1.hour = AlarmClockFu.db.profile.alarm1.hour - 12
							end
						end,
						disabled = AlarmClockFu:IsTwentyFour(),
					},
					pm = {
						type = "toggle",
						name = "PM",
						desc = "Given time is in the PM",
						get = function()
							return not AlarmClockFu.db.profile.alarm1.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm1.am = false
							if AlarmClockFu.db.profile.alarm1.hour < 12 then
								AlarmClockFu.db.profile.alarm1.hour = AlarmClockFu.db.profile.alarm1.hour + 12
							end
						end,
						disabled = AlarmClockFu:IsTwentyFour(),
					},
					message = {
						order = 3,
						type = "text",
						name = "Message",
						desc = "Alarm 1 Message",
						usage = "<message, string>",
						get = function()
							return AlarmClockFu.db.profile.alarm1.message
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm1.message = v
						end
					}
				}
			},
			alarm2 = {
				type = "group",
				name = "Alarm 2 Settings",
				desc = "Configure Alarm 2",
				args = {
					hour = {
						order = 1,
						type = "text",
						name = "Hour",
						desc = "Set the alarm hour (1 to 12)",
						usage = "<hour>",
						get = function()
							return AlarmClockFu.db.profile.alarm2.hour
						end,
						set = function(v)
							v = tonumber(v)
 							if v > 12 then
								v = v - 12
							end
							
							if AlarmClockFu:IsTwentyFour() then
								if v > 12 then
									AlarmClockFu.db.profile.alarm2.hour = v
								else
									if v == 12 then
										v = 0
									else
										v = v + 12
									end
								end
							else
								if v == 12 then
									v = 0
								else
									v = v + 12
								end
							end

							AlarmClockFu.db.profile.alarm2.hour = v
						end,
					},
					minute = {
						order = 2,
						type = "text",
						name = "Minute",
						desc = "Set the alarm minute (0 to 59)",
						usage = "<minute>",
						get = function()
							return AlarmClockFu.db.profile.alarm2.minute
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm2.minute = v
						end,
					},
					enabled = {
						order = 0,
						type = "toggle",
						name = "Enable Alarm",
						desc = "Toggle the alarm on or off",
						get = function()
							return AlarmClockFu.db.profile.alarm2.enabled
						end,
						set = function()
							AlarmClockFu.db.profile.alarm2.enabled = not AlarmClockFu.db.profile.alarm2.enabled
						end
					},
					am = {
						type = "toggle",
						name = "AM",
						desc = "Given time is in the AM",
						get = function()
							return AlarmClockFu.db.profile.alarm2.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm2.am = true
							if AlarmClockFu.db.profile.alarm2.hour > 12 then
								AlarmClockFu.db.profile.alarm2.hour = AlarmClockFu.db.profile.alarm2.hour - 12
							end
						end,
					},
					pm = {
						type = "toggle",
						name = "PM",
						desc = "Given time is in the PM",
						get = function()
							return not AlarmClockFu.db.profile.alarm2.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm2.am = false
							if AlarmClockFu.db.profile.alarm2.hour < 12 then
								AlarmClockFu.db.profile.alarm2.hour = AlarmClockFu.db.profile.alarm2.hour + 12
							end							
						end,
					},
					message = {
						order = 3,
						type = "text",
						name = "Message",
						desc = "Alarm 2 Message",
						usage = "<message, string>",
						get = function()
							return AlarmClockFu.db.profile.alarm2.message
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm2.message = v
						end
					}
				}
			},
			alarm3 = {
				type = "group",
				name = "Alarm 3 Settings",
				desc = "Configure Alarm 3",
				args = {
					hour = {
						order = 1,
						type = "text",
						name = "Hour",
						desc = "Set the alarm hour (1 to 12)",
						usage = "<hour>",
						get = function()
							return AlarmClockFu.db.profile.alarm3.hour
						end,
						set = function(v)
							v = tonumber(v)
							
							if v > 12 then
								v = v - 12
							end							
							
							if AlarmClockFu:IsTwentyFour() then
								if v > 12 then
									AlarmClockFu.db.profile.alarm3.hour = v
								else
									if v == 12 then
										v = 0
									else
										v = v + 12
									end
								end
							else
								if v == 12 then
									v = 0
								else
									v = v + 12
								end
							end

							AlarmClockFu.db.profile.alarm3.hour = v
						end,
					},
					minute = {
						order = 2,
						type = "text",
						name = "Minute",
						desc = "Set the alarm minute (0 to 59)",
						usage = "<minute>",
						get = function()
							return AlarmClockFu.db.profile.alarm3.minute
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm3.minute = v
						end,
					},
					enabled = {
						order = 0,
						type = "toggle",
						name = "Enable Alarm",
						desc = "Toggle the alarm on or off",
						get = function()
							return AlarmClockFu.db.profile.alarm3.enabled
						end,
						set = function()
							AlarmClockFu.db.profile.alarm3.enabled = not AlarmClockFu.db.profile.alarm3.enabled
						end
					},
					am = {
						type = "toggle",
						name = "AM",
						desc = "Given time is in the AM",
						get = function()
							return AlarmClockFu.db.profile.alarm3.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm3.am = true
							if AlarmClockFu.db.profile.alarm3.hour > 12 then
								AlarmClockFu.db.profile.alarm3.hour = AlarmClockFu.db.profile.alarm3.hour - 12
							end							
						end,
					},
					pm = {
						type = "toggle",
						name = "PM",
						desc = "Given time is in the PM",
						get = function()
							return not AlarmClockFu.db.profile.alarm3.am
						end,
						set = function()
							AlarmClockFu.db.profile.alarm3.am = false
							if AlarmClockFu.db.profile.alarm3.hour < 12 then
								AlarmClockFu.db.profile.alarm3.hour = AlarmClockFu.db.profile.alarm3.hour + 12
							end								
						end,
					},
					message = {
						order = 3,
						type = "text",
						name = "Message",
						desc = "Alarm 3 Message",
						usage = "<message, string>",
						get = function()
							return AlarmClockFu.db.profile.alarm3.message
						end,
						set = function(v)
							AlarmClockFu.db.profile.alarm3.message = v
						end
					}
				}
			},
			countdown = {
				type = 'group',
				name = "Countdown Settings",
				desc = "Configure a countdown alarm",
				args = {
					hours = {
						type = 'text',
						name = "Hours",
						desc = "Sets the number of hours to count down",
						usage = "<hours>",
						get = function()
							return AlarmClockFu.db.profile.countdown.hours
						end,
						set = function(v)
							v = tonumber(v)
							AlarmClockFu.db.profile.countdown.hours = v
							AlarmClockFu.db.profile.countdown.hour = mod(tonumber(date("%H")) + v, 12)
							
							if AlarmClockFu.db.profile.countdown.hour == 0 then
								AlarmClockFu.db.profile.countdown.hour = 12
							end
						end
					},
					minutes = {
						type = 'text',
						name = "Minutes",
						desc = "Sets the number of minutes to count down",
						usage = "<minutes>",
						get = function()
							return AlarmClockFu.db.profile.countdown.minutes
						end,
						set = function(v)
							v = tonumber(v)
							AlarmClockFu.db.profile.countdown.minutes = v
							
							-- get the time now
							local hour = tonumber(date("%H"))
							local minute = tonumber(date("%M"))
							
							-- alarm minute = current minute + minute offset
							AlarmClockFu.db.profile.countdown.minute = mod(minute + v, 60)
							
							if (AlarmClockFu.db.profile.countdown.hours == 0) and (v >= 60) then
								AlarmClockFu.db.profile.hours = v / 60
								AlarmClockFu.db.profile.hour = hour + (v / 60)
							elseif (AlarmClockFu.db.profile.countdown.hours == 0) and (v < 60) then
								AlarmClockFu.db.profile.hour = tonumber(date("%H"))
							end
						end
					},
					enabled = {
						type = 'toggle',
						name = "Enabled",
						desc = "Toggle the countdown alarm on or off",
						get = function()
							return AlarmClockFu.db.profile.countdown.enabled
						end,
						set = function()
							AlarmClockFu.db.profile.countdown.enabled = not AlarmClockFu.db.profile.countdown.enabled
							if AlarmClockFu.db.profile.countdown.enabled == true then
								local hour = tonumber(date("%H"))
								local minute = tonumber(date("%M"))
								AlarmClockFu.db.profile.countdown.hour = hour + AlarmClockFu.db.profile.countdown.hours
								AlarmClockFu.db.profile.countdown.minute = minute + AlarmClockFu.db.profile.countdown.minutes
							end
						end
					}
				}
			}
		}
	}
	
	AlarmClockFu:SetConfigTable(config)
	AlarmClockFu.OnMenuRequest = config
	
	-- set up alarm dialog boxes
	StaticPopupDialogs["ALARMCLOCKFU_ALARM1"] = {
		text = AlarmClockFu.db.profile.alarm1.message,
		button1 = "OK",
		button2 = "Snooze",
		OnCancel = AlarmClockFu.Snooze,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1
	}

	StaticPopupDialogs["ALARMCLOCKFU_ALARM2"] = {
		text = AlarmClockFu.db.profile.alarm2.message,
		button1 = "OK",
		button2 = "Snooze",
		OnCancel = AlarmClockFu.Snooze,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1
	}

	StaticPopupDialogs["ALARMCLOCKFU_ALARM3"] = {
		text = AlarmClockFu.db.profile.alarm3.message,
		button1 = "OK",
		button2 = "Snooze",
		OnCancel = AlarmClockFu.Snooze,
		timeout = 0,
		whileDead = 1,
		hideOnEscape = 1
	}
end

-- Format the given hour / minute into a string to be displayed
function AlarmClockFu:FormatTime(hour, minute)
	if self:IsTwentyFour() then
		-- we want to display a twenty four hour time
		return string.format("|cffffffff%02d|r:|cffffffff%02d|r", hour, minute)
	else
		local am = ""
		
		if hour > 12 then
			am = "PM"
		else
			am = "AM"
		end
		
		-- display 12 hour format
		hour = mod(hour, 12)
		if hour == 0 then
			hour = 12
		end
		
		return string.format("|cffffffff%02d|r:|cffffffff%02d|r %s", hour, minute, am)
	end
end

-- Get the current local time
-- Should be accurate to the second
function AlarmClockFu:GetLocalTime()
	local hour = tonumber(date("%H"))
	local minute = tonumber(date("%M"))
	return self:FormatTime(hour, minute)
end

-- Get the current server time
-- This tends to be inaccurate as to the current minute
function AlarmClockFu:GetServerTime()
	local hour, minute = GetGameTime()
	return self:FormatTime(hour, minute)
end

-- Alarm helper method (checks to see if an alarm is ready)
function AlarmClockFu:IsAlarmReady(alarm)
	local minute = tonumber(date("%M"))
	local tf = self.db.profile.twentyFour

	local ret = false

	if tf == false then
		self.db.profile.twentyFour = true
	end

	if alarm == ALARM_ONE then
		ret = ((self:FormatTime(self.db.profile.alarm1.hour, self.db.profile.alarm1.minute) == self:GetLocalTime()) and (lastAlarmMinute ~= minute) and self.db.profile.alarm1.enabled)
	elseif alarm == ALARM_TWO then
		ret = ((self:FormatTime(self.db.profile.alarm2.hour, self.db.profile.alarm2.minute) == self:GetLocalTime()) and (lastAlarmMinute ~= minute) and self.db.profile.alarm2.enabled)
	else
        ret = ((self:FormatTime(self.db.profile.alarm3.hour, self.db.profile.alarm3.minute) == self:GetLocalTime()) and (lastAlarmMinute ~= minute) and self.db.profile.alarm3.enabled)
	end
	
	self.db.profile.twentyFour = tf
	
	return ret
end

-- Countdown helper method
function AlarmClockFu:IsCountdownReady(  )
	local hour = tonumber(date("%H"))
	local minute = tonumber(date("%M"))
	
	return (self:FormatTime(self.db.profile.countdown.hour, self.db.profile.countdown.minute) == self:GetLocalTime()) and (lastAlarmMinute ~= minute) and (self.db.profile.countdown.enabled)
end

-- Called once per second via the timer set in AlarmClockFu:OnEnable
-- Sets the FuBar text to the current time
function AlarmClockFu:OnUpdateFuBarText()
	local str = ""
	local ret
	
	local hour = tonumber(date("%H"))
	local minute = tonumber(date("%M"))
	
	if self:IsAlarmReady(ALARM_ONE) then
		-- alarm 1 needs to be triggered
		StaticPopup_Show("ALARMCLOCKFU_ALARM1")
		PlaySound("ReadyCheck")
		
		lastAlarmMinute = minute
		activeAlarm = ALARM_ONE
	elseif self:IsAlarmReady(ALARM_TWO) then
		StaticPopup_Show("ALARMCLOCKFU_ALARM2")
		PlaySound("ReadyCheck")
		
		lastAlarmMinute = minute
		activeAlarm = ALARM_TWO
	elseif self:IsAlarmReady(ALARM_THREE) then
		StaticPopup_Show("ALARMCLOCKFU_ALARM3")
		PlaySound("ReadyCheck")

		lastAlarmMinute = minute
		activeAlarm = ALARM_THREE
	end
	
	if self:IsCountdownReady() then
		-- countdown is ready, needs to be triggered
		local msg
		
		if self.db.profile.countdown.hours == 0 then
			msg = string.format("It has been %d minutes: %s", self.db.profile.countdown.minutes, self.db.profile.countdown.message)
		else
			msg = string.format("It has been %d hours and %d minutes", self.db.profile.countdown.hours, self.db.profile.countdown.minutes, self.db.profile.countdown.message)
		end
		
		StaticPopupDialogs["ALARMCLOCKFU_COUNTDOWN"] = {
			text = msg,
			button1 = "OK",
			timeout = 0,
			whileDead = 1,
			hideOnEscape = 1,
			sound = "ReadyCheck"
		}
		
		StaticPopup_Show("ALARMCLOCKFU_COUNTDOWN")
		
		lastAlarmMinute = minute
		activeAlarm = ALARM_COUNTDOWN
		-- Disable the countdown (should not be recurring by default
		AlarmClockFu.db.profile.countdown.enabled = false
	end

	if self:IsBothTimes() then
		if self:IsLocalTime() then
			str = string.format("%s || %s", self:GetLocalTime(), self:GetServerTime())
		else
			str = string.format("%s || %s", self:GetServerTime(), self:GetLocalTime())
		end
	else
		if self:IsLocalTime() then
			str = self:GetLocalTime()
		else
			str = self:GetServerTime()
		end
	end
	
	self:SetFuBarText(str)
end

-- FuBar Tooltip Display
function AlarmClockFu:OnUpdateFuBarTooltip()
	-- insert the current date
	Tablet:AddCategory():AddLine(
		'text', date("%A, %B %d, %Y"),
		'textR', 1,
		'textG', 1,
		'textB', 1,
		'justify', "CENTER"
	)

	local cat = Tablet:AddCategory(
		'columns', 2,
		'child_textR', 1,
		'child_textG', 1,
		'child_textB', 0,
		'child_text2R', 1,
		'child_text2G', 1,
		'child_text2B', 1
	)
	
	cat:AddLine(
		'text', "Local Time",
		'text2', self:GetLocalTime()
	)
	
	cat:AddLine(
		'text', "Server Time",
		'text2', self:GetServerTime()
	)
	
	if (self.db.profile.alarm1.enabled) then
		cat:AddLine(
			'text', "Alarm 1",
			'text2', self:FormatTime(self.db.profile.alarm1.hour, self.db.profile.alarm1.minute)
		)
	end
	
	if self.db.profile.alarm2.enabled then
		cat:AddLine(
			'text', "Alarm 2",
			'text2', self:FormatTime(self.db.profile.alarm2.hour, self.db.profile.alarm2.minute)
		)
	end
	
	if self.db.profile.alarm3.enabled then
		cat:AddLine(
			'text', "Alarm 3",
			'text2', self:FormatTime(self.db.profile.alarm3.hour, self.db.profile.alarm3.minute)
		)
	end
	
	if self.db.profile.countdown.enabled then
		cat:AddLine(
			'text', "Countdown",
			'text2', self:FormatTime(self.db.profile.countdown.hour, self.db.profile.countdown.minute)
		)
	end
end