--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
-- Gets a specific setting of a buttontable, if there's more then one button, will take the setting of the first one. accepts either a table or a string
function FlexBar2:GetSetting(ButtonTable, Setting)
	if(type(ButtonTable) == "table") then
		ButtonName = ButtonTable[1]
	else
		ButtonName = ButtonTable;
	end
	if(type(ButtonName) ~= "string" or type(Setting) ~= "string") then return nil; end
	return FlexBar2.db.profile.Buttons[ButtonName][Setting];
end

-- Apply a setting to a buttontable or to a button name
function FlexBar2:ApplySetting(ButtonTable, Setting, Value)
	if(type(ButtonTable) == "table") then
		for _, ButtonName in ipairs(ButtonTable) do
			FlexBar2.db.profile.Buttons[ButtonName][Setting] = Value;
		end
		return true;
	elseif(type(ButtonTable) == "string") then
		FlexBar2.db.profile.Buttons[ButtonTable][Setting] = Value;
		return true;
	else
		return false;
	end
end

-- Sort a button table by name
function FlexBar2.SortButtonTable(Button1Name, Button2Name)
	local Button1Number = tonumber(Button1Name);
	local Button2Number = tonumber(Button2Name);
	if(Button1Number and Button2Number) then
		return Button1Number < Button2Number;
	else
		return Button1Name < Button2Name;
	end
end

-- Get a buttontable out of a range of buttons, mainly used for configuration
function FlexBar2:ToButtonTable(ButtonRange)
	-- If the given range is a string, try to match it
	local ButtonTable;
	if(type(ButtonRange) == "string") then
		ButtonTable = {};
		-- If we get "*" then get a buttontable out of FlexBar2.Buttons
		if(ButtonRange == "*") then
			ButtonTable = {};
			for Name, Object in pairs(FlexBar2.Buttons) do
				table.insert(ButtonTable, Name);
			end
			-- try to match a range (number)-(number)
		elseif(ButtonRange:find("%d+-%d+")) then
			local _, _, First, Last = ButtonRange:find("^(%d+)-(%d+)$");
			for ButtonNumber = First, Last do
				table.insert(ButtonTable, tostring(ButtonNumber));
			end
			-- Try to match a table (name);(name);(name);...
		elseif(ButtonRange:find("[%w_];[%w_]")) then
			ButtonTable = { string.split(";", ButtonRange) };
			-- No match, just assume the given string is a valid button name, for now.
		else
			ButtonTable = { ButtonRange };
		end
		-- Its a table, try to convert
	elseif(type(ButtonRange) == "table") then
		ButtonTable = {};
		for Name, Object in pairs(ButtonRange) do
			table.insert(ButtonTable, Name);
		end
		-- Failed to convert, lets use the original table again in case it was already a button table and it only needs sorting
		if(#ButtonTable == 0) then
			ButtonTable = ButtonRange;
		end
		-- Not a string or not a table, fail
	else
		return false;
	end
	-- Make sure the matched buttons all matched the validity rules (only digits , letters & underscores) before returning
	for Key, ButtonName in ipairs(ButtonTable) do
		if(ButtonName == "mouseover") then
			local FocusFrame = GetMouseFocus();
			if(FocusFrame and FocusFrame.Object and FocusFrame.Object.Name) then
				ButtonTable[Key] = FocusFrame.Object.Name;
			else
				return false;
			end
		end
		if(not ButtonName:find("^[%w_]+$")) then
			return false;
		end
	end
	-- Sort the keys by name so configuration is done in order
	table.sort(ButtonTable, FlexBar2.SortButtonTable);
	return ButtonTable;
end

-- Convert the name of a group to a range of buttons, used in the slashcommand engine to substitute a group name for a list of buttons
function FlexBar2:GroupNameToButtonRange(GroupName)
	-- Make sure the group exists
	if(type(FlexBar2.Groups[GroupName]) == "table" and type(FlexBar2.Groups[GroupName].MemberList) == "table") then
		-- Convert the member list into a valid button table
		local ButtonTable = FlexBar2:ToButtonTable(FlexBar2.Groups[GroupName].MemberList);
		if(type(ButtonTable) == "table") then
			-- Concat the button table into a string if the buttontable is valid
			return table.concat(ButtonTable, ";");
		end
	end
	return false;
end
	
-- Make sure a button exists, handy for slashcommands & gui config
function FlexBar2:ButtonsExist(Button)
	if(type(Button) == "string") then
		Button = FlexBar2:ToButtonTable(Button);
	end
	if(type(Button) == "table") then
           local ButtonList = FlexBar2.Buttons;
		for _, ButtonName in ipairs(Button) do
			if(ButtonList[ButtonName] == nil) then return false; end
		end
		return true;
	else
		return false;
	end
end
	
-- Make sure a button is grouped, handy for slashcommands & gui config
function FlexBar2:ButtonIsGrouped(Button)
	if(type(Button) == "string") then
		Button = FlexBar2:ToButtonTable(Button);
	end
	if(type(Button) == "table") then
		local ButtonList = FlexBar2.Buttons;
		for _, ButtonName in ipairs(Button) do
			if(type(ButtonList[ButtonName].Group) == "table") then return true; end
		end
		return false;
	else
		return false;
	end
end
	
-- Table manipulation functions, they should be pretty clear in what they do
function FlexBar2:ValueIsInTable(Table, Value)
	for n, v in pairs(Table) do
		if(v == Value) then return true; end
	end
	return false;
end

function FlexBar2:RemoveByValue(tbl, val)
	if(type(tbl) ~= "table") then return false; end
	for n, v in pairs(tbl) do
		if(v == val) then
			table.remove(tbl, n);
			return true;
		end
	end
	return false;
end

function FlexBar2:GetKeyByValue(tbl, val)
	for n, v in pairs(tbl) do
		if(v == val) then
			return n;
		end
	end
	return false;
end

-- Debug func
function FlexBar2:Debug(...)
	if(self.Debugging) then
		self:Print(...);
	end
end
