--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Tooltip =  FlexBar2:NewModule("Tooltip");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings.
Tooltip.Version = 1;

Tooltip.ButtonMixin = {};

function Tooltip.ButtonMixin:Deactivate()
	self:DisableTooltip();
end

function Tooltip.ButtonMixin:SetDefaults()
	self.InfoTable.ShowTooltip = true;
end

function Tooltip.ButtonMixin:LoadSettings()
	if(self.InfoTable.ShowTooltip == true) then
		self:EnableTooltip();
	else
		self:DisableTooltip();
	end	
end

function Tooltip.ButtonMixin:EnableTooltip()
	self:RegisterScript("OnEnter","ShowTooltip");
	self:RegisterScript("OnLeave","HideTooltip");
end

function Tooltip.ButtonMixin:DisableTooltip()
	self:UnregisterScript("OnEnter","ShowTooltip");
	self:UnregisterScript("OnLeave","HideTooltip");
end

--- Try to show the tooltip for this button. If we don't maange, hide it.
function Tooltip.ButtonMixin:ShowTooltip()
	GameTooltip_SetDefaultAnchor(GameTooltip, self.Frame);
	local ActionType = self:GetModifiedAttribute("type");
	
	if(ActionType == "spell") then
		self:SetSpellTooltip(self:GetModifiedAttribute("spell"));
	elseif(ActionType == "item") then
		self:SetItemTooltip(self:GetModifiedAttribute("item"));
	elseif(ActionType == "macro") then
		self:SetMacroTooltip(self:GetModifiedAttribute("macro"));
	-- Nothing meaningfull to show, make sure tooltip is hidden
	else
		self:HideTooltip();
	end
end

--- Set the tooltip to the spell with the given name.
-- This gets called by ShowTooltip when applicable.
function Tooltip.ButtonMixin:SetSpellTooltip(SpellName)
	local SpellLink = GetSpellLink(SpellName);
	if(SpellLink) then
		GameTooltip:SetHyperlink(SpellLink);
		GameTooltip:Show();
	end
end

--- Set the tooltip to the spell with the given itemlink.
-- This gets called by ShowTooltip when applicable.
function Tooltip.ButtonMixin:SetItemTooltip(Itemlink)
	Itemlink = Itemlink:gsub("^itemlink:", "");
	if(Itemlink) then
		GameTooltip:SetHyperlink(Itemlink);
		GameTooltip:Show();
	end
end

--- Set the tooltip to the spell with the given macroname
-- This gets called by ShowTooltip when applicable.
function Tooltip.ButtonMixin:SetMacroTooltip(MacroName)
	if(MacroName) then
		local MacroSpell = GetMacroSpell(MacroName);
		if(MacroSpell) then
			self:SetSpellTooltip(MacroSpell);
		else
			local MacroItem = GetMacroItem(MacroName);
			if(MacroItem) then
				self:SetItemTooltip(select(2, GetItemInfo(MacroItem)));
			else
				GameTooltip:SetText(MacroName);
			end
		end
		GameTooltip:Show();
	end
end

function Tooltip.ButtonMixin:HideTooltip()
	GameTooltip:Hide()
end


