--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local TextSubs = FlexBar2:NewModule("TextSubs", "LibRockEvent-1.0");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings.
TextSubs.Version = 3;

local TextSubList = {};
TextSubs.TextSubList = TextSubList;

-- Register a new sub
function TextSubs:RegisterTextSub(TextSub, Replace)
    if((type(Replace) == "function" or type(Replace) == "string") and type(TextSub) == "string") then
		FlexBar2:Debug("Registering new textsub: ", TextSub);
		TextSubList[TextSub] = Replace;
	else
		FlexBar2:Debug("failure to register ", TextSub, " as new textsub");
    end
    TextSubs:DispatchEvent("NewTextSub", TextSub, Replace);
end

function TextSubs:UpdateTextSub(TextSub, ...)
	-- Loop through all buttons and call UpdateTextSub on them with the same arguments as here
	for ButtonName, ButtonRef in pairs(FlexBar2.Buttons) do
		ButtonRef:UpdateTextSub(TextSub, ...);
	end
end

-- Declare the button mixin
TextSubs.ButtonMixin = {};

function TextSubs.ButtonMixin:Load()
	-- Textsub cache of currently set subs
	self.TextSubs = {};
	for Key = 1, 5 do
		self.TextSubs[Key] = {SubCache = {}, PrefixList = {}, SuffixList = {}, ParsedText = ""};
	end
	
	local TextFields = {};
	self.TextFields = TextFields;
	-- Declare the first text field at the TOPLEFT
	local Text1 = self.Frame:CreateFontString(nil, "ARTWORK");
	Text1:SetFontObject("NumberFontNormal");
	Text1:SetJustifyH("LEFT");
	Text1:SetPoint("TOPLEFT", self.Frame, "TOPLEFT");
	self.TextFields[1] = Text1;
    
	-- Declare the second text field at the TOPRIGHT
	local Text2 = self.Frame:CreateFontString(nil, "ARTWORK");
	Text2:SetFontObject("NumberFontNormal");
	Text2:SetJustifyH("RIGHT");
	Text2:SetPoint("TOPRIGHT", self.Frame, "TOPRIGHT");
	self.TextFields[2] = Text2;
    
	-- Declare the third text field at the BOTTOMRIGHT
	local Text3 = self.Frame:CreateFontString(nil, "ARTWORK");
	Text3:SetFontObject("NumberFontNormal");
	Text3:SetJustifyH("RIGHT");
	Text3:SetPoint("BOTTOMRIGHT", self.Frame, "BOTTOMRIGHT");
	self.TextFields[3] = Text3;

	-- Declare the fourth text field at BOTTOMLEFT
	local Text4 = self.Frame:CreateFontString(nil, "ARTWORK");
	Text4:SetFontObject("NumberFontNormal");
	Text4:SetJustifyH("LEFT");
	Text4:SetPoint("BOTTOMLEFT", self.Frame, "BOTTOMLEFT");
	self.TextFields[4] = Text4;
        
	-- Declare the fifth text field at CENTER
	local Text5 = self.Frame:CreateFontString(nil, "ARTWORK");
	Text5:SetFontObject("NumberFontNormal");
	Text5:SetJustifyH("CENTER");
	Text5:SetPoint("CENTER", self.Frame, "CENTER");
	self.TextFields[5] = Text5;
end

local ButtonUpdateTextSub = function(self, Event, NameSpace, Dispatcher, Button) if(self == Button) then self:UpdateTextSub(); end end;
local ButtonUpdateNewSub = function(self, Event, NameSpace, Dispatcher, TextSub) self:UpdateTextSub(TextSub); end;
function TextSubs.ButtonMixin:Activate()
	-- Listen to the ActionChanged event and refresh all subs on that event
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonUpdateTextSub);
	self:AddEventListener("FlexBar2_TextSubs", "NewTextSub", ButtonUpdateNewSub);
	for _, TextField in ipairs(self.TextFields) do
		TextField:Show();
	end
	self:UpdateTextSub();
end

function TextSubs.ButtonMixin:Deactivate()
	for _, TextField in ipairs(self.TextFields) do
		TextField:Hide();
	end
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonUpdateTextSub);
end

function TextSubs.ButtonMixin:SetDefaults()
	local TextSubs = {};
	self.InfoTable.TextSubs = TextSubs;
	-- Text: Text on the sub
	TextSubs[1] = { Text = "", Red = 1, Green = 1, Blue = 1, Alpha = 1};
	TextSubs[2] = { Text = "$(key)", Red = 1, Green = 1, Blue = 1, Alpha = 1};
	TextSubs[3] = { Text = "$(itemcount)", Red = 1, Green = 1, Blue = 1, Alpha = 1};
	TextSubs[4] = { Text = "$(macroname)", Red = 1, Green = 1, Blue = 1, Alpha = 1};
	TextSubs[5] = { Text = "$(cd)", Red = 1, Green = 1, Blue = 1, Alpha = 1};
end

function TextSubs.ButtonMixin:LoadSettings()
	local TextSubs = self.InfoTable.TextSubs;
	for i = 1, 5 do
		local CurrentField = TextSubs[i];
		self:SetTextFieldText(i, CurrentField.Text);
		self:SetTextFieldColor(i, CurrentField.Red, CurrentField.Green, CurrentField.Blue, CurrentField.Alpha);
	end
end

function TextSubs.ButtonMixin:UpdateSettings(OldVersion, NewVersion)
	-- Update old remap format to the new one
	if(OldVersion == 1) then
		local TextSubs = self.InfoTable.TextSubs;
		for i = 1, 5 do
			local Text = TextSubs[i];
			local CurrentField = {};
			TextSubs[i] = CurrentField;
			CurrentField.Text = Text;
			CurrentField.Red = 1;
			CurrentField.Blue = 1;
			CurrentField.Green = 1;
			CurrentField.Alpha = 1;
		end
		OldVersion = 2;
	end
	if(OldVersion == 2) then
		if(self.InfoTable.TextSubs[4].Text == "") then
			self.InfoTable.TextSubs[4].Text = "$(macroname)";
		end
		OldVersion = 3;
	end
	return true;
end

-- Set a textsub field, generate an unformatted string to format it later on
function TextSubs.ButtonMixin:SetTextFieldText(Field, Text)
	if(not FlexBar2:GetModuleState("TextSubs")) then return end
	-- Make sure field is between 1 & 5 and Text is a string
	if(Field > 5 or Field < 1 or type(Text) ~= "string") then return; end
	local SubInfo = self.TextSubs[Field];
	local PrefixList = SubInfo.PrefixList;
	local SuffixList = SubInfo.SuffixList;
	local SubCache = SubInfo.SubCache;
	-- Clear the current Prefix & Suffix cache
	for Key, Value in pairs(PrefixList) do
		PrefixList[Key] = nil;
		SuffixList[Key] = nil;
		SubCache[Key] = nil;
	end
	
	local ParsedText = Text;
	-- Loop through all sub matches of the $(...) format
	for Match in string.gmatch(Text, "%$%((.-)%)") do
		-- Find out the prefix and the suffix
		local Prefix, Suffix = string.match(Match, "(%a*) ?(.*)");
		table.insert(PrefixList, Prefix);
		table.insert(SuffixList, Suffix);
		-- Replace what we just found by %s for string.match later on, also make sure only one replacement is done maximum
		ParsedText = string.gsub(ParsedText, "%$%(" .. Match .."%)", "%%%s", 1);
	end
	FlexBar2:Debug("Parsed Text: [", ParsedText, "]");
	-- Save our parsing results
	SubInfo.ParsedText = ParsedText;
	SubInfo.Text = Text;
	self:UpdateTextSub();
	self:MatchTextField(Field);
end

function TextSubs.ButtonMixin:GetTextFieldText(Field)
	return self.TextSubs[Field].Text;
end

function TextSubs.ButtonMixin:SetTextFieldColor(Field, r, g, b, a)
	self.TextFields[Field]:SetTextColor(r, g, b, a);
end

function TextSubs.ButtonMixin:UpdateTextSub(TextSub, ...)
	-- Only update if the module is active
	if(not FlexBar2:GetModuleState("TextSubs")) then return end
	-- Loop through all fields
	for Field = 1, 5 do
		-- Boolean local wether to rematch or not
		local PlanifyRematch = false;
		local SubInfo = self.TextSubs[Field];
		-- Loop through all prefixes, continue if the prefix matches the current sub
		for Key, Prefix in ipairs(SubInfo.PrefixList) do
			if(Prefix == TextSub or TextSub == nil) then
				-- In case TextSub wasnt passed
				local TextSub = Prefix;
				local OldValue = SubInfo.SubCache[Key];
				local Suffix = SubInfo.SuffixList[Key];
				-- Sub replacement is a function, set its return value
				if(type(TextSubList[TextSub]) == "function") then
					SubInfo.SubCache[Key] = TextSubList[TextSub](self, Suffix, ...) or "";
				-- Sub replacement is a string, just set its value
				elseif(type(TextSubList[TextSub]) == "function") then
					SubInfo.SubCache[Key] = TextSubList[TextSub];
				else
					SubInfo.SubCache[Key] = "$(" .. Prefix .. " " .. Suffix .. ")";
				end
				-- If a rematch is currently not planned and if the Old value is not equal to the new one, planify one
				if(PlanifyRematch == false and OldValue ~= SubInfo.SubCache[Key]) then FlexBar2:Debug("Rematching: ", TextSub); PlanifyRematch = true; end
			end
		end
		if(PlanifyRematch) then self:MatchTextField(Field); end
	end
end

function TextSubs.ButtonMixin:MatchTextField(Field)
	-- Anti-idiot check
	if(Field > 5 or Field < 1) then return; end
	local SubInfo = self.TextSubs[Field];
	-- Format the string with the cached subs
	self.TextFields[Field]:SetText(SubInfo.ParsedText:format(unpack(SubInfo.SubCache)));
end
