--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
-- Slashcommand handler: thanks to Mairelon for the inspiration on the API (though most code was written myself except for ToTable)
local SlashCommands = {};
FlexBar2.SlashCommands = SlashCommands;

-- Two tables keep track of all registered commands & groups
SlashCommands.CommandList = {};
SlashCommands.GroupList = {};
function SlashCommands.Dispatch(Prefix, Command)
	FlexBar2:Debug(Prefix, Command);
	-- Refuse to do anything at all in combat
	if(InCombatLockdown()) then FlexBar2:Print("Configuration is not possible in combat"); return; end
	-- Make sure Command is a string
	if(type(Command) ~= "string") then FlexBar2:Print("Error: Command error. Please report"); return false; end
	-- Find out what function was called and what its arguments are
	local _, _, Function, Arguments = Command:find("^([^ ]+)[ ]*(.*)");
	if(Function) then Function, FunctionValue = string.split("=", Function); end
	-- One day, cakes will rule the world
	if(Function == "sparta") then FlexBar2:Print("/flex THIS IS CAAAAKETOWN /flex"); return; end
	-- If the user asked for help, call the help directly and return
	if(Function == "help" or Command == "") then SlashCommands:Help(Arguments); return; end
	-- Make sure the asked function is registered and valid before continueing
	if(type(SlashCommands.CommandList[Function]) == "table" and type(SlashCommands.CommandList[Function].Function) == "function") then
		-- Convert the param=value format into a table
		local Params = SlashCommands:ToTable(Arguments or "");
		if(FunctionValue) then Params[Function] = FunctionValue; end
		if(SlashCommands:Verify(Function, Params, true) == false) then return false; end
		FlexBar2:Debug(Params);
		-- Call the function
		SlashCommands.CommandList[Function].Function(Params);
	else
		-- If all fails, give the user some help
		SlashCommands:Help();
	end
end
	
-- Register slashcommands with Rock
FlexBar2:AddSlashCommand(SlashCommands.Dispatch, "/fb", "/flexbar", "/flexbar2", "/fb2");
	
-- Help function
function SlashCommands:Help(Group)
	FlexBar2:Debug("Group:", Group)
	-- If a group is passed we tell the user all avail commands in that group
	if(type(Group) == "string" and Group ~= "") then
		FlexBar2:Print("Available commands: ");
		for Command, InfoTable in pairs(SlashCommands.CommandList) do
			if(Group == InfoTable.Group) then
				if(InfoTable.Help) then
				-- Found help for the command, print it out
				FlexBar2:Print("/FlexBar2 " .. Command .. " " .. InfoTable.Help);
				else
					-- No help specified for this command meh, give some default message that won't enlighten the user at all.
					FlexBar2:Print("/FlexBar2 " .. Command .. ": no help available");
				end
			end
		end
	-- No group was passed, print a generic help listing all possible groups
	elseif(Group == "" or type(Group) ~= "string") then
		FlexBar2:Print("Usage: /fb help <group>.");
		FlexBar2:Print("Please Choose a group to get further help");
		for GroupName, _ in pairs(SlashCommands.GroupList) do
			FlexBar2:Print(GroupName);
		end
	end	
end

function SlashCommands:ToTable(msg) -- nearly unmodified code from Mairelon's utility class used in FB1
	if msg == nil then return {} end
	-- Replace \n by a newline
	msg = string.gsub(msg, "\\n", "\n");
	-- pattern to return name=
	local pattern = "([%a_][%w_]*)="
	local index = 1
	local params = {}
	local firsti, lasti, capture = string.find(msg,pattern)
	-- while we have a name=, process the info after it.
	while capture do
		local varname=string.lower(capture)
		index = index+lasti
		firsti, lasti, capture = string.find(string.sub(msg, index),pattern)
		if not firsti then firsti=string.len(msg) else firsti=firsti-2 end
		local str = string.sub(msg,index, index+firsti)
		if(string.find(str, "[ ]+$")) then -- code to strip out spaces at the end
			str = string.sub(str, 1, (string.find(str, "()[ ]+$")) - 1);
		end
		-- str contain the rest of the equality ( varname = str )
		params[varname] = str;
	end
	return params;
end

function SlashCommands:Verify(Function, Params, Output)
	-- Sanity checks to make sure everything is set up currectly
	if(type(SlashCommands.CommandList[Function]) ~= "table"
		or type(SlashCommands.CommandList[Function].Function) ~= "function"
		or type(SlashCommands.CommandList[Function].Params) ~= "table") then
		if(Output) then FlexBar2:Print("Slashcommand not set up correctly, cancelling"); end
		return false;
	end

	-- If a group has been set in the slashcommand we convert it to a set of buttons, this makes button and group interchangable
        if(type(Params["button"]) ~= "string" and type(Params["group"]) == "string") then
            Params.button = FlexBar2:GroupNameToButtonRange(Params["group"]);
        end

	-- Loop through every param that has to be verified
	for Param, Info in pairs(SlashCommands.CommandList[Function].Params) do

		-- Make sure the param is set
		if(type(Params[Param]) ~= "string") then
			-- Param is not set but is required, failure
			if(Info.Required == true) then
				if(Output) then FlexBar2:Print("Required parameter [" .. Param .. "] not set, cancelling"); end
				return false;
			end
		end

		-- Set the default value if the param is not set and a default val has been declared
		if(type(Info.DefaultValue) == "string" and type(Params[Param]) ~= "string") then
			Params[Param] = Info.DefaultValue;
		end
		-- Only do further checking if Param has been set
		if(type(Params[Param]) == "string") then
			-- Make sure the param is in the table if allowed values, if it has been set
			if(type(Info.AllowedValues) == "table") then
				if(not FlexBar2:ValueIsInTable(Info.AllowedValues, Params[Param])) then
					if(Output) then FlexBar2:Print("[" .. Params[Param] .. "] is not in the list of allowed values for " .. Param); end
					return false;
				end
			end
			-- If a verify func has been set, make sure it returns true for the param, if it doesn't, then we outpit its return value
			if(type(Info.VerifyFunc) == "function") then
				local VerifyReturn = Info.VerifyFunc(Params[Param]);
				if(VerifyReturn ~= true) then
					if(Output) then FlexBar2:Print("Verification failed for [" .. Param .. "]: " .. (VerifyReturn or ("], Invalid value [" .. Params[Param] .. "]"))); end
					return false;
				end
			end
		end
	end
	return true;
end

--[[
	Command: The name of the command we are adding
	Function: The command that this will be passed to
	Group: What group the function is in (for the help box)
	Help: function usage which will be displayed in the /FlexBar2 help <command> message
--]]
function SlashCommands:AddCommand(Command, Function, Group, Help)
	if(type(Command) ~= "string" or type(Function) ~= "function" or SlashCommands.CommandList[Command] == "table") then
		return false;
	else
		Command = Command:lower();
	end
	if(type(Group) ~= "string") then Group = "main"; end
	SlashCommands.GroupList[Group] = true;
	SlashCommands.CommandList[Command] = {Function = Function, Group = Group, Help = Help, Params = {}};
end
--[[
	Command: The name of the command we are adding parameters to
	Param: The name of the parameter we are adding
	Required: wether to error out if the parameter is not set
	Types: what type the parameter can be (string or table [ranges are seen as tables])
	AllowedValues: what values are allowed for this parameter (example: {"true", "false"})
	Default: what default value the parameter takes if it is ommited
--]]
function SlashCommands:AddParam(Command, Param, Required, AllowedValues, DefaultValue, VerifyFunc) 
	-- Sanity check
	if(type(Command) ~= "string"
		or type(Param) ~= "string" or type(SlashCommands.CommandList[Command]) ~= "table" or type(SlashCommands.CommandList[Command].Params[Param]) == "table") then
			return false;
		end

	-- Add the command to the list
	SlashCommands.CommandList[Command].Params[Param] =
	{
		Required = Required,
		AllowedValues = AllowedValues,
		DefaultValue = DefaultValue,
		VerifyFunc = VerifyFunc,
	};
end

function SlashCommands:SetAlias(Command, Alias) 
	-- Sanity check
	if(type(Command) ~= "string" or type(Alias) ~= "string" or type(SlashCommands.CommandList[Command]) ~= "table" or type(SlashCommands.CommandList[Alias]) == "table") then return; end
	SlashCommands.CommandList[Alias] = SlashCommands.CommandList[Command];
end
	
-- Slashcommand utility function to make sure buttons exist
function SlashCommands.VerifyExistance(Button)
	if(not FlexBar2:ButtonsExist(Button)) then
		return "One or more buttons do not exist";
	else
		return true;
	end
end
	
-- Slashcommand utility function to make sure buttons do NOT exist
function SlashCommands.VerifyNonExistance(Button)
	if(FlexBar2:ButtonsExist(Button)) then
	    return "One or more buttons already exist";
	else
	    return true;
	end
end

-- Slashcommand utility function to make sure buttons are NOT grouped
function SlashCommands.VerifyNotGrouped(Button)
        local Existance = SlashCommands.VerifyExistance(Button);
        if(Existance ~= true) then
            return Existance;
        else
            if(not FlexBar2:ButtonIsGrouped(Button)) then
                return true;
            else
                return "One or more buttons are already grouped";
            end
        end
end
    
-- Slashcommand utility function to make sure buttons are grouped
function SlashCommands.VerifyGrouped(Button)
	local Existance = SlashCommands.VerifyExistance(Button);
	if(Existance ~= true) then
		return Existance;
	else
		if(FlexBar2:ButtonIsGrouped(Button)) then
			return true;
		else
			return "One or more buttons are not grouped";
		end
	end
end
	
-- Slashcommand utility function to make sure a module exists
function SlashCommands.IsModule(Module)
	return FlexBar2:HasModule(Module);
end
   
-- /fb create handler
local function SlashCommand_Create(Params)
	for _, ButtonName in ipairs(FlexBar2:ToButtonTable(Params.button)) do
		FlexBar2:Create(ButtonName);
	end
end

-- /fb delete handler
local function SlashCommand_Delete(Params)
	for _, ButtonName in ipairs(FlexBar2:ToButtonTable(Params.button)) do
		FlexBar2:Delete(ButtonName);
	end
end

-- /fb list handler
local function SlashCommand_List(Params)
	local ButtonList = FlexBar2.Buttons;
	if(Params.button) then
		FlexBar2:Print("current buttons");
		local ButtonTable = FlexBar2:ToButtonTable(Params.button);
		-- Get a button table out of the button list so everything is sorted
		for _, ButtonName in ipairs(ButtonTable) do
			FlexBar2:Print("[" .. ButtonName .. "]" .. ": " .. (ButtonList[ButtonName].Group and ButtonList[ButtonName].Group.Name or "Ungrouped"));
		end
	else
		FlexBar2:Print("current groups");
		for GroupName, Group in pairs(FlexBar2.Groups) do
			FlexBar2:Print("[" .. GroupName .. "]");
		end
	end
end
	
-- /fb group handler
local function SlashCommand_Group(Params)
	local GroupName = Params.group;
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	if(FlexBar2.Groups[GroupName]) then 
		for _, ButtonName in ipairs(ButtonTable) do
			FlexBar2:AddButtonToGroup(GroupName, ButtonName);
		end
	else
		FlexBar2:CreateGroup(GroupName, ButtonTable);
	end
end

-- /fb ungroup handler
local function SlashCommand_Ungroup(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	for _, ButtonName in ipairs(ButtonTable) do
	FlexBar2:RemoveButtonFromGroup(ButtonName);
	end
end

-- /fb setmodule handler
local function SlashCommand_SetModule(Params)
	if(not Params.button or Params.button == "*") then
		if(Params.enabled == "true") then
			FlexBar2:SetModuleState(Params.module, true);
		else
			FlexBar2:SetModuleState(Params.module, false);
		end
	else
		local ButtonTable = FlexBar2:ToButtonTable(Params.button);
		local ButtonList = FlexBar2.Buttons;
		local Module = FlexBar2:GetModule(Params.module);
		-- Note, no sanity checking done to see if the module was already active or not
		if(Params.enabled == "true") then
			for _, ButtonName in ipairs(ButtonTable) do
				Module.ButtonMixin.Activate(ButtonList[ButtonName]);
			end
		else
			for _, ButtonName in ipairs(ButtonTable) do
				Module.ButtonMixin.Deactivate(ButtonList[ButtonName]);
			end
		end
	end
end

-- /fb debug handler
local function SlashCommand_Debug(Params)
	if(Params.debug == "true") then
		FlexBar2.Debugging = true;
		else
		FlexBar2.Debugging = false;
	end
end

-- /fb setprofile handler
local function SlashCommand_SetProfile(Params)
	FlexBar2:SetProfile(Params.profile);
end

local function SlashCommand_GetProfile()
	local Profile = FlexBar2:GetProfile();
	if(Profile == "char") then FlexBar2:Print("char/" .. UnitName("player") .. " - " .. GetRealmName());
	else FlexBar2:Print(Profile); end
end

-- /fb copyprofile handler
local function SlashCommand_CopyProfile(Params)
	FlexBar2:Disable();
	local Profile = Params.profile;
	if(Profile == "char") then
		Profile = "char/" .. UnitName("player") .. " - " .. GetRealmName();
	end
	FlexBar2:CopyProfile(Profile);
	FlexBar2:Enable();
end

local function SlashCommand_ResetProfile(Params)
	FlexBar2:Disable();
	if(Params.profile == FlexBar2:GetProfile() or not Params.profile) then
		FlexBar2:ResetDatabase(Params.profile);	
	else
		FlexBar2:RemoveProfile(Params.profile);
	end
	FlexBar2:Enable();
end


local function SlashCommand_Blizz(Params)
	if(Params.show == "true") then
		FlexBar2.db.profile.Blizz = true;
		MainMenuBar:Show();
	else
		MainMenuBar:Hide();
		FlexBar2.db.profile.Blizz = false;
	end
end

local function SlashCommand_GetRevision(Params)
	FlexBar2:Print(FlexBar2.Rev);	
end

local function SlashCommand_Enable(Params)
	FlexBar2:Enable();	
end

local function SlashCommand_Disable(Params)
	FlexBar2:Disable();	
end


-- Localize those handy utility funcs
local VerifyExistance = SlashCommands.VerifyExistance;
local VerifyNonExistance = SlashCommands.VerifyNonExistance;
local VerifyGrouped = SlashCommands.VerifyGrouped;
local VerifyNotGrouped = SlashCommands.VerifyNotGrouped;
local IsModule = SlashCommands.IsModule;
-- Register /fb create command
SlashCommands:AddCommand("create", SlashCommand_Create, "main", "button=<Buttons>");
	SlashCommands:AddParam("create", "button", true, nil, nil, VerifyNonExistance);
-- Register /fb delete command
SlashCommands:AddCommand("delete", SlashCommand_Delete, "main", "button=<Buttons>");
SlashCommands:SetAlias("delete", "remove");
SlashCommands:SetAlias("delete", "rm");
SlashCommands:SetAlias("delete", "del");
	SlashCommands:AddParam("delete", "button", true, nil, nil, VerifyExistance);
-- Register /fb list command
SlashCommands:AddCommand("list", SlashCommand_List, "main", "list");
	SlashCommands:AddParam("list", "button", false, nil, nil, VerifyExistance);
-- Register /fb group command
SlashCommands:AddCommand("group", SlashCommand_Group, "main", "button=<Buttons> group=<Group name>");
       SlashCommands:AddParam("group", "button", true, nil, nil, VerifyNotGrouped);
       SlashCommands:AddParam("group", "group", true, nil, nil, function(Name) if(string.match(Name, "^[%w_]+$")) then return true; else return "Invalid group name"; end end);
-- Register /fb ungroup command
SlashCommands:AddCommand("ungroup", SlashCommand_Ungroup, "main", "ungroup group=<Group>");
	SlashCommands:AddParam("ungroup", "button", true, nil, nil, VerifyGrouped);
SlashCommands:AddCommand("setmodule", SlashCommand_SetModule, "main", "module=<ModuleName> enabled=<true/false>");
SlashCommands:SetAlias("setmodule", "module");
	SlashCommands:AddParam("setmodule", "module", true, nil, nil, IsModule);
	SlashCommands:AddParam("setmodule", "enabled", true, {"true", "false"}, nil, nil);
	SlashCommands:AddParam("setmodule", "button", false, nil, nil, VerifyExistance);
SlashCommands:AddCommand("setdebugging", SlashCommand_Debug, "debug", "debug=<true/false>");
SlashCommands:SetAlias("setdebugging", "setdebug");
SlashCommands:SetAlias("setdebugging", "debug");
SlashCommands:SetAlias("setdebugging", "debugging");
	SlashCommands:AddParam("setdebugging", "debug", true, {"true", "false"}, nil, nil);
SlashCommands:AddCommand("setprofile", SlashCommand_SetProfile, "main", "profile=<profile>");
SlashCommands:SetAlias("setprofile", "profile");
	SlashCommands:AddParam("setprofile", "profile", true, nil, nil, nil);
SlashCommands:AddCommand("getprofile", SlashCommand_GetProfile, "main", "");
SlashCommands:AddCommand("resetprofile", SlashCommand_ResetProfile, "main", "");
SlashCommands:AddCommand("reset", SlashCommand_ResetProfile, "main", "");
	SlashCommands:AddParam("resetprofile", "profile", false, nil, nil, nil);
SlashCommands:AddCommand("copyprofile", SlashCommand_CopyProfile, "main", "profile=<profile>");
	SlashCommands:AddParam("copyprofile", "profile", true, nil, nil, nil);
SlashCommands:AddCommand("blizz", SlashCommand_Blizz, "main", "show=<true/false>");
SlashCommands:SetAlias("blizz", "blizz");
	SlashCommands:AddParam("blizz", "show", false, {"true", "false"}, "false", nil);
SlashCommands:AddCommand("getrevision", SlashCommand_GetRevision, "debug", "");
SlashCommands:SetAlias("getrevision", "getrev");
SlashCommands:SetAlias("getrevision", "rev");
SlashCommands:AddCommand("enable", SlashCommand_Enable, "main", "");
SlashCommands:AddCommand("disable", SlashCommand_Disable, "main", "");

