--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Keybindings =  FlexBar2:NewModule("Keybindings");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings.
Keybindings.Version = 1;
local BindingCache = {};

if(FlexBar2:HasModule("TextSubs")) then
	TextSubs = FlexBar2:GetModule("TextSubs");
end

-- Steal Ace2's validity check function for our own use
function Keybindings:ValidateBinding(text)
	if(text == nil or text == "NONE")then
		return nil
	end
	text = text:upper()
	local shift, ctrl, alt
	local modifier
	while true do
		if text == "-" then
			break
		end
		modifier, text = strsplit('-', text, 2)
		if text then
			if modifier ~= "SHIFT" and modifier ~= "CTRL" and modifier ~= "ALT" then
				return false
			end
			if modifier == "SHIFT" then
				if shift then
					return false
				end
				shift = true
			end
			if modifier == "CTRL" then
				if ctrl then
					return false
				end
				ctrl = true
			end
			if modifier == "ALT" then
				if alt then
					return false
				end
				alt = true
			end
		else
			text = modifier
			break
		end
	end
	if not text:find("^F%d+$") and text ~= "CAPSLOCK" and text:len() ~= 1 and (text:byte() < 128 or text:len() > 4) and not _G["KEY_" .. text] then
		return false
	end
	local s = text
	if shift then
		s = "SHIFT-" .. s
	end
	if ctrl then
		s = "CTRL-" .. s
	end
	if alt then
		s = "ALT-" .. s
	end
	return s
end


-- Declare the button mixin
Keybindings.ButtonMixin = {};

-- Load currently set binding
function Keybindings.ButtonMixin:LoadSettings()
	self:SetBinding(self.InfoTable.KeyBinding);
end

-- Clear binding cache for this button when deactived and unbind it
function Keybindings.ButtonMixin:Deactivate()
	local Key = self:GetBinding();
	if(Key) then
		BindingCache[Key] = nil;
		ClearOverrideBindings(self.Frame);
	end
end
-- Set a binding
function Keybindings.ButtonMixin:SetBinding(Key)
	-- If user uses MM then set the key to BUTTON3
	if(Key == "MM") then Key = "BUTTON3"; end
	-- Get the 'old' binding if any
	local SavedBinding = self.InfoTable.KeyBinding;
	FlexBar2:Debug("Setting binding to [" , Key,  "] after verif [", Keybindings:ValidateBinding(Key) ,"] from old binding [", SavedBinding, "]");
	-- Clear any bindings currently on the frame before setting the new one
	ClearOverrideBindings(self.Frame);
	-- Make sure the binding is valid before setting it
	local Key = Keybindings:ValidateBinding(Key);
	if(Key) then
		FlexBar2:Debug("Verif passed");
		-- Set it !
		SetOverrideBindingClick(self.Frame, nil, Key, self.Frame:GetName());
	end
	if(type(SavedBinding) == "string") then -- Clear the old binding from the cache
		BindingCache[SavedBinding] = nil;
	end
	-- Clear the new binding from any old buttons it might have been set to
	if(type(Key) == "string") then
		if(type(BindingCache[Key]) == "table") then
			BindingCache[Key]:SetBinding();
		end
		BindingCache[Key] = self;
	end
	-- Save our newly bound key
	self.InfoTable.KeyBinding = Key;
	-- Update the $(key) text sub
	if(TextSubs) then self:UpdateTextSub("key"); end
end

function Keybindings.ButtonMixin:GetBinding()
	return self.InfoTable.KeyBinding;
end
