--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Highlighting =  FlexBar2:NewModule("Highlighting");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings. Though there are no settings for this module, it has to be declared
Highlighting.Version = 1;
-- Get a local version of SpecialEvents-Aura
local Aura = Rock("LibSpecialEvents-Aura-3.0");

-- Note: Both Inner & Outer Priority will be configurable in the future
-- Define priority for inner checks (mana, range & usable)
local InnerPriority = {"Unusable", "NoMana", "NoRange"};
-- Define priority for outer checks (channeling, casting, autoattacking, buffed, debuffed, equipped)
local OuterPriority = {"Channeling", "Casting","AutoCasting", "ItemBuff", "TargetBuff", "PlayerBuff", "Debuff", "Equipped"};
local RangeGuess;

-- Register all colors with paintchips, scheduled to be configurable in the future
function Highlighting:OnInitialize()
	-- Keep a hash table of RGB values for highlights
	local ColorList = {};
	self.ColorList = ColorList;
	ColorList.PlayerBuff = {0/255, 255/255, 0/255};
	ColorList.TargetBuff = {0/255, 0/255, 200/255};
	ColorList.ItemBuff = {186/255, 0/255, 211/255};
	ColorList.Debuff = {255/255 , 0/255, 0/255};
	ColorList.Equipped = {255/255 , 255/255, 0/255};
	ColorList.NoMana = {0/255 , 0/255, 150/255};
	ColorList.NoRange = {255/255 , 0/255, 0/255};
	ColorList.Unusable = {100/255 , 100/255, 100/255};
	ColorList.Casting = {255/255, 255/255, 255/255};
	ColorList.Channeling = {152/255, 245/255, 255/255};
	ColorList.AutoCasting = {238/255, 118/255, 0/255};
end

-- Declare the button mixin
Highlighting.ButtonMixin = {};

local ButtonCheckUsable = function(Button, Event, NameSpace, Source, UnitId) if(UnitId == "player") then self:CheckUsable(); end end;
local ButtonCheckEquipped = function(Button, Event, NameSpace, Source, UnitId) if(UnitId == "player") then self:CheckEquipped(); end end;
local ButtonCheckCasting = function(Button, Event, NameSpace, Source, UnitId) if(UnitId == "player") then self:CheckCasting(); end end;
local ButtonCheckChanneling = function(Button, Event, NameSpace, Source, UnitId) if(UnitId == "player") then self:CheckChanneling(); end end;
local ButtonCheckPlayerBuff = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckPlayerBuff(); end end;
local ButtonCheckTargetBuff = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckTargetBuff(); end end;
local ButtonCheckItemBuff = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckItemBuff(); end end;
local ButtonCheckDebuff = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckDebuff(); end end;
local ButtonCheckEquipped = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckEquipped(); end end;
local ButtonCheckUsable = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:CheckUsable(); end end;

local InitHighlights = function(self)
	self:CheckEquipped();
	self:CheckPlayerBuff();
	self:CheckDebuff();
	self:CheckUsable();
	self:CheckItemBuff();
end

local CheckTargetBuff = function(self, Event, NameSpace, Dispatcher, Unit) if(Unit == "target") then self:CheckTargetBuff(); end end;
local CheckTargetDebuff = function(self, Event, NameSpace, Dispatcher, Unit) if(Unit == "target") then self:CheckDebuff(); end end;


function Highlighting.ButtonMixin:Activate()
	-- Setup a frame for outer highlights
	local Frame = self.Frame;
	local CheckedTexture = Frame:CreateTexture(nil, "OVERLAY");
	Frame.CheckedTexture = CheckedTexture;
	CheckedTexture:SetAllPoints(Frame);
	CheckedTexture:SetTexture("Interface\\Buttons\\CheckButtonHilight");
	CheckedTexture:SetBlendMode("ADD");
	-- Variables to store what checks are currently active
	self.InnerCheck = {};
	self.OuterCheck = {};
	-- Update the outer highlight when clicked, as clicking might check/uncheck it
	self:RegisterScript("PostClick", "UpdateOuterHighlight");
	-- Update range highlight every 0.3 sec with FB2_UPDATE
	self:AddEventListener("FlexBar2_EventManager", "UPDATE", "CheckRange");
	-- Update usability on mana, rage & energy updates (this includes mana)
	self:AddEventListener("UNIT_MANA", "CheckUsable");
	self:AddEventListener("UNIT_RAGE", "CheckUsable");
	self:AddEventListener("UNIT_ENERGY", "CheckUsable");
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffGained", "CheckUsable");
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffLost", "CheckUsable");
	-- Check usability on shapeshifts
	self:AddEventListener("UPDATE_SHAPESHIFT_FORM", "CheckUsable");
	self:AddEventListener("UPDATE_STEALTH", "CheckUsable");
	-- Check usability on enter/leave combat
	self:AddEventListener("PLAYER_REGEN_ENABLED", "CheckUsable");
	self:AddEventListener("PLAYER_REGEN_DISABLED", "CheckUsable");
	-- Check usability when combo points change
	self:AddEventListener("PLAYER_COMBO_POINTS", "CheckUsable");
	-- Check for usable when using a taxi and when u stop using a taxi
	self:AddEventListener("PLAYER_CONTROL_GAINED", "CheckUsable");
	self:AddEventListener("PLAYER_CONTROL_LOST", "CheckUsable");
	-- Update equiped items when inventory changes
	self:AddEventListener("UNIT_INVENTORY_CHANGED", "CheckEquipped");
	-- Also check for usability changes when the inv changes
	self:AddEventListener("UNIT_INVENTORY_CHANGED", "CheckUsable");
	-- Update casting on spellcasting start & stop
	-- Currently buggy so has to be done OnUpdate :'(
	self:AddEventListener("FlexBar2_EventManager", "UPDATE", "CheckCasting");

	--self:AddEventListener("UNIT_SPELLCAST_START", ButtonCheckCasting);
	--self:AddEventListener("UNIT_SPELLCAST_STOP", ButtonCheckCasting);
	--]]
	-- Update channeling on channeling start & stop
	-- Currently buggy so has to be done OnUpdate :(
	self:AddEventListener("FlexBar2_EventManager", "UPDATE", "CheckChanneling");

	--self:AddEventListener("UNIT_SPELLCAST_CHANNEL_START", ButtonCheckChanneling);
	--self:AddEventListener("UNIT_SPELLCAST_CHANNEL_STOP", ButtonCheckChanneling);
	-- Update autoattacking (or wanding, shooting, whatever) with StartAutoRepeat and EnterCombat events
	self:AddEventListener("START_AUTOREPEAT_SPELL", "CheckStartAutoCasting");
	self:AddEventListener("STOP_AUTOREPEAT_SPELL", "CheckStopAutoCasting");
	self:AddEventListener("PLAYER_ENTER_COMBAT", "CheckStartAutoCasting");
	self:AddEventListener("PLAYER_LEAVE_COMBAT", "CheckStopAutoCasting");
	-- Update buffs when SpecialEvent triggers on player buffs
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffGained", "CheckPlayerBuff");
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffLost", "CheckPlayerBuff");
	-- Update on target buffs too
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "UnitBuffGained", CheckTargetBuff);
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "UnitBuffLost", CheckTargetBuff);
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "AuraTargetChanged", "CheckTargetBuff");

	-- Update usability when the player gains debuffs (hint: forbearance)
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerDebuffGained", "CheckUsable");
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerDebuffLost", "CheckUsable");
	-- Update debuffs when the target gains a debuff
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "UnitDebuffGained", CheckTargetDebuff);
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "UnitDebuffLost", CheckTargetDebuff);
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "AuraTargetChanged", "CheckDebuff");
	-- Check for item buffs
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerItemBuffGained", "CheckItemBuff");
	self:AddEventListener("LibSpecialEvents-Aura-3.0", "PlayerItemBuffLost", "CheckItemBuff");
	-- Update everything if the action changed for this button
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckPlayerBuff);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckTargetBuff);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckDebuff);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckChanneling);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckCasting);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckEquipped);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckUsable);
	self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckItemBuff);

	-- Highlight upon loading too
	if(Rock("LibRockEvent-1.0"):IsFullyInitialized()) then
		InitHighlights(self);
	else
		self:AddEventListener("LibRockEvent-1.0", "FullyInitialized", InitHighlights);
	end
end

function Highlighting.ButtonMixin:Deactivate()
	self:RemoveEventListener("FlexBar2_EventManager", "UPDATE", "CheckRange");
	self:RemoveEventListener("UNIT_INVENTORY_CHANGED", ButtonCheckEquipped);
	self:RemoveEventListener("UNIT_MANA", "CheckUsable");
	self:RemoveEventListener("UNIT_RAGE", "CheckUsable");
	self:RemoveEventListener("UNIT_ENERGY", "CheckUsable");
	self:RemoveEventListener("FlexBar2_EventManager", "UPDATE", "CheckCasting");
	self:RemoveEventListener("FlexBar2_EventManager", "UPDATE", "CheckChanneling");
	self:RemoveEventListener("START_AUTOREPEAT_SPELL", "CheckStartAutoCasting");
	self:RemoveEventListener("STOP_AUTOREPEAT_SPELL", "CheckStopAutoCasting");
	self:RemoveEventListener("PLAYER_ENTER_COMBAT", "CheckStartAutoCasting");
	self:RemoveEventListener("PLAYER_LEAVE_COMBAT", "CheckStopAutoCasting");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffGained", "CheckPlayerBuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerBuffLost", "CheckPlayerBuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerDebuffGained", "CheckUsable");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerDebuffLost", "CheckUsable");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "UnitDebuffGained", "CheckDebuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "UnitDebuffLost", CheckTargetDebuff);
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "AuraTargetChanged", CheckTargetDebuff);
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerItemBuffGained", "CheckItemBuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "PlayerItemBuffLost", "CheckItemBuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "UnitBuffGained", "CheckTargetBuff");
	self:RemoveEventListener("LibSpecialEvents-Aura-3.0", "UnitBuffLost", "CheckTargetBuff");
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckPlayerBuff);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckTargetBuff);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckDebuff);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckChanneling);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckCasting);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckEquipped);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckUsable);
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckItemBuff);
	self:RemoveEventListener("UPDATE_SHAPESHIFT_FORMS", "CheckUsable");
	self:RemoveEventListener("UPDATE_STEALTH", "CheckUsable");
	self:RemoveEventListener("PLAYER_COMBO_POINTS", "CheckUsable");
	self:RemoveEventListener("PLAYER_REGEN_ENABLED", "CheckUsable");
	self:RemoveEventListener("PLAYER_REGEN_DISABLED", "CheckUsable");
	self:RemoveEventListener("UNIT_INVENTORY_CHANGED", "CheckUsable");

	local InnerCheck = self.InnerCheck;
	for Key in ipairs(InnerCheck) do
		InnerCheck[Key] = nil;
	end
	local OuterCheck = self.OuterCheck;
	for Key in ipairs(InnerCheck) do
		OuterCheck[Key] = nil;
	end
	self:UpdateOuterHighlight();
	self:UpdateInnerHighlight();
end

local function InnerSort(a, b)
	return FlexBar2:GetKeyByValue(InnerPriority, a) < FlexBar2:GetKeyByValue(InnerPriority, b);
end

local function OuterSort(a, b)
	return FlexBar2:GetKeyByValue(OuterPriority, a) < FlexBar2:GetKeyByValue(OuterPriority, b);
end

function Highlighting.ButtonMixin:SetInnerHighlight(CheckType, State)
	-- State is true, or an equivalent of true
	if(State) then
		-- Make sure this highlight is not already active
		if(not FlexBar2:ValueIsInTable(self.InnerCheck, CheckType)) then
			-- insert the highlight into the table
			table.insert(self.InnerCheck, CheckType);
			-- sort the table with the priorities in our priority tables declared at the top
			table.sort(self.InnerCheck, InnerSort);
		end
	-- State is false or an equivalent of false
	else
		-- Remove the highlight
		FlexBar2:RemoveByValue(self.InnerCheck, CheckType);
	end
	self:UpdateInnerHighlight();
end

function Highlighting.ButtonMixin:UpdateInnerHighlight()
	-- find out if there's still a highlight active after doing our stuff
	if(type(self.InnerCheck[1]) == "string") then
		-- If there's a highlight active, set it
		self.Frame.Icon:SetVertexColor(unpack(Highlighting.ColorList[self.InnerCheck[1]]));
	else
		-- If there's no highlight active, back to plain boring white
		self.Frame.Icon:SetVertexColor(1, 1, 1);
	end
end


function Highlighting.ButtonMixin:SetOuterHighlight(CheckType, State)
	-- State is true, or an equivalent of true
	if(State) then
		-- Make sure this highlight is not already active
		if(not FlexBar2:ValueIsInTable(self.OuterCheck, CheckType)) then
			-- insert the highlight into the table
			table.insert(self.OuterCheck, CheckType);
			-- sort the table with the priorities in our priority tables declared at the top
			table.sort(self.OuterCheck, OuterSort);
		end
	else
		FlexBar2:RemoveByValue(self.OuterCheck, CheckType)
	end
	-- Update the outer highlight
	self:UpdateOuterHighlight();
end

-- Make sure the outer highlight is set correctly, called on postclick and when a highlight changes
function Highlighting.ButtonMixin:UpdateOuterHighlight()
	if(type(self.OuterCheck[1]) == "string") then
		self.Frame.CheckedTexture:Show();
		self.Frame.CheckedTexture:SetVertexColor(unpack(Highlighting.ColorList[self.OuterCheck[1]]));
	else
		self.Frame.CheckedTexture:Hide();
	end
end


-- Check if the user is currently auto casting something, be it autoattack, autowand or autoshot
function Highlighting.ButtonMixin:CheckStartAutoCasting()
	local Spell = self:GetModifiedAttribute("spell"); 
	if(Spell) then
		self:SetOuterHighlight("AutoCasting", IsAutoRepeatSpell(Spell) or IsAttackSpell(Spell));
	else
		self:SetOuterHighlight("AutoCasting", false);
	end
end

-- Triggered when autocasting stops
function Highlighting.ButtonMixin:CheckStopAutoCasting()
	self:SetOuterHighlight("AutoCasting", false);
end

-- Grab a local copy
local GetSpellInfo = FlexBar2:GetModule("SpellCache").GetSpellInfo;
function Highlighting.ButtonMixin:CheckCasting()
	local CurrentSpellName, CurrentSpellRank = UnitCastingInfo("player");
	local CurrentSpell;
	-- Only continue if we're actualy casting stuff
	if(CurrentSpellName) then
		if(CurrentSpellRank) then
			CurrentSpell = CurrentSpellName .. "(" .. CurrentSpellRank .. ")";
		else
			CurrentSpell = CurrentSpellName;
		end
		-- Ok, the picture here is, sometimes we don't specify a rank, so we have to figure what rank is the highest one, do this will GetSpellName
		local ButtonSpell = self:GetModifiedAttribute("spell");
		if(ButtonSpell) then
			local ButtonSpellName, ButtonSpellRank = GetSpellName(GetSpellInfo(nil, ButtonSpell) or "", BOOKTYPE_SPELL);
			if(ButtonSpellRank) then
				ButtonSpell = ButtonSpellName .. "(" .. ButtonSpellRank .. ")";
			else
				ButtonSpell = ButtonSpellName;
			end
			self:SetOuterHighlight("Casting", ButtonSpell == CurrentSpell);
		else
			self:SetOuterHighlight("Casting", false);
		end
	else
		self:SetOuterHighlight("Casting", false);
	end
end

-- Checks if the user is currently casting something
--function Highlighting.ButtonMixin:CheckCasting()
	-- the IsCurrentSpell function used here comes out in 2.3
	--[[
	local ActionType = self:GetModifiedAttribute("type");
	if(ActionType == "spell") then
		self:SetOuterHighlight("Casting", IsCurrentSpell(self:GetModifiedAttribute("spell")));
	elseif(ActionType == "item") then
		self:SetOuterHighlight("Casting", IsCurrentItem(self:GetModifiedAttribute("itemlink")));
	elseif(ActionType == "macro") then
		-- Not supported yet
		self:SetOuterHighlight("Casting", false);
	else
		self:SetOuterHighlight("Casting", false);
	end-]]
--end

-- Checks for channeled spells
function Highlighting.ButtonMixin:CheckChanneling()
	local CurrentSpellName, CurrentSpellRank = UnitChannelInfo("player");
	local CurrentSpell;
	-- Only continue if we're actualy casting stuff
	if(CurrentSpellName) then
		if(CurrentSpellRank) then
			CurrentSpell = CurrentSpellName .. "(" .. CurrentSpellRank .. ")";
		else
			CurrentSpell = CurrentSpellName;
		end
		-- Ok, the picture here is, sometimes we don't specify a rank, so we have to figure what rank is the highest one, do this will GetSpellName
		local ButtonSpell = self:GetModifiedAttribute("spell")
		if(ButtonSpell) then
			local ButtonSpellName, ButtonSpellRank = GetSpellName(GetSpellInfo(nil, ButtonSpell) or "", BOOKTYPE_SPELL);
			if(ButtonSpellRank) then
				ButtonSpell = ButtonSpellName .. "(" .. ButtonSpellRank .. ")";
			else
				ButtonSpell = ButtonSpellName;
			end
			self:SetOuterHighlight("Channeling", ButtonSpell == CurrentSpell);
		else
			self:SetOuterHighlight("Channeling", false);
		end
	else
		self:SetOuterHighlight("Channeling", false);
	end
end

-- Checks for itembuffs on the player
function Highlighting.ButtonMixin:CheckItemBuff()
	local MainBuff, MainRank = Aura:GetPlayerMainHandItemBuff();
	local OffBuff, OffRank = Aura:GetPlayerOffHandItemBuff();
	if(not MainBuff and not OffBuff) then
		self:SetOuterHighlight("ItemBuff", false);
		return;
	end
	local ButtonType = self:GetModifiedAttribute("type");
	if(not ButtonType) then self:SetOuterHighlight("ItemBuff", false); return; end
	if(ButtonType == "macro") then self:SetOuterHighlight("ItemBuff", false); --[[ TODO: Add code here in 2.3]] return; end
	local ButtonAction = self:GetModifiedAttribute(ButtonType);
	if((MainBuff or OffBuff) and ButtonAction) then
		if(ButtonType == "spell") then
			local ButtonSpell = ButtonAction;
			local ButtonSpellName, ButtonSpellRank = GetSpellName(GetSpellInfo(nil, ButtonSpell) or "", BOOKTYPE_SPELL);
			if(ButtonSpellRank) then
				-- Get the localized version of "Rank"
				local ButtonSpell = ButtonSpellName .. "(" .. ButtonSpellRank .. ")";
				-- Try with the Weapon suffix (shammy self buffs)
				if(MainBuff and ButtonSpell == (MainBuff .. " Weapon" .. (MainRank and "(" .. RANK .. " " .. MainRank .. ")" or ""))) then
					self:SetOuterHighlight("ItemBuff", true);
					FlexBar2:Debug("Found " , MainBuff, " on main hand");
					return;
				end
				if(OffBuff and ButtonSpell == (OffBuff .. " Weapon" .. (OffRank and "(" .. RANK .. " " .. OffRank .. ")" or ""))) then 
					self:SetOuterHighlight("ItemBuff", true);
					FlexBar2:Debug("Found " , MainBuff, " on off hand");

					return;
				end
				-- Try with totem suffix (shammy totem buffs)
				if(MainBuff and ButtonSpell == (MainBuff .. " Totem" .. (MainRank and "(" .. RANK .. " " .. MainRank .. ")" or ""))) then
					self:SetOuterHighlight("ItemBuff", true);
					FlexBar2:Debug("Found " , MainBuff, " on main hand");
					return;	
				end

				if(OffBuff and ButtonSpell == (OffBuff .. " Totem" .. (OffRank and "(" .. RANK .. " " .. OffRank .. ")" or ""))) then 
					self:SetOuterHighlight("ItemBuff", true);
					FlexBar2:Debug("Found " , MainBuff, " on off hand");
					return;
				end
				-- Fail
				self:SetOuterHighlight("ItemBuff", false);
				return;
			else
				self:SetOuterHighlight("ItemBuff", ButtonSpell == MainBuff);
				return;
			end
		elseif(ButtonType == "item") then
			local ItemName = GetItemInfo(ButtonAction);
			self:SetOuterHighlight("ItemBuff", ItemName == MainBuff or ItemName == OffBuff);
		end
	end
	-- Fallback
	self:SetOuterHighlight("ItemBuff", false);
end


-- Checks for buffs on the player
function Highlighting.ButtonMixin:CheckPlayerBuff()
	FlexBar2:Debug("checking for buffs");
	local Icon = self:GetActionTexture();
	if(Icon ~= nil and Aura:UnitHasBuff("player", nil, nil, Icon)) then
		self:SetOuterHighlight("PlayerBuff", true);
	else
		self:SetOuterHighlight("PlayerBuff", false);
	end
end

-- Checks for buffs on the player
function Highlighting.ButtonMixin:CheckTargetBuff()
	FlexBar2:Debug("checking for buffs");
	local Icon = self:GetActionTexture();
	-- Only highlight with targetbuff if the buff is on the target AND if the target is not the player
	if(Icon ~= nil and Aura:UnitHasBuff("target", nil, nil, Icon) and not UnitIsUnit("player", "target")) then
		self:SetOuterHighlight("TargetBuff", true);
	else
		self:SetOuterHighlight("TargetBuff", false);
	end
end



-- Checks for debuffs on the target
function Highlighting.ButtonMixin:CheckDebuff()
	local Icon = self:GetActionTexture();
	if(Icon ~= nil and Aura:UnitHasDebuff("target", nil, nil, Icon)) then
		self:SetOuterHighlight("Debuff", true);
	else
		self:SetOuterHighlight("Debuff", false);
	end
end

-- Check for equiped items
function Highlighting.ButtonMixin:CheckEquipped()
	local Item = self:GetModifiedAttribute("itemlink");
	if(Item and IsEquippedItem(Item)) then
		self:SetOuterHighlight("Equipped", true);
	else
		self:SetOuterHighlight("Equipped", false);
	end
end

-- Check if the action is in range
function Highlighting.ButtonMixin:CheckRange()
	local ButtonType = self:GetModifiedAttribute("type");
	local IsInRange;
	if(ButtonType == "spell") then
		local Spell = self:GetModifiedAttribute("spell") or "";
		IsInRange = IsSpellInRange(Spell, "target");
	elseif(ButtonType == "item") then
		local Item = self:GetModifiedAttribute("itemlink") or "";
		-- bug here the local was shadowing the real variable a few lines up
		--local IsInRange = IsItemInRange(Item, "target"); 
		IsInRange = IsItemInRange(Item, "target"); 
	elseif(ButtonType == "macro") then
		local Macro = self:GetModifiedAttribute("macro");
		if(Macro) then
			local Spell,Rank = GetMacroSpell(Macro);
			if(Spell) then
				if(Rank) then 
					Spell_cat=Spell .. "(" .. Rank .. ")";
				else
					Spell_cat=Spell .. "( )";
				end
				IsInRange = IsSpellInRange(Spell_cat, "target");
			else
				local Item,Itemlink = GetMacroItem(Macro);
				if(Itemlink) then 
					IsInRange = IsItemInRange(Item, "target"); 
				else
					self:SetInnerHighlight("NoRange", false);
					return;
				end
			end
		 end
	else
		self:SetInnerHighlight("NoRange", false);
		return;
	end
	self:SetInnerHighlight("NoRange", IsInRange == 0);
end

-- CHeck if the action is usable 
function Highlighting.ButtonMixin:CheckUsable()
	local ButtonType = self:GetModifiedAttribute("type");
	local Usable, NoMana;
	if(ButtonType == "spell") then
		local Spell = self:GetModifiedAttribute("spell");
		if(Spell) then Usable, NoMana = IsUsableSpell(Spell); end
	elseif(ButtonType == "item") then
		local Item = self:GetModifiedAttribute("itemlink");
		Usable, NoMana = IsUsableItem(Item);
	elseif(ButtonType == "macro") then
		local Macro = self:GetModifiedAttribute("macro");
		if(Macro) then
			local Spell,Rank = GetMacroSpell(Macro);
			if(Spell) then
				if(Rank) then 
					Spell_cat=Spell .. "(" .. Rank .. ")";
				else
					Spell_cat=Spell .. "( )";
				end
				Usable, NoMana = IsUsableSpell(Spell_cat);
			else
				local Item,Itemlink = GetMacroItem(Macro);
				if(Itemlink) then 
					Usable, NoMana = IsUsableItem(Itemlink);
				else
					self:SetInnerHighlight("NoMana", false);
					self:SetInnerHighlight("Unusable", false);
					return;
				end
			end
		 end
	else
		self:SetInnerHighlight("NoMana", false);
		self:SetInnerHighlight("Unusable", false);
		return;
	end
	if(Usable) then
		self:SetInnerHighlight("Unusable", false);
		self:SetInnerHighlight("NoMana", false);
	else
		self:SetInnerHighlight("Unusable", true);
		if(NoMana) then
			self:SetInnerHighlight("NoMana", true);
		else
			self:SetInnerHighlight("NoMana", false);
		end
	end
end
