--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local EventManager =  FlexBar2:NewModule("EventManager", "LibRockEvent-1.0", "LibRockTimer-1.0");
EventManager.Version = 1;

EventManager.EventTable = {};
local EventTable = EventManager.EventTable;

function EventManager:OnInitialize()
    -- Schedule an updater event so that stuff without event can update
    EventManager:AddRepeatingTimer(0.3, function() self:DispatchEvent("UPDATE"); end)
end

EventManager.ButtonMixin = {};

function EventManager.ButtonMixin:AddEventListener(NameSpace, Event, MethodName)
	if(MethodName == nil) then
		MethodName = Event;
		Event = NameSpace;
		NameSpace = "Blizzard";
	end
	FlexBar2:Debug("Registering " , Event, " with " , MethodName);
	-- Sanity check
	if((type(MethodName) == "string" and type(self[MethodName]) == "function") or type(MethodName) == "function") then
		if(not EventManager:HasEventListener(Event)) then
			-- Create the event handling function that AceEvent will trigger, will loop through all buttons for the event and trigger all registered methods for it
			local EventFunc = function(...)
			if(Event ~= "UPDATE") then
				FlexBar2:Debug("Triggering ", Event);
			end
				-- Loop through all buttons
				for Button , EventTable in pairs(EventTable[NameSpace][Event]) do
					-- Loop through all methods for the event found
					for _, MethodName in pairs(EventTable) do
						if(type(MethodName) == "string") then
							Button[MethodName](Button, Event, ...);
						elseif(type(MethodName) == "function") then
							MethodName(Button, Event, ...);
						end
					end
				end
			end
			-- Actualy start listening to the event now
			EventManager:AddEventListener(NameSpace, Event, EventFunc);
		end
		-- Now we're sure the event is registered, lets store the methodname and the button which registered it
		if(type(EventTable[NameSpace]) ~= "table") then
			EventTable[NameSpace] = {};
		end
		if(type(EventTable[NameSpace][Event]) ~= "table") then
			EventTable[NameSpace][Event] = {};
		end
		if(type(EventTable[NameSpace][Event][self]) ~= "table") then
			EventTable[NameSpace][Event][self] = {};
		end
		table.insert(EventTable[NameSpace][Event][self], MethodName);
	else
		-- Failure, lets spit out some debug shit
		FlexBar2:Debug("Failure to register event [" , Event, "] with [", MethodName, "] as method");
		return false;
	end
end

function EventManager.ButtonMixin:RemoveEventListener(NameSpace, Event, MethodName)
	if(MethodName == nil) then
		MethodName = Event;
		Event = NameSpace;
		NameSpace = "Blizzard";
	end
	-- If we had a method name passed, remove said method
	if(NameSpace and MethodName and Event) then
		if(EventTable[NameSpace] and EventTable[NameSpace][Event] and EventTable[NameSpace][Event][self]) then
			return FlexBar2:RemoveByValue(EventTable[NameSpace][Event][self], MethodName);
		end
	elseif(Event) then -- if we dont have a methodname passed but do have an event passed, remove it all
		for n,v in pairs(EventTable[NameSpace][Event][self]) do
			table.remove(EventTable[NameSpace][Event][self], n);
		end
	end
end

function EventManager.ButtonMixin:HasEventListener(NameSpace, Event, MethodName)
	if(MethodName == nil) then
		MethodName = Event;
		Event = NameSpace;
		NameSpace = "Blizzard";
	end
	if(NameSpace and MethodName and Event and EventTable[NameSpace] and EventTable[NameSpace][Event] and EventTable[NameSpace][Event][self]) then
		return FlexBar2:ValueIsInTable(EventTable[NameSpace][Event][self], MethodName);	
	end
end

function EventManager.ButtonMixin:DispatchEvent(Event, ...)
    EventManager:DispatchEvent(Event, ...);
end

-- Scripts are hooked using SetScript, as to not interfer with whatever the state button's scripts are
function EventManager.ButtonMixin:RegisterScript(Trigger, MethodName, ShareWithGroup) 
	-- Make sure ShareWithGroup is not nil if omitted.
	if(ShareWithGroup == nil) then ShareWithGroup = false; end
	if(MethodName == nil) then MethodName = Trigger; end
	-- Sanity checks, if MethodName is a string, make sure the method exists. a function is also valid
	if((type(MethodName) == "string" and type(self[MethodName]) == "function") or type(MethodName) == "function") then
		if(type(self.ScriptTable) ~= "table") then
			self.ScriptTable = {};
		end
		-- Now the script handler is registered for sure, lets add the one we got passed to our list to trigger
		if(type(self.ScriptTable[Trigger]) ~= "table") then
			self.ScriptTable[Trigger] = {};
			local SetScript = type(self.Frame:GetScript(Trigger)) == "func" and self.Frame.HookScript or self.Frame.SetScript;
			SetScript(self.Frame, Trigger, function(...)
				-- Loop through all registered methods
				for MethodName , ShareWithGroup in pairs(self.ScriptTable[Trigger]) do						
					-- Dont trigger debug code of the script is OnUpdate (which is unlikely anyway, as there is an UPDATE event around, but you never know)
					if(Trigger ~= "OnUpdate") then FlexBar2:Debug("Triggering ", MethodName, " For Handler: ", Trigger); end
					-- Wether the script should be shared with its group (this is only used in a few cases but it can be handy)
					if(ShareWithGroup) then
						if(type(MethodName) == "string") then
							-- Loop through all group members and trigger it for them too
							for _, GroupMember in pairs(self.GroupMemberList) do
								GroupMember[MethodName](GroupMember, Trigger, ...);
							end
						elseif(type(MethodName) == "function") then
							-- Same as above, but a function has been directly passed instead of a method name
							for _, GroupMember in pairs(self.GroupMemberList) do
								MethodName(GroupMember, Trigger, ...);
							end
						end
					else
						if(type(MethodName) == "string") then
							-- Trigger script for this button
							self[MethodName](self, Trigger, ...);
						elseif(type(MethodName) == "function") then
							-- Same as above, but a function has been directly passed instead of a method name
							MethodName(self, Trigger, ...);
						end
					end
				end
			end);
		end
		self.ScriptTable[Trigger][MethodName] = ShareWithGroup;
		FlexBar2:Debug("Registering ", Trigger, " ," , MethodName);
		return true;
	else
		-- Failed, spit out debug crap and return false
		FlexBar2:Debug("Failed at registering ", Trigger, " ," , MethodName);
		return false;
	end
end

function EventManager.ButtonMixin:UnregisterScript(Trigger, MethodName)
	-- Nothing to possibly unregister if scripttable is nil, so return;
	if(type(self.ScriptTable) ~= "table") then return; end
	-- MethodName was passed, remove said method
	if(MethodName and Trigger and self.ScriptTable[Trigger]) then
		self.ScriptTable[Trigger][MethodName] = nil;
	elseif(Trigger and self.ScriptTable[Trigger]) then -- Only trigger was passed, remove it all
		for n,v in pairs(self.ScriptTable[Trigger]) do
			table.remove(self.ScriptTable[Trigger], n);
		end
	end
end
