--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
FlexBar2 = Rock:NewAddon("FlexBar2", "LibRockDB-1.0", "LibRockConsole-1.0", "LibRockModuleCore-1.0");
FlexBar2.Rev = "$Rev: 55549 $";
local CurrentVersion = 1;
do -- Section [1]: Loading process 	
	FlexBar2:SetDatabase("FlexBar2_DB");
	FlexBar2:SetDefaultProfile("char");
	FlexBar2:SetDatabaseDefaults('profile', {
		Buttons = {},
		Groups = {},
		Modules = {},
	});

	function FlexBar2:OnEnable()
		-- This is set to true when i'm coding, false when I'm commiting :)
		self.Debugging = false;
		-- Create all the tables we need for intializing
		self.ButtonMixinList = {};
		self.GroupMixinList = {};
		self.Buttons = {};
		self.UnnamedButtons = {};
		self.Groups = {};
		self.ButtonCache = {};
		-- Loop through all modules and find if certain stuff has been declared, if they have, add them to the table.
		-- this includes ButtonMixin:Activate(), ButtonMixin:Deactivate(), ButtonMixin:LoadSettings(), ButtonMixin:SetDefaults(), GroupMixin:Activate(), ButtonWaterfall & Groupwaterfall
		FlexBar2:Debug("Detecting modules");
		for _, Name in ipairs(self.ModuleList) do
			local Module = self:GetModule(Name);
			self:Debug("     Found Module: " .. Name);
			if(type(Module.ButtonMixin) == "table") then
				table.insert(self.ButtonMixinList, Module.ButtonMixin);
				Module.ButtonMixin.MixinName = Name;
			end
			if(type(Module.GroupMixin) == "table") then
				table.insert(self.GroupMixinList, Module.GroupMixin);
				Module.GroupMixin.MixinName = Name;
			end
		end
		
		-- Create the button superclass which will inherit from all registered ButtonMixins.
		self.ButtonSuperClass = self.OO.Class(self.ButtonMixinList);
			function self.ButtonSuperClass:Initialize(Name, ...)
				FlexBar2:Debug("          Initializing button", Name, ...);
				self:SetName(Name);
				self:Load(...);
				self:Enable(...);
			end
			-- Used to set the name of a button, can also be used for renaming a button, it is recommended to disable to button before renaming and reenabling it afterwards.
			-- If you don't you'll get some VERY weird behaviour.
			function self.ButtonSuperClass:SetName(Name)
				-- If a name is declared then we set InfoTable to the one in the database and put the button into our table
				if(type(Name) == "string") then
					self.InfoTable = FlexBar2.db.profile.Buttons[Name];
					self.Name = Name;
					FlexBar2.Buttons[Name] = self;
				else
					self.Name = self.uid;
					self.InfoTable = {};
					FlexBar2.UnnamedButtons[self.Name] = self;
				end
			end
			function self.ButtonSuperClass:EnableModules(...)
				local Name = self.Name;
				local ModulesState = FlexBar2.db.profile.Modules;
				if(Name and type(FlexBar2.db.profile.Buttons[Name].Modules) ~= "table") then FlexBar2.db.profile.Buttons[Name].Modules = {}; end
				local ModuleList = Name and FlexBar2.db.profile.Buttons[Name].Modules or {};
				for _, ModuleName in ipairs(FlexBar2.ModuleList) do
					if(ModulesState[ModuleName] ~= false) then
						local Module = FlexBar2:GetModule(ModuleName);
						local ModuleVersion = Module.Version;
						local ButtonMixin = Module.ButtonMixin;
						-- If the module has a buttonmixin continue
						if(type(ButtonMixin) == "table") then
							-- If the module has never been previously loaded yet OR if the ModuleVersion integrer has been changed, call SetDefaults(), this code allows to reset the settings of one module when needed without touching the others.
							if((type(ModuleList) ~= "table" or ModuleList[ModuleName] ~= ModuleVersion)) then
								-- If there is a function for updating settings and the module was previously installed we attempt to update settings to the new version, on failure (return false) try to reset defaults. Also reset defaults if the module doesn't support updating settings
								if(type(ButtonMixin.UpdateSettings) == "function" and type(ModuleList[ModuleName]) == "number") then
									-- Attempt an upgrade, if it fails we set the defaults back
									if(ButtonMixin.UpdateSettings(self, ModuleList[ModuleName], ModuleVersion, Name, ...) ~= true and type(ButtonMixin.SetDefaults) == "function") then
										ButtonMixin.SetDefaults(self, Name, ...);
									end
								elseif(type(ButtonMixin.SetDefaults) == "function") then
									ButtonMixin.SetDefaults(self, Name, ...);
								end
							end
							if(type(ButtonMixin.LoadSettings) == "function") then ButtonMixin.LoadSettings(self, Name, ...); end
							if(type(ButtonMixin.Activate) == "function") then ButtonMixin.Activate(self, Name, ...); end
						end
						-- Save the data version of the module  in the saved vars
						ModuleList[ModuleName] = ModuleVersion;
						end
				end
			end
			function self.ButtonSuperClass:Enable(...)
				self:EnableModules(...);
			end

			function self.ButtonSuperClass:Disable()
				self:Deactivate();
				self.Group = nil;
				self.GroupMemberList = nil;
			end
		-- Create the group superclass which inherits from all registered GroupMixins
		self.GroupSuperClass = self.OO.Class(self.GroupMixinList);
			function self.GroupSuperClass:Initialize(Name, ButtonTable, ...)
				FlexBar2:Debug("          Initializing group", Name, ButtonTable);
				
				-- Store the group into flexbar's group list
				FlexBar2.Groups[Name] = self;

				-- Store the group name internally
				self.Name = Name;

				-- Store how much members the group has
				self.Count = #ButtonTable;

				-- Declare the infotable internally for easy access later on
				self.InfoTable = FlexBar2.db.profile.Groups[Name];

				-- Create a member list which will contain all members of the group in a 'Name = Object' relation. This is derived from the ButtonTable that has been passed
				local MemberList = {};
				self.MemberList = MemberList
				local ButtonList = FlexBar2.Buttons;
				for _, ButtonName in ipairs(ButtonTable) do
					local Button = ButtonList[ButtonName];
					-- Add the button we just found to the memberlist
					MemberList[ButtonName] = ButtonList[ButtonName];
					-- Make sure button exists, if it doesnt remove the button from the group
					if(Button) then
						-- 'Tell' the button what group it's in by declaring Button.Group
						Button.Group = self;
						-- Add a list of group members to the button in Button.GroupMemberList
						Button.GroupMemberList = MemberList;
					else
						FlexBar2:RemoveButtonFromGroup(ButtonName);
					end
				end
				self:Load(Name, ButtonTable, ...);
				self:Activate(Name, ButtonTable, ...);
			end		
		self:Enable();
	end

	function FlexBar2:Enable()
		if(self.Enabled == true) then return; end
		-- Enable modules
		local ModulesState = self.db.profile.Modules;
		for Key, ModuleName in ipairs(FlexBar2.ModuleList) do
			local Module = FlexBar2:GetModule(ModuleName);
			if(ModulesState[ModuleName] ~= false and type(Module.Activate) == "function") then
				Module:Activate();
			end
		end
		-- Reset settings if current data version is bigger then the one in the saved vars
		if(FlexBar2.db.profile.DataVersion == nil or CurrentVersion > FlexBar2.db.profile.DataVersion) then
			FlexBar2:ResetDatabase("profile");
			FlexBar2.db.profile.DataVersion = CurrentVersion;
		end

		-- Hide blizz bars if told to
		if(self.db.profile.Blizz == false) then MainMenuBar:Hide(); end
	
		-- Loop through all buttons we're suposed to create in the saved vars (and create them ofc -.-)
		for Name, InfoTable in pairs(self.db.profile.Buttons) do
			if(InfoTable.Generate == true) then
				self:Debug("     Generating " .. Name);
				self:Generate(Name);
				self:Debug("Generated");
			end
		end

		-- Loop through all groups we're suposed to create and create em
		for Name, MemberList in pairs(self.db.profile.Groups) do
			FlexBar2:CreateGroup(Name, MemberList);
		end

		
		for Key, ModuleName in ipairs(FlexBar2.ModuleList) do
			local Module = FlexBar2:GetModule(ModuleName);
			if(ModulesState[ModuleName] ~= false and type(Module.PostActivate) == "function") then
				Module:PostActivate();
			end
		end
		self.Enabled = true;
	end

	function FlexBar2:Disable()
		if(self.Enabled ~= true) then return; end
		local ModulesState = self.db.profile.Modules;
		-- Disable modules
		for Key, ModuleName in ipairs(FlexBar2.ModuleList) do
			local Module = FlexBar2:GetModule(ModuleName);
			if(ModulesState[ModuleName] ~= false and type(Module.Deactivate) == "function") then
				Module:Deactivate();
			end
		end
		-- Disable all buttons
		for ButtonName, Button in pairs(FlexBar2.Buttons) do
			Button:Disable();
			table.insert(FlexBar2.ButtonCache, Button);
			FlexBar2.Buttons[ButtonName] = nil;
		end
		-- Disband and groups, but don't save their disbanding
		for GroupName, Group in pairs(FlexBar2.Groups) do
			for ButtonName, Button in pairs(Group.MemberList) do
				Button.Group = nil;
				Button.GroupMemberList = nil;
			end
			Group:Deactivate();
			FlexBar2.Groups[GroupName] = nil;
		end
		self.Enabled = false;
	end

	-- Put all buttons in cache and remove them from their groups and don't save anything of it
	function FlexBar2:OnProfileDisable()
		self:Disable();
	end

	function FlexBar2:OnProfileEnable()
		self:Enable();
	end

	-- Needed to keep track of which order modules were loaded in, as we have to activate their mixins in the order they were loaded
	FlexBar2.ModuleList = {};
	function FlexBar2:OnModuleCreated(Name, Module)
		table.insert(self.ModuleList, Name);
	end
end

do -- Section [2]: Module handlers
	function FlexBar2:SetModuleState(ModuleName, State)
		local Module = FlexBar2:GetModule(ModuleName);
		FlexBar2:Debug(State, FlexBar2:GetModuleState(ModuleName), State == true and not FlexBar2:GetModuleState(ModuleName), State == false and FlexBar2:GetModuleState(ModuleName));
		if(State == true and not FlexBar2:GetModuleState(ModuleName)) then
			self.db.profile.Modules[ModuleName] = State;
			local ButtonMixin = Module.ButtonMixin;
			local GroupMixin = Module.GroupMixin;
			if(ButtonMixin and ButtonMixin.Activate) then
				for ButtonName, ButtonObject in pairs(FlexBar2.Buttons) do
					ButtonMixin.Activate(ButtonObject);
				end
			end
			if(GroupMixin) then
				for GroupName, GroupObject in pairs(FlexBar2.Groups) do
					GroupMixin.Activate(ButtonObject);
				end
			end
		elseif(State == false and FlexBar2:GetModuleState(ModuleName)) then
			self.db.profile.Modules[ModuleName] = State;
			local ButtonMixin = Module.ButtonMixin;
			local GroupMixin = Module.GroupMixin;
			if(ButtonMixin) then
				for ButtonName, ButtonObject in pairs(FlexBar2.Buttons) do
					ButtonMixin.Deactivate(ButtonObject);
				end
			end
			if(GroupMixin) then
				for GroupName, GroupObject in pairs(FlexBar2.Groups) do
					GroupMixin.Deactivate(ButtonObject);
				end
			end
		end
	end

	function FlexBar2:GetModuleState(Module)
		local State = self.db.profile.Modules[Module];
		return State == nil and true or State;
	end
end

do -- Section [3: Button control 
	-- Create a button and make sure its recreated at next login too
	function FlexBar2:Create(Name)
		-- Make sure the button doesn't already exist
		if(Name and type(FlexBar2.db.profile.Buttons[Name]) ~= "table") then
			FlexBar2.db.profile.Buttons[Name] = { Generate = true};
			return FlexBar2:Generate(Name);
		elseif(not Name) then
			return FlexBar2:Generate();
		else
			FlexBar2:Print("Button already exists");
			return false;
		end
	end
	
	-- Delete a button, run whatever has been set in the deactivation mixin, remove the button from its group and make sure it never comes back :)
	function FlexBar2:Delete(Name)
		-- only do something if the button actualy exists
		if(type(FlexBar2.db.profile.Buttons[Name]) == "table") then
			FlexBar2.db.profile.Buttons[Name] = nil;
			local Button = FlexBar2.Buttons[Name];
			-- Remove the button from any groups its in
			FlexBar2:RemoveButtonFromGroup(Name);
			-- Disable the button
			Button:Disable();
			-- nil the button's entry in the listing and add it to the button cache instead
			FlexBar2.Buttons[Name] = nil;
			table.insert(FlexBar2.ButtonCache, Button);
			return true;
		else
			-- Someone is trying to delete a button that doesnt exist, tell him
			FlexBar2:Print("Cannot delete non-existant button");
			return false;
		end
		
	end

	-- Generate a button, load its modules and wipe their settings if needed
	function FlexBar2:Generate(Name)
		local Button;
		-- Make sure to only do something if the button doesnt exist
		if(Name and self.Buttons[Name] ~= nil) then
			self:Print("          Error: Button already exists, aborting");
			return false;
		end
		-- Initialize an instance of the superclass
		if(#self.ButtonCache == 0) then
			FlexBar2:Debug("No cached buttons found, making a new one");
			Button = self.ButtonSuperClass:new(Name);
		else
			FlexBar2:Debug("found a cached button, using it");
			Button = table.remove(self.ButtonCache);
			Button:SetName(Name);
			Button:Enable();
		end
		return Button;
	end
end

do -- Section [4]: Group control
	function FlexBar2:CreateGroup(Name, ButtonTable)
		self.db.profile.Groups[Name] = ButtonTable;
		self.GroupSuperClass:new(Name, ButtonTable);
	end
	
	function FlexBar2:DisbandGroup(Name)
		-- Make sure the grp exists first
		if(type(FlexBar2.Groups[Name]) == "table") then
			local Group = FlexBar2.Groups[Name];
			-- Remove every button from the group, fast, no need to keep the count or anything like RemoveButtonFromGroup does, we're disbanding it anyway
			for ButtonName, Button in pairs(Group.MemberList) do
				Button.Group = nil;
				Button.GroupMemberList = nil;
			end
			Group:Deactivate();
			FlexBar2.Groups[Name] = nil;
			FlexBar2.db.profile.Groups[Name] = nil;
		end
	end

	function FlexBar2:AddButtonToGroup(Group, Name)
		-- Make sure the button is not already grouped and the group exists before adding;
		if(not FlexBar2:ButtonIsGrouped(Name) and type(FlexBar2.Groups[Group]) == "table") then
			-- Localize
			local Group = FlexBar2.Groups[Group];
			local Button = FlexBar2.Buttons[Name];
			-- Add the button to the list of members of the group
			Group.MemberList[Name] = Button;
			-- Increment the member count of this group by one
			Group.Count = Group.Count + 1;
			-- Declare Group and GroupMemberList on the button
			Button.Group = Group;
			Button.GroupMemberList = Group.MemberList;
			-- Save the new group member
			table.insert(Group.InfoTable, Name);
		end
	end

	function FlexBar2:RemoveButtonFromGroup(Name)
		local Button = FlexBar2.Buttons[Name];
		-- Make sure its grouped first
		if(type(Button.GroupMemberList) == "table") then
			-- nil out the button from the member list
			Button.GroupMemberList[Name] = nil;
			local Group = Button.Group;
			-- Decrement the count by one
			Group.Count = Group.Count - 1;
			-- remove the button from the saved button member list
			FlexBar2:RemoveByValue(Button.Group.InfoTable, Name);
			-- nil out Group & GroupMemberList
			Button.Group = nil;
			Button.GroupMemberList = nil;
			-- If the count reached 0, just disband the group, its useless anyway :)
			if(Group.Count == 0) then
				FlexBar2:DisbandGroup(Group.Name);
			end
		end
	end
end
