--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
-- SlashCommands --
local SlashCommands = FlexBar2.SlashCommands;
-- /fb show command
local function SlashCommand_Show(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local Attributes = FlexBar2:GetModule("Attributes");
	local ShowStates;
	if(Params.state) then
		ShowStates = { string.split(",", Params.state) };
		local CurrentState = FlexBar2:GetModule("Attributes").SecureStateDriverFrame:GetAttribute("state");
		for Key, State in ipairs(ShowStates) do
			if(State == "current") then
				ShowStates[Key] = CurrentState;
			end
		end
	end
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:Show(ShowStates);
	end
	Attributes:RefreshState();
end

-- /fb hide command
local function SlashCommand_Hide(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local Attributes = FlexBar2:GetModule("Attributes");
	local HideStates;
	if(Params.state) then
		HideStates = { string.split(",", Params.state) };
		local CurrentState = FlexBar2:GetModule("Attributes").SecureStateDriverFrame:GetAttribute("state");
		for Key, State in ipairs(HideStates) do
			if(State == "current") then
				HideStates[Key] = CurrentState;
			end
		end
	end
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:Hide(HideStates);
	end
	Attributes:RefreshState();
end

-- /fb lock command
local function SlashCommand_Lock(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	FlexBar2:ApplySetting(ButtonTable, "Lock", true);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:Lock();
	end
end

-- /fb unlock command
local function SlashCommand_Unlock(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	FlexBar2:ApplySetting(ButtonTable, "Lock", false);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:Unlock();
	end
end


-- /fb getposition command
local function SlashCommand_GetPosition(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		local Button = ButtonList[ButtonName];
		local X, Y = Button:GetPosition();
		FlexBar2:Print(ButtonName.. ":(".. X, Y .. ")");
	end
end

local function SlashCommand_SetPosition(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	local AnchorButton = Params.anchor or ButtonTable[1];
	local Anchor = ButtonList[AnchorButton];
	local AnchorX, AnchorY = Anchor:GetPosition();
	local X = tonumber(Params.x) or AnchorX;
	local Y = tonumber(Params.y) or AnchorY;

	-- Only move if position was specified in "stupid" mode
	if(Params.smart == "false") then
		if(Params.x ~= nil) then FlexBar2:ApplySetting(ButtonTable, "X", Params.x); end
		if(Params.y ~= nil) then FlexBar2:ApplySetting(ButtonTable, "Y", Params.y); end
	elseif(Params.smart == "true") then
		local ButtonPositions = {};
		for Key, ButtonName in ipairs(ButtonTable) do
			local Button = ButtonList[ButtonName];
			-- Only do something if the button we're computing is not the anchor
			if(Button ~= Anchor) then
				-- Get the button's position relative to the anchor
				ButtonX, ButtonY = Button:GetPosition();
				local Scale = Button:GetScale();
				local RelativeX = ButtonX - AnchorX;
				local RelativeY = ButtonY - AnchorY;
				local NewX = X + RelativeX;
				local NewY = Y + RelativeY;
				Button.InfoTable.X = NewX;
				Button.InfoTable.Y = NewY;
				Button:SetPosition(NewX, NewY);
			else
				Button.InfoTable.X = X;
				Button.InfoTable.Y = Y;
				Button:SetPosition(X, Y);
			end
		end

	end
end


--  /fb setscale command
local function SlashCommand_SetScale(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local Scale = tonumber(Params.scale);
	FlexBar2:ApplySetting(ButtonTable, "Scale", Scale);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:SetScale(Scale);
	end
end

--  /fb setalpha command
local function SlashCommand_SetAlpha(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local Alpha = tonumber(Params.alpha);
	FlexBar2:ApplySetting(ButtonTable, "Alpha", Alpha);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:SetAlpha(Alpha);
	end
end


-- /fb tocursor command
local function SlashCommand_ToCursor(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		local Button = ButtonList[ButtonName]
		X, Y = Button:FrameToCursor();
		FlexBar2:ApplySetting(ButtonName, "X", X);
		FlexBar2:ApplySetting(ButtonName, "Y", Y);
	end
end

-- /fb setbuttontype command
local function SlashCommand_SetButtonType(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		ButtonList[ButtonName]:SetType(Params.type);
	end
end

local function SlashCommand_Line(Params)
	local Rows = tonumber(Params.rows);
	local Columns = tonumber(Params.columns);
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	-- Fail if we have a number of columns & rows specified but the number of buttons is bigger then the amount of buttons we can line up in those
	if(Columns and Rows and Rows * Columns < #ButtonTable) then FlexBar2:Print("can not match asked amount of buttons with the allowed rows & columns"); return; end
	-- Compute the number of columns needed to have X rows
	if(Rows and not Columns) then Columns = math.ceil(#ButtonTable / Rows); end
	-- Compute the number of rows needed to have X columns
	if(Columns and not Rows) then Rows = math.ceil(#ButtonTable / Columns); end
	local StartX = ButtonList[ButtonTable[1]].InfoTable.X;
	local StartY = ButtonList[ButtonTable[1]].InfoTable.Y;
	local Space = tonumber(Params.space);
	if((Columns or Rows)) then
		local LastScale = ButtonList[ButtonTable[1]]:GetScale();
		local Column = 1;
		local Row = 1;
		local LastX = StartX;
		local LastY = StartY;
		for Key, ButtonName in ipairs(ButtonTable) do
			if(Key ~= 1) then
				-- Do nothing for the first button, we position the others relative to it
				local Button = ButtonList[ButtonName];
				-- Go down a row if the current one is full
				if(Column == Columns) then
					Column = 1;
					--Row = Row + 1;
					LastY = LastY - Space * LastScale;
					LastX = StartX;
				elseif(Row == Rows) then
					Row = 1;
					LastX = LastX + Space * LastScale;
					LastY = StartY;
				else
					LastX = LastX + Space * LastScale;
					Column = Column + 1;
				end
				LastScale = Button:GetScale();
				Button:SetPosition(LastX, LastY);
				Button.InfoTable.X = LastX;
				Button.InfoTable.Y = LastY;
			end
		end
	-- Only columns specified
	--[[elseif(Rows) then
		local LastScale = ButtonList[ButtonTable[1]]--[[:GetScale();
		local Row = 1;
		local LastX = StartX;
		local LastY = StartY;
		for Key, ButtonName in ipairs(ButtonTable) do
			-- Do nothing for the first button, we position the others relative to it
			local Button = ButtonList[ButtonName];	
			if(Key ~= 1) then
				-- Go down a row if the current one is full
				if(Row == Rows) then
					Row = 1;
					LastX = LastX + Space * LastScale;
					LastY = StartY;
				else
					LastY = LastY - Space * LastScale;
					Row = Row + 1;
				end
				Button:SetPosition(LastX, LastY);
				Button.InfoTable.X = LastX;
				Button.InfoTable.Y = LastY;
			end
			LastScale = Button:GetScale();
		end]]
	else
		FlexBar2:Print("Need either number of rows or number of columns declared");
	end
end

local function SlashCommand_Ring(Params) 
	local Radius = tonumber(Params.radius); 
	local ButtonTable = FlexBar2:ToButtonTable(Params.button); 
	local ButtonList = FlexBar2.Buttons; 
	local DeltaAngle = (360 / #ButtonTable); 
	local CurrAngle = 0; 
	local CenterX = tonumber(Params.centerx); 
	local CenterY = tonumber(Params.centery); 
	-- There was some sanity checking here. I should write some for the new function. But there isn't. Beware, buffer overflows off the starboard bow! 
	local StartX = CenterX + Radius; 
	local StartY = CenterY; 
	if((Radius)) then

		local LastX = StartX; 
		local LastY = StartY; 
		for Key, ButtonName in ipairs(ButtonTable) do

			local Button = ButtonList[ButtonName]; 
			LastX = LastX + Radius*math.cos(math.rad(CurrAngle)); 
			LastY = LastY + Radius*math.sin(math.rad(CurrAngle)); 
			CurrAngle = CurrAngle + DeltaAngle; 
			Button:SetPosition(LastX, LastY); 
			Button.InfoTable.X = LastX; 
			Button.InfoTable.Y = LastY; 

		end 

	else

		FlexBar2:Print("Need radius declared"); 

	end 
end


local function SlashCommand_SetTexture(Params)
	local ButtonTable = FlexBar2:ToButtonTable(Params.button);
	local ButtonList = FlexBar2.Buttons;
	for _, ButtonName in ipairs(ButtonTable) do
		FlexBar2:Print(Params.texture);
		local Button = ButtonList[ButtonName];
		Button.InfoTable.Texture = Params.texture;
		Button:SetIcon(Params.texture);
	end
end

-- Localize this neat verify function for faster access
local VerifyExistance = FlexBar2.SlashCommands.VerifyExistance;
-- AddCommand format :AddCommand(Command, Function, Group, Help)
-- AddParam format :AddParam(Command, Param, Required, AllowedValues, DefaultValue, VerifyFunc) 
-- everything here should be self explainative knowing the syntax
SlashCommands:AddCommand("show", SlashCommand_Show, "layout", "button=<Button>");
	SlashCommands:AddParam("show", "button", false, nil, "*", VerifyExistance)
SlashCommands:AddCommand("hide", SlashCommand_Hide, "layout", "button=<Button>");
	SlashCommands:AddParam("hide", "button", false, nil, "*", VerifyExistance)
SlashCommands:AddCommand("lock", SlashCommand_Lock, "layout", "button=<Button>");
	SlashCommands:AddParam("lock", "button", false, nil, "*", VerifyExistance)
SlashCommands:AddCommand("unlock", SlashCommand_Unlock, "layout", "button=<Button>");
	SlashCommands:AddParam("unlock", "button", false, nil, "*", VerifyExistance)
SlashCommands:AddCommand("getposition", SlashCommand_GetPosition, "layout", "button=<Button>");
SlashCommands:SetAlias("getposition", "getpos");
	SlashCommands:AddParam("getposition", "button", true, nil, nil, VerifyExistance)
SlashCommands:AddCommand("setposition", SlashCommand_SetPosition, "layout", "button=<Button> x=<x coordinate> y=<y coordinate> (smart=<true/false>)");
SlashCommands:SetAlias("setposition", "position");
SlashCommands:SetAlias("setposition", "pos");
SlashCommands:SetAlias("setposition", "setpos");
	SlashCommands:AddParam("setposition", "button", true, nil, nil, VerifyExistance);
	SlashCommands:AddParam("setposition", "anchor", false, nil, nil, VerifyExistance);
	SlashCommands:AddParam("setposition", "x", false, nil, nil, function(X) if(string.match(X, "^%d*%.?%d+$")) then return true; else return "Invalid X"; end end);
	SlashCommands:AddParam("setposition", "y", false, nil, nil, function(Y) if(string.match(Y, "^%d*%.?%d+$")) then return true; else return "Invalid Y"; end end);
	SlashCommands:AddParam("setposition", "smart", false, {"true", "false"}, "true", nil);
SlashCommands:AddCommand("setscale", SlashCommand_SetScale, "layout", "button=<Button> scale=<Scale>");
SlashCommands:SetAlias("setscale", "scale");
    SlashCommands:AddParam("setscale", "button", true, nil, nil, VerifyExistance)
    SlashCommands:AddParam("setscale", "scale", false, nil, "1", function(Scale) if(string.match(Scale, "^%d*%.?%d+$")) then return true; else return "Invalid scale"; end end);
SlashCommands:AddCommand("setalpha", SlashCommand_SetAlpha, "layout", "button=<Button> scale=<Scale>");
SlashCommands:SetAlias("setalpha", "alpha");
    SlashCommands:AddParam("setalpha", "button", true, nil, nil, VerifyExistance)
    SlashCommands:AddParam("setalpha", "alpha", false, nil, "1", function(Alpha) if(string.match(Alpha, "^%d*%.?%d+$")) then return true; else return "Invalid alpha"; end end);
SlashCommands:AddCommand("tocursor", SlashCommand_ToCursor, "layout", "button=<Button>");
SlashCommands:AddCommand("setbuttontype", SlashCommand_SetButtonType, "layout", "button=<Button> type=<Type>");
SlashCommands:SetAlias("setbuttontype", "buttontype");
	SlashCommands:AddParam("setbuttontype", "button", true, nil, nil, VerifyExistance);
	SlashCommands:AddParam("setbuttontype", "type", true, nil, nil, nil);
SlashCommands:AddCommand("line", SlashCommand_Line, "layout", "button=<Button> (rows=X) (Columns=Y) (space=Z)");
	SlashCommands:AddParam("line", "button",  true, nil, nil, VerifyExistance);
	SlashCommands:AddParam("line", "rows", false, nil, nil, function(Rows) return tonumber(Rows) ~= nil end);
	SlashCommands:AddParam("line", "columns", false, nil, nil, function(Columns) return tonumber(Columns) ~= nil end);
	SlashCommands:AddParam("line", "space", false, nil, "37", function(Space) return tonumber(Space) ~= nil end);
SlashCommands:AddCommand("ring", SlashCommand_Ring, "layout", "button=<Button> radius=<r>, centerx=<x>, centery=<y>");
	SlashCommands:AddParam("ring", "button", true, nil, nil, VerifyExistence); 
	SlashCommands:AddParam("ring", "radius", true, nil, nil, function(Radius) return tonumber(Radius) ~= nil end); 
	SlashCommands:AddParam("ring", "centerx", false, nil, "500", function(CenterX) return tonumber(CenterX) ~= nil end); 
	SlashCommands:AddParam("ring", "centery", false, nil, "400", function(CenterY) return tonumber(CenterY) ~= nil end);
SlashCommands:AddCommand("settexture", SlashCommand_SetTexture, "layout", "texture=<texture>");
SlashCommands:SetAlias("settexture", "texture");
	SlashCommands:AddParam("settexture", "button",  true, nil, nil, VerifyExistance);
	SlashCommands:AddParam("settexture", "texture", false, nil, "", nil);
