--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Button = FlexBar2:NewModule("Button", "LibRockEvent-1.0");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings
Button.Version = 3;

function Button:Activate()
	self:AddEventListener("FlexBar2_SpellCache", "SpellCacheUpdate", "UpdateShowStates");
end

function Button:PostActivate()
	self:UpdateShowStates();
end

function Button:Deactivate()
	self:RemoveEventListener("FlexBar2_SpellCache", "SpellCacheUpdate", "UpdateShowStates");
end

-- This function takes care of updating the showstates of all buttons and refresh the state afterwards, this is ran when new spells are learned
function Button:UpdateShowStates()
	for _, Button in pairs(FlexBar2.Buttons) do
		Button:GenerateShowStates();
	end
	FlexBar2:GetModule("Attributes"):RefreshState();
end

-- Create a button mixin
Button.ButtonMixin = {};

-- This is called while loading the mixin
function Button.ButtonMixin:Load()
	FlexBar2:Debug("               Setting up layout");

	-- Create the actual frame, inheriting from SecureActionButtonTemplate. Set a name for it, as sadly, some blizzard functions require a frame name instead of a reference
	local Frame = CreateFrame("Button", "FB2_BTN_" .. self.uid, UIParent, "SecureActionButtonTemplate");
	self.Frame = Frame;
	Frame.Object = self;
	
	-- Strata and sizing, drag registering
	Frame:SetFrameStrata("MEDIUM");
	Frame:SetFrameLevel(1);
	Frame:SetWidth(36);
	Frame:SetHeight(36);

	-- Textures setup
	Frame:SetPushedTexture("Interface\\Buttons\\UI-Quickslot-Depress");
	Frame:SetHighlightTexture("Interface\\Buttons\\ButtonHilight-Square");
	Frame:GetHighlightTexture():SetBlendMode("ADD");

	-- Icon setup
	Frame.Icon = self.Frame:CreateTexture(nil, "BACKGROUND")
	Frame.Icon:SetAllPoints(Frame);

	-- Cooldown ticker setup
	Frame.Cooldown = CreateFrame("Cooldown", nil, self.Frame, "CooldownFrameTemplate");

	-- NormalTexture setup
	self.Frame:SetNormalTexture("Interface\\Buttons\\UI-Quickslot2");
	local NormalTexture = self.Frame:GetNormalTexture();
	NormalTexture:SetWidth(60);
	NormalTexture:SetHeight(60);
	NormalTexture:ClearAllPoints();
	NormalTexture:SetPoint("CENTER", 1, -1);

	-- Button settings
	Frame:RegisterForClicks("AnyUp");
	Frame:EnableMouse(true);
	Frame:SetClampedToScreen(true);
	Frame:SetMovable(true);

	-- Dragger frame
	Frame.Dragger = CreateFrame("Frame");
	local Dragger = Frame.Dragger;
	Dragger.Object = self;
	Dragger:SetFrameStrata("MEDIUM");
	Dragger:SetFrameLevel(3);
	Dragger:SetAllPoints(Frame);
	Dragger.Texture = Dragger:CreateTexture(nil,'Overlay');
	Dragger.Texture:SetTexture(0, 0.6, 0, 0.5);
	Dragger.Texture:SetAllPoints(Dragger);
	Dragger:EnableMouse(true);
	Dragger:SetClampedToScreen(true);
	Dragger:SetMovable(false);

	-- Declare a text field to show the button's name
	local ButtonNameField = self.Frame.Dragger:CreateFontString(nil, "ARTWORK");
	self.ButtonNameField = ButtonNameField;
	ButtonNameField:SetFontObject("NumberFontNormal");
	ButtonNameField:SetJustifyH("CENTER");
	ButtonNameField:SetPoint("CENTER", self.Frame, "CENTER");
	-- Set scripts for dragging
	Dragger:SetScript("OnMouseUp", function(Button, MouseClick) if(MouseClick == "LeftButton") then self:StopDragging(); end end);
	Dragger:SetScript("OnMouseDown", function(Button, MouseClick)
		-- Drag
		if(MouseClick == "LeftButton") then
			-- Start dragging
			self:StartDragging();
		end
	end);
	-- Support to receive actions while button is unlocked
	self.Frame.Dragger:SetScript("OnReceiveDrag", function() self:PlaceAction(); end);
end

-- Register 
function Button.ButtonMixin:Activate()
	self:RegisterScript("OnShow", "CheckLock");
	self:RegisterScript("OnHide", "Lock");
	self.ButtonNameField:SetText(self.Name);
end


-- Deactivator functions, in case the button gets deactivated, hide it
function Button.ButtonMixin:Deactivate()
	self.Frame:Hide();
	self:UnregisterScript("OnShow", "CheckLock");
	self:UnregisterScript("OnHide", "Lock");
end

-- SetDefaults function, restore everything back to normal
function Button.ButtonMixin:SetDefaults()
	self.InfoTable.Show = true;
	self.InfoTable.Lock = false;
	self.InfoTable.Texture = nil;
	local ButtonNumber = tonumber(self.Name);
	-- Little conditional, if the BUtton's name matches a number, then set its position according to it, this makes everything look prettyer when using numbers as button name
	local StartHeight = UIParent:GetHeight() - UIParent:GetHeight() / 4;
	local StartWidth = UIParent:GetWidth() / 6;
	if(ButtonNumber) then
		self.InfoTable.X = StartWidth + (37 * (ButtonNumber - ((math.ceil(ButtonNumber / 12) - 1) * 12) - 1));
		self.InfoTable.Y = StartHeight - (37 * (math.ceil(ButtonNumber / 12) - 1));
	else
		self.InfoTable.X = StartWidth;
		self.InfoTable.Y = StartHeight;
	end
	self.InfoTable.Scale = 1;
	-- Set default button type to actionbutton, all that stuff is handled by the action module
	self.InfoTable.Type = "Action";
	self.InfoTable.ShowStates = {};
end

function Button.ButtonMixin:UpdateSettings(OldVersion, NewVersion)
	-- Update old remap format to the new one
	if(OldVersion == 1 or OldVersion == 2) then
		self.InfoTable.ShowStates = {};
		OldVersion = 3;
	end
	return true;
end

-- Apply the saved position, status & scale settings
function Button.ButtonMixin:LoadSettings()
	self:GenerateShowStates();
	-- TODO: Find another way to make the buttons show up again after they have been cached
	FlexBar2:GetModule("Attributes"):RefreshState();
	if(self.InfoTable.Lock == true) then
		self:Lock();
	else
		self:Unlock();
	end
	self:SetScale(self.InfoTable.Scale or 1);
	self:SetAlpha(self.InfoTable.Alpha or 1);
	self:SetPosition(self.InfoTable.X, self.InfoTable.Y);
end

-- Change button type + trigger event for it
function Button.ButtonMixin:SetType(Type)
	if(Type ~= self.InfoTable.Type) then
		self.InfoTable.Type = Type;
		self:DispatchEvent("TypeChanged", self, Type);
	end
end

-- Start dragging method
function Button.ButtonMixin:StartDragging()
	-- Check if the button has a group
	if(type(self.GroupMemberList) == "table") then
		-- Get this frame's positioning
		local MasterX, MasterY = self:GetPosition();
		FlexBar2:Debug("Moving button", self.Name, "StartPos: ", MasterX, MasterY);
		-- Loop through all group members
		for _, GroupMember in pairs(self.GroupMemberList) do
			-- Only do something if the group member is not this button, this is important because we cant anchor a button to itself
			if(GroupMember ~= self) then
				local SlaveFrame = GroupMember.Frame;
				-- Get the group member's position
				SlaveX, SlaveY = GroupMember:GetPosition();
				local Scale = SlaveFrame:GetScale();
				local NewX = (SlaveX - MasterX) / Scale;
				local NewY = (SlaveY - MasterY) / Scale;
				-- Set the member's position relative to this frame
				SlaveFrame:ClearAllPoints();
				SlaveFrame:SetPoint("BOTTOMLEFT", self.Frame , "BOTTOMLEFT", NewX, NewY);
				FlexBar2:Debug("   Dragging ", GroupMember.Name, " along anchored to", NewX, NewY);
			end
		end
	end
	-- DRAG !
	self.Frame:StartMoving();
end

function Button.ButtonMixin:StopDragging()
	-- Check if the button has a group
	if(type(self.GroupMemberList) == "table") then
		FlexBar2:Debug("Groupstop");
		-- Get this frame's positioning
		local MasterX, MasterY = self:GetPosition();
		FlexBar2:Debug("stopping move of button", self.Name, "EndPos: ", MasterX, MasterY);
		-- Loop through all group members
		for _, GroupMember in pairs(self.GroupMemberList) do
			-- Only do something if the group member is not this button, this is important because we cant anchor a button to itself
			if(GroupMember ~= self) then
				local SlaveFrame = GroupMember.Frame;
				-- Get the group member's positioning
				local NewX, NewY = GroupMember:GetPosition();
				SlaveFrame.Object:SetPosition(NewX, NewY);
				SlaveFrame.Object.InfoTable.X = NewX;
				SlaveFrame.Object.InfoTable.Y = NewY;
				FlexBar2:Debug("Stopping dragging for [", _ , "], reanchorred to: ", NewX, NewY);
			end
		end
	end
	-- STOP DRAGGING !
	self.Frame:StopMovingOrSizing();
	-- Lets save
	local X, Y = self:GetPosition();
	self.InfoTable.X = X;
	self.InfoTable.Y = Y;
	self:SetPosition();
end

function Button.ButtonMixin:FrameToCursor()
	local X, Y = GetCursorPosition();
	local UIScale = UIParent:GetScale();
	-- Apply UIScale and get the coords relative to BOTTOMLEFT
	X = X / UIScale;
	Y = Y / UIScale;
	self:SetPosition(X, Y);
	return X, Y;
end

function Button.ButtonMixin:SetPosition(X, Y)
	if(type(X) ~= "number") then X = self.InfoTable.X end
	if(type(Y) ~= "number") then Y = self.InfoTable.Y end
	local Scale = self:GetScale();
	self.Frame:ClearAllPoints();
	self.Frame:SetPoint("BOTTOMLEFT", UIParent, "BOTTOMLEFT", X / Scale, Y / Scale);
end

function Button.ButtonMixin:GetPosition()
	-- Always try to return a position relative to TOPLEFT
	local Frame = self.Frame;
	local Scale = self:GetScale();
	return Frame:GetLeft() * Scale, Frame:GetBottom() * Scale;
end

-- Show & hide only do showstates changing, all the rest is done OnShow & OnHide, this is to support showing & hiding due to attribute changes correctly
function Button.ButtonMixin:Show(States)
	local ShowStates = self.InfoTable.ShowStates;
	if(not States) then
		local StateList = FlexBar2:GetModule("Attributes").StateList;
		for Key, StateTable in ipairs(StateList) do
			ShowStates[StateTable[1]] = true;
		end
	else
		for _, StateName in ipairs(States) do
			ShowStates[StateName] = true;
		end
	end
	self:GenerateShowStates();
end

function Button.ButtonMixin:Hide(States)
	local ShowStates = self.InfoTable.ShowStates;
	if(not States) then
		local StateList = FlexBar2:GetModule("Attributes").StateList;
		for Key, StateTable in ipairs(StateList) do
			ShowStates[StateTable[1]] = false;
		end
	else
		for _, StateName in ipairs(States) do
			ShowStates[StateName] = false;
		end
	end
	self:GenerateShowStates();
end

function Button.ButtonMixin:GenerateShowStates()
	local StateList = FlexBar2:GetModule("Attributes").StateList;
	local ShowStates = self.InfoTable.ShowStates;
	local StateAttribute;
	for Key, StateTable in ipairs(StateList) do
		local StateName = StateTable[1];
		-- If the state is not listed then default to showing it, this is handy for new states
		local StateShow = ShowStates[StateName];
		if(StateShow ~= false) then
			if(Key == 1) then
				StateAttribute = StateName;
			else
				StateAttribute = StateAttribute .. "," .. StateName;
			end
		else
			if(Key == 1) then
				StateAttribute = "!" .. StateName;
			else
				StateAttribute = StateAttribute .. ",!" .. StateName;
			end
		end
	end
	self:SetAttribute("showstates", StateAttribute, false);
end

function Button.ButtonMixin:CheckLock()
	-- Restore lock settings
	if(self.InfoTable.Lock == true) then
		self:Lock();
	else
		self:Unlock();
	end
end

function Button.ButtonMixin:Lock()
	self.Frame.Dragger:Hide();
end

function Button.ButtonMixin:Unlock()
	if(self.Frame:IsShown()) then
		self.Frame.Dragger:Show();
	end
end

function Button.ButtonMixin:GetIcon(Icon)
	return self.Frame.Icon:GetTexture(Icon);
end

function Button.ButtonMixin:SetIcon(Icon, IgnoreSavedTexture)
	self.Frame.Icon:SetTexture(IgnoreSavedTexture and Icon or (self.InfoTable.Texture or Icon));
end

function Button.ButtonMixin:SetScale(Scale)
	self.Frame:SetScale(Scale);
	-- Reposition the button with the new scale
	FlexBar2:Debug("Repositioning button with new scale");
	self:SetPosition();
end

-- Gets scale
function Button.ButtonMixin:GetScale()
	return self.Frame:GetScale();
end

function Button.ButtonMixin:SetAlpha(Alpha)
	self.Frame:SetAlpha(Alpha);
end

-- Gets scale
function Button.ButtonMixin:GetAlpha()
	return self.Frame:GetAlpha();
end


-- Small group mixin for showing the group name above it's first button
Button.GroupMixin = {};

function Button.GroupMixin:Activate(Name, ButtonTable)
	local ButtonList = FlexBar2.Buttons;
	local GroupNameField = ButtonList[ButtonTable[1]].Frame.Dragger:CreateFontString(nil, "ARTWORK");
	GroupNameField:SetFontObject("NumberFontNormal");
	GroupNameField:SetJustifyH("CENTER");
	GroupNameField:SetPoint("CENTER", ButtonList[ButtonTable[1]].Frame.Dragger, "CENTER", 0, 20);
	GroupNameField:SetText(Name);
	self.GroupNameField = GroupNameField;
end

function Button.GroupMixin:Deactivate()
	self.GroupNameField:Hide();
end
