--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Attributes =  FlexBar2:NewModule("Attributes", "LibRockEvent-1.0");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings
Attributes.Version = 4;

-- Get an instance of this lovely SecureStateDriverFrame for use with stance, shift & stealth changes
local SecureStateDriverFrame = CreateFrame("Frame", nil, UIParent, "SecureStateDriverTemplate");
Attributes.SecureStateDriverFrame = SecureStateDriverFrame;
-- Function called after FlexBar2's OnInitialize by Ace2
function Attributes:Activate()
	self.States = {};
	self.TypeOrder = {};
	self.StateButtonCache = {};
	SecureStateDriverFrame:SetAttribute("statemap-state", "$input");
	self:AddEventListener("FlexBar2_SpellCache", "SpellCacheUpdate", "UpdateStates");
	self:UpdateStates();
end

function Attributes:Deactivate()
	self:RemoveEventListener("FlexBar2_SpellCache", "SpellCacheUpdate", "UpdateStates");
end

function Attributes:RefreshState()
	SecureStateDriverFrame:SetAttribute("state", SecureStateDriverFrame:GetAttribute("state"));
end

function Attributes:AddState(Type, Name, Params)
	if(not self.States[Type]) then
		self.States[Type] = {};
		tinsert(self.TypeOrder, Type);

	end
	if(self.States[Type][Name]) then return false; end
	tinsert(self.States[Type], {Name, Params});
end

function Attributes:RemoveState(Type, Name)
	if(Type and self.States[Type]) then
		if(Name) then
			for Key, StateTable in ipairs(self.States[Type]) do
				if(StateTable[1] == Name) then
					tremove(self.States[Type], Key);
					return true;
				end
			end
			return false;
		else
			FlexBar2:RemoveByValue(self.TypeOrder, Type);
			self.States[Type] = nil;
		end
	end
end

function Attributes:UpdateStates()
	FlexBar2:Debug("Updating state driver");
	self:RemoveState("shapeshift");
	self:RemoveState("target");
	local _, Class = UnitClass("player");
	-- Add shapeshift forms to the current states
	for State = 1, GetNumShapeshiftForms() do
		local _ , FormName = GetShapeshiftFormInfo(State);
		FormName = FormName:gsub(" ", "");
		if(Class == "DRUID" and (State == 2 or State == 3)) then
			self:AddState("shapeshift", "Stealth", "stance:" .. State .. ",stealth:1");
		end
		self:AddState("shapeshift", FormName, "stance:" .. State);
	end
	-- Specific shadowform support code as its not seen as a shapeshift
	if(Class == "PRIEST") then
		local _, _, _, _, ShadowformRank = GetTalentInfo(3, 18);
		if(ShadowformRank) then
			self:AddState("shapeshift", "ShadowForm", "stance:1");
		end
	end
	self:AddState("shapeshift", "Unshifted", "");
	-- Add friendly/hostile/no target remaps
	self:AddState("target", "harm", "harm");
	self:AddState("target", "help","help");
	self:AddState("target", "none", "");
	self:GenerateStateDriver();
	self:GenerateStateButton();
end

function Attributes:GenerateStateDriver()
	FlexBar2:Debug("Generating state driver");
	local StateList = {};
	self.StateList = StateList;
	local StateDriver = "";
	-- Help nested loops conquer the universe !
	local TypeOrder = self.TypeOrder;
	for MainKey = 1, #TypeOrder do
		local Type = TypeOrder[MainKey];
		local States = self.States[Type];
		local CurrStateList = {};
		for SubKey = 1, #States do
			local StateTable = States[SubKey];
			local State = StateTable[1];
			local Params = StateTable[2];
			-- Consider old states and add to them
			for Key = 1, #StateList do
				local SubStateTable = StateList[Key];
				local PrecStateName = SubStateTable[1];
				local PrecStateParams = SubStateTable[2];
				if(Params ~= "" and PrecStateParams ~= "") then
					tinsert(CurrStateList, {PrecStateName .. "-" .. State, PrecStateParams .. "," .. Params});
				elseif(PrecStateParams == "" and Params ~= "") then
					tinsert(CurrStateList, {PrecStateName .. "-" .. State, Params});
				else
					tinsert(CurrStateList, {PrecStateName .. "-" .. State, PrecStateParams});
				end
				-- Get rid of the old state in favor of the new one
				tinsert(CurrStateList, {PrecStateName, false});
			end
			-- If this is the first time we loop, then we can add the "pure" state name into the list
			if(MainKey == 1) then tinsert(CurrStateList, {State, Params}); end
		end
		-- Merge the CurrStateList into the main one
		for Key = 1, #CurrStateList do

			local CurrStateTable = CurrStateList[Key];
			local State = CurrStateTable[1];
			local Params = CurrStateTable[2];
			if(Params) then
				tinsert(StateList, {State, Params});
			else
				for Key = 1, #StateList do
					local StateTable = StateList[Key];
					if(StateTable[1] == State) then
						tremove(StateList, Key);
						break;
					end
				end
			end
		end
	end
	for Key = 1, #StateList do
		local StateTable = StateList[Key];
		local State = StateTable[1];
		local Params = StateTable[2];
		if(Params) then
			if(Params ~= "") then
				StateDriver = StateDriver .. "[" .. Params .. "]" .. State .. ";";
			else
				StateDriver = StateDriver .. State .. ";";
			end
		end
	end
	Attributes.StateDriver = StateDriver;
	RegisterStateDriver(SecureStateDriverFrame, "state", StateDriver);
	return StateDriver;
end

function Attributes:GenerateStateButton()
	-- Clear cache before regenerating
	local StateButtonCache = Attributes.StateButtonCache;
	for Key in pairs(StateButtonCache) do
		StateButtonCache[Key] = nil;
	end
	for _, Button in pairs(FlexBar2.Buttons) do
		FlexBar2:Debug("Generating state button for ", Button.Name);
		Button:GenerateStateButton();
	end
end


-- Declare the button mixin
Attributes.ButtonMixin = {};

-- During initialization add this button to the state driver's list of buttons to manage
function Attributes.ButtonMixin:Load()
	SecureStateDriverFrame:SetAttribute("addchild", self.Frame);
	self.AttributeList = {};
	self.AttributeCache = {};
end

local function ScheduleAttributeRecache(self) self.RecacheAttributes = true; end;
function Attributes.ButtonMixin:Activate()
	-- Generate the state button attribute based on saved states
	self:GenerateStateButton();
	-- Save current modifiers
	self.ModifierPrefix = self:GetModifierPrefix();
	self.ModifierSuffix = self:GetModifierSuffix();
	-- Register events
	self:RegisterScript("OnAttributeChanged", ScheduleAttributeRecache);
	self:RegisterScript("OnAttributeChanged", "CheckModifierAttributes");
	self:AddEventListener("PLAYER_TARGET_CHANGED", "CheckModifierAttributes");
	self:AddEventListener("MODIFIER_STATE_CHANGED", "CheckModifierAttributes");
end

function Attributes.ButtonMixin:Deactivate()
	self:UnregisterScript("OnAttributeCache", ScheduleAttributeRecache);
	self:RemoveEventListener("PLAYER_TARGET_CHANGED", "CheckModifierAttributes");
	self:RemoveEventListener("MODIFIER_STATE_CHANGED", "CheckModifierAttributes");
	self:ClearAllAttributes();
end

function Attributes.ButtonMixin:UpdateSettings(OldVersion, NewVersion)
	-- Update old remap format to the new one
	if(OldVersion == 1) then
		self.InfoTable.Remaps = {};
		self.InfoTable.Remaps.shapeshift = self.InfoTable.RemapShapeshift;
		self.InfoTable.Remaps.target = self.InfoTable.RemapTarget;
		self.InfoTable.RemapShapeshift = nil;
		self.InfoTable.RemapTarget = nil;
		OldVersion = 2;
	end
	-- Map left & right clicks together
	if(OldVersion == 2) then
		local InfoTable = self.InfoTable.Attributes;
		local NewAttributes = {};
		for AttributeName, AttributeValue in pairs(InfoTable) do
			local AttributeName, AttributeSuffix = string.match(AttributeName, "^(.-)-(.*)$");
			if(AttributeName == "*type" or AttributeName == "*spell" or AttributeName == "*macro" or AttributeName == "*item" or AttributeName == "*itemlink") then
				NewAttributes[AttributeName .. "-RightButton-" .. AttributeSuffix] = AttributeValue;	
			end
		end
		for AttributeName, AttributeValue in pairs(NewAttributes) do
			InfoTable[AttributeName] = AttributeValue;
		end
		OldVersion = 3;
	end
	-- Map left clicks to middle clicks
	if(OldVersion == 3) then
		local InfoTable = self.InfoTable.Attributes;
		local NewAttributes = {};
		for AttributeName, AttributeValue in pairs(InfoTable) do
			local AttributeName, AttributeSuffix = string.match(AttributeName, "^(.-)-(.*)$");
			if((AttributeName == "*type" or AttributeName == "*spell" or AttributeName == "*macro" or AttributeName == "*item" or AttributeName == "*itemlink")
				and not string.match(AttributeSuffix, "^RightButton")) then
				-- Map the found values to the middle mouse button
				NewAttributes[AttributeName .. "-MiddleButton-" .. AttributeSuffix] = AttributeValue;
			end
		end
		for AttributeName, AttributeValue in pairs(NewAttributes) do
			InfoTable[AttributeName] = AttributeValue;
		end
		OldVersion = 4;
	end
	return true;
end

-- Sets state changing to false by default and all attributes to nil
function Attributes.ButtonMixin:SetDefaults()
	self.InfoTable.Attributes = {};
	self.InfoTable.StateButton = false;
	self.InfoTable.Remaps = {};
end

-- Load saved settings
function Attributes.ButtonMixin:LoadSettings()
	for Attribute, Value in pairs(self.InfoTable.Attributes) do
		self:SetAttribute(Attribute, Value, false);
	end
end

function Attributes.ButtonMixin:SaveAttributeName(Trigger, Frame, Attribute)
	self.AttributeList[Attribute] = true;
end

function Attributes.ButtonMixin:ClearAllAttributes()
	for Attribute, _  in pairs(self.AttributeList) do
		self:SetAttribute(Attribute, "", false);
		self.AttributeCache[Attribute] = nil;
	end
end

local function StatesAreEqual(tbl1, tbl2)
	for Key, Value in pairs(tbl1) do
		if(Value and not tbl2[Key]) then return false; end 
	end
	for Key, Value in pairs(tbl2) do
		if(Value and not tbl1[Key]) then return false; end
	end
	return true;
end
-- Generate the state button attribute for this button, this is done on the fly to avoid complications with various combinations

function Attributes.ButtonMixin:GenerateStateButton()
	self:UnregisterScript("OnAttributeChanged", ScheduleAttributeRecache);
	self:UnregisterScript("OnAttributeChanged", "CheckModifierAttributes");
	local StateButton;
	local StateButton2;
	local StateButton3;
	local TypeOrder = Attributes.TypeOrder;
	local RemapInfo = self.InfoTable.Remaps;
	local Generate = true;
	FlexBar2:Debug("Generating state button attribute", self.Name);
	-- Check the statebutton cache if a statebutton equal to this one doesn't already have been generated, use it if it has instead of bothering to generate it again
	for Key, StateButtons in pairs(Attributes.StateButtonCache) do
		if(StatesAreEqual(Key, RemapInfo)) then
			Generate = false;
			FlexBar2:Debug("Using cache of ", self.Name);
			StateButton = StateButtons[1];
			StateButton2 = StateButtons[2];
			StateButton3 = StateButtons[3];
			break;
		end
	end
	if(Generate) then
		local StateList = {};
		self.StateList = StateList;
		FlexBar2:Debug("Couldn't find in cache, generating: ", self.Name);
		StateButton = "";
		StateButton2 = "";
		StateButton3 = "";
		for MainKey = 1, #TypeOrder do
			local Type = TypeOrder[MainKey];
			local States = Attributes.States[Type];
			local CurrStateList = {};
			local RemapName = TypeOrder[MainKey];
			for SubKey = 1, #States do
				local StateTable = States[SubKey];
				local State = StateTable[1];
				local Remap = RemapInfo[RemapName] or false;
				local FallBackState = States[#States][1];
				FlexBar2:Debug(State, Remap, FallBackState);
				-- Consider old states and add to them
				for PrecStateName, PrecStateRemap in pairs(StateList) do
					local SubStateTable = StateList[Key];
					CurrStateList[PrecStateName .. "-" .. State] = Remap and PrecStateRemap .. "-" .. State or PrecStateRemap .. "-" .. FallBackState;
					-- Get rid of the old state in favor of the new one
					CurrStateList[PrecStateName] = false;
				end
				-- If this is the first time we loop, then we can add the "pure" state name into the list
				if(MainKey == 1) then CurrStateList[State] = Remap and State or FallBackState; end
			end
			-- Merge the CurrStateList into the main one
			for StateName, MapName in pairs(CurrStateList) do
				if(MapName) then
					StateList[StateName] = MapName;
				else
					StateList[StateName] = nil;
				end
			end
		end
		for StateName, MapName in pairs(StateList) do
			StateButton = StateButton .. StateName .. ":" .. MapName .. ";";
			StateButton2 = StateButton2 .. StateName .. ":RightButton-" .. MapName .. ";";
			StateButton3 = StateButton3 .. StateName .. ":MiddleButton-" .. MapName .. ";";
		end
		-- Store the generated statebuttons in cache for later reusage
		Attributes.StateButtonCache[RemapInfo] = {StateButton, StateButton2, StateButton3};
	end
	FlexBar2:Debug("Setting state button to " , StateButton);
	self:SetAttribute("statebutton", StateButton, false);
	self:SetAttribute("statebutton2", StateButton2, false);
	self:SetAttribute("statebutton3", StateButton3, false);
	-- Check modifier attributes & planify a recache
	self:CheckModifierAttributes();
	self.RecacheAttributes = true; 
	self:RegisterScript("OnAttributeChanged", ScheduleAttributeRecache);
	self:RegisterScript("OnAttributeChanged", "CheckModifierAttributes");
end

function Attributes.ButtonMixin:GetModifierPrefix()
	return self.ModifierPrefix or SecureButton_GetModifierPrefix();
end

function Attributes.ButtonMixin:GetModifierSuffix()
	return self.ModifierSuffix or SecureButton_GetButtonSuffix(SecureStateChild_GetEffectiveButton(self.Frame));
end


function Attributes.ButtonMixin:CheckModifierAttributes()
	local OldPrefix = self.ModifierPrefix or "";
	local OldSuffix = self.ModifierSuffix or "";
	local NewPrefix = SecureButton_GetModifierPrefix();
	local NewSuffix = SecureButton_GetButtonSuffix(SecureStateChild_GetEffectiveButton(self.Frame));
	if(OldPrefix ~= NewPrefix or OldSuffix ~= NewSuffix) then
		-- Update the prefix & suffix cache then trigger an event that a ModifiedAttribute has changed
		self.ModifierPrefix = NewPrefix;
		self.ModifierSuffix = NewSuffix;
		self:DispatchEvent("ModifierAttributeChanged", self, OldPrefix, OldSuffix, NewPrefix, NewSuffix);
	end
end

-- Gets an attribute on the button, usable both in & out of combat, but not very usefull in combat as SetAttribute is not avail
function Attributes.ButtonMixin:GetAttribute(Attribute)
	local AttributeCache = self.AttributeCache;
	if(self.RecacheAttributes == true) then
		for Key in pairs(AttributeCache) do
			AttributeCache[Key] = nil;
		end
	end
	return AttributeCache[Attribute] or self.Frame:GetAttribute(Attribute);
end

-- This will call :GetAttribute with the button prefix (ctrl-alt-shift-) before it and the button suffix (the current state) after it. if nothing is found with a certain prefix * is used as prefix
function Attributes.ButtonMixin:GetModifiedAttribute(Attribute)
	if(Attribute) then
		return self:GetAttribute(self:GetModifierPrefix() .. Attribute .. self:GetModifierSuffix()) or 
		self:GetAttribute("*" .. Attribute .. self:GetModifierSuffix());
	end
end

-- Set an attribute on the frame, not usable in combat
function Attributes.ButtonMixin:SetAttribute(Attribute, Value, Save)
	self.Frame:SetAttribute(Attribute, Value);
	if(Save ~= false) then self.InfoTable.Attributes[Attribute] = Value; end
	FlexBar2:Debug("Setting Attribute [", Attribute, "] to ", Value);
	-- Put the attribute in the cache so it can be removed if needed
	if(Value ~= "") then self.AttributeList[Attribute] = true; end
	self.AttributeCache[Attribute] = Value;
end

-- Same as :GetAttribute except it sets the attribute instead of getting it :)
function Attributes.ButtonMixin:SetModifiedAttribute(Attribute, Value, Save)
	local Prefix = self:GetModifierPrefix();
	if(Prefix == "") then Prefix = "*"; end
	self:SetAttribute(Prefix .. Attribute .. self:GetModifierSuffix(), Value, Save); 
end
