--[[
	This file is part of FlexBar2.

	FlexBar2 is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	FlexBar2 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with FlexBar2.  If not, see <http://www.gnu.org/licenses/>.
]]
local Action =  FlexBar2:NewModule("Action");
-- If this var is incremented, SetDefaults() will be called for this module thus resetting all its settings.
Action.Version = 2;

-- Get a local version of the text subs module if available

-- PickupAction function, this takes an action type (macro/item/spell) as first argument and its value (macro id/itemlink/spell name) as second
function Action:PickupAction(ActionType, ActionAttribute)
	-- Make sure the cursor is clear before picking anything up
    ClearCursor();
    FlexBar2:Debug("PickupAction:", ActionType, ActionAttribute);
    if(ActionAttribute) then
		-- Pickup the spell with its name
        if(ActionType == "spell") then
            PickupSpell(ActionAttribute);
		-- A bit more complicated, pickup an item using its cached BagID & BagSlot or using its cached InvSlot
        elseif(ActionType == "item" or ActionType == "itemlink") then
            local ItemInfo = FlexBar2:GetModule("ItemCache"):GetItemInfo(ActionAttribute);
            FlexBar2:Debug("Item: ",ActionAttribute);
            if(type(ItemInfo) == "table") then
                if(ItemInfo.Bag and ItemInfo.BagSlot) then
                    PickupContainerItem(ItemInfo.Bag, ItemInfo.BagSlot);
                elseif(ItemInfo.InvSlot) then
                    PickupInventoryItem(ItemInfo.InvSlot);
                else
                    FlexBar2:Print("Item currently not in the inventory, so can not pickup. Sorry :(");
                end                
            else
                FlexBar2:Print("Item currently not in the inventory, so can not pickup. Sorry :(");
            end
		-- Pickup the macro using its ID
        elseif(ActionType == "macro") then
            PickupMacro(ActionAttribute);
        end
    end
end

-- Declare the button mixin
Action.ButtonMixin = {};

-- Local version of functions as to not duplicate them once for each button
local ButtonUpdateActionTexture = function(self, Event, NameSpace, Dispatcher, Button) if(Button == self) then self:UpdateActionTexture(); end end;
local ButtonCheckCooldown = function(self, Event, NameSpace, Dispatcher, Button)
	if(Button == self) then
		self:CheckCooldown();
	end
end;
local ButtonUpdateItemCountSub = function(self) self:UpdateTextSub("itemcount"); end;
local ButtonModifierAttributeChanged = function(self, Event, NameSpace, Dispatcher, Button, OldPrefix, OldSuffix, NewPrefix, NewSuffix) if(Button == self) then self:CheckAction(OldPrefix, OldSuffix, NewPrefix, NewSuffix); end end;
local ButtonTypeChanged = function(self, Event, NameSpace, Dispatcher, Button, Type)
	if(self == Button) then
		if(Type == "Action") then
			Action.ButtonMixin.Activate(self);
		else
			Action.ButtonMixin.Deactivate(self);
		end
	end
end
function Action.ButtonMixin:Activate()
	-- Don't unregister this event on Deactivate, the button might get an actionbutton again later but still make sure to only register it once
	if(not self.ActionRegisteredTypeChanged) then
		self:AddEventListener("FlexBar2_EventManager", "TypeChanged", ButtonTypeChanged);
		self.ActionRegisteredTypeChanged = true;
	end
	if(self.InfoTable.Type == "Action") then
		-- Update button texture on shapeshifts as sometimes an Icon can change without the actual action name changing (think Attack for druids)
		self:AddEventListener("UPDATE_SHAPESHIFT_FORM", "UpdateActionTexture");
		-- Register for dragging so action's can be dragged out
		self.Frame:RegisterForDrag("LeftButton");
		-- Update the cooldown spinner when we get this event
		self:AddEventListener("ACTIONBAR_UPDATE_COOLDOWN", "UpdateCooldownSpinner");
		-- Update the texture of the button when the action changed
		self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonUpdateActionTexture);
		-- A cache of old actions is kept, see later (at the PreClick_DisableButton function)
		self.OldActionCache = {};
		-- Update the itemcount text sub with the BAG_UPDATE event and the FB2_ActionChanged event
		if(TextSubs) then self:AddEventListener("BAG_UPDATE", ButtonUpdateItemCountSub); end
		-- Listen to changing modifier attributes and check if the action on the button changed
		self:AddEventListener("FlexBar2_EventManager", "ModifierAttributeChanged", ButtonModifierAttributeChanged);
		self:AddEventListener("SPELL_UPDATE_COOLDOWN", "CheckCooldown");
		self:AddEventListener("BAG_UPDATE_COOLDOWN", "CheckCooldown");
		self:AddEventListener("FlexBar2_EventManager", "ActionChanged", ButtonCheckCooldown);
		-- Update the texture of buttons when equipped items change
		self:AddEventListener("UNIT_INVENTORY_CHANGED", "UpdateActionTexture");
		if(FlexBar2:HasModule("TextSubs")) then self:CheckCooldown(); end
		self:UpdateActionTexture();
		self:UpdateCooldownSpinner();
		self.Frame.Dragger:HookScript("OnMouseDown", 	
		function(Button, MouseClick)
			self:PlaceAction(MouseClick);
		end);

	end
end

function Action.ButtonMixin:Deactivate()
	self:RemoveEventListener("UPDATE_SHAPESHIFT_FORM", "UpdateActionTexture");
	-- Update the cooldown spinner when we get this event
	self:RemoveEventListener("ACTIONBAR_UPDATE_COOLDOWN", "UpdateCooldownSpinner");
	-- Update the texture of the button when the action changed
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", ButtonUpdateActionTexture);
	-- Update the itemcount text sub with the BAG_UPDATE event and the FB2_ActionChanged event
	self:RemoveEventListener("BAG_UPDATE", ButtonUpdateItemCountSub);
	-- Listen to changing modifier attributes and check if the action on the button changed
	self:RemoveEventListener("FlexBar2_EventManager", "ModifierAttributeChanged", ButtonModifierAttributeChanged);
	self:RemoveEventListener("SPELL_UPDATE_COOLDOWN", "CheckCooldown");
	self:RemoveEventListener("BAG_UPDATE_COOLDOWN", "CheckCooldown");
	self:RemoveEventListener("FlexBar2_EventManager", "ActionChanged", "CheckCooldown");

	self:SetIcon("");
end



-- ActionBUttons are locked by default
function Action.ButtonMixin:SetDefaults()
    self.InfoTable.LockAction = false;
end

-- Load the lock setting & apply it
function Action.ButtonMixin:LoadSettings()
    if(self.InfoTable.LockAction == true) then
        self:LockAction();
    else
        self:UnlockAction();
    end
end

function Action.ButtonMixin:CheckCooldown()
	local StartTime, Duration, Enable = self:GetActionCooldown();
	self:UpdateCooldownSpinner();
	if(FlexBar2:HasModule("TextSubs")) then
		if(StartTime and StartTime > 0 and Duration > 0 and Enable > 0) then
			self.CooldownStart = StartTime;
			self.CooldownDuration = Duration;
			self.CooldownEnable = Enable;
			self.EndTime = StartTime + Duration;
			if(not self:HasEventListener("FlexBar2_EventManager", "UPDATE", "UpdateCooldown")) then self:AddEventListener("FlexBar2_EventManager", "UPDATE", "UpdateCooldown"); end
		else
			self.CooldownStart = 0;
			self.CooldownDuration = 0;
			self.CooldownEnable = 0;
			self.EndTime = 0;
		end
	end
end

function Action.ButtonMixin:UpdateCooldown()
	if(self.EndTime and GetTime() > self.EndTime) then
		self:RemoveEventListener("FlexBar2_EventManager", "UPDATE", "UpdateCooldown");
	end
	FlexBar2:Debug("updating cd on button", self.Name);
	self:UpdateTextSub("cd");
end

function Action.ButtonMixin:CheckAction(OldPrefix, OldSuffix, NewPrefix, NewSuffix)
	local OldType = self:GetAttribute(OldPrefix .. "type" .. OldSuffix) or self:GetAttribute("*" .. "type" .. OldSuffix) or "";
	local NewType = self:GetAttribute(NewPrefix .. "type" .. NewSuffix) or self:GetAttribute("*" .. "type" .. NewSuffix) or "";
	-- Get the old action, on failure try using a wildcard as prefix
	local OldAction = self:GetAttribute(OldPrefix .. OldType .. OldSuffix) or self:GetAttribute("*" .. OldType .. OldSuffix);
	-- Get new action, on failure try using a wildcard as prefix
	local NewAction = self:GetAttribute(NewPrefix .. NewType .. NewSuffix) or self:GetAttribute("*" .. NewType .. NewSuffix);
	-- Dispatch the ActionChanged event if old & new actions aren't equal
	if(OldAction ~= NewAction) then
		self:DispatchEvent("ActionChanged", self);
	end
end

-- Unregister all drag scripts when the button is locked
function Action.ButtonMixin:LockAction()
	-- Disable regular dragging
    self:UnregisterScript("OnDragStart", "PickupCurrentAction");
    self:UnregisterScript("OnReceiveDrag", "PlaceAction");
    self:UnregisterScript("PreClick", "PreClick_DisableButton");
    self:UnregisterScript("PostClick", "PostClick_PickOldAction");
end

-- Register all drag scripts when the button is unlocked
function Action.ButtonMixin:UnlockAction()
	-- Enable regular dragging
    self:RegisterScript("OnDragStart", "PickupCurrentAction");
    self:RegisterScript("OnReceiveDrag", "PlaceAction");
    self:RegisterScript("PreClick", "PreClick_DisableButton");
    self:RegisterScript("PostClick", "PostClick_PickOldAction");	
end

-- Disable the button, this should not be used for log disablings. only works OUT of combat
function Action.ButtonMixin:DisableAction()
    if(InCombatLockdown() == nil) then
        self.DisabledActionTypeLeft = self:GetModifiedAttribute("type");
	self.DisabledActionTypeRight = self:GetModifiedAttribute("type-RightButton");
	self.DisabledActionTypeMiddle = self:GetModifiedAttribute("type-MiddleButton");
        self:SetModifiedAttribute("type", nil, false);
	self:SetModifiedAttribute("type-RightButton", nil, false);
	self:SetModifiedAttribute("type-MiddleButton", nil, false);
    end
end

-- Reenable the button. only works OUT of combat
function Action.ButtonMixin:EnableAction()
	if(InCombatLockdown() == nil) then
		if(self.DisabledActionTypeLeft ~= nil) then
			self:SetModifiedAttribute("type", self.DisabledActionTypeLeft, false);
			self.DisabledActionTypeLeft = nil;
		end
		if(self.DisabledActionTypeRight ~= nil) then
			self:SetModifiedAttribute("type-RightButton", self.DisabledActionTypeRight, false);
			self.DisabledActionTypeRight = nil;
		end
		if(self.DisabledActionTypeMiddle ~= nil) then
			self:SetModifiedAttribute("type-MiddleButton", self.DisabledActionTypeMiddle, false);
			self.DisabledActionTypeMiddle = nil;
		end
	end
end

-- Get the texture that's suposed to currently be on the button
function Action.ButtonMixin:GetActionTexture()
	local ActionType = self:GetModifiedAttribute("type");
	FlexBar2:Debug(ActionType);
	-- If its a spell, get the spell's texture using its name
	if(ActionType == "spell") then
		local Spell = self:GetModifiedAttribute("spell");
		if(Spell and Spell ~= "") then
			return GetSpellTexture(Spell) or "Interface\\Icons\\INV_Misc_QuestionMark";
		else
			return "";
		end
		-- If its an item get its texture with GetItemInfo using its itemlink
	elseif(ActionType == "item") then
		local ItemLink = self:GetModifiedAttribute("itemlink");
		if(ItemLink and ItemLink ~= "") then
			local _, _, _, _, _, _, _, _, _, ItemTexture = GetItemInfo(ItemLink);
			return ItemTexture or "Interface\\Icons\\INV_Misc_QuestionMark";
		else
			return "";
		end
		-- If its a macro get the texture using the macro ID
	elseif(ActionType == "macro") then
		local MacroName = self:GetModifiedAttribute("macro");
		if(MacroName and MacroName ~= "") then
			local _, MacroTexture, _, _ = GetMacroInfo(MacroName);
			return MacroTexture or "Interface\\Icons\\INV_Misc_QuestionMark";
		else
			local MacroText = self:GetModifiedAttribute("macrotext");
			if(MacroText and MacroText ~= "") then
				-- No textures for custom macro's, if the user didn't specify any put up a question mark
				return "Interface\\Icons\\INV_Misc_QuestionMark";
			else
				return "";
			end
		end
	else
		return "";
	end
end

-- set the actionbutton's texture using GetActionTexture
function Action.ButtonMixin:UpdateActionTexture()
	local Icon = self:GetActionTexture();
	FlexBar2:Debug("Updating Icon to: " , Icon);
	self:SetIcon(Icon);
end

-- Get an actionbutton's cooldown
function Action.ButtonMixin:GetActionCooldown()
	local ActionType = self:GetModifiedAttribute("type");
	-- If its a spell use GetSpellCooldown with the spell's name
	if(ActionType == "spell") then
		local Spell = self:GetModifiedAttribute("spell");
		if(Spell) then return GetSpellCooldown(Spell); end
		-- If its an item either use GetContainerItemCooldown or GetInventoryItemCooldown depending on wether the item is equiped or not
	elseif(ActionType == "item") then
		local Itemlink = self:GetModifiedAttribute("itemlink");
		if(Itemlink) then return GetItemCooldown(Itemlink); end
	elseif(ActionType == "macro") then
		local Macro = self:GetModifiedAttribute("macro");
		if(Macro) then
			local Spell,Rank = GetMacroSpell(Macro);
			if(Spell) then
				if(Rank) then 
					Spell_cat=Spell .. "(" .. Rank .. ")";
				else
					Spell_cat=Spell .. "( )";
				end
				return GetSpellCooldown(Spell_cat);
			else
				local Item,Itemlink = GetMacroItem(Macro);
				if(Itemlink) then return GetItemCooldown(Itemlink); end
			end
		 end
	end
end

-- Update the button's cooldown spinner using GetActionCooldown's return stuff
function Action.ButtonMixin:UpdateCooldownSpinner() 
	local CooldownFrame = self.Frame.Cooldown;
	local start, duration, enable = self:GetActionCooldown();
	FlexBar2:Debug("Updating cooldown spinner: ", start, duration, enable);
	if(start ~= nil and duration ~= nil and enable ~= nil and start > 0 and duration > 0 and enable > 0) then
		CooldownFrame:SetCooldown(start, duration);
		CooldownFrame:Show();
	else
		CooldownFrame:Hide();
	end
end

-- Clear the button's action for this state only. only works OUT of combat
function Action.ButtonMixin:ClearAction()
    if(InCombatLockdown() == nil) then
        self:SetModifiedAttribute("type", nil);
        self:SetModifiedAttribute("spell", nil);
        self:SetModifiedAttribute("macro", nil);
        self:SetModifiedAttribute("item", nil);
        self:SetModifiedAttribute("itemlink", nil);
	-- Clear rightbutton stuff too
	self:SetModifiedAttribute("type-RightButton", nil);
        self:SetModifiedAttribute("spell-RightButton", nil);
        self:SetModifiedAttribute("macro-RightButton", nil);
        self:SetModifiedAttribute("item-RightButton", nil);
	self:SetModifiedAttribute("itemlink-RightButton", nil);
	-- Clear the middle button also
	self:SetModifiedAttribute("type-MiddleButton", nil);
        self:SetModifiedAttribute("spell-MiddleButton", nil);
        self:SetModifiedAttribute("macro-MiddleButton", nil);
        self:SetModifiedAttribute("item-MiddleButton", nil);
	self:SetModifiedAttribute("itemlink-MiddleButton", nil);

	-- Update the textute after nil'ing out all this stuff
        self:UpdateActionTexture();
    end
end

-- Disable the button on preclick if the cursor contains anything, this allows us to put the action on the button without triggering a click. only works OUT of combat
function Action.ButtonMixin:PreClick_DisableButton(Event, Button, MouseClick)
	if(GetCursorInfo() ~= nil and InCombatLockdown() == nil) then
		local OldActionCache = self.OldActionCache;
		OldActionCache.ActionType = self:GetModifiedAttribute("type");
		OldActionCache.ActionAttribute = self:GetModifiedAttribute(OldActionCache.ActionType);
		-- Place any actions currently on the cursor on the button
		self:PlaceAction(MouseClick);
		-- Disable the button so it wont catch the click and trigger the newly placed action
		self:DisableAction();
	end
end

-- Reenable the button on postclick so it can catch clicks again
function Action.ButtonMixin:PostClick_PickOldAction()
	local OldActionCache = self.OldActionCache;
	if(OldActionCache.ActionType ~= nil and OldActionCache.ActionType ~= "" and OldActionCache.ActionAttribute ~= nil and OldActionCache.ActionAttribute ~= "") then
		Action:PickupAction(OldActionCache.ActionType, OldActionCache.ActionAttribute);
		OldActionCache.ActionType = nil;
		OldActionCache.ActionAttribute = nil;
	end
		-- Reenable the button so it can catch clicks again
	self:EnableAction();
end

-- Pickup the current action from the button to the cursor; only works OUT of combat
function Action.ButtonMixin:PickupCurrentAction()
	if(InCombatLockdown() == nil) then
		-- Get Action's type
		local ActionType = self:GetModifiedAttribute("type");
		-- Pickup the action using its type and the type's value
		Action:PickupAction(ActionType, self:GetModifiedAttribute(ActionType));
		-- Clear any actions currently on the cursor
		self:ClearAction();
		-- Update cooldown spinner & texture
		self:UpdateCooldownSpinner();
		self:UpdateActionTexture();
		-- Trigger an event for this, so other modules can behave accordingly, pass self as a paremeter so a check can be made to only update this button
		self:DispatchEvent("ActionChanged", self);
	end
end

-- Place the action currently on the cursor to the button. only works OUT of combat
function Action.ButtonMixin:PlaceAction(MouseClick)
	if(InCombatLockdown() == nil) then
		-- Get info from the cursor, does not return anything for pets, so unsupported
		local ActionType, ActionId, ActionBook = GetCursorInfo()
		FlexBar2:Debug("PlaceAction:", ActionType, ActionId, ActionBook);
		-- If there is currently an action on the button, pick it up
		if(ActionType ~= nil) then self:PickupCurrentAction(); end
		-- ActionType is a spell, get the spell's name & rank, format it into a SpellName(Rank X) format and save it
		if(ActionType == "spell") then
			self:SetModifiedAttribute("type", "spell");
			self:SetModifiedAttribute("type-RightButton", "spell");
			self:SetModifiedAttribute("type-MiddleButton", "spell");


			local SpellName, SpellRank = GetSpellName(ActionId, ActionBook);
			-- Parse a rank number out of Rank X and get the localized version of "Rank"
			SpellRank = string.match(SpellRank, "^.* (%d+)$");
			SpellRank = tonumber(SpellRank);
			-- If the spell is the highest rank available to the player, dont specify the rank, that way it'll autorank up when the user learns a new rank
			if(type(SpellRank) == "number") then
				-- Find out if the spell is the highest rank by requesting the texture of the spell one rank higher, if this isn't nil, then a rank above exists
				local HigherRankExists = GetSpellTexture(SpellName .. "(" .. RANK .. " " .. SpellRank + 1 .. ")") ~= nil;
				if(HigherRankExists) then
					SpellName = SpellName .. "(" .. RANK .. " " .. SpellRank .. ")";
				else
					SpellName = SpellName .. "()";
				end
			end
			self:SetModifiedAttribute("spell", SpellName);
			self:SetModifiedAttribute("spell-RightButton", SpellName);
			self:SetModifiedAttribute("spell-MiddleButton", SpellName);
			-- Get the itemlink & save it. the itemlink attribute is not needed but is kept so to not have to remove itemlink: from the front of the item attribute when the raw item link is needed (blizzard funcs :p)
		elseif(ActionType == "item") then
			self:SetModifiedAttribute("type", "item");
			self:SetModifiedAttribute("type-RightButton", "item");
			self:SetModifiedAttribute("type-MiddleButton", "item");
			local ItemLink = string.gsub(ActionBook, "(.*):(.-)$", "%1:0");
			self:SetModifiedAttribute("item", "itemlink:" .. ItemLink);
			self:SetModifiedAttribute("item-RightButton", "itemlink:" .. ItemLink);
			self:SetModifiedAttribute("item-MiddleButton", "itemlink:" .. ItemLink);
			self:SetModifiedAttribute("itemlink", ItemLink);
			self:SetModifiedAttribute("itemlink-RightButton", ItemLink);
			self:SetModifiedAttribute("itemlink-MiddleButton", ItemLink);
		-- Macro, just save the macro's ID
		elseif(ActionType == "macro") then
			self:SetModifiedAttribute("type", "macro");
			local MacroName = GetMacroInfo(ActionId);
			self:SetModifiedAttribute("macro", MacroName);
			self:SetModifiedAttribute("type-RightButton", "macro");
			self:SetModifiedAttribute("type-MiddleButton", "macro");
			self:SetModifiedAttribute("macro-RightButton", MacroName);
			self:SetModifiedAttribute("macro-MiddleButton", MacroName);
		end
		-- Update cooldown spinner & button texture
		self:UpdateCooldownSpinner();
		self:UpdateActionTexture();
		-- Trigger an event for this, so other modules can behave accordingly, pass self as a paremeter so a check can be made to only update this button
		self:DispatchEvent("ActionChanged", self);
	else
		FlexBar2:Print("Actions cannot be changed in combat, sorry");
	end
end


