-------------------------------------------------------------------------------
-- Upvalues
-------------------------------------------------------------------------------

local button
local setupEvent = 'PLAYER_ENTERING_WORLD'

-------------------------------------------------------------------------------
-- Setup
-------------------------------------------------------------------------------

local function Setup()
	button:UnregisterEvent(setupEvent)
	button:SetScript('OnEvent', nil)

	local function Print(msg, ...)
		DEFAULT_CHAT_FRAME:AddMessage('|cffffff78FeedIt:|r '..msg)
	end
	
	local function Feedback(msg, ...)
		UIErrorsFrame:AddMessage('|cffffff78[FeedIt]|r |cffffCC00'..msg:format(...)..'|r')
	end

	local Debug = function() end

	local IS_WRATH_BUILD = select(4, GetBuildInfo()) >= 30000

	-- Localization
	local NOT_AN_HUNTER = 'You are not an hunter. FeedIt is disabled.'
	local FEED_PET_UNKNOWN = 'You do not know the Feed Pet spell. FeedIt is disabled.'
	local DEAD_PET = '%s is dead.'
	local PET_IS_HAPPY = '%s is already happy.'
	local PET_IS_EATING = '%s is already eating.'
	local NO_SUITABLE_FOOD = 'Could not find suitable food for %s.'
	local BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT
				
	local locale = GetLocale()
	if locale == 'enUS' or locale == 'enGB' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = 'Bread', 'Cheese', 'Fish', 'Fruit', 'Fungus', 'Meat'

	elseif locale == 'frFR' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = 'Pain', 'Fromage', 'Poisson', 'Fruit', 'Champignon', 'Viande'
		NOT_AN_HUNTER = "Vous n'êtes pas un chasseur. FeedIt désactivé."
		FEED_PET_UNKNOWN = "Vous ne connaissez pas Nourrir le familier. FeedIt désactivé."
		DEAD_PET = '%s est mort.'
		PET_IS_HAPPY = '%s est déjà heureux.'
		PET_IS_EATING = '%s est en train de manger.'
		NO_SUITABLE_FOOD = 'Pas de nourriture adaptée pour %s.'
	
	elseif locale == 'deDE' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = 'Brot', 'Käse', 'Fisch', 'Frucht', 'Pilz', 'Fleisch'
		-- NOT_AN_HUNTER = 'You are not an hunter. FeedIt is disabled.'
		-- FEED_PET_UNKNOWN = 'You do not know the Feed Pet spell. FeedIt is disabled.'
		-- DEAD_PET = '%s is dead.'
		-- PET_IS_HAPPY = '%s is already happy.'
		-- PET_IS_EATING = '%s is already eating.'
		-- NO_SUITABLE_FOOD = 'Could not find suitable food for %s.'
		
	elseif locale == 'koKR' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = '빵', '치즈', '물고기', '과일', '곰팡이류', '고기'
		-- NOT_AN_HUNTER = 'You are not an hunter. FeedIt is disabled.'
		-- FEED_PET_UNKNOWN = 'You do not know the Feed Pet spell. FeedIt is disabled.'
		-- DEAD_PET = '%s is dead.'
		-- PET_IS_HAPPY = '%s is already happy.'
		-- PET_IS_EATING = '%s is already eating.'
		-- NO_SUITABLE_FOOD = 'Could not find suitable food for %s.'

	elseif locale == 'zhCN' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = '面包', '奶酪', '鱼类', '水果', '蘑菇', '肉类'
		-- NOT_AN_HUNTER = 'You are not an hunter. FeedIt is disabled.'
		-- FEED_PET_UNKNOWN = 'You do not know the Feed Pet spell. FeedIt is disabled.'
		-- DEAD_PET = '%s is dead.'
		-- PET_IS_HAPPY = '%s is already happy.'
		-- PET_IS_EATING = '%s is already eating.'
		-- NO_SUITABLE_FOOD = 'Could not find suitable food for %s.'

	elseif locale == 'zhTW' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = '麵包', '乳酪', '魚類', '水果', '菌類', '肉類'
		NOT_AN_HUNTER = '你不是一個獵人。 FeedIt 停用。'
		FEED_PET_UNKNOWN = '你尚未習得 餵養寵物。 FeedIt 停用。'
		DEAD_PET = '%s 死亡。'
		PET_IS_HAPPY = '%s 感到高興。'
		PET_IS_EATING = '%s 進食中。'
		NO_SUITABLE_FOOD = '未發現適合 %s 的食物。'

	elseif locale == 'esES' then
		BREAD, CHEESE, FISH, FRUIT, FUNGUS, MEAT = 'Pan', 'Queso', 'Pescado', 'Fruta', 'Hongo', 'Carne'
		-- NOT_AN_HUNTER = 'You are not an hunter. FeedIt is disabled.'
		-- FEED_PET_UNKNOWN = 'You do not know the Feed Pet spell. FeedIt is disabled.'
		-- DEAD_PET = '%s is dead.'
		-- PET_IS_HAPPY = '%s is already happy.'
		-- PET_IS_EATING = '%s is already eating.'
		-- NO_SUITABLE_FOOD = 'Could not find suitable food for %s.'

	else
		Print('FeedIt has not support for your locale ('..locale..'). It is disabled.')
		return
	end

	-- Disable if not an hunter
	if select(2, UnitClass('player')) ~= 'HUNTER' then
		Print(NOT_AN_HUNTER)
		return
	end

	-- Check Feed Pet spell
	local feedPetSpell = GetSpellInfo(GetSpellInfo(6991))
	if not feedPetSpell then
		Print(FEED_PET_UNKNOWN)
		return
	end
	local feedPetEffect = GetSpellInfo(1539)

	-- Create our food sets
	local foodSets = {	
		conjured = {}, -- special case
		-- Some items that pets do not like
		blacklist = { [6807] = true }
	}
	do
		-- Data ripped from LibPeriodicTable-3.1 revision 77769
		local sourceSets = {
			[BREAD] = {
				["Consumable.Food.Edible.Bread.Basic"]="19696:50,20857:61,4540:61,4541:243,4542:552,4544:874,16169:874,4601:1392,8950:2148,23160:2148,29394:7500,27855:4320,28486:4320,24072:243,30816:61,29449:7500",
				["Consumable.Food.Edible.Bread.Combo.Conjured"]="34062:7500",
				["Consumable.Food.Edible.Bread.Combo.Non-Conjured"]="13724:2148,19301:4410,34780:7500",
				["Consumable.Food.Edible.Bread.Conjured"]="1113:243,1114:552,1487:874,5349:61,8075:1392,8076:2148,22895:4320,22019:7500",
			},
			[CHEESE] = {
				["Consumable.Food.Edible.Cheese.Basic"]="2070:61,414:243,17406:243,422:552,1707:874,3927:1392,8932:2148,27857:4320,29448:7500,30458:4320",
			},
			[FISH] = {
				["Consumable.Food.Edible.Fish.Basic"]="19996:50,1326:243,6316:243,21552:1392,16766:1392,2682:294,4592:243,4593:552,4594:874,5095:243,6290:61,6887:1392,787:61,8364:874,8957:2148,8959:1392,13546:1392,13930:1392,13935:2148,27661:4320,27858:4320,12238:243,13933:2148,29452:7500,33048:7500,35285:7500",
				["Consumable.Food.Edible.Fish.Combo"]="21071:155,2682:294,21153:567,33053:7500",
				["Consumable.Food.Inedible.Fish"]="2675,4603:874,4655,5468,5503,5504,6289:61,6291:30,6303:30,6308:243,6317:61,6361:61,6362:552,7974,8365:552,12206,13754:874,13755:874,13756:874,13758:874,13759:874,13760:874,13889:1392,13893:1392,15924,13890,13888:1392,27422,27435,27437,27438,27439,27429,27425",
			},
			[FRUIT] = {
				["Consumable.Food.Edible.Fruit.Basic"]="29450:7500,19994:50,22324:2148,4536:61,4537:243,4538:552,4539:874,4602:1392,8953:2148,16168:1392,21033:2148,21031:2148,21030:1392,29393:4320,27856:4320",
				["Consumable.Food.Edible.Fruit.Combo"]="28112:4410,20031:2550",
			},
			[FUNGUS] = {
				["Consumable.Food.Edible.Fungus.Basic"]="4604:61,4605:243,4606:552,4607:874,4608:1392,8948:2148,27859:4320,30355:7500,29453:7500",
				["Consumable.Food.Edible.Fungus.Combo"]="3448:294",
			},
			[MEAT] = {
				["Consumable.Food.Edible.Meat.Basic"]="7097:61,21235:50,19995:50,2679:61,17407:874,6807:874,19224:874,19223:61,117:61,2287:243,19304:243,2681:61,2685:552,3770:552,3771:874,19305:552,4599:1392,5478:552,6890:243,8952:2148,19306:1392,9681:1392,9681:61,29451:7500,30610:4320,27854:4320,23495:61,17119:243,11444:2148,32685:7500,32686:7500,33254:7500",
				["Consumable.Food.Inedible.Meat"]="729,20424,769,1015,1081,2672,2673,2677,2924,3173,3404,3667,3712,3730,3731,4739,5051,5465,5467,5469,5470,5471,12037,12184,12202,12203,12204,12205,12208,12223,27669,27671,27677,27678,22644,24477,27668,23676,31670,27674,31671,27682,27681,35562",
			},
		}

		local GetSetString
		local PT3 = LibStub and LibStub('LibPeriodicTable-3.1')
		if PT3 then
			GetSetString = function(name) return PT3:GetSetString(name) end
		else
			GetSetString = function(name, default) return default end
		end

		for foodType,sets in pairs(sourceSets) do
			foodSets[foodType] = {}
			for name, default in pairs(sets) do
				for item in GetSetString(name, default):gmatch('[^,]+') do
					local id, value = item:match('^(%d+):(%d+)')
					id, value = tonumber(id) or tonumber(item), tonumber(value) or true
					if not foodSets.blacklist[id] then
						foodSets[foodType][id] = value
					end
				end
				if name:match('.Conjured') then
					for id in pairs(foodSets[foodType]) do
						foodSets.conjured[id] = true
					end
				end
			end
		end
	end

	local GetFoodSets
	do
		local tmp = {}
		function GetFoodSets(...)
			local n = select('#', ...)
			for i = 1, n do
				tmp[i] = foodSets[select(i, ...)]
			end
			for i = n+1,#tmp do
				tmp[i] = nil
			end
			return #tmp > 0 and tmp or nil
		end
	end

	local function GetBestPetFood()
		local petFoods = GetFoodSets(GetPetFoodTypes())
		if not petFoods then
			return
		end
		local requiredFoodLevel = UnitLevel('pet') - 30
		local playerHealth = UnitHealthMax('player')

		local bestFood, bestScore = nil, 0

		-- Check every items of the sets
		for i,set in ipairs(petFoods) do
			for id,value in pairs(set) do
				local count = GetItemCount(id)
				if count > 0 then
					local name, _, _, level, reqLevel = GetItemInfo(id)
					if level >= requiredFoodLevel then
						-- Fuzzy scoring
						local score = 0
						if type(value) == 'number' then
							-- Edible food, try to give lower level food first
							score = score + 100 * (playerHealth - value) / playerHealth
						else
							-- Unedible food always scores 100
							score = score + 100
						end
						-- Try to get rid of smaller piles first
						score = score - 2 * count
						-- Conjured items get an additionnal score boost
						if foodSets.conjured[id] then
							score = score + 50
						end

						Debug('Found food %q, score %d', name, score)
						if not bestFood or score > bestScore then
							bestFood, bestScore = name, score
						end
					end
				end
			end
		end
		return bestFood
	end

	-------------------------------------------------------------------------------
	-- Button setup
	-------------------------------------------------------------------------------

	local function SetFood(item)
		if item then
			button:SetAttribute('type', 'spell')
			button:SetAttribute('spell', feedPetSpell)
			button:SetAttribute('target-item', item)
			Debug('Setup for %q (%q)', item, feedPetSpell)
		elseif button:GetAttribute('spell') then
			button:SetAttribute('type', ATTRIBUTE_NOOP)
			button:SetAttribute('spell', nil)
			button:SetAttribute('target-item', nil)
			Debug('Cleansed')
		end
	end

	local IsPetEating
	if IS_WRATH_BUILD then
		function IsPetEating()
			local expirationTime = select(7, UnitBuff('pet', feedPetEffect))
			return expirationTime, expirationTime
		end
	else
		function IsPetEating()
			for i = 1,32 do
				local name, _, _, _, _, timeLeft = UnitBuff('pet',i)
				if name == feedPetEffect then
					return true, GetTime() + timeLeft
				elseif not name then
					return
				end
			end
		end
	end

	local cooldown = 0
	local function OnPreClick()
		if InCombatLockdown() or cooldown >= GetTime() or not UnitExists('pet') then return end
		cooldown = GetTime() + 1.5
		if UnitIsDeadOrGhost('pet') then
			return Feedback(DEAD_PET, UnitName('pet'))
		end
		local happiness = GetPetHappiness()
		if happiness and happiness >= 3 then
			return Feedback(PET_IS_HAPPY, UnitName('pet'))
		end
		local isEating, expirationTime = IsPetEating()
		if isEating then
			cooldown = expirationTime
			return Feedback(PET_IS_EATING, UnitName('pet'))
		else
			local food = GetBestPetFood()
			if food then
				SetFood(food)
			else
				return Feedback(NO_SUITABLE_FOOD, UnitName('pet'))
			end
		end
	end

	local function OnPostClick()
		if InCombatLockdown() then return end
		SetFood(nil)
	end

	-------------------------------------------------------------------------------
	-- Create the button and go
	-------------------------------------------------------------------------------

	if IS_WRATH_BUILD then
		button:SetScript("PreClick", OnPreClick)
		button:SetScript("PostClick", OnPostClick)
	else
		-- gross 2.4 broken /click workaround
		button:SetScript("PreClick", function()
			UIParent:UnregisterEvent('ADDON_ACTION_FORBIDDEN')
			OnPreClick()
		end)
		button:SetScript("PostClick", function()
			OnPostClick()
			UIParent:RegisterEvent('ADDON_ACTION_FORBIDDEN')
		end)
	end

	Debug('Loaded and enabled')
end

-------------------------------------------------------------------------------
-- Bootstrap
-------------------------------------------------------------------------------

button = CreateFrame('Button', 'FeedItButton', UIParent, "SecureActionButtonTemplate")
if IsLoggedIn() then
	Setup()
else
	button:RegisterEvent(setupEvent)
	button:SetScript('OnEvent', Setup)
end
