--[[-----------------------------------------------------------------------------------------------------------------
FSRTimer.lua

A customizable timer bar for the caster 5-second rule.

	(c)2008 John Mann <john.d.mann@gmail.com>
---------------------------------------------------------------------------------------------------------------------]]
FSRTimer				= {}
local self				= FSRTimer
local _
local timer				= nil
local db				= nil
local mana				= UnitMana("player")
self.castTime			= GetTime()
local timerDuration		= 5.000
local timerVar			= timerDuration

local function Print(msg)
	DEFAULT_CHAT_FRAME:AddMessage("|cFFFFD700FSRT:|r "..msg)
end

function FSRTimer:Timer(elapsed)
	FSRTimer:UpdateTimer(timerVar)
	timerVar = timerVar - elapsed
	if (timerVar < 0) then
		FSRTimer:TimerReset() -- Stop timer
	end
end

function FSRTimer:UpdateTimer(curTime)
	-- Bar
	timer.FillFG:SetVertexColor(1, 0, 0, 1)
	timer.Fill:SetValue(curTime)
	-- Text
	curTime = string.format("%.1f", curTime)
	timer.Counter:SetText(curTime)
end

function FSRTimer:TimerReset(start, full)
	--local db = FSRTimerDB
	timer:SetScript("OnUpdate", nil) -- Turn off our timer update listener
	-- Reset our duration
	timerVar = timerDuration
	timer.Fill:SetValue(timerVar)
	local counterText = " "
	if db.counter.showZero == 1 then counterText = "0.0" end
	timer.Counter:SetText(counterText)
	-- Make the bar a bright green
	timer.FillFG:SetVertexColor(0, 1, 0, 1)
	-- Start listening, if our timer needs to start up again
	if start then
		timer:SetScript("OnUpdate", function(self, elapsed) FSRTimer:Timer(elapsed) end)
	end
end

function FSRTimer:CheckMana()
	local curMana = UnitMana("player")
	if curMana == UnitManaMax("player") then
		-- Stop the timer and make it a dark green if the player's mana is full.
		FSRTimer:TimerReset(false)
		FSRTimer.timer.FillFG:SetVertexColor(0, 0.25, 0, 1)
		return
	elseif (curMana < mana) and ((GetTime() - FSRTimer.castTime) < 0.5) then
		--[[
			Here is where we determine if the mana loss was due to a spell being cast.  Originally, the
			method used to trigger the timer was borrowed from RegenFu, but I found that it caused the
			timer to start more often than it should have.  While the window for the check is rather
			large here, at 0.5 seconds, smaller amounts prevented the timer from starting at most of my
			own latency ranges.  This gives it a bit more leeway, but is still precision enough to
			prevent most false starts.
		--]]
		FSRTimer:TimerReset(true)
	end
	
	
	-- Save a reference to our current mana to check against later.
	mana = curMana
end

function FSRTimer:UNIT_MANA(self, unit)
	if unit == "player" and UnitPowerType("player") == 0 then
		FSRTimer:CheckMana()
	end
end

function FSRTimer:UNIT_SPELLCAST_SUCCEEDED(self, unit)
	--[[
		This saves the time when a spell was cast.  This is used to find out if the player lost mana due to
		a spell cast, or if it was something else which caused the mana loss.  This is determined in the
		function :CheckMana().
	--]]
	if unit == "player" then
		FSRTimer.castTime = GetTime()
	end
end

function FSRTimer:VARIABLES_LOADED()
	FSRTimer:LoadSettings()
end

local function SavePosition()
	db.anchorPoint, _, db.anchorTo, db.xOffset, db.yOffset = timer:GetPoint()
	FSRTimer:LoadSettings()
end

local function UpdateLayout()
	--local db = FSRTimerDB
	local timer = FSRTimer.timer
	timer:SetWidth(db.width)
	timer:SetHeight(db.height)
	timer:ClearAllPoints()
	timer:SetPoint(db.anchorPoint, UIParent, db.anchorTo, db.xOffset, db.yOffset)
	timer.Fill:SetOrientation(db.orientation)
	local counterText = " "
	if db.counter.showZero == 1 then counterText = "0.0" end
	timer.Counter:SetText(counterText)
	timer.Counter:ClearAllPoints()
	timer.Counter:SetPoint(db.counter.anchorPoint, timer, db.counter.anchorTo, db.counter.xOffset, db.counter.yOffset)
	timer.Counter:SetAlpha(db.counter.alpha)
	if db.shown == 1 then timer:Show() else timer:Hide() end
end

function FSRTimer:LoadSettings()
	if not FSRTimerDB		then FSRTimerDB			= {}			end
	if not db				then db					= FSRTimerDB	end
	-- Timer settings
	if not db.shown			then db.shown			= 1				end
	if not db.locked		then db.locked			= 0				end
	if not db.width			then db.width			= 200			end
	if not db.height		then db.height			= 10			end
	if not db.anchorPoint	then db.anchorPoint		= "CENTER"		end
	if not db.anchorTo		then db.anchorTo 		= "CENTER"		end
	if not db.xOffset		then db.xOffset			= 0				end
	if not db.yOffset		then db.yOffset			= 0				end
	if not db.orientation	then db.orientation		= "Horizontal"	end
	-- Counter settings
	if not db.counter				then db.counter				= {}		end
	if not db.counter.showZero		then db.counter.showZero	= 1			end
	if not db.counter.anchorPoint	then db.counter.anchorPoint	= "RIGHT"	end
	if not db.counter.anchorTo		then db.counter.anchorTo	= "LEFT"	end
	if not db.counter.xOffset		then db.counter.xOffset		= -4		end
	if not db.counter.yOffset		then db.counter.yOffset		= 0			end
	if not db.counter.alpha			then db.counter.alpha		= 1			end
	
	if not timer.isMoving then
		timer:ClearAllPoints()
		timer:SetPoint(db.anchorPoint, UIParent, db.anchorTo, db.xOffset, db.yOffset)
	end
	
	UpdateLayout()
end

function FSRTimer:SlashHandler(cmd)
	local action, value, actionEnd, _
	--local db = FSRTimerDB
	cmd = strlower(cmd)
	
	if (cmd == "") then cmd = "menu" end
	
	-- Get command
	actionEnd, _, _, _ = string.find(cmd, " ") -- If there's a space, then we have an action and a value
	if actionEnd then -- Split if a space
		action = string.sub(cmd, 1, (actionEnd-1))
		value = string.sub(cmd, (actionEnd+1))
	else -- Just pass it on if not
		action = cmd
	end
	
	-- Show/Hide the timer
	if action == "show" then if db.shown == 1 then db.shown = 0 else db.shown = 1 end end
	-- Lock/Unlock the timer
	if action == "lock" then if db.locked == 1 then db.locked = 0 else db.locked = 1 end end
	-- Timer height/width
	if action == "width" and (tonumber(value) > 0) then db.width = tonumber(value) end
	if action == "height" and (tonumber(value) > 0) then db.height = tonumber(value) end
	if action == "horizontal" then db.orientation = "Horizontal" end
	if action == "vertical" then db.orientation = "Vertical" end
	-- Counter text
	if action == "counter" then
		local p, t, x, y = nil
		-- On/Off
		if value == "off" then	db.counter.alpha = 0 end
		if value == "on" then	db.counter.alpha = 1 end
		if value == "showzero" then if db.counter.showZero == 1 then db.counter.showZero = 0 else db.counter.showZero = 1 end end
		-- Position
		if value == "topleft"then		db.anchorPoint = "BOTTOMRIGHT";	db.counter.anchorTo = "TOPLEFT"			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "top" then			db.anchorPoint = "BOTTOM";		db.counter.anchorTo = "TOP";			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "topright" then		db.anchorPoint = "BOTTOMLEFT";	db.counter.anchorTo = "TOPRIGHT";		db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "left" then			db.anchorPoint = "RIGHT";		db.counter.anchorTo = "LEFT";			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "right" then		db.anchorPoint = "LEFT";		db.counter.anchorTo = "RIGHT";			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "bottomleft" then	db.anchorPoint = "TOPRIGHT";	db.counter.anchorTo = "BOTTOMLEFT";		db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "bottom" then		db.anchorPoint = "TOP";			db.counter.anchorTo = "BOTTOM";			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "bottomright" then	db.anchorPoint = "TOPLEFT";		db.counter.anchorTo = "BOTTOMRIGHT";	db.counter.xOffset =  0;	db.counter.yOffset =  0	end
		if value == "insideleft" then	db.anchorPoint = "LEFT";		db.counter.anchorTo = "LEFT";			db.counter.xOffset =  4;	db.counter.yOffset =  0	end
		if value == "insideright" then	db.anchorPoint = "RIGHT";		db.counter.anchorTo = "RIGHT";			db.counter.xOffset = -4;	db.counter.yOffset =  0	end
		if value == "center" then		db.anchorPoint = "CENTER";		db.counter.anchorTo = "CENTER";			db.counter.xOffset =  0;	db.counter.yOffset =  0	end
	end
	-- Menu
	if action == "menu" then
		Print("Help Text")
		Print("Hold shift while dragging to move the timer")
		Print("/fsrt show (toggle)")
		Print("/fsrt lock (toggle)")
		Print("/fsrt height|width <pixels>")
		Print("/fsrt horizontal|vertical (fill direction)")
		Print("/fsrt counter <on|off>")
		Print("/fsrt counter showzero (Toggles showing or hiding the counter when it reaches zero)")
		Print("/fsrt counter <position> (Relative to the timer bar: topleft, top, topright, left, right, bottomleft, bottom, bottomright, insideleft, insideright, or center)")
	end
	
	UpdateLayout()
end

do
	-- Container frame
	self.timer = CreateFrame("Button", "FSRTimer", UIParent)
	timer = self.timer
	-- Set up events
	timer:RegisterEvent("VARIABLES_LOADED")
	timer:RegisterEvent("UNIT_MANA")
	timer:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
	timer:SetScript("OnEvent", function (...) FSRTimer[event](...) end)
	-- Frame setup
	timer.isMoving = false -- Set to true if frame is in motion
	timer:SetWidth(200)
	timer:SetHeight(10)
	timer:SetFrameStrata("BACKGROUND")
	timer:SetClampedToScreen(true)
	timer:EnableMouse(true)
	timer:SetMovable(true)
	timer:SetScript("OnMouseDown", function ()
		if IsShiftKeyDown() then
			if db.locked == 0 then
				timer.isMoving = true
				timer:StartMoving()
			else
				Print("Frame is locked. /fsrt lock to toggle")
			end
		end
	end)
	timer:SetScript("OnMouseUp", function ()
		if timer.isMoving then
			timer:StopMovingOrSizing()
			timer.isMoving = false
			SavePosition()
		end
	end)
	timer:ClearAllPoints()
	timer:SetPoint("CENTER", UIParent, "CENTER", 0, 0)
		-- The fill bar
		timer.Fill = CreateFrame("StatusBar", "FSRTimerFill", timer)
		timer.Fill:SetOrientation("Horizontal")
		timer.Fill:SetAllPoints()
		timer.Fill:SetMinMaxValues(0, timerDuration)
		timer.Fill:SetStatusBarColor(1, 1, 1, 1)
			-- Fill bar's foreground
			timer.FillFG = timer.Fill:CreateTexture("FSRTimerFillFG")
			timer.FillFG:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
			timer.FillFG:SetVertexColor(0, 0.25, 0, 1)
			timer.FillFG:SetAllPoints()
			timer.Fill:SetStatusBarTexture(timer.FillFG)
			-- Fill bar's background
			timer.FillBG = timer.Fill:CreateTexture("FSRTimerFillBG", "BACKGROUND")
			timer.FillBG:SetTexture(0, 0, 0, 1)
			timer.FillBG:SetAllPoints()
		-- Counter text
		timer.Counter = timer.Fill:CreateFontString("FSRTimerCounter", "ARTWORK", "GameFontHighlight")
		timer.Counter:SetJustifyH("RIGHT")
		timer.Counter:SetPoint("RIGHT", timer.Fill, "LEFT", -4, 0)
		--timer.Counter:SetText("0.0")
		timer.Counter:SetAlpha(1)
	-- Show the frame
	timer:Show()
	-- Slash command
	SLASH_FSRTIMER1 = "/fsrtimer"
	SLASH_FSRTIMER2 = "/fsrt"
	SlashCmdList["FSRTIMER"] = function(cmd)
		FSRTimer:SlashHandler(cmd)
	end
	
	Print("loaded")
end

