--[[

Flightpath Puzzle League v1.2b by BlankDiploma

This mod calls itself "TetrisAttack" internally because, well, that's what it is. 
I just figured a different name might be good on the outside because have you SEEN Nintendo's legal department? Talk about rabid.

When I started this project, I had never used LUA or XML before, so it was kind of a learn-as-you-go thing, 
and since I come from a C++ background it was more of a learn-to-hate-LUA-as-you-go thing. But hey, it works, and that's the important thing!
With that said, if you're wondering why my coding method and use of various LUA features varies from function to function, it's because as I learned the language I
tried my hand at different ways of doing familiar things.

If you have any questions about the mod or anything you find in this file, feel free to contact me. My info is in the readme file.


~BlankDiploma

--]]

--OK, let's start out with tables that hold all sorts of game data. Here's the game field, textures in use, and their analogues for the multiplayer display.
--The game field is 6 columns by 24 rows: 12 visible and 12 buffer above the screen to hold falling trash.
local blockTable = {}
local blockTextures = {}
local blockTexturesMP = {}
local newBlockTable = {} --"New" blocks are the ones that rise up from below the screen.
local newBlockTextures = {}
local newBlockTexturesMP = {}
local cursorTexture = nil --Texture used for the cursor and win/lose images
local MPcursorTexture = nil
local numRows = 24
local numRowsVisible = 12

--How many blocks are currently disappearing, and which ones are they?
local numBlocksToClear = 0
local blocksToClear = {}

--How many blocks do we need to check this frame for potential matches, and where are they?
local numBlocksToCheck = 0
local blocksToCheck = {}

--How many blocks are currently falling, and where are they? I decided to do this one by columns because you only ever need to consider one column at a time, and this approach makes it faster.
--Or at least, that's the theory.
local fallingBlocksPerColumn = {0, 0, 0, 0, 0, 0}
local fallingTable = {{}, {}, {}, {}, {}, {}} -- One table for each column

--How many chunks of garbage do we have, and where are they?
local numGarbageBlocks = 0
local garbageTable = {}

--This stuff handles queueing up garbage so it doesn't fall all at once, and doesn't try to get inserted into blocks that are already full.
local numGarbageToCreate = 0
local garbageQueue = {}

--This handles the number of combos currently active. When no blocks are being cleared or falling, we know it's safe to resolve all open combos and turn them into trash for our opponent.
local numCombos = 0
local currentCombos = {}

--How many current animations do we have, and where are they? Used for both combos/chains and the powerup messages. Also stores the number of textures currently created for recycling purposes.
local numAnimations = 0
local maxComboAnims = 0
local currentAnims = {} -- ["x"], ["y"], ["texture"], ["fontstring"], ["value"], ["color"], ["timer"]

local editColor = 1 --Currently selected block color for creating puzzles.
local currentPuzzleIndex = 1 --Which puzzle do we have loaded?
local numMovesLeft = 0 --How many moves left in the current puzzle?
local numBlocksLeft = 0 --How many blocks are left to clear in the current puzzle?
local heartbeat = 0 --How long has it been since we heard from somebody in MP?

--These are all self-explanatory.
TetrisAttack_SavedPuzzles = {}
TetrisAttack_EndlessHighScores = {{}, {}, {}, {}, {}, {}, {}, {}, {}, {}}
TetrisAttack_FlightpathHighScores = {}
TetrisAttack_SavedSettings = {}

--Used to transmit race/sex data over MP so sound effects can be played for your opponent.
local raceSfxCodes = {"OrcMale", "OrcFemale",
					"Troll", 
					"Tauren", 
					"Scourge", 
					"BloodElf", 
					"Human", 
					"Gnome", 
					"NightElf", 
					"Draenai", 
					"Dwarf"}
local genderCodes = {"Male", "Female"}
local raceIndex = {["Orc"] = 0, ["Troll"] = 1, ["Tauren"] = 2, ["Undead"] = 3, ["Blood Elf"] = 4, ["Human"] = 5, ["Gnome"] = 6, ["Night Elf"] = 7, ["Draenai"] = 8, ["Dwarf"] = 9}
local raceGenderCodes = {["Orc1"] = 1, --Used to turn race into a number that can be used to index the sound effect table below.
					["Orc2"] = 2,
					["Troll1"] = 3,
					["Troll2"] = 4,
					["Tauren1"] = 5,
					["Tauren2"] = 6,
					["Undead1"] = 7,
					["Undead2"] = 8,
					["Blood Elf1"] = 9,
					["Blood Elf2"] = 10,
					["Human1"] = 11,
					["Human2"] = 12,
					["Dwarf1"] = 13,
					["Dwarf2"] = 14,
					["Night Elf1"] = 15,
					["Night Elf2"] = 16,
					["Gnome1"] = 17,
					["Gnome2"] = 18,
					["Draenei1"] = 19,
					["Draenei2"] = 20}
					
--This next bit used to be a bunch of local variables, but then I ran over the 200 local variable limit. Live and learn, I guess.
local soundEffects = {["blockfall"] = "Sound\\interface\\uMiniMapZoom.wav", 
["garbageSmall"] = "Sound\\Effects\\DeathImpacts\\mDeathImpactSmallDirtA.wav",
 ["garbageLarge"] = "Sound\\Effects\\DeathImpacts\\mDeathImpactGiantStoneA.wav",
 ["blockFlip"] = "Sound\\interface\\PickUp\\PutDownFoodGeneric.wav",
 ["blockClear"] = "", --No sound effect for this, I can't find one that isn't obnoxious.
 ["danger"] = "Sound\\Doodad\\NightmareBellOpen.wav",
 ["victory"] = "Sound\\Spells\\PVPVictory",
 ["defeat"] = "Sound\\Spells\\PVPVictory",
 ["challenge"] = "Sound\\interface\\iPlayerInviteA.wav",
 ["garbageClear"] = "Sound\\Spells\\BlizzardImpact1c.wav",
 ["ohShit"] = "Sound\\Creature\\Illidan\\BLACK_Illidan_04.wav",
 ["ding"] = "Sound\\Spells\\LevelUp.wav",
 ["bloodlust"] = "Sound\\Spells\\bloodlust_player_cast_head.wav",
 ["corruption"] = "Sound\\Spells\\Curse.wav",
 ["tranquility"] = "Sound\\Spells\\Tranquility.wav",
 ["baseCombo"] = "Sound\\Effects\\DeathImpacts\\mDeathImpactSmallGrassA.wav",
 ["puzzleWin"] = "",
 ["puzzleLose"] = "Sound\\Spells\\GhostlyStrikeImpact.wav"}

--Hand-picked sound effects for different combo levels.
local comboSounds = {{"Sound\\Character\\Orc\\OrcVocalMale\\OrcMaleYes02.wav", 
									"Sound\\Character\\Orc\\OrcVocalMale\\OrcMaleCongratulations01.wav", 
									"Sound\\Character\\Orc\\OrcVocalMale\\OrcMaleCheer01.wav", 
									"Sound\\Character\\Orc\\OrcMale\\OrcMaleLaugh01.wav"},
					{"Sound\\Character\\Orc\\OrcVocalFemale\\OrcFemaleYes02.wav", 
									"Sound\\Character\\Orc\\OrcVocalFemale\\OrcFemaleCongratulations01.wav", 
									"Sound\\Character\\Orc\\OrcVocalFemale\\OrcFemaleCheer01.wav", 
									"Sound\\Character\\Orc\\Female\\OrcFemaleLaugh01.wav"},
					{"Sound\\Character\\Troll\\TrollVocalMale\\TrollMaleYes04.wav", 
									"Sound\\Character\\Troll\\TrollVocalMale\\TrollMaleCongratulations02.wav", 
									"Sound\\Character\\Troll\\TrollVocalMale\\TrollMaleCheer01.wav", 
									"Sound\\Character\\Troll\\TrollMaleLaugh01.wav"},
					{"Sound\\Character\\Troll\\TrollVocalFemale\\TrollFemaleYes02.wav", 
									"Sound\\Character\\Troll\\TrollVocalFemale\\TrollFemaleCongratulations03.wav", 
									"Sound\\Character\\Troll\\TrollVocalFemale\\TrollFemaleCheer02.wav", 
									"Sound\\Character\\Troll\\TrollFemaleLaugh01.wav"},
					{"Sound\\Character\\Tauren\\TaurenVocalMale\\TaurenMaleYes01.wav", 
									"Sound\\Character\\Tauren\\TaurenVocalMale\\TaurenMaleCongratulations01.wav", 
									"Sound\\Character\\Tauren\\TaurenVocalMale\\TaurenMaleCheer01.wav", 
									"Sound\\Character\\Tauren\\TaurenMale\\TaurenMaleLaugh01.wav"},
					{"Sound\\Character\\Tauren\\TaurenVocalFemale\\TaurenFemaleYes02.wav", 
									"Sound\\Character\\Tauren\\TaurenVocalFemale\\TaurenFemaleCongratulations02.wav", 
									"Sound\\Character\\Tauren\\TaurenVocalFemale\\TaurenFemaleCheer01.wav", 
									"Sound\\Character\\Tauren\\Female\\TaurenFemaleLaugh01.wav"},
					{"Sound\\Character\\Scourge\\ScourgeVocalMale\\UndeadMaleYes01.wav", 
									"Sound\\Character\\Scourge\\ScourgeVocalMale\\UndeadMaleCongratulations02.wav", 
									"Sound\\Character\\Scourge\\ScourgeVocalMale\\UndeadMaleCheer01.wav", 
									"Sound\\Character\\Scourge\\ScourgeMale\\UndeadMaleLaugh01.wav"},
					{"Sound\\Character\\Scourge\\ScourgeVocalFemale\\UndeadFemaleYes01.wav", 
									"Sound\\Character\\Scourge\\ScourgeVocalFemale\\UndeadFemaleCongratulations03.wav", 
									"Sound\\Character\\Scourge\\ScourgeVocalFemale\\UndeadFemaleCheer02.wav", 
									"Sound\\Character\\Scourge\\ScourgeFemale\\UndeadFemaleLaugh01.wav"},
					{"Sound\\Character\\BloodElf\\BloodElfMaleNod02.wav", 
									"Sound\\Character\\BloodElf\\BloodElfMaleCongratulations02.wav", 
									"Sound\\Character\\BloodElf\\BloodElfMaleCheer01.wav", 
									"Sound\\Character\\BloodElf\\BloodElfMaleLaugh01.wav"},
					{"Sound\\Character\\BloodElf\\BloodElfFemaleYes03.wav", 
									"Sound\\Character\\BloodElf\\BloodElfFemaleCongratulations02.wav", 
									"Sound\\Character\\BloodElf\\BloodElfFemaleCheer01.wav", 
									"Sound\\Character\\BloodElf\\BloodElfFemaleLaugh01.wav"},
					{"Sound\\Character\\Human\\HumanVocalMale\\HumanMaleYes01.wav", 
									"Sound\\Character\\Human\\HumanVocalMale\\HumanMaleCongratulations03.wav", 
									"Sound\\Character\\Human\\HumanVocalMale\\HumanMaleCheer01.wav", 
									"Sound\\Character\\Human\\Male\\HumanMaleLaugh01.wav"},
					{"Sound\\Character\\Human\\HumanVocalFemale\\HumanFemaleYes01.wav", 
									"Sound\\Character\\Human\\HumanVocalFemale\\HumanFemaleCongratulations02.wav", 
									"Sound\\Character\\Human\\HumanVocalFemale\\HumanFemaleCheer02.wav", 
									"Sound\\Character\\Human\\Female\\HumanFemaleLaugh01.wav"},
					{"Sound\\Character\\Dwarf\\DwarfVocalMale\\DwarfMaleYes01.wav", 
									"Sound\\Character\\Dwarf\\DwarfVocalMale\\DwarfMaleCongratulations04.wav", 
									"Sound\\Character\\Dwarf\\DwarfVocalMale\\DwarfMaleCheer01.wav", 
									"Sound\\Character\\Dwarf\\DwarfMale\\DwarfMaleLaugh01.wav"},
					{"Sound\\Character\\Dwarf\\DwarfVocalFemale\\DwarfFemaleYes02.wav", 
									"Sound\\Character\\Dwarf\\DwarfVocalFemale\\DwarfFemaleCongratulations04.wav", 
									"Sound\\Character\\Dwarf\\DwarfVocalFemale\\DwarfFemaleCheer01.wav", 
									"Sound\\Character\\Dwarf\\DwarfFemale\\DwarfFemaleLaugh01.wav"},
					{"Sound\\Character\\NightElf\\NightElfVocalMale\\NightElfMaleYes01.wav", 
									"Sound\\Character\\NightElf\\NightElfVocalMale\\NightElfMaleCongratulations03.wav", 
									"Sound\\Character\\NightElf\\NightElfVocalMale\\NightElfMaleCheer01.wav", 
									"Sound\\Character\\NightElf\\NightElfMale\\NightElfMaleLaugh01.wav"},
					{"Sound\\Character\\NightElf\\NightElfVocalFemale\\NightElfFemaleYes01.wav", 
									"Sound\\Character\\NightElf\\NightElfVocalFemale\\NightElfFemaleCongratulations03.wav", 
									"Sound\\Character\\NightElf\\NightElfVocalFemale\\NightElfFemaleCheer02.wav", 
									"Sound\\Character\\NightElf\\NightElfFemale\\NightElfFemaleLaugh01.wav"},
					{"Sound\\Character\\Gnome\\GnomeVocalMale\\GnomeMaleCheer02.wav", 
									"Sound\\Character\\Gnome\\GnomeVocalMale\\GnomeMaleCongratulations02.wav", 
									"Sound\\Character\\Gnome\\GnomeVocalMale\\GnomeMaleCheer01.wav", 
									"Sound\\Character\\Gnome\\GnomeMaleLaugh01.wav"},
					{"Sound\\Character\\Gnome\\GnomeVocalFemale\\GnomeFemaleYes02.wav", 
									"Sound\\Character\\Gnome\\GnomeVocalFemale\\GnomeFemaleCongratulations01.wav", 
									"Sound\\Character\\Gnome\\GnomeVocalFemale\\GnomeFemaleCheer01.wav", 
									"Sound\\Character\\Gnome\\GnomeFemaleLaugh01.wav"},
					{"Sound\\Character\\Draenei\\DraeneiMaleYes01.wav", 
									"Sound\\Character\\Draenei\\DraeneiMaleCongratulations02.wav", 
									"Sound\\Character\\Draenei\\DraeneiMaleCheer01.wav", 
									"Sound\\Character\\Draenei\\DraeneiMaleLaugh01.wav"},
					{"Sound\\Character\\Draenei\\DraeneiFemaleYes01.wav", 
									"Sound\\Character\\Draenei\\DraeneiFemaleCongratulations03.wav", 
									"Sound\\Character\\Draenei\\DraeneiFemaleCheer01.wav", 
									"Sound\\Character\\Draenei\\DraeneiFemaleLaugh01.wav"}}
--Just in case we try to access a sound that doesn't exist.
comboSounds[0] = {"","","",""}
local myRaceGender = 0 --Player's race/gender index.
local opponentRaceGender = 0

local blockColor = {{1, 1, 0}, {1, 0, 0}, {.2, .2, 1}, {0, 1, 0}, {1, 0, 1}, {0, 1, 1}, {1, 1, 1}, {1, 1, 1}, {.5, .5, .5}} --Vertex colors for blocks.
local difficultyColors = {{0, 1, 0}, {1, 1, 0}, {1, 0, 0}} --Text colors for easy/normal/hard
local qualityColors = {{.62, .62, .62}, {1, 1, 1}, {.12, 1, 0}, {0, .5, 1}, {.7, .28, .97}, {.94, .47, 0}, {1, 0, 0}} --WoW's item-quality colors, used for combos.
local comboAnimSpeed = 64 --How fast combos animate in pixels/s
local difficultyText = {"Easy", "Normal", "Hard"} --Changes an index into something readable.
local sinValues = {} --Stores precomputer sin values to avoid calling slow function during onupdate.

--Safety values in case something goes wrong
blockColor[0] = {0,0,0}
difficultyColors[0] = {1,1,1}
difficultyText[0] = "Error"

--OK, from here on out a lot of variables have self-explanatory names, so I'm not going to comment anything that's obvious.
local blockTypes = 6 --Current number of colors in play.
local isPaused = 0
local preGameTimer = -1 --Countdown before the game starts.
local frameUpdates = 0 --Should we be updating only one frame? Used for debugging.
local playerScore = 0

--Multiplayer stuff
local multiplayerUpdateInterval = .25 --How often do we send data in multiplayer?
local multiplayerUpdateTimer = 0 --How much time has passed since last update.
local MPgarbageUpdateTimer = 0 --How much time has passed since we last sent garbage. Garbage-sending interval is 3x the multiplayer data interval.
local currentOpponent = ""
local lastOpponent = ""
local currentOpponentSpeed = 0
local currentOpponentDifficulty = 0
local MPAvgLatency = 0	--Roughly estimated average latency between clients.
local MPfieldMoving = true --Is the opponent's field moving upwards, according to their last transmitted data?
local MPgarbageCode = "" --Garbage to send in multiplayer, encoded as a string.

local chatMsgFieldPrefix = "TAMPFLD"	--Prefixes for multiplayer communication. They're kind of long because I wanted to make them unique and also, you know, meaningful.
local chatMsgBouncePrefix = "TAMPBNC"
local chatMsgGarbagePrefix = "TAMPGRB"
local chatMsgHandshakePrefix = "TAMPHND"
local chatMsgGreetingPrefix = "TAMPHI"
local chatMsgPingPrefix = "TAMPPNG"
local chatMsgScorePrefix = "TAMPSCR"
local chatMsgScoreReqPrefix = "TAMPREQ"
local chatMsgChallengePrefix = "TAMPCHL"

--This next table holds conversions from a number of blocks to a set of garbage to drop on the opponent.
local MPsmallGarbageCodes = {"03", "04", "05", "0303", "0304", "0404", "030303", "030304", "030404", "03030303", "03030304", "03030404", "03030405", "03040405", "03040505", "030303040405", "030304040505", "030404040505", "030404050505"}
local speedyShift = 0 --If the player hit the Bump button, manually raise the field one column at a high speed and then stop.

--Game speed info. Inside each index, the values are: [1]Speed the field moves at, [2]Block falling speed, [3]How long it takes for blocks to disappear, and [4]the speed at which the death buffer drains.
local gameSpeeds = {{5, 512, 1.5, 1},
				{6, 512, 1.5, 1},
				{7, 512, 1.5, 1},
				{8, 512, 1.5, 1},
				{9, 512, 1.25, 2},
				{10, 512, 1.25, 2},
				{12, 512, 1.25, 2},
				{14, 640, 1, 2},
				{16, 640, 1, 2},
				{18, 640, 1, 2},
				{20, 640, 1, 2},
				{22, 640, 1, 2},
				{24, 640, 1, 2},
				{26, 640, 1, 2},
				{28, 640, 1, 2},
				{30, 768, .75, 2.5},
				{32, 768, .75, 2.5},
				{34, 768, .75, 2.5},
				{36, 768, .75, 2.5},
				{38, 768, .75, 2.5},
				{40, 768, .75, 2.5},
				{42, 768, .75, 2.5},
				{44, 768, .75, 2.5},
				{46, 768, .75, 2.5},
				{48, 1024, .75, 3},
				{50, 1024, .75, 3},
				{52, 1024, .5, 3},
				{54, 1024, .5, 3},
				{56, 1024, .5, 3},
				{58, 1024, .5, 3}}
local difficulty = {{1000, 10, 5}, {750, 8, 6}, {300, 4, 6}} --Points it takes for a speed increase, max death buffer, number of colors in play

local fieldSpeed = 1
local currentSpeed = 1
local startingSpeed = 1
local blockClearDelay = 1.5 --These variables hold the currently-selected difficulty settings. They are referenced so often that I didn't want to have to access a table every time, since I was unfamiliar with how much of a performance hit LUA takes with table access.
local fallingDelay = .25
local blockFallSpeed = 8
local bufferDrainSpeed = 1
local maxDeathBuffer = 6
local currentDeathBuffer = 6

local garbageTextureCodes = {"z", "x", "c", --Letters used to transmit the shape of garbage blocks to our opponent.
							"s", "d", "f",
							"e", "r", "t",
							"v", "g", "y"}
local garbageTextureDecodes = {["z"] = "LB", ["x"] = "B", ["c"] = "RB", --Inputting a letter from the above table gives us the proper texture filename to append.
							["s"] = "L", ["d"] = "", ["f"] = "R",
							["e"] = "LT", ["r"] = "T", ["t"] = "RT",
							["v"] = "LTB", ["g"] = "TB", ["y"] = "RTB"}
local currentDifficulty = 2

local lastBlockClearedRow = 0	--What was the last block we cleared? Used for combos.
local lastBlockClearedColumn = 0

local lastComboRow = 0	--Where was the last combo we placed? Used to prevent overlapping.
local lastComboColumn = 0

local basePointsPerSpeedIncrease = 500	--How many points it takes to increase speed, and how close the player is to doing so. basePoints is lowered every time you get a combo in endless mode to add a sort of auto-adapting difficulty.
local pointsTillNextSpeedIncrease = 500

local totalBlocksThisFrame = 0 --How many blocks we cleared this frame.

local freezeTime = 0 --How much freezetime the player has generated through combos.
local colorizeBlocks = 1
local useClassIcons = 1
local gameInProgress = 0		--1 = endless, 2 = puzzle, 3 = MP

local allowInput = false --Allow the player to flip blocks?
local cursorX = 3
local cursorY = 8
local MPcursorX = 3
local MPcursorY = 8
local fieldOffset = 0 -- increases until 31, then resets to zero and bumps the whole field up a tile.
local MPfieldOffset = 0
local lastHandshakeTime = 0 --Used to compute average latency.
local dangerPulseTimer = 0	--Used to time the pulsing of the background.
local opponentDangerPulseTimer = 0
local inDanger = 0	--Has the screen been filled recently?
local friendHasFPL = {}	--List of friends and their status as far as mod version goes.
local miniMapDrag = false --Are we dragging the minimap?

--These are local version of saved variables. More specifically, these are the default values.
local playSounds = 1
local enableMouse = 1
local fascistKeyboard = 1
local enableMinimap = 1
local useClassIcons = 0
local combatPanic = 1
local autoPause = 1
local autoOpenFlightpath = 1
local upKey = "W"
local downKey = "S"
local leftKey = "A"
local rightKey = "D"
local flipKey = "ENTER"
local bumpKey = "Q"
local pauseKey = "X"
local useFriendsList = 1
local ignoreChallenges = false
local verbose = 1 --Output multiplayer messages?

local keyToBind = 0 --Are we binding a key?
local keysInOrder = {"upKey", "downKey", "leftKey", "rightKey", "flipKey", "bumpKey", "pauseKey"} --Names of keys in order.
local keyDescsInOrder = {"Press Up Key", "Press Down Key", "Press Left Key", "Press Right Key", "Press Flip Key", "Press Bump Key", "Press Pause Key"} --Prompts.
local selectedFriend = "" --Which friend is selected on the list?
local pregameCleanup = 0 --Are we currently doing ultraspeed processing to remove matches and falling blocks before the screen fades in?
local multiplayerSeed = 0 --Which random seed to use for the board. Bit of a misnomer, since this can also be used for flightpath games.

--Variables which track the fancy-pants animations that play when a game ends.
local deathAnim = 0
local deathAnimOffset = 0
local deathAnimVelocity = 0
local deathAnimAccel = 0
local opponentDeathAnim = 0
local opponentDeathAnimOffset = 0
local opponentDeathAnimVelocity = 0
local opponentDeathAnimAccel = 0

--YOU ARE NOT PREPARED
local ohShitTimer = 0

--Currently-displayed high score lists.
local topFriendScores = {{["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}}
local endlessScoreDisplay = {{["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}, {["name"] = "name", ["score"] = 0}}

--Where we're flying from and to.
local srcName = ""
local destName = ""

--Handles cooldown on initiating another friends list check.
local lastFriendCheck = 0
local friendCheckCooldown = 15

local currentPlayerName = "genericname"

--Handles the appearance of ?-blocks.
local bonusBlockChance = 0
local totalBonusThisFrame = 0
local totalBonusOnField = 0

--Send a powerup notification?
local sendSpecial = "0"

--Timers for powerups. I should have done this with just one timer, but that wouldn't allow multiple ones to be active at the same time.
local tranqTimer = 0
local bloodTimer = 0
local corrTimer = 0
local opponentTranqTimer = 0
local opponentBloodTimer = 0
local opponentCorrTimer = 0

--Are we playing a flightpath game?
local taxiGameInProgress = false

--Combo notifications to send.
local animsToSend = ""

local currentDemo = {} --Current demo being played/recorded.
local currentDemoPos = 0 --Current action in current demo.
local demoStart = 0 --Time the current demo started.
local demoPlay = false --Are we playing a demo?
local demoRecord = false --Are we recording a demo?

--Handles turning the cursor into a lose/win message and dropping it from above the screen.
local cursorAnimPos = 0
local cursorAnimVelocity = 0

--Hard-coded number of tutorials and puzzles. Yeah, yeah, I know.
local numTutorials = 16
local numOfficialPuzzles = 35

--Are we currently animating either background? Had to add this to handle multiple colors trying to work at the same time.
local opponentBackgroundPulse = false
local backgroundPulse = false

--Variable added to make sure corruption blocks don't make a tower and kill you instantly.
local corruptionIndex = 1

--Cooldown on issuing/receiving challenges.
local challengeTimer = 0

--Have we created the raid frame icons yet?
local raidInitialized = false

--How long until we're willing to broadcast our own name again?
local raidRefreshCooldown = 0 

--Has a raid member been added since we last let everybody know we're alive?
local raidMemberAdded = false 
local numRaidMembers = 0

--Maintain a local indicator of how far the friends list is scrolled down so we can tell if it's changed.
--local friendListOffset = 1

local currentVersion = 11

--Define hooked functions.
local TetrisAttack_TakeTaxiNode_Original
local TetrisAttack_ClickFriendButton_Original

BINDING_HEADER_TAHEADER = "Flightpath Puzzle League"
BINDING_NAME_TAOPEN = "Toggle the game window"
BINDING_NAME_TALEFT = "Move cursor left\n|cffbfbfbf(replaced by in-mod settings)|r"
BINDING_NAME_TARIGHT = "Move cursor right\n|cffbfbfbf(replaced by in-mod settings)|r"
BINDING_NAME_TAUP = "Move cursor up\n|cffbfbfbf(replaced by in-mod settings)|r"
BINDING_NAME_TADOWN = "Move cursor down\n|cffbfbfbf(replaced by in-mod settings)|r"
BINDING_NAME_TAFLIP = "Flip blocks at cursor\n|cffbfbfbf(replaced by in-mod settings)|r"
BINDING_NAME_TAPAUSE = "Pause\n|cffbfbfbf(replaced by in-mod settings)|r"

--[[
		
		It's FUNCTION TIME!
		
		These few functions are local either because I wanted them to be secured from calling via the in-game /script,
		or because I've read a bit about how global scope is slower, and some of these functions are called a ridiculous amount.
		
--]]

--Is this block currently disappearing?
local function TetrisAttack_BlockIsBeingCleared(row, column)
	for block = 1, numBlocksToClear do
		if  ((blocksToClear[block]["row"] == row) and (blocksToClear[block]["column"] == column) and (blocksToClear[block]["delay"] > 0)) then return true end
	end
	return false
end

--Is this block in the pre-fall phase where it still has a physical presence on the board?
local function TetrisAttack_BlockIsSuspended(row, column)
	for block = 1, fallingBlocksPerColumn[column] do
		if  (fallingTable[column][block]["row"] == row and fallingTable[column][block]["column"] == column and fallingTable[column][block]["delay"] > 0) then return true end
	end
	for block = 1, numGarbageBlocks do
		if  (row >= garbageTable[block]["bottom"] and
			row <= garbageTable[block]["top"] and
			column >= garbageTable[block]["left"] and
			column <= garbageTable[block]["right"]) then return true end
	end
	return false
end

--What's the lowest row with nothing in it, not counting falling blocks?
local function TetrisAttack_FindLowestEmptyRow()
	local row = numRows
	local sum = 0
	while (sum == 0 and row > 0) do
		row = row - 1
		for i = 1, 6 do
			sum = sum + ((blockTable[row][i] >= 0 and blockTable[row][i]) or 0)
		end
	end
	return (row + 1)
end

--Is a given section of a row empty? For garbage collisions.
local function TetrisAttack_isAreaEmpty(row, left, right)
	local sum = 0
	for i = left, right do
		sum = sum + ((blockTable[row][i] >= 0 and blockTable[row][i]) or 0)
	end
	if (sum > 0) then return false else return true end
end

--Does the given block match anything? If passed a combo index, this function will continue the current combo-in-progress.
--Also handles updating of combo values and queueing garbage to be sent for combos greater than x1, as well as the appropriate bonus points.
--Awarding points for simply clearing blocks is done in the UpdateGame function.
local function TetrisAttack_CheckForMatches(row, column, combo)
	local clearingBonus = false
	if (TetrisAttack_BlockIsSuspended(row, column)) then return end
	combo = combo or 0
	local blockColor = blockTable[row][column]
	if (blockColor == 0 or blockColor == 7 or blockColor == 9) then return end
	if (blockColor == 8) then clearingBonus = true end
	local vertmatches = {}
	local vertindex = 0
	local horizmatches = {}
	local horizindex = 0
	local startingBlockCleared = 0
	local totalBlocks = 0
	if (column > 1) then
		local currentColumn = column-1	--Search to left
	
		while(currentColumn > 0 and 
				blockTable[row][currentColumn] == blockColor and 
				not TetrisAttack_BlockIsBeingCleared(row, currentColumn) and
				not TetrisAttack_BlockIsSuspended(row, currentColumn)) do
			horizindex = horizindex + 1
			horizmatches[horizindex] = currentColumn
			currentColumn = currentColumn - 1
		end
	end

	if (row < numRows) then
		local currentRow = row+1	--Search up
	
		while(currentRow < 13 and 
				blockTable[currentRow][column] == blockColor and 
				not TetrisAttack_BlockIsBeingCleared(currentRow, column) and
				not TetrisAttack_BlockIsSuspended(currentRow, column)) do
			vertindex = vertindex + 1
			vertmatches[vertindex] = currentRow
			currentRow = currentRow + 1
		end
	end
	
	vertindex = vertindex + 1
	horizindex = horizindex + 1
	horizmatches[horizindex] = column
	vertmatches[vertindex] = row

	if (column < 6) then
		local currentColumn = column+1	--Search to right
	
		while(currentColumn < 7 and
				blockTable[row][currentColumn] == blockColor and 
				not TetrisAttack_BlockIsBeingCleared(row, currentColumn) and
				not TetrisAttack_BlockIsSuspended(row, currentColumn)) do
			horizindex = horizindex + 1
			horizmatches[horizindex] = currentColumn
			currentColumn = currentColumn + 1
		end
	end

	if (row > 1) then
		local currentRow = row-1	--Search down
	
		while(currentRow > 0 and 
				blockTable[currentRow][column] == blockColor and 
				not TetrisAttack_BlockIsBeingCleared(currentRow, column) and
				not TetrisAttack_BlockIsSuspended(currentRow, column)) do
			vertindex = vertindex + 1
			vertmatches[vertindex] = currentRow
			currentRow = currentRow - 1
		end
	end
	
	if (horizindex > 2) then
		totalBlocks = totalBlocks + horizindex
	end
	if (vertindex > 2) then
		totalBlocks = totalBlocks + vertindex
	end
	if (horizindex > 2 and vertindex > 2) then
		totalBlocks = totalBlocks - 1
	end
	
	if (totalBlocks > 2 and pregameCleanup == 0) then
		lastBlockClearedRow = row
		lastBlockClearedColumn = column
		if (combo == 0) then
			numCombos = numCombos + 1
			currentCombos[numCombos] = 0
			combo = numCombos
		end
		if (gameInProgress == 2) then
			numBlocksLeft = numBlocksLeft - totalBlocks
		end
		totalBlocksThisFrame = totalBlocksThisFrame + totalBlocks
		if (clearingBonus) then totalBonusThisFrame = totalBonusThisFrame+totalBlocks end
		local addedScore = totalBlocks*10
		local bonusTime = 0
		if (currentCombos[combo] >= 0) then -- Increase combo!
			currentCombos[combo] = currentCombos[combo] + 1
			if (currentCombos[combo] > 1) then
				local color = currentCombos[combo] - 1
				if (playSounds == 1 and color == 1) then 
					PlaySoundFile(soundEffects["baseCombo"])
				elseif (playSounds == 1 and color < 6) then 
					PlaySoundFile(comboSounds[myRaceGender][color-1]) 
				end
				if (color>6) then color = 6 end
				local text = "x" .. currentCombos[combo]
				TetrisAttack_CreateComboAnim(lastBlockClearedRow, lastBlockClearedColumn, text, color)
				local mpText = string.format("%02d",lastBlockClearedRow) .. lastBlockClearedColumn
				mpText = mpText .. color .. string.format("%02d",currentCombos[combo])
				animsToSend = animsToSend .. mpText
				bonusTime = blockClearDelay
				addedScore = totalBlocks*currentCombos[combo]*(10+currentSpeed)
				if (TetrisAttack_FindLowestEmptyRow() > numRowsVisible - 2) then bonusTime = bonusTime * 1.5 end
			end
			currentCombos[combo] = -currentCombos[combo] -- Mark this combo as frozen until next frame, so clearing 6 blocks in a single frame doesn't give you a 2x combo.
		end
		if (gameInProgress ~= 2) then --Don't give freezetime or points in puzzle mode.
			if (bonusTime > 0) then
				freezeTime = freezeTime + bonusTime
				TetrisAttackSpeedText:SetText("Freeze Time:")
				TetrisAttackSpeedDisplay:SetText(math.floor(freezeTime))
			end
			playerScore = playerScore + addedScore
			pointsTillNextSpeedIncrease = pointsTillNextSpeedIncrease - addedScore
		end
	end
	if horizindex > 2 then
		for k=1, horizindex do
			--TetrisAttack_ClearGarbage(horizmatches[k], combo)
			numBlocksToClear = numBlocksToClear + 1
			if (not blocksToClear[numBlocksToClear]) then
				blocksToClear[numBlocksToClear] = {["row"] = row, 
												["column"] = horizmatches[k],
												["combo"] = combo,
												["delay"] = blockClearDelay}
			else
				blocksToClear[numBlocksToClear]["row"] = row
				blocksToClear[numBlocksToClear]["column"] = horizmatches[k]
				blocksToClear[numBlocksToClear]["combo"] = combo
				blocksToClear[numBlocksToClear]["delay"] = blockClearDelay
			end
			TetrisAttack_TriggerGarbage(row, horizmatches[k], combo)
			if (horizmatches[k] == column) then  startingBlockCleared = 1 end
		end
	end
	if vertindex > 2 then
		for k=1, vertindex do
			--TetrisAttack_ClearGarbage(vertmatches[k], combo)
			if (vertmatches[k] ~= row or startingBlockCleared == 0) then
				numBlocksToClear = numBlocksToClear + 1
			if (not blocksToClear[numBlocksToClear]) then
				blocksToClear[numBlocksToClear] = {["row"] = vertmatches[k], 
													["column"] = column,
													["combo"] = combo,
													["delay"] = blockClearDelay}
			else
				blocksToClear[numBlocksToClear]["row"] = vertmatches[k]
				blocksToClear[numBlocksToClear]["column"] = column
				blocksToClear[numBlocksToClear]["combo"] = combo
				blocksToClear[numBlocksToClear]["delay"] = blockClearDelay
			end
				TetrisAttack_TriggerGarbage(vertmatches[k], column, combo)
			end
		end
	end
end

--Given a garbage chunk index and a table, count up all the garbage chunks that are connected to the original in some way. Recursive.
--Returns the total number of chunks to clear.
local function TetrisAttack_CountGarbage(row, column, endTable)
	local top, bottom, left, right = 0,0,0,0
	local numBlocks = 0
	for i = 1, numGarbageBlocks do
		top = garbageTable[i]["top"]
		bottom = garbageTable[i]["bottom"]
		left = garbageTable[i]["left"]
		right = garbageTable[i]["right"]
		if (garbageTable[i]["index"] == 0 and garbageTable[i]["offset"] == 0 and bottom <= numRowsVisible) then
			if ((row-1 >= bottom and row-1 <= top and column >= left and column <= right) or
				(row+1 >= bottom and row+1 <= top and column >= left and column <= right) or
				(row >= bottom and row <= top and column-1 >= left and column-1 <= right) or
				(row >= bottom and row <= top and column+1 >= left and column+1 <= right)) then
				tinsert(endTable, i)
				numBlocks = numBlocks + ((top-bottom)+1)*((right-left)+1)
				garbageTable[i]["index"] = 1
				for y = bottom, top do
					for x = left, right do
						numBlocks = numBlocks + TetrisAttack_CountGarbage(y, x, endTable)
					end
				end
			end
		end
	end
	return numBlocks
end

--Bump the player field up a bit, depending on current framerate. Also handle death buffer and return whether or not the field moved at all.
local function TetrisAttack_MovePlayerField(elapsed)
	local fieldMoved = false
	if (TetrisAttack_FindLowestEmptyRow() > numRowsVisible) then
		currentDeathBuffer = currentDeathBuffer - elapsed * bufferDrainSpeed
		if (currentDeathBuffer < 0) then
			TetrisAttack_TriggerDeathAnim()
		end
	else
		if (currentDeathBuffer < maxDeathBuffer) then
			currentDeathBuffer = currentDeathBuffer + (elapsed * (1/(bufferDrainSpeed * 3)))
		end
		if (speedyShift == 1) then
			fieldOffset = fieldOffset + 90/GetFramerate()
		else
			fieldOffset = fieldOffset + (fieldSpeed/GetFramerate() * (((bloodTimer > 0) and 1.1) or ((tranqTimer > 0) and .8) or 1))
		end
		fieldMoved = true
		if fieldOffset >= 32 then
			while (fieldOffset > 32) do
				TetrisAttack_bumpField()
				fieldOffset = fieldOffset - 32
			end
		end
		for column=1, 6 do
			--newBlockTextures[column]:SetTexCoord(0.1,0.9,0.1,((fieldOffset+1)/32) * 0.9)
			newBlockTextures[column]:SetTexCoord(0,1,0,((fieldOffset+1)/32))
			newBlockTextures[column]:SetHeight(fieldOffset+1)
		end
	end
	return fieldMoved
end

--Move a block texture back to its rightful place.
local function TetrisAttack_ResetBlockPos(row, column)
	if (row > 0 and row < 13 and column > 0 and column < 7) then
		blockTextures[row][column]:ClearAllPoints()
		blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) + 10 + 32)
	end
end

--Make sure a block is using the correct texture and color.
local function TetrisAttack_UpdateBlock(row, column, colorIndex)
	if (row > 0 and row < 13 and column > 0 and column < 7) then
		colorIndex = colorIndex or blockTable[row][column]
		if (colorIndex < 0) then colorIndex = 0 end
		
		if (colorIndex == 8) then
			blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockbonus")
			blockTextures[row][column]:SetVertexColor(1,1,1,1)
		elseif (colorIndex == 0 or useClassIcons == 1) then
			blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
			blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
		elseif (colorizeBlocks == 1) then
			blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
			blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
		end
		if (pregameCleanup == 1) then --If we're doing pregame cleanup, mirror all changes we make in the multiplayer display as well, to simulate the correct starting blocks.
			if (colorIndex == 0 or useClassIcons == 1) then
				blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
				blockTexturesMP[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
			elseif (colorizeBlocks == 1) then
				blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
				blockTexturesMP[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
			end
		end
	end
end

--Handle the bestowing of the various powerups in multiplayer.
local function TetrisAttack_Powerup()
	MPgarbageCode = MPgarbageCode .. "06"
	totalBonusOnField = totalBonusOnField - totalBonusThisFrame
	if (totalBonusThisFrame == 4) then
		--Bloodlust
		backgroundPulse = true
		if (playSounds == 1) then PlaySoundFile(soundEffects["bloodlust"]) end
		bloodTimer = bloodTimer + 10
		sendSpecial = "1"
		TetrisAttack_CreateComboAnim(8, 3.5, "Bloodlust!", 7, 0)
	elseif (totalBonusThisFrame == 5) then
		--Corruption
		if (playSounds == 1) then PlaySoundFile(soundEffects["corruption"]) end
		opponentCorrTimer = opponentCorrTimer + (20 - MPAvgLatency)
		sendSpecial = "2"
		TetrisAttack_CreateComboAnim(8, 10, "Corruption!", 5, 0)
		opponentBackgroundPulse = true
	elseif (totalBonusThisFrame >= 6) then
		--Tranquility
		backgroundPulse = true
		if (playSounds == 1) then PlaySoundFile(soundEffects["tranquility"]) end
		sendSpecial = "3"
		tranqTimer = tranqTimer + 15
		TetrisAttack_ReplaceBlocks(5, 4)
		TetrisAttack_ReplaceBlocks(6, 3)
		blockTypes = 4
		TetrisAttack_CreateComboAnim(8, 3.5, "Tranquility!", 3, 0)
	end
end

--Check all timers to see if they've run their course, then reset them and do stuff based on which timer it was.
local function TetrisAttack_CheckTimers(elapsed)
	if (challengeTimer > 0) then 
		challengeTimer = challengeTimer - elapsed
		if (challengeTimer <= 0) then
			challengeTimer = 0
			TetrisAttackChallengeAcceptFrame:Hide()
			if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: " .. currentOpponent .. "'s challenge expired...", 1, 1, 1) end
			currentOpponent = ""
		end
	end
	if (raidRefreshCooldown > 0) then 
		raidRefreshCooldown = raidRefreshCooldown - elapsed
		if (raidRefreshCooldown <= 0 and raidMemberAdded) then
			raidRefreshCooldown = 0
			TetrisAttack_SendRaidGreeting()
		end
	end
	if (preGameTimer > 0) then
		preGameTimer = preGameTimer - elapsed
		if (preGameTimer < 0) then
			TetrisAttackCommentText:SetText("GO!")
			TetrisAttackCommentText:SetTextColor(0.2, 1.0, 0.2, 1.0)
			allowInput = true
		elseif (preGameTimer < 3) then
			TetrisAttackCommentText:SetText(math.ceil(preGameTimer))
			TetrisAttackField:SetAlpha(1)
			TetrisAttackOpponentField:SetAlpha(1)
		elseif (preGameTimer < 4) then
			TetrisAttackField:SetAlpha(4-preGameTimer)
			TetrisAttackOpponentField:SetAlpha(4-preGameTimer)
		end
		if (pregameCleanup == 0) then return false end --If we're still cleaning up, return info to the calling function.
	end
	if (preGameTimer > -1) then
		preGameTimer = preGameTimer - elapsed
		if (preGameTimer <= -1) then
			TetrisAttackCommentText:SetText("")
		end
	end
	if (tranqTimer > 0) then
		dangerPulseTimer = dangerPulseTimer + 1
		tranqTimer = tranqTimer - elapsed
		if (tranqTimer <= 0) then blockTypes = difficulty[currentDifficulty][3] end --Reset number of block colors if the timer's up.
		TetrisAttackFieldBackground:SetTexture(0, 0.8, 0, sinValues[dangerPulseTimer])
		if (dangerPulseTimer >= 120) then dangerPulseTimer = 0 end
	elseif (bloodTimer > 0) then
		dangerPulseTimer = dangerPulseTimer + 1
		bloodTimer = bloodTimer - elapsed
		TetrisAttackFieldBackground:SetTexture(0.8, 0, 0, sinValues[dangerPulseTimer])
		if (dangerPulseTimer >= 120) then dangerPulseTimer = 0 end
	elseif (corrTimer > 0) then
		dangerPulseTimer = dangerPulseTimer + 1
		local corrTick = math.floor(corrTimer)
		corrTimer = corrTimer - elapsed
		if ((math.floor(corrTimer) ~= corrTick) and (math.floor(corrTimer) % 2 == 0)) then TetrisAttack_QueueGarbage(2) end
		TetrisAttackFieldBackground:SetTexture(0.6, 0, 0.8, sinValues[dangerPulseTimer])
		if (dangerPulseTimer >= 120) then dangerPulseTimer = 0 end
	elseif (inDanger == 1) then
		dangerPulseTimer = dangerPulseTimer + 1
		TetrisAttackFieldBackground:SetTexture(0.8, 0, 0, sinValues[dangerPulseTimer])
		if (dangerPulseTimer >= 120) then dangerPulseTimer = 0 end
	end
	if ((tranqTimer <= 0) and (bloodTimer <= 0) and (corrTimer <= 0) and (inDanger == 0) and backgroundPulse) then --Only reset the background if nothing is flashing.
		TetrisAttackFieldBackground:SetTexture(.05, .05, .05, .75)
		backgroundPulse = false
	end
	return true
end

--The big one. This baby calls just about every other function in here, once per frame. I had to split it up into multiple sub-functions because of the 60-upvalue limit. Oh well.
local function TetrisAttack_UpdateGame(elapsed)
	local pauseField
	local fieldMoved = false
	if gameInProgress > 0 and (isPaused == 0 or frameUpdates == 1) then
		if (demoPlay) then	--Handle playing the current demo.
			if (currentDemo[currentDemoPos*2] <= GetTime()-demoStart) then
				TetrisAttack_DecodeDemoAction(currentDemo[currentDemoPos*2-1])
				currentDemoPos = currentDemoPos+1
				if (currentDemo[currentDemoPos*2-1] == 0) then 
					demoPlay = false; currentDemoPos = 1 
					TetrisAttack1:EnableKeyboard(fascistKeyboard)
				end
			end
		end
		TetrisAttack_UpdateComboAnims(elapsed)
		frameUpdates = 0 --If we updated one frame while paused, reset it until manually triggered again.
		for theCombo = 1, numCombos do
			if currentCombos[theCombo] < 0 then currentCombos[theCombo] = -currentCombos[theCombo] end --Unfreeze all combos
		end
		if (numBlocksToCheck >= 1) then
			for i = 1, numBlocksToCheck do
				TetrisAttack_CheckForMatches(blocksToCheck[i][1], blocksToCheck[i][2], blocksToCheck[i][3]) --Check all blocks for matches that need it
			end
			numBlocksToCheck = 0
		end
		if (gameInProgress ~= 2) then
			for i = 1,numGarbageBlocks do
				if (garbageTable[i]["delay"] ~= 0 and garbageTable[i]["bottom"] > 0) then --If we have active garbage chunks that are clearing, don't move the field.
					pauseField = true
				end
			end
		end
		if (numBlocksToClear < 1 and not pauseField) then
			if (fallingBlocksPerColumn[1] + 
					fallingBlocksPerColumn[2] + 
					fallingBlocksPerColumn[3] + 
					fallingBlocksPerColumn[4] + 
					fallingBlocksPerColumn[5] + 
					fallingBlocksPerColumn[6] == 0) then --At this point, we know there's no falling blocks and no clearing blocks, so all combos can be deleted.
				TetrisAttack_FinishCombos()
				if (pregameCleanup == 1) then --Unfreeze sound effects and set the speed to the correct values.
					pregameCleanup = 0
					playSounds = -playSounds
					TetrisAttack_ChangeGameSpeed(currentSpeed)
					return
				end
				local topRow = TetrisAttack_FindLowestEmptyRow()
				if (topRow > numRowsVisible-2 and inDanger == 0 and gameInProgress ~= 2) then --ONOES
					if (playSounds == 1) then PlaySoundFile(soundEffects["danger"]) end
					inDanger = 1
					backgroundPulse = true
				elseif (topRow <= numRowsVisible - 4 and inDanger == 1) then
					TetrisAttackFieldBackground:SetTexture(0.05, 0.05, 0.05, 0.75)
					inDanger = 0
				end
				if (gameInProgress == 2) then --Check for puzzle mode win/loss
					if (numBlocksLeft <= 0 and numGarbageBlocks < 1) then
						TetrisAttack_WinPuzzle()
					elseif (numBlocksLeft > 0 and numMovesLeft <= 0) then
						TetrisAttack_LosePuzzle()
					end
				end
			end
			if (speedyShift == 1 or (gameInProgress ~= 2 and freezeTime <= 0)) then	-- don't do this stuff in puzzle mode
				fieldMoved = TetrisAttack_MovePlayerField(elapsed)
			end
			if (freezeTime > 0) then --If we've got freezetime, expend it and update the display.
				freezeTime = freezeTime - elapsed 
				TetrisAttackSpeedDisplay:SetText(math.floor(freezeTime))
				if (freezeTime <= 0) then
					TetrisAttackSpeedText:SetText("Speed:")
					TetrisAttackSpeedDisplay:SetText(currentSpeed)
				end
			end
		else
			TetrisAttack_UpdateClears(elapsed) --Make blocks fade out
			TetrisAttack_PointsAndCombos() --Another sub-function created to solve the problem of having more than 60 upvalues.
		end
		if gameInProgress > 0 then
			if (IsAltKeyDown()) then TetrisAttack_UnstickBlocks() end --Hey, this turned out not to be needed since I solved the falling blocks bug, but I'm leaving it in in case I need it.
			TetrisAttack_UpdateFallers(elapsed)
			TetrisAttack_UpdateGarbage(elapsed)
			TetrisAttackField:ClearAllPoints()
			TetrisAttackField:SetPoint("CENTER", TetrisAttack1, "CENTER", 0, fieldOffset) --Oh, hey, I guess I should explain this. Only falling blocks move every frame, the other movement you see is actually the frame they're anchored to being pushed up.
			if (gameInProgress == 3) then
				heartbeat = heartbeat + elapsed --Ba-bump. Ba-bump.
				if (heartbeat > 1.5) then
					TetrisAttackCommentText:SetText("LAG WARNING")
					TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
				end
				if (heartbeat > 5) then
					TetrisAttackCommentText:SetText("Connection lost :(")
					TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
					TetrisAttack_SendDisconnect()
					lastOpponent = currentOpponent
					currentOpponent = ""
					TetrisAttack_TriggerWinAnim()
				end
				TetrisAttack_UpdateOpponentBackground(elapsed)
				if MPfieldOffset < 32 then --Update the opponent's field to the value we predict it will be at when we next hear from them.
					if (MPfieldMoving) then MPfieldOffset = MPfieldOffset + gameSpeeds[currentOpponentSpeed][1]/GetFramerate() * (((opponentBloodTimer > 0) and 1.1) or ((opponentTranqTimer > 0) and .8) or 1) end
					for column = 1, 6 do
						newBlockTexturesMP[column]:SetTexCoord(0,1,0,((MPfieldOffset+1)/32))
						newBlockTexturesMP[column]:SetHeight(MPfieldOffset+1)
					end
				end
				TetrisAttackOpponentField:ClearAllPoints()
				TetrisAttackOpponentField:SetPoint("CENTER", TetrisAttack2, "CENTER", 0, MPfieldOffset)
				multiplayerUpdateTimer = multiplayerUpdateTimer + elapsed
				MPgarbageUpdateTimer = MPgarbageUpdateTimer + elapsed
				if (multiplayerUpdateTimer > multiplayerUpdateInterval) then
					TetrisAttack_SendEncodedGrid(fieldMoved)
					multiplayerUpdateTimer = multiplayerUpdateTimer - multiplayerUpdateInterval
				end
				if (MPgarbageUpdateTimer > multiplayerUpdateInterval * 3) then
					TetrisAttack_SendGarbage()
					MPgarbageUpdateTimer = MPgarbageUpdateTimer - multiplayerUpdateInterval*3
				end
			end
		end
	end
end

--[[

	No more local functions. Either I ran out of local variable namespace, which is the case for most of these functions, or they needed to be global so they could be called from the outside.
	
	I have since cleared up a good deal of local namespace, so I may make more of these local in future versions.
	
--]]

--This function handles awarding points and combo garbage for clearing more than 3 blocks in a single frame.
function TetrisAttack_PointsAndCombos()
	if (totalBlocksThisFrame > 0 and gameInProgress ~= 2) then
		local bulkBonus = (totalBlocksThisFrame - 3) * (2*currentSpeed)		--For readability's sake
		playerScore = playerScore + bulkBonus
		pointsTillNextSpeedIncrease = pointsTillNextSpeedIncrease - bulkBonus
		TetrisAttack_UpdateEndlessScoreDisplay(playerScore)
		if (pointsTillNextSpeedIncrease < 0) then
			pointsTillNextSpeedIncrease = basePointsPerSpeedIncrease
			TetrisAttack_ChangeGameSpeed(currentSpeed + 1)
		end
		if (totalBlocksThisFrame > 3) then
			local text = "" .. totalBlocksThisFrame
			TetrisAttack_CreateComboAnim(lastBlockClearedRow, lastBlockClearedColumn, text, 1)
			local mpText = string.format("%02d",lastBlockClearedRow) .. lastBlockClearedColumn
			mpText = mpText .. "0" .. string.format("%02d",totalBlocksThisFrame)
			animsToSend = animsToSend .. mpText
			local bonusTime = blockClearDelay * (totalBlocksThisFrame/(currentDifficulty + 3))
			if (TetrisAttack_FindLowestEmptyRow() > numRowsVisible - 2) then bonusTime = bonusTime * 1.5 end --award extra freezetime when in danger.
			freezeTime = freezeTime + bonusTime
			TetrisAttackSpeedText:SetText("Freeze Time:")
			TetrisAttackSpeedDisplay:SetText(math.floor(freezeTime))
			if (playSounds == 1) then PlaySoundFile(soundEffects["baseCombo"]) end
			if (gameInProgress == 3) then
				if (totalBlocksThisFrame > 23) then
					MPgarbageCode = MPgarbageCode .. "0707"
				else
					if (currentDifficulty < currentOpponentDifficulty or bloodTimer > 0) then --Modify garbage based on bloodlust and difficulty.
						MPgarbageCode = MPgarbageCode .. MPsmallGarbageCodes[totalBlocksThisFrame - 2]
					elseif (currentDifficulty > currentOpponentDifficulty) then
						if (totalBlocksThisFrame > 4) then
							MPgarbageCode = MPgarbageCode .. MPsmallGarbageCodes[totalBlocksThisFrame - 4]
						end
					else
						MPgarbageCode = MPgarbageCode .. MPsmallGarbageCodes[totalBlocksThisFrame - 3]
					end
				end
			end
		end
		totalBlocksThisFrame = 0
		if (totalBonusThisFrame >= 3) then
			TetrisAttack_Powerup()
			totalBonusThisFrame = 0
		end
	elseif (gameInProgress == 2 and totalBlocksThisFrame > 0) then --Still do animations in puzzle mode, but nothing else.
		if (totalBlocksThisFrame > 3) then
			local text = "" .. totalBlocksThisFrame
			TetrisAttack_CreateComboAnim(lastBlockClearedRow, lastBlockClearedColumn, text, 1)
			if (playSounds == 1) then PlaySoundFile(soundEffects["baseCombo"]) end
		end
		if (totalBonusThisFrame >= 3) then
			TetrisAttack_Powerup()
			totalBonusThisFrame = 0
		end
		totalBlocksThisFrame = 0
	end
end

--OnUpdate basically just calls everything else, while also handling those animations for the win and loss messages, since I didn't really think they deserved their own functions.
function TetrisAttack_OnUpdate(self, elapsed)
	if (miniMapDrag) then TetrisAttack_UpdateMinimapButton() end
	--if (FriendsFrame:IsShown() and (friendListOffset ~= FauxScrollFrame_GetOffset(FriendsFrameFriendsScrollFrame))) then
	--	TetrisAttack_FriendsList_Update()
	--end
	if (not TetrisAttack_CheckTimers(elapsed)) then return end
	if (deathAnim ~= 0) then TetrisAttack_UpdateDeathAnims(elapsed) end
	if (cursorAnimPos ~= 0) then
		cursorAnimPos = cursorAnimPos + cursorAnimVelocity/GetFramerate()
		if (cursorAnimPos > -138 and cursorAnimPos < -10) then 
			cursorTexture:SetTexCoord(0, 1, 1 + (cursorAnimPos+10)/128, 1);
			cursorTexture:SetHeight(-(cursorAnimPos+10))
			if (gameInProgress == 3) then
				MPcursorTexture:SetTexCoord(0, 1, 1 + (cursorAnimPos+10)/128, 1);
				MPcursorTexture:SetHeight(-(cursorAnimPos+10))
			end
		elseif (cursorAnimPos < -138) then 
			cursorTexture:SetTexCoord(0, 1, 0, 1)
			cursorTexture:SetHeight(128) 
			if (gameInProgress == 3) then
				MPcursorTexture:SetTexCoord(0, 1, 0, 1)
				MPcursorTexture:SetHeight(128) 
			end
		end
		cursorAnimVelocity = cursorAnimVelocity - 256/GetFramerate()
		cursorTexture:ClearAllPoints()
		cursorTexture:SetPoint("BOTTOM", TetrisAttackFieldBorder, "TOP", 0, cursorAnimPos)
		if (gameInProgress == 3) then
		MPcursorTexture:ClearAllPoints()
		MPcursorTexture:SetPoint("BOTTOM", TetrisAttackOpponentFieldBorder, "TOP", 0, cursorAnimPos)
		end
		if (cursorAnimPos <= -190) then
			if (playSounds == 1) then PlaySoundFile(soundEffects["garbageSmall"]) end
			cursorAnimPos = -190
			cursorAnimVelocity = -cursorAnimVelocity/4 --bounce
			if (abs(cursorAnimVelocity) < 32) then cursorAnimPos = 0 end
		end
	end
	if (deathAnim == 0) then TetrisAttack_UpdateGame(elapsed) end --If we're not dying, call the update function.
end

--Update the pretty animations that play when you win or lose.
function TetrisAttack_UpdateDeathAnims(elapsed)
	local hideRow, shrinkRow, disappearingRow
	if (deathAnim == 1) then
		TetrisAttackField:ClearAllPoints()
		TetrisAttackField:SetPoint("CENTER", TetrisAttack1, "CENTER", 0, deathAnimOffset)
		if (deathAnimOffset < 0) then
			local modOffset = -((-deathAnimOffset) % 32)
			for row=1, numRowsVisible do
					hideRow = abs(deathAnimOffset/32)
					shrinkRow = ceil(abs(deathAnimOffset)/32)
				for column=1, 6 do
					if (row < hideRow) then
						blockTextures[row][column]:Hide()
					elseif (row == shrinkRow) then
						blockTextures[row][column]:SetTexCoord(0,1,0,((33+modOffset)/32))
						blockTextures[row][column]:SetHeight(33+modOffset)
					end
				end
			end
			if (deathAnimOffset < -2000) then --Stay a while and listen!
				deathAnim = 0 
				if (taxiGameInProgress) then 
					TetrisAttack_TaxiEnd() 
				else
					TetrisAttack_YouLose()
				end
			end
		end
		deathAnimVelocity = deathAnimVelocity + (deathAnimAccel * elapsed)
		deathAnimOffset = deathAnimOffset + (deathAnimVelocity * elapsed)
	elseif (deathAnim == -1) then
		if (deathAnimOffset > 0) then
			disappearingRow = math.ceil(deathAnimOffset/32)
			for column = 1, 6 do
				blockTextures[disappearingRow][column]:SetAlpha((deathAnimOffset - disappearingRow*32)/32)
				if (disappearingRow < numRowsVisible) then blockTextures[disappearingRow+1][column]:Hide() end
			end
			deathAnimOffset = deathAnimOffset + deathAnimVelocity/GetFramerate()
			if (deathAnimOffset <= 0) then
				for column = 1, 6 do
					blockTextures[1][column]:Hide()
				end
			end
		elseif (deathAnimOffset < -100) then
			deathAnim = 0 
			TetrisAttack_YouWin()
		else
			deathAnimOffset = deathAnimOffset + deathAnimVelocity/GetFramerate()
		end
	end
	if (opponentDeathAnim == 1) then
		TetrisAttackOpponentField:ClearAllPoints()
		TetrisAttackOpponentField:SetPoint("CENTER", TetrisAttack2, "CENTER", 0, opponentDeathAnimOffset)
		if (opponentDeathAnimOffset < 0) then
			local modOffset = -((-opponentDeathAnimOffset) % 32)
			for row=1, numRowsVisible do
					hideRow = abs(opponentDeathAnimOffset/32)
					shrinkRow = ceil(abs(opponentDeathAnimOffset)/32)
				for column=1, 6 do
					if (row < hideRow) then
						blockTexturesMP[row][column]:Hide()
					elseif (row == shrinkRow) then
						blockTexturesMP[row][column]:SetTexCoord(0,1,0,((33+modOffset)/32))
						blockTexturesMP[row][column]:SetHeight(33+modOffset)
					end
				end
			end
			if (opponentDeathAnimOffset < -768) then 
				opponentDeathAnim = 0 
			end
		end
		opponentDeathAnimVelocity = opponentDeathAnimVelocity + (opponentDeathAnimAccel * elapsed)
		opponentDeathAnimOffset = opponentDeathAnimOffset + (opponentDeathAnimVelocity * elapsed)
	elseif (opponentDeathAnim == -1) then
		if (opponentDeathAnimOffset > 0) then
			disappearingRow = math.ceil(opponentDeathAnimOffset/32)
			for column = 1, 6 do
				blockTexturesMP[disappearingRow][column]:SetAlpha((opponentDeathAnimOffset - disappearingRow*32)/32)
				if (disappearingRow < numRowsVisible) then blockTexturesMP[disappearingRow+1][column]:Hide() end
			end
			opponentDeathAnimOffset = opponentDeathAnimOffset + opponentDeathAnimVelocity/GetFramerate()
			if (opponentDeathAnimOffset <= 0) then
				for column = 1, 6 do
					blockTexturesMP[1][column]:Hide()
				end
			end
		elseif (opponentDeathAnimOffset < -64) then
			opponentDeathAnim = 0 
		else
			opponentDeathAnimOffset = opponentDeathAnimOffset + opponentDeathAnimVelocity/GetFramerate()
		end
	end
end

--Show the given tooltip when we mouseover.
function TetrisAttack_ShowCTRLtip(obj, text)
	if (IsControlKeyDown()) then
		GameTooltip:SetOwner( obj, "ANCHOR_RIGHT")
		GameTooltip:AddLine(text, 1, 1, 1, 1)
		GameTooltip:Show()
	end
end

--I implemented this as a bandaid for falling blocks sticking mid-air, but that problem has since been fixed. Keeping it around just in case.
function TetrisAttack_UnstickBlocks()
	local row
	local color
	for column = 1, 6 do
		row = 1
		color = 1
		while (color > 0 and row <= numRows) do
			color = blockTable[row][column]
			row = row+1
		end
		TetrisAttack_StartFalling(row, column, 0, .01)
	end
end

--Pretty self-explanatory.
function TetrisAttack_UpdateOpponentBackground(elapsed)
	if (opponentTranqTimer > 0) then
		opponentDangerPulseTimer = opponentDangerPulseTimer + 1
		opponentTranqTimer = opponentTranqTimer - elapsed
		TetrisAttackOpponentFieldBackground:SetTexture(0, 0.8, 0, sinValues[opponentDangerPulseTimer])
		if (opponentDangerPulseTimer >= 120) then opponentDangerPulseTimer = 0 end
	elseif (opponentBloodTimer > 0) then
		opponentDangerPulseTimer = opponentDangerPulseTimer + 1
		opponentBloodTimer = opponentBloodTimer - elapsed
		TetrisAttackOpponentFieldBackground:SetTexture(0.8, 0, 0, sinValues[opponentDangerPulseTimer])
		if (opponentDangerPulseTimer >= 120) then opponentDangerPulseTimer = 0 end
	elseif (opponentCorrTimer > 0) then
		opponentDangerPulseTimer = opponentDangerPulseTimer + 1
		opponentCorrTimer = opponentCorrTimer - elapsed
		TetrisAttackOpponentFieldBackground:SetTexture(0.6, 0, 0.8, sinValues[opponentDangerPulseTimer])
		if (opponentDangerPulseTimer >= 120) then opponentDangerPulseTimer = 0 end
	end
	if (opponentTranqTimer <= 0 and opponentCorrTimer <= 0 and opponentBloodTimer <= 0 and opponentBackgroundPulse) then
		TetrisAttackOpponentFieldBackground:SetTexture(.05, .05, .05, .75) 
		opponentBackgroundPulse = false
	end
end

--Resolve all open combos, awarding garbage and accelerating the rate of gamespeed increases. 
function TetrisAttack_FinishCombos()
	for i = 1, numCombos do
		if (currentCombos[i] > 1) then
			if (gameInProgress == 3) then
				local size = currentCombos[i]+4
				if (currentDifficulty > currentOpponentDifficulty) then
					MPgarbageCode = MPgarbageCode .. (((size-1 < 10) and "0") or "") .. size-1
				elseif (bloodTimer > 0) then
					MPgarbageCode = MPgarbageCode .. (((size+1 < 10) and "0") or "") .. size+1
				else
					MPgarbageCode = MPgarbageCode .. (((size < 10) and "0") or "") .. size
				end
				currentCombos[i] = 0
			end
			if (basePointsPerSpeedIncrease > 100 and gameInProgress == 1) then
				basePointsPerSpeedIncrease = basePointsPerSpeedIncrease - 5*currentCombos[i]
			end
		end
	end
	numCombos = 0
end

--Replace all blocks of one color with another. For tranquility.
function TetrisAttack_ReplaceBlocks(oldColor, newColor)
	for row = 1, numRows do
		for column = 1, 6 do
			if (blockTable[row][column] == oldColor and not TetrisAttack_BlockIsBeingCleared(row, column)) then
				blockTable[row][column] = newColor
				TetrisAttack_UpdateBlock(row, column)
				TetrisAttack_CheckForMatches(row, column)
			end
		end
	end
	for column = 1, 6 do
		if (newBlockTable[column] == oldColor) then
			newBlockTable[column] = newColor
			if (useClassIcons == 1)
			then
				newBlockTextures[column]:SetVertexColor(blockColor[newColor][1]*.5,blockColor[newColor][2]*.5,blockColor[newColor][3]*.5, 1)
				newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. newColor)
			else
				newBlockTextures[column]:SetVertexColor(blockColor[newColor][1]*.5,blockColor[newColor][2]*.5,blockColor[newColor][3]*.5, 1)
				newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
			end
		end
	end
	for column = 1, 6 do
		for i = 1, fallingBlocksPerColumn[column] do
			if (fallingTable[column][i]["color"] == oldColor and fallingTable[column][i]["column"] > 0) then
				fallingTable[column][i]["color"] = newColor
				TetrisAttack_UpdateBlock(fallingTable[column][i]["column"], fallingTable[column][i]["row"], newColor)
			end
		end
	end
end

--Bump the field up one row to give the illusion of smooth scrolling. Called when fieldOffset >= 32.
function TetrisAttack_bumpField()
	if (speedyShift == 1) then bonusBlockChance = bonusBlockChance + 5 end --More ?-blocks when you manually bump.
	speedyShift = 0
	local TetrisAttack_UpdateBlock = TetrisAttack_UpdateBlock
	for row = numRows, 2, -1 do	--Shift all blocks up by one.
		for column = 1, 6 do
			blockTable[row][column] = blockTable[row-1][column]
			TetrisAttack_UpdateBlock(row, column)
		end
	end
	for i = 1, 6 do
		if (fallingBlocksPerColumn[i] > 0) then
			for j = 1, fallingBlocksPerColumn[i] do
				TetrisAttack_ResetBlockPos(fallingTable[i][j]["row"], fallingTable[i][j]["column"])
				if (fallingTable[i][j]["row"] > 1) then fallingTable[i][j]["row"] = fallingTable[i][j]["row"] + 1 end
			end
		end
	end
	for i = 1, numGarbageBlocks do
		if (garbageTable[i]["bottom"] > 0) then
			for j = garbageTable[i]["left"],garbageTable[i]["right"] do
				TetrisAttack_ResetBlockPos(garbageTable[i]["bottom"], j)
			end
			garbageTable[i]["bottom"] = garbageTable[i]["bottom"] + 1
			garbageTable[i]["top"] = garbageTable[i]["top"] + 1
		end
		TetrisAttack_UpdateGarbageChunk(i)
	end
	local lastRandom1 = 0
	local lastRandom2 = 0
	for column = 1, 6 do	--Move new blocks up
		blockTable[1][column] = newBlockTable[column]
		TetrisAttack_UpdateBlock(1, column)
		if ((math.random(100) < bonusBlockChance) and (gameInProgress == 3) and (totalBonusOnField < 6)) then --Create ?-blocks in multiplayer.
			bonusBlockChance = currentSpeed/1.5
			if (bonusBlockChance > 10) then bonusBlockChance = 10 end 
			totalBonusOnField = totalBonusOnField + 1
			newBlockTable[column] = 8
		else
			newBlockTable[column] = math.random(blockTypes)
		end
		while (newBlockTable[column] == lastRandom1 and newBlockTable[column] == lastRandom2) do
			newBlockTable[column] = math.random(blockTypes)
		end
		lastRandom2 = lastRandom1
		lastRandom1 = newBlockTable[column]
		newBlockTextures[column]:SetVertexColor(blockColor[lastRandom1][1]*.5,blockColor[lastRandom1][2]*.5,blockColor[lastRandom1][3]*.5, 1)
		newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. lastRandom1)
		if (newBlockTable[column] == 8) then
			newBlockTextures[column]:SetVertexColor(1,1,1,1)
			newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockbonus")
		elseif (newBlockTable[column] == 0 or useClassIcons == 1) then
			newBlockTextures[column]:SetVertexColor(blockColor[lastRandom1][1]*.5,blockColor[lastRandom1][2]*.5,blockColor[lastRandom1][3]*.5, 1)
			newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. lastRandom1)
		elseif (colorizeBlocks == 1) then
			newBlockTextures[column]:SetVertexColor(blockColor[lastRandom1][1]*.5,blockColor[lastRandom1][2]*.5,blockColor[lastRandom1][3]*.5, 1)
			newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
		end
	end
	for column = 1, 6 do
		totalBlocks = TetrisAttack_CheckForMatches(1, column)
	end
	TetrisAttack_CursorUp() --Move the cursor too.
end

--Updates the given garbage chunk's graphics.
function TetrisAttack_UpdateGarbageChunk(chunk)
	local top = garbageTable[chunk]["top"]
	local bottom = garbageTable[chunk]["bottom"]
	if (top == 0 or bottom == 0) then return end
	local left = garbageTable[chunk]["left"]
	local right = garbageTable[chunk]["right"]
	local sides = ""
	for row = bottom, top do
		for column = left, right do
			if (row <= numRowsVisible) then
				sides = "" .. ((((column == left) and "L") or ((column == right) and "R")) or "")
				sides = sides .. (((row == top) and "T") or "")
				sides = sides .. (((row == bottom) and "B") or "")
				blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockgarbage" .. sides)
				blockTextures[row][column]:SetVertexColor(blockColor[7][1],blockColor[7][2],blockColor[7][3], 1)
			end
		end
	end
end

--Updates an individual block with garbage graphics.
function TetrisAttack_UpdateGarbageBlock(row, column, sides)
	blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockgarbage" .. sides)
	blockTextures[row][column]:SetVertexColor(blockColor[7][1],blockColor[7][2],blockColor[7][3], 1)
end

--Causes a block to start falling. After the given delay, the block will be removed from the field and handled as a seperate case by UpdateFallers.
--It will also leave a trail of negative numbers as it falls which allow the game to easily sends its position over multiplayer without having to loop through all falling blocks every frame.
function TetrisAttack_StartFalling(row, column, combo, delay)
	combo = combo or 0
	if (combo == 0) then
		numCombos = numCombos + 1
		currentCombos[numCombos] = 0
		combo = numCombos
	end
	delay = delay or fallingDelay
	if (pregameCleanup == 1) then delay = .01 end
	if (row <= numRows and row > 1 and not TetrisAttack_BlockIsBeingCleared(row, column) and not TetrisAttack_BlockIsSuspended(row, column)) then
		if (blockTable[row][column] > 0) then
			if (blockTable[row][column] ~= 7) then
				local num = fallingBlocksPerColumn[column] + 1
				fallingBlocksPerColumn[column] = num
				if (not fallingTable[column][num]) then --Make a new one if we have to.
					fallingTable[column][num] = {["row"] = row,
												["column"] = column,
												["offset"] = 0,
												["color"] = blockTable[row][column],
												["combo"] = combo,
												["delay"] = delay}	
					if (delay == 0) then --If we got a delay of 0, give this block a head start and then give it a delay of .0001 to still let things work correctly.
						fallingTable[column][num]["offset"] = blockFallSpeed/GetFramerate()
						fallingTable[column][num]["delay"] = 0.0001
					else
						fallingTable[column][num]["offset"] = 0
						fallingTable[column][num]["delay"] = delay
					end
				else		
					if (delay == 0) then
						fallingTable[column][num]["offset"] = blockFallSpeed/GetFramerate()
						fallingTable[column][num]["delay"] = 0.0001
					else
						fallingTable[column][num]["offset"] = 0
						fallingTable[column][num]["delay"] = delay
					end
					fallingTable[column][num]["row"] = row
					fallingTable[column][num]["column"] = column
					fallingTable[column][num]["color"] = blockTable[row][column]
					fallingTable[column][num]["combo"] = combo
				end
			end
		end
		TetrisAttack_StartFalling(row+1, column, combo, delay)
		
	end
end

--I thought about making this local, but decided against it. Handles all active falling blocks, including collision detection, movement, updating textures and all that jazz.
--[[local--]] function TetrisAttack_UpdateFallers(time)
	local numSkipped = 0;
	for column = 1, 6 do
		if (fallingBlocksPerColumn[column] > 0) then
			numSkipped = 0
			for j = 1, fallingBlocksPerColumn[column] do
				local row = fallingTable[column][j]["row"]
				if (row > 1 and row <= numRows and column >= 1 and column <= 6) then
					local delay = fallingTable[column][j]["delay"]
					local offset = fallingTable[column][j]["offset"]
					if (delay > 0) then
						delay = delay - time
						if (delay <= 0) then
							if (blockTable[row-1][column] <= 0) then
								blockTable[row][column] = -fallingTable[column][j]["color"]
								offset = offset + blockFallSpeed/GetFramerate()
								fallingTable[column][j]["offset"] = offset
								if (row <= numRowsVisible) then
									blockTextures[row][column]:ClearAllPoints()
									blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) - (offset - 10) + 32)
								end
								if (offset > 32) then
									while (offset > 32) do
										TetrisAttack_ResetBlockPos(row, column)
										TetrisAttack_UpdateBlock(row, column)
										if (blockTable[row-2][column] > 0) then -- collision
											if (playSounds == 1) then PlaySoundFile(soundEffects["blockfall"]) end
											fallingTable[column][j]["row"] = 0
											fallingTable[column][j]["column"] = 0
											offset = 0
											blockTable[row-1][column] = fallingTable[column][j]["color"]
											if (blockTable[row][column] <= 0) then blockTable[row][column] = 0 end
											TetrisAttack_UpdateBlock(row-1, column)
											TetrisAttack_QueueForChecking(row-1, column, fallingTable[column][j]["combo"])
										else
											offset = offset - 32
											row = row - 1
											fallingTable[column][j]["row"] = row
											fallingTable[column][j]["offset"] = offset
											if (blockTable[row+1][column] <= 0) then blockTable[row+1][column] = 0 end
											blockTable[row][column] = -fallingTable[column][j]["color"]
											TetrisAttack_UpdateBlock(row, column, fallingTable[column][j]["color"])
											if (row <= numRowsVisible) then
												blockTextures[row][column]:ClearAllPoints()
												blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) - (offset - 10) + 32)
											end
										end
									end
								end
								TetrisAttack_StartFalling(row+1, column, 1, .01)
							else
								fallingTable[column][j]["row"] = 0
								fallingTable[column][j]["column"] = 0
								TetrisAttack_QueueForChecking(row, column, fallingTable[column][j]["combo"])
							end
						end
						fallingTable[column][j]["delay"] = delay
					else
						offset = offset + blockFallSpeed/GetFramerate()
						if (blockTable[row-1][column] <= 0) then
							if (offset > 32) then
								while (offset > 32) do
									TetrisAttack_ResetBlockPos(row, column)
									TetrisAttack_UpdateBlock(row, column)
									if (blockTable[row-2][column] > 0) then -- collision
										if (playSounds == 1) then PlaySoundFile(soundEffects["blockfall"]) end
										fallingTable[column][j]["row"] = 0
										fallingTable[column][j]["column"] = 0
										offset = 0
										blockTable[row-1][column] = fallingTable[column][j]["color"]
										if (blockTable[row][column] <= 0) then blockTable[row][column] = 0 end
										TetrisAttack_UpdateBlock(row-1, column)
										TetrisAttack_QueueForChecking(row-1, column, fallingTable[column][j]["combo"])
									else
										offset = offset - 32
										row = row - 1
										fallingTable[column][j]["row"] = row
										fallingTable[column][j]["offset"] = offset
										if (blockTable[row+1][column] <= 0) then blockTable[row+1][column] = 0 end
										blockTable[row][column] = -fallingTable[column][j]["color"]
										TetrisAttack_UpdateBlock(row, column, fallingTable[column][j]["color"])
										if (row <= numRowsVisible) then
											blockTextures[row][column]:ClearAllPoints()
											blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) - (offset - 10) + 32)
										end
									end
								end
							else
								fallingTable[column][j]["offset"] = offset
								if (row <= numRowsVisible) then
									blockTextures[row][column]:ClearAllPoints()
									blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) - (offset - 10) + 32)
								end
							end
						else -- surprise collision
							if (playSounds == 1) then PlaySoundFile(soundEffects["blockfall"]) end
							TetrisAttack_ResetBlockPos(row, column)
							TetrisAttack_UpdateBlock(row, column)
							fallingTable[column][j]["row"] = 0
							fallingTable[column][j]["column"] = 0
							blockTable[row][column] = fallingTable[column][j]["color"]
							if (blockTable[row+1][column] <= 0) then blockTable[row+1][column] = 0 end
							TetrisAttack_UpdateBlock(row, column)
							TetrisAttack_QueueForChecking(row, column, fallingTable[column][j]["combo"])
						end
					end
				else
					numSkipped = numSkipped + 1
				end
			end
		end
		if (numSkipped >= fallingBlocksPerColumn[column]) then --When all falling blocks are concluded, reset the number.
			fallingBlocksPerColumn[column] = 0
		end
	end
end

--Mark a block to be checked later.
function TetrisAttack_QueueForChecking(row, column, combo)
	numBlocksToCheck = numBlocksToCheck + 1
	if (not blocksToCheck[numBlocksToCheck]) then
		blocksToCheck[numBlocksToCheck] = {row, column, combo}
	else
		blocksToCheck[numBlocksToCheck][1] = row
		blocksToCheck[numBlocksToCheck][2] = column
		blocksToCheck[numBlocksToCheck][3] = combo
	end
end

--Put garbage in the queue to be dropped as soon as there's room.
function TetrisAttack_QueueGarbage(size)
	numGarbageToCreate = numGarbageToCreate + 1
	local width = ((size > 6) and 6) or size
	local height = ((size > 6) and size - 5) or 1
	if (height > 5 and playSounds == 1) then 
		PlaySoundFile(soundEffects["ohShit"])
		ohShitTimer = 2
	end
	if (not garbageQueue[numGarbageToCreate]) then
		garbageQueue[numGarbageToCreate] = {width, height}
	else
		garbageQueue[numGarbageToCreate][1] = width
		garbageQueue[numGarbageToCreate][2] = height
	end
end

--Update all blocks which are currently disappearing. Again, I thought about making this local, but decided against it.
--[[local--]] function TetrisAttack_UpdateClears(time)
	local row, column, combo
	local colorIndex = 0
	local skippedBlocks = 0
	for i = 1, numBlocksToClear do
		if (blocksToClear[i]["delay"] > 0) then
			row = blocksToClear[i]["row"]
			column = blocksToClear[i]["column"]
			combo = blocksToClear[i]["combo"]
			colorIndex = blockTable[row][column]
			
			blocksToClear[i]["delay"] = blocksToClear[i]["delay"] - time
			blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], blocksToClear[i]["delay"]/blockClearDelay)
			if (blocksToClear[i]["delay"] < 0) then
				blockTable[row][column] = 0
				TetrisAttack_UpdateBlock(row, column)
				blocksToClear[i]["row"] = 0
				blocksToClear[i]["column"] = 0
				TetrisAttack_StartFalling(row+1, column, combo, .15)
			end
		else
			skippedBlocks = skippedBlocks + 1
			if (skippedBlocks == numBlocksToClear) then numBlocksToClear = 0 end
		end
	end
end

--Attempt to flip a block that's stationary with one that's falling mid-air, if any are in range.
--[[local--]] function TetrisAttack_FlipFallingBlocks(Y, X)
	local didNothing = true
	if (blockTable[Y][X] <= 0 and blockTable[Y][X+1] > 0) then
		for block = 1, fallingBlocksPerColumn[X] do
			local row = fallingTable[X][block]["row"]
			local column = fallingTable[X][block]["column"]
			local offset = fallingTable[X][block]["offset"]
			local color = fallingTable[X][block]["color"]
			if ((Y == row and offset < 16) or (Y == row-1 and offset > 16)) then -- We've got a block in range.
				if (playSounds == 1) then PlaySoundFile(soundEffects["blockFlip"]) end
				fallingTable[X][block]["row"] = 0
				fallingTable[X][block]["column"] = 0
				if (blockTable[Y+1][X] < 0) then blockTable[Y+1][X] = 0 end
				blockTable[Y][X] = blockTable[Y][X+1]
				blockTable[Y][X+1] = color
				blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block0")
				TetrisAttack_ResetBlockPos(row, column)
				TetrisAttack_UpdateBlock(Y, X)
				TetrisAttack_UpdateBlock(Y, X+1)
				TetrisAttack_StartFalling(Y, X)
				TetrisAttack_StartFalling(Y+1, X)
				TetrisAttack_CheckForMatches(Y, X + 1)
				didNothing = false
			elseif ((Y == row and offset > 16) or (Y == row - 1 and offset < 16)) then
				didNothing = false
			end
		end
	elseif (blockTable[Y][X] > 0 and blockTable[Y][X+1] <= 0) then
		for block = 1, fallingBlocksPerColumn[X+1] do
			local row = fallingTable[X+1][block]["row"]
			local column = fallingTable[X+1][block]["column"]
			local offset = fallingTable[X+1][block]["offset"]
			local color = fallingTable[X+1][block]["color"]
			if ((Y == row and offset < 16) or (Y == row-1 and offset > 16)) then -- We've got a block in range.
				if (playSounds == 1) then PlaySoundFile(soundEffects["blockFlip"]) end
				fallingTable[X+1][block]["row"] = 0
				fallingTable[X+1][block]["column"] = 0
				if (blockTable[Y+1][X+1] < 0) then blockTable[Y+1][X+1] = 0 end
				blockTable[Y][X+1] = blockTable[Y][X]
				blockTable[Y][X] = color
				blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block0")
				TetrisAttack_ResetBlockPos(row, column)
				TetrisAttack_UpdateBlock(Y, X)
				TetrisAttack_UpdateBlock(Y, X+1)
				TetrisAttack_StartFalling(Y, X+1)
				TetrisAttack_StartFalling(Y+1, X+1)
				TetrisAttack_CheckForMatches(Y, X)
				didNothing = false
			elseif ((Y == row and offset > 16) or (Y == row - 1 and offset < 16)) then
				didNothing = false
			end
		end
	end
	return didNothing
end

--Flip the two blocks at the cursor after checking a ridiculous number of reason why you shouldn't be allowed to.
function TetrisAttack_CursorFlip()
	if (demoRecord) then
		currentDemo[currentDemoPos*2-1] = 5
		currentDemo[currentDemoPos*2] = GetTime() - demoStart
		currentDemoPos = currentDemoPos+1
		currentDemo[currentDemoPos*2+1] = 0
		currentDemo[currentDemoPos*2+2] = 0
	end
	if (gameInProgress == 0 or allowInput == false or (gameInProgress == 2 and numMovesLeft == 0) or isPaused == 1 or deathAnim ~= 0) then
		return
	end
	if (TetrisAttack_BlockIsBeingCleared(cursorY, cursorX) or TetrisAttack_BlockIsBeingCleared(cursorY, cursorX+1)) then
		return
	end
	if (TetrisAttack_BlockIsSuspended(cursorY, cursorX) or TetrisAttack_BlockIsSuspended(cursorY, cursorX+1)) then
		return
	end
	if (not TetrisAttack_FlipFallingBlocks(cursorY, cursorX)) then
		return
	end
	if (blockTable[cursorY][cursorX] == 7 or blockTable[cursorY][cursorX+1] == 7) then
		return
	end
	if (blockTable[cursorY][cursorX] < 0) then blockTable[cursorY][cursorX] = 0 end
	if (blockTable[cursorY][cursorX+1] < 0) then blockTable[cursorY][cursorX+1] = 0 end
	
	--make a new combo, since we're manually moving the blocks
	numCombos = numCombos + 1
	currentCombos[numCombos] = 0
	local combo = numCombos
	
	local temp = blockTable[cursorY][cursorX]
	if (playSounds == 1) then PlaySoundFile(soundEffects["blockFlip"]) end
	blockTable[cursorY][cursorX] = blockTable[cursorY][cursorX+1]
	blockTable[cursorY][cursorX+1] = temp
	if (gameInProgress == 2 and blockTable[cursorY][cursorX] ~= blockTable[cursorY][cursorX+1]) then
		numMovesLeft = numMovesLeft - 1
		TetrisAttackPuzzleMovesDisplay:SetText(numMovesLeft)
	end
	TetrisAttack_UpdateBlock(cursorY, cursorX)
	TetrisAttack_UpdateBlock(cursorY, cursorX+1)
	
	if (blockTable[cursorY - 1][cursorX] <= 0) then
		TetrisAttack_StartFalling(cursorY, cursorX)
	else
		TetrisAttack_CheckForMatches(cursorY, cursorX, combo)
	end
	
	if (blockTable[cursorY - 1][cursorX + 1] <= 0) then
		TetrisAttack_StartFalling(cursorY, cursorX + 1)
	else
		TetrisAttack_CheckForMatches(cursorY, cursorX + 1, combo)
	end
	
	if (blockTable[cursorY][cursorX] <= 0 and cursorY < numRowsVisible) then
		TetrisAttack_StartFalling(cursorY +1, cursorX)
	end
	if (blockTable[cursorY][cursorX+1] <= 0 and cursorY < numRowsVisible) then
		TetrisAttack_StartFalling(cursorY +1, cursorX+1)
	end
end

--Hide a game window. Not very well done, but this was like the first function I even wrote in LUA, and it still works.
function TetrisAttack_Hide(num)
	local frame = getglobal("TetrisAttack" .. num)
	frame:Hide()
	if (autoPause) then TetrisAttack_Pause(1) end
end 

--Toggle a game window. Same deal.
function TetrisAttack_Toggle(num)
	local frame = getglobal("TetrisAttack" .. num)
	if (frame) then
		if(frame:IsVisible()) then
			frame:Hide()
			if (gameInProgress == 1 or gameInProgress == 2) then TetrisAttack_Pause(1) end
		else
			frame:Show()
		end
	end
end

--Slash commands.
function TetrisAttack_SlashCommandHandler(msg)
	if (msg == "ResetEndless") then	
		out("FPL: Endless mode scores cleared!")
		for i = 1, 10 do
			TetrisAttack_EndlessHighScores[i]["name"] = ""
			TetrisAttack_EndlessHighScores[i]["score"] = 0
			TetrisAttack_EndlessHighScores[i]["difficulty"] = 0
		end
	elseif (msg == "ResetTaxi") then
		out("FPL: Taxi mode scores cleared!")
		TetrisAttack_FlightpathHighScores = {}
	elseif (msg == "RaidUpdate") then
		TetrisAttack_SendRaidGreeting(true)
	else
		TetrisAttack_Toggle(1)
	end
end

--The rest of the events are defined later to be sure none of them fire before VARIABLES_LOADED.
function TetrisAttack_OnLoad()
	TetrisAttackBaseFrame:RegisterEvent("VARIABLES_LOADED")
	currentPlayerName = UnitName("player")
	
	SLASH_PUZZLELEAGUE1 = "/fpl"
	SlashCmdList["PUZZLELEAGUE"] = TetrisAttack_SlashCommandHandler
end

--Player clicked the "forfeit" button.
function TetrisAttack_Forfeit()
	if (gameInProgress == 3) then
		SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "FORFEIT" .. "~" .. currentVersion, "WHISPER", currentOpponent)
	end
	TetrisAttack_TriggerDeathAnim()
end

--Handle UI events. Most of this deals with multiplayer communication and stuff like that.
function TetrisAttack_OnEvent(event)
	if (event == "CHAT_MSG_ADDON") then
		if (arg1 == chatMsgFieldPrefix and arg3 == "WHISPER" and gameInProgress == 3) then
			TetrisAttack_MPHeartbeat()
			TetrisAttack_UpdateMPDisplay(arg2)
		end
		if (arg1 == chatMsgGreetingPrefix and arg3 == "WHISPER") then
			local message, version = strsplit("~", arg2)
			if (message == "PING") then --Respond to pings with high score information and a version check.
				SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "PONG" .. "~" .. currentVersion, "WHISPER", arg4)
				SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScorePrefix, " ~ ~ ~ ~ ~" .. TetrisAttack_EndlessHighScores[1]["name"] .. "~" .. TetrisAttack_EndlessHighScores[1]["score"] .. "~" .. TetrisAttack_EndlessHighScores[1]["difficulty"], "WHISPER", arg4)
			elseif (message == "WIN" and gameInProgress == 3) then
				TetrisAttack_TriggerWinAnim()
			elseif (message == "BUSY") then
				if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: " .. arg4 .. " is busy with another opponent!", 1, 1, 1) end
				currentOpponent = ""
				challengeTimer = 0
			elseif (message == "DISCONNECT" and gameInProgress == 3) then
				TetrisAttackCommentText:SetText("Connection lost :(")
				TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
				lastOpponent = currentOpponent
				currentOpponent = ""
				TetrisAttack_TriggerWinAnim()
			elseif (message == "PAUSE" and gameInProgress == 3) then
				TetrisAttack_Pause(1, true)
				TetrisAttackCommentText:SetText(currentOpponent .. " paused the game!")
			elseif (message == "UNPAUSE" and gameInProgress == 3) then
				TetrisAttack_Pause(0, true)
				TetrisAttackCommentText:SetText("")
			elseif (message == "FORFEIT" and gameInProgress == 3) then
				TetrisAttackCommentText:SetText(currentOpponent .. " gave up!")
				TetrisAttack_TriggerWinAnim()
			end
			TetrisAttack_RecvGreeting(arg4, tonumber(version)) --Register this person as having the mod.
		elseif (arg1 == chatMsgGreetingPrefix and arg3 == "RAID") then
			local version, gametype = strsplit("~", arg2)
			TetrisAttack_RecvRaidGreeting(arg4, tonumber(version), tonumber(gametype))
		elseif (arg1 == chatMsgChallengePrefix and arg3 == "WHISPER") then
			if (gameInProgress == 3) then
				SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgGreetingPrefix, "BUSY" .. "~" .. currentVersion, "WHISPER", arg4)
				return
			end
			local message, diff, speed, gender, version = strsplit("~", arg2)
			if (tonumber(version) > currentVersion) then
				DEFAULT_CHAT_FRAME:AddMessage("FPL: Received a challenge from " .. arg4 .. ", but they're using a newer version than you. Go download it!", 1, 0, 0)
				SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgChallengePrefix, "NEWVERSION~ ~ ~ ~" .. version, "WHISPER", arg4)
				return
			elseif (tonumber(version) < currentVersion) then
				DEFAULT_CHAT_FRAME:AddMessage("FPL: Received a challenge from " .. arg4 .. ", but they're using an older version than you.", 1, 1, 1)
				SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgChallengePrefix, "OLDVERSION~ ~ ~ ~" .. version, "WHISPER", arg4)
				return
			end
			diff = tonumber(diff)
			speed = tonumber(speed)
			if (message == "ACCEPT") then --Your challenge was accepted.
				if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: " .. arg4 .. " has accepted your challenge!", 1, 1, 1) end
				challengeTimer = 0
				TetrisAttack_UpdateOpponentInfo(arg4, diff, speed)
				opponentRaceGender = tonumber(gender)
				currentOpponentSpeed = speed
				currentOpponentDifficulty = diff
				TetrisAttack1:Show()
				currentOpponent = arg4
				TetrisAttack_InitMultiplayerComm()
			elseif (message == "DECLINE") then
				if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: " .. arg4 .. " declined your challenge :(", 1, 1, 1) end
				challengeTimer = 0
				currentOpponent = ""
			elseif (message == "NEWVERSION") then
				DEFAULT_CHAT_FRAME:AddMessage("FPL: Challenge failed, " .. arg4 .. " is using version " .. tonumber(version)/10 .. ", which is outdated.", 1, 1, 1)
				challengeTimer = 0
				currentOpponent = ""
			elseif (message == "OLDVERSION") then
				DEFAULT_CHAT_FRAME:AddMessage("FPL: Challenge failed, " .. arg4 .. " is using version " .. tonumber(version)/10 .. ". Go download it!", 1, 0, 0)
				challengeTimer = 0
				currentOpponent = ""
			elseif (message == "CHALLENGE") then --A challenge has been received!
				inInstance, instanceType = IsInInstance()
				if (currentOpponent == "" and (challengeTimer == 0 or arg4 == currentOpponent) and not ignoreChallenges and not (instanceType == "arena") and not (instanceType == "raid" and InCombatLockdown())) then
					challengeTimer = 15
					TetrisAttackChallengeAcceptFrame:Show()
					TetrisAttack_UpdateChallengeInfo(diff, speed)
					TetrisAttack_UpdateOpponentInfo(arg4, diff, speed)
					opponentRaceGender = tonumber(gender)
					currentOpponentSpeed = speed
					currentOpponentDifficulty = diff
					TetrisAttackChallengeName:SetText(arg4)
					if (playSounds) then PlaySoundFile(soundEffects["challenge"]) end
					currentOpponent = arg4
				else
					SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgGreetingPrefix, "BUSY" .. "~" .. currentVersion, "WHISPER", arg4)
				end
			end
		elseif (arg1 == chatMsgGarbagePrefix and arg3 == "WHISPER"  and gameInProgress == 3) then
			TetrisAttack_MPHeartbeat()
			TetrisAttack_RecvGarbage(arg2)
		elseif (arg1 == chatMsgHandshakePrefix and arg3 == "WHISPER") then
			currentOpponent = arg4
			TetrisAttack1:Show()
			TetrisAttack2:Show()
			TetrisAttack_MPHeartbeat()
			TetrisAttack_RecvHandshake(arg2)
		elseif (arg1 == chatMsgPingPrefix and arg3 == "WHISPER") then
			MPAvgLatency = (MPAvgLatency + (GetTime() - lastHandshakeTime)/2) / 2
			lastHandshakeTime = GetTime()
			currentOpponent = arg4
			TetrisAttack_MPHeartbeat()
			local rep = tonumber(arg2)
			if (rep < 5) then 
				SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgPingPrefix, "" .. rep + 1, "WHISPER", arg4)
			elseif (rep == 5) then
				SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgPingPrefix, "" .. rep + 1, "WHISPER", arg4)
				TetrisAttack_StartMultiplayer(0)
			else
				TetrisAttack_StartMultiplayer(MPAvgLatency)
			end
		elseif (arg1 == chatMsgScoreReqPrefix and arg3 == "WHISPER") then
			TetrisAttack_RecvHighScoreRequest(arg4, arg2)
		elseif (arg1 == chatMsgScorePrefix and arg3 == "WHISPER") then
			TetrisAttack_RecvHighScore(arg4, arg2)
		end
	elseif (event == "VARIABLES_LOADED") then
		TetrisAttack_Init()
	elseif (event == "PLAYER_REGEN_DISABLED") then
		if (combatPanic == 1) then TetrisAttack_Hide(1) end
	elseif (event == "PLAYER_CONTROL_GAINED") then
		TetrisAttack_TaxiEnd()
	elseif (event == "CHAT_MSG_SYSTEM") then --Try to detect players logging out.
		local dcMatch = strmatch(arg1, "'.*'")
		if (dcMatch) then dcMatch = strsub(dcMatch, 2, -2) end
		if (dcMatch and dcMatch == currentOpponent) then
			TetrisAttackCommentText:SetText("Connection lost :(")
			TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
			currentOpponent = ""
			TetrisAttack_TriggerWinAnim()
		end
	elseif (event == "PLAYER_ENTERING_WORLD") then --If somebody zones while playing multiplayer, for God's sake, just disconnect them.
		if (gameInProgress == 1) then
			TetrisAttack_YouLose()
		elseif (gameInProgress == 3) then
			TetrisAttack_SendDisconnect()
			lastOpponent = currentOpponent
			currentOpponent = ""
			TetrisAttack_YouLose()
		end
	elseif (event == "PLAYER_LEAVING_WORLD") then --buh-bye
		if (gameInProgress == 3) then
			TetrisAttack_SendDisconnect()
			currentOpponent = ""
			TetrisAttack_YouLose()	
		end
	--elseif (event == "FRIENDLIST_UPDATE") then
	--	TetrisAttack_FriendsList_Update()
	elseif (event == "RAID_ROSTER_UPDATE") then
		if (numRaidMembers == 0) then
			numRaidMembers = GetNumRaidMembers()
			raidMemberAdded = true
			raidRefreshCooldown = 1
		elseif (GetNumRaidMembers() ~= numRaidMembers) then
			numRaidMembers = GetNumRaidMembers()
			if (raidRefreshCooldown == 0) then
				TetrisAttack_SendRaidGreeting()
				raidRefreshCooldown = 5
			else
				raidMemberAdded = true
			end
		end
		TetrisAttack_UpdateRaidIcons()
	end
end

function TetrisAttack_SendRaidGreeting(askForUpdate)
	inInstance, instanceType = IsInInstance()
	if (GetNumRaidMembers() > 0 and instanceType ~= "pvp") then
		if (askForUpdate) then
			SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "" .. 0 .. "~" .. gameInProgress, "RAID")
		else
			SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "" .. currentVersion .. "~" .. gameInProgress, "RAID")
		end
	end
end

function TetrisAttack_RecvRaidGreeting(name, version, gameType)
	inInstance, instanceType = IsInInstance()
	if (instanceType == "pvp") then return end
	if (version == 0) then
		if (raidRefreshCooldown == 0) then
			TetrisAttack_SendRaidGreeting()
			raidRefreshCooldown = 5
		else
			raidMemberAdded = true
		end
		return
	end
	if (version == currentVersion) then
		if (gameType == 3) then
			friendHasFPL[name] = 3
		else
			friendHasFPL[name] = 1
		end
	elseif (version < currentVersion) then
		friendHasFPL[name] = 2
	else
		friendHasFPL[name] = -1
	end
	TetrisAttack_UpdateRaidIcons()
end

--Updates the challenge window indicators with opponent's settings.
function TetrisAttack_UpdateChallengeInfo(diff, speed)
	TetrisAttackChallengeDifficultyIndicator:SetPoint("CENTER", TetrisAttackChallengeAcceptOverlay, "CENTER", -83 + (diff-1)*48, -2)
	TetrisAttackChallengeSpeedIndicator:SetPoint("CENTER", TetrisAttackChallengeAcceptOverlay, "CENTER", -83 + (speed-1)*3.3, -37)
end

--Update the opponent info window with their stats.
function TetrisAttack_UpdateOpponentInfo(name, diff, speed)
	TetrisAttackOpponentNameDisplay:SetText(name)
	TetrisAttackOpponentDifficultyDisplay:SetText(difficultyText[diff])
	TetrisAttackOpponentDifficultyDisplay:SetTextColor(TetrisAttack_GetDifficultyColor(diff))
	TetrisAttackOpponentSpeedDisplay:SetText(speed)
end

--Change the starting speed to the given value, and update all sliders.
function TetrisAttack_ChangeStartingSpeed(speed)
	startingSpeed = speed
	if (TetrisAttack_GameSpeedChallengeSlider) then TetrisAttack_GameSpeedChallengeSlider:SetValue(speed) end
	if (TetrisAttack_GameSpeedSlider) then TetrisAttack_GameSpeedSlider:SetValue(speed) end
	if (TetrisAttack_MPGameSpeedSlider) then TetrisAttack_MPGameSpeedSlider:SetValue(speed) end
	if (TetrisAttack_FlightpathGameSpeedSlider) then TetrisAttack_FlightpathGameSpeedSlider:SetValue(speed) end
end

--Update the rate of multiplayer data.
function TetrisAttack_ChangeMPInterval(num)
	multiplayerUpdateInterval = 1/num
end

--Change the current difficulty, update all local vars and update all sliders.
function TetrisAttack_ChangeDifficulty(diff)
	if (TetrisAttack_GameDifficultyChallengeSlider) then TetrisAttack_GameDifficultyChallengeSlider:SetValue(diff) end
	if (TetrisAttack_GameDifficultySlider) then TetrisAttack_GameDifficultySlider:SetValue(diff) end
	if (TetrisAttack_MPGameDifficultySlider) then TetrisAttack_MPGameDifficultySlider:SetValue(diff) end
	if (TetrisAttack_FlightpathGameDifficultySlider) then TetrisAttack_FlightpathGameDifficultySlider:SetValue(diff) end
	basePointsPerSpeedIncrease = difficulty[diff][1]
	pointsTillNextSpeedIncrease = difficulty[diff][1]
	maxDeathBuffer = difficulty[diff][2]
	currentDeathBuffer = difficulty[diff][2]
	blockTypes = difficulty[diff][3]
	currentDifficulty = diff
	TetrisAttackDifficultyDisplay:SetText(difficultyText[diff])
	TetrisAttackDifficultyDisplay:SetTextColor(TetrisAttack_GetDifficultyColor(diff))
end

--Pauses the game, obeying various user options and gamestates.
function TetrisAttack_Pause(num, dontSend)
	dontSend = dontSend or false
	if (demoRecord) then
		TetrisAttack_StopRecording()
	end
	isPaused = num or math.abs(isPaused - 1)
	if (isPaused == 1) then 
		TetrisAttack1:EnableKeyboard(0)
		TetrisAttackPaused:Show() 
		if (gameInProgress == 3 and not dontSend) then
			SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "PAUSE" .. "~" .. currentVersion, "WHISPER", currentOpponent)
		end
	else 
		TetrisAttack1:EnableKeyboard(fascistKeyboard)
		TetrisAttackPaused:Hide() 
		if (gameInProgress == 3 and not dontSend) then
			SuperChatThrottleLib:SendAddonMessage("NORMAL", chatMsgGreetingPrefix, "UNPAUSE" .. "~" .. currentVersion, "WHISPER", currentOpponent)
		end
	end
end

--No comment needed. Really.
function TetrisAttack_UpdateCursor()
	cursorTexture:ClearAllPoints()
	cursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackField, "BOTTOMLEFT", (cursorX*32) - 52, (cursorY-1) * 32 - 6)
end

--[[

	These functions all move the cursor around, and contain code that handles recording demos.
	
--]]
function TetrisAttack_CursorLeft()
	if (deathAnim ~= 0) then return end
	if (demoRecord) then
		currentDemo[currentDemoPos*2-1] = 3
		currentDemo[currentDemoPos*2] = GetTime() - demoStart
		currentDemoPos = currentDemoPos+1
		currentDemo[currentDemoPos*2+1] = 0
		currentDemo[currentDemoPos*2+2] = 0
	end
	if (cursorX > 1 and gameInProgress > 0) then -- and isPaused == 0) then
		cursorX = cursorX - 1
		TetrisAttack_UpdateCursor()
	end
end

function TetrisAttack_CursorRight()
	if (deathAnim ~= 0) then return end
	if (demoRecord) then
		currentDemo[currentDemoPos*2-1] = 4
		currentDemo[currentDemoPos*2] = GetTime() - demoStart
		currentDemoPos = currentDemoPos+1
		currentDemo[currentDemoPos*2+1] = 0
		currentDemo[currentDemoPos*2+2] = 0
	end
	if (cursorX < 5 and gameInProgress > 0) then -- and isPaused == 0) then
		cursorX = cursorX + 1
		TetrisAttack_UpdateCursor()
	end
end

function TetrisAttack_CursorUp()
	if (deathAnim ~= 0) then return end
	if (demoRecord) then
		currentDemo[currentDemoPos*2-1] = 1
		currentDemo[currentDemoPos*2] = GetTime() - demoStart
		currentDemoPos = currentDemoPos+1
		currentDemo[currentDemoPos*2+1] = 0
		currentDemo[currentDemoPos*2+2] = 0
	end
	if (cursorY < numRowsVisible and gameInProgress > 0) then -- and isPaused == 0) then
		cursorY = cursorY + 1
		TetrisAttack_UpdateCursor()
	end
end

function TetrisAttack_CursorDown()
	if (deathAnim ~= 0) then return end
	if (demoRecord) then
		currentDemo[currentDemoPos*2-1] = 2
		currentDemo[currentDemoPos*2] = GetTime() - demoStart
		currentDemoPos = currentDemoPos+1
		currentDemo[currentDemoPos*2+1] = 0
		currentDemo[currentDemoPos*2+2] = 0
	end
	if (cursorY > 1 and gameInProgress > 0) then -- and isPaused == 0) then
		cursorY = cursorY - 1
		TetrisAttack_UpdateCursor()
	end
end

--Change the game speed and all related values.
function TetrisAttack_ChangeGameSpeed(speed)
	if (speed <= #gameSpeeds) then
		fieldSpeed					= gameSpeeds[speed][1]
		blockFallSpeed				= gameSpeeds[speed][2]
		blockClearDelay				= gameSpeeds[speed][3]
		bufferDrainSpeed			= gameSpeeds[speed][4]
		bonusBlockChance = speed/1.5
		if (bonusBlockChance > 10) then bonusBlockChance = 10 end
		currentSpeed = speed
		if (freezeTime <= 0) then TetrisAttackSpeedDisplay:SetText(speed) end
	end
end

--Mouse handler. gameInProgress == 4 is unattainable in release copies, it's used for editing puzzles and tutorials on my end.
function TetrisAttack_OnMouseDown(frame, button)
	local column,row  = TetrisAttack_GetBlockAtCursor()
	if (not IsShiftKeyDown()) then
		local column,row  = TetrisAttack_GetBlockAtCursor()
		if (column < 1 or column > 6 or row < 1 or row > numRowsVisible) then return end
		if (gameInProgress == 4) then
			if (button == "LeftButton") then
				blockTable[row][column] = editColor
			else
				blockTable[row][column] = 0
			end
			TetrisAttack_UpdateBlock(row, column)
		elseif (gameInProgress > 0 and isPaused == 0) then
			if ((button == "RightButton" and column > 1) or column == 6) then
				cursorX = column-1
				cursorY = row
				TetrisAttack_UpdateCursor()
				TetrisAttack_CursorFlip()
			elseif ((button == "LeftButton" and column < 6) or column == 1) then
				cursorX = column
				cursorY = row
				TetrisAttack_UpdateCursor()
				TetrisAttack_CursorFlip()
			end
		end
	else
		TetrisAttack1:StartMoving()
	end
end

--Keydown handler. Only ever called if the used has enabled trapping keyboard input.
function TetrisAttack_OnKeyDown(key)
	if (keyToBind ~= 0) then
		TetrisAttack_SavedSettings[keysInOrder[keyToBind]] = key
		keyToBind = keyToBind+1
		if (keyToBind > 7) then 
			TetrisAttack_LoadSettings()
			TetrisAttack1:EnableKeyboard(false)
			TetrisAttackKeyBindingsText:SetText("---Current Key Bindings---")
			keyToBind = 0 
		else
			TetrisAttackKeyBindingsText:SetText("---" .. keyDescsInOrder[keyToBind] .. "---")
		end
		return
	end
	if (key == "ESCAPE") then TetrisAttack_Hide(1) end
	if (demoPlay) then return end
	if (key == upKey) then
		TetrisAttack_CursorUp()
	elseif (key == leftKey) then
		TetrisAttack_CursorLeft()
	elseif (key == rightKey) then
		TetrisAttack_CursorRight()
	elseif (key == downKey) then
		TetrisAttack_CursorDown()
	elseif (key == flipKey) then
		TetrisAttack_CursorFlip()
	elseif (key == bumpKey and (gameInProgress ~= 2 or currentPuzzleIndex == 14)) then
		speedyShift = 1
	elseif (key == pauseKey) then
		TetrisAttack_Pause()
	else
		local key1, key2 = GetBindingKey("TAOPEN")
		if (key == key1 or key == key2) then
			TetrisAttack_Toggle(1)
		end
	end
end

--Trigger all the garbage that CountGarbage() told us is connected to the given block.
--[[local--]] function TetrisAttack_TriggerGarbage(row, column, combo)
	delay = delay or blockClearDelay
	local top, bottom, left, right = 0,0,0,0
	local matchTable = {}
	local index = 0
	local numBlocksTotal = TetrisAttack_CountGarbage(row, column, matchTable)
	local numBlocksProcessed = 0
	for i = 1, #matchTable do
		index = matchTable[i]
		top = garbageTable[index]["top"]
		bottom = garbageTable[index]["bottom"]
		left = garbageTable[index]["left"]
		right = garbageTable[index]["right"]
		garbageTable[index]["delay"] = blockClearDelay + (numBlocksProcessed * blockClearDelay/5)
		numBlocksProcessed = numBlocksProcessed + ((top-bottom)+1)*((right-left)+1)
		garbageTable[index]["delayAfter"] = (numBlocksTotal - numBlocksProcessed) * blockClearDelay/5 + .01
		garbageTable[index]["combo"] = combo
		garbageTable[index]["index"] = 1
	end
end

--Update all garbage in the same manner of falling blocks, except with added code to handle the "Oompf-Oompf-Oompf" when you destroy it.
--[[local--]] function TetrisAttack_UpdateGarbage(time)
	local skipped = 0
	local collision = false
	local bottom, top, left, right, offset, delay, index, combo = 0,0,0,0,0,0,0,0
	local highestTop = 0
	local sides = ""
	for i = 1, numGarbageBlocks do
		collision = false
		bottom = garbageTable[i]["bottom"]
		top = garbageTable[i]["top"]
		if (highestTop < top) then highestTop = top end
		left = garbageTable[i]["left"]
		right = garbageTable[i]["right"]
		offset = garbageTable[i]["offset"]
		delay = garbageTable[i]["delay"]
		delayAfter = garbageTable[i]["delayAfter"]
		index = garbageTable[i]["index"]
		combo = garbageTable[i]["combo"]
		if (bottom == 0) then
			skipped = skipped + 1
		elseif (delay > 0) then
			delay = delay - time
			if (delay <= 0) then
				local row = bottom + math.floor((index-1)/6)
				local column = mod(index-1, 6)+left
				garbageTable[i]["index"] = index + 1
				if (playSounds == 1) then PlaySoundFile(soundEffects["garbageClear"]) end
				if (row <= numRows) then
					if (row == bottom) then
						blockTable[row][column] = math.random(blockTypes)
						if (column > 2) then
							while((blockTable[row][column] == blockTable[row][column-1]) and
									(blockTable[row][column] == blockTable[row][column-2])) do			
								blockTable[row][column] = math.random(blockTypes)
							end
						end
						TetrisAttack_UpdateBlock(row, column)
					else
						blockTable[row][column] = 7
						if (row <= numRowsVisible) then
							sides = "" .. ((((column == left) and "L") or ((column == right) and "R")) or "")
							sides = sides .. (((row == top) and "T") or "")
							sides = sides .. (((row == bottom+1) and "B") or "")
							TetrisAttack_UpdateGarbageBlock(row, column, sides)
						end
					end
				end
				if (index+1 <= ((top-bottom)+1)*((right-left)+1)) then
					delay = blockClearDelay/5
				end
			end
			garbageTable[i]["delay"] = delay
			garbageTable[i]["offset"] = offset
		elseif (delayAfter > 0) then
			delayAfter = delayAfter - time
			garbageTable[i]["delayAfter"] = delayAfter
			if (delayAfter <= 0) then
				garbageTable[i]["bottom"] = 0
				garbageTable[i]["top"] = 0
				if (top ~= bottom) then
					TetrisAttack_CreateBabyGarbage(top, bottom+1, left, right)
				end
				for fallBlock = left, right do
					TetrisAttack_StartFalling(bottom, fallBlock, combo)
				end
			end
		elseif (TetrisAttack_isAreaEmpty(bottom-1, left, right)) then -- we're falling
			if (offset == 0) then	-- falling for the first time
				for row = bottom, top do
					for column = left, right do
						blockTable[row][column] = -7
						if (row == top) then TetrisAttack_StartFalling(row+1, column, 1, 0) end
					end
				end
			end
			offset = offset + blockFallSpeed/GetFramerate()
			if (offset > 32) then
				while (offset > 32) do
					if (not TetrisAttack_isAreaEmpty(bottom-2, left, right)) then -- collision
						if (playSounds == 1) then 
							if (top-bottom < 2) then PlaySoundFile(soundEffects["garbageSmall"]) 
							else PlaySoundFile(soundEffects["garbageLarge"]) 
							end
						end
						offset = 0
						for row = bottom, top do
							for column = left, right do
								TetrisAttack_ResetBlockPos(row, column)
								blockTable[row-1][column] = 7
							end
						end
						for column = left, right do
							TetrisAttack_UpdateBlock(top, column)
							if (blockTable[top][column] <= 0) then blockTable[top][column] = 0 end
						end
					else
						offset = offset - 32
						for column = left, right do 
							TetrisAttack_ResetBlockPos(top, column) 
							TetrisAttack_UpdateBlock(top, column)
							if (blockTable[top][column] <= 0) then blockTable[top][column] = 0 end
							blockTable[bottom-1][column] = -7
						end
						
						for row = bottom, top do
							for column = left, right do
								--TetrisAttack_UpdateBlock(row-1, column, 7)
								if (row <= numRowsVisible) then
									blockTextures[row-1][column]:ClearAllPoints()
									blockTextures[row-1][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-2) * 32) - (offset - 10) + 32)
								end
							end
						end
					end
					garbageTable[i]["bottom"] = bottom - 1
					garbageTable[i]["top"] = top - 1
					garbageTable[i]["offset"] = offset
				end
				TetrisAttack_UpdateGarbageChunk(i)
			else
				garbageTable[i]["offset"] = offset
				for row = bottom, top do
					for column = left, right do
						if (row <= numRowsVisible) then
							blockTextures[row][column]:ClearAllPoints()
							blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) - (offset - 10) + 32)
						end
					end
				end
			end
		elseif (offset > 0) then -- surprise collision
			if (playSounds == 1) then 
				if (top-bottom < 2) then PlaySoundFile(soundEffects["garbageSmall"]) 
				else PlaySoundFile(soundEffects["garbageLarge"]) 
				end
			end
			offset = 0
			for row = bottom, top do
				for column = left, right do
					TetrisAttack_ResetBlockPos(row, column)
					blockTable[row][column] = 7
				end
			end
			garbageTable[i]["offset"] = 0
		end
	end
	if (skipped == numGarbageBlocks) then
		numGarbageBlocks = 0
	end
	if (numGarbageToCreate > 0) then --Create garbage if there's enough room.
		local width = garbageQueue[1][1]
		local height = garbageQueue[1][2]
		if (height > 5 and ohShitTimer > 0) then
			ohShitTimer = ohShitTimer - time
		else
			if (highestTop <= (numRows-1) - height) then
				tremove(garbageQueue, 1)
				numGarbageToCreate = numGarbageToCreate - 1
				TetrisAttack_CreateGarbage(width, height)
			end
		end
	end
end

--Make some trash and then drop it on the player.
function TetrisAttack_CreateGarbage(width, height)
	numGarbageBlocks = numGarbageBlocks + 1
	if (currentDeathBuffer < .5) then currentDeathBuffer = .5 end
	local leftEdge = math.random((6-width) + 1)
	if (width == 2) then
		leftEdge = corruptionIndex
		corruptionIndex = corruptionIndex+2
		if (corruptionIndex > 5) then corruptionIndex = 1 end
	end
	local rightEdge = (leftEdge + width) - 1
	local topEdge = numRows - 1
	local bottomEdge = (topEdge - height) + 1
	if (not garbageTable[numGarbageBlocks]) then
		garbageTable[numGarbageBlocks] = {["top"] = topEdge,
										["left"] = leftEdge,
										["bottom"] = bottomEdge,
										["right"] = rightEdge,
										["offset"] = 0,
										["delay"] = 0,
										["delayAfter"] = 0,
										["index"] = 0,
										["combo"] = 0}
	else
		garbageTable[numGarbageBlocks]["top"] = topEdge
		garbageTable[numGarbageBlocks]["left"] = leftEdge
		garbageTable[numGarbageBlocks]["bottom"] = bottomEdge
		garbageTable[numGarbageBlocks]["right"] = rightEdge
		garbageTable[numGarbageBlocks]["offset"] = 0
		garbageTable[numGarbageBlocks]["delay"] = 0
		garbageTable[numGarbageBlocks]["delayAfter"] = 0
		garbageTable[numGarbageBlocks]["index"] = 0
		garbageTable[numGarbageBlocks]["combo"] = 0
	end
end

--Make a subset of garbage. Used when a big enough chunk was cleared and spawns a little one in its place.
function TetrisAttack_CreateBabyGarbage(topEdge, bottomEdge, leftEdge, rightEdge)
	numGarbageBlocks = numGarbageBlocks + 1
	if (not garbageTable[numGarbageBlocks]) then
		garbageTable[numGarbageBlocks] = {["top"] = topEdge,
										["left"] = leftEdge,
										["bottom"] = bottomEdge,
										["right"] = rightEdge,
										["offset"] = 0,
										["delay"] = 0,
										["delayAfter"] = 0,
										["index"] = 0,
										["combo"] = 0}
	else
		garbageTable[numGarbageBlocks]["top"] = topEdge
		garbageTable[numGarbageBlocks]["left"] = leftEdge
		garbageTable[numGarbageBlocks]["bottom"] = bottomEdge
		garbageTable[numGarbageBlocks]["right"] = rightEdge
		garbageTable[numGarbageBlocks]["offset"] = 0
		garbageTable[numGarbageBlocks]["delay"] = 0
		garbageTable[numGarbageBlocks]["delayAfter"] = 0
		garbageTable[numGarbageBlocks]["index"] = 0
		garbageTable[numGarbageBlocks]["combo"] = 0
	end
end

--HA HA, YOU SUCK
function TetrisAttack_YouLose()
	TetrisAttack1:EnableKeyboard(0)
	TetrisAttackMainMenu:Show()
	TetrisAttackEndless:Hide()
	TetrisAttack2:Hide()
	cursorTexture:Hide()
	if (gameInProgress == 1) then --endless mode
		if (currentDifficulty == 1) then
			playerScore = math.floor(playerScore * .8)
		elseif (currentDifficulty == 3) then
			playerScore = math.floor(playerScore * 1.2)
		end
		TetrisAttack_AddNewPersonalScore(playerScore, currentDifficulty)
	elseif (gameInProgress == 3) then --multiplayer
        PanelTemplates_SetTab(TetrisAttackMainMenu, 2);
        TetrisAttackMainMenuTabPage1:Hide();
        TetrisAttackMainMenuTabPage2:Show();
        TetrisAttackMainMenuTabPage3:Hide();
        MainMenuHeader:SetText("Multiplayer")
    end
	gameInProgress = 0
	TetrisAttack_SendRaidGreeting()
	if (currentOpponent ~= "") then
		lastOpponent = currentOpponent
	end
	currentOpponent = ""
	inDanger = 0
	TetrisAttackFieldBackground:SetTexture(0.05, 0.05, 0.05, 0.75)
	TetrisAttack_UpdateComboAnims(0, true) --clear all combos
end

--Show/hide the +/- 20% messages for easy and hard modes.
function TetrisAttack_UpdateEndlessScoreDisplay(score)
	TetrisAttackScoreDisplay:SetText(score)
	if (currentDifficulty == 1) then
		TetrisAttackScoreEasy:Show()
	elseif (currentDifficulty == 3) then
		TetrisAttackScoreHard:Show()
	end
end

--Process a high score that we received from a friend.
function TetrisAttack_AddNewFriendScore(score, name, difficulty)
	--out("Adding new friend score: " .. name .. ", " .. score)
	local tempName, tempScore, tempDiff
	local noSeed = false
	for i = 6, 10 do
		if (name == TetrisAttack_EndlessHighScores[i]["name"]) then
			if (score <= TetrisAttack_EndlessHighScores[i]["score"]) then
				return
			else
				TetrisAttack_EndlessHighScores[i]["score"] = score
				TetrisAttack_EndlessHighScores[i]["difficulty"] = difficulty
				noSeed = true
			end
		end
	end
	if (score > TetrisAttack_EndlessHighScores[10]["score"]) then
	if (not noSeed) then
		TetrisAttack_EndlessHighScores[10]["name"] = name
		TetrisAttack_EndlessHighScores[10]["difficulty"] = difficulty
		TetrisAttack_EndlessHighScores[10]["score"] = score
	end
		for i = 9, 6, -1 do
			if (TetrisAttack_EndlessHighScores[i+1]["score"] > TetrisAttack_EndlessHighScores[i]["score"]) then
				tempName = TetrisAttack_EndlessHighScores[i]["name"]
				tempScore = TetrisAttack_EndlessHighScores[i]["score"]
				tempDiff = TetrisAttack_EndlessHighScores[i]["difficulty"]
				TetrisAttack_EndlessHighScores[i]["name"] = TetrisAttack_EndlessHighScores[i+1]["name"]
				TetrisAttack_EndlessHighScores[i]["score"] = TetrisAttack_EndlessHighScores[i+1]["score"]
				TetrisAttack_EndlessHighScores[i]["difficulty"] = TetrisAttack_EndlessHighScores[i+1]["difficulty"]
				TetrisAttack_EndlessHighScores[i+1]["name"] = tempName
				TetrisAttack_EndlessHighScores[i+1]["score"] = tempScore
				TetrisAttack_EndlessHighScores[i+1]["difficulty"] = tempDiff
			end 
		end
	end
	TetrisAttack_RefreshEndlessScoreDisplay()
end

--Process one of our own high scores.
function TetrisAttack_AddNewPersonalScore(score, diff)
	local tempName, tempScore, tempDiff
	local newScorePos = 5
	local noSeed = false
	for i = 1, 5 do
		if (score == TetrisAttack_EndlessHighScores[i]["score"]) then
			return
		end
	end
	if (score > TetrisAttack_EndlessHighScores[5]["score"]) then
		TetrisAttack_EndlessHighScores[5]["name"] = currentPlayerName
		TetrisAttack_EndlessHighScores[5]["difficulty"] = diff
		TetrisAttack_EndlessHighScores[5]["score"] = score
		for i = 4, 1, -1 do
			if (TetrisAttack_EndlessHighScores[i+1]["score"] > TetrisAttack_EndlessHighScores[i]["score"]) then
				newScorePos = i
				tempName = TetrisAttack_EndlessHighScores[i]["name"]
				tempScore = TetrisAttack_EndlessHighScores[i]["score"]
				tempDiff = TetrisAttack_EndlessHighScores[i]["difficulty"]
				TetrisAttack_EndlessHighScores[i]["name"] = TetrisAttack_EndlessHighScores[i+1]["name"]
				TetrisAttack_EndlessHighScores[i]["score"] = TetrisAttack_EndlessHighScores[i+1]["score"]
				TetrisAttack_EndlessHighScores[i]["difficulty"] = TetrisAttack_EndlessHighScores[i+1]["difficulty"]
				TetrisAttack_EndlessHighScores[i+1]["name"] = tempName
				TetrisAttack_EndlessHighScores[i+1]["score"] = tempScore
				TetrisAttack_EndlessHighScores[i+1]["difficulty"] = tempDiff
			end 
		end
	end
	TetrisAttack_RefreshEndlessScoreDisplay()
	if (newScorePos == 1) then
		if (playSounds == 1) then PlaySoundFile(soundEffects["ding"]) end
		TetrisAttackEndlessScores:SetText("NEW HIGH SCORE!")
		TetrisAttack_BroadcastNewHighScore(currentPlayerName, score, diff)
	end
end

--Change the display mode of the main scoreboard.
function TetrisAttack_ChangeEndlessScoreDisplay(min, max, text)
	endlessScoreMin = min
	endlessScoreMax = max
	endlessScoreText = text
	TetrisAttack_RefreshEndlessScoreDisplay()
end

--Redraw the endless-mode high scores display.
function TetrisAttack_RefreshEndlessScoreDisplay()
	TetrisAttack_ClearEndlessScoreboard()
	for i = endlessScoreMin, endlessScoreMax do
		TetrisAttack_UpdateEndlessScoreboard(TetrisAttack_EndlessHighScores[i]["name"], TetrisAttack_EndlessHighScores[i]["score"], TetrisAttack_EndlessHighScores[i]["difficulty"])
	end
	local _G = getfenv()
	local nameText
	local scoreText
	for i = 1, 5 do
		nameText = _G["TetrisAttackEndlessName" .. i]
		scoreText = _G["TetrisAttackEndlessScore" .. i]
		if (endlessScoreDisplay[i]["score"] > 0) then
			nameText:Show()
			scoreText:Show()
			nameText:SetText(endlessScoreDisplay[i]["name"] .. ":")
			scoreText:SetText(endlessScoreDisplay[i]["score"])
			scoreText:SetTextColor(TetrisAttack_GetDifficultyColor(endlessScoreDisplay[i]["difficulty"]))
		end
	end
	TetrisAttackEndlessScores:SetText(endlessScoreText)
end

--yeah, this really doesn't need a comment.
function TetrisAttack_ClearEndlessScoreboard()
	local _G = getfenv()
	for i = 1, 5 do
		nameText = _G["TetrisAttackEndlessName" .. i]
		scoreText = _G["TetrisAttackEndlessScore" .. i]
		endlessScoreDisplay[i]["name"] = ""
		endlessScoreDisplay[i]["score"] = 0
		endlessScoreDisplay[i]["difficulty"] = 0
		nameText:Hide()
		scoreText:Hide()
	end
end

--Actually add a score to the current scoreboard display table and bubble it to the top.
function TetrisAttack_UpdateEndlessScoreboard(name, score, diff)
	if (not name or not score or not diff) then return end
	local tempName, tempScore, tempDiff
	if (score > endlessScoreDisplay[5]["score"]) then
		endlessScoreDisplay[5]["name"] = name
		endlessScoreDisplay[5]["score"] = score
		endlessScoreDisplay[5]["difficulty"] = diff
		for i = 4, 1, -1 do
			if (endlessScoreDisplay[i+1]["score"] > endlessScoreDisplay[i]["score"]) then
				tempName = endlessScoreDisplay[i]["name"]
				tempScore = endlessScoreDisplay[i]["score"]
				tempDiff = endlessScoreDisplay[i]["difficulty"]
				endlessScoreDisplay[i]["name"] = endlessScoreDisplay[i+1]["name"]
				endlessScoreDisplay[i]["score"] = endlessScoreDisplay[i+1]["score"]
				endlessScoreDisplay[i]["difficulty"] = endlessScoreDisplay[i+1]["difficulty"]
				endlessScoreDisplay[i+1]["name"] = tempName
				endlessScoreDisplay[i+1]["score"] = tempScore
				endlessScoreDisplay[i+1]["difficulty"] = tempDiff
			end 
		end
	end
end

--Start the winner animation.
function TetrisAttack_TriggerWinAnim()
	for column = 1, 6 do
		newBlockTextures[column]:Hide()
	end
	if (playSounds == 1) then PlaySoundFile(soundEffects["victory"]) end
	cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youwin.tga")
	cursorTexture:SetWidth(128)
	cursorTexture:SetHeight(128)
	TetrisAttack_DropCursor()
	deathAnimVelocity = -128
	deathAnimOffset = numRowsVisible*32
	deathAnim = -1
	if (gameInProgress == 3) then
		for column = 1, 6 do
			newBlockTexturesMP[column]:Hide()
		end
		MPcursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youlose.tga")
		MPcursorTexture:SetWidth(128)
		MPcursorTexture:SetHeight(128)
		opponentDeathAnimVelocity = 24
		opponentDeathAnimAccel = -384
		opponentDeathAnimOffset = fieldOffset
		opponentDeathAnim = 1
	end
end

--HOORAY
function TetrisAttack_YouWin()
	TetrisAttack1:EnableKeyboard(0)
	TetrisAttackMainMenu:Show()
	TetrisAttackEndless:Hide()
	TetrisAttack2:Hide()
	cursorTexture:Hide()
	if (gameInProgress == 3) then
        PanelTemplates_SetTab(TetrisAttackMainMenu, 2);
        TetrisAttackMainMenuTabPage1:Hide();
        TetrisAttackMainMenuTabPage2:Show();
        TetrisAttackMainMenuTabPage3:Hide();
        MainMenuHeader:SetText("Multiplayer")
    end
	gameInProgress = 0
	TetrisAttack_SendRaidGreeting()
	if (currentOpponent ~= "") then
		lastOpponent = currentOpponent
	end
	currentOpponent = ""
	inDanger = 0
	TetrisAttackFieldBackground:SetTexture(0.05, 0.05, 0.05, 0.75)
	TetrisAttack_UpdateComboAnims(0, true)
end

--Player wins a puzzle.
function TetrisAttack_WinPuzzle()
	if (currentPuzzleIndex > numTutorials) then
		if (playSounds == 1) then PlaySoundFile(soundEffects["puzzleWin"]) end
		cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youwin.tga")
		cursorTexture:SetWidth(128)
		cursorTexture:SetHeight(128)
		TetrisAttack_DropCursor()
		gameInProgress = 0
		TetrisAttackFieldBackground:SetTexture(0, .6, 0, .75)
	else
		TetrisAttack_StartPuzzle(currentPuzzleIndex)
	end
	TetrisAttack_UpdateComboAnims(0, true)
end

--OK, honestly, this is just getting silly, I'm not going to comment functions like this.
function TetrisAttack_Rematch()
	if (lastOpponent ~= "") then
		currentOpponent = lastOpponent
		TetrisAttack_SendChallenge(currentOpponent)
	else
		if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: No matches played recently!", 1, 1, 1) end
	end
end

function TetrisAttack_LosePuzzle()
	if (currentPuzzleIndex > numTutorials) then
		if (playSounds == 1) then PlaySoundFile(soundEffects["puzzleLose"]) end
		cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youlose.tga")
		cursorTexture:SetWidth(128)
		cursorTexture:SetHeight(128)
		TetrisAttack_DropCursor()
		gameInProgress = 0
		TetrisAttackFieldBackground:SetTexture(.6, 0, 0, .75)
	else
		TetrisAttack_StartPuzzle(currentPuzzleIndex)
	end
	TetrisAttack_UpdateComboAnims(0, true)
end

--Set the cursor to drop from above the screen to animate the win/loss messages.
function TetrisAttack_DropCursor()
	cursorAnimPos = 64
	cursorAnimVelocity = -128
	cursorTexture:ClearAllPoints()
	cursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "TOPLEFT", (3*32) - 52, cursorAnimPos)
	cursorTexture:SetHeight(1)
	cursorTexture:SetTexCoord(0,1,0,.01)
	if (gameInProgress == 3) then TetrisAttack_DropOpponentCursor() end
end

--Do the same for the opponent's cursor.
function TetrisAttack_DropOpponentCursor()
	MPcursorTexture:ClearAllPoints()
	MPcursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackOpponentFieldBorder, "TOPLEFT", (3*32) - 52, cursorAnimPos)
	MPcursorTexture:SetHeight(1)
	MPcursorTexture:SetTexCoord(0,1,0,.01)
end

--Wipe the flightpath scores clean.
function TetrisAttack_ClearFlightpathScoreboard()
	local _G = getfenv()
	for i = 1, 5 do
		nameText = _G["TetrisAttackFlightpathName" .. i]
		scoreText = _G["TetrisAttackFlightpathScore" .. i]
		topFriendScores[i]["name"] = ""
		topFriendScores[i]["score"] = 0
		nameText:Hide()
		scoreText:Hide()
	end
end

--Physically add a score to the board and bubble it up from the bottom.
function TetrisAttack_UpdateFlightpathScoreboard(name, score, diff)
	local tempName, tempScore, tempDiff
	for i = 1, 5 do
		if ((score == topFriendScores[i]["score"]) and (name == topFriendScores[i]["name"])) then
			return
		end
	end
	if (score > topFriendScores[5]["score"]) then
		topFriendScores[5]["name"] = name
		topFriendScores[5]["score"] = score
		topFriendScores[5]["difficulty"] = diff
		for i = 4, 1, -1 do
			if (topFriendScores[i+1]["score"] > topFriendScores[i]["score"]) then
				tempName = topFriendScores[i]["name"]
				tempScore = topFriendScores[i]["score"]
				tempDiff = topFriendScores[i]["difficulty"]
				topFriendScores[i]["name"] = topFriendScores[i+1]["name"]
				topFriendScores[i]["score"] = topFriendScores[i+1]["score"]
				topFriendScores[i]["difficulty"] = topFriendScores[i+1]["difficulty"]
				topFriendScores[i+1]["name"] = tempName
				topFriendScores[i+1]["score"] = tempScore
				topFriendScores[i+1]["difficulty"] = tempDiff
			end 
		end
	end
	local _G = getfenv()
	local nameText
	local scoreText
	for i = 1, 5 do
		nameText = _G["TetrisAttackFlightpathName" .. i]
		scoreText = _G["TetrisAttackFlightpathScore" .. i]
		if (topFriendScores[i]["score"] > 0) then
			nameText:Show()
			scoreText:Show()
			nameText:SetText(topFriendScores[i]["name"] .. ":")
			scoreText:SetText(topFriendScores[i]["score"])
			scoreText:SetTextColor(TetrisAttack_GetDifficultyColor(topFriendScores[i]["difficulty"]))
		end
	end
end

--Start a new flightpath game,unless we're already doing something.
function TetrisAttack_NewGameFlightpath()
	TetrisAttackFlightpathPrompt:Hide()
	if (gameInProgress ~= 3) then
		TetrisAttack_NewGameEndless()
		taxiGameInProgress = true
		TetrisAttackFlightpath:Show()
		if (useFriendsList == 1) then
			for name, enabled in pairs(friendHasFPL) do
				if (enabled == 1) then
					TetrisAttack_SendHighScoreRequest(name, srcName .. "~" .. destName)
				end
			end
		end
	end
end

--Reset all variables that could possibly cause problems if a game was started when they were still holding non-default information.
function TetrisAttack_ScrubGameVariables()
	TetrisAttackCommentText:SetText("")
	cursorX = 3
	cursorY = 8
	MPcursorX = 3
	MPcursorY = 8
	playerScore = 0
	speedyShift = 0
	cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\TAcursor")
	cursorTexture:SetTexCoord(0,1,0,1)
	cursorTexture:SetWidth(128)
	cursorTexture:SetHeight(64)
	MPcursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\TAcursor")
	MPcursorTexture:SetTexCoord(0,1,0,1)
	MPcursorTexture:SetWidth(128)
	MPcursorTexture:SetHeight(64)
	numBlocksLeft = 0
	taxiGameInProgress = false
	animsToSend = ""
	currentDemo = {}
	currentDemoPos = 0
	demoStart = 0
	demoPlay = false
	demoRecord = false
	cursorAnimPos = 0
	cursorAnimVelocity = 0
	currentDeathBuffer = maxDeathBuffer
	basePointsPerSpeedIncrease = difficulty[currentDifficulty][1]
	pointsTillNextSpeedIncrease = difficulty[currentDifficulty][1]
	allowInput = false
	fieldOffset = 0
	multiplayerUpdateTimer = 0
	numBlocksToClear = 0
	numBlocksToCheck = 0
	fallingBlocksPerColumn = {0, 0, 0, 0, 0, 0}
	numGarbageBlocks = 0
	numGarbageToCreate = 0
	numCombos = 0
	TetrisAttack_ClearTables()
	TetrisAttack_UpdateComboAnims(0, true)
	heartbeat = 0
	preGameTimer = -1
	frameUpdates = 0
	MPgarbageCode = ""
	inDanger = 0
	deathAnim = 0
	deathAnimOffset = 0
	deathAnimVelocity = 0
	deathAnimAccel = 0
	ohShitTimer = 0
	dangerPulseTimer = 0
	lastHandshakeTimer = 0
	bonusBlockChance = 0
	totalBonusThisFrame = 0
	totalBonusOnField = 0
	sendSpecial = "0"
	tranqTimer = 0
	bloodTimer = 0
	corrTimer = 0
	opponentTranqTimer = 0
	opponentBloodTimer = 0
	opponentCorrTimer = 0
	opponentDangerPulseTimer = 0
	opponentDeathAnim = 0
	opponentDeathAnimOffset = 0
	opponentDeathAnimVelocity = 0
	opponentDeathAnimAccel = 0
	TetrisAttackOpponentFieldBackground:SetTexture(.05, .05, .05, .75)
	freezeTime = 0
	TetrisAttackComment1:SetText("")
end

--Clear all the important data tables to make sure you don't start a new game and immediately get flattened by garbage from a game you just finished playing.
function TetrisAttack_ClearTables()
	for i = 1, #blocksToClear do
		blocksToClear[i]["row"] = 0
		blocksToClear[i]["combo"] = 0
		blocksToClear[i]["column"] = 0
		blocksToClear[i]["delay"] = 0
	end
	for i = 1, #blocksToCheck do
		blocksToCheck[i][1] = 0
		blocksToCheck[i][2] = 0
		blocksToCheck[i][3] = 0
	end
	for i = 1, #garbageTable do
		garbageTable[i]["top"] = 0
		garbageTable[i]["left"] = 0
		garbageTable[i]["bottom"] = 0
		garbageTable[i]["right"] = 0
		garbageTable[i]["offset"] = 0
		garbageTable[i]["delay"] = 0
		garbageTable[i]["delayAfter"] = 0
		garbageTable[i]["index"] = 0
		garbageTable[i]["combo"] = 0
	end
	for i = 1, #currentCombos do
		currentCombos[i] = 0
	end
	for i = 1, #garbageQueue do
		garbageQueue[i][1] = 0
		garbageQueue[i][2] = 0
	end
	for i = 1, 6 do
		for j = 1, #(fallingTable[i]) do
			fallingTable[i][j]["row"] = 0
			fallingTable[i][j]["column"] = 0
			fallingTable[i][j]["offset"] = 0
			fallingTable[i][j]["color"] = 0
			fallingTable[i][j]["combo"] = 0
			fallingTable[i][j]["delay"] = 0
		end
	end
	for column = 1, 6 do
		for row = 1, numRows do
			blockTable[row][column] = 0
		end
	end
end

--Set variables to temporary values for speedy processing of the pregame block setup.
function TetrisAttack_BeginPregameCleanup()
	pregameCleanup = 1
	playSounds = -playSounds
	blockClearDelay = .05
end

function TetrisAttack_ShowMainMenu()
	TetrisAttackMainMenu:Show()
	TetrisAttackPuzzleInfo:Hide()
	TetrisAttackFlightpath:Hide()
	TetrisAttackComment1:SetText("")
	TetrisAttackCommentText:SetText("")
	gameInProgress = 0
	TetrisAttack1:EnableKeyboard(0)
end

--Start a new endless game, ensuring that all variables and textures are what they should be, and generating a random starting board from either a given seed or the current time.
function TetrisAttack_NewGameEndless()
	gameInProgress = 1
	TetrisAttackMainMenu:Hide()
	TetrisAttackEndless:Show()
	TetrisAttack1:Show()
	TetrisAttackField:SetAlpha(0)
	TetrisAttackField:SetPoint("CENTER", TetrisAttack1, "CENTER", 0, 0)
	TetrisAttackFlightpath:Hide()
	TetrisAttackOpponentField:SetAlpha(0)
	TetrisAttackPuzzleInfo:Hide()
	TetrisAttackFieldBackground:SetTexture(0.05, 0.05, 0.05, 0.75)
	TetrisAttack_ScrubGameVariables()
	TetrisAttack_UpdateEndlessScoreDisplay(0)
	TetrisAttack_ChangeGameSpeed(startingSpeed)
	TetrisAttack_BeginPregameCleanup()
	TetrisAttackScoreEasy:Hide()
	TetrisAttackScoreHard:Hide()
	preGameTimer = 5
	TetrisAttackCommentText:SetText("Ready!")
	TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
	TetrisAttack_Pause(0)
	if (multiplayerSeed > 0) then 
		math.randomseed(multiplayerSeed)
		multiplayerSeed = 0
	end
	for row = 1, numRows do
		for column = 1, 6 do
			blockTable[row][column] = 0
			if (row < 7) then
				blockTable[row][column] = math.random(blockTypes)
				local colorIndex = abs(blockTable[row][column])
				if (blockTable[row][column] <= 0 or useClassIcons == 1)
				then
					blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
					blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
				elseif (colorizeBlocks == 1) then
					blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
					blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
				end
				blockTextures[row][column]:Show()
				blockTextures[row][column]:SetAlpha(1)
				blockTextures[row][column]:SetTexCoord(0,1,0,1)
				blockTextures[row][column]:SetHeight(32)
				blockTextures[row][column]:ClearAllPoints()
				blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) + 10 + 32)
			elseif (row <= numRowsVisible) then
				blockTextures[row][column]:Show()
				blockTextures[row][column]:SetAlpha(1)
				blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block0")
				blockTextures[row][column]:ClearAllPoints()
				blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) + 10 + 32)
				blockTextures[row][column]:SetTexCoord(0,1,0,1)
				blockTextures[row][column]:SetHeight(32)
			end
			lastBlock = block
		end
	end
	for column=1, 6 do
		newBlockTable[column] = math.random(blockTypes) --Possible values are 0-8: 0 empty 1-7 blocks 8 garbage
		local colorIndex = abs(newBlockTable[column])
		
		if (newBlockTable[column] <= 0 or useClassIcons == 1)
		then
			newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
			newBlockTextures[column]:SetVertexColor(blockColor[colorIndex][1]*.5,blockColor[colorIndex][2]*.5,blockColor[colorIndex][3]*.5, 1)
		elseif (colorizeBlocks == 1) then
			newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
			newBlockTextures[column]:SetVertexColor(blockColor[colorIndex][1]*.5,blockColor[colorIndex][2]*.5,blockColor[colorIndex][3]*.5, 1)
		end
		newBlockTextures[column]:ClearAllPoints()
		newBlockTextures[column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20,10)
		newBlockTextures[column]:SetTexCoord(0,1,0,0)
		newBlockTextures[column]:SetHeight(.01)
		newBlockTextures[column]:Show()
		lastBlock = block
	end
	cursorTexture:Show()
	cursorTexture:ClearAllPoints()
	cursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", (cursorX*32) - 52, ((cursorY-1) * 32) - 6)
	MPcursorTexture:Show()
	MPcursorTexture:ClearAllPoints()
	MPcursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackOpponentFieldBorder, "BOTTOMLEFT", (MPcursorX*32) - 52, ((MPcursorY-1) * 32) - 6)
	for row = 1, 7 do
		for column = 1, 6 do
			TetrisAttack_CheckForMatches(row, column)
		end
	end
end

--Translate a cursor position into a block index.
function TetrisAttack_GetBlockAtCursor() -- Thanks MapNotes for showing me how to get cursor position relative to a frame!
	local x, y = GetCursorPosition();
	x = x / (TetrisAttackField:GetEffectiveScale());
	y = y / (TetrisAttackField:GetEffectiveScale());
	local centerX, centerY = TetrisAttackField:GetCenter();
	x = (x - (centerX - (216/2))) - 12;
	y = 430 - (centerY + (408/2) - y );
	x = math.floor(x / 32) + 1
	y = math.floor(y / 32)
	return x, y
end

function TetrisAttack_SetEditColor(num)
	editColor = num
end

--Change how many moves are left in puzzle mode.
function TetrisAttack_ChangeRemainingMoves(num)
	numMovesLeft = numMovesLeft + num
	TetrisAttackScoreDisplay:SetText(numMovesLeft)
end

function TetrisAttack_RecordDemo()
	demoRecord = true
	demoPlay = false
	currentDemoPos = 1
	demoStart = GetTime()
	currentDemo = {0, 0}
end

function TetrisAttack_StopRecording()
	demoRecord = false
	if (not TetrisAttack_SavedPuzzles[currentPuzzleIndex]["demo"]) then TetrisAttack_SavedPuzzles[currentPuzzleIndex]["demo"] = {} end
	for Key, Val in pairs(currentDemo) do
		TetrisAttack_SavedPuzzles[currentPuzzleIndex]["demo"][Key] = Val
	end
end

--Play the current demo.
function TetrisAttack_PlayDemo()
	TetrisAttack_StartPuzzle(currentPuzzleIndex)
	if (currentDemo[1] ~= 0) then
		demoPlay = true
		demoStart = GetTime()
		currentDemoPos = 1
		TetrisAttack1:EnableKeyboard(1)
	end
end

--Play the current demo.
function TetrisAttack_PlayCustomDemo()
	TetrisAttack_StartPuzzle(currentPuzzleIndex, true)
	if (currentDemo[1] ~= 0) then
		demoPlay = true
		demoStart = GetTime()
		currentDemoPos = 1
		TetrisAttack1:EnableKeyboard(1)
	end
end

--Translate a demo action index into actually doing something.
function TetrisAttack_DecodeDemoAction(action)
	if (action == 1) then
		TetrisAttack_CursorUp()
	elseif (action == 2) then
		TetrisAttack_CursorDown()
	elseif (action == 3) then
		TetrisAttack_CursorLeft()
	elseif (action == 4) then
		TetrisAttack_CursorRight()
	elseif (action == 5) then
		TetrisAttack_CursorFlip()
	elseif (action == 6) then
		speedyShift = 1
	elseif (action == 7) then
		TetrisAttack_ReloadPuzzle()
	end
end

--Commented out so that end users can't go messing with puzzles.
function TetrisAttack_SaveCurrentPuzzle()
	local index
	if (IsAltKeyDown()) then
	index = #TetrisAttack_SavedPuzzles+1
	else
	index = currentPuzzleIndex
	end
	local code = ""
	for row = 1, numRowsVisible do
		for column = 1, 6 do
		code = code .. blockTable[row][column]
		end
	end
	if (#TetrisAttack_SavedPuzzles < index) then
		TetrisAttack_SavedPuzzles[index] = {}
	end
	TetrisAttack_SavedPuzzles[index]["numMoves"] = numMovesLeft
	TetrisAttack_SavedPuzzles[index]["code"] = code
	TetrisAttack_SavedPuzzles[index]["demo"] = {{0, 0}}
	out("Puzzle saved at index " .. index)
	out(numMovesLeft)
	out(code)
end

--Load the next puzzle.
function TetrisAttack_NextPuzzle()
	if (currentPuzzleIndex < numOfficialPuzzles and currentPuzzleIndex ~= numTutorials) then 
		currentPuzzleIndex = currentPuzzleIndex + 1
		if (currentPuzzleIndex > numTutorials) then 
			TetrisAttack_LoadPuzzle(currentPuzzleIndex-numTutorials) 
		else
			TetrisAttack_LoadTutorial(currentPuzzleIndex) 
		end
	end
end

--Load the previous puzzle.
function TetrisAttack_PrevPuzzle()
	if (currentPuzzleIndex > 1 and currentPuzzleIndex ~= numTutorials + 1) then 
		currentPuzzleIndex = currentPuzzleIndex - 1
		if (currentPuzzleIndex > numTutorials) then 
			TetrisAttack_LoadPuzzle(currentPuzzleIndex-numTutorials) 
		else
			TetrisAttack_LoadTutorial(currentPuzzleIndex) 
		end
	end
end

--Load the next puzzle.
function TetrisAttack_NextCustomPuzzle()
	if (currentPuzzleIndex < #TetrisAttack_SavedPuzzles) then 
		currentPuzzleIndex = currentPuzzleIndex + 1
		TetrisAttack_LoadCustomPuzzle(currentPuzzleIndex) 
	end
end

--Load the previous puzzle.
function TetrisAttack_PrevCustomPuzzle()
	if (currentPuzzleIndex > 1) then 
		currentPuzzleIndex = currentPuzzleIndex - 1
		TetrisAttack_LoadCustomPuzzle(currentPuzzleIndex) 
	end
end

function TetrisAttack_ReloadPuzzle()
	TetrisAttack_StartPuzzle(currentPuzzleIndex)
end

function TetrisAttack_LoadTutorial(num)
	TetrisAttack_StartPuzzle(num)
	numMovesLeft = 99
	TetrisAttackPuzzleMovesText:Hide()
	TetrisAttackPuzzleMovesDisplay:Hide()
	TetrisAttackPuzzleDifficultyText:Hide()
	TetrisAttackPuzzleDifficultyDisplay:Hide()
	TetrisAttackPuzzleNumber:SetText("Tutorial " .. currentPuzzleIndex)
	TetrisAttack_PlayDemoButton:SetText("Play Demo")
	TetrisAttack_ResetButton:Show()
end

function TetrisAttack_LoadPuzzle(num)
	TetrisAttack_StartPuzzle(num+numTutorials)
	TetrisAttackPuzzleMovesText:Show()
	TetrisAttackPuzzleMovesDisplay:Show()
	TetrisAttackPuzzleDifficultyText:Show()
	TetrisAttackPuzzleDifficultyDisplay:Show()
	TetrisAttackPuzzleMovesText:SetText("Moves Left:")
	TetrisAttackPuzzleDifficultyText:SetText("Difficulty:")
	TetrisAttackPuzzleNumber:SetText("Puzzle " .. currentPuzzleIndex-numTutorials)
	TetrisAttack_PlayDemoButton:SetText("Give Up")
	TetrisAttack_ResetButton:Show()
end

function TetrisAttack_LoadCustomPuzzle(num)
	TetrisAttack_StartPuzzle(num, true)
	TetrisAttackPuzzleMovesText:Show()
	TetrisAttackPuzzleMovesDisplay:Show()
	TetrisAttackPuzzleDifficultyText:Show()
	TetrisAttackPuzzleDifficultyDisplay:Show()
	TetrisAttackPuzzleMovesText:SetText("Moves Left:")
	TetrisAttackPuzzleDifficultyText:SetText("Difficulty:")
	TetrisAttackPuzzleNumber:SetText("Custom Puzzle " .. currentPuzzleIndex)
	TetrisAttack_PlayDemoButton:SetText("Give Up")
	TetrisAttack_ResetButton:Show()
end

--Start a puzzle or tutorial, since they're the same thing.
function TetrisAttack_StartPuzzle(num, custom)
	custom = custom or false
	if (numOfficialPuzzles >= num) then
		TetrisAttack_ScrubGameVariables()
		gameInProgress = 2
		currentPuzzleIndex = num
		local code, diff, text
		if (not custom) then
			code, numMovesLeft, diff, text = TetrisAttack_GetPuzzleInfo(num, currentDemo)
		else
			code, numMovesLeft, diff, text = TetrisAttack_GetCustomPuzzleInfo(num, currentDemo)
		end
		TetrisAttackComment1:SetText(text)
		TetrisAttackPuzzleDifficultyDisplay:SetText(difficultyText[diff])
		TetrisAttackPuzzleDifficultyDisplay:SetTextColor(TetrisAttack_GetDifficultyColor(diff))
		TetrisAttackPuzzleMovesDisplay:SetText(numMovesLeft)
		TetrisAttackMainMenu:Hide()
		TetrisAttackEndless:Hide()
		TetrisAttack1:Show()
		TetrisAttack2:Hide()
		TetrisAttackPuzzleInfo:Show()
		TetrisAttackField:SetAlpha(1)
		TetrisAttackField:SetPoint("CENTER", TetrisAttack1, "CENTER", 0, 0)
		TetrisAttackFlightpath:Hide()
		TetrisAttackOpponentField:SetAlpha(0)
		TetrisAttackFieldBackground:SetTexture(0.05, 0.05, 0.05, 0.75)
		TetrisAttackScoreEasy:Hide()
		TetrisAttackScoreHard:Hide()
		TetrisAttackMainMenu:Hide()
		TetrisAttack_ChangeGameSpeed(5)
		speedyShift = 0
		allowInput = true
		TetrisAttack_Pause(0)
		for row = 1, numRowsVisible do
			for column = 1, 6 do
				index = (row-1)*6 + column
				colorIndex = tonumber(strsub(code, index, index))
				blockTable[row][column] = colorIndex
				if (colorIndex > 0 and colorIndex < 9) then numBlocksLeft = numBlocksLeft + 1 end
				if (colorIndex > 0) then
					if (blockTable[row][column] == 8) then
						blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockbonus")
						blockTextures[row][column]:SetVertexColor(1,1,1,1)
					elseif (blockTable[row][column] == 0 or useClassIcons == 1) then
						blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
						blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
					elseif (colorizeBlocks == 1) then
						blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
						blockTextures[row][column]:SetVertexColor(blockColor[colorIndex][1],blockColor[colorIndex][2],blockColor[colorIndex][3], 1)
					end
					blockTextures[row][column]:Show()
					blockTextures[row][column]:SetAlpha(1)
					blockTextures[row][column]:ClearAllPoints()
					blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) + 10 + 32)
					blockTextures[row][column]:SetTexCoord(0,1,0,1)
					blockTextures[row][column]:SetHeight(32)
				else
					blockTextures[row][column]:Show()
					blockTextures[row][column]:SetAlpha(1)
					blockTextures[row][column]:SetTexCoord(0,1,0,1)
					blockTextures[row][column]:SetHeight(32)
					blockTextures[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block0")
					blockTextures[row][column]:ClearAllPoints()
					blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, ((row-1) * 32) + 10 + 32)
				end
				lastBlock = block
			end
		end
		for row = numRowsVisible+1, numRows do
			for column = 1, 6 do
				blockTable[row][column] = 0
			end
		end
		for column=1, 6 do
			newBlockTable[column] = math.random(blockTypes) --Possible values are 0-8: 0 empty 1-7 blocks 8 garbage
			local colorIndex = abs(newBlockTable[column])
			
			if (newBlockTable[column] <= 0 or useClassIcons == 1)
			then
				newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. colorIndex)
				newBlockTextures[column]:SetVertexColor(blockColor[colorIndex][1]*.5,blockColor[colorIndex][2]*.5,blockColor[colorIndex][3]*.5, 1)
			elseif (colorizeBlocks == 1) then
				newBlockTextures[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
				newBlockTextures[column]:SetVertexColor(blockColor[colorIndex][1]*.5,blockColor[colorIndex][2]*.5,blockColor[colorIndex][3]*.5, 1)
			end
			newBlockTextures[column]:ClearAllPoints()
			newBlockTextures[column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20,10)
			newBlockTextures[column]:SetTexCoord(0,1,0,0)
			newBlockTextures[column]:SetHeight(.01)
			newBlockTextures[column]:Show()
			lastBlock = block
		end
		cursorTexture:Show()
		cursorTexture:ClearAllPoints()
		cursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", (cursorX*32) - 52, ((cursorY-1) * 32) - 6)
	end
	if (num == 15) then TetrisAttack_RecvGarbage("03040506") end
end

--Start the animation where all the blocks slide off the screen.
function TetrisAttack_TriggerDeathAnim()
	if (gameInProgress == 3 and currentOpponent ~= "") then
		SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgGreetingPrefix, "WIN" .. "~" .. currentVersion, "WHISPER", currentOpponent)
	end
	if (playSounds == 1) then PlaySoundFile(soundEffects["defeat"]) end
	for column = 1, 6 do
		newBlockTextures[column]:Hide()
	end
	TetrisAttack1:EnableKeyboard(false)
	cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youlose.tga")
	cursorTexture:SetWidth(128)
	cursorTexture:SetHeight(128)
	TetrisAttack_DropCursor()
	deathAnimVelocity = 24
	deathAnimAccel = -384
	deathAnimOffset = fieldOffset
	deathAnim = 1
	if (gameInProgress == 3) then
		for column = 1, 6 do
			newBlockTexturesMP[column]:Hide()
		end
		MPcursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\youwin.tga")
		MPcursorTexture:SetWidth(128)
		MPcursorTexture:SetHeight(128)
		opponentDeathAnimVelocity = -128
		opponentDeathAnimOffset = numRowsVisible*32
		opponentDeathAnim = -1
	end
end

--Initialize variables for a new combo animation... you know, "x2!", "x3!", "Bloodlust!" Anyway, if showTex is 0, it hides the little background texture.
function TetrisAttack_CreateComboAnim(row, column, text, color, showTex)
	showTex = showTex or 1
	if (lastComboRow == row and lastComboColumn == column) then row = row + 1 end
	lastComboRow = row
	lastComboColumn = column
	local anim = 0
	local newTexture
	local newString
	for i = 1, maxComboAnims do
		if (currentAnims[i]["timer"] <= 0) then
			anim = i
			break
		end
	end
	local _G = getfenv()
	if (anim == 0) then
		maxComboAnims = maxComboAnims+1
		currentAnims[maxComboAnims] = {["x"] = (column*32) - 28,
									 ["y"] = ((row) * 32) + 10 + fieldOffset - 41, 
									 ["texture"] = TetrisAttackFieldBorder:CreateTexture("TetrisAttackComboAnim".. numAnimations, "ARTWORK"),
									 ["fontstring"] = TetrisAttackFieldBorder:CreateFontString("TetrisAttackComboString".. numAnimations, "OVERLAY", "WorldMapTextFont"), 
									 ["timer"] = 1.5}
		newTexture = _G["TetrisAttackComboAnim" .. numAnimations]
		newString = _G["TetrisAttackComboString" .. numAnimations]
		newTexture:SetTexture("Interface\\BUTTONS\\UI-Quickslot")
		newTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", (column*32) - 28, ((row) * 32) + 10 + fieldOffset - 41)
		newTexture:SetWidth(50)
		newTexture:SetHeight(50)
		if (showTex == 0) then newTexture:Hide() end
		newString:SetText(text)
		newString:SetJustifyH("CENTER")
		newString:SetTextColor(qualityColors[color][1], qualityColors[color][2], qualityColors[color][3])
		newString:SetPoint("CENTER", TetrisAttackFieldBorder, "BOTTOMLEFT", (column*32) - 28 + 23, ((row) * 32) + 10 + fieldOffset - 16)
	else
		newTexture = _G["TetrisAttackComboAnim" .. anim]
		newString = _G["TetrisAttackComboString" .. anim]
		if (showTex == 1) then newTexture:Show() end
		newString:Show()
		newTexture:SetAlpha(1)	
		newString:SetAlpha(1)
		newTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", (column*32) - 28, ((row) * 32) + 10 + fieldOffset - 41)
		newTexture:SetWidth(50)
		newTexture:SetHeight(50)
		newString:SetText(text)
		newString:SetTextColor(qualityColors[color][1], qualityColors[color][2], qualityColors[color][3])
		newString:SetPoint("CENTER", TetrisAttackFieldBorder, "BOTTOMLEFT", (column*32) - 28 + 23, ((row) * 32) + 10 + fieldOffset - 16)
		currentAnims[anim]["x"] = (column*32) - 28
		currentAnims[anim]["y"] = ((row) * 32) + 10 + fieldOffset - 41
		currentAnims[anim]["timer"] = 1.5
	end
end

--Update all current combo anims and mark them as available for repurposing when they finish.
--[[local--]] function TetrisAttack_UpdateComboAnims(elapsed, clearAll)
	clearAll = clearAll or false
	local timer, x, y, tex, str
	for i = 1, maxComboAnims do
		if (currentAnims[i]["timer"] > 0) then
			timer = currentAnims[i]["timer"] - elapsed
			currentAnims[i]["timer"] = timer
			tex = _G["TetrisAttackComboAnim" .. i]
			str = _G["TetrisAttackComboString" .. i]
			if (timer <= 0 or clearAll) then
				currentAnims[i]["timer"] = 0
				tex:Hide()
				str:Hide()
			elseif (timer < 1) then
				x = currentAnims[i]["x"]
				y = currentAnims[i]["y"] + comboAnimSpeed/GetFramerate() * timer/1.5
				currentAnims[i]["y"] = y
				tex:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", x, y)
				str:SetPoint("CENTER", TetrisAttackFieldBorder, "BOTTOMLEFT", x + 23, y + 25)
				tex:SetAlpha(timer/1)
				str:SetAlpha(timer/1)
			else
				x = currentAnims[i]["x"]
				y = currentAnims[i]["y"] + comboAnimSpeed/GetFramerate() * timer/1.5
				currentAnims[i]["y"] = y
				tex:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", x, y)
				str:SetPoint("CENTER", TetrisAttackFieldBorder, "BOTTOMLEFT", x + 23, y + 25)
			end
		end
	end
end

--Translate a row and column into a letter suitable for sending in multiplayer. Allows the transmitting of the proper shape of garbage blocks quickly and only using a single character of data per square.
function TetrisAttack_GetGarbageTextureCode(row, column)
	local index = 5
	for i = 1, numGarbageBlocks do
		local top = garbageTable[i]["top"]
		local bottom = garbageTable[i]["bottom"]
		local left = garbageTable[i]["left"]
		local right = garbageTable[i]["right"]
		if (row <= top and row >= bottom and column <= right and column >= left) then
			if (row == top and row == bottom) then
				index = index + 6
			elseif (row == top) then
				index = index + 3
			elseif (row == bottom) then
				index = index - 3
			end
			if (column == left) then
				index = index - 1
			elseif (column == right) then
				index = index + 1 
			end
			return garbageTextureCodes[index]
		end
	end
	return ""
end

--[[

		Turn the current game field into a solid string of numbers and letters to be sent to the other client. The format is:
		-current field offset, 3 characters. (Two-digit number plus possible minus sign indicating that the field is not rising.)
		-our current game speed, 2 characters
		-cursor X and Y positions, 4 characters.
		-code for the special attack to send, if any. 1 character.
		-full view of the current field including "new" blocks. 12*6 + 1*6 = 78 characters.
		-data about any animations we've triggered since last update, which is 6 characters per animation.
		Total: 88 + 6/anim
		
--]]

--[[local--]] function TetrisAttack_SendEncodedGrid(fieldMoved)
	if (opponentName == "") then
		TetrisAttack_TriggerWinAnim()
		TetrisAttackCommentText:SetText("Connection lost :(")
		TetrisAttackCommentText:SetTextColor(1, 0.2, 0.2, 1)
		return
	end
	local GetGarbageTextureCode = TetrisAttack_GetGarbageTextureCode
	local code = "" .. (((fieldOffset < 10) and "0") or "") .. math.floor(fieldOffset)
	local code = (fieldMoved and string.format("%03d", math.ceil(fieldOffset))) or ("-" .. string.format("%02d", math.ceil(fieldOffset)))
	code = code .. string.format("%02d", currentSpeed) .. string.format("%02d", cursorX) .. string.format("%02d", cursorY)
	code = code .. sendSpecial
	sendSpecial = "0"
	local y = 0
	local x = 0
	for column = 1, 6 do
		code = code .. newBlockTable[column]
	end
	for row = 1, numRowsVisible do
		for column = 1, 6 do
			if (abs(blockTable[row][column]) == 7) then
				code = code .. GetGarbageTextureCode(row, column)
			else
				code = code .. abs(blockTable[row][column])
			end
		end
	end
	code = code .. animsToSend
	animsToSend = ""
	SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgFieldPrefix, code, "WHISPER", currentOpponent, "NOQUEUE")
end

function TetrisAttack_IgnoreChallenges(ignore)
	ignoreChallenges = ignore
end

--Set up the multiplayer display for a new game.
function TetrisAttack_InitMPDisplay()
	local index = 0
	MPfieldOffset = 0
	MPfieldMoving = 0
	MPcursorX = cursorX
	MPcursorY = cursorY
	MPcursorTexture:ClearAllPoints()
	MPcursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackOpponentField, "BOTTOMLEFT", (MPcursorX*32) - 52, ((MPcursorY-1) * 32) - 6)
	TetrisAttackOpponentField:SetPoint("CENTER", TetrisAttack2, "CENTER", 0, 0)
	for column = 1, 6 do
		color = newBlockTable[column]
	
		if (color == 0 or useClassIcons == 1)
		then
			newBlockTexturesMP[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. color)
			newBlockTexturesMP[column]:SetVertexColor(blockColor[color][1]*0.5,blockColor[color][2]*0.5,blockColor[color][3]*0.5, 1)
		elseif (colorizeBlocks == 1) then
			newBlockTexturesMP[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
			newBlockTexturesMP[column]:SetVertexColor(blockColor[color][1]*0.5,blockColor[color][2]*0.5,blockColor[color][3]*0.5, 1)
		end
		newBlockTexturesMP[column]:ClearAllPoints()
		newBlockTexturesMP[column]:SetPoint("TOPLEFT", TetrisAttackOpponentField, "BOTTOMLEFT", (column*32) - 20,10)
		newBlockTexturesMP[column]:SetTexCoord(0,1,0,0)
		newBlockTexturesMP[column]:SetHeight(.01)
		newBlockTexturesMP[column]:Show()
		newBlockTexturesMP[column]:SetAlpha(1)
	end
	for row = 1, numRowsVisible do
		for column = 1, 6 do
		color = abs(blockTable[row][column])
		
			if (color == 0 or useClassIcons == 1)
			then
				blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. color)
				blockTexturesMP[row][column]:SetVertexColor(blockColor[color][1],blockColor[color][2],blockColor[color][3], 1)
			elseif (colorizeBlocks == 1) then
				blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
				blockTexturesMP[row][column]:SetVertexColor(blockColor[color][1],blockColor[color][2],blockColor[color][3], 1)
			end
			blockTexturesMP[row][column]:SetHeight(32)
			blockTexturesMP[row][column]:SetTexCoord(0,1,0,1)
			blockTexturesMP[row][column]:Show()
			blockTexturesMP[row][column]:SetAlpha(1)
		end
	end
end

--Process an opponent's powerup.
function TetrisAttack_RecvSpecialAttack(type)
	if (type == 1) then
		if (playSounds == 1) then PlaySoundFile(soundEffects["bloodlust"]) end
		opponentBloodTimer = opponentBloodTimer + (10 - MPAvgLatency)
		TetrisAttack_CreateComboAnim(8, 10, "Bloodlust!", 7, 0)
		opponentBackgroundPulse = true
	elseif (type == 2) then
		backgroundPulse = true
		if (playSounds == 1) then PlaySoundFile(soundEffects["corruption"]) end
		corrTimer = corrTimer + 20
		TetrisAttack_CreateComboAnim(8, 3.5, "Corruption!", 5, 0)
	elseif (type == 3) then
		if (playSounds == 1) then PlaySoundFile(soundEffects["tranquility"]) end
		opponentTranqTimer = opponentTranqTimer + (16 - MPAvgLatency)
		TetrisAttack_CreateComboAnim(8, 10, "Tranquility!", 3, 0)
		opponentBackgroundPulse = true
	end
end

--Update the multiplayer display using the known pattern of data sent by the other client. The pattern is defined above, before the sending function.
--[[local--]] function TetrisAttack_UpdateMPDisplay(code)
	local index = 0
	local offset = tonumber(strsub(code, 1, 3))
	currentOpponentSpeed = tonumber(strsub(code, 4, 5))
	TetrisAttack_UpdateOpponentInfo(currentOpponent, currentOpponentDifficulty, currentOpponentSpeed)
	MPfieldOffset = math.abs(offset) + (MPAvgLatency * gameSpeeds[currentOpponentSpeed][1] * (((opponentBloodTimer > 0) and 1.1) or ((opponentTranqTimer > 0) and .8) or 1))
	MPfieldMoving = (offset >= 0)
	MPcursorX = tonumber(strsub(code, 6, 7))
	MPcursorY = tonumber(strsub(code, 8, 9))
	TetrisAttack_RecvSpecialAttack(tonumber(strsub(code, 10, 10)))
	MPcursorTexture:ClearAllPoints()
	MPcursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackOpponentField, "BOTTOMLEFT", (MPcursorX*32) - 52, ((MPcursorY-1) * 32) - 6)
	if (MPfieldOffset > 31) then MPfieldOffset = 31 end
	for column = 1, 6 do
		color = tonumber(strsub(code, column+10, column+10))
		newBlockTexturesMP[column]:SetTexCoord(0,1,0,((MPfieldOffset+1)/32))
		newBlockTexturesMP[column]:SetHeight(MPfieldOffset+1)
		if (color == 8) then
			newBlockTexturesMP[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockbonus")
			newBlockTexturesMP[column]:SetVertexColor(1,1,1,1)
		elseif (color == 0 or useClassIcons == 1) then
			newBlockTexturesMP[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. color)
			newBlockTexturesMP[column]:SetVertexColor(blockColor[color][1]*0.5,blockColor[color][2]*0.5,blockColor[color][3]*0.5, 1)
		elseif (colorizeBlocks == 1) then
			newBlockTexturesMP[column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
			newBlockTexturesMP[column]:SetVertexColor(blockColor[color][1]*0.5,blockColor[color][2]*0.5,blockColor[color][3]*0.5, 1)
		end
	end
	for row = 1, numRowsVisible do
		for column = 1, 6 do
			index = (row-1)*6 + column
			colorStr = strsub(code, index+16, index+16)
			color = tonumber(colorStr)
			if (color) then --If it was a number, treat it normally.
				if (color == 8) then
					blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockbonus")
					blockTexturesMP[row][column]:SetVertexColor(1,1,1,1)
				elseif (color == 0 or useClassIcons == 1) then
					blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block" .. color)
					blockTexturesMP[row][column]:SetVertexColor(blockColor[color][1],blockColor[color][2],blockColor[color][3], 1)
				elseif (colorizeBlocks == 1) then
					blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockplain")
					blockTexturesMP[row][column]:SetVertexColor(blockColor[color][1],blockColor[color][2],blockColor[color][3], 1)
				end
			else
					blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\blockgarbageO" .. garbageTextureDecodes[colorStr]) -- If it's garbage, use the translator.
					blockTexturesMP[row][column]:SetVertexColor(1,1,1,1)
			end
		end
	end
	if (strlen(code) > 88) then --Check for any animation data.
		for i = 1, (strlen(code)-88)/6 do
			TetrisAttack_ProcessMPComboCode(strsub(code, 88 + (i*6) - 5, 88 + i*6))
		end			
	end
end

--Turn an opponent's combo info into an animation.
function TetrisAttack_ProcessMPComboCode(code)
	local row = tonumber(strsub(code, 1, 2))
	local column =  tonumber(strsub(code, 3, 3))
	local color =  tonumber(strsub(code, 4, 4))
	local text = ""
	if (color > 0) then
		text =  "x" .. tonumber(strsub(code, 5, 6))
		if (currentOpponentDifficulty > currentDifficulty) then bonusBlockChance = bonusBlockChance + 2 end
	else
		color = 1
		text =  "" .. tonumber(strsub(code, 5, 6))
		if (currentOpponentDifficulty > currentDifficulty) then bonusBlockChance = bonusBlockChance + 1 end
	end
	if (playSounds == 1 and color == 1) then 
		PlaySoundFile(soundEffects["baseCombo"])
	elseif (playSounds == 1 and color < 6) then 
		PlaySoundFile(comboSounds[opponentRaceGender][color-1]) 
	end
	TetrisAttack_CreateComboAnim(row, column + 6.5, text, color)
end

--Start up multiplayer communication.
function TetrisAttack_InitMultiplayerComm()
	local down, up, lag = GetNetStats()
	MPAvgLatency = lag / 1000
	multiplayerSeed = math.floor(GetTime())
	TetrisAttack_SendHandshake(multiplayerSeed)
end

--SET PHASERS TO FUN
function TetrisAttack_StartMultiplayer(delay)
	TetrisAttack_NewGameEndless()
	gameInProgress = 3
	TetrisAttack_SendRaidGreeting()
	TetrisAttack_InitMPDisplay()
	preGameTimer = preGameTimer - delay
	TetrisAttack2:Show()
end

--Send the initial handshake with our multiplayer seed and startingspeed.
function TetrisAttack_SendHandshake(seed)
	local code = "" .. seed .. " " .. startingSpeed
	lastHandshakeTime = GetTime()
	SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgHandshakePrefix, code, "WHISPER", currentOpponent)
end

--Process what was sent with the above.
function TetrisAttack_RecvHandshake(code)
	local seed, speed = strsplit(" ", code)
	multiplayerSeed = tonumber(seed)
	lastHandshakeTime = GetTime()
	SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgPingPrefix, "1", "WHISPER", currentOpponent)
end

--Deal with a high score we received from a friend.
function TetrisAttack_RecvHighScore(commName, string)
	local src, dest, name, score, diff, endlessName, endless, endlessDiff = strsplit("~", string)
	diff = tonumber(diff)
	endless = tonumber(endless)
	endlessDiff = tonumber(endlessDiff)
	score = tonumber(score)
	if (endless > 0) then TetrisAttack_AddNewFriendScore(endless, endlessName, endlessDiff) end
	if (src == srcName and dest == destName and name ~= " ") then
		if (not TetrisAttack_FlightpathHighScores[destName][srcName]) then
			TetrisAttack_FlightpathHighScores[destName][srcName] = {["name"] = name, ["score"] = score,  ["difficulty"] = diff}
		end
		if (not TetrisAttack_FlightpathHighScores[srcName][destName]) then
			TetrisAttack_FlightpathHighScores[srcName][destName] = {["name"] = name, ["score"] = score,  ["difficulty"] = diff}
		else
			if (score > TetrisAttack_FlightpathHighScores[srcName][destName]["score"]) then
				if (TetrisAttack_FlightpathHighScores[srcName][destName]["name"] == currentPlayerName) then
					TetrisAttack_FlightpathHighScores[srcName][destName]["myName"] = TetrisAttack_FlightpathHighScores[srcName][destName]["name"]
					TetrisAttack_FlightpathHighScores[srcName][destName]["myScore"] = TetrisAttack_FlightpathHighScores[srcName][destName]["score"]
					TetrisAttack_FlightpathHighScores[srcName][destName]["myDiff"] = TetrisAttack_FlightpathHighScores[srcName][destName]["difficulty"]
					TetrisAttack_FlightpathHighScores[destName][srcName]["myName"] = TetrisAttack_FlightpathHighScores[srcName][destName]["name"]
					TetrisAttack_FlightpathHighScores[destName][srcName]["myScore"] = TetrisAttack_FlightpathHighScores[srcName][destName]["score"]
					TetrisAttack_FlightpathHighScores[destName][srcName]["myDiff"] = TetrisAttack_FlightpathHighScores[srcName][destName]["difficulty"]
				end
				TetrisAttack_FlightpathHighScores[srcName][destName]["name"] = name
				TetrisAttack_FlightpathHighScores[srcName][destName]["score"] = score
				TetrisAttack_FlightpathHighScores[srcName][destName]["difficulty"] = diff
				TetrisAttack_FlightpathHighScores[destName][srcName]["name"] = name
				TetrisAttack_FlightpathHighScores[destName][srcName]["score"] = score
				TetrisAttack_FlightpathHighScores[destName][srcName]["difficulty"] = diff
			end
		end
		TetrisAttack_UpdateFlightpathScoreboard(name, score, diff)
	end
end

--Ask friends for high scores.
function TetrisAttack_SendHighScoreRequest(name, string)
	SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScoreReqPrefix, string, "WHISPER", name)
end

--let everybody know how cool you are
function TetrisAttack_BroadcastNewHighScore(scoreName, score, diff)
	if (useFriendsList == 1) then
		for name, enabled in pairs(friendHasFPL) do
			if (enabled == 1) then
				SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScorePrefix, " ~ ~ ~ ~ ~" .. scoreName .. "~" .. score .. "~" .. diff, "WHISPER", name)
			end
		end
	end
end

--If we got a request for a high score, check to see if we know of any for that route and then bounce it back with our own endless mode scores added on the end.
function TetrisAttack_RecvHighScoreRequest(commName, string)
	local src, dest = strsplit("~", string)
	local score, name, diff
	if ((src == " ") or (dest == " ")) then
		SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScorePrefix, string .. "~ ~ ~ ~" .. TetrisAttack_EndlessHighScores[1]["name"] .. "~" .. TetrisAttack_EndlessHighScores[1]["score"] .. "~" .. TetrisAttack_EndlessHighScores[1]["difficulty"], "WHISPER", commName)
	end
	if (TetrisAttack_FlightpathHighScores[src]) then
		if (TetrisAttack_FlightpathHighScores[src][dest]) then
			name = TetrisAttack_FlightpathHighScores[src][dest]["name"]
			score = TetrisAttack_FlightpathHighScores[src][dest]["score"]
			diff = TetrisAttack_FlightpathHighScores[src][dest]["difficulty"]
			SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScorePrefix, string .. "~" .. name .. "~" .. score .. "~" .. diff .. "~" .. TetrisAttack_EndlessHighScores[1]["name"] .. "~" .. TetrisAttack_EndlessHighScores[1]["score"] .. "~" .. TetrisAttack_EndlessHighScores[1]["difficulty"], "WHISPER", commName)
			return
		end
	end
	if (TetrisAttack_FlightpathHighScores[dest]) then
		if (TetrisAttack_FlightpathHighScores[dest][src]) then
			name = TetrisAttack_FlightpathHighScores[dest][src]["name"]
			score = TetrisAttack_FlightpathHighScores[dest][src]["score"]
			diff = TetrisAttack_FlightpathHighScores[dest][src]["difficulty"]
			SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgScorePrefix, string .. "~" .. name .. "~" .. score .. "~" .. diff .. "~" .. TetrisAttack_EndlessHighScores[1]["name"] .. "~" .. TetrisAttack_EndlessHighScores[1]["score"] .. "~" .. TetrisAttack_EndlessHighScores[1]["difficulty"], "WHISPER", commName)
			return
		end
	end
end

function TetrisAttack_RecvGarbage(code)
	local num = strlen(code)/2
	for index = 1, num do
		TetrisAttack_QueueGarbage(tonumber(strsub(code, index*2-1, index*2)))
	end
end

function TetrisAttack_SendGarbage()
	if (MPgarbageCode ~= "") then
		SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgGarbagePrefix, MPgarbageCode, "WHISPER", currentOpponent)
		MPgarbageCode = ""
	end
end

--Get a heartbeat and clear any lag warnings present.
function TetrisAttack_MPHeartbeat()
	if (heartbeat > 1) then
		TetrisAttackCommentText:SetText("")
	end
	heartbeat = 0
end

--Checking your friends list for the presence of this mod.
function TetrisAttack_CheckFriendsForFPL()
	for i=1, GetNumFriends() do
		name, level, class, area, connected, status = GetFriendInfo(i);
		if (connected) then
			if ((not friendHasFPL[name]) or friendHasFPL[name] == 0) then
				TetrisAttack_SendGreeting(name)
			end
		else
			friendHasFPL[name] = 0
		end
	end
end

--Send a greeting to a person. Greetings are mostly used to check what version they have.
function TetrisAttack_SendGreeting(name)
	SuperChatThrottleLib:SendAddonMessage("BULK", chatMsgGreetingPrefix, "PING" .. "~" .. currentVersion, "WHISPER", name)
end

function TetrisAttack_SendFriendChallenge()
	TetrisAttack_SendChallenge(selectedFriend)
end

--Send the user's response to their opponent.
function TetrisAttack_RespondToChallenge(msg)
	if (challengeTimer ~= 0) then
		challengeTimer = 0
		TetrisAttackChallengeAcceptFrame:Hide()
		SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgChallengePrefix, msg .. "~" .. currentDifficulty .. "~" .. startingSpeed .. "~" .. myRaceGender .. "~" .. currentVersion, "WHISPER", currentOpponent)
		if (msg == "DECLINE") then currentOpponent = "" end
	end
end

function TetrisAttack_SendDisconnect()
	SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgGreetingPrefix, "DISCONNECT" .. "~" .. currentVersion, "WHISPER", currentOpponent)
end

function TetrisAttack_SendChallenge(name)
	if (name == "") then return end
	if (challengeTimer > 0) then
		DEFAULT_CHAT_FRAME:AddMessage("FPL: Only one challenge can be sent or considered at a time.", 1, 1, 1)
		return
	end
	currentOpponent = name
	challengeTimer = 15
	if (verbose == 1) then DEFAULT_CHAT_FRAME:AddMessage("FPL: Sending challenge to " .. currentOpponent .. "...", 1, 1, 1) end
	local code = "CHALLENGE" .. "~" .. currentDifficulty .. "~" .. startingSpeed .. "~" .. myRaceGender .. "~" .. currentVersion
	SuperChatThrottleLib:SendAddonMessage("ALERT", chatMsgChallengePrefix, code, "WHISPER", currentOpponent)
end

--Friends list function, which adds text to friends' names who have the mod.

function TetrisAttack_FriendsList_Update()
	local friendListOffset = FauxScrollFrame_GetOffset(FriendsFrameFriendsScrollFrame);
	local friendIndex;
	if (useFriendsList == 1) then
		name, level, class, area, connected, status = GetFriendInfo(GetSelectedFriend())
		if ((friendHasFPL[name] == 1 or friendHasFPL[name] == 3) and connected) then
			TetrisAttackChallengeButton:Enable()
			selectedFriend = name
		else
			TetrisAttackChallengeButton:Disable()
		end
			
		for i=1, FRIENDS_TO_DISPLAY, 1 do
			friendIndex = friendListOffset + i;
			name, level, class, area, connected, status = GetFriendInfo(friendIndex);
			local friendText = _G["FriendsFrameFriendButton" .. i .. "FPLEnabled"]
			if ((friendHasFPL[name] == 1 or friendHasFPL[name] == 3) and connected) then
				friendText:Show()
				friendText:SetText("FPL Installed!")
				friendText:SetTextColor(0, 1, 0)
			elseif (friendHasFPL[name] == 2 and connected) then
				friendText:Show()
				friendText:SetText("Older FPL Version")
				friendText:SetTextColor(.5, .5, .5)
			elseif (friendHasFPL[name] == -1 and connected) then
				friendText:Show()
				friendText:SetText("Newer FPL Version!")
				friendText:SetTextColor(1, 0, 0)
			else
				friendText:Hide()
			end
		end
		if (lastFriendCheck + friendCheckCooldown < GetTime()) then
			lastFriendCheck = GetTime()
			TetrisAttack_CheckFriendsForFPL()
		end
	end
end

--Because defining a PushedTexture for this stupid button didn't work at all, I had to write this.
function TetrisAttack_ClickButton(button, isDown)	--WHY DOESN'T THE DAMN PUSHED TEXTURE JUST WORK LIKE IT'S SUPPOSED TO AASGHAGHRGRHRGRHG
	if (button == "LeftButton" and isDown) then
		TetrisAttackChallengeButton:SetHighlightTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\buttonpress", "DISABLE")
	elseif (button == "LeftButton" and not isDown) then
		TetrisAttackChallengeButton:SetHighlightTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\buttonhilight", "DISABLE")
	end
end

--Process a greeting and update the friends list accordingly.
function TetrisAttack_RecvGreeting(name, version)
	if (version == currentVersion) then
		friendHasFPL[name] = 1
	elseif (version < currentVersion) then
		friendHasFPL[name] = 2
	else
		friendHasFPL[name] = -1
	end
	TetrisAttack_FriendsList_Update()
end

--Set up the friends list additions.
function TetrisAttack_InitFriendsListModifications()
	local _G = getfenv()
	for i=1, FRIENDS_TO_DISPLAY, 1 do
		local friendButton = _G["FriendsFrameFriendButton" .. i]
		friendButton:CreateFontString("FriendsFrameFriendButton" .. i .. "FPLEnabled", "ARTWORK", "GameFontNormal")
		local friendText = _G["FriendsFrameFriendButton" .. i .. "FPLEnabled"]
		friendText:SetText("FPL Installed!")
		friendText:ClearAllPoints()
		friendText:SetPoint("RIGHT", friendButton, "RIGHT", 0, 0)
		friendText:SetTextColor(0.0, 1.0, 0.0, 1.0)
		friendText:Hide();
	end
	TetrisAttackChallengeButton:SetDisabledTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\buttondisable")
	TetrisAttackChallengeButton:SetHighlightTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\buttonhilight", "DISABLE")
	TetrisAttackChallengeButton:SetNormalTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\buttonnormal")
end

--Set up the raid frame additions.
function TetrisAttack_InitRaidFrameModifications()
	local _G = getfenv()
	local raidButton, raidName, raidIcon
	for i=1, MAX_RAID_MEMBERS do
		raidButton = _G["RaidGroupButton" .. i]
		raidButton:SetAttribute("shift-type1", "click")
		raidButton:SetAttribute("clickbutton", TetrisAttackRaidChallengeButton)
		raidName = _G["RaidGroupButton" .. i .. "Name"]
		raidButton:CreateTexture("RaidGroupButton" .. i .. "FPLIcon", "OVERLAY")
		raidIcon = _G["RaidGroupButton" .. i .. "FPLIcon"]
		raidIcon:ClearAllPoints()
		raidIcon:SetPoint("RIGHT", raidName, "LEFT", 0, -1)
		raidIcon:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\raidicon")
		raidIcon:SetVertexColor(0,1,0)
		raidIcon:SetWidth(16)
		raidIcon:SetHeight(16)
		raidIcon:Show();
	end
	raidInitialized = true
end

function TetrisAttack_SendRaidChallenge()
	local _G = getfenv()
	local name = UnitName(GameTooltip:GetUnit())
	if (friendHasFPL[name] == 1) then
		TetrisAttack_SendChallenge(name)
	end
end

--Update the raid frame icons.
function TetrisAttack_UpdateRaidIcons()
	if (not raidInitialized) then
		TetrisAttack_InitRaidFrameModifications()
	end
	local _G = getfenv()
	local raidButton, raidName, raidIcon, name, rank, subgroup, level, class, fileName, zone, online, isDead, role, isML
	for i=1, MAX_RAID_MEMBERS do
		name, rank, subgroup, level, class, fileName, zone, online, isDead, role, isML = GetRaidRosterInfo(i)
		raidIcon = _G["RaidGroupButton" .. i .. "FPLIcon"]
		if (name and friendHasFPL[name] and online) then
			if (friendHasFPL[name]) then
				if (friendHasFPL[name] == 1) then
					raidIcon:SetVertexColor(0,1,0)
				elseif (friendHasFPL[name] == 2) then --older
					raidIcon:SetVertexColor(.7,.7,.7)
				elseif (friendHasFPL[name] == -1) then --newer
					raidIcon:SetVertexColor(1,0,0)
				elseif (friendHasFPL[name] == 3) then --newer
					raidIcon:SetVertexColor(1,.5,1)
				end
			end
			raidIcon:Show();
		else
			raidIcon:Hide();
		end
	end
end

--Initialize a flightpath game, checking if any high scores exist and displaying them.
function TetrisAttack_TaxiStart()
	local score = 0
	local name = ""
	local diff = 0
	local myName = ""
	local myScore = 0
	local myDiff = 0
	local scoreFound = false
	taxiGameInProgress = true
	if (TetrisAttack_FlightpathHighScores[srcName]) then
		if (TetrisAttack_FlightpathHighScores[srcName][destName]) then
			TetrisAttackFlightpathPromptText:SetText("Highest score on this route:")
			score = TetrisAttack_FlightpathHighScores[srcName][destName]["score"]
			name = TetrisAttack_FlightpathHighScores[srcName][destName]["name"]
			diff = TetrisAttack_FlightpathHighScores[srcName][destName]["difficulty"]
			myName = TetrisAttack_FlightpathHighScores[srcName][destName]["myName"] or myName
			myScore = TetrisAttack_FlightpathHighScores[srcName][destName]["myScore"] or myScore
			myDiff = TetrisAttack_FlightpathHighScores[srcName][destName]["myDiff"] or myDiff
			TetrisAttackFlightpathPromptScore:SetText(name .. ": " .. score)
			TetrisAttackFlightpathPromptScore:Show()
			scoreFound = true
		end
	else
		TetrisAttack_FlightpathHighScores[srcName] = {}
	end
	if (TetrisAttack_FlightpathHighScores[destName]) then
		if (TetrisAttack_FlightpathHighScores[destName][srcName]) then
			TetrisAttackFlightpathPromptText:SetText("Highest score on this route:")
			score = TetrisAttack_FlightpathHighScores[destName][srcName]["score"]
			name = TetrisAttack_FlightpathHighScores[destName][srcName]["name"]
			diff = TetrisAttack_FlightpathHighScores[destName][srcName]["difficulty"]
			myName = TetrisAttack_FlightpathHighScores[destName][srcName]["myName"] or myName
			myScore = TetrisAttack_FlightpathHighScores[destName][srcName]["myScore"] or myScore
			myDiff = TetrisAttack_FlightpathHighScores[destName][srcName]["myDiff"] or myDiff
			TetrisAttack_FlightpathHighScores[srcName][destName] = {["name"] = name, ["score"] = score, ["difficulty"] = diff,["myName"] = myName, ["myScore"] = myScore, ["myDiff"] = myDiff}
			TetrisAttackFlightpathPromptScore:SetText(name .. ": " .. score)
			TetrisAttackFlightpathPromptScore:Show()
			scoreFound = true
		end
	else
		TetrisAttack_FlightpathHighScores[destName] = {}
	end
	if (not scoreFound) then
		TetrisAttackFlightpathPromptText:SetText("No high scores for this route!")
		TetrisAttackFlightpathPromptScore:Hide()
	else
		TetrisAttackFlightpathPromptScore:SetTextColor(TetrisAttack_GetDifficultyColor(diff))
	end
	TetrisAttackFlightpathScores:SetText("--High Scores--")
	TetrisAttack_ClearFlightpathScoreboard()
	TetrisAttack_UpdateFlightpathScoreboard(name, score, diff)
	TetrisAttack_UpdateFlightpathScoreboard(myName, myScore, myDiff)
	TetrisAttackFlightpathPrompt:Show()
end

function TetrisAttack_GetDifficultyColor(diff)
	diff = tonumber(diff)
	if (diff >= 1 and diff <= 3) then
		return difficultyColors[diff][1], difficultyColors[diff][2], difficultyColors[diff][3]
	else
		return 1, 1, 1
	end
end

--Hooked function for taking a flightpath to a given node. Grabs the source and destination names and stores them for further use.
function TetrisAttack_TakeTaxi(node)
	local zoneName
	if (autoOpenFlightpath == 1 and gameInProgress == 0) then
		node = node or 1
		destName = TaxiNodeName(node)
		for i = 1, NumTaxiNodes() do
			if (TaxiNodeGetType(i) == "CURRENT") then
				srcName = TaxiNodeName(i)
			end
		end
		srcName, zoneName = strsplit(",", srcName)	 --Why would we store the name of the zone? That's silly!
		destName, zoneName = strsplit(",", destName)
		TetrisAttack_TaxiStart()
	end
	TetrisAttack_FriendsList_Update()
	multiplayerSeed = TaxiNodeCost(node) + node
	TetrisAttack_TakeTaxiNode_Original(node)
end

function TetrisAttack_ClickFriendButton(button)
	TetrisAttack_ClickFriendButton_Original(button)
	TetrisAttack_FriendsList_Update()
end

--Aww, you don't want to play? Why not? Is it something I said?
function TetrisAttack_CancelTaxiGame()
	srcName = ""
	destName = ""
	taxiGameInProgress = false
end

--End a taxi game when you land and handle high score processing.
function TetrisAttack_TaxiEnd()
	local newHigh = false
	if (taxiGameInProgress) then
		TetrisAttack_YouLose()
		taxiGameInProgress = false
		if (not TetrisAttack_FlightpathHighScores[srcName]) then TetrisAttack_FlightpathHighScores[srcName] = {} end
		if (not TetrisAttack_FlightpathHighScores[srcName][destName]) then
			TetrisAttack_FlightpathHighScores[srcName][destName] = {["name"] = currentPlayerName, ["score"] = playerScore, ["difficulty"] = currentDifficulty}
			newHigh = true
		else
			if (playerScore > TetrisAttack_FlightpathHighScores[srcName][destName]["score"]) then
				newHigh = true
				TetrisAttack_FlightpathHighScores[srcName][destName]["name"] = currentPlayerName
				TetrisAttack_FlightpathHighScores[srcName][destName]["score"] = playerScore
				TetrisAttack_FlightpathHighScores[srcName][destName]["difficulty"] = currentDifficulty
				TetrisAttack_FlightpathHighScores[srcName][destName]["myName"] = ""
				TetrisAttack_FlightpathHighScores[srcName][destName]["myScore"] = 0
				TetrisAttack_FlightpathHighScores[srcName][destName]["myDiff"] = 0
			end
		end
		if (not TetrisAttack_FlightpathHighScores[destName]) then TetrisAttack_FlightpathHighScores[destName] = {} end
		if (not TetrisAttack_FlightpathHighScores[destName][srcName]) then
			TetrisAttack_FlightpathHighScores[destName][srcName] = {["name"] = currentPlayerName, ["score"] = playerScore, ["difficulty"] = currentDifficulty}
		else
			if (playerScore > TetrisAttack_FlightpathHighScores[destName][srcName]["score"]) then
				newHigh = true
				TetrisAttack_FlightpathHighScores[destName][srcName]["name"] = currentPlayerName
				TetrisAttack_FlightpathHighScores[destName][srcName]["score"] = playerScore
				TetrisAttack_FlightpathHighScores[destName][srcName]["difficulty"] = currentDifficulty
				TetrisAttack_FlightpathHighScores[destName][srcName]["myName"] = ""
				TetrisAttack_FlightpathHighScores[destName][srcName]["myScore"] = 0
				TetrisAttack_FlightpathHighScores[destName][srcName]["myDiff"] = 0
			end
		end
		srcName = ""
		destName = ""
		if (newHigh) then
			TetrisAttack_UpdateFlightpathScoreboard("Last game", playerScore, currentDifficulty)
			TetrisAttackFlightpathScores:SetText("New high score!")
			if (playSounds == 1) then PlaySoundFile(soundEffects["ding"]) end
		else
			TetrisAttack_UpdateFlightpathScoreboard("Last game", playerScore, currentDifficulty)
		end
		TetrisAttack_ScrubGameVariables()
	end
end

--A whole bunch of stuff like registering events, creating textures and setting important things to their default values. Also loading saved settings and making the options menu reflect that.
function TetrisAttack_Init()
	DEFAULT_CHAT_FRAME:AddMessage("Welcome to Flightpath Puzzle League!", .5, .8, 1)
	if (not TetrisAttack_SavedSettings["firstRun"]) then
		TetrisAttack_SavedSettings["firstRun"] = 1
		TetrisAttackWelcomeMessage:SetText("Welcome to FPL! To get started, use this minimap button,   ---> \n   bind a key, or use /fpl to open the game window.\nCheck out the Tutorials if you've never played before.")
	end
	for i = 1, 120 do	--Precompute values to prevent costly calls to sin() during onupdate
		sinValues[i] = math.sin(math.rad(i*3))
	end
	TetrisAttackBaseFrame:RegisterEvent("CHAT_MSG_ADDON")
	TetrisAttackBaseFrame:RegisterEvent("PLAYER_REGEN_DISABLED")
	TetrisAttackBaseFrame:RegisterEvent("PLAYER_CONTROL_GAINED")
	TetrisAttackBaseFrame:RegisterEvent("PLAYER_ENTERING_WORLD")
	TetrisAttackBaseFrame:RegisterEvent("CHAT_MSG_SYSTEM")
	TetrisAttackBaseFrame:RegisterEvent("FRIENDLIST_UPDATE")
	TetrisAttackBaseFrame:RegisterEvent("RAID_ROSTER_UPDATE")
	TetrisAttackMinimapButton:SetPoint("TOPLEFT", "Minimap", "TOPLEFT", -8, -107)
	TetrisAttack_ChangeDifficulty(2)
	TetrisAttack_LoadSettings()
	TetrisAttack_InitOptions()
	TetrisAttackComment1:SetWidth(196)
	TetrisAttackMultiplayerNotice:SetText("You can also use the button on\n your friends list to challenge\n friends who have FPL installed.")
	for i = 1, 10 do
		if (not TetrisAttack_EndlessHighScores[i]["name"]) then
			TetrisAttack_EndlessHighScores[i]["name"] = ""
			TetrisAttack_EndlessHighScores[i]["score"] = 0
			TetrisAttack_EndlessHighScores[i]["difficulty"] = 0
		end
	end
	local myRace, myRaceEn = UnitRace("player")
	myRaceGender = raceGenderCodes[myRace .. (UnitSex("player")-1)]
	local newTexture
	local newString
	for i = 1, 10 do
		maxComboAnims = 10
		currentAnims[i] = {["x"] = 0,
							["y"] = 0, 
							["texture"] = TetrisAttackFieldBorder:CreateTexture("TetrisAttackComboAnim".. i, "ARTWORK"),
							["fontstring"] = TetrisAttackFieldBorder:CreateFontString("TetrisAttackComboString".. i, "OVERLAY", "BossEmoteNormalHuge"), 
							["timer"] = 0}
		newTexture = _G["TetrisAttackComboAnim" .. i]
		newString = _G["TetrisAttackComboString" .. i]
		newTexture:SetTexture("Interface\\BUTTONS\\UI-Quickslot")
		newString:Hide()
		newString:SetJustifyH("CENTER")
		newTexture:Hide()
	end
	TetrisAttack_ChangeEndlessScoreDisplay(1, 5, "--Personal Scores--")
	englishFaction, localizedFaction = UnitFactionGroup("player")
	tinsert(UISpecialFrames,"TetrisAttack1");
	tinsert(UISpecialFrames,"TetrisAttackChallengeAcceptFrame");
	TetrisAttack_InitFriendsListModifications()
	--TetrisAttack_InitRaidFrameModifications()
	TetrisAttack_TakeTaxiNode_Original = TakeTaxiNode
	hooksecurefunc("FriendsList_Update", TetrisAttack_FriendsList_Update)
--	TetrisAttack_ClickFriendButton_Original = FriendsFrameFriendButton_OnClick
--	FriendsFrameFriendButton_OnClick = TetrisAttack_ClickFriendButton
	TakeTaxiNode = TetrisAttack_TakeTaxi
	
	soundEffects["puzzleWin"] = "Sound\\Spells\\CrowdCheer" .. englishFaction .. "2.wav"
	soundEffects["victory"] = soundEffects["victory"] .. englishFaction .. ".wav"
	if (englishFaction == "Horde") then
		soundEffects["defeat"] = soundEffects["defeat"] .. "Alliance.wav"
	else
		soundEffects["defeat"] = soundEffects["defeat"] .. "Horde.wav"
	end
	
	TetrisAttack_MPNameBox:SetAutoFocus(false)
	TetrisAttack1:EnableKeyboard(0)
	
	for row = 1, numRows do
		blockTable[row] = {1}
		if (row <= numRowsVisible) then
			blockTextures[row] = {1}
			blockTexturesMP[row] = {1}
			for column = 1, 6 do
				local index = column + 6*(row-1)
				blockTextures[row][column] = TetrisAttackField:CreateTexture("TetrisAttackField_block" .. index, "BACKGROUND")
				blockTextures[row][column]:SetWidth(32)
				blockTextures[row][column]:SetHeight(32)
				blockTextures[row][column]:ClearAllPoints()
				blockTextures[row][column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20, (row * 32) + 10)
			
				blockTexturesMP[row][column] = TetrisAttackOpponentField:CreateTexture("TetrisAttackField_block" .. index, "BACKGROUND")
				blockTexturesMP[row][column]:SetWidth(32)
				blockTexturesMP[row][column]:SetHeight(32)
				blockTexturesMP[row][column]:ClearAllPoints()
				blockTexturesMP[row][column]:SetPoint("TOPLEFT", TetrisAttackOpponentField, "BOTTOMLEFT", (column*32) - 20, (row * 32) + 10)
				blockTexturesMP[row][column]:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\block0")
				blockTexturesMP[row][column]:SetVertexColor(blockColor[1][1],blockColor[1][2],blockColor[1][3], color)
				
			end
		end
	end
	blockTable[0] = {1,1,1,1,1,1}
	for column = 1, 6 do
		newBlockTextures[column] = TetrisAttackField:CreateTexture("TetrisAttackField_newBlock" .. column , "BACKGROUND")
		newBlockTextures[column]:SetWidth(32)
		newBlockTextures[column]:ClearAllPoints()
		newBlockTextures[column]:SetPoint("TOPLEFT", TetrisAttackField, "BOTTOMLEFT", (column*32) - 20 , 10)
		newBlockTextures[column]:SetTexCoord(0,1,0,0)
		newBlockTextures[column]:SetHeight(.01)
		
		newBlockTexturesMP[column] = TetrisAttackOpponentField:CreateTexture("TetrisAttackField_newBlock" .. column , "BACKGROUND")
		newBlockTexturesMP[column]:SetWidth(32)
		newBlockTexturesMP[column]:ClearAllPoints()
		newBlockTexturesMP[column]:SetPoint("TOPLEFT", TetrisAttackOpponentField, "BOTTOMLEFT", (column*32) - 20 , 10)
		newBlockTexturesMP[column]:SetTexCoord(0,1,0,0)
		newBlockTexturesMP[column]:SetHeight(.01)
		lastBlock = block
	end
	
	local _G = getfenv()
	for color = 1, 6 do
		local tex = _G["TAEditButtonTexture" .. color]
		tex:SetVertexColor(blockColor[color][1],blockColor[color][2],blockColor[color][3], 1)
	end
	cursorTexture = TetrisAttackFieldBorder:CreateTexture("TetrisAttackField_cursor", "ARTWORK")
	cursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\TAcursor")
	cursorTexture:SetWidth(128)
	cursorTexture:SetHeight(64)
	cursorTexture:ClearAllPoints()
	cursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackFieldBorder, "BOTTOMLEFT", (cursorX*32) - 52, ((cursorY-1) * 32) - 6)
	cursorTexture:Hide()
	MPcursorTexture = TetrisAttackOpponentFieldBorder:CreateTexture("TetrisAttackField_cursor", "ARTWORK")
	MPcursorTexture:SetTexture("Interface\\AddOns\\FPPuzzleLeague\\Images\\TAcursor")
	MPcursorTexture:SetWidth(128)
	MPcursorTexture:SetHeight(64)
	MPcursorTexture:ClearAllPoints()
	MPcursorTexture:SetPoint("BOTTOMLEFT", TetrisAttackOpponentFieldBorder, "BOTTOMLEFT", (MPcursorX*32) - 52, ((MPcursorY-1) * 32) - 6)
	MPcursorTexture:Hide()
end

function TetrisAttack_MinimapDrag(isDragging)
	if (IsShiftKeyDown) then
		miniMapDrag = isDragging
	else
		miniMapDrag = false
	end
end  

--Changes a saved setting to the value given. Called by the XML file quite a bit.
function TetrisAttack_ChangeSavedSetting(setting, value)
	TetrisAttack_SavedSettings[setting] = (value or 0) -- Don't save nil, because that would break the load function.
	TetrisAttack_LoadSettings()
end
	
function TetrisAttack_LoadSettings()
	autoPause = TetrisAttack_SavedSettings["autoPause"] or autoPause
	combatPanic = TetrisAttack_SavedSettings["combatPanic"] or combatPanic
	autoOpenFlightpath = TetrisAttack_SavedSettings["autoOpenFlightpath"] or autoOpenFlightpath
	enableMinimap = TetrisAttack_SavedSettings["enableMinimap"] or enableMinimap
	playSounds = TetrisAttack_SavedSettings["playSounds"] or playSounds
	useClassIcons = TetrisAttack_SavedSettings["useClassIcons"] or useClassIcons
	enableMouse = TetrisAttack_SavedSettings["enableMouse"] or enableMouse
	fascistKeyboard = TetrisAttack_SavedSettings["fascistKeyboard"] or fascistKeyboard
	useFriendsList = TetrisAttack_SavedSettings["useFriendsList"] or fascistKeyboard
	verbose = TetrisAttack_SavedSettings["verbose"] or fascistKeyboard
	
	upKey = TetrisAttack_SavedSettings["upKey"] or upKey
	downKey = TetrisAttack_SavedSettings["downKey"] or downKey
	leftKey = TetrisAttack_SavedSettings["leftKey"] or leftKey
	rightKey = TetrisAttack_SavedSettings["rightKey"] or rightKey
	flipKey = TetrisAttack_SavedSettings["flipKey"] or flipKey
	bumpKey = TetrisAttack_SavedSettings["bumpKey"] or bumpKey
	pauseKey = TetrisAttack_SavedSettings["pauseKey"] or pauseKey
	if (fascistKeyboard == 1) then 
		TetrisAttackKeyBindingsText:SetText("---Current Key Bindings---")
		r = 1.0
		g = .82
		b = 0
	else
		TetrisAttackKeyBindingsText:SetText("--Use WoW's Keybind Menu--")
		r = .5
		g = .5
		b = .5
	end
	
	TetrisAttackKeyUpText:SetTextColor(r, g, b)
	TetrisAttackKeyDownText:SetTextColor(r, g, b)
	TetrisAttackKeyLeftText:SetTextColor(r, g, b)
	TetrisAttackKeyRightText:SetTextColor(r, g, b)
	TetrisAttackKeyFlipText:SetTextColor(r, g, b)
	TetrisAttackKeyBumpText:SetTextColor(r, g, b)
	TetrisAttackKeyPauseText:SetTextColor(r, g, b)
	
	TetrisAttackKeyUpText:SetText("Up:     " .. upKey)
	TetrisAttackKeyDownText:SetText("Down: " .. downKey)
	TetrisAttackKeyLeftText:SetText("Left:   " .. leftKey)
	TetrisAttackKeyRightText:SetText("Right: " .. rightKey)
	TetrisAttackKeyFlipText:SetText("Flip:    " .. flipKey)
	TetrisAttackKeyBumpText:SetText("Bump: " .. bumpKey)
	TetrisAttackKeyPauseText:SetText("Pause: " .. pauseKey)
	
	
	TetrisAttack1:EnableMouse(enableMouse == 1)
	TetrisAttackMinimapButton:SetPoint("TOPLEFT", "Minimap", "TOPLEFT", (TetrisAttack_SavedSettings["MinimapX"] or -8), (TetrisAttack_SavedSettings["MinimapY"] or -108))
	if (enableMinimap == 1) then TetrisAttackMinimapButton:Show() else TetrisAttackMinimapButton:Hide() end
end

--Check all appropriate check boxes.
function TetrisAttack_InitOptions()
	TetrisAttackAutoPauseCheckButton:SetChecked(autoPause)
	TetrisAttackCombatPanicCheckButton:SetChecked(combatPanic)
	TetrisAttackAutoFlightpathCheckButton:SetChecked(autoOpenFlightpath)
	TetrisAttackUseFriendsCheckButton:SetChecked(useFriendsList)
	TetrisAttackEnableSoundsCheckButton:SetChecked(playSounds)
	TetrisAttackEnableMinimapCheckButton:SetChecked(enableMinimap)
	TetrisAttackUseClassIconsCheckButton:SetChecked(useClassIcons)
	TetrisAttackEnableMouseCheckButton:SetChecked(enableMouse)
	TetrisAttackReplaceHotkeysCheckButton:SetChecked(fascistKeyboard)
	TetrisAttackVerboseCheckButton:SetChecked(verbose)
end

function TetrisAttack_BindKeys()
	TetrisAttack1:EnableKeyboard(true)
	keyToBind = 1
	TetrisAttackKeyBindingsText:SetText("---" .. keyDescsInOrder[keyToBind] .. "---")
end

--Blatantly lifted from Bongos2, and was apparently lifted from Trinity before that. Thanks trinity!
function TetrisAttack_UpdateMinimapButton()
	local pos;
	local xpos,ypos = GetCursorPosition()
	local xmin,ymin = Minimap:GetLeft(), Minimap:GetBottom()

	xpos = xmin - xpos / Minimap:GetEffectiveScale() + 70
	ypos = ypos / Minimap:GetEffectiveScale() - ymin - 70

	pos = atan2(ypos, xpos)

	xpos = 80 * cos(pos)
	ypos = 80 * sin(pos)
	TetrisAttack_SavedSettings["MinimapX"] = 52 - xpos
	TetrisAttack_SavedSettings["MinimapY"] = ypos - 55
	TetrisAttackMinimapButton:SetPoint("TOPLEFT", "Minimap", "TOPLEFT", 52 - xpos, ypos - 55)
end