﻿--[[
Equipment Evaluator
DPS Calculations by Lerkur (lerkur at hotmail dot com)
Based on code by Kevin Clement <kevin.clement@gmail.com> (Arcuss)
With some data and formulas taken from Rating Buster

]]
EquipEval = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceDebug-2.0", "AceHook-2.1")
EquipEval.version = "1.0.4"
EquipEval.L = AceLibrary("AceLocale-2.2"):new("EquipEval")

local Bonus = AceLibrary("LibItemBonus-2.0")
local TipHooker = AceLibrary("TipHooker-1.0")
local L = EquipEval.L

-- Enable or disable debugging
EquipEval:SetDebugging(false)

-- Defaults
local profileDefaults = {
	EnemyLevelDifference			= 0,
	EnemyParryPercentage			= 5,
	EnemyDodgePercentage			= 5,
	EnemyArmorPercentage			= 20,
	EnemyBaseCritPercentage			= 5,
	EnemyResilience					= 0,
	EnemyResistance					= 0,
	EnemyIsBleedImmune				= 0,
	EnemyIsPoisonImmune				= 0,
	EnemyIsRootImmune				= 0,
	EnemyFractionWhiteDamage		= 0.5,
	EnemyHitsPerSecond				= 0.5,
	EnemyArmorPenetration			= 0,

	BonusHitPercent					= 0,
	BonusSpellHitPercent			= 0,
	BonusStatPercent				= 0,
	BonusCritDamagePercent			= 0,
	BonusCritChancePercent			= 0,
	BonusSpellCritChancePercent		= 0,
	BonusAP							= 0,
	BonusAPPercent					= 0,
	BonusHP							= 0,
	BonusWeaponSkill				= 0,
	BonusExpertise					= 0,
	BonusMana						= 0,
	BonusMP5						= 0,
	BonusSpellDamage				= 0,

	DisplayDPSTotal					= 0,
	DisplayHealingTotal				= -2,
	DisplayMitTotal					= -2,
	ItemRecalculationDelay			= 2,

	IncludeStrikes					= 1,
	PermanentMangleOnTarget			= 0,
	BurstFactor						= 1,
	ManaPotionUse					= 1,
	NumberOfAdditionalTargets		= 0,

	EnergyGenerationPerTick			= 20,

	PercentTimeIn5SecRule			= 80,
	PercentIdleTimeWhileHealing		= 20,
	ManaVolatilityDampingFactor		= 0.75,
	DPMtoDPSBalanceFactor			= 0.1,
	CastTimePadding					= 0.1,

	TotalDisplayed					= true,
	DPSDisplayed					= true,
	MitigationDisplayed				= true,
	HealingDisplayed				= true,
	HealingDurationDisplayed		= false,
	ChosenAttacksDisplayed			= true,
	ChosenHealsDisplayed			= true,
	ScanCurrentGearBonuses			= true,
	CompareToCurrentItems			= true,
	CalculateOnUseEffects			= false,

	AllowSearingPain				= false,
	CalculateSpiritTapEffect		= true,
	WindfuryTotemUptime				= 0,
	ImprovedWindfuryTotemRank		= 0,
	
	MakeGemAssumptions				= true,
	AssumedRedGemLink				= "None",
	AssumedYellowGemLink			= "None",
	AssumedBlueGemLink				= "None",
	AssumedMetaGemLink				= "None",
	
	AutoSetHealingVars				= false,
	AutoSet5SRTrackingSeconds		= 1800,
	AutoSetIdleTrackingFights		= 50,
	AutoSetHealingTimeThreshold		= 5,
	
	DisplayTotalItemRating			= false,
	ItemRatingWeightDPS				= 1,
	ItemRatingWeightMit				= 1,
	ItemRatingWeightHeal			= 1,
}


EquipEval:RegisterDB("EquipEvalDB")
EquipEval:RegisterDefaults('profile', profileDefaults)


---------------------------------------------------------------------------------------
-- Returns the default value for the given variable
---------------------------------------------------------------------------------------
function EquipEval:GetDefaultValue(variable)
	return profileDefaults[variable]
end


---------------------------------------------------------------------------------------
-- Toggles the config window on and off
---------------------------------------------------------------------------------------
function EquipEvalOptions_Toggle()
	if(EquipEvalOptionsFrame:IsVisible()) then
		EquipEvalOptionsFrame:Hide();
	else
		EquipEvalOptionsFrame:Show();
	end
end

---------------------------------------------------------------------------------------
-- Toggles the gem options window on and off
---------------------------------------------------------------------------------------
function EquipEvalGemOptions_Toggle()
	if(EquipEvalGemOptionsFrame:IsVisible()) then
		EquipEvalGemOptionsFrame:Hide();
	else
		EquipEvalGemOptionsFrame:Show();
	end
end

---------------------------------------------------------------------------------------
-- Toggles the rating options window on and off
---------------------------------------------------------------------------------------
function EquipEvalRatingOptions_Toggle()
	if(EquipEvalRatingOptionsFrame:IsVisible()) then
		EquipEvalRatingOptionsFrame:Hide();
	else
		EquipEvalRatingOptionsFrame:Show();
	end
end


function EquipEval:OnInitialize()
    -- Called when the addon is loaded

	-- Register simple slash command
	SlashCmdList["EQUIPEVAL"] = 	function(msg)
						EquipEval:SlashCommandHandler(msg);
					end
	SLASH_EQUIPEVAL1 = "/ee";
	SLASH_EQUIPEVAL2 = "/equipeval";

end

function EquipEval:SlashCommandHandler(msg)
	if (msg == "debug") then
		if EquipEval:IsDebugging() then
			EquipEval:SetDebugging(false)
			out("EquipEval: Debugging disabled.")
		else
			EquipEval:SetDebugging(true)
			out("EquipEval: Debugging enabled.")
		end
	elseif (msg == "debug 0") or (msg == "debug off") then
		EquipEval:SetDebugging(false)
		out("EquipEval: Debugging disabled.")
	elseif (msg == "debug 1") or (msg == "debug on") then
		EquipEval:SetDebugging(true)
		out("EquipEval: Debugging enabled.")
	elseif (msg =="variables") then
		for index,value in pairs(EquipEval.db.profile) do
			out("EquipEval: "..index.." = "..tostring(value))
		end
	elseif string.sub(msg, 1, 3) == "get" then
		local var = string.sub(msg, 5)
		if var ~= nil then
			if EquipEval.db.profile[var] ~= nil then
				out("EquipEval: "..var.." = ".. tostring(EquipEval.db.profile[var]))
			else
				out("EquipEval: No such variable")
			end
		else
			out("EquipEval: No variable given")
		end
	elseif string.sub(msg, 1, 3) == "set" then
		local var = nil
		local val = nil
		
		local loc = string.find(string.sub(msg, 5), " ")
		if loc then		
			var = string.sub(msg, 5, (loc + 3))
			val = string.sub(msg, (loc + 5))
		end
		
		if var ~= nil and val ~= nil then
			if EquipEval.db.profile[var] ~= nil then
				if val == "true" then
					val = true
				elseif val == "false" then
					val = false
				elseif tonumber(val) then
					val = tonumber(val)
				else
					val = strtrim(val)
				end
				EquipEval.db.profile[var] = val
				out("EquipEval: "..var.." = "..tostring(EquipEval.db.profile[var]))
			else
				out("EquipEval: No such variable")
			end
		else
			out("EquipEval: Variable and value needed")
		end
	elseif string.sub(msg, 1, 3) == "gem" then
		EquipEvalGemOptions_Toggle()
	elseif string.sub(msg, 1, 6) == "rating" then
		EquipEvalRatingOptions_Toggle()
	else
		EquipEvalOptions_Toggle();
	end
end


function EquipEval:OnEnable()
    -- Called when the addon is enabled
    TipHooker:Hook(self.ProcessTooltip, "item")
    if not EquipEval:IsHooked(Bonus, "GetUnitEquipment") then
		EquipEval:Hook(Bonus, "GetUnitEquipment", EquipEval.IBLGetUnitEquipmentHook)
	end
	if not EquipEval:IsHooked(Bonus, "BuildBonusSet") then
		EquipEval:Hook(Bonus, "BuildBonusSet", EquipEval.IBLBuildBonusSetHook)
	end
end

function EquipEval:OnDisable()
    -- Called when the addon is disabled
    TipHooker:Unhook(self.ProcessTooltip, "item")
    if EquipEval:IsHooked(Bonus, "GetUnitEquipment") then
		EquipEval:Unhook(Bonus, "GetUnitEquipment")
	end
	if EquipEval:IsHooked(Bonus, "BuildBonusSet") then
		EquipEval:Unhook(Bonus, "BuildBonusSet")
	end
end


---------------------------------------------------------------------------------------
-- Does a shallow copy of a table
---------------------------------------------------------------------------------------
function EquipEval:CopyTable(object)
	if type(object) ~= "table" then
		return object
	end
	
	local NewTable = {}
	for index,value in pairs(object) do
		NewTable[index] = EquipEval:CopyTable(value)
	end
	
	return NewTable
end


---------------------------------------------------------------------------------------
-- Used to calculate DPS given an info object that came out of ItemBonusLib
--
-- infoObj: already parsed info object
---------------------------------------------------------------------------------------
function EquipEval:CalculateDPS(infoObj, WeaponSlotName, WeaponHands, itemSubType)
	local CurrentLevel = UnitLevel("player")
	local LevelRating = EquipEval:GetLevelRating(CurrentLevel)
	local LevelRatingDS = EquipEval:GetLevelRatingDefensiveStats(CurrentLevel)
	local _, englishClass = UnitClass("player");

	local dpsChange = 0
	local dpsTotal = 0
	local mitChange = 0
	local mitTotal = 0
	local healChange = 0
	local healTotal = 0
	local OutputLines = { Healing = {}, Mitigation = {}, DPS = {}}
	local InfoLine = nil
	
	WeaponSlotName = WeaponSlotName or ""
	WeaponHands = WeaponHands or 0
	

	if englishClass == "HUNTER" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateHunterDPS(infoObj, CurrentLevel, LevelRating, WeaponSlotName, WeaponHands)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateHunterMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
	elseif englishClass == "ROGUE" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateRogueDPS(infoObj, CurrentLevel, LevelRating, WeaponSlotName, WeaponHands, itemSubType)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateRogueMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
	elseif englishClass == "SHAMAN" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateShamanDPS(infoObj, CurrentLevel, LevelRating, WeaponSlotName, WeaponHands, itemSubType)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateShamanMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
		if EquipEval.db.profile["HealingDisplayed"] then
			healChange, healTotal, InfoLine = EquipEval:CalculateShamanHealing(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Healing)
		end
	elseif englishClass == "DRUID" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateDruidDPS(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateDruidMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
		if EquipEval.db.profile["HealingDisplayed"] then
			healChange, healTotal, InfoLine = EquipEval:CalculateDruidHealing(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Healing)
		end
	elseif englishClass == "PALADIN" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculatePaladinDPS(infoObj, CurrentLevel, LevelRating, WeaponSlotName, WeaponHands, itemSubType)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculatePaladinMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
		if EquipEval.db.profile["HealingDisplayed"] then
			healChange, healTotal, InfoLine = EquipEval:CalculatePaladinHealing(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Healing)
		end
	elseif englishClass == "WARRIOR" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateWarriorDPS(infoObj, CurrentLevel, LevelRating, WeaponSlotName, WeaponHands, itemSubType)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateWarriorMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
	elseif englishClass == "MAGE" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateMageDPS(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateMageMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
	elseif englishClass == "PRIEST" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculatePriestDPS(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculatePriestMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
		if EquipEval.db.profile["HealingDisplayed"] then
			healChange, healTotal, InfoLine = EquipEval:CalculatePriestHealing(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Healing)
		end
	elseif englishClass == "WARLOCK" then
		if EquipEval.db.profile["DPSDisplayed"] then
			dpsChange, dpsTotal, InfoLine = EquipEval:CalculateWarlockDPS(infoObj, CurrentLevel, LevelRating)
			EquipEval:AddStringToTable(InfoLine, OutputLines.DPS)
		end
		if EquipEval.db.profile["MitigationDisplayed"] then
			mitChange, mitTotal, InfoLine = EquipEval:CalculateWarlockMit(infoObj, CurrentLevel, LevelRating, LevelRatingDS)
			EquipEval:AddStringToTable(InfoLine, OutputLines.Mitigation)
		end
	end
	

	return dpsChange, dpsTotal, mitChange, mitTotal, healChange, healTotal, OutputLines
end



---------------------------------------------------------------------------------------
-- Formulates the display string to be used in the tooltip
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayString(Base, Original, Enchant, Socket, Set, Header, preChar, postChar, decimals)
	local DisplayString = ""
	local FormatString = "%."..tostring(decimals).."f"	
	
	local EnchantString = format(FormatString, Enchant)..postChar
	if Enchant > 0 then EnchantString = "+"..EnchantString end
	
	local SocketString = format(FormatString, Socket)..postChar
	if Socket > 0 then SocketString = "+"..SocketString end
	
	local SetString = format(FormatString, Set)..postChar
	if Set > 0 then SetString = "+"..SetString end
	
	if preChar == "+" and Base < 0 then
		preChar = ""
	end

	local DisplayString = string.format(Header..preChar.."%s", format(FormatString, Base)..postChar)
	if Enchant ~= 0 then
		DisplayString = string.format("%s %s(%s)|r", DisplayString, GREEN_FONT_COLOR_CODE, EnchantString)
	end
	if Socket ~= 0 then
		DisplayString = string.format("%s %s(%s)|r", DisplayString, HIGHLIGHT_FONT_COLOR_CODE, SocketString)
	end
	if Set ~= 0 then
		DisplayString = string.format("%s %s(%s)|r", DisplayString, LIGHTYELLOW_FONT_COLOR_CODE, SetString)
	end

	return DisplayString
end


---------------------------------------------------------------------------------------
-- Formulates the left display string for DPS
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringLeftDPS(dpsOriginal, dpsSet, dpsEnchant, dpsSocket, dpsBase, dpsOriginalT, dpsSetT, dpsEnchantT, dpsSocketT, dpsBaseT)

	local Header = "DPS: "
	local preChar = ""
	local postChar = ""
	local decimals = 2
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayDPSTotal"] < 0 then
		Multiplier = 100 / (dpsOriginalT - dpsOriginal)
		preChar = "+"
		postChar = "%"
	elseif EquipEval.db.profile["DisplayDPSTotal"] > 0 then
		dpsBase, dpsOriginal, dpsEnchant, dpsSocket, dpsSet = dpsBaseT, dpsOriginalT, dpsEnchantT, dpsSocketT, dpsSetT
	end

	return EquipEval:FormulateDisplayString(Multiplier * dpsBase, Multiplier * dpsOriginal, Multiplier * dpsEnchant, Multiplier * dpsSocket, Multiplier * dpsSet, Header, preChar, postChar, decimals)
end

---------------------------------------------------------------------------------------
-- Formulates the right display string for DPS
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringRightDPS(dpsBase, dpsGrandTotal, dpsGrandTotalT)
	
	if dpsBase == dpsGrandTotal then return "" end	
	
	local Header = "Total DPS: "
	local preChar = ""
	local postChar = ""
	local decimals = 2
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayDPSTotal"] < 0 then
		Multiplier = 100 / (dpsGrandTotalT - dpsGrandTotal)
		preChar = "+"
		postChar = "%"
	elseif EquipEval.db.profile["DisplayDPSTotal"] > 0 then
		dpsGrandTotal = dpsGrandTotalT
	end	

	return EquipEval:FormulateDisplayString(Multiplier * dpsGrandTotal, 0, 0, 0, 0, Header, preChar, postChar, decimals)
end


---------------------------------------------------------------------------------------
-- Formulates the left display string for Mitigation
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringLeftMit(mitOriginal, mitSet, mitEnchant, mitSocket, mitBase, mitOriginalT, mitSetT, mitEnchantT, mitSocketT, mitBaseT)

	local Header = "Mit: "
	local preChar = ""
	local postChar = "%"
	local decimals = 3
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayMitTotal"] < -1 then
		Multiplier = 100 /(mitOriginalT - mitOriginal)
		Header = "TTL: "
		preChar = "+"
	elseif EquipEval.db.profile["DisplayMitTotal"] < 0 then
		Multiplier = 100 / (100 - (mitOriginalT - mitOriginal))
		preChar = "+"
	elseif EquipEval.db.profile["DisplayMitTotal"] > 0 then
		mitBase, mitOriginal, mitEnchant, mitSocket, mitSet = mitBaseT, mitOriginalT, mitEnchantT, mitSocketT, mitSetT
	end

	return EquipEval:FormulateDisplayString(Multiplier * mitBase, Multiplier * mitOriginal, Multiplier * mitEnchant, Multiplier * mitSocket, Multiplier * mitSet, Header, preChar, postChar, decimals)
end

---------------------------------------------------------------------------------------
-- Formulates the right display string for Mitigation
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringRightMit(mitBase, mitGrandTotal, mitGrandTotalT)

	if mitBase == mitGrandTotal then return "" end	

	local Header = "Total Mit: "
	local preChar = ""
	local postChar = "%"
	local decimals = 3
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayMitTotal"] < -1 then
		Multiplier = 100 / (mitGrandTotalT - mitGrandTotal)
		Header = "Total TTL: "
		preChar = "+"
	elseif EquipEval.db.profile["DisplayMitTotal"] < 0 then
		Multiplier = 100 / (100 - (mitGrandTotalT - mitGrandTotal))
		preChar = "+"
	elseif EquipEval.db.profile["DisplayMitTotal"] > 0 then
		mitGrandTotal = mitGrandTotalT
	end

	return EquipEval:FormulateDisplayString(Multiplier * mitGrandTotal, 0, 0, 0, 0, Header, preChar, postChar, decimals)
end


---------------------------------------------------------------------------------------
-- Formulates the left display string for healing
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringLeftHeal(healOriginal, healSet, healEnchant, healSocket, healBase, healOriginalT, healSetT, healEnchantT, healSocketT, healBaseT)

	local Header = "HPS: "
	local preChar = ""
	local postChar = ""
	local decimals = 2
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayHealingTotal"] < -1 then
		Header = "Healing: "
	end
	if EquipEval.db.profile["DisplayHealingTotal"] < 0 then
		Multiplier = 100 / (healOriginalT - healOriginal)
		preChar = "+"
		postChar = "%"
	elseif EquipEval.db.profile["DisplayHealingTotal"] > 0 then
		healBase, healOriginal, healEnchant, healSocket, healSet = healBaseT, healOriginalT, healEnchantT, healSocketT, healSetT
	end

	return EquipEval:FormulateDisplayString(Multiplier * healBase, Multiplier * healOriginal, Multiplier * healEnchant, Multiplier * healSocket, Multiplier * healSet, Header, preChar, postChar, decimals)
end

---------------------------------------------------------------------------------------
-- Formulates the right display string for healing
---------------------------------------------------------------------------------------
function EquipEval:FormulateDisplayStringRightHeal(healBase, healGrandTotal, healGrandTotalT)
	
	if healBase == healGrandTotal then return "" end	
	
	local Header = "Total HPS: "
	local preChar = ""
	local postChar = ""
	local decimals = 2
	
	local Multiplier = 1
	if EquipEval.db.profile["DisplayHealingTotal"] < -1 then
		Header = "Total Healing: "
	end
	if EquipEval.db.profile["DisplayHealingTotal"] < 0 then
		Multiplier = 100 / (healGrandTotalT - healGrandTotal)
		preChar = "+"
		postChar = "%"
	elseif EquipEval.db.profile["DisplayHealingTotal"] > 0 then
		healGrandTotal = healGrandTotalT
	end	

	return EquipEval:FormulateDisplayString(Multiplier * healGrandTotal, 0, 0, 0, 0, Header, preChar, postChar, decimals)
end


---------------------------------------------------------------------------------------
-- Appends TextTable to the bottom of "tooltip"
---------------------------------------------------------------------------------------
function EquipEval:AddLinesToTooltip(tooltip, TextTable)
	if #TextTable > 0 then
		for line_number = 1, #TextTable do
			tooltip:AddLine(TextTable[line_number])
		end
	end

	return
end

---------------------------------------------------------------------------------------
-- Adds a string to a table of strings
---------------------------------------------------------------------------------------
function EquipEval:AddStringToTable(string, stringTable)
	if not stringTable then stringTable = { } end
	if type(stringTable) ~= "table" then stringTable = {tostring(stringTable)} end
	if not string then return stringTable end
	
	if type(string) == "table" then
		for index, value in pairs(string) do
			stringTable = EquipEval:AddStringToTable(value, stringTable)
		end
	else
		table.insert(stringTable, tostring(string))
	end

	return stringTable
end

---------------------------------------------------------------------------------------
-- Runs whenever a tooltip is displayed.  This is controlled by TipHooker and calls into
-- this method.
--
-- tooltip: Tooltip object (table)
-- name: Name of the item (string)
-- link: ItemLink (string)
---------------------------------------------------------------------------------------
local RecentLinks = { }
local CacheCleared = nil
function EquipEval.ProcessTooltip(tooltip, name, link)
	local origInfo, enchantInfo, socketInfo, setInfo
	
	local dpsOriginal, dpsSet, dpsEnchant, dpsSocket, dpsBase, dpsGrandTotal = 0
	local dpsOriginalT, dpsSetT, dpsEnchantT, dpsSocketT, dpsBaseT, dpsGrandTotalT = 0
	
	local mitOriginal, mitSet,  mitEnchant, mitSocket, mitBase, mitGrandTotal = 0
	local mitOriginalT, mitSetT, mitEnchantT, mitSocketT, mitBaseT, mitGrandTotalT = 0
	
	local healOriginal, healSet, healEnchant, healSocket, healBase, healGrandTotal = 0
	local healOriginalT, healSetT, healEnchantT, healSocketT, healBaseT, healGrandTotalT = 0
	
	local dpsCurrent, mitCurrent, healCurrent
	local linesToAdd = { Info = {}, Healing = {}, Mitigation = {}, DPS = {}, Rating = {} }
	local InfoLine = nil

	-- If this is the first scan since initializing, clear the item cache
	if not CacheCleared then
		EquipEval:AddAdditionalPatterns()
		Bonus:ClearCache()
		CacheCleared = GetTime()
	end

	-- return if this is a tooltip without a name, happens on empty inventory
	if not name then return end
	
	-- If this is a link without an item string, return nothing
	if not EquipEval:ExtractItemString(link) then return end
	
	-- If this is a repeated tooltip, don't recalculate within the recalculation delay
	if not RecentLinks[link] or (GetTime() - RecentLinks[link].Time) >= EquipEval.db.profile["ItemRecalculationDelay"] then
		RecentLinks[link] = { Text = { }, Time = GetTime() }
	else
		EquipEval:AddLinesToTooltip(tooltip, RecentLinks[link].Text)
		tooltip:Show()
		return
	end
	
	--Check to see if we are acquiring gems for the gem options panel
	if EquipEval.db.profile["GemAcquisitionActive"] then
		EquipEval.db.profile["GemAcquisitionActive"]:SetText(link)
	end
	
	-- grab the bonus info: standard scan
	-- if there were problems return nothing
	origInfo = EquipEval:ScanItemLink(link)
	if not origInfo then return end
	
	-- handle sets
	setInfo = { bonuses = {}, set = nil}
	setInfo = EquipEval:MergeItemInfo(setInfo, origInfo)
	EquipEval:AddActivatedSetBonusToInfo(setInfo)
	
	--Handle item replacement
	local RIorigInfo, RIenchantInfo, RIsocketInfo, RIsetInfo, InfoLine, ItemSlotName, WeaponHands, itemSubType = EquipEval:GetReplacedItemInfos(link, setInfo)
	EquipEval:AddStringToTable(InfoLine, linesToAdd.Info)

	dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(origInfo, RIorigInfo), ItemSlotName, WeaponHands, itemSubType)
	dpsGrandTotal = dpsOriginal
	dpsGrandTotalT = dpsOriginalT
	mitGrandTotal = mitOriginal
	mitGrandTotalT = mitOriginalT
	healGrandTotal = healOriginal
	healGrandTotalT = healOriginalT

	-- prints out the stats on the item if debugging is enabled
	EquipEval:PrintStats(origInfo, "Orig Info")
	EquipEval:PrintStats(RIorigInfo, "Replacing Info")
	

	-- if there wasnt a total and there isnt
	-- the potential for set bonus, then return
	if (dpsOriginal == 0 and mitOriginal == 0 and healOriginal == 0) and not origInfo.set then return end


	--calculate set effect
	dpsCurrent, dpsCurrentT, mitCurrent, mitCurrentT, healCurrent, healCurrentT, InfoLine = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(setInfo, RIsetInfo), ItemSlotName, WeaponHands, itemSubType)
	local ItemRating = 100 * EquipEval:CalculateItemRating(dpsCurrent, dpsCurrentT, mitCurrent, mitCurrentT, healCurrent, healCurrentT)
	--EquipEval:AddStringToTable(InfoLine, linesToAdd)
	dpsSet = dpsCurrent - dpsOriginal
	dpsSetT = dpsCurrentT - dpsOriginalT
	mitSet = mitCurrent - mitOriginal
	mitSetT = mitCurrentT - mitOriginalT
	healSet = healCurrent - healOriginal
	healSetT = healCurrentT - healOriginalT

	dpsGrandTotal = dpsCurrent
	dpsGrandTotalT = dpsCurrentT
	mitGrandTotal = mitCurrent
	mitGrandTotalT = mitCurrentT
	healGrandTotal = healCurrent
	healGrandTotalT = healCurrentT


	-- get an info object that is the item without enchants
	enchantInfo=EquipEval:ScanItemLink(EquipEval:ClearEnchantsFromLink(link))
	if not enchantInfo then return end
	dpsCurrent, dpsCurrentT, mitCurrent, mitCurrentT, healCurrent, healCurrentT = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(enchantInfo, RIenchantInfo), ItemSlotName, WeaponHands, itemSubType)
	dpsEnchant = dpsOriginal - dpsCurrent
	dpsEnchantT = dpsOriginalT - dpsCurrentT
	mitEnchant = mitOriginal - mitCurrent
	mitEnchantT = mitOriginalT - mitCurrentT
	healEnchant = healOriginal - healCurrent
	healEnchantT = healOriginalT - healCurrentT

	-- get an info object that is the item without sockets
	socketInfo=EquipEval:ScanItemLink(EquipEval:ClearSocketsFromLink(link), "No Sockets")
	if not socketInfo then return end
	dpsCurrent, dpsCurrentT, mitCurrent, mitCurrentT, healCurrent, healCurrentT = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(socketInfo, RIsocketInfo), ItemSlotName, WeaponHands, itemSubType)
	dpsSocket = dpsOriginal - dpsCurrent
	dpsSocketT = dpsOriginalT - dpsCurrentT
	mitSocket = mitOriginal - mitCurrent
	mitSocketT = mitOriginalT - mitCurrentT
	healSocket = healOriginal - healCurrent
	healSocketT = healOriginalT - healCurrentT
	
	-- the base is the original minus the sockets and enchants
	dpsBase = dpsOriginal - dpsEnchant - dpsSocket
	dpsBaseT = dpsOriginalT - dpsEnchantT - dpsSocketT
	mitBase = mitOriginal - mitEnchant - mitSocket
	mitBaseT = mitOriginalT - mitEnchantT - mitSocketT
	healBase = healOriginal - healEnchant - healSocket
	healBaseT = healOriginalT - healEnchantT - healSocketT


	-- formulate display strings
	local leftString = EquipEval:FormulateDisplayStringLeftDPS(dpsOriginal, dpsSet, dpsEnchant, dpsSocket, dpsBase, dpsOriginalT, dpsSetT, dpsEnchantT, dpsSocketT, dpsBaseT)
	local leftStringMit = EquipEval:FormulateDisplayStringLeftMit(mitOriginal, mitSet, mitEnchant, mitSocket, mitBase, mitOriginalT, mitSetT, mitEnchantT, mitSocketT, mitBaseT)
	local leftStringHeal = EquipEval:FormulateDisplayStringLeftHeal(healOriginal, healSet, healEnchant, healSocket, healBase, healOriginalT, healSetT, healEnchantT, healSocketT, healBaseT)

	local rightString = EquipEval:FormulateDisplayStringRightDPS(dpsBase, dpsGrandTotal, dpsGrandTotalT)
	local rightStringMit = EquipEval:FormulateDisplayStringRightMit(mitBase, mitGrandTotal, mitGrandTotalT)
	local rightStringHeal = EquipEval:FormulateDisplayStringRightHeal(healBase, healGrandTotal, healGrandTotalT)
	
	local ItemRatingString = nil
	if EquipEval.db.profile['DisplayTotalItemRating'] then
		ItemRatingString = "Rating: "
		
		if ItemRating < 0 then
			ItemRatingString = ItemRatingString..RED_FONT_COLOR_CODE
		else
			ItemRatingString = ItemRatingString..GREEN_FONT_COLOR_CODE.."+"
		end
		
		ItemRatingString = ItemRatingString..format("%.2f", ItemRating).."|r"
	end


	if dpsGrandTotal == 0 and mitGrandTotal == 0 and healGrandTotal == 0 then return end

	--accumulate lines	
	if dpsGrandTotal ~= 0 then
		EquipEval:AddStringToTable(InfoLine.DPS, linesToAdd.DPS)
		EquipEval:AddStringToTable(leftString, linesToAdd.DPS)	
	end
	if dpsBase ~= dpsGrandTotal and EquipEval.db.profile['TotalDisplayed'] then
		EquipEval:AddStringToTable(rightString, linesToAdd.DPS)
	end

	if healGrandTotal ~= 0 then
		EquipEval:AddStringToTable(InfoLine.Healing, linesToAdd.Healing)
		EquipEval:AddStringToTable(leftStringHeal, linesToAdd.Healing)
	end
	if healBase ~= healGrandTotal and EquipEval.db.profile['TotalDisplayed'] then
		EquipEval:AddStringToTable(rightStringHeal, linesToAdd.Healing)
	end

	if mitGrandTotal ~= 0 and EquipEval.db.profile['MitigationDisplayed'] then
		EquipEval:AddStringToTable(InfoLine.Mitigation, linesToAdd.Mitigation)
		EquipEval:AddStringToTable(leftStringMit, linesToAdd.Mitigation)
	end
	if mitBase ~= mitGrandTotal and EquipEval.db.profile['TotalDisplayed'] and EquipEval.db.profile['MitigationDisplayed'] then
		EquipEval:AddStringToTable(rightStringMit, linesToAdd.Mitigation)
	end
	
	if ItemRatingString then
		EquipEval:AddStringToTable(ItemRatingString, linesToAdd.Rating)
	end
	
	
	-- add all the lines we accumulated
	local needSpace = #linesToAdd.Info > 1 or #linesToAdd.DPS > 1 or #linesToAdd.Healing > 1 or #linesToAdd.Mitigation > 1
	EquipEval:AddStringToTable(" ", RecentLinks[link].Text)
	if #linesToAdd.Info > 0 then
		EquipEval:AddStringToTable(linesToAdd.Info, RecentLinks[link].Text)
		EquipEval:AddStringToTable(" ", RecentLinks[link].Text)
	end
	if #linesToAdd.DPS > 0 then
		EquipEval:AddStringToTable(linesToAdd.DPS, RecentLinks[link].Text)
		if needSpace then EquipEval:AddStringToTable(" ", RecentLinks[link].Text) end
	end
	if #linesToAdd.Healing > 0 then
		EquipEval:AddStringToTable(linesToAdd.Healing, RecentLinks[link].Text)
		if needSpace then EquipEval:AddStringToTable(" ", RecentLinks[link].Text) end
	end
	if #linesToAdd.Mitigation > 0 then
		EquipEval:AddStringToTable(linesToAdd.Mitigation, RecentLinks[link].Text)
		if needSpace then EquipEval:AddStringToTable(" ", RecentLinks[link].Text) end
	end
	if #linesToAdd.Rating > 0 then
		EquipEval:AddStringToTable(linesToAdd.Rating, RecentLinks[link].Text)
	end

	if RecentLinks[link].Text[#RecentLinks[link].Text] == " " then
		RecentLinks[link].Text[#RecentLinks[link].Text] = nil
	end

	EquipEval:AddLinesToTooltip(tooltip, RecentLinks[link].Text)
	tooltip:Show()
end



