




---------------------------------------------------------------------------------------
-- Takes a Spell Table and a Rank and returns the spell details
---------------------------------------------------------------------------------------
function EquipEval:GetSpellDetails(SpellTable, SpellRank)
	local Level = EquipEval:GetSpellStat(SpellTable, SpellRank, "Level")
	local Coefficient = EquipEval:GetSpellStat(SpellTable, SpellRank, "Coefficient")
	local OverTimeCoefficient = EquipEval:GetSpellStat(SpellTable, SpellRank, "OverTimeCoefficient")
	local CastTime = EquipEval:GetSpellStat(SpellTable, SpellRank, "CastTime")
	local Duration = EquipEval:GetSpellStat(SpellTable, SpellRank, "Duration")
	local CoolDown = EquipEval:GetSpellStat(SpellTable, SpellRank, "CoolDown")
	local ManaCost = EquipEval:GetSpellStat(SpellTable, SpellRank, "ManaCost")
	local DirectHealAmount = EquipEval:GetSpellStat(SpellTable, SpellRank, "DirectHealAmount")
	local HealOverTimeAmount = EquipEval:GetSpellStat(SpellTable, SpellRank, "HealOverTimeAmount")
	local DirectDamageAmount = EquipEval:GetSpellStat(SpellTable, SpellRank, "DirectDamageAmount")
	local DamageOverTimeAmount = EquipEval:GetSpellStat(SpellTable, SpellRank, "DamageOverTimeAmount")
	local MaxTargets = EquipEval:GetSpellStat(SpellTable, SpellRank, "MaxTargets")
	local AdditionalTargetFactor = EquipEval:GetSpellStat(SpellTable, SpellRank, "AdditionalTargetFactor")


	return Level, Coefficient, OverTimeCoefficient, CastTime, Duration, CoolDown, ManaCost, DirectHealAmount, HealOverTimeAmount, DirectDamageAmount, DamageOverTimeAmount, MaxTargets, AdditionalTargetFactor
end

---------------------------------------------------------------------------------------
-- Takes a Spell Table, Rank, and a Stat name
--
-- Returns the stat's details
---------------------------------------------------------------------------------------
function EquipEval:GetSpellStat(SpellTable, SpellRank, StatName)

	if not SpellTable then
		return 0
	elseif not SpellTable["All"] then
		return 0
	elseif not SpellTable[SpellRank] then
		return SpellTable["All"][StatName] or 0
	end

	return SpellTable[SpellRank][StatName] or SpellTable["All"][StatName] or 0
end



---------------------------------------------------------------------------------------
-- Takes a Spell Table and returns the highest rated rank of heal
---------------------------------------------------------------------------------------
function EquipEval:ChooseHeal(SpellTable, CurrentLevel, ManaCostModifier, EffectModifier, CastTimeModifier, CoefficientModifier, CoefficientOverTimeModifier, PlusHealing, CritChance, CritModifier, Haste, DirectHealBonus, HealOverTimeBonus)
	local RankName = ""
	local Rating = 0
	local Healing = 0
	local Mana = 0
	local CastTime = 0
	local Duration = 0
	local CoolDown = 0
	local MaxTargets = 0


	CurrentLevel = CurrentLevel or UnitLevel("player")
	ManaCostModifier = ManaCostModifier or 1
	EffectModifier = EffectModifier or 1
	CastTimeModifier = CastTimeModifier or 0
	CoefficientModifier = CoefficientModifier or 0
	CoefficientOverTimeModifier = CoefficientOverTimeModifier or 0
	PlusHealing = PlusHealing or 0
	CritChance = CritChance or 0
	CritModifier = CritModifier or 0.5
	Haste = Haste or 0
	DirectHealBonus = DirectHealBonus or 0
	HealOverTimeBonus = HealOverTimeBonus or 0


	for SpellRank = 1, #SpellTable do
		local SpellLevel, SpellCoefficient, SpellOverTimeCoefficient, SpellCastTime, SpellDuration, SpellCoolDown, SpellManaCost, SpellDirectHealAmount, SpellHealOverTimeAmount, SpellDirectDamageAmount, SpellDamageOverTimeAmount, SpellMaxTargets, SpellAdditionalTargetFactor = EquipEval:GetSpellDetails(SpellTable, SpellRank)
		if SpellLevel > CurrentLevel then
			break
		end

		local DownRankModifier = EquipEval:GetDownrankModifier(SpellLevel, CurrentLevel)
		
		SpellCoefficient = (SpellCoefficient + CoefficientModifier) * DownRankModifier
		SpellOverTimeCoefficient = (SpellOverTimeCoefficient + CoefficientOverTimeModifier) * DownRankModifier

		SpellCastTime = (SpellCastTime - CastTimeModifier) / (1 + Haste)
		SpellCastTime = max(EquipEval:GetSpellGCD(Haste), SpellCastTime) + EquipEval.db.profile["CastTimePadding"]

		SpellDirectHealAmount = SpellDirectHealAmount + DirectHealBonus * DownRankModifier + SpellCoefficient * PlusHealing
		SpellDirectHealAmount = (1 + CritChance * CritModifier) * SpellDirectHealAmount
		SpellHealOverTimeAmount = SpellHealOverTimeAmount + HealOverTimeBonus * DownRankModifier
		SpellHealOverTimeAmount = SpellHealOverTimeAmount + SpellOverTimeCoefficient * PlusHealing

		local TotalHealing = EffectModifier * (SpellDirectHealAmount + SpellHealOverTimeAmount)
		for TargetNumber = 2, min(SpellMaxTargets, (1 + EquipEval.db.profile["NumberOfAdditionalTargets"])) do
			TotalHealing = TotalHealing + TotalHealing * SpellAdditionalTargetFactor ^ (TargetNumber - 1)
		end

		SpellManaCost = ManaCostModifier * SpellManaCost

		local CurSpellRating = EquipEval:GetSpellRating(TotalHealing, SpellManaCost, SpellCastTime)

		if CurSpellRating > Rating then
			Rating = CurSpellRating
			RankName = "(Rank "..SpellRank..")"

			Healing = TotalHealing
			Mana = SpellManaCost
			CastTime = SpellCastTime
			Duration = SpellDuration
			CoolDown = SpellCoolDown
			MaxTargets = SpellMaxTargets
		end
	end

	return RankName, Healing, Mana, CastTime, Duration, CoolDown, MaxTargets, Rating

end






---------------------------------------------------------------------------------------
-- Takes spell specifics and returns a rating that represents DPS and DPM efficiency
---------------------------------------------------------------------------------------
function EquipEval:GetSpellRating(SpellDmg, SpellMana, CastTime)
	if SpellMana == 0 or CastTime == 0 then
		return 0
	end

	--range of this is -1 (DPM) to 1 (DPS)
	local DPMtoDPSBalance = EquipEval.db.profile["DPMtoDPSBalanceFactor"]

	SpellMana = SpellMana ^ (1 - DPMtoDPSBalance)
	CastTime = CastTime ^ (1 + DPMtoDPSBalance)

	return (SpellDmg^2 / (SpellMana * CastTime))
end


---------------------------------------------------------------------------------------
-- Takes character level and spell level and returns the modifier to +dmg/healing
-- due to the difference between the two.
---------------------------------------------------------------------------------------
function EquipEval:GetDownrankModifier(SpellLevel, CharacterLevel)
	CharacterLevel = CharacterLevel or UnitLevel("player")
	SpellLevel = SpellLevel or 1

	local DownrankModifier = max(0, min(1, ((SpellLevel + 11) / CharacterLevel)))
	local Level20Modifier = max(0, min(1, (1 - ((20 - SpellLevel) * 0.0375))))

	return (DownrankModifier * Level20Modifier)
end


---------------------------------------------------------------------------------------
-- Copies an existing Spell Cycle to a new variable
---------------------------------------------------------------------------------------
function EquipEval:CopySpellCycle(SpellCycle)
	local NewSpellCycle = { }
	
	for SpellName,TimeFraction in pairs(SpellCycle) do
		NewSpellCycle[SpellName] = TimeFraction
	end
	
	return NewSpellCycle
end

---------------------------------------------------------------------------------------
-- Copies an existing Spell List to a new variable
---------------------------------------------------------------------------------------
function EquipEval:CopySpellList(SpellList)
	local NewSpellList = { }
	
	for SpellName,SpellTable in pairs(SpellList) do
		NewSpellList[SpellName] = { }
		for index,value in pairs(SpellTable) do
			NewSpellList[SpellName][index] = value
		end
	end
	
	return NewSpellList
end

---------------------------------------------------------------------------------------
-- Returns the stats of an appropriately leveled Windfury Totem weapon enchantment
---------------------------------------------------------------------------------------
function EquipEval:GetWindfuryTotemStats(Level)
	local BonusAP = 0
	local ProcChance = 0
	
	if EquipEval.db.profile["WindfuryTotemUptime"] > 0 then
		ProcChance = EquipEval.db.profile["WindfuryTotemUptime"] * (0.2 * EquipEval.db.profile["BurstFactor"])
		ProcChance = min(1, ProcChance)
		
		if Level >= 70 then
			BonusAP = 445
		elseif Level >= 61 then
			BonusAP = 375
		elseif Level >= 52 then
			BonusAP = 315
		elseif Level >= 42 then
			BonusAP = 229
		else
			BonusAP = 122
		end
		
		BonusAP = BonusAP * (1 + 0.15 * EquipEval.db.profile["ImprovedWindfuryTotemRank"])
		BonusAP = BonusAP * (1 + EquipEval.db.profile["BonusAPPercent"] / 100)
	end
	
	return ProcChance, BonusAP
end


