
local Bonus = AceLibrary("LibItemBonus-2.0")
local UnitBeingScannedIsPlayer = false
local PlayerSetCountTable = nil


local AdditionalPatternsPassive = {
	{ pattern = "Adds (.*) damage per second", effect = "RANGEDDPS" },
	{ pattern = "Improves haste rating by (%d+)%.", effect = "CR_HASTE" },
	{ pattern = "Counterweight %(%+(%d+) Haste Rating%)", effect = "CR_HASTE" },
	{ pattern = "Increases healing done by up to (%d+) and damage done by up to (%d+) for all magical spells and effects%.", effect = {"HEAL", "DMG"} },
	{ pattern = "Use: Permanently attach clefthide armor onto pants to increase stamina by +(%d+) and agility by (%d+)%.", effect = {"STA", "AGI"} },
	{ pattern = "Use: Permanently attach nethercleft leg armor onto pants to increase Stamina by +(%d+) and Agility by (%d+)%.", effect = {"STA", "AGI"} },
	{ pattern = "Use: Permanently attach nethercobra armor onto pants to increase attack power by +(%d+) and critical strike rating by (%d+)%.", effect = {"ATTACKPOWER", "CR_CRIT"} },
	{ pattern = "Use: Permanently attach cobrahide armor onto pants to increase attack power by +(%d+) and critical strike rating by (%d+)%.", effect = {"ATTACKPOWER", "CR_CRIT"} },
	{ pattern = "Use: Permanently embroiders spellthread into pants, increasing healing by up to (%d+), spell damage by up to (%d+),  and Stamina by (%d+)%.", effect = {"HEAL", "DMG", "STA"} },
	{ pattern = "Use: Permanently embroiders spellthread into pants, increasing spell damage and healing by up to (%d+) and Stamina by (%d+)%.", effect = {"HEAL_AND_DMG", "STA"} },
	
	{ pattern = "Use: Increases damage and healing done by magical spells and effects by up to (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_HEAL_AND_DMG", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases spell damage by up to (%d+) and healing by up to (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases spell damage done by up to (%d+) and healing done by up to (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases attack power by (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases your melee and ranged attack power by (%d+)%.  Effect lasts for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases armor by (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_ARMOR_BONUS", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases agility by (%d+) for (%d+) sec%. %((%d+) Mins Cooldown%)", effect = {"TEMP_AGI", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	
	{ pattern = "Use: Increases damage and healing done by magical spells and effects by up to (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_HEAL_AND_DMG", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases spell damage by up to (%d+) and healing by up to (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases spell damage done by up to (%d+) and healing done by up to (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases attack power by (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases your melee and ranged attack power by (%d+)%.  Effect lasts for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases armor by (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_ARMOR_BONUS", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	{ pattern = "Use: Increases agility by (%d+) for (%d+) sec%. %((%d+) Min Cooldown%)", effect = {"TEMP_AGI", "TEMP_DURATION", "TEMP_COOLDOWN"} },
	
	{ pattern = "Use: Increases damage and healing done by magical spells and effects by up to (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_HEAL_AND_DMG", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases spell damage by up to (%d+) and healing by up to (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases spell damage done by up to (%d+) and healing done by up to (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_DMG", "TEMP_HEAL", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases attack power by (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases your melee and ranged attack power by (%d+)%.  Effect lasts for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_ATTACKPOWER", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases armor by (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_ARMOR_BONUS", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	{ pattern = "Use: Increases agility by (%d+) for (%d+) sec%. %((%d+) Min (%d+) Secs Cooldown%)", effect = {"TEMP_AGI", "TEMP_DURATION", "TEMP_COOLDOWN", "TEMP_COOLDOWN_SECONDS"} },
	
	{ pattern = "%+(%d+) Intellect & Chance to restore mana on spellcast", effect = {"INT", "IED_TOKEN"}},
}

local AdditionalPatternsOther = {
	{ pattern = "\"Matches a Red Socket%.\"", effect = "IS_GEM_RED", value = 1 },
	{ pattern = "\"Matches a Blue Socket%.\"", effect = "IS_GEM_BLUE", value = 1 },
	{ pattern = "\"Matches a Yellow Socket%.\"", effect = "IS_GEM_YELLOW", value = 1 },
	{ pattern = "\"Matches a Red or Blue Socket%.\"", effect = {"IS_GEM_RED", "IS_GEM_BLUE"}, value = {1, 1} },
	{ pattern = "\"Matches a Blue or Red Socket%.\"", effect = {"IS_GEM_RED", "IS_GEM_BLUE"}, value = {1, 1} },
	{ pattern = "\"Matches a Yellow or Blue Socket%.\"", effect = {"IS_GEM_YELLOW", "IS_GEM_BLUE"}, value = {1, 1} },
	{ pattern = "\"Matches a Blue or Yellow Socket%.\"", effect = {"IS_GEM_YELLOW", "IS_GEM_BLUE"}, value = {1, 1} },
	{ pattern = "\"Matches a Red or Yellow Socket%.\"", effect = {"IS_GEM_RED", "IS_GEM_YELLOW"}, value = {1, 1} },
	{ pattern = "\"Matches a Yellow or Red Socket%.\"", effect = {"IS_GEM_RED", "IS_GEM_YELLOW"}, value = {1, 1} },
	{ pattern = "\"Only fits in a meta gem slot%.\"", effect = "IS_GEM_META", value = 1 },
	{ pattern = "\"Matches a Red, Yellow or Blue Socket%.\"", effect = {"IS_GEM_RED", "IS_GEM_YELLOW", "IS_GEM_BLUE"}, value = {1, 1, 1} },
	{ pattern = "Chance to Increase Spell Cast Speed", effect = "BONUS_CR_SPELLHASTE", value = 21.3333 },
	--{ pattern = "Mongoose", effect = "MONGOOSE_TOKEN", value = (1.2 * 15 / 60) },
	--{ pattern = "Executioner", effect = "EXECUTIONER_TOKEN", value = (1 * 15 / 60) },
}


---------------------------------------------------------------------------------------
-- Adds EquipEval's additional patterns to the LibItemBonus pattern table
---------------------------------------------------------------------------------------
local PatternsUpdated = false
function EquipEval:AddAdditionalPatterns()
	if not PatternsUpdated then
		for index, value in pairs(AdditionalPatternsPassive) do
			table.insert(Bonus.patterns.PATTERNS_PASSIVE, value)
		end
		for index, value in pairs(AdditionalPatternsOther) do
			table.insert(Bonus.patterns.PATTERNS_OTHER, value)
		end
		
		PatternsUpdated = true
	end
end


---------------------------------------------------------------------------------------
-- Pulls the Item string out of the link string, this is pulled from wowwiki
-- if it had problems, just return the orginal link
-- 
-- link: larget link string with formatting and such
---------------------------------------------------------------------------------------
function EquipEval:ExtractItemString(link)
	local found, _, itemString = string.find(link, "^|%x+|H(.+)|h%[.+%]")
	return itemString
end

---------------------------------------------------------------------------------------
-- Takes a link and will pull out the item string, and strip the enchant from it
-- It will return the new itemString without enchant
-- 
-- link: larget link string with formatting and such
---------------------------------------------------------------------------------------
function EquipEval:ClearEnchantsFromLink(link)
	if not link then return nil end
	local fullItemString = EquipEval:ExtractItemString(link)
	local matchString = "^(item:%d+:)%d+:"
	
	local newString = string.gsub(fullItemString, matchString, "%10:")
	return newString
end

---------------------------------------------------------------------------------------
-- Takes a link and will pull out the item string, and strip the sockets from it
-- It will return the new itemString without sockets
-- 
-- link: larget link string with formatting and such
---------------------------------------------------------------------------------------
function EquipEval:ClearSocketsFromLink(link)
	if not link then return nil end
	local fullItemString = EquipEval:ExtractItemString(link)
	local matchString = "^(item:%d+:%d+:)%d+:%d+:%d+:%d+:"
	
	local newString = string.gsub(fullItemString, matchString, "%10:0:0:0:")
	return newString
end

---------------------------------------------------------------------------------------
-- Used in debugging to print the stats of a moused over object
--
-- infoObj: already parsed info object
-- objType: what type of object is this
---------------------------------------------------------------------------------------
function EquipEval:PrintStats(infoObj, objType)
	if not self:IsDebugging() or not infoObj then return end
	self:Debug(objType.." ---------------------")
	for bonus, value in pairs(infoObj.bonuses) do
		self:Debug(bonus.." : "..value)
	end
end

---------------------------------------------------------------------------------------
-- Used in debugging to print the stats of an info object using the out() function
--
-- infoObj: already parsed info object
-- objType: what type of object is this
---------------------------------------------------------------------------------------
function EquipEval:PrintStatsOut(infoObj, objType)
	if not infoObj then return end
	out(objType.." ---------------------")
	for bonus, value in pairs(infoObj.bonuses) do
		out(bonus.." : "..value)
	end
end




---------------------------------------------------------------------------------------
-- Retrieves bonus info from an info object or only from certain slots
-- of current gear.
---------------------------------------------------------------------------------------
function EquipEval:GetSpecificBonus(BonusName, infoObj, SlotName)
	local totalBonusValue = 0
	local currentBonusValue = 0
	local newBonusValue = 0
	
	if infoObj then
		newBonusValue = newBonusValue + (infoObj.bonuses[BonusName] or 0)
		totalBonusValue = totalBonusValue + newBonusValue
	end
	
	
	if EquipEval.db.profile["ScanCurrentGearBonuses"] and SlotName then
		local info = Bonus:GetBonusDetails(BonusName)
		for slot, value in pairs(info) do
			if slot == SlotName or SlotName == "All" then
				totalBonusValue = totalBonusValue + value
				currentBonusValue = currentBonusValue + value
			end
		end
	end
	
	return totalBonusValue, currentBonusValue, newBonusValue
	
end

---------------------------------------------------------------------------------------
-- Retrieves bonus info from current gear and from an info object
---------------------------------------------------------------------------------------
function EquipEval:GetSpecialBonus(BonusName, infoObj)
	local totalBonusValue, baseBonusValue, additionalBonusValue = EquipEval:GetSpecificBonus(BonusName, infoObj, "All")
	return totalBonusValue, baseBonusValue, additionalBonusValue
end


---------------------------------------------------------------------------------------
-- Retrieves weapon stats from an info object
---------------------------------------------------------------------------------------
function EquipEval:GetWeaponStats(infoObj)
	local minDmg = 0
	local maxDmg = 0
	local speed = 1
	local meleedmg = 0

	if infoObj then
		for bonus, value in pairs(infoObj.bonuses) do
			if (value > 0) then
				if bonus == "WEAPON_MIN" then
					minDmg = value
				elseif bonus == "WEAPON_MAX" then
					maxDmg = value
				elseif bonus == "WEAPON_SPEED" then
					speed = value
				elseif bonus == "MELEEDMG" then
					meleedmg = value
				end
			end
		end
	end
	
	minDmg = minDmg + meleedmg
	maxDmg = maxDmg + meleedmg
	
	local avgDmg = (minDmg + maxDmg) / 2
	local dps = avgDmg / speed
	if dps == 0 then
		speed = 0
	end
	return dps, speed, avgDmg, minDmg, maxDmg
end

---------------------------------------------------------------------------------------
-- Returns the slot that a weapon would go into.
---------------------------------------------------------------------------------------
function EquipEval:GetWeaponSlot(itemEquipLoc)
	if not itemEquipLoc then
		return 0, "None", 0
	end

	local SlotName = "None"
	local Hands = 0
	
	local _, englishClass = UnitClass("player");
	local CurrentLevel = UnitLevel("player")
	local CanDualWield = (englishClass == "HUNTER" and CurrentLevel >= 20) or (englishClass == "ROGUE" and CurrentLevel >= 10) or (englishClass == "WARRIOR" and CurrentLevel >= 20) or (englishClass == "SHAMAN" and EquipEval:TalentRank(2,18) == 1)
	
	local itemEquipLocMH = "Empty"
	local itemEquipLocOH = "Empty"
	local itemLevelMH = 0
	local itemLevelOH = 0
	local mainHandLink = GetInventoryItemLink("player",GetInventorySlotInfo("MainHandSlot"))
	local offHandLink = GetInventoryItemLink("player",GetInventorySlotInfo("SecondaryHandSlot"))
	
	if mainHandLink then
		_, _, _, itemLevelMH, _, _, _, _, itemEquipLocMH = GetItemInfo(mainHandLink)
	end
	if offHandLink then
		_, _, _, itemLevelOH, _, _, _, _, itemEquipLocOH = GetItemInfo(offHandLink)
	end
	
	if string.find(itemEquipLoc, "WEAPON") ~= nil then
		Hands = 1
		if itemEquipLoc == "INVTYPE_2HWEAPON" then
			Hands = 2
			SlotName = "MainHand"
		elseif itemEquipLoc == "INVTYPE_WEAPONMAINHAND" then
			SlotName = "MainHand"
		elseif itemEquipLoc == "INVTYPE_WEAPONOFFHAND" then
			if CanDualWield then
				SlotName = "SecondaryHand"
			end
		elseif itemEquipLoc == "INVTYPE_WEAPON" then
			if itemEquipLocMH == "INVTYPE_2HWEAPON" or itemEquipLocMH == "Empty" then
				SlotName = "MainHand"
			else
				if CanDualWield then
					if itemEquipLocOH == "Empty" then
						SlotName = "SecondaryHand"
					else
						SlotName = "OneHand"
					end
				else
					SlotName = "MainHand"
				end
			end
		end
	elseif itemEquipLoc == "INVTYPE_RANGED" then
		SlotName = "Ranged"
	elseif itemEquipLoc == "INVTYPE_SHIELD" or itemEquipLoc == "INVTYPE_HOLDABLE" then
		SlotName = "SecondaryHand"
		Hands = 0
		
		if itemEquipLocMH == "INVTYPE_2HWEAPON" then
			Hands = 2
		end
	end
	
	
	return SlotName, Hands
end


---------------------------------------------------------------------------------------
-- Returns the slot that an item would go into
---------------------------------------------------------------------------------------
function EquipEval:GetItemSlot(itemEquipLoc)
	local SlotName = ""
	local Hands = 0
	
	SlotName, Hands = EquipEval:GetWeaponSlot(itemEquipLoc)
	
	if itemEquipLoc and SlotName == "None" then
		SlotName = getglobal(itemEquipLoc)
		if not SlotName then
			SlotName = "None"
		elseif SlotName == "Thrown" then
			SlotName = "Ranged"
		end
	end
	
	return SlotName, Hands
end



---------------------------------------------------------------------------------------
-- Returns the item info object of the item(s) that "link" will be replacing
-- also returns any notes about replacement.
---------------------------------------------------------------------------------------
function EquipEval:GetReplacedItemInfos(link, EquipInfo)

	local ReplacementSelectionLine = nil
	local NumReplaced = 0
	local origInfo = nil
	local enchantInfo = nil
	local socketInfo = nil
	local setInfo = nil
	local replacedItemInfo1 = nil
	local replacedItemInfo2 = nil
	local replacedItemLink1, replacedItemLink2 = EquipEval:GetPossibleReplacedItemLinks(link)
	
	if replacedItemLink1 then
		replacedItemInfo1 = EquipEval:MergeItemInfo(replacedItemInfo1, EquipEval:ScanItemLink(replacedItemLink1))
	end
	if replacedItemLink2 then
		replacedItemInfo2 = EquipEval:MergeItemInfo(replacedItemInfo2, EquipEval:ScanItemLink(replacedItemLink2))
	end
	
	local _, _, _, _, _, _, itemSubType, _, itemEquipLoc = GetItemInfo(link)
	local ItemSlotName, WeaponHands = EquipEval:GetItemSlot(itemEquipLoc)
	if WeaponHands == "None" then WeaponHands = 0 end
	
	if replacedItemInfo1 and replacedItemInfo2 then
		if itemEquipLoc == "INVTYPE_2HWEAPON" then
			origInfo = EquipEval:MergeItemInfo(replacedItemInfo1, replacedItemInfo2)
			NumReplaced = 2
		elseif itemEquipLoc == "INVTYPE_WEAPON" then
			if EquipInfo.set ~= replacedItemInfo1.set then
				EquipEval:AddActivatedSetBonusToInfo(replacedItemInfo1, -1)
			else
				EquipEval:AddActivatedSetBonusToInfo(replacedItemInfo1, 1)
			end
			if EquipInfo.set ~= replacedItemInfo2.set then
				EquipEval:AddActivatedSetBonusToInfo(replacedItemInfo2, -1)
			else
				EquipEval:AddActivatedSetBonusToInfo(replacedItemInfo2, 1)
			end
			local _, TempDPSTotal1 = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(EquipInfo, replacedItemInfo1), "MainHand", WeaponHands, itemSubType)
			local _, TempDPSTotal2 = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(EquipInfo, replacedItemInfo2), "SecondaryHand", WeaponHands, itemSubType)
			NumReplaced = 1
			
			if TempDPSTotal1 > TempDPSTotal2 then
				ItemSlotName = "MainHand"
				origInfo = replacedItemInfo1
				ReplacementSelectionLine = "Equipped as Mainhand"
				replacedItemLink2 = nil
			else
				ItemSlotName = "SecondaryHand"
				origInfo = replacedItemInfo2
				ReplacementSelectionLine = "Equipped as Offhand:"
				replacedItemLink1 = nil
			end
		else
			local Item1Rating = 0
			local Item2Rating = 0
			dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(EquipInfo, replacedItemInfo1), ItemSlotName, WeaponHands, itemSubType)
			Item1Rating = EquipEval:CalculateItemRating(dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT)
			dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT = EquipEval:CalculateDPS(EquipEval:GetItemInfoDifference(EquipInfo, replacedItemInfo2), ItemSlotName, WeaponHands, itemSubType)
			Item2Rating = EquipEval:CalculateItemRating(dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT)
			NumReplaced = 1
			
			if Item1Rating > Item2Rating then
				origInfo = replacedItemInfo1
				replacedItemLink2 = nil
			else
				origInfo = replacedItemInfo2
				replacedItemLink1 = nil
			end
		end
	elseif (replacedItemInfo1 or replacedItemInfo2) and not (ItemSlotName == "Finger" or ItemSlotName == "Trinket") then
		origInfo = replacedItemInfo1 or replacedItemInfo2
		NumReplaced = 1
	else
		replacedItemLink1 = nil
		replacedItemLink2 = nil
	end
	
	setInfo = EquipEval:MergeItemInfo(setInfo, origInfo)
	if EquipInfo.set ~= setInfo.set then
		EquipEval:AddActivatedSetBonusToInfo(setInfo, -NumReplaced)
	else
		EquipEval:AddActivatedSetBonusToInfo(setInfo, NumReplaced)
	end
	
	if replacedItemLink1 or replacedItemLink2 then
		if replacedItemLink1 then
			enchantInfo = EquipEval:MergeItemInfo(enchantInfo, EquipEval:ScanItemLink(EquipEval:ClearEnchantsFromLink(replacedItemLink1)))
			socketInfo = EquipEval:MergeItemInfo(socketInfo, EquipEval:ScanItemLink(EquipEval:ClearSocketsFromLink(replacedItemLink1), "No Sockets"))
		end
		if replacedItemLink2 then
			enchantInfo = EquipEval:MergeItemInfo(enchantInfo, EquipEval:ScanItemLink(EquipEval:ClearEnchantsFromLink(replacedItemLink2)))
			socketInfo = EquipEval:MergeItemInfo(socketInfo, EquipEval:ScanItemLink(EquipEval:ClearSocketsFromLink(replacedItemLink2), "No Sockets"))
		end
	else
		enchantInfo = origInfo
		socketInfo = origInfo
	end
	
	return origInfo, enchantInfo, socketInfo, setInfo, ReplacementSelectionLine, ItemSlotName, WeaponHands, itemSubType
end


---------------------------------------------------------------------------------------
-- This takes an info object of an item, finds out if that item is responsible
-- for adding a set bonus.  And then adds the set bonus to that item's info.
---------------------------------------------------------------------------------------
function EquipEval:AddActivatedSetBonusToInfo(info, PiecesAdded)
	PiecesAdded = PiecesAdded or 1
	
	if info then
		local setCount = EquipEval:GetSetCount(info.set)		
		local setBonuses = Bonus:GetSetBonuses(info.set)
		
		--Add set bonus that this item activates
		if setBonuses then
			for piecesRequired, bonuses in pairs(setBonuses) do
				for bonusName, value in pairs(bonuses) do
					if (PiecesAdded > 0 and setCount < piecesRequired and setCount + PiecesAdded >= piecesRequired)
					or (PiecesAdded < 0 and setCount >= piecesRequired and setCount + PiecesAdded < piecesRequired) then
						EquipEval:AddBonusToInfo(info, bonusName, value)
					end
				end
			end
		end
	end

	return
end

---------------------------------------------------------------------------------------
-- Returns the item link(s) of the item(s) that "link" may be replacing
---------------------------------------------------------------------------------------
function EquipEval:GetPossibleReplacedItemLinks(link)
	local replacedLink = nil
	local replacedLink2 = nil

	if link and EquipEval.db.profile["CompareToCurrentItems"] then
		local _, _, _, _, _, _, _, _, itemEquipLoc = GetItemInfo(link) 
		local SlotName, Hands = EquipEval:GetItemSlot(itemEquipLoc)
		
		if SlotName == "Trinket" or SlotName == "Finger" then
			replacedLink = GetInventoryItemLink("player",GetInventorySlotInfo(SlotName.."0Slot"))
			replacedLink2 = GetInventoryItemLink("player",GetInventorySlotInfo(SlotName.."1Slot"))
		elseif Hands == 2 or SlotName == "OneHand" then
			replacedLink = GetInventoryItemLink("player",GetInventorySlotInfo("MainHandSlot"))
			replacedLink2 = GetInventoryItemLink("player",GetInventorySlotInfo("SecondaryHandSlot"))
		elseif SlotName == "Head" or SlotName == "Neck" or SlotName == "Shoulder" or SlotName == "Back" or SlotName == "Chest" or SlotName == "Wrist" or SlotName == "Hands" or SlotName == "Waist" or SlotName == "Legs" or SlotName == "Feet" or SlotName == "MainHand" or SlotName == "SecondaryHand" or SlotName == "Ranged" or SlotName == "Ammo" then
			replacedLink = GetInventoryItemLink("player",GetInventorySlotInfo(SlotName.."Slot"))
			local _, _, _, _, _, _, _, _, itemEquipLoc = GetItemInfo(link) 
		end
	end

	return replacedLink, replacedLink2
end

---------------------------------------------------------------------------------------
-- Subtracts two info objects
---------------------------------------------------------------------------------------
function EquipEval:GetItemInfoDifference(info1, info2)
	return EquipEval:MergeItemInfo(info1, info2, 1, -1)
end


---------------------------------------------------------------------------------------
-- Merges Info Objects
--
-- The multipliers are optional
---------------------------------------------------------------------------------------
function EquipEval:MergeItemInfo(info1, info2, multiplier1, multiplier2)
	local infoObj = { bonuses = {}, set = nil, inactive_gem_bonus = {}}
	
	if info1 then
		if info1.bonuses then
			for bonus, value in pairs(info1.bonuses) do
				EquipEval:AddBonusToInfo(infoObj, bonus, value, multiplier1)
			end
		end
		if info1.inactive_gem_bonus then
			for bonus, value in pairs(info1.inactive_gem_bonus) do
				EquipEval:AddBonusToInfo(infoObj, bonus, value, multiplier1, "inactive_gem_bonus")
			end
		end
		
		infoObj.set = info1.set
	end
	
	if info2 then
		if info2.bonuses then
			for bonus, value in pairs(info2.bonuses) do
				EquipEval:AddBonusToInfo(infoObj, bonus, value, multiplier2)
			end
		end
		if info2.inactive_gem_bonus then
			for bonus, value in pairs(info2.inactive_gem_bonus) do
				EquipEval:AddBonusToInfo(infoObj, bonus, value, multiplier2, "inactive_gem_bonus")
			end
		end
		
		
		if infoObj.set and infoObj.set ~= info2.set then
			infoObj.set = nil
		else
			infoObj.set = info2.set
		end
	end
	
	EquipEval:CorrectWeaponDamage(infoObj)
	return infoObj
end

---------------------------------------------------------------------------------------
-- Rolls MELEEDMG into WEAPON_MIN and WEAPON_MAX
---------------------------------------------------------------------------------------
function EquipEval:CorrectWeaponDamage(info)
	
	if info then	
		if info.bonuses["MELEEDMG"] and info.bonuses["WEAPON_MIN"] and info.bonuses["WEAPON_MAX"] then
			info.bonuses["WEAPON_MIN"] = info.bonuses["WEAPON_MIN"] + info.bonuses["MELEEDMG"]
			info.bonuses["WEAPON_MAX"] = info.bonuses["WEAPON_MAX"] + info.bonuses["MELEEDMG"]
			info.bonuses["MELEEDMG"] = 0
		end
	end
	
	return
end


---------------------------------------------------------------------------------------
-- Adds an item bonus to an info object
---------------------------------------------------------------------------------------
function EquipEval:AddBonusToInfo(info, bonus, value, multiplier, BonusTableName)
	BonusTableName = BonusTableName or "bonuses"
	multiplier = multiplier or 1
	value = value or 0
	
	if info and bonus and info[BonusTableName] and value ~= 0 then	
		if bonus == "WEAPON_MAX" or bonus == "WEAPON_MIN" or bonus == "WEAPON_SPEED" then
			info[BonusTableName][bonus] = info[BonusTableName][bonus] or value
		else			
			info[BonusTableName][bonus] = (info[BonusTableName][bonus] or 0) + value * multiplier
		end
	end
	
	return
end

---------------------------------------------------------------------------------------
-- Subtracts an item bonus from an info object
---------------------------------------------------------------------------------------
function EquipEval:SubtractBonusFromInfo(info, bonus, value, multiplier, BonustTableName)
	EquipEval:AddBonusToInfo(info, bonus, value, ((multiplier or 1) * -1), BonustTableName)
	
	return
end



---------------------------------------------------------------------------------------
-- Used to calculate Stats given an info object that came out of ItemBonusLib
--
-- infoObj: already parsed info object
---------------------------------------------------------------------------------------
function EquipEval:CalculateStats(infoObj)
	local str = 0
	local agi = 0
	local ap = 0
	local rap = 0
	local fap = 0
	local crit = 0
	local hit = 0
	local meleedmg = 0
	local critbonus = 0
	local spelldmg = 0
	local holydmg = 0
	local int = 0
	local haste = 0
	local weapSkill = 0
	local ignorearmor = 0
	local expertise = 0

	local TotalValue = 0
	local UnusedValue = 0

	-- go through each bonus, if we find it in our 
	-- profile with a value then multiply it
	for bonus, value in pairs(infoObj.bonuses) do
		if (value ~= 0) then
			if string.sub(bonus, 1, 6) == "BONUS_" then
				bonus = string.sub(bonus, 7)
			end
			
			if bonus == "STR" then
				str = str + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "AGI" then
				agi = agi + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "ATTACKPOWER" then
				ap = ap + value
				rap = rap + value
			elseif bonus == "RANGEDATTACKPOWER" then
				rap = rap + value
			elseif bonus == "ATTACKPOWERFERAL" then
				fap = fap + value
			elseif bonus == "CR_CRIT" then
				crit = crit + value
			elseif bonus == "CR_HIT" then
				hit = hit + value
			elseif bonus == "MELEEDMG" then
				meleedmg = meleedmg + value
			elseif bonus == "CRITDMG" then
				critbonus = critbonus + value
			elseif bonus == "DMG" then
				spelldmg = spelldmg + value
			elseif bonus == "HOLYDMG" then
				holydmg = holydmg + value
			elseif bonus == "INT" then
				int = int + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "CR_HASTE" then
				haste = haste + value
			elseif bonus == "IGNOREARMOR" then
				ignorearmor = ignorearmor + value
			elseif bonus == "CR_EXPERTISE" then
				expertise = expertise + value
			elseif bonus == "CR_WEAPON" then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_DAGGER" and IsEquippedItemType("Daggers") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_SWORD" and IsEquippedItemType("One-Handed Swords") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_SWORD_2H" and IsEquippedItemType("Two-Handed Swords") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_AXE" and IsEquippedItemType("One-Handed Axes") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_AXE_2H" and IsEquippedItemType("Two-Handed Axes") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_MACE" and IsEquippedItemType("One-Handed Maces") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_MACE_2H" and IsEquippedItemType("Two-Handed Maces") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_BOW" and IsEquippedItemType("Bows") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_XBOW" and IsEquippedItemType("Crossbows") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_GUN" and IsEquippedItemType("Guns") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_UNARMED" and IsEquippedItemType("Fist Weapons") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_STAFF" and IsEquippedItemType("Staves") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_POLEARM" and IsEquippedItemType("Polearms") then
				weapSkill = weapSkill + value
			elseif bonus == "CR_WEAPON_FERAL" then
				local _, englishClass = UnitClass("player");
				if englishClass == "DRUID" and UnitPowerType("player") ~= 0 then
					weapSkill = weapSkill + value
				end
			else
				UnusedValue = UnusedValue + abs(value)
			end
			
			TotalValue = TotalValue + abs(value)
		end
	end

	TotalValue = TotalValue - UnusedValue

	return TotalValue, str, agi, ap, rap, fap, crit, hit, meleedmg, spelldmg, holydmg, int, haste, critbonus, weapSkill, ignorearmor, expertise
end

---------------------------------------------------------------------------------------
-- Used to calculate Stats given an info object that came out of ItemBonusLib
--
-- infoObj: already parsed info object
---------------------------------------------------------------------------------------
function EquipEval:CalculateMitigationStats(infoObj)
	local agi = 0
	local dodge = 0
	local parry = 0
	local block = 0
	local armor = 0
	local defense = 0
	local resilience = 0
	local addedArmor = 0
	local stam = 0
	local health = 0
	local int = 0

	local TotalValue = 0
	local UnusedValue = 0

	-- go through each bonus, if we find it in our 
	-- profile with a value then multiply it
	for bonus, value in pairs(infoObj.bonuses) do
		if (value ~= 0) then
			if string.sub(bonus, 1, 6) == "BONUS_" then
				bonus = string.sub(bonus, 7)
			end
			
			if bonus == "AGI" then
				agi = agi + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "CR_DODGE" then
				dodge = dodge + value
			elseif bonus == "CR_PARRY" then
				parry = parry + value
			elseif bonus == "CR_BLOCK" then
				block = block + value
			elseif bonus == "ARMOR" then
				if armor == 0 then
					armor = armor + value
				else
					addedArmor = addedArmor + value
				end
			elseif bonus == "BASE_ARMOR" then
				armor = armor + value
			elseif bonus == "ARMOR_BONUS" then
				addedArmor = addedArmor + value
			elseif bonus == "CR_DEFENSE" then
				defense = defense + value
			elseif bonus == "CR_RESILIENCE" then
				resilience = resilience + value
			elseif bonus == "STA" then
				stam = stam + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "INT" then
				int = int + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "HEALTH" then
				health = health + value
			else
				UnusedValue = UnusedValue + abs(value)
			end
			
			TotalValue = TotalValue + abs(value)
		end
	end

	TotalValue = TotalValue - UnusedValue

	return 	TotalValue, agi, dodge, parry, block, armor, defense, resilience, addedArmor, stam, health, int
end

---------------------------------------------------------------------------------------
-- Used to calculate Stats given an info object that came out of ItemBonusLib
--
-- infoObj: already parsed info object
---------------------------------------------------------------------------------------
function EquipEval:CalculateCasterStats(infoObj)
	local int = 0
	local spi = 0
	local mana = 0
	local mp5 = 0
	local crit = 0
	local holycrit = 0
	local naturecrit = 0
	local heal = 0
	local spelldmg = 0
	local haste = 0
	local hit = 0
	local frostdmg = 0
	local shadowdmg = 0
	local firedmg = 0
	local str = 0
	local agi = 0
	local stam = 0
	local arcanedmg = 0
	local spellpen = 0
	local naturedmg = 0
	local holydmg = 0
	local critbonus = 0

	local TotalValue = 0
	local UnusedValue = 0

	-- go through each bonus, if we find it in our 
	-- profile with a value then multiply it
	for bonus, value in pairs(infoObj.bonuses) do
		if (value ~= 0) then
			if string.sub(bonus, 1, 6) == "BONUS_" then
				bonus = string.sub(bonus, 7)
			end
		
			if bonus == "INT" then
				int = int + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "SPI" then
				spi = spi + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "STR" then
				str = str + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "AGI" then
				agi = agi + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "STA" then
				stam = stam + value * (1 + EquipEval.db.profile["BonusStatPercent"] / 100)
			elseif bonus == "MANA" then
				mana = mana + value
			elseif bonus == "MANAREG" then
				mp5 = mp5 + value
			elseif bonus == "CR_SPELLCRIT" then
				crit = crit + value
			elseif bonus == "HOLYCRIT" then
				holycrit = holycrit + value
			elseif bonus == "NATURECRIT" then
				naturecrit = naturecrit + value
			elseif bonus == "HEAL" then
				heal = heal + value
			elseif bonus == "DMG" then
				spelldmg = spelldmg + value
			elseif bonus == "CR_SPELLHASTE" then
				haste = haste + value
			elseif bonus == "CR_SPELLHIT" then
				hit = hit + value
			elseif bonus == "FROSTDMG" then
				frostdmg = frostdmg + value
			elseif bonus == "SHADOWDMG" then
				shadowdmg = shadowdmg + value
			elseif bonus == "FIREDMG" then
				firedmg = firedmg + value
			elseif bonus == "ARCANEDMG" then
				arcanedmg = arcanedmg + value
			elseif bonus == "NATUREDMG" then
				naturedmg = naturedmg + value
			elseif bonus == "HOLYDMG" then
				holydmg = holydmg + value
			elseif bonus == "SPELLPEN" then
				spellpen = spellpen + value
			elseif bonus == "SPELLCRITDMG" then
				critbonus = critbonus + value
			else
				UnusedValue = UnusedValue + abs(value)
			end
			
			TotalValue = TotalValue + abs(value)
		end
	end

	TotalValue = TotalValue - UnusedValue

	return TotalValue, int, spi, mana, mp5, crit, holycrit, naturecrit, heal, spelldmg, haste, hit, frostdmg, shadowdmg, firedmg, str, stam, arcanedmg, spellpen, naturedmg, holydmg, critbonus, agi

end




---------------------------------------------------------------------------------------
-- Takes a set name and returns the number of items of that set you have equipped
---------------------------------------------------------------------------------------
function EquipEval:GetSetCount(set)
	local setCount = 0
	
	if set then
		if PlayerSetCountTable then
			setCount = (PlayerSetCountTable[set] or 0)
		elseif EquipEval.db.profile["ScanCurrentGearBonuses"] then
			local eq = Bonus:GetUnitEquipment("player")
			local _, set_count_table = Bonus:BuildBonusSet(eq)
			
			setCount = (set_count_table[set] or 0)
		end
	end


	return (setCount or 0)
end

---------------------------------------------------------------------------------------
-- A function that is hooked onto the ItemBonusLib method "GetUnitEquipment"
-- that sets a boolean value to true if the unit being scanned is the player
-- and false otherwise.
---------------------------------------------------------------------------------------
function EquipEval:IBLGetUnitEquipmentHook(unit)
	UnitBeingScannedIsPlayer = unit == "player"
	return EquipEval.hooks[Bonus].GetUnitEquipment(Bonus, unit)
end

---------------------------------------------------------------------------------------
-- A function that is hooked onto the ItemBonusLib method "BuildBonusSet"
-- that intercepts the returned table of set_count and stores it locally
-- if it is the player's set_count
---------------------------------------------------------------------------------------
function EquipEval:IBLBuildBonusSetHook(eq)
	EquipEval:AddAdditionalPatterns()

	local details, set_count = EquipEval.hooks[Bonus].BuildBonusSet(Bonus, eq)
	if UnitBeingScannedIsPlayer then
		PlayerSetCountTable = set_count
	end
	
	return details, set_count
end


---------------------------------------------------------------------------------------
-- Takes a link, passes it to the Bonus library to be scanned
-- and does some post processing.
---------------------------------------------------------------------------------------
function EquipEval:ScanItemLink(link, NoSockets)
	local InfoObj = Bonus:ScanItemLink(link)
	if not InfoObj then return nil end
	
	--Handle "Use:" effects such as trinkets
	if EquipEval.db.profile["CalculateOnUseEffects"] and (InfoObj.bonuses["TEMP_COOLDOWN"] or InfoObj.bonuses["TEMP_COOLDOWN_SECONDS"]) then
		local CoolDown = (InfoObj.bonuses["TEMP_COOLDOWN"] or 0) + (InfoObj.bonuses["TEMP_COOLDOWN_SECONDS"] or 0) / 60
		local Duration = InfoObj.bonuses["TEMP_DURATION"]
		
		InfoObj.bonuses["TEMP_COOLDOWN"] = nil
		InfoObj.bonuses["TEMP_COOLDOWN_SECONDS"] = nil
		InfoObj.bonuses["TEMP_DURATION"] = nil
		
		for bonus, value in pairs(InfoObj.bonuses) do
			if string.sub(bonus, 1, 5) == "TEMP_" then
				local BonusName = string.sub(bonus, 6)
				BonusName = "BONUS_"..BonusName

				InfoObj.bonuses[BonusName] = (InfoObj.bonuses[BonusName] or 0) + max(0, (value * Duration / (60 * CoolDown)))
				InfoObj.bonuses[bonus] = nil
			end
		end
		
	end
	
	--Handle combined "HEAL_AND_DMG"
	if InfoObj.bonuses["HEAL_AND_DMG"] then
		InfoObj.bonuses["HEAL"] = (InfoObj.bonuses["HEAL"] or 0) + InfoObj.bonuses["HEAL_AND_DMG"]
		InfoObj.bonuses["DMG"] = (InfoObj.bonuses["DMG"] or 0) + InfoObj.bonuses["HEAL_AND_DMG"]
		InfoObj.bonuses["HEAL_AND_DMG"] = nil
	end
	if InfoObj.bonuses["BONUS_HEAL_AND_DMG"] then
		InfoObj.bonuses["BONUS_HEAL"] = (InfoObj.bonuses["BONUS_HEAL"] or 0) + InfoObj.bonuses["BONUS_HEAL_AND_DMG"]
		InfoObj.bonuses["BONUS_DMG"] = (InfoObj.bonuses["BONUS_DMG"] or 0) + InfoObj.bonuses["BONUS_HEAL_AND_DMG"]
		InfoObj.bonuses["BONUS_HEAL_AND_DMG"] = nil
	end
	
	--Handle Insightful Earthstorm Diamond Token
	if InfoObj.bonuses["IED_TOKEN"] then
		InfoObj.bonuses["BONUS_MANAREG"] = (InfoObj.bonuses["BONUS_MANAREG"] or 0) + InfoObj.bonuses["IED_TOKEN"] /12 * 25
		InfoObj.bonuses["IED_TOKEN"] = nil
	end
	
	--Handle gem assumption
	if not NoSockets and EquipEval.db.profile["MakeGemAssumptions"] then
		local AssumptionsMade = false
		local ColorsMatch = true
		local NewInfo = { bonuses = {}, set = nil}
		NewInfo = EquipEval:MergeItemInfo(NewInfo, InfoObj)
		
		local GemColors = { "Red", "Yellow", "Blue", "Meta" }
		for ColorNumber = 1, #GemColors do
			local ColorAssumptionsMade = false
			local ColorMatches = false
			
			NewInfo, ColorAssumptionsMade, ColorMatches = EquipEval:MakeGemAssumptions(NewInfo, GemColors[ColorNumber])
			
			AssumptionsMade = (AssumptionsMade or ColorAssumptionsMade)
			ColorsMatch = (ColorsMatch and ColorMatches)
		end
		
		if ColorsMatch and NewInfo.inactive_gem_bonus then
			for bonus, value in pairs(NewInfo.inactive_gem_bonus) do
				EquipEval:AddBonusToInfo(NewInfo, bonus, value)
				NewInfo.inactive_gem_bonus[bonus] = nil
			end
		end
		
		if AssumptionsMade then
			return NewInfo
		end
	end
	
	return InfoObj
end


---------------------------------------------------------------------------------------
-- Updates an item info with the additional bonuses from assumed gems
---------------------------------------------------------------------------------------
function EquipEval:MakeGemAssumptions(ItemInfo, SocketColor)
	local AssumptionsMade = false
	local ColorMatch = true
	local SocketName = "EMPTY_SOCKET_"..strupper(SocketColor)
	local AssumedLink = EquipEval.db.profile[("Assumed"..SocketColor.."GemLink")] or "None"
	
	if ItemInfo.bonuses[SocketName] and ItemInfo.bonuses[SocketName] ~= 0 then
		ColorMatch = false
		if EquipEval:ExtractItemString(AssumedLink) then
			AssumptionsMade = true
			
			local GemInfo = EquipEval:ScanItemLink(AssumedLink, "No Sockets")
			ItemInfo = EquipEval:MergeItemInfo(ItemInfo, GemInfo, 1, ItemInfo.bonuses[SocketName])
			ItemInfo.bonuses[SocketName] = nil
			
			ColorMatch = EquipEval:CheckColorMatch(GemInfo, SocketColor)
		end
	end
	
	return ItemInfo, AssumptionsMade, ColorMatch
end


---------------------------------------------------------------------------------------
-- Updates an item info with the additional bonuses from assumed gems
---------------------------------------------------------------------------------------
function EquipEval:CheckColorMatch(GemInfo, SocketColor)
	local ColorMatched = false
	SocketColor = "IS_GEM_"..strupper(SocketColor)
	
	if GemInfo and GemInfo.bonuses and GemInfo.bonuses[SocketColor] and GemInfo.bonuses[SocketColor] > 0 then
		ColorMatched = true
	end
	
	return ColorMatched
end


---------------------------------------------------------------------------------------
-- Takes item effects and returns a total item rating
---------------------------------------------------------------------------------------
function EquipEval:CalculateItemRating(dpsOriginal, dpsOriginalT, mitOriginal, mitOriginalT, healOriginal, healOriginalT)
	local ItemRating = EquipEval.db.profile['ItemRatingWeightDPS'] * dpsOriginal / max(0.001, dpsOriginalT - dpsOriginal) + EquipEval.db.profile['ItemRatingWeightMit'] * mitOriginal / max(0.001, mitOriginalT - mitOriginal) + EquipEval.db.profile['ItemRatingWeightHeal'] * healOriginal / max(0.001, healOriginalT - healOriginal)
	return ItemRating
end


