
local Bonus = AceLibrary("LibItemBonus-2.0")



local eqevBaseManaRegenTable = {
0.034965, 0.034191, 0.033465, 0.032526, 0.031661, 0.031076, 0.030523, 0.029994, 0.029307, 0.028661, 
0.027584, 0.026215, 0.025381, 0.0243, 0.023345, 0.022748, 0.021958, 0.021386, 0.02079, 0.020121, 
0.019733, 0.019155, 0.018819, 0.018316, 0.017936, 0.017576, 0.017201, 0.016919, 0.016581, 0.016233, 
0.015994, 0.015707, 0.015464, 0.015204, 0.014956, 0.014744, 0.014495, 0.014302, 0.014094, 0.013895, 
0.013724, 0.013522, 0.013363, 0.013175, 0.012996, 0.012853, 0.012687, 0.012539, 0.012384, 0.012233, 
0.012113, 0.011973, 0.011859, 0.011714, 0.011575, 0.011473, 0.011342, 0.011245, 0.01111, 0.010999, 
0.0107, 0.010522, 0.01029, 0.010119, 0.009968, 0.009808, 0.009651, 0.009553, 0.009445, 0.009327 }



---------------------------------------------------------------------------------------
-- Calculates the Level Rating
---------------------------------------------------------------------------------------
function EquipEval:GetLevelRating(currentLevel)
	currentLevel = max(currentLevel, 10)

	local LevelRating = (-3/82 * currentLevel + 131/41)
	if currentLevel < 60 and currentLevel ~= 8 then
		LevelRating = 52 / (currentLevel - 8)
	end

	return LevelRating
end

---------------------------------------------------------------------------------------
-- Calculates the Level Rating for Defensive Stats
---------------------------------------------------------------------------------------
function EquipEval:GetLevelRatingDefensiveStats(currentLevel)
	currentLevel = max(currentLevel, 34)

	local LevelRating = (-3/82 * currentLevel + 131/41)
	if currentLevel < 60 and currentLevel ~= 8 then
		LevelRating = 52 / (currentLevel - 8)
	end

	return LevelRating
end

---------------------------------------------------------------------------------------
-- Returns the number of points in a specified talent
---------------------------------------------------------------------------------------
function EquipEval:TalentRank(tab, number)
	local _, _, _, _, currRank, _ = GetTalentInfo(tab, number);
	
	return currRank
end


---------------------------------------------------------------------------------------
-- Calculates the difference between enemy skill and character skill
---------------------------------------------------------------------------------------
function EquipEval:GetSkillDifference(skillRating, currentLevel)
	currentLevel = currentLevel or UnitLevel("player")
	currentLevel = max(1, currentLevel)
	skillRating = skillRating or currentLevel * 5

	return (5 * (currentLevel + self.db.profile["EnemyLevelDifference"]) - skillRating)
end

---------------------------------------------------------------------------------------
-- Calculates the Miss chance due to the difference between enemy skill
-- and character skill
---------------------------------------------------------------------------------------
function EquipEval:GetMissFromSkillDifference(skillDifference)
	skillDifference = skillDifference or EquipEval:GetSkillDifference()

	if skillDifference > 10 then
		return ((skillDifference - 10) * 0.4 + 2) / 100
	end

	if skillDifference >= 0 then
		return (skillDifference * 0.1 / 100)
	end

	return (skillDifference * 0.04 / 100)

end

---------------------------------------------------------------------------------------
-- Calculates the Dodge chance due to the difference between enemy skill
-- and character skill
---------------------------------------------------------------------------------------
function EquipEval:GetDodgeFromSkillDifference(skillDifference)
	skillDifference = skillDifference or EquipEval:GetSkillDifference()

	if skillDifference >= 0 then
		return (skillDifference * 0.1 / 100)
	end

	return (skillDifference * 0.04 / 100)

end

---------------------------------------------------------------------------------------
-- Calculates the Parry chance due to the difference between enemy skill
-- and character skill
---------------------------------------------------------------------------------------
function EquipEval:GetParryFromSkillDifference(skillDifference)
	skillDifference = skillDifference or EquipEval:GetSkillDifference()

	if skillDifference >= 0 then
		return (skillDifference * 0.3 / 100)
	end

	return (skillDifference * 0.04 / 100)

end

---------------------------------------------------------------------------------------
-- Calculates the chance and severity of glancing blows
--
-- This uses formulas taken from http://www.wowwiki.com/Glancing_blow (10/31/07)
---------------------------------------------------------------------------------------
function EquipEval:GetGlancingBlowInfo()
	local SkillDif = EquipEval.db.profile["EnemyLevelDifference"] * 5
	local GlanceModifierHigh = min(0.99, max(0.2, (1.2 - 0.03 * SkillDif)))
	local GlanceModifierLow = min(0.91, max(0.01, (1.3 - 0.05 * SkillDif)))
	local GlanceSeverity = (GlanceModifierLow + GlanceModifierHigh) / 2

	local GlanceChance = min(1, max(0, ((10 + SkillDif) / 100)))
	GlanceChance = GlanceChance / EquipEval.db.profile["BurstFactor"]

	return GlanceChance, GlanceSeverity

end



---------------------------------------------------------------------------------------
-- Returns the bonus to critical strike damage gained through gear
---------------------------------------------------------------------------------------
function EquipEval:GetBonusCritDamage()
	if not EquipEval.db.profile["ScanCurrentGearBonuses"] then
		return 0
	end

	local totalCritDmgBonus = 0

	local info = Bonus:GetBonusDetails("CRITDMG")
	for slot, value in pairs(info) do
		totalCritDmgBonus = totalCritDmgBonus + value
	end

	return (totalCritDmgBonus / 100)

end

---------------------------------------------------------------------------------------
-- Returns the bonus to spell critical strike damage gained through gear
---------------------------------------------------------------------------------------
function EquipEval:GetBonusSpellCritDamage()
	if not EquipEval.db.profile["ScanCurrentGearBonuses"] then
		return 0
	end

	local totalCritDmgBonus = 0

	local info = Bonus:GetBonusDetails("SPELLCRITDMG")
	for slot, value in pairs(info) do
		totalCritDmgBonus = totalCritDmgBonus + value
	end

	return (totalCritDmgBonus / 100)

end


---------------------------------------------------------------------------------------
-- Calculates the Modifer to Damage due to Armor
---------------------------------------------------------------------------------------
function EquipEval:GetArmorModifer(ignoreArmor)

	local reductionPercent = 0
	local level = UnitLevel("player") + EquipEval.db.profile["EnemyLevelDifference"]
	local armor = EquipEval:GetArmorFromDamageReduction()

	ignoreArmor = ignoreArmor or 0

	armor = max(0, (armor - ignoreArmor))
	reductionPercent = EquipEval:GetDamageReductionFromArmor(armor, level)

	return reductionPercent
end

---------------------------------------------------------------------------------------
-- Calculates the Reduction in Damage Taken due to Armor
---------------------------------------------------------------------------------------
function EquipEval:GetDamageReductionFromArmor(armor, level)
	local reductionPercent = 0
	if level >= 60 then
		reductionPercent = armor / (armor + 400 + 85 * (level + 4.5 * (level - 59)))
	else
		reductionPercent = armor / (armor + 400 + 85 * level)
	end
	return min(reductionPercent, 0.75)
end

---------------------------------------------------------------------------------------
-- Calculates the amount of Armor from damage reduction percent
---------------------------------------------------------------------------------------
function EquipEval:GetArmorFromDamageReduction(reductionPercent, level)
	local armor = 0

	reductionPercent = reductionPercent or EquipEval.db.profile["EnemyArmorPercentage"]/100
	level = level or UnitLevel("player") + EquipEval.db.profile["EnemyLevelDifference"]

	if reductionPercent >= 1 then
		reductionPercent = 0.999999
	end

	if level >= 60 then
		armor = (-467.5 * reductionPercent * (level - 47.417112299465)) / (reductionPercent - 1)
	else
		armor = -5 * reductionPercent * (17 * level + 80) / (reductionPercent - 1)
	end

	return armor
end

---------------------------------------------------------------------------------------
-- Returns the current armor penetration gained through gear
---------------------------------------------------------------------------------------
function EquipEval:GetArmorPenetration()
	if not EquipEval.db.profile["ScanCurrentGearBonuses"] then
		return 0
	end

	local totalPenetration = 0

	local info = Bonus:GetBonusDetails("IGNOREARMOR")
	for slot, value in pairs(info) do
		totalPenetration = totalPenetration + value
	end

	return totalPenetration
end


---------------------------------------------------------------------------------------
-- Returns the current amount of expertise the player has
---------------------------------------------------------------------------------------
function EquipEval:GetExpertise()
	return ((GetExpertise() + EquipEval.db.profile["BonusExpertise"]) * 0.0025)
end



---------------------------------------------------------------------------------------
-- Returns the Chance to Hit a PvE Opponent with Spells
---------------------------------------------------------------------------------------
function EquipEval:GetSpellHitChance(LevelDifference)
	LevelDifference = LevelDifference or EquipEval.db.profile["EnemyLevelDifference"]

	local HitChance = 0.96 - (LevelDifference * 0.01)
	if LevelDifference >= 3 then
		HitChance = HitChance - (LevelDifference - 2) * 0.10
	end

	return max(0.01, min(0.99, HitChance))
end

---------------------------------------------------------------------------------------
-- Returns the damage modifier due to spell resistance and level difference
---------------------------------------------------------------------------------------
function EquipEval:GetResistanceModifier(SpellPenetration, CurrentLevel, LevelDifference, EnemyResistance)
	SpellPenetration = SpellPenetration or GetSpellPenetration()
	CurrentLevel = CurrentLevel or UnitLevel("player")
	LevelDifference = LevelDifference or EquipEval.db.profile["EnemyLevelDifference"]
	EnemyResistance = EnemyResistance or EquipEval.db.profile["EnemyResistance"]


	EnemyResistance = EnemyResistance + max(0, (5 * LevelDifference)) - min(SpellPenetration, EnemyResistance)

	local ResistanceModifier = 1 - min(0.75, ((EnemyResistance / (CurrentLevel * 5)) * 0.75))
	ResistanceModifier = max(0.01, (1 - (1 - ResistanceModifier) / EquipEval.db.profile["BurstFactor"]))
	ResistanceModifier = min(1, ResistanceModifier)

	return ResistanceModifier
end


---------------------------------------------------------------------------------------
-- Returns the MP5 equivalent from potion use
---------------------------------------------------------------------------------------
function EquipEval:GetMP5FromPotionUse(Level)
	if EquipEval.db.profile["ManaPotionUse"] == 0 then
		return 0
	end

	Level = Level or UnitLevel("player")

	local mp5 = 0

	if Level >= 55 then
		mp5 = (1800 + 3000) / 2
	elseif Level >= 49 then
		mp5 = (1350 + 2250) / 2
	elseif Level >= 41 then
		mp5 = (900 + 1500) / 2
	elseif Level >= 31 then
		mp5 = (700 + 900) / 2
	elseif Level >= 22 then
		mp5 = (455 + 585) / 2
	elseif Level >= 14 then
		mp5 = (280 + 360) / 2
	elseif Level >= 5 then
		mp5 = (140 + 180) / 2
	end

	mp5 = mp5 * 5 / 120

	return mp5 * EquipEval.db.profile["ManaPotionUse"]
end


---------------------------------------------------------------------------------------
-- Returns the maximum possible chance to hit an opponent.
---------------------------------------------------------------------------------------
function EquipEval:MaxHitChance(expertiseEffect, skillDifference, isHunter)
	local maxHitChance = EquipEval:GetHitStats(expertiseEffect, skillDifference, isHunter)
	return maxHitChance
end



---------------------------------------------------------------------------------------
-- Returns the maximum possible chance to hit an opponent and the effect
-- of dodge and parry.
---------------------------------------------------------------------------------------
function EquipEval:GetHitStats(expertiseEffect, skillDifference, isHunter)
	local maxHitChance = 1
	local parryChance = 0
	local dodgeChance = 0
	
	isHunter = isHunter or 0

	if isHunter == 0 then
		if skillDifference == nil then
			skillDifference = EquipEval:GetSkillDifference()
		end

		if self.db.profile["EnemyParryPercentage"] >= 0 then
			parryChance = self.db.profile["EnemyParryPercentage"]/100 + EquipEval:GetParryFromSkillDifference(skillDifference) - expertiseEffect
			parryChance = max(0, parryChance)
		end
		if self.db.profile["EnemyDodgePercentage"] >= 0 then
			dodgeChance = self.db.profile["EnemyDodgePercentage"]/100 + EquipEval:GetDodgeFromSkillDifference(skillDifference) - expertiseEffect
			dodgeChance = max(0, dodgeChance)
		end
	end

	maxHitChance = maxHitChance - (parryChance + dodgeChance)
	maxHitChance = min(1, max(0, maxHitChance))

	return maxHitChance, parryChance, dodgeChance
end



---------------------------------------------------------------------------------------
-- Find the total Mitigation given mitigation stats
---------------------------------------------------------------------------------------
function EquipEval:FindMitigation(hitChance, dodgeChance, parryChance, blockChance, critChance, critBonus, crushChance, armorPercent, stanceModifier, dmgBase)
	if EquipEval.db.profile["BurstFactor"] <= 0 then
		EquipEval.db.profile["BurstFactor"] = 1
	end
	
	--Handle enemy armor penetration
	local EnemyLevel = UnitLevel("player") + self.db.profile["EnemyLevelDifference"]
	local armorValue = EquipEval:GetArmorFromDamageReduction(armorPercent, EnemyLevel)
	armorPercent = EquipEval:GetDamageReductionFromArmor(max(armorValue - self.db.profile["EnemyArmorPenetration"], 0), EnemyLevel)

	local damageFraction = EquipEval.db.profile["EnemyFractionWhiteDamage"]
	local critChanceYellow = 0
	local crushChanceYellow = 0
	local avoidance = 0

	--To Do:  Review effect of blocks on hit tables

	hitChance = max(0.01, (1 - (1 - hitChance) / EquipEval.db.profile["BurstFactor"]))
	avoidance = avoidance + (1 - hitChance)

	dodgeChance = dodgeChance / EquipEval.db.profile["BurstFactor"]
	dodgeChance = max(0, min(dodgeChance, (1 - avoidance)))
	avoidance = avoidance + dodgeChance

	parryChance = parryChance / EquipEval.db.profile["BurstFactor"]
	parryChance = max(0, min(parryChance, (1 - avoidance)))
	avoidance = avoidance + parryChance

	blockChance = blockChance / EquipEval.db.profile["BurstFactor"]
	blockChance = max(0, min(blockChance, (1 - avoidance)))

	critChance = critChance * EquipEval.db.profile["BurstFactor"]
	crushChance = crushChance * EquipEval.db.profile["BurstFactor"]

	critChance = max(0, min(critChance, (1 - (avoidance + blockChance))))
	crushChance = max(0, min(crushChance, (1 - (avoidance + blockChance + critChance))))
	critChanceYellow = max(0, min(critChance, (1 - (dodgeChance + parryChance + blockChance))))
	crushChanceYellow = max(0, min(crushChance, (1 - (dodgeChance + parryChance + blockChance + critChance))))
	critChance = (damageFraction) * critChance + (1 - damageFraction) * critChanceYellow
	crushChance = (damageFraction) * crushChance + (1 - damageFraction) * crushChanceYellow

	
	local totalMit = (1 - ((hitChance - (dodgeChance + parryChance)) + (critChance * critBonus) + (crushChance * 0.5)) * (1 - armorPercent) * stanceModifier / dmgBase)
	totalMit = min(totalMit, 0.9999999)
		
	return totalMit
end




---------------------------------------------------------------------------------------
-- Returns the MP5 produced outside of the 5 second rule by level, spirit, and intellect
---------------------------------------------------------------------------------------
function EquipEval:GetBaseMP5FromStats(Spirit, Intellect, Level, BonusSpirit, BonusIntellect)
	local stat = 0

	Level = Level or UnitLevel("player")
	BonusSpirit = BonusSpirit or 0
	BonusIntellect = BonusIntellect or 0
	
	_, stat = UnitStat("player", 5)
	Spirit = (Spirit or stat) + BonusSpirit
	
	_, stat = UnitStat("player", 4)
	Intellect = (Intellect or stat) + BonusIntellect

	return 5 * (0.001 + sqrt(Intellect) * Spirit * eqevBaseManaRegenTable[Level])
end


---------------------------------------------------------------------------------------
-- Returns the difference in mp5 between two levels of spirit and intellect
---------------------------------------------------------------------------------------
function EquipEval:GetDeltaMP5FromStats(NewSpirit, NewIntellect, OldSpirit, OldIntellect, Level)
	Level = Level or UnitLevel("player")
	
	if not (OldSpirit and OldIntellect) then
		local stat = 0
		
		_, stat = UnitStat("player", 5)
		OldSpirit = stat
		NewSpirit = NewSpirit + OldSpirit
		
		_, stat = UnitStat("player", 4)
		OldIntellect = stat
		NewIntellect = NewIntellect + OldIntellect
	end
	
	local OldMP5 = EquipEval:GetBaseMP5FromStats(OldSpirit, OldIntellect, Level)
	local NewMP5 = EquipEval:GetBaseMP5FromStats(NewSpirit, NewIntellect, Level)

	return (NewMP5 - OldMP5)
end


---------------------------------------------------------------------------------------
-- Returns the length of the global cooldown, given spell haste
---------------------------------------------------------------------------------------
function EquipEval:GetSpellGCD(SpellHaste)
	SpellHaste = SpellHaste or (GetCombatRatingBonus(CR_HASTE_SPELL) / 100)
	return max(1, (1.5 / (1 + SpellHaste)))
end



---------------------------------------------------------------------------------------
-- Returns the current intellect of the player
---------------------------------------------------------------------------------------
function EquipEval:GetTotalIntellect()
	local base, stat, posBuff, negBuff = UnitStat("player",4)
	return stat
end
