--[[---------------------------------------------------------------------------------
 Detox
 written by Maia/Proudmoore, Rabbit/Magtheridon EU and others.
 Code based on the Decursive 1.9.7 core code by Archarodim and Quu.
 The name 'Detox' was provided by Tem.
------------------------------------------------------------------------------------]]

local vmajor,vminor = "0.4", tonumber(string.sub("$Revision: 63576 $", 12, -3))

--[[---------------------------------------------------------------------------------
 locals
------------------------------------------------------------------------------------]]

local L                 = AceLibrary("AceLocale-2.2"):new("Detox")
local BS                = LibStub("LibBabble-Spell-3.0"):GetLookupTable()
local BC                = LibStub("LibBabble-Class-3.0"):GetLookupTable()
local tablet            = AceLibrary("Tablet-2.0")
local aura              = nil -- AceLibrary("SpecialEvents-Aura-2.0")
local RL                = nil -- AceLibrary("Roster-2.1")

local soundPlayed       = false
local hasSpells         = false
local spells            = {} -- will be filled with available spells
local priorityIndex     = {} -- this table references name to index of the priority table
local priority          = {} -- this integer keyed table will hold subtables containing name, unitid and priority for each unit in our roster

local DebuffTypeColor      = {
	none    = { r = 0.80, g = 0, b = 0 },
	Magic   = { r = 0.20, g = 0.60, b = 1.00 },
	Curse   = { r = 0.60, g = 0.00, b = 1.00 },
	Disease = { r = 0.60, g = 0.40, b = 0 },
	Poison  = { r = 0.00, g = 0.60, b = 0 },
}

local invisible = {
	[BS["Prowl"]]                 = true,
	[BS["Stealth"]]               = true,
	[BS["Shadowmeld"]]            = true,
}

local ignore = {
	[BS["Banish"]]                = true,
	[BS["Phase Shift"]]           = true,
	[L["Frost Trap Aura"]]        = true,
}

local cleaningSpells = {
	[BS["Cure Disease"]]          = true,
	[BS["Abolish Disease"]]       = true,
	[BS["Purify"]]                = true,
	[BS["Cleanse"]]               = true,
	[BS["Dispel Magic"]]          = true,
	[BS["Cure Poison"]]           = true,
	[BS["Abolish Poison"]]        = true,
	[BS["Remove Lesser Curse"]]   = true,
	[BS["Remove Curse"]]          = true,
	[BS["Purge"]]                 = true,
	[BS["Devour Magic"]]          = true,
	[BS["Stoneform"]]             = true,
}

BINDING_HEADER_DETOX              = "Detox"
BINDING_NAME_PRIORITYTOGGLE       = L["Toggle target priority"]
BINDING_NAME_PRIORITYTOGGLEGROUP  = L["Toggle target group priority"]
BINDING_NAME_PRIORITYTOGGLECLASS  = L["Toggle target class priority"]

--[[---------------------------------------------------------------------------------
 defaults and AceOptions table
------------------------------------------------------------------------------------]]

local defaults = {
	showPriorities = true,
	ignoreStealthed = true,
	checkAbolish = true,
	ignorePets = false,
	liveDisplay = 5,
	filter = true,
	range = true,
	sound = true,
	priorityNames = {},
	priorityGroups = {},
	priorityClasses = {},
	debuffsInSkipList = {
		BS["Silence"],
		L["Ancient Hysteria"],
		L["Ignite Mana"],
		L["Tainted Mind"],
		L["Magma Shackles"],
		L["Cripple"],
		L["Dust Cloud"],
		L["Widow's Embrace"],
		L["Curse of Tongues"],
		L["Sonic Burst"],
		L["Thunderclap"],
		L["Delusions of Jin'do"]
	},
	skipByClass = {
		WARRIOR = {
			[L["Ancient Hysteria"]]   = true,
			[L["Ignite Mana"]]        = true,
			[L["Tainted Mind"]]       = true,
			[L["Widow's Embrace"]]    = true,
			[L["Curse of Tongues"]]   = true,
			[L["Delusions of Jin'do"]]= true,
		},
		ROGUE = {
			[BS["Silence"]]           = true,
			[L["Ancient Hysteria"]]   = true,
			[L["Ignite Mana"]]        = true,
			[L["Tainted Mind"]]       = true,
			[L["Widow's Embrace"]]    = true,
			[L["Curse of Tongues"]]   = true,
			[L["Sonic Burst"]]        = true,
			[L["Delusions of Jin'do"]]= true,
		},
		HUNTER = {
			[L["Magma Shackles"]]     = true,
			[L["Delusions of Jin'do"]]= true,
		},
		MAGE = {
			[L["Magma Shackles"]]     = true,
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Thunderclap"]]        = true,
			[L["Delusions of Jin'do"]]= true,
		},
		WARLOCK = {
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Thunderclap"]]        = true,
			[L["Delusions of Jin'do"]]= true,
		},
		DRUID = {
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Thunderclap"]]        = true,
			[L["Delusions of Jin'do"]]= true,
		},
		PALADIN = {
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Delusions of Jin'do"]]= true,
		},
		PRIEST = {
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Thunderclap"]]        = true,
			[L["Delusions of Jin'do"]]= true,
		},
		SHAMAN = {
			[L["Cripple"]]            = true,
			[L["Dust Cloud"]]         = true,
			[L["Delusions of Jin'do"]]= true,
		}
	}
}

local function get(key)
	return Detox.db.profile[key]
end

local function set(key, value)
	Detox.db.profile[key] = value
end

local options = {
	type = "group",
	handler = Detox,
	args = {
		livelistoptions = {
			type = "group",
			name = L["Live list"],
			desc = L["Options for the live list."],
			order = 110,
			args = {
				show = {
					type = "toggle",
					name = L["Show live list"],
					desc = L["Detaches the live list from the Detox icon."],
					get = "IsTooltipDetached",
					set = "ToggleTooltipDetached",
					order = 100,
				},
				sound = {
					type = "toggle",
					name = L["Play sound"],
					desc = L["Play sound if unit needs decursing"],
					get = get,
					set = set,
					order = 101,
					passValue = "sound",
				},
				showpriorizations = {
					type = "toggle",
					name = L["Show priorities"],
					desc = L["Displays who is prioritized in the live list."],
					get = get,
					set = set,
					order = 102,
					passValue = "showPriorities",
				},
				live = {
					type = 'range',
					name = L["Max debuffs shown"],
					desc = L["Defines the max number of debuffs to display in the live list."],
					get = get,
					set = set,
					passValue = "liveDisplay",
					min = 0,
					max = 20,
					step = 1,
					isPercent = false,
					order = 103,
				},
			},
		},
		priority = {
			type = "group",
			name = L["Priority"],
			desc = L["These units will be priorized when curing."],
			order = 120,
			args = {}
		},
		filter = {
			type = "group",
			name = L["Filter"],
			desc = L["Options for filtering various debuffs and conditions."],
			order = 130,
			args = {
				filter = {
					type = "toggle",
					name = L["Filter by type"],
					desc = L["Only show debuffs you can cure."],
					get = get,
					set = set,
					passValue = "filter",
					order = 103,
				},
				range = {
					type = "toggle",
					name = L["Filter by range"],
					desc = L["Only show units in range."],
					get = get,
					set = set,
					passValue = "range",
					order = 104,
				},
				stealth = {
					type = "toggle",
					name = L["Filter stealthed units"],
					desc = L["It is recommended not to cure stealthed units."],
					get = get,
					set = set,
					passValue = "ignoreStealthed",
					order = 105,
				},
				abolish = {
					type = "toggle",
					name = L["Filter Abolished units"],
					desc = L["Skip units that have an active Abolish buff."],
					get = get,
					set = set,
					passValue = "checkAbolish",
					order = 106,
				},
				pets = {
					type = "toggle",
					name = L["Filter pets"],
					desc = L["Pets are also your friends."],
					get = function() return Detox.db.profile.ignorePets end,
					set = function()
						Detox.db.profile.ignorePets = not Detox.db.profile.ignorePets
						Detox:UpdatePriority()
					end,
					order = 107
				},
				debuff = {
					type = "group",
					name = L["Debuff"],
					desc = L["Filter by debuff and class."],
					order = 400,
					args = {}
				},
			},
		},
	},
}

--[[---------------------------------------------------------------------------------
 Initialization
------------------------------------------------------------------------------------]]

Detox = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceDB-2.0", "AceConsole-2.0", "FuBarPlugin-2.0")
local Detox = Detox

-- stuff for FuBar:
Detox.hasIcon = true
Detox.defaultPosition = "LEFT"
Detox.defaultMinimapPosition = 250
Detox.cannotDetachTooltip = false
Detox.tooltipHiddenWhenEmpty = true
Detox.hideWithoutStandby = true
Detox.clickableTooltip = true
Detox.independentProfile = true

function Detox:OnInitialize()
	self:RegisterDB("DetoxDB")
	self:RegisterDefaults('profile', defaults)
	self:RegisterChatCommand('/detox', options, "DETOX")
	self.OnMenuRequest = options
end


function Detox:OnEnable()
	if not aura then aura = AceLibrary("SpecialEvents-Aura-2.0") end
	if not RL then RL = AceLibrary("Roster-2.1") end
	self:RegisterEvent("SPELLS_CHANGED", "RescanSpellbook", 1)
	self:RegisterEvent("LEARNED_SPELL_IN_TAB", "RescanSpellbook", 1)
	self:RegisterEvent("RosterLib_UnitChanged")
	self:RegisterEvent("SpecialEvents_UnitDebuffLost", "UpdateDisplay")
	self:RegisterEvent("SpecialEvents_UnitDebuffGained", "UpdateDisplay")
	self:ParseSpellbook()
	self:PopulateSkipList()
end

--[[---------------------------------------------------------------------------------
 Events
------------------------------------------------------------------------------------]]

function Detox:RosterLib_UnitChanged(unitid, name, class, subgroup, rank, oldName, oldUnitid, oldClass, oldSubgroup, oldRank)
	if name == nil then -- Someone left the raid
		for i, v in ipairs(priority) do
			if not RL.roster[priority[i].name] then
				priorityIndex[priority[i].name] = nil
				priority[i] = nil
				table.remove(priority, i)
			end
		end
	end
	self:UpdatePriority()
end

--[[---------------------------------------------------------------------------------
 Spellbook
------------------------------------------------------------------------------------]]


function Detox:ParseSpellbook()
	spells = {}
	hasSpells = false
	local BookType    = BOOKTYPE_SPELL
	local break_flag  = false
	local i = 1
	local maxRankMagic      = 0
	local maxRankEnemyMagic = 0
	local _, playerClass = UnitClass("player")
	while not break_flag do
		while (true) do --  HUH?
			local name, rank = GetSpellName(i, BookType)
			if not name then
				if BookType == BOOKTYPE_PET then
					break_flag = true
					break
				end
				BookType = BOOKTYPE_PET
				i = 1
				break
			end
			if cleaningSpells[name] then
				hasSpells = true
				spells["cooldown"] = {i, BookType}
				if name == BS["Cure Disease"] or name == BS["Purify"] then
					spells["disease1"] = {i, BookType, name}
				end
				if name == BS["Abolish Disease"] or ( playerClass == "PALADIN" and name == BS["Cleanse"] ) then
					spells["disease2"] = {i, BookType, name}
				end
				if name == BS["Cure Poison"] or name == BS["Purify"] then
					spells["poison1"] = {i, BookType, name}
				end
				if name == BS["Abolish Poison"] or ( playerClass == "PALADIN" and name == BS["Cleanse"] ) then
					spells["poison2"] = {i, BookType, name}
				end
				if name == BS["Remove Curse"] or name == BS["Remove Lesser Curse"] then
					spells["curse"] = {i, BookType, name}
				end
				if ( playerClass == "PALADIN" and name == BS["Cleanse"] ) or name == BS["Dispel Magic"] or name == BS["Devour Magic"] then
					if rank and rank:find(L["Rank (%d+)"]) then
						local ranknum = rank:gsub(L["Rank (%d+)"], "%1")
						if tonumber(ranknum) > maxRankMagic then
							spells["magic"] = {i, BookType, name}
							maxRankMagic = tonumber(ranknum)
						end
					else
						spells["magic"] = {i, BookType, name}
					end
				end
				if name == BS["Dispel Magic"] or ( playerClass == "SHAMAN" and name == BS["Purge"] ) or name == BS["Devour Magic"] then
					if rank and rank:find(L["Rank (%d+)"]) then
						local ranknum = rank:gsub(L["Rank (%d+)"], "%1")
						if tonumber(ranknum) > maxRankEnemyMagic then
							spells["enemymagic"] = {i, BookType, name}
							maxRankEnemyMagic = tonumber(ranknum)
						end
					else
						spells["enemymagic"] = {i, BookType, name}
					end
				end
			end
			i = i + 1
		end
	end
end

function Detox:RescanSpellbook()
	if not hasSpells then return end
	local valid = true
	for spell in pairs(spells) do
		if GetSpellName(spells[spell][1], spells[spell][2]) ~= spells[spell][3] then 
			valid = false
		end
	end
	if not valid then
		self:ParseSpellbook()
	end
end

--[[---------------------------------------------------------------------------------
 Cleaning
------------------------------------------------------------------------------------]]

function Detox:UnitCurable(unit)
	if not UnitExists(unit) then return false end
	if not UnitIsVisible(unit) then return false end
	if self.db.profile.range and not CheckInteractDistance(unit, 4) then return false end
	-- check if unit is stealthed
	if self.db.profile.ignoreStealthed then
		for buff in pairs(invisible) do
			if aura:UnitHasBuff(unit, buff) then return false end
		end
	end
	-- check if we need to ignore unit
	for debuff in pairs(ignore) do
		if aura:UnitHasDebuff(unit, debuff) then return false end
	end
	-- check for debuffs we want to skip for specific classes when in combat
	if UnitAffectingCombat("player") then
		local _, class = UnitClass(unit)
		if self.db.profile.skipByClass[class] then
			for debuff in pairs(self.db.profile.skipByClass[class]) do
				if self.db.profile.skipByClass[class][debuff] then
					if aura:UnitHasDebuff(unit, debuff) then return false end
				end
			end
		end
	end
	if self.db.profile.checkAbolish and (aura:UnitHasBuff(unit, BS["Abolish Poison"]) or aura:UnitHasBuff(unit, BS["Abolish Disease"])) then return false end
	return true
end

--[[---------------------------------------------------------------------------------
 Debuff skipping
------------------------------------------------------------------------------------]]


function Detox:AddSkippedDebuff(debuffName)
	for k, name in pairs(self.db.profile.debuffsInSkipList) do
		if name == debuffName then return end
	end
	table.insert(self.db.profile.debuffsInSkipList, debuffName)
	self:PopulateSkipList()
end

-- XXX Please note that we don't remove the class ignores for this debuff.
-- XXX Should we?
function Detox:RemoveSkippedDebuff(debuffName)
	for k, name in pairs(self.db.profile.debuffsInSkipList) do
		if name == debuffName then
			table.remove(self.db.profile.debuffsInSkipList, k)
		end
	end
	self:PopulateSkipList()
end


function Detox:PopulateSkipList()
	local classes = { "Warrior", "Priest", "Druid", "Shaman", "Paladin", "Mage", "Warlock", "Hunter", "Rogue" }

	if options.args.filter.args.debuff.args.add then
		options.args.filter.args.debuff.args.add = nil
		for k, debuff in pairs(self.db.profile.debuffsInSkipList) do
			options.args.filter.args.debuff.args.remove[debuff] = nil
		end
		options.args.filter.args.debuff.args.remove = nil
	end

	-- |class| is also the spacer some time during the loop
	for class in pairs(options.args.filter.args.debuff.args) do
		for debuff in pairs(options.args.filter.args.debuff.args[class]) do
			-- If we just added a debuff, it will not be in the menu yet, so
			-- check in case we try to reclaim it.
			if options.args.filter.args.debuff.args[class][debuff] then
				options.args.filter.args.debuff.args[class][debuff] = nil
			end
		end
		options.args.filter.args.debuff.args[class] = nil
	end

	-- create add option in dewdrop
	options.args.filter.args.debuff.args.add = {
		type = 'text',
		name = L["Add"],
		desc = L["Adds a new debuff to the class submenus."],
		get = false,
		set = "AddSkippedDebuff",
		usage = L["<debuff name>"],
		order = 101,
	}

	-- create remove option in dewdrop
	options.args.filter.args.debuff.args.remove = {
		type = "group",
		name = L["Remove"],
		desc = L["Removes a debuff from the class submenus."],
		args = {},
		order = 102,
		pass = true,
		func = "RemoveSkippedDebuff",
	}

	-- create list of debuffs in subfolder
	for k, debuff in pairs(self.db.profile.debuffsInSkipList) do
		options.args.filter.args.debuff.args.remove.args[debuff] = {
			type = 'execute',
			name = debuff,
			desc = L["Remove %s from the class submenus."]:format(debuff),
		}
	end
	
	-- create spacer
	options.args.filter.args.debuff.args.spacer = { type = "header", order = 103 }

	-- create list of debuffs
	for k, debuff in pairs(self.db.profile.debuffsInSkipList) do
		options.args.filter.args.debuff.args[debuff] = {
			type = "group",
			name = debuff,
			desc = L["Classes to filter for: %s."]:format(debuff),
			args = {},
			order = 200,
			pass = true,
			get = function(key)
				return self.db.profile.skipByClass[key:upper()][debuff]
			end,
			set = function(key, value)
				self.db.profile.skipByClass[key:upper()][debuff] = value
			end,
		}
		-- create submenu for classes
		for k, class in pairs(classes) do
			options.args.filter.args.debuff.args[debuff].args[class] = {
				type = "toggle",
				name = BC[class],
				desc = L["Toggle filtering %s on %s."]:format(debuff, BC[class]),
			}
		end
	end
end


--[[---------------------------------------------------------------------------------
 Priority
------------------------------------------------------------------------------------]]

function Detox:PriorityPrint(target, priority)
	if priority then
		self:Print(L["%s was added to the priority list."]:format(target))
	else
		self:Print(L["%s has been removed from the priority list."]:format(target))
	end
end

function Detox:PriorityToggle(targetType)
	if not UnitExists("target") then
		self:Print(L["Can't add/remove current target to priority list, it doesn't exist."])
		return
	end
	local target = UnitName("target")
	if not targetType then
		if not RL.roster[target] then
			self:Print(L["Can't add/remove current target to priority list, it's not in your raid."])
			return
		end
		self.db.profile.priorityNames[target] = not self.db.profile.priorityNames[target]
		self:PriorityPrint(target, self.db.profile.priorityNames[target])
	else
		local targetGroup = nil
		local targetClass = nil

		for n, u in pairs(RL.roster) do
			if u and u.name and u.class ~= "PET" then
				if not targetGroup and u.name == target then
					targetGroup = u.subgroup
					targetClass = u.class
					break
				end
			end
		end

		if targetType == "class" and targetClass then
			self.db.profile.priorityClasses[targetClass] = not self.db.profile.priorityClasses[targetClass]
			self:PriorityPrint(L["Class %s"]:format(targetClass), self.db.profile.priorityClasses[targetClass])
		elseif targetType == "group" and targetGroup then
			self.db.profile.priorityGroups[targetGroup] = not self.db.profile.priorityGroups[targetGroup]
			self:PriorityPrint(L["Group %s"]:format(targetGroup), self.db.profile.priorityGroups[targetGroup])
		else
			self:Print(L["Can't add/remove current target to priority list, it's not in your raid."])
			return
		end
	end
	self:UpdatePriority()
end

function Detox:UpdatePriority()
	self:CreateOptionsTable()
	-- update and sort all units. this task has a few steps:
	local datasubtable, index
	for name, u in pairs(RL.roster) do
		-- see if that unit is in our priorityIndex table
		index = priorityIndex[name]
		-- either use the existing subtable or create a new one
		datasubtable = index and priority[index] or {}
		datasubtable.pri = self:GetPriority(name)
		datasubtable.unitid = u.unitid
		if not index then
			-- our subtable still needs a name
			datasubtable.name = name
			-- add that subtable to the priority table
			table.insert(priority, datasubtable)
			priorityIndex[name] = #priority  -- maybe not needed, but for safety reasons it can't hurt.
		end
	end
	-- now re-sort the priority table
	table.sort(priority, function(a, b) return a.pri > b.pri end)
	-- save the current index for all names
	for k,v in ipairs(priority) do
		priorityIndex[v.name] = k
	end

	self:UpdateDisplay()
end


function Detox:CreateOptionsTable()
	-- we need to update the dewdrop menu, so lets clear the old one and inject new stuff:
	for class in pairs(options.args.priority.args) do
		for unit in pairs(options.args.priority.args[class].args) do
			options.args.priority.args[class].args[unit] = nil
		end
		options.args.priority.args[class] = nil
	end

	if not options.args.priority.args.group then
		options.args.priority.args.group = {
			type = "group",
			name = L["Groups"],
			desc = L["Prioritize by group."],
			order = 100,
			args = {},
			pass = true,
			get = function(key)
				return Detox.db.profile.priorityGroups[key]
			end,
			set = function(key, value)
				self.db.profile.priorityGroups[key] = value
				self:UpdatePriority()
			end,
		}
		for i = 1, 8 do
			options.args.priority.args.group.args[i] = {
				type = "toggle",
				name = L["Group %s"]:format(i),
				desc = L["Prioritize group %s."]:format(i),
				order = 100 + i,
			}
		end
	end

	-- create subgroups
	for name, unit in pairs(RL.roster) do
		if unit and unit.class ~= "PET" then
			if not options.args.priority.args[unit.class] then
				local c = UnitClass(unit.unitid)
				options.args.priority.args[unit.class] = {
					type = "group",
					name = c,
					desc = c,
					order = 101,
					args = {},
					pass = true,
					get = function(key)
						return self.db.profile.priorityClasses[key]
					end,
					set = function(key, value)
						self.db.profile.priorityClasses[key] = value
						self:UpdatePriority()
					end,
				}

				-- Create "all <class>" item
				if not options.args.priority.args[unit.class].args[unit.class] then
					options.args.priority.args[unit.class].args[unit.class] = {
						type = "toggle",
						name = L["Every %s"]:format(c),
						desc = L["Prioritize every %s."]:format(c),
						order = 100,
					}
				end
			end
			options.args.priority.args[unit.class].args[name] = {
				type = "toggle",
				name = name,
				desc = L["Prioritize %s."]:format(name),
				order = 101,
			}
		end
	end
end


function Detox:GetPriority(name)
	local playergroup = (RL.roster[UnitName("player")] and RL.roster[UnitName("player")].subgroup) or 1
	local pri
	local grp = RL.roster[name].subgroup
	local cls = RL.roster[name].class
	-- priority range 1000 - 0.
	if     name == UnitName("player") then  pri = 800
	elseif playergroup == grp         then  pri = 700
	elseif playergroup < grp          then  pri = 700 - (grp - playergroup)*10
	else                                    pri = 700 - (grp + 8 - playergroup)*10
	end
	if cls == "PET" then
		if self.db.profile.ignorePets then pri = 0 else pri = pri - 200 end
	end
	if self.db.profile.priorityNames[name] then pri = pri + 200 end  -- unit in individual priority list
	if self.db.profile.priorityClasses[cls] then pri = pri + 200 end  -- unit in prioritized class
	if self.db.profile.priorityGroups[grp] then pri = pri + 200 end  -- unit in prioritized group
	return pri
end


--[[---------------------------------------------------------------------------------
 Live display and other FuBar stuff
------------------------------------------------------------------------------------]]


function Detox:OnTooltipUpdate()
	local cat = nil
	local lines = 0
	for k,v in ipairs(priority) do
		if v.pri == 0 or lines > self.db.profile.liveDisplay then break end
		if UnitIsVisible(v.unitid) and self:UnitCurable(v.unitid) then
			for debuffname, count, type, texture in aura:DebuffIter(v.unitid) do
				if ( not self.db.profile.filter and ( type == "Magic" or type == "Poison" or type == "Disease" or type == "Curse" ) )
				or ( type == "Magic" and spells["magic"] )
				or ( type == "Poison" and ( spells["poison1"] or spells["poison2"] ) )
				or ( type == "Disease" and ( spells["disease1"] or spells["disease2"] ) )
				or ( type == "Curse" and spells["curse"] )
				then
					local r,g,b = self:GetRaidColors(v.unitid)
					if not count or count == 0 then count = 1 end
					lines = lines + 1
					if not cat then
						cat = tablet:AddCategory(
							"columns", 2,
							"showWithoutChildren", false,
							"hideBlankLine", false,
							"child_hasCheck", true,
							"child_checked", true,
							"child_justify", "LEFT",
							"child_justify2", "LEFT"
						)
					end
					cat:AddLine(
						"text", string.format("%s (%dx)", debuffname, count),
						"textR", DebuffTypeColor[type].r,
						"textG", DebuffTypeColor[type].g,
						"textB", DebuffTypeColor[type].b,
						"text2", UnitName(v.unitid),
						"text2R", r,
						"text2G", g,
						"text2B", b,
						"checkIcon", texture
					)
					-- update fubar display (icon + text) to display the first
					-- debuff we find.
					if lines == 1 then
						self:SetIcon(texture)
						self:SetText(string.format("|cff%02x%02x%02x%s|r", DebuffTypeColor[type].r*255, DebuffTypeColor[type].g*255, DebuffTypeColor[type].b*255, UnitName(v.unitid)))
					end
				end
			end
		end
	end
	-- play sound
	if self.db.profile.sound then
		if lines == 0 then
			soundPlayed = false
		elseif not soundPlayed then
			PlaySoundFile("Sound\\interface\\AuctionWindowOpen.wav")
			soundPlayed = true
		end
	end
	-- Reset FuBar text and display
	if lines == 0 then
		self:SetIcon(true)
		self:SetText("Detox")
	end
	-- Show the prioritized units
	if self.db.profile.showPriorities then
		local priCat = nil
		local text = nil
		for k,v in ipairs(priority) do
			if v.pri > 700 then
				if not priCat then priCat = tablet:AddCategory("columns", 1, "text", L["Priorities"]) end
				local name = v.name
				if not text then
					text = name
				else
					text = text..", "..name
				end
			end
		end
		if priCat then priCat:AddLine("text", text..".", "wrap", true) end
	end
end

function Detox:GetRaidColors(unit)
	local _,class = UnitClass(unit)
	if RAID_CLASS_COLORS[class] then
		return RAID_CLASS_COLORS[class].r, RAID_CLASS_COLORS[class].g, RAID_CLASS_COLORS[class].b
	else
		return 0.5, 0.5, 0.5
	end
end

