--[[
 Copyright 2008 Geir H. Pettersen
 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
]]

--[[
Name: DemocraticReadyCheck
Revision: $Revision: 74442 $
Author: Geir H. Pettersen
Email: ghenningp [at] gmail [dot] com
Description: Make your whole party/raid aware of whats happening during and after a readycheck.
]]

----------------------------------
-- Libraries
----------------------------------
local Waterfall = AceLibrary:HasInstance("Waterfall-1.0") and AceLibrary("Waterfall-1.0")
DemocraticReadyCheck = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0")

----------------------------------
-- Config options
----------------------------------
local options = {
    handler = DemocraticReadyCheck,
    type='group',
    args = {
        win = {
			type = 'execute',
			name = 'Options Window',
			desc = 'Shows the Options Window',
			func = "OpenWaterfall",
			guiHidden = true,
		},
        messages = {
            type = 'group',
            name = 'Message config',
            desc = 'Configuration of ReadyCheck messages',
            args = {
                initiate = {
                    type = 'text',
                    name = "ReadyCheck initiate message type",
                    desc = "Set how ReadyCheck initiate message should be sent",
                    get = "GetInitiateType",
                    set = "SetInitiateType",
                    validate = {
                        ["DISABLED"] = "Disabled",
                        ["CHAT"] = "Chat",
                        ["WARN"] = "Warning"
                    },
                    order = 10,
                },
                confirm = {
                    type = 'text',
                    name = "ReadyCheck confirm message type",
                    desc = "Set how ReadyCheck confirm (per character) message should be sent",
                    get = "GetConfirmType",
                    set = "SetConfirmType",
                    validate = {
                        ["DISABLED"] = "Disabled",
                        ["CHAT"] = "Chat",
                        ["WARN"] = "Warning"
                    },
                    order = 20,
                },
                complete = {
                    type = 'text',
                    name = "ReadyCheck complete message type",
                    desc = "Set how ReadyCheck complete message should be sent",
                    get = "GetCompleteType",
                    set = "SetCompleteType",
                    validate = {
                        ["DISABLED"] = "Disabled",
                        ["CHAT"] = "Chat",
                        ["WARN"] = "Warning"
                    },
                    order = 30,
                },
                finalstatus = {
                    type = 'text',
                    name = "ReadyCheck final status message type",
                    desc = "Set how ReadyCheck final status message should be sent",
                    get = "GetFinalStatusType",
                    set = "SetFinalStatusType",
                    validate = {
                        ["DISABLED"] = "Disabled",
                        ["CHAT"] = "Chat",
                        ["WARN"] = "Warning"
                    },
                    order = 40,
                },
                texts = {
                    type = 'group',
                    name = 'Message text config',
                    desc = 'Configuration of ReadyCheck message texts',
                    order = 50,
                    args = {
                        header = {
                            type = 'header',
                            name = "%s is replaced by character name(s)",
                            order = 1,
                        },
                        initiate = {
                            type = 'text',
                            name = "initiate",
                            desc = "Set the readycheck initiate text",
                            usage = "<Your message here>",
                            get = "GetInitiateStr",
                            set = "SetInitiateStr",
                            order = 10,
                        },
                        confirmYes = {
                            type = 'text',
                            name = "confirmYes",
                            desc = "Set the readycheck confirmYes text",
                            usage = "<Your message here>",
                            get = "GetConfirmYesStr",
                            set = "SetConfirmYesStr",
                            order = 20,
                        },
                        confirmNo = {
                            type = 'text',
                            name = "confirmNo",
                            desc = "Set the readycheck confirmNo text",
                            usage = "<Your message here>",
                            get = "GetConfirmNoStr",
                            set = "SetConfirmNoStr",
                            order = 30,
                        },
                        complete = {
                            type = 'text',
                            name = "complete",
                            desc = "Set the readycheck complete text",
                            usage = "<Your message here>",
                            get = "GetCompleteStr",
                            set = "SetCompleteStr",
                            order = 40,
                        },
                        finalStatusNo = {
                            type = 'text',
                            name = "finalStatusNo",
                            desc = "Set the readycheck finalStatusNo text",
                            usage = "<Your message here>",
                            get = "GetFinalStatusNoStr",
                            set = "SetFinalStatusNoStr",
                            order = 50,
                        },
                        finalStatusAfk = {
                            type = 'text',
                            name = "finalStatusAfk",
                            desc = "Set the readycheck finalStatusAfk text",
                            usage = "<Your message here>",
                            get = "GetFinalStatusAfkStr",
                            set = "SetFinalStatusAfkStr",
                            order = 60,
                        },
                        finalStatusOk = {
                            type = 'text',
                            name = "finalStatusOk",
                            desc = "Set the readycheck finalStatusOk text",
                            usage = "<Your message here>",
                            get = "GetFinalStatusOkStr",
                            set = "SetFinalStatusOkStr",
                            order = 70,
                        },
                    },
                },
            },
        },
    },
}

----------------------------------
-- Initialization
----------------------------------
DemocraticReadyCheck:RegisterChatCommand("/drc","/democraticreadycheck", options)
DemocraticReadyCheck:RegisterDB("DemocraticReadyCheckDB", "DemocraticReadyCheckDBPC")
DemocraticReadyCheck:RegisterDefaults('profile', {
    initiateType = "DISABLED",
    confirmType = "DISABLED",
    completeType = "CHAT",
    finalStatusType = "WARN",
    initiateStr = "%s has initiated a ready check.",
    confirmYesStr = "%s is ready.",
    confirmNoStr = "%s is NOT ready.",
    completeStr = "Ready check completed.",
    finalStatusNoStr = "%s is NOT ready.",
    finalStatusAfkStr = "%s is AFK.",
    finalStatusOkStr = "Everyone is ready!",
})

----------------------------------
-- DemocraticReadyCheck object
----------------------------------

----------------------
-- Addon maintainance
----------------------
function DemocraticReadyCheck:OnInitialize()
    -- Called when the addon is loaded
    if Waterfall then
		Waterfall:Register("DemocraticReadyCheck",
		"aceOptions", options,
		"title", "DemocraticReadyCheck Options")
	end
end

function DemocraticReadyCheck:OnEnable()
    -- Called when the addon is enabled
    self:RegisterEvent("READY_CHECK")
    self:RegisterEvent("READY_CHECK_CONFIRM")
    self:RegisterEvent("READY_CHECK_FINISHED")
end

function DemocraticReadyCheck:OnDisable()
    -- Called when the addon is disabled
end

function DemocraticReadyCheck:OpenWaterfall()
    if Waterfall then
        Waterfall:Open("DemocraticReadyCheck")
    else
        self:Print("Waterfall-1.0 is required to access the GUI.")
    end
end

----------------------
-- Value get/setters
----------------------
function DemocraticReadyCheck:GetInitiateType()
    return self.db.profile.initiateType
end

function DemocraticReadyCheck:SetInitiateType(newValue)
    self.db.profile.initiateType = newValue
end

function DemocraticReadyCheck:GetConfirmType()
    return self.db.profile.confirmType
end

function DemocraticReadyCheck:SetConfirmType(newValue)
    self.db.profile.confirmType = newValue
end

function DemocraticReadyCheck:GetCompleteType()
    return self.db.profile.completeType
end

function DemocraticReadyCheck:SetCompleteType(newValue)
    self.db.profile.completeType = newValue
end

function DemocraticReadyCheck:GetFinalStatusType()
    return self.db.profile.finalStatusType
end

function DemocraticReadyCheck:SetFinalStatusType(newValue)
    self.db.profile.finalStatusType = newValue
end

function DemocraticReadyCheck:GetInitiateStr()
    return self.db.profile.initiateStr
end

function DemocraticReadyCheck:SetInitiateStr(newValue)
    self.db.profile.initiateStr = newValue
end

function DemocraticReadyCheck:GetConfirmYesStr()
    return self.db.profile.confirmYesStr
end

function DemocraticReadyCheck:SetConfirmYesStr(newValue)
    self.db.profile.confirmYesStr = newValue
end

function DemocraticReadyCheck:GetConfirmNoStr()
    return self.db.profile.confirmNoStr
end

function DemocraticReadyCheck:SetConfirmNoStr(newValue)
    self.db.profile.confirmNoStr = newValue
end

function DemocraticReadyCheck:GetCompleteStr()
    return self.db.profile.completeStr
end

function DemocraticReadyCheck:SetCompleteStr(newValue)
    self.db.profile.completeStr = newValue
end

function DemocraticReadyCheck:GetFinalStatusNoStr()
    return self.db.profile.finalStatusNoStr
end

function DemocraticReadyCheck:SetFinalStatusNoStr(newValue)
    self.db.profile.finalStatusNoStr = newValue
end

function DemocraticReadyCheck:GetFinalStatusAfkStr()
    return self.db.profile.finalStatusAfkStr
end

function DemocraticReadyCheck:SetFinalStatusAfkStr(newValue)
    self.db.profile.finalStatusAfkStr = newValue
end

function DemocraticReadyCheck:GetFinalStatusOkStr()
    return self.db.profile.finalStatusOkStr
end

function DemocraticReadyCheck:SetFinalStatusOkStr(newValue)
    self:Print("SetFinalStatusOkStr(" .. newValue .. ")")
    self.db.profile.finalStatusOkStr = newValue
end
----------------------
-- Business
----------------------
function DemocraticReadyCheck:READY_CHECK(characterName, arg2)
    -- Called when a ready check is performed
    self:sendMessage(string.format(self.db.profile.initiateStr, characterName), self.db.profile.initiateType)
    self.notReady = {}
    self.ready = {}
    self.afk = self:getFullGroup()
    self:afkRemove(characterName)
end

function DemocraticReadyCheck:READY_CHECK_CONFIRM(prtyMN, status)
    -- Called when a player confirms ready status
    local unitName = UnitName(self:getUnitPrefix() .. prtyMN)
    self:afkRemove(unitName)
    local stStr
    if status == 1 then
       stStr = self.db.profile.confirmYesStr
       table.insert(self.ready, unitName)
    else
       stStr = self.db.profile.confirmNoStr
       table.insert(self.notReady, unitName)
    end
    self:sendMessage(string.format(stStr, unitName), self.db.profile.confirmType)
end

function DemocraticReadyCheck:READY_CHECK_FINISHED()
    -- Called when ready check completes
    self:sendMessage(self.db.profile.completeStr, self.db.profile.completeType)
    local stStr = nil
    local charListStr = nil
    if (table.getn(self.notReady) > 0) then
        stStr = self.db.profile.finalStatusNoStr
        charListStr = strjoin(", ", unpack(self.notReady))
    elseif (table.getn(self.afk) > 0) then
        stStr = self.db.profile.finalStatusAfkStr
        charListStr = strjoin(", ", unpack(self.afk))
    else
        stStr = self.db.profile.finalStatusOkStr
    end
    self:sendMessage(string.format(stStr, charListStr), self.db.profile.finalStatusType)

    self.notReady = nil
    self.ready = nil
    self.afk = nil 
end

function DemocraticReadyCheck:sendMessage(message, sendType)
    -- Called when something has happened that may result in a message
    if (sendType == "CHAT") then
        SendChatMessage(message, self:getUnitPrefix())
    elseif (sendType == "WARN") then
        SendChatMessage(message, "RAID_WARNING")
    end
end

function DemocraticReadyCheck:getUnitPrefix()
    if UnitInRaid("player") == 1 then
       return "RAID"
    else
       return "PARTY"
    end
end

function DemocraticReadyCheck:afkRemove(name)
    for i=1, table.getn(self.afk) do
        if (name == self.afk[i]) then
            table.remove(self.afk, i)
            break
        end
    end
end

function DemocraticReadyCheck:getFullGroup()
    local groupCount
    local fullGroup = {}
    if UnitInRaid("player") == 1 then
        groupCount = GetNumRaidMembers()
    else
        groupCount = GetNumPartyMembers()
    end
    for i=1, groupCount do
        local unitName = UnitName(self:getUnitPrefix() .. i)
        table.insert(fullGroup, unitName)
    end
    return fullGroup
end