--[[
See this page for theory:
http://www.wowwiki.com/Crushing_Blow
]]

local Tablet = AceLibrary("Tablet-2.0")
local L = Rock("LibRockLocale-1.0"):GetTranslationNamespace("CrushChance")

CrushChance = Rock:NewAddon("CrushChance", "LibRockConfig-1.0", "LibRockDB-1.0", "LibFuBarPlugin-3.0","LibRockEvent-1.0")

CrushChance:SetDatabase("CrushChanceDB")
CrushChance:SetDatabaseDefaults('profile', {
   x=400,
   y=400,
   floatFrameEnabled = true,
   mobLevel=0,
   crushR=0.96,
   crushG=0.0,
   crushB=0.06,
   hitR=1,
   hitG=0.52,
   hitB=0,
   okR=0,
   okG=1,
   okB=0,
   dualWield = false,
   decimalPlaces = 1
})

CrushChance.version = "2.1"
CrushChance.date = string.sub("$Date: 2008-02-02 16:05:52 -0800 (Sat, 02 Feb 2008) $", 8, 17)

CrushChance:SetFuBarOption('hasIcon', true)
CrushChance:SetFuBarOption('iconPath', 'Interface\\Icons\\Ability_Rogue_CheatDeath')
CrushChance:SetFuBarOption('tooltipType', "Tablet-2.0")

function CrushChance:OnInitialize()
	local optionsTable = {
		name = "CrushChance",
		desc = self.notes,
		handler = CrushChance,
      icon = 'Interface\\Icons\\Ability_Rogue_CheatDeath',
		type = 'group',
		args = {
		   mobLevel = {
		      type = 'range',
            name = L['Mob Level'],
            desc = L['The mob level to assume, if zero uses the level of your latest target'],
            get = "GetLevel",
            set = "SetLevel",
            min = 0,
            max = 73,
            step = 1,
            order = 1,
		   },
         dualWield = {
            type = 'toggle',
            name = L['Mob Dual Wields'],
            desc = L['Assume the mob is dual wielding'],
            get = 'GetDualWield',
            set = 'SetDualWield',
            order = 2
         },
         enableFloatFrame = {
            type = 'toggle',
            name = L['Enable floating crush chance number'],
            desc = L['Enable the floating crush chance number frame.'],
            get = 'IsFloatFrameEnabled',
            set = 'SetFloatFrameEnabled',
            order = 3
         },
         resetPosition = {
            type = 'execute',
            name = L['Reset floating number position'],
            desc = L['Resets the position of the floating crush chance number.'],
            func = 'ResetPosition',
            disabled = function()
               return not CrushChance:IsFloatFrameEnabled()
            end,
            order = 4
         },
         decimalPlaces = {
            type = 'range',
            name = L['Decimal Places'],
            desc = L['The number of decimal places to show.'],
            get = 'GetDecimalPlaces',
            set = 'SetDecimalPlaces',
            min = 0,
            max = 3,
            step = 1,
            order = 5,
         },
			okColor = {
				type = 'color',
				name = L["Uncrushable Color"],
				desc = L["Set the text color for when you're uncrushable."],
				get = "GetOkColor",
				set = "SetOkColor",
            order = 6
			},
			crushColor = {
				type = 'color',
				name = L["Crushable Color"],
				desc = L["Set the text color for when you're crushable."],
				get = "GetCrushColor",
				set = "SetCrushColor",
            order = 7
			},
			hitColor = {
				type = 'color',
				name = L["Hitable Color"],
				desc = L["Set the color for when you're hitable."],
				get = "GetHitColor",
				set = "SetHitColor",
            order = 8
			}                           
		}
	}
	self:SetConfigTable(optionsTable)
	self.OnMenuRequest = optionsTable
   self:SetConfigSlashCommand(L["/crushchance"])
end

function CrushChance:OnEnable()
   if self.db.profile.floatFrameEnabled then
      self:GetFloatFrame()
   end

   self.decimalFormat = "%."..self.db.profile.decimalPlaces.."f"
   
   --SetLevel calls UpdateLevelBasedNumbers, which calls UpdateChance
   self:OnInventoryChanged()
   self:SetLevel(self.db.profile.mobLevel)
   
   self:AddEventListener("Blizzard","PLAYER_AURAS_CHANGED","UpdateChance",0.25)
   self:AddEventListener("Blizzard","UNIT_INVENTORY_CHANGED","OnInventoryChanged",0.5)
   --Gained defense skill point
   self:AddEventListener("Blizzard","CHAT_MSG_SKILL","UpdateChance",0.25)
end

function CrushChance:OnInventoryChanged()
   local il = GetInventoryItemLink("player", 17)
   if il then
      local _, _, _, _, _, _, itemSubType, _, _, _ = GetItemInfo(il) 
      self.shieldEquiped = itemSubType=='Shields'
   else
      self.shieldEquiped = false 
   end
   
   if self.mobLevel ~= nil then
      self:UpdateChance()
   end
end


function CrushChance:OnDisable()
   self:RemoveAllEventListeners()
   if self.floatFrame then
      self.floatFrame:Hide()
   end	   
end

function CrushChance:PLAYER_TARGET_CHANGED()
   local targetLevel = UnitLevel("target")
   if targetLevel == -1 then
      return
   end
   self.mobLevel = targetLevel
   self:UpdateLevelBasedNumbers()
end

-- Options Getters & Setters
function CrushChance:IsFloatFrameEnabled()
   return self.db.profile.floatFrameEnabled
end

function CrushChance:SetFloatFrameEnabled(enabled)
   self.db.profile.floatFrameEnabled = enabled
   local f = self:GetFloatFrame()
   if enabled then
      f:Show()
   else
      f:Hide()
   end
end

function CrushChance:GetHitColor()
	return self.db.profile.hitR, self.db.profile.hitG, self.db.profile.hitB
end

function CrushChance:GetHitHexColor()
	return string.format("%02x%02x%02x", self.db.profile.hitR * 255, self.db.profile.hitG * 255, self.db.profile.hitB * 255)
end

function CrushChance:SetHitColor(r, g, b)
	self.db.profile.hitR = r
	self.db.profile.hitG = g
	self.db.profile.hitB = b
   self:UpdateDisplays()
end

function CrushChance:GetCrushColor()
	return self.db.profile.crushR, self.db.profile.crushG, self.db.profile.crushB
end

function CrushChance:GetCrushHexColor()
	return string.format("%02x%02x%02x", self.db.profile.crushR * 255, self.db.profile.crushG * 255, self.db.profile.crushB * 255)
end

function CrushChance:SetCrushColor(r, g, b)
	self.db.profile.crushR = r
	self.db.profile.crushG = g
	self.db.profile.crushB = b
   self:UpdateDisplays()	
end

function CrushChance:GetOkColor()
	return self.db.profile.okR, self.db.profile.okG, self.db.profile.okB
end

function CrushChance:GetOkHexColor()
	return string.format("%02x%02x%02x", self.db.profile.okR * 255, self.db.profile.okG * 255, self.db.profile.okB * 255)
end

function CrushChance:SetOkColor(r, g, b)
	self.db.profile.okR = r
	self.db.profile.okG = g
	self.db.profile.okB = b
   self:UpdateDisplays()
end

function CrushChance:GetLevel()
   return self.db.profile.mobLevel
end

function CrushChance:SetLevel(mobLevel)
   self.db.profile.mobLevel = mobLevel
   if mobLevel == 0 then
      self.mobLevel = 73
      self:AddEventListener("Blizzard","PLAYER_TARGET_CHANGED","PLAYER_TARGET_CHANGED")
   else
      self:RemoveEventListener("Blizzard","PLAYER_TARGET_CHANGED")
      self.mobLevel = mobLevel
   end
   self:UpdateLevelBasedNumbers()
end

function CrushChance:GetDualWield()
   return self.db.profile.dualWield
end

function CrushChance:SetDualWield(dw)
   self.db.profile.dualWield = dw
   if self.floatFrame then
      if dw then
         self.floatFrame.dualWieldTex:Show()
      else
         self.floatFrame.dualWieldTex:Hide()
      end
   end
   self:UpdateChance()
end

function CrushChance:UpdateLevelBasedNumbers()
   --The number to adjust dodge,parry,block by
   self.adjustNum=(UnitLevel('player') - self.mobLevel)*0.2
      
   self:UpdateChance()
end

function CrushChance:GetDecimalPlaces()
   return self.db.profile.decimalPlaces
end

function CrushChance:SetDecimalPlaces(v)
   self.db.profile.decimalPlaces = v
   self.decimalFormat = "%."..v.."f"
   self:UpdateDisplays()
end

function CrushChance:UpdateChance()
   --The percent adjustment from defense applying to miss, crit, dodge,block,parry
   local defenseAdjust = (UnitDefense('player')+GetCombatRatingBonus(CR_DEFENSE_SKILL)-5*self.mobLevel)*0.04
   local x = 100
   self.missChance=5+defenseAdjust
   if self.db.profile.dualWield then
      self.missChance = self.missChance + 20
   end
   if self.missChance <= 0 then
      self.missChance = 0
   else
      x = x - self.missChance   
   end
   
   self.dodgeChance = GetDodgeChance()+self.adjustNum
   if self.dodgeChance <= 0 then
      self.dodgeChance = 0
   else
      x = x - self.dodgeChance
      if x <= 0 then
         self.dodgeChance = self.dodgeChance + x
         self.parryChance = 0
         self.critChance = 0
         self.blockChance = 0
         self.crushChance = 0
         self.hitChance = 0
         self:UpdateDisplays()
         return
      end
   end
   
   self.parryChance = GetParryChance()+self.adjustNum
   if self.parryChance <= 0 then
      self.parryChance = 0
   else
      x = x - self.parryChance
      if x <= 0 then
         self.parryChance = self.parryChance + x
         self.critChance = 0
         self.blockChance = 0
         self.crushChance = 0
         self.hitChance = 0
         self:UpdateDisplays()
         return      
      end
   end  
   
   self.critChance = 5 - defenseAdjust - GetCombatRatingBonus(CR_CRIT_TAKEN_MELEE)
   if self.critChance <= 0 then
      self.critChance = 0
   else
      x = x - self.critChance
      if x <= 0 then
         self.critChance = self.critChance + x
         self.blockChance = 0
         self.crushChance = 0
         self.hitChance = 0
         self:UpdateDisplays()
         return      
      end
   end  

   if self.shieldEquiped then
      self.blockChance = GetBlockChance()+self.adjustNum
   else
      self.blockChance = 0
   end
   
   if self.blockChance <= 0 then
      self.blockChance = 0
   else
      x = x - self.blockChance
      if x <= 0 then
         self.blockChance = self.blockChance + x
         self.crushChance = 0
         self.hitChance = 0
         self:UpdateDisplays()
         return      
      end
   end
         
   local defense = UnitDefense('player')
   -- See: http://forums.wow-europe.com/thread.html?topicId=14551489&sid=1
   if defense > 5*UnitLevel('player') then
      defense = 5*UnitLevel('player')
   end
   self.crushChance = (self.mobLevel*5-defense)*2 - 15
   if self.crushChance <= 0 then
      self.crushChance = 0
   else
      x = x - self.crushChance
      if x <= 0 then
         self.crushChance = self.crushChance + x
         self.hitChance = 0
         self:UpdateDisplays()
         return      
      end
   end
   
   self.hitChance = x
   
   self:UpdateDisplays()
end

function CrushChance:UpdateDisplays()
   self:UpdateFloaterText()
   self:UpdateFuBarText()
   self:UpdateFuBarTooltip()
end

function CrushChance:UpdateFloaterText()
   if self.floatFrame then
      self.floatFrame.chanceText:SetText(string.format("|cff%s"..self.decimalFormat.."|r",
         self.crushChance > 0 and self:GetCrushHexColor() or self:GetOkHexColor(),self.crushChance))
   end
end


function CrushChance:OnUpdateFuBarText()
   self:SetFuBarText(string.format("|cff%s"..self.decimalFormat.."|r", 
      self.crushChance > 0 and self:GetCrushHexColor() or self:GetOkHexColor(),self.crushChance))
end

function CrushChance:OnUpdateFuBarTooltip()

   local cat = Tablet:AddCategory(
		'columns', 2,
		'child_textR', 1,
		'child_textG', 0.85,
		'child_textB', 0.24,
		'child_text2R', 1,
		'child_text2G', 1,
		'child_text2B', 1
	)
      
   local s = L['Mob Level']
   if self.db.profile.dualWield then
      s=s..' '..L['(DW)']
   end   
   cat:AddLine(
		'text', s,
		'text2', self.mobLevel
	)
   
   cat:AddLine(
      'text', L['Miss'],
      'text2', string.format(self.decimalFormat,self.missChance)
   )
      
	cat:AddLine(
	   'text', L['Dodge'],
		'text2', string.format(self.decimalFormat, self.dodgeChance )
	)
   cat:AddLine(
      'text', L['Parry'],
      'text2', string.format(self.decimalFormat, self.parryChance )
   )
   
   cat:AddLine(
      'text', L['Crit'],
      'text2', string.format(self.decimalFormat, self.critChance )
   )
         
	cat:AddLine(
	   'text', L['Block'],
		'text2', string.format(self.decimalFormat, self.blockChance )
	)
	
   
   local r,g,b
   if self.crushChance > 0 then
      r,g,b = self:GetCrushColor()
   else
      r,g,b = self:GetOkColor()
   end
   
   cat:AddLine(
      'text',L['Crush'],
      'text2', string.format(self.decimalFormat, self.crushChance ),
      'textR',r,
      'textG',g,
      'textB',b,
      'text2R',r,
      'text2G',g,
      'text2B',b      
   )

   if self.hitChance > 0 then
      r,g,b = self:GetHitColor()
   else
      r,g,b = self:GetOkColor()
   end
   
   cat:AddLine(
      'text',L['Hit'],
      'text2', string.format(self.decimalFormat, self.hitChance),
      'textR',r,
      'textG',g,
      'textB',b,
      'text2R',r,
      'text2G',g,
      'text2B',b      
   )
   
end

function CrushChance:GetFloatFrame()
   if self.floatFrame == nil then
      --create frame
      local f = CreateFrame("Frame",nil,UIParent)
      f:SetPoint("TOPLEFT",UIParent,"BOTTOMLEFT",
           self.db.profile.x,self.db.profile.y)
      f:SetWidth(60)
      f:SetHeight(20)
         
      --Chance Text
      local t = f:CreateFontString(nil,"OVERLAY","GameFontNormal")
      t:SetAllPoints(f)
      t:SetJustifyH("LEFT")
      f.chanceText = t
         
      --Dual-Wield Texture
      local tex = f:CreateTexture()
      tex:SetTexture("Interface\\Buttons\\UI-Checkbox-SwordCheck")
      tex:SetTexCoord(0,0.593,0,0.5)
      tex:SetPoint("RIGHT",t,"LEFT")
      tex:SetWidth(12)
      tex:SetHeight(10)
      tex:Hide()
      f.dualWieldTex = tex
      
      if self.db.profile.dualWield then
         tex:Show()
      end
      
      f:EnableMouse(true)
      f:SetMovable(true)		
         
      f:SetScript("OnMouseDown",
         function()
            if arg1 == "RightButton" or IsModifierKeyDown() then
               f:StartMoving()
            else
               CrushChance:SetDualWield(not CrushChance:GetDualWield())
            end
         end)
         
      f:SetScript("OnMouseUp",
         function()
            f:StopMovingOrSizing()
            CrushChance.db.profile.x = f:GetLeft()
            CrushChance.db.profile.y = f:GetTop()            
         end)
            
      f:Show()
      self.floatFrame = f
   end
   
   return self.floatFrame
end

function CrushChance:ResetPosition()
   local f = self.floatFrame
   if f == nil then
      return
   end
	f:SetPoint("TOPLEFT",UIParent,"BOTTOMLEFT",
		UIParent:GetWidth()/2, UIParent:GetHeight()/2)	
end
