CooldownWatch = {}
CooldownWatchVar = {}
--CooldownWatchInit = {} -- nil after initalization, recreated on shutdown

local CooldownWatch = DongleStub("Dongle-1.2"):New("CooldownWatch",CooldownWatch)
local db
CooldownWatchVar.version = 4

local frame, frameByName, inventoryframe, frameBySlot, seconds, minutes

frame = {}
frameByName = {}
inventoryframe = {}
frameBySlot = {}
seconds, minutes = {}, {}
for i=1,120 do
	tinsert(seconds, i.."s")
end
for i=1,120 do
	tinsert(minutes, i.."m")
end

CooldownWatch.frame = frame


----------------------------------------------------------------------------
-- Module system
-- Modules define table entries of CooldownWatch, mostly functions.
-- A module may be located in an addon, in this case the addon is loaded
-- when the table entry is indexed.
-- Each module is initialized with "private" data, so that this data may 
-- be used as upvalues.
----------------------------------------------------------------------------


local modules = {
	Config = {
		addon = "CooldownWatchConfig",
		data = {
			"Test",
			"RemoveWatch",
			"RemoveAll",
			'MoveFrame',
			'GetConfigMode',
			'ToggleConfigMode',
			'ConfigMode',
			'RecreateAll',
		},	
		status = 'not loaded',
		dependencies = {'SaveConfig', 'InitConfig'},
	},
	Save = {
		--addon = "CooldownWatchPersistence",
		data = {
			"SaveData",
			"Disable",
		},
		status = 'load',
		dependencies = {'SaveConfig'},
	},
	SaveConfig = {
		data = {
		},
		status = 'not loaded',
	},
	Init = {
		data = {
			"LoadData",
			"UnloadInit",
			"FirstTime",
			"Initialize",
			"UpdateLoadData",
		},
		status = 'load',
		dependencies = {'InitConfig'},
	},
	InitConfig = {
		data = {
			--[[
			"AddWatch",
			"AddEquipWatch",
			"AddItemWatch",
			"AddSpellWatch",
			"CreateWatchFrame",
			]]
		},
		status = 'load',
	},
}
CooldownWatch.modules = modules

local function initialize(modulename)
	local module = modules[modulename]
	if module.dependencies then
		for _,depname in ipairs(module.dependencies) do
			initialize(depname)
		end
	end
	if module.status =='loaded' then
		if type(module.init) == "function" then
			--CooldownWatch:Print("initializing module "..modulename)
			module.init(frame, frameByName, inventoryframe, frameBySlot, seconds, minutes)
			module.init = nil
		else
			--CooldownWatch:Print("couldn't initialize module "..modulename)
		end
		module.status = 'initialized'
	end
end

local function __index(table, key)
	for modulename, module in pairs(modules) do
		for _, entry in pairs(module.data) do
			if entry == key then
				-- load if necessary
				if module.addon and module.status=='not loaded' then
					if module.dependencies then
						for _,dep in ipairs(module.dependencies) do
							if modules[dep].status=='not loaded' then
								modules[dep].status='load'
								--CooldownWatch:Print("dependency "..dep)
							end
						end
					end
					CooldownWatch:Print("loading addon "..module.addon)
					module.status = 'load'

					LoadAddOn(module.addon)
				end
				if module.status == 'load' then
					module.status = 'loaded'
				end
				if module.dependencies then
					for _,dep in ipairs(module.dependencies) do
						if modules[dep].status=='load' then
							modules[dep].status='loaded'
						end
					end
				end
				initialize(modulename)
				return rawget(table, key)
			end
		end
	end
end
setmetatable(CooldownWatch, {__index=__index})


----------------------------------------------------------------------------
-- List layouts
----------------------------------------------------------------------------
CooldownWatch.listlayout = {}
do
	local Expand = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPLEFT", self.anchor, "BOTTOMLEFT", 0, -2)
	end
	local Collapse = function(self)
		self:ClearAllPoints()
		self:SetPoint("BOTTOMLEFT", self.anchor, "BOTTOMLEFT")
	end
	CooldownWatch.listlayout['TOP_BOTTOM'] = { Collapse = Collapse, Expand = Expand }
end
do
	local Expand = function(self)
		self:ClearAllPoints()
		self:SetPoint("BOTTOMLEFT", self.anchor, "TOPLEFT", 0, 2)
	end
	local Collapse = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPLEFT", self.anchor, "TOPLEFT")
	end
	CooldownWatch.listlayout['BOTTOM_TOP'] = { Collapse = Collapse, Expand = Expand }
end
do
	local Expand = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPLEFT", self.anchor, "TOPRIGHT", 2, 0)
	end
	local Collapse = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPRIGHT", self.anchor, "TOPRIGHT")
	end
	CooldownWatch.listlayout['LEFT_RIGHT'] = { Collapse = Collapse, Expand = Expand }
end
do
	local Expand = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPRIGHT", self.anchor, "TOPLEFT", -2, 0)
	end
	local Collapse = function(self)
		self:ClearAllPoints()
		self:SetPoint("TOPLEFT", self.anchor, "TOPLEFT")
	end
	CooldownWatch.listlayout['RIGHT_LEFT'] = { Collapse = Collapse, Expand = Expand }
end

-- TODO
local develop = nil

function CooldownWatch:Trace(...)
	self:Debug(3,...)
end

--[[
interesting event sequences:

SPELL_UPDATE_COOLDOWN
BAG_UPDATE_COOLDOWN(container ID or nil)
PET_BAR_UPDATE_COOLDOWN
ACTIONBAR_UPDATE_COOLDOWN

If a spell is used:
UNIT_SPELLCAST_SUCCEEDED("player", spellname)
UNIT_SPELLCAST_STOP("player", spellname)
ACTIONBAR_UPDATE_COOLDOWN
SPELL_UPDATE_COOLDOWN


If an equippable item is used:
UNIT_SPELLCAST_SUCCEEDED("player", buffname) buffname is the name of the trinket specific buff.
SPELL_UPDATE_COOLDOWN
BAG_UPDATE_COOLDOWN()

If an item is equipped
ITEM_LOCK_CHANGED(bag,slot) lock the bag item
SPELL_UPDATE_COOLDOWN()
BAG_UPDATE_COOLDOWN()
UNIT_INVENTORY_CHANGED("player")
ITEM_LOCK_CHANGED(slot) lock the inventory item

If an unequippable item is used
UNIT_SPELLCAST_SUCCEEDED("player", spellname) spellname seems to be specific to the item type, and is not the item name
SPELL_UPDATE_COOLDOWN()
BAG_UPDATE_COOLDOWN()
]]

-- check all items
function CooldownWatch:CheckItems()
	for i=1,#inventoryframe do
		inventoryframe[i].CheckCooldown()
	end
end

-- check inventory slot
function CooldownWatch:CheckInventory(event, slot)
	local f = frameBySlot[slot]
	if f then
		f.CheckCooldown()
		if f.icon then
			f.icon:SetTexture(f.GetTexture(f.id, f.idtype))
		end
	end
end

-- tell a spell frame to update on SPELL_UPDATE_COOLDOWN
function CooldownWatch:CheckSpell(event, unitid, spellname)
	if unitid~="player" then
		return
	end
	local cooldownframe = frameByName[spellname]
	if cooldownframe then
		cooldownframe:RegisterEvent("SPELL_UPDATE_COOLDOWN")
	end
end

do
	local equipListener,spellListener,itemListener
	local listening
	function CooldownWatch:StartEventListening()
		if not equipListener then
			equipListener = CreateFrame("Frame",nil,CooldownWatchFrame)
			equipListener:SetScript("OnEvent", CooldownWatch.CheckInventory)
			spellListener = CreateFrame("Frame",nil,CooldownWatchFrame)
			spellListener:SetScript("OnEvent", CooldownWatch.CheckSpell)
			itemListener = CreateFrame("Frame",nil,CooldownWatchFrame)
			itemListener:SetScript("OnEvent", CooldownWatch.CheckItems)
		end
		equipListener:RegisterEvent("ITEM_LOCK_CHANGED")
		spellListener:RegisterEvent("UNIT_SPELLCAST_SUCCEEDED")
		itemListener:RegisterEvent("BAG_UPDATE_COOLDOWN")
		listening = true
	end
	function CooldownWatch:StopEventListening()
		if listening then
			equipListener:UnregisterEvent("ITEM_LOCK_CHANGED")
			spellListener:UnregisterEvent("UNIT_SPELLCAST_SUCCEEDED")
			itemListener:UnregisterEvent("BAG_UPDATE_COOLDOWN")
		end
		listening = nil
	end
end

function CooldownWatch:Enable()
	CooldownWatch:StartEventListening()
	if self.LoadData then
		self:LoadData()
		CooldownWatch:UnloadInit()
	end
end

function CooldownWatch.modules.Init.init(...)
function CooldownWatch:UpdateLoadData()
	self.UpdateLoadData = nil

	if not db.version then
		-- ignore the update code, watches have to be recreated.
		CooldownWatchInit.updatecode = nil
		db.version = 1
	end
	if db.version == 1 then
		CooldownWatchInit.updatecode = nil
		db.width = db.width+db.height
		db.version = 2
	end
	if db.version == 2 then
		CooldownWatchInit.updatecode = nil
		db.version = 3
	end
	if db.version == 3 then
		db.frameData = CooldownWatchInit.frameData
		CooldownWatchInit = nil
		db.version = 4
	end
end


function CooldownWatch:Initialize()
	local firstTime

	--develop = UnitName("player") == "Devora"
	if develop then
		self:Print("develop mode")
	end
	db = CooldownWatchVar
	if not db.width then firstTime = true end
	if not db.width then db.width = 108 end
	if not db.height then db.height = 18 end
	if not db.statusbarcolor then db.statusbarcolor = {1,1,1,0.3} end
	if not db.backdropcolor then db.backdropcolor = {1,1,1,0.1} end
	if not db.textcolor then db.textcolor = {1,1,1,1} end
	if not db.textshadowcolor then db.textshadowcolor = {0,0,0,1} end
	if not db.bartexture then db.bartexture = [[Interface\Addons\CooldownWatch\Smooth]] end
	if not db.layout then db.layout = 'TOP_BOTTOM' end
	if not db.font then db.font = "Fonts\\FRIZQT__.TTF" end
	if not db.fontsize then db.fontsize = 10 end

	local f = CreateFrame("Frame","CooldownWatchFrame",UIParent)
	do
		-- Bug fix: At the 2. frame after a /rl, the elapsed parameter is bogus, which leads to wrong cooldowns.
		-- This seems to be a flaw in Blizz code.
		local numframestowait = 10
		local numframes = 0
		local function onupdatebugfix(_, elapsed)
			numframes=numframes+1
			if numframestowait>1 then
				numframestowait=numframestowait-1
				if elapsed>5 then 
					numframestowait = 1 
				end
				return
			end
			--CooldownWatch:Print('checking after',numframes,'frames')
			for i,f in ipairs(frame) do 
				f.Stop() 
				f.CheckCooldown() 
			end
			f:SetScript('OnUpdate', nil)
		end
		f:SetScript('OnUpdate', onupdatebugfix)
	end
	self:UpdateLoadData()

	local sco = self:InitializeSlashCommand("CooldownWatch Configuration", "CDW", "cw", "cooldown", "cooldownwatch")
	sco:RegisterSlashHandler("CooldownWatch Configuration", ".*", function() CooldownWatch:ToggleConfigMode() end)

	if CooldownWatch:IsDebugEnabled() then
		setglobal("f",frame)
	end
	if develop then
		setglobal("f",frame)
		setglobal("c",CooldownWatch)
		setglobal("db",db)
		if develop then
			return
		end
		local abbrev = {}
		for fname,f in pairs(CooldownWatch) do
			if f==rawget(CooldownWatch,fname) and type(fname)=="string" and type(f)=="function" then
				local name = fname:gsub("[a-z]",""):lower()
				while CooldownWatch[name] do
					name = name.."x"
				end
				abbrev[fname] = name
			end
		end
		for fname,name in pairs(abbrev) do
			CooldownWatch[name] = CooldownWatch[fname]
		end
	end

	if firstTime then
		CooldownWatch:FirstTime()
	end
	CooldownWatch.initialized = true
end


function CooldownWatch:UnloadInit()
	self.LoadData = nil
	self.UpdateLoadData = nil
	self.FirstTime = nil
	self.Initialize = nil

	if not self.modules.Config.status=='loaded' then
		-- unload InitConfig
		self.CreateWatchFrame = nil
		self.AddWatch = nil
		self.GetOnUpdate = nil
		self.UnloadInit = nil
		modules.InitConfig.status = 'not loaded'
	end
end

function CooldownWatch:FirstTime()
	self:Print('Usage:')
	local function p(s) ChatFrame1:AddMessage(s) end
	p([[Enter /cooldownwatch or /cw to toggle the config mode. A small anchor frame is visible if you're in config mode.]])
	p([[Move the cooldown frames: Drag the anchor.]])
	p([[Add a cooldown: Drag and drop the spell or item to the anchor frame to add it at the end.]])
	p([[Or drop it on another cooldown to insert it at that position, and use that cooldown's settings.]])
	p([[Note: If you drag an item from your bags, the item itself will be added as the cooldown. If you drag an item from your character frame, the equipment slot it came from will be watched. That way, you can watch for example the cooldowns of the trinkets you're currently wearing.]])
	p([[Remove a cooldown: Middle click the cooldown frame.]])
	p([[Reorder cooldowns: Drag the cooldown frame.]])
	p([[Access cooldown specific options: Right-click the cooldown frame.]])
	p([[Access global options: Right-click the anchor frame.]])
	self:ConfigMode(true)
	p([[IMPORTANT: While you're in config mode, cooldowns won't get updated. You have to leave config mode in order to make cooldowns work.]])
end

function CooldownWatch:LoadData()
	if db.debug then
		self:EnableDebug(db.debug)
		self:Print("EnableDebug",db.debug)
	end

	CooldownWatchAnchor.layout = db.layout
	CooldownWatchFont:SetFont(db.font, db.fontsize, 'OUTLINE')
	CooldownWatchFont:SetTextColor(unpack(db.textcolor))
	CooldownWatchFont:SetShadowColor(unpack(db.textshadowcolor))

	if db.frameData then
		self:Debug(1,'creating stuff')
		for frameid,data in ipairs(db.frameData) do 
			local options = {
				nobar = data.nobar,
				label = data.label,
				autohide = data.autohide,
				autohidebar = data.autohidebar,
				duration = data.duration,
				remaining = data.remaining,
			}

			self:AddWatch(data.watchtype, data.watchname, options)
		end
	end
end
end


