local sm = LibStub("LibSharedMedia-2.0")
if AceLibrary:HasInstance("Waterfall-1.0") then
	local waterfall = AceLibrary("Waterfall-1.0")
end
local L = AceLibrary("AceLocale-2.2"):new("CooldownTimers")

CooldownTimers.Options = {
	type = 'group',
	args = {
		options = {
			type = 'execute',
			name = L["Open GUI"],
			desc = L["Opens the Waterfall GUI for configuration"],
			func = function()
				CooldownTimers:OpenWaterfall()
			end,
			wfHidden = true,
			hidden = function()
				return (not waterfall)
			end,
			order = 1,
		},
		maxtime = {
			type = 'text',
			name = L["Max Time"],
			desc = L["Set the max time for a cooldown to be tracked."],
			get = function()
				return CooldownTimers.db.profile.maxtime
			end,
			set = function(s)
				CooldownTimers.db.profile.maxtime = tonumber(s)
			end,
			validate = function(s)
				return not string.find(s, "%D")
			end,
			usage = L["<time>"],
			order = -3,
		},
		mintime = {
			type = 'text',
			name = L["Min Time"],
			desc = L["Set the min time for a cooldown to be tracked."],
			get = function()
				return CooldownTimers.db.profile.mintime
			end,
			set = function(s)
				CooldownTimers.db.profile.mintime = tonumber(s)
			end,
			validate = function(s)
				return not string.find(s, "%D")
			end,
			usage = L["<time>"],
			order = -3,
		},
		fixgroups = {
			type = 'execute',
			name = L["Fix Groups"],
			desc = L["Fix skills that have groups that no longer exist (moving them to groups that do exist)."],
			func = function()
				CooldownTimers:FixGroups()
			end,
			order = -2,
		},
		menu = {
			type = 'execute',
			name = L["Show Menu"],
			desc = L["Shows the configuration menu"],
			func = function()
				CooldownTimers:OpenMenu()
			end,
			order = -2,
		},
		sound = {
			type = 'toggle',
			name = L["Enable Sound"],
			desc = L["Disable or enable the sound that plays when a cooldown is finished"],
			get = function()
				return CooldownTimers.db.profile.sound
			end,
			set = function(v)
				CooldownTimers.db.profile.sound = v
			end,
		},
    condense = {
      type = 'toggle',
      name = L["Auto Condense Groups"],
      desc = L["Try to have the addon determine what should be condensed, tries to use the last skill/item used"],
      get = function()
        return CooldownTimers.db.profile.autogroup;
      end,
      set = function(v)
        CooldownTimers.db.profile.autogroup = v;
        CooldownTimers:PopulateCooldowns();
      end,
    },
		defaults = {
			type = 'group',
			name = L["Default Bar Settings"],
			desc = L["Sets the default bar look, used after custom group settings and individual skill settings"],
			order = 100,
			args = {
				texture = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Sets the status bar texture."],
					get = function()
						return CooldownTimers.db.profile.barOptions.texture
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.texture = v
					end,
					validate = sm:List("statusbar"),
					order = 1,
				},
				colors = {
					type = 'text',
					name = L["Colors"],
					desc = L["Sets the fade colors. Needs at least two, but will take as many colors as you want.\n"]..L["Usage: <color1> <color2> ..."],
					get = function()
						local colors
						for k,v in ipairs(CooldownTimers.db.profile.barOptions.colors) do
							if not colors then
								colors = v
							else
								colors = colors.." "..v
							end
						end
						return colors
					end,
					set = function(s)
						CooldownTimers.db.profile.barOptions.colors = {}
						for color in string.gmatch(s, "%a+") do
							table.insert(CooldownTimers.db.profile.barOptions.colors, color)
						end
					end,
					validate = function(s)
						return not string.find(s, "%a+ %a+")
					end,
					usage = L["Usage: <color1> <color2> ..."],
					order = 2,
				},
				fadetime = {
					type = 'text',
					name = L["Fade Time"],
					desc = L["Sets how long bars take to fade after the bar completes.\n"]..L["Usage: <fadetime> (in seconds)"],
					get = function()
						return CooldownTimers.db.profile.barOptions.fade
					end,
					set = function(s)
						CooldownTimers.db.profile.barOptions.fade = tonumber(s)
					end,
					validate = function(s)
						return string.find(s, "^%d+%.?%d*")
					end,
					usage = L["Usage: <fadetime> (in seconds)"],
					order = 3,
				},
				sliderspacer = {
					type = 'header',
					name = " ",
					order = 4,
				},
				barwidth = {
					type = 'range',
					name = L["Bar Width"],
					desc = L["Sets the default bar width."],
					min = 32,
					max = 200,
					step = 1,
					get = function()
						return CooldownTimers.db.profile.barOptions.barwidth
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.barwidth = v
					end,
					order = 5,
				},
				barheight = {
					type = 'range',
					name = L["Bar Height"],
					desc = L["Sets the default bar height."],
					min = 16,
					max = 64,
					step = 1,
					get = function()
						return CooldownTimers.db.profile.barOptions.barheight
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.barheight = v
					end,
					order = 6,
				},
				barscale = {
					type = 'range',
					name = L["Bar Scale"],
					desc = L["Sets the default bar scale."],
					min = 0.5,
					max = 2.0,
					step = 0.1,
					get = function()
						return CooldownTimers.db.profile.barOptions.scale
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.scale = v
					end,
					order = 7,
				},
				checkboxspacer = {
					type = 'header',
					name = " ",
					order = 8,
				},
				pulse = {
					type = 'toggle',
					name = L["Pulse on hit"],
					desc = L["Will pulse a bar of a skill that the player attempts to use while still on cooldown"],
					get = function()
						return CooldownTimers.db.profile.pulseoncooldown
					end,
					set = function(v)
						CooldownTimers.db.profile.pulseoncooldown = v
						if CooldownTimers.db.profile.pulseoncooldown and not CooldownTimers:IsEventRegistered("CHAT_MSG_SPELL_FAILED_LOCALPLAYER") then
							CooldownTimers:RegisterEvent("CHAT_MSG_SPELL_FAILED_LOCALPLAYER", "OnSpellFail")
						elseif not CooldownTimers.db.profile.pulsecooldown and CooldownTimers:IsEventRegistered("CHAT_MSG_SPELL_FAILED_LOCALPLAYER") then
							CooldownTimers:UnregisterEvent("CHAT_MSG_SPELL_FAILED_LOCALPLAYER")
						end
					end,
					order = 9,
				},
				stack = {
					type = 'toggle',
					name = L["Grow Downwards"],
					desc = L["Whether the bars will stack up, or stack down"],
					get = function()
						return not CooldownTimers.db.profile.barOptions.up
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.up = not v
					end,
					order = 10,
				},
				sort = {
					type = 'toggle',
					name = L["Sort and Collapse Bars"],
					desc = L["Whether the bars will be automatically sorted and automatically collapse"]..L["\n(See the left side for additional options when this is disabled)"],
					get = function()
						return CooldownTimers.db.profile.barOptions.collapse
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.collapse = v
					end,
					order = 11,
				},
				collapsespacer = {
					type = 'header',
					name = " ",
					order = 12,
				},
				gap = {
					type = 'range',
					name = L["Bar Gap"],
					desc = L["Sets the default space between bars."]..L["\n(Only used when bars do not automatically collapse.)"],
					hidden = function()
						return CooldownTimers.db.profile.barOptions.collapse
					end,
					min = 0,
					max = 32,
					step = 1,
					get = function()
						return CooldownTimers.db.profile.barOptions.bargap
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.bargap = v
					end,
					order = 13,
				},
				columns = {
					type = 'range',
					name = L["Bar Columns"],
					desc = L["Sets the number of bar columns."]..L["\n(Only used when bars do not automatically collapse.)"],
					hidden = function()
						return CooldownTimers.db.profile.barOptions.collapse
					end,
					min = 1,
					max = 5,
					step = 1,
					get = function()
						return CooldownTimers.db.profile.barOptions.columns
					end,
					set = function(v)
						CooldownTimers.db.profile.barOptions.columns = v
					end,
					order = 14,
				},
			},
		},
		--We remove this section from the command line because it is way to confusing to configure there.
		wfgroups = {
			type = 'group',
			name = L["Group Settings"],
			desc = L["Sets the settings for a particular group."],
			order = 100,
			cmdHidden = true,
			args = {
				name = {
					type = 'text',
					name = L["Group Name"],
					desc = L["Name of the group you would like to change settings for"],
					order = 2,
					get = function()
						if not next(CooldownTimers.Options.args.wfgroups.args.name.validate) then
							for k,v in pairs(CooldownTimers.db.profile.groups) do
								if not v.disabled then
									tinsert(CooldownTimers.Options.args.wfgroups.args.name.validate, k)
								end
							end
						end
						return selectedgroup
					end,
					set = function(v)
						selectedgroup = v
					end,
					validate = {},
				},
				new = {
					type = 'text',
					name = L["Create New Group"],
					desc = L["Make a new group to show cooldowns in.\n"]..L["Group names must contain only letters"],
					order = 1,
					get = function()
						return
					end,
					set = function(s)
						CooldownTimers.db.profile.groups[s] = {}
						tinsert(CooldownTimers.Options.args.wfgroups.args.name.validate, s)
						CooldownTimers:MakeAnchor(s, CooldownTimers.db.profile.groups[s])
					end,
					validate = function(s)
						return (not string.find(s, "%A") and not CooldownTimers.db.profile.groups[s]) or (CooldownTimers.db.profile.groups[s] and not CooldownTimers.db.profile.groups[s].disabled)
					end,
					usage = "<group name> (Numbers are not allowed, and make sure the group doesn't already exist)",
					--wfHidden = function() return not (selected == "") end,
				},
				buttonspacer = {
					type = 'header',
					name = " ",
					order = 4,
				},
				delete = {
					type = 'execute',
					name = L["Delete Group"],
					desc = L["Delete this group (NOT RESTORABLE)\nDefault groups cannot be deleted."],
					order = 3,
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					disabled = function()
						return (selectedgroup == "CDT")
					end,
					func = function()
						--Make sure that the selected option is still in the validate table before trying to remove it, and 
						--Even with disabled set to activate when the value of selectedgroup is "CDT", it is still possible to remove the CDT group,
						--this takes care of that.
						if selectedgroup ~= "CDT" and selectedgroup ~= "" and selectedgroup ~= nil then
							--Create the confirmation dialog
							StaticPopupDialogs["CDT_DELETE_GROUP_CHECK"] = {
								text = L["Are you sure you want to delete group: "]..selectedgroup,
								button1 = L["Yes"],
								button2 = L["No"],
								OnAccept = function()
									if selectedgroup == "PetCooldowns" or selectedgroup == "ItemCooldowns" then
										CooldownTimers.db.profile.groups[selectedgroup].disabled = true --otherwise it'll just be recreated by default
									else	
										CooldownTimers.db.profile.groups[selectedgroup] = nil
									end
									CooldownTimers.anchors[selectedgroup]:Hide()
									for k,v in pairs(CooldownTimers.db.class.cooldowns) do
										if v.group == selectedgroup then
											v.group = "CDT"
										end
									end
									for k,v in pairs(CooldownTimers.db.profile.itemcooldowns) do
										if v.group == selectedgroup then
											v.group = "CDT"
										end
									end
									for k,v in pairs(CooldownTimers.db.char.petcooldowns) do
										if v.group == selectedgroup then
											v.group = "CDT"
										end
									end
									for k,v in pairs(CooldownTimers.Options.args.wfgroups.args.name.validate) do
										if v == selectedgroup then
											tremove(CooldownTimers.Options.args.wfgroups.args.name.validate, k)
										end
									end
									selectedgroup = ""
								end,
								showAlert = true,
								timeout = 0,
							}
							--Show the confirmation dialog
							StaticPopup_Show("CDT_DELETE_GROUP_CHECK")
						end
					end,
				},
				locked = {
					type = 'toggle',
					name = L["Locked"],
					desc = L["Shows/Hides the group anchor."],
					order = 5,
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					get = function()
						if selectedgroup ~= "" and selectedgroup ~= nil then
							return CooldownTimers.db.profile.groups[selectedgroup].locked
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].locked = v
						if CooldownTimers.db.profile.groups[selectedgroup].locked then
							CooldownTimers.anchors[selectedgroup]:Hide()
						else
							CooldownTimers.anchors[selectedgroup]:Show()
						end
					end,
				},
				texture = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Sets the status bar texture."],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 6,
					get = function()
						if not next(CooldownTimers.Options.args.wfgroups.args.texture.validate) then
							tinsert(CooldownTimers.Options.args.wfgroups.args.texture.validate, L["Use Default"])
							for _,v in pairs(sm:List("statusbar")) do
								tinsert(CooldownTimers.Options.args.wfgroups.args.texture.validate, v)
							end
						end
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].texture == nil then
								return L["Use Default"]
							else
								return CooldownTimers.db.profile.groups[selectedgroup].texture
							end
						end
					end,
					set = function(v)
						if v == L["Use Default"] then
							CooldownTimers.db.profile.groups[selectedgroup].texture = nil
						else
							CooldownTimers.db.profile.groups[selectedgroup].texture = v
						end
					end,
					validate = {},
				},
				colors = {
					type = 'text',
					name = L["Colors"],
					desc = L["Sets the fade colors. Needs at least two, but will take as many colors as you want.\n"]..L["Usage: <color1> <color2> ..."],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 7,
					get = function()
						local colors
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].colors then
								for k,v in pairs(CooldownTimers.db.profile.groups[selectedgroup].colors) do
									if not colors then
										colors = v
									else 
										colors = colors.." "..v
									end
								end
							else
								colors = ""
							end
						end
						if colors == "" then
							return L["Use Default"]
						else
							return colors
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.profile.groups[selectedgroup].colors = nil
						else
							CooldownTimers.db.profile.groups[selectedgroup].colors = {}
							for color in string.gmatch(s, "%a+") do
								table.insert(CooldownTimers.db.profile.groups[selectedgroup].colors, color)
							end
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%a+ %a+"))
					end,
					usage = L["Usage: <color1> <color2> ..."],
				},
				fade = {
					type = 'text',
					name = L["Custom "]..L["Fade Time"],
					desc = L["Sets how long bars take to fade after the bar completes.\n"]..L["Usage: <fadetime> (in seconds)"],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 8,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].fade == nil then
								return L["Use Default"]
							else
								return CooldownTimers.db.profile.groups[selectedgroup].fade
							end
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.profile.groups[selectedgroup].fade = nil
						else
							CooldownTimers.db.profile.groups[selectedgroup].fade = tonumber(s)
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%d+%.?%d*"))
					end,
					usage = L["Usage: <fadetime> (in seconds)"],
				},
				barwidth = {
					type = 'range',
					name = L["Bar Width"],
					desc = L["Sets the default bar width."],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 9,
					min = 32,
					max = 200,
					step = 1,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].barwidth then
								return CooldownTimers.db.profile.groups[selectedgroup].barwidth
							else
								return CooldownTimers.db.profile.barOptions.barwidth
							end
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].barwidth = v
					end,
				},
				barheight = {
					type = 'range',
					name = L["Bar Height"],
					desc = L["Sets the default bar height."],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 10,
					min = 16,
					max = 64,
					step = 1,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].barheight then
								return CooldownTimers.db.profile.groups[selectedgroup].barheight
							else
								return CooldownTimers.db.profile.barOptions.barheight
							end
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].barheight = v
					end,
				},
				barscale = {
					type = 'range',
					name = L["Bar Scale"],
					desc = L["Sets the default bar scale."],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 11,
					min = 0.5,
					max = 2.0,
					step = 0.1,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].scale then
								return CooldownTimers.db.profile.groups[selectedgroup].scale
							else
								return CooldownTimers.db.profile.barOptions.scale
							end
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].scale = v
					end,
				},
				stack = {
					type = 'toggle',
					name = L["Grow Downwards"],
					desc = L["Whether the bars will stack up, or stack down"],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 12,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].up ~= nil then
								return (not CooldownTimers.db.profile.groups[selectedgroup].up)
							else
								return (not CooldownTimers.db.profile.barOptions.up)
							end
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].up = not v
						CooldownTimers:SetCandyBarGroupGrowth(selectedgroup, (not CooldownTimers.db.profile.groups[selectedgroup].up))
					end,
				},
				sort = {
					type = 'toggle',
					name = L["Sort and Collapse Bars"],
					desc = L["Whether the bars will be automatically sorted and automatically collapse"],
					wfHidden = function()
						return (selectedgroup == "") or (selectedgroup == nil)
					end,
					order = 13,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup] ~= nil then
							if CooldownTimers.db.profile.groups[selectedgroup].collapse ~= nil then
								return CooldownTimers.db.profile.groups[selectedgroup].collapse
							else 
								return CooldownTimers.db.profile.barOptions.collapse
							end
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].collapse = v
					end,
				},
				collapsespacer = {
					type = 'header',
					name = " ",
					order = 14,
				},
				gap = {
					type = 'range',
					name = L["Bar Gap"],
					desc = L["Sets the default space between bars."]..L["\n(Only used when bars do not automatically collapse.)"],
					wfHidden = function()
						return (selectedgroup == "") or (CooldownTimers.db.profile.groups[selectedgroup] == nil) or (CooldownTimers.db.profile.groups[selectedgroup].collapse)	end,
					order = 15,
					min = 0,
					max = 32,
					step = 1,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup].bargap then
							return CooldownTimers.db.profile.groups[selectedgroup].bargap
						else
							return CooldownTimers.db.profile.barOptions.bargap
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].bargap = v
					end,
				},
				columns = {
					type = 'range',
					name = L["Bar Columns"],
					desc = L["Sets the number of bar columns."]..L["\n(Only used when bars do not automatically collapse.)"],
					wfHidden = function()
						return (selectedgroup == "") or (CooldownTimers.db.profile.groups[selectedgroup] == nil) or (CooldownTimers.db.profile.groups[selectedgroup].collapse)
					end,
					min = 1,
					max = 5,
					step = 1,
					get = function()
						if CooldownTimers.db.profile.groups[selectedgroup].columns then
							return CooldownTimers.db.profile.groups[selectedgroup].columns
						else
							return CooldownTimers.db.profile.barOptions.columns
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.groups[selectedgroup].columns = v
					end,
					order = 16,
				},
			},
		},
		announce = {
			type = 'group',
			name = L["Announce Settings"],
			desc = L["Settings for the Announce display"],
			order = 400,
			args = {
				enabled = {
					type = 'toggle',
					name = L["Enabled"],
					desc = L["Enables and disables the announcement of a cooldown completion."],
					order = 1,
					get = function()
						return CooldownTimers.db.profile.announce.enabled
					end,
					set = function(v)
						CooldownTimers.db.profile.announce.enabled = v
					end,
				},
				locked = {
					type = 'toggle',
					name = L["Locked"],
					desc = L["Shows/Hides the announce anchor. Without locking the announce will never dissapear!"],
					order = 2,
					wfHidden = function()
						return not CooldownTimers.db.profile.announce.enabled
					end,
					get = function()
					if not CooldownTimers.db.profile.announce.locked then
							if CooldownTimers.db.profile.announce.enabled then
								CooldownTimers.announce.anchor:Show()
								CooldownTimers.announce.frame:Show()
								CooldownTimers.announce.frame:SetAlpha(1)
								CooldownTimers.announce.alpha = 1
							else
								CooldownTimers.announce.anchor:Hide()
							end
						end
						return CooldownTimers.db.profile.announce.locked
					end,
					set = function(v)
						CooldownTimers.db.profile.announce.locked = v
					end,
				},
				editspacer = {
					type = 'header',
					name = " ",
					order = 3,
				},
				message = {
					type = 'text',
					name = L["Display String"],
					desc = L["What you want the announce to say. Use %s for the skill name."],
					usage = "",
					order = 4,
					wfHidden = function()
						return not CooldownTimers.db.profile.announce.enabled
					end, 
					get = function()
						return CooldownTimers.db.profile.announce.announceString
					end,
					set = function(s)
						CooldownTimers.db.profile.announce.announceString = s
					end,
				},
				sliderspacer = {
					type = 'header',
					name = " ",
					order = 5,
				},
				fadetime = {
					type = 'range',
					name = L["Fade Time"],
					desc = L["How long until the announce begins fading."],
					order = 6,
					wfHidden = function()
						return not CooldownTimers.db.profile.announce.enabled
					end,
					min = 0,
					max = 10,
					step = 0.5,
					get = function()
						return CooldownTimers.db.profile.announce.fade
					end,
					set = function(v)
						CooldownTimers.db.profile.announce.fade = v
					end,
				},
				scale = {
					type = 'range',
					name = L["Announce Scale"]..L["\nSee the readme for important notes about this option."],
					desc = L["Scale of the announce text."],
					order = 7,
					wfHidden = function()
						return not CooldownTimers.db.profile.announce.enabled
					end,
					min = 0.5,
					max = 2,
					step = 0.1,
					get = function()
						return CooldownTimers.db.profile.announce.scale
					end,
					set = function(v)
						CooldownTimers.db.profile.announce.scale = v
					end,
				},
			},
		},
		wfskills = {
			type = 'group',
			name = L["Skill Cooldowns"],
			desc = L["Sets the settings for individual cooldowns. Enable/Disable cooldowns here."],
			order = 500,
			cmdHidden = true,
			args = {
				name = {
					type = 'text',
					name = L["Skill Name"],
					desc = L["Name of the skill you would like to change settings for"],
					order = 2,
					get = function()
						if not next(CooldownTimers.Options.args.wfskills.args.name.validate) then
							for k in pairs(CooldownTimers.db.class.cooldowns) do
								tinsert(CooldownTimers.Options.args.wfskills.args.name.validate, k)
							end
						end
						return selectedskill
					end,
					set = function(v)
						selectedskill = v
					end,
					validate = {},
				},
				enabled = {
					type = 'toggle',
					name = L["Enabled"],
					desc = L["Enables and disables tracking of individual skill cooldowns."],
					order = 3,
					wfHidden = function()
						return (selectedskill == "")
					end,
					get = function()
						if selectedskill ~= "" and selectedskill ~= nil then
							return not CooldownTimers.db.class.cooldowns[selectedskill].disabled
						end
					end,
					set = function(v)
						CooldownTimers.db.class.cooldowns[selectedskill].disabled = not v
					end,
				},
				dropdownspacer = {
					type = 'header',
					name = " ",
					order = 4,
				},
				group = {
					type = 'text',
					name = L["Group"],
					desc = L["Cooldown group to put selected skill in"],
					order = 5,
					wfHidden = function()
						return (selectedskill == "")
					end,
					disabled = function()
						if selectedskill ~= "" and selectedskill ~= nil then
							return CooldownTimers.db.class.cooldowns[selectedskill].disabled
						end
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfskills.args.group.validate) then
							for k,v in pairs(CooldownTimers.db.profile.groups) do
								if not v.disabled then
									tinsert(CooldownTimers.Options.args.wfskills.args.group.validate, k)
								end
							end
						end
						return CooldownTimers.db.class.cooldowns[selectedskill].group
					end,
					set = function(v)
						CooldownTimers.db.class.cooldowns[selectedskill].group = v
					end,
					validate = {},
				},
				texture = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Sets the status bar texture."],
					order = 6,
					wfHidden = function()
						return (selectedskill == "")
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfskills.args.texture.validate) then
							tinsert(CooldownTimers.Options.args.wfskills.args.texture.validate, L["Use Default"])
							for _,v in pairs(sm:List("statusbar")) do
								tinsert(CooldownTimers.Options.args.wfskills.args.texture.validate, v)
							end
						end
						if CooldownTimers.db.class.cooldowns[selectedskill].texture == nil then
							return L["Use Default"]
						else
							return CooldownTimers.db.class.cooldowns[selectedskill].texture
						end
					end,
					set = function(v)
						if v == L["Use Default"] then
							CooldownTimers.db.class.cooldowns[selectedskill].texture = nil
						else
							CooldownTimers.db.class.cooldowns[selectedskill].texture = v
						end
					end,
					validate = {},
				},
				colors = {
					type = 'text',
					name = L["Colors"],
					desc = L["Sets the fade colors. Needs at least two, but will take as many colors as you want.\n"]..L["Usage: <color1> <color2> ..."],
					wfHidden = function()
						return (selectedskill == "")
					end,
					disabled = function()
						if selectedskill ~= "" and selectedskill ~= nil then
							return CooldownTimers.db.class.cooldowns[selectedskill].disabled
						end
					end,
					order = 7,
					get = function()
						local colors
						if CooldownTimers.db.class.cooldowns[selectedskill].colors then
							for k,v in pairs(CooldownTimers.db.class.cooldowns[selectedskill].colors) do
								if not colors then
									colors = v
								else 
									colors = colors.." "..v
								end
							end
						else
							colors = ""
						end
						if colors == "" then
							return L["Use Default"]
						else
							return colors
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.class.cooldowns[selectedskill].colors = nil
						else
							CooldownTimers.db.class.cooldowns[selectedskill].colors = {}
							for color in string.gmatch(s, "%a+") do
								table.insert(CooldownTimers.db.class.cooldowns[selectedskill].colors, color)
							end
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%a+ %a+"))
					end,
					usage = L["Usage: <color1> <color2> ..."],
				},
				fadetime = {
					type = 'text',
					name = L["Custom "]..L["Fade Time"],
					desc = L["Sets how long bars take to fade after the bar completes.\n"]..L["Usage: <fadetime> (in seconds)"],
					wfHidden = function()
						return (selectedskill == "")
					end,
					disabled = function()
						if selectedskill ~= "" and selectedskill ~= nil then
							return CooldownTimers.db.class.cooldowns[selectedskill].disabled
						end
					end,
					order = 8,
					get = function()
						if CooldownTimers.db.class.cooldowns[selectedskill].fade == nil then
							return L["Use Default"]
						else
							return CooldownTimers.db.class.cooldowns[selectedskill].fade
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.class.cooldowns[selectedskill].fade = nil
						else
							CooldownTimers.db.class.cooldowns[selectedskill].fade = tonumber(s)
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%d+%.?%d*"))
					end,
					usage = L["Usage: <fadetime> (in seconds)"],
				},
				newcondense = {
					type = 'text',
					name = L["New Condense Group"],
					desc = L["Create a new Condense Group with\nthis as a member"],
					wfHidden = function()
						return (selectedskill == "")
					end,
					order = 9,
					get = function()
						return
					end,
					set = function(s)
						CooldownTimers.db.class.cooldowns[selectedskill] = nil
						CooldownTimers.db.class.skillgroups[selectedskill] = s
						CooldownTimers:UpdateData()
					end,
					usage = L["<group name> (Numbers are not allowed, and make sure the group doesn't already exist)"],
				},
				condensegroup = {
					type = 'text',
					name = L["Condense Group"],
					desc = L["Select a group to condense the selected skill into\n(e.g. for shared cooldowns)"],
					wfHidden = function()
						return (selectedskill == "")
					end,
					order = 10,
					get = function()
						CooldownTimers.groups = {}
						for _,v in pairs(CooldownTimers.db.class.skillgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						for _,v in pairs(CooldownTimers.db.profile.itemgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						if not next(CooldownTimers.Options.args.wfskills.args.condensegroup.validate) then
							for k in pairs(CooldownTimers.groups) do
								tinsert(CooldownTimers.Options.args.wfskills.args.condensegroup.validate, k)
							end
						end
						return
					end,
					set = function(v)
						CooldownTimers.db.class.cooldowns[selectedskill] = nil
						CooldownTimers.db.class.skillgroups[selectedskill] = v
						CooldownTimers:UpdateData()
					end,
					validate = {},
				},
			},
		},
		wfitems = {
			type = 'group',
			name = L["Item Cooldowns"],
			desc = L["Sets the settings for individual cooldowns. Enable/Disable cooldowns here."],
			order = 600,
			cmdHidden = true,
			disabled = function()
				return not next(CooldownTimers.db.profile.itemcooldowns)
			end,
			args = {
				name = {
					type = 'text',
					name = L["Item Name"],
					desc = L["Name of the item you would like to change settings for"],
					order = 2,
					get = function()
						if not next(CooldownTimers.Options.args.wfitems.args.name.validate) then
							for k in pairs(CooldownTimers.db.profile.itemcooldowns) do
								tinsert(CooldownTimers.Options.args.wfitems.args.name.validate, k)
							end
						end
						return selecteditem
					end,
					set = function(v)
						selecteditem = v
					end,
					validate = {},
				},
				enabled = {
					type = 'toggle',
					name = L["Enabled"],
					desc = L["Enables and disables tracking of individual item cooldowns."],
					order = 3,
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					get = function()
						if selecteditem ~= "" and selecteditem ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
							return (not CooldownTimers.db.profile.itemcooldowns[selecteditem].disabled)
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.itemcooldowns[selecteditem].disabled = not v
					end,
				},
				dropdownspacer = {
					type = 'header',
					name = " ",
					order = 4,
				},
				group = {
					type = 'text',
					name = L["Group"],
					desc = L["Cooldown group to put selected item in"],
					order = 5,
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					disabled = function()
						if selecteditem ~= "" and selecteditem ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
							return (CooldownTimers.db.profile.itemcooldowns[selecteditem].disabled)
						end
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfitems.args.group.validate) then
							for k,v in pairs(CooldownTimers.db.profile.groups) do
								if not v.disabled then
									tinsert(CooldownTimers.Options.args.wfitems.args.group.validate, k)
								end
							end
						end
						if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
							return CooldownTimers.db.profile.itemcooldowns[selecteditem].group
						end
					end,
					set = function(v)
						CooldownTimers.db.profile.itemcooldowns[selecteditem].group = v
					end,
					validate = {},
				},
				texture = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Sets the status bar texture."],
					order = 6,
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfitems.args.texture.validate) then
							tinsert(CooldownTimers.Options.args.wfitems.args.texture.validate, L["Use Default"])
							for _,v in pairs(sm:List("statusbar")) do
								tinsert(CooldownTimers.Options.args.wfitems.args.texture.validate, v)
							end
						end
						if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem].texture == nil then
							return L["Use Default"]
						else
							if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
								return CooldownTimers.db.profile.itemcooldowns[selecteditem].texture
							end
						end
					end,
					set = function(v)
						if v == L["Use Default"] then
							CooldownTimers.db.profile.itemcooldowns[selecteditem].texture = nil
						else
							CooldownTimers.db.profile.itemcooldowns[selecteditem].texture = v
						end
					end,
					validate = {},
				},
				colors = {
					type = 'text',
					name = L["Colors"],
					desc = L["Sets the fade colors. Needs at least two, but will take as many colors as you want.\n"]..L["Usage: <color1> <color2> ..."],
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					disabled = function()
						if selecteditem ~= "" and selecteditem ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
							return (CooldownTimers.db.profile.itemcooldowns[selecteditem].disabled)
						end
					end,
					order = 7,
					get = function()
						local colors
						if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem].colors then
							for k,v in pairs(CooldownTimers.db.profile.itemcooldowns[selecteditem].colors) do
								if not colors then
									colors = v
								else 
									colors = colors.." "..v
								end
							end
						else
							colors = ""
						end
						if colors == "" then
							return L["Use Default"]
						else
							return colors
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.profile.itemcooldowns[selecteditem].colors = nil
						else
							CooldownTimers.db.profile.itemcooldowns[selecteditem].colors = {}
							for color in string.gmatch(s, "%a+") do
								table.insert(CooldownTimers.db.profile.itemcooldowns[selecteditem].colors, color)
							end
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%a+ %a+"))
					end,
					usage = L["Usage: <color1> <color2> ..."],
				},
				fadetime = {
					type = 'text',
					name = L["Custom "]..L["Fade Time"],
					desc = L["Sets how long bars take to fade after the bar completes.\n"]..L["Usage: <fadetime> (in seconds)"],
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					disabled = function()
						if selecteditem ~= "" and selecteditem ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
							return (CooldownTimers.db.profile.itemcooldowns[selecteditem].disabled)
						end
					end,
					order = 8,
					get = function()
						if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil and CooldownTimers.db.profile.itemcooldowns[selecteditem].fade == nil then
							return L["Use Default"]
						else
							if CooldownTimers.db.profile.itemcooldowns[selecteditem] ~= nil then
								return CooldownTimers.db.profile.itemcooldowns[selecteditem].fade
							end
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.profile.itemcooldowns[selecteditem].fade = nil
						else
							CooldownTimers.db.profile.itemcooldowns[selecteditem].fade = tonumber(s)
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%d+%.?%d*"))
					end,
					usage = L["Usage: <fadetime> (in seconds)"],
				},
				newcondense = {
					type = 'text',
					name = L["New Condense Group"],
					desc = L["Create a new Condense Group with\nthis as a member"],
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					order = 9,
					get = function()
						return
					end,
					set = function(s)
						CooldownTimers.db.profile.itemcooldowns[selecteditem] = nil
						CooldownTimers.db.profile.itemgroups[selecteditem] = s
						CooldownTimers:UpdateData()
					end,
					usage = "<group name> (Numbers are not allowed, and make sure the group doesn't already exist)",
				},
				condensegroup = {
					type = 'text',
					name = L["Condense Group"],
					desc = L["Select a group to condense the selected item into\n(e.g. for shared cooldowns)"],
					wfHidden = function()
						return (selecteditem == "") or (CooldownTimers.db.profile.itemcooldowns[selecteditem] == nil)
					end,
					order = 10,
					get = function()
						CooldownTimers.groups = {}
						for _,v in pairs(CooldownTimers.db.class.skillgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						for _,v in pairs(CooldownTimers.db.profile.itemgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						if not next(CooldownTimers.Options.args.wfitems.args.condensegroup.validate) then
							for k in pairs(CooldownTimers.groups) do
								tinsert(CooldownTimers.Options.args.wfitems.args.condensegroup.validate, k)
							end
						end
						return
					end,
					set = function(v)
						CooldownTimers.db.profile.itemcooldowns[selecteditem] = nil
						CooldownTimers.db.profile.itemgroups[selecteditem] = v
						CooldownTimers:UpdateData()
					end,
					validate = {},
				},
			},
		},
		wfpetcds = {
			type = 'group',
			name = L["Pet Cooldowns"],
			desc = L["Sets the settings for individual cooldowns. Enable/Disable cooldowns here."],
			order = 700,
			cmdHidden = true,
			disabled = function()
				return not next(CooldownTimers.db.char.petcooldowns)
			end,
			args = {
				name = {
					type = 'text',
					name = L["Pet Cooldown Name"],
					desc = L["Name of the pet cooldown you would like to change settings for"],
					order = 2,
					get = function()
						if not next(CooldownTimers.Options.args.wfpetcds.args.name.validate) then
							for k in pairs(CooldownTimers.db.char.petcooldowns) do
								tinsert(CooldownTimers.Options.args.wfpetcds.args.name.validate, k)
							end
						end
						return selectedpetcd
					end,
					set = function(v)
						selectedpetcd = v
					end,
					validate = {},
				},
				enabled = {
					type = 'toggle',
					name = L["Enabled"],
					desc = L["Enables and disables tracking of individual skill cooldowns."],
					order = 3,
					wfHidden = function()
						return (selectedpetcd == "")
					end,
					get = function()
						if selectedpetcd ~= "" and selectedpetcd ~= nil then
							return not CooldownTimers.db.char.petcooldowns[selectedpetcd].disabled
						end
					end,
					set = function(v)
						CooldownTimers.db.char.petcooldowns[selectedpetcd].disabled = not v
					end,
				},
				dropdownspacer = {
					type = 'header',
					name = " ",
					order = 4,
				},
				group = {
					type = 'text',
					name = L["Group"],
					desc = L["Cooldown group to put selected pet skill in"],
					order = 5,
					wfHidden = function()
						return (selectedpetcd == "")
					end,
					disabled = function()
						if selectedpetcd ~= "" and selectedpetcd ~= nil then
							return (CooldownTimers.db.char.petcooldowns[selectedpetcd].disabled)
						end
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfpetcds.args.group.validate) then
							for k,v in pairs(CooldownTimers.db.profile.groups) do
								if not v.disabled then
									tinsert(CooldownTimers.Options.args.wfpetcds.args.group.validate, k)
								end
							end
						end
						return CooldownTimers.db.char.petcooldowns[selectedpetcd].group
					end,
					set = function(v)
						CooldownTimers.db.char.petcooldowns[selectedpetcd].group = v
					end,
					validate = {},
				},
				texture = {
					type = 'text',
					name = L["Bar Texture"],
					desc = L["Sets the status bar texture."],
					order = 6,
					wfHidden = function()
						return (selectedpetcd == "")
					end,
					get = function()
						if not next(CooldownTimers.Options.args.wfpetcds.args.texture.validate) then
							tinsert(CooldownTimers.Options.args.wfpetcds.args.texture.validate, L["Use Default"])
							for _,v in pairs(sm:List("statusbar")) do
								tinsert(CooldownTimers.Options.args.wfpetcds.args.texture.validate, v)
							end
						end
						if CooldownTimers.db.char.petcooldowns[selectedpetcd].texture == nil then
							return L["Use Default"]
						else
							return CooldownTimers.db.char.petcooldowns[selectedpetcd].texture
						end
					end,
					set = function(v)
						if v == L["Use Default"] then
							CooldownTimers.db.char.petcooldowns[selectedpetcd].texture = nil
						else
							CooldownTimers.db.char.petcooldowns[selectedpetcd].texture = v
						end
					end,
					validate = {},
				},
				colors = {
					type = 'text',
					name = L["Colors"],
					desc = L["Sets the fade colors. Needs at least two, but will take as many colors as you want.\n"]..L["Usage: <color1> <color2> ..."],
					wfHidden = function()
						return (selectedpetcd == "")
					end,
					disabled = function()
						if selectedpetcd ~= "" and selectedpetcd ~= nil then
							return (CooldownTimers.db.char.petcooldowns[selectedpetcd].disabled)
						end
					end,
					order = 7,
					get = function()
						local colors
						if CooldownTimers.db.char.petcooldowns[selectedpetcd].colors then
							for k,v in pairs(CooldownTimers.db.char.petcooldowns[selectedpetcd].colors) do
								if not colors then
									colors = v
								else 
									colors = colors.." "..v
								end
							end
						else
							colors = ""
						end
						if colors == "" then
							return L["Use Default"]
						else
							return colors
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.char.petcooldowns[selectedpetcd].colors = nil
						else
							CooldownTimers.db.char.petcooldowns[selectedpetcd].colors = {}
							for color in string.gmatch(s, "%a+") do
								table.insert(CooldownTimers.db.char.petcooldowns[selectedpetcd].colors, color)
							end
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%a+ %a+"))
					end,
					usage = L["Usage: <color1> <color2> ..."],
				},
				fadetime = {
					type = 'text',
					name = L["Custom "]..L["Fade Time"],
					desc = L["Sets how long bars take to fade after the bar completes.\n"]..L["Usage: <fadetime> (in seconds)"],
					wfHidden = function()
						return (selectedpetcd == "")
					end,
					disabled = function()
						if selectedpetcd ~= "" and selectedpetcd ~= nil then
							return CooldownTimers.db.char.petcooldowns[selectedpetcd].disabled
						end
					end,
					order = 8,
					get = function()
						if CooldownTimers.db.char.petcooldowns[selectedpetcd].fade == nil then
							return L["Use Default"]
						else
							return CooldownTimers.db.char.petcooldowns[selectedpetcd].fade
						end
					end,
					set = function(s)
						if s == L["Use Default"] then
							CooldownTimers.db.char.petcooldowns[selectedpetcd].fade = nil
						else
							CooldownTimers.db.char.petcooldowns[selectedpetcd].fade = tonumber(s)
						end
					end,
					validate = function(s)
						return (string.match(s, L["Use Default"])) or (string.find(s, "^%d+%.?%d*"))
					end,
					usage = L["Usage: <fadetime> (in seconds)"],
				},
			},
		},
		wfcondensegrps = {
			type = 'group',
			name = L["Condense Groups"],
			desc = L["Use this to condense multiple skills/items with the same cooldown into a single bar."],
			order = 800,
			cmdHidden = true,
      disabled = function()
        return CooldownTimers.db.profile.autogroup;
      end,
			args = {
				group = {
					type = 'text',
					name = L["Condense Group"],
					desc = L["Select a condense group to remove an item from"],
					order = 2,
					get = function()
						CooldownTimers.groups = {}
						for _,v in pairs(CooldownTimers.db.class.skillgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						for _,v in pairs(CooldownTimers.db.profile.itemgroups) do
							if not CooldownTimers.groups[v] then
								CooldownTimers.groups[v] = {}
							end
						end
						if not next(CooldownTimers.Options.args.wfcondensegrps.args.group.validate) then
							for k in pairs(CooldownTimers.groups) do
								tinsert(CooldownTimers.Options.args.wfcondensegrps.args.group.validate, k)
							end
						end
						return selectedcondensegrp
					end,
					set = function(v)
						selectedcondensegrp = v
						CooldownTimers.Options.args.wfcondensegrps.args.cooldown.validate = {}
					end,
					validate = {},
				},
				cooldown = {
					type = 'text',
					name = L["Cooldown"],
					desc = L["Select an ability or item to remove from this condense group"],
					wfHidden = function()
						return (selectedcondensegrp == "")
					end,
					order = 3,
					get = function()
						if not next(CooldownTimers.Options.args.wfcondensegrps.args.cooldown.validate) then
							for k, v in pairs(CooldownTimers.db.class.skillgroups) do
								if v == selectedcondensegrp then
									tinsert(CooldownTimers.Options.args.wfcondensegrps.args.cooldown.validate, k)
								end
							end
							for k, v in pairs(CooldownTimers.db.profile.itemgroups) do
								if v == selectedcondensegrp then
									tinsert(CooldownTimers.Options.args.wfcondensegrps.args.cooldown.validate, k)
								end
							end
						end
						return selectedcooldown
					end,
					set = function(v)
						selectedcooldown = v
					end,
					validate = {},
				},
				remove = {
					type = 'execute',
					name = L["Remove"],
					desc = L["Remove this item from the condense group."],
					wfHidden = function()
						return (selectedcooldown == "")
					end,
					order = 4,
					func = function()
						CooldownTimers.db.profile.itemgroups[selectedcooldown] = nil
						CooldownTimers.db.class.skillgroups[selectedcooldown] = nil
						CooldownTimers:PopulateCooldowns()
						CooldownTimers:UpdateData()
					end,
				},
			},
		},
	},
}

CooldownTimers:RegisterChatCommand({"/cooldowntimers", "/cdt"}, CooldownTimers.Options)