 --[[ 

 Name: ClosetGnomesZones

 Description: ClosetGnomeBigWigs changed to support Zones.  Basically a straight rip of Rabbit's code 
 so he gets 99.5% of the credit.

 Updates:

    - moved most of the configuration to a submenu under ClostGnome
    - added mount checking code again with a toggleable option :)

 ]]

if not ClosetGnome then return end

-------------------------------------------------------------------------------
-- Locals                                                                    --
-------------------------------------------------------------------------------

local L = AceLibrary("AceLocale-2.2"):new("ClosetGnomeZone")
local queuedSet = nil

-------------------------------------------------------------------------------
-- Addon declaration                                                         --
-------------------------------------------------------------------------------

ClosetGnomeZone = AceLibrary("AceAddon-2.0"):new("AceConsole-2.0", "AceDB-2.0", "AceEvent-2.0")
ClosetGnomeZone:RegisterDB("ClosetGnomeZonePCDB")
ClosetGnomeZone:RegisterDefaults("char", {
	confirmEquip = true,
	enabled = true,
	mounted = false,
	assignments = {},
})

ClosetGnomeZone.Opts = {
	type = "group",
	name = L["Zone"],
	desc = L["Makes your ClosetGnome aware of the zones in this dangerous world."],
	order = 103.1,
	args = {
		enabled = {
			type = "toggle",
			name = L["Enabled"],
			desc = L["Enable set switching when changing zones"],
			get = function() return ClosetGnomeZone.db.char.enabled end,
			set = function(v) ClosetGnomeZone.db.char.enabled = v end,
			order = 100,
		},
		spacer = { type = "header", order = 110 },
		list = {
			type = 'group',
			name = L["Assignments"],
			desc = L["List the current zone-to-set assignments."],
			args = {},
			order = 140,
			disabled = function() return next(ClosetGnomeZone.db.char.assignments) == nil end
		},
		spacer2 = { type = "header", order = 150 },
		confirm = {
			type = "toggle",
			name = L["Confirm Autoequip"],
			desc = L["Open a confirmation dialog before autoequipping a zone set."],
			get = function() return ClosetGnomeZone.db.char.confirmEquip end,
			set = function(v) ClosetGnomeZone.db.char.confirmEquip = v end,
			order = 160,
		},
		mounted = {
			type = "toggle",
			name = L["Switch while Mounted"],
			desc = L["Switch sets while mounted"],
			get = function() return ClosetGnomeZone.db.char.mounted end,
			set = function(v) ClosetGnomeZone.db.char.mounted = v end,
			order = 170,
		},
	},
}
-------------------------------------------------------------------------------
-- Initialization                                                            --
-------------------------------------------------------------------------------

function ClosetGnomeZone:OnInitialize()
	ClosetGnome.OnMenuRequest.args.zone = ClosetGnomeZone.Opts
end

function ClosetGnomeZone:OnEnable()
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA", "ZoneChangedNewArea")
	self:RegisterEvent("ClosetGnome_AddSet", "Update")
	self:RegisterEvent("ClosetGnome_DeleteSet", "DeleteSet")

	StaticPopupDialogs["ClosetGnomeZoneConfirm"] = {
		text = L["You have entered %s. Do you want to equip %s?\n\nYou can disable this confirmation dialog to autoequip with /cg zone confirm."],
		button1 = L["Equip"],
		button2 = L["Cancel"],
		sound = "levelup2",
		whileDead = 0,
		hideOnEscape = 1,
		timeout = 0,
		OnAccept = function()
			ClosetGnome:WearSet(queuedSet)
			if CGFu then CGFu:Update() end
		end,
	}
	self:Update()
end

-------------------------------------------------------------------------------
-- Command handlers                                                          --
-------------------------------------------------------------------------------

function ClosetGnomeZone:AddAssignment(set)
	local zone = GetRealZoneText()
	if zone and set then
		if set == L["None"] then
			self:DeleteAssignment(zone)
		else
			self.db.char.assignments[zone] = set
		end
	end
	ClosetGnomeZone:Update()
end

function ClosetGnomeZone:DeleteSet(set)
	zone = GetRealZoneText()
	if zone and set then
		for k,v in pairs(self.db.char.assignments) do
			if v == set then
				self.db.char.assignments[k] = nil
				break
			end
		end
	end
	self:Update()
end

function ClosetGnomeZone:DeleteAssignment(zone)
	if zone and self.db.char.assignments[zone] then
		self.db.char.assignments[zone] = nil
	end
	ClosetGnomeZone:Update()
end

function ClosetGnomeZone:DefaultAssignment(v)
	self.db.char.assignments[L["Default"]] = v ~= L["None"] and v or nil
	ClosetGnomeZone:Update()
end

function ClosetGnomeZone:Update()
	local zone = GetRealZoneText()
	if zone == nil or zone == "" then
		self:ScheduleEvent(self.Update,5,self)
		return
	end

	local sets = {}
	for k, v in pairs(ClosetGnome.db.char.set) do
		table.insert(sets, k)
	end
	table.sort(sets)
	table.insert(sets, 1, L["None"])

	ClosetGnome.OnMenuRequest.args.zone.args.assign = {
		order = 120,
		type = 'text',
		desc = string.format(L["Set to use for %s"], zone),
		name = string.format(L["Assign Set to %s"], zone),
		validate = sets,
		get = function() return self.db.char.assignments[zone] or L["None"] end,
		set = function(v) ClosetGnomeZone:AddAssignment(v) end,
	}

	ClosetGnome.OnMenuRequest.args.zone.args.default = {
		order = 130,
		type = 'text',
		desc = L["Set to use for unassigned zones"],
		name = L["Default Set"],
		validate = sets,
		get = function() return self.db.char.assignments[L["Default"]] or L["None"] end,
		set = function(v) ClosetGnomeZone:DefaultAssignment(v) end,
	}

	ClosetGnome.OnMenuRequest.args.zone.args.list.args = {}
	for k, v in pairs(self.db.char.assignments) do
		if ClosetGnome.db.char.set[v] then
			local menuKey = k:gsub(" ", "")
			ClosetGnome.OnMenuRequest.args.zone.args.list.args[menuKey] = {
				type = "execute",
				name = k.." - "..v,
				desc = L["Ctrl-Click to Delete"],
				func = function()
					if IsControlKeyDown() then
						ClosetGnomeZone:DeleteAssignment(k)
					end
				end,
			}
		else
			self.db.char.assignments[k] = nil
		end
	end
end

-------------------------------------------------------------------------------
-- Events                                                                    --
-------------------------------------------------------------------------------

local function mountCheck()
	if not IsMounted() and not IsFlying() then
		ClosetGnomeZone:CancelScheduledEvent("CGZMountCheck")
		ClosetGnomeZone:ZoneChangedNewArea()
	end
end

function ClosetGnomeZone:ZoneChangedNewArea()
	self:Update()

	local mounted = IsMounted() or IsFlying()
	if UnitIsDeadOrGhost("player")
	or not self.db.char.enabled
	or (not self.db.char.mounted and mounted) then
		if mounted then
			self:ScheduleRepeatingEvent("CGZMountCheck", 5, mountCheck)
		end
		return
	end

	local zone = GetRealZoneText()

	if zone == nil or zone == "" then
		-- zone hasn't been loaded yet, try again in 5 secs.
		self:ScheduleEvent(self.ZoneChangedNewArea,5,self)
		-- self:Print("Unable to determine zone - retrying in 5 secs")
		return
	end

	local setName = self.db.char.assignments[zone] or self.db.char.assignments[L["Default"]]

	if not setName or ClosetGnome:IsSetFullyEquipped(setName) then return end

	if ClosetGnome:HasSet(setName) then
		if self.db.char.confirmEquip then
			queuedSet = setName
			StaticPopup_Show("ClosetGnomeZoneConfirm", "|cffd9d919"..zone.."|r", "|cffd9d919"..setName.."|r")
		else
			self:Print(string.format(L["Entered %s, equipping set %s."], zone, setName))
			ClosetGnome:WearSet(setName)
			if CGFu then CGFu:Update() end
		end
	end
end

