--[[-------------------------------------------------------------------------
-- ClosetGnome_Switcher - Swaps your set to match buff names. Can be easily
--   used to enable special outfits for stances and shapeshift forms.
--
--  by PProvost
--
--  Last Modified: $Date: 2008-04-10 14:46:29 -0400 (Thu, 10 Apr 2008) $
--  Revision: $Revision: 68962 $
--
-- TODO:
--  * N/A
--
--  Known Bugs:
--  * N/A
--]]-------------------------------------------------------------------------


--[[ Local Variables and Constants ]]
local MAJOR_VERSION = "1.0"
local MINOR_VERSION = tonumber((string.gsub("$Revision: 68962 $", "^.-(%d+).-$", "%1")))

local buffRegister = {}
local lastForm = ""
local currSet = nil
local lastSet = nil

if type(ClosetGnomeSwitcherDBPC) ~= "table" then
	ClosetGnomeSwitcherDBPC = {}
	ClosetGnomeSwitcherDBPC.enabled = true
end

local _G = getfenv(0)

--[[ Utility Libraries ]]
local L = AceLibrary("AceLocale-2.2"):new("ClosetGnomeSwitcher")

local SEEA = AceLibrary("SpecialEvents-Aura-2.0")
local drink = GetSpellInfo(30024) -- Just one of the few hundred drink spells
local food = GetSpellInfo(6410) -- Just one of the few hundred food spells

--[[ Initialize Addon instance ]]
ClosetGnomeSwitcher = AceLibrary("AceAddon-2.0"):new("AceEvent-2.0", "AceConsole-2.0", "AceModuleCore-2.0", "AceHook-2.1")

function ClosetGnomeSwitcher:OnInitialize()
	self.version = MAJOR_VERSION .. "." .. MINOR_VERSION
	self.sets = {}

	self.Opts = {
		handler = self,
		name = L["Switcher"],
		desc = L["Tells your closet gnome when to switch outfits for you."],
		type = "group",
		order = 106,
		pass = true,
		get = function(key)
			if key == "enabled" then
				return ClosetGnomeSwitcherDBPC[key]
			else
				return ClosetGnomeSwitcherDBPC[key] or "None"
			end
		end,
		set = function(key, value)
			if key == "enabled" then
				self:ToggleActive(value)
			else
				ClosetGnomeSwitcherDBPC[key] = value ~= "None" and value or nil
			end
		end,
		args = {
			defaultSet = {
				type = "text",
				name = L["Default"],
				desc = L["The default set to use when none of the others are applied."],
				usage = "<setName>",
				validate = self.sets,
				disabled = "ShouldDisable",
			},
			logoutSet = {
				type = "text",
				name = L["Logout"],
				desc = L["Set to apply when logging out."],
				usage = "<setName>",
				validate = self.sets,
				disabled = "ShouldDisable",
			},
			loginSet = {
				type = "text",
				name = L["Login"],
				desc = L["Set to apply when logging in."],
				usage = "<setName>",
				validate = self.sets,
				disabled = "ShouldDisable",
			},
			[food] = {
				type = "text",
				name = food,
				desc = L["The set to use when eating."],
				usage = "<setName>",
				validate = self.sets,
				disabled = "ShouldDisable",
			},
			[drink] = {
				type = "text",
				name = drink,
				desc = L["The set to use when drinking."],
				usage = "<setName>",
				validate = self.sets,
				disabled = "ShouldDisable",
			},
			enabled = {
				type = "toggle",
				name = L["Enabled"],
				desc = L["Enable/disable ClosetGnome_Switcher."],
				order = -1,
			},
		}
	}

	ClosetGnome.OnMenuRequest.args.switcher = self.Opts
end

--[[function ClosetGnomeSwitcher:UPDATE_SHAPESHIFT_FORM()
	self:Print("Form changed: ".. self:GetCurrentShapeshiftForm())
end]]

function ClosetGnomeSwitcher:OnEnable()
	if not ClosetGnomeSwitcherDBPC.enabled then return end
	--self:RegisterBucketEvent("UPDATE_SHAPESHIFT_FORM", 0.2)

	--self:RegisterEvent("UPDATE_SHAPESHIFT_FORM")

	-- For warrior stance changes:
	self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")

	self:RegisterEvent("ClosetGnome_DeleteSet", "UpdateSet")
	self:RegisterEvent("ClosetGnome_AddSet", "UpdateSet")
	self:RegisterEvent("ClosetGnome_WearSet")

	self:RegisterEvent("SpecialEvents_PlayerBuffGained")
	self:RegisterEvent("SpecialEvents_PlayerBuffLost")

	if AceLibrary("AceEvent-2.0"):IsFullyInitialized() then
		self:AceEvent_FullyInitialized()
	end
	self:RegisterEvent("AceEvent_FullyInitialized")

	if type(_G["Logout"]) == "function" then
		self:Hook("Logout", "OnLogout", true)
	end
	if type(_G["Quit"]) == "function" then
		self:Hook("Quit", "OnQuit", true)
	end

	self:RegisterForBuff(self, drink)
	self:RegisterForBuff(self, food)
end

function ClosetGnomeSwitcher:ToggleActive(value)
	if value and not ClosetGnomeSwitcherDBPC.enabled then
		ClosetGnomeSwitcherDBPC.enabled = value

		self:OnEnable()
	elseif ClosetGnomeSwitcherDBPC.enabled and not value then
		ClosetGnomeSwitcherDBPC.enabled = value

		self:UnregisterAllEvents()
		self:UnhookAll()
	end
end

function ClosetGnomeSwitcher:IsActive()
	return ClosetGnomeSwitcherDBPC.enabled
end

function ClosetGnomeSwitcher:BuffGained(name)
	return ClosetGnomeSwitcherDBPC[name]
end

function ClosetGnomeSwitcher:BuffLost(name)
	if (name == food and self:HasSet(food)) or (name == drink and self:HasSet(drink)) then
		return true
	end
end

function ClosetGnomeSwitcher:HasSet(key)
	return ClosetGnomeSwitcherDBPC[key] and ClosetGnomeSwitcherDBPC[key] ~= "None" or nil
end

function ClosetGnomeSwitcher:ShouldDisable()
	return not self:IsActive() or #ClosetGnomeSwitcher.sets == 0
end

local function WearSet(setName)
	if ClosetGnome:HasSet(setName) then
		return ClosetGnome:WearSet(setName)
	end
	return false
end

function ClosetGnomeSwitcher:OnQuit(...)
	if ClosetGnomeSwitcherDBPC and ClosetGnomeSwitcherDBPC.logoutSet then
		WearSet(ClosetGnomeSwitcherDBPC.logoutSet)
	end
	self.hooks["Quit"](...)
end

function ClosetGnomeSwitcher:OnLogout(...)
	if ClosetGnomeSwitcherDBPC and ClosetGnomeSwitcherDBPC.logoutSet then
		WearSet(ClosetGnomeSwitcherDBPC.logoutSet)
	end
	self.hooks["Logout"](...)
end

function ClosetGnomeSwitcher:RegisterForBuff(module, buff)
	if not buffRegister[buff] then buffRegister[buff] = {} end
	buffRegister[buff][module] = true
end

-- Maintain a sorted list of sets, preceded by a "None" option
function ClosetGnomeSwitcher:AceEvent_FullyInitialized()
	if self.setsPopulated then return end
	self.setsPopulated = true
	for k, v in pairs(ClosetGnome.db.char.set) do
		table.insert(self.sets, k)
	end
	table.sort(self.sets)
	table.insert(self.sets, 1, "None")

	if ClosetGnomeSwitcherDBPC and ClosetGnomeSwitcherDBPC.loginSet then
		WearSet(ClosetGnomeSwitcherDBPC.loginSet)
	end
end

function ClosetGnomeSwitcher:UpdateSet(name)
	if ClosetGnome:HasSet(name) then
		table.remove(self.sets, 1)
		table.insert(self.sets, name)
		table.sort(self.sets)
		table.insert(self.sets, 1, "None")
	else
		for k,v in ipairs(self.sets) do
			if v == name then
				table.remove(self.sets, k)
			end
		end
	end
end

-- hook into ClosetGnome:WearSet() since no CurrentSet() api exists
function ClosetGnomeSwitcher:ClosetGnome_WearSet(name)
	currSet = name
end

function ClosetGnomeSwitcher:GetCurrentShapeshiftForm()
	local numForms = GetNumShapeshiftForms()
	local name, active = "Humanoid", nil
	for i = 1, numForms do
		name, active = select(2, GetShapeshiftFormInfo(i))
		if active == 1 then return name end
	end
	return name
end

function ClosetGnomeSwitcher:COMBAT_LOG_EVENT_UNFILTERED(timestamp, eventtype, sourceguid, sourcename, sourceflags, destguid, destname, destflags, ...)
	if self[eventtype] and CombatLog_Object_IsA(destflags, COMBATLOG_FILTER_MINE) then
		-- We have a handler, and the event relates to us.
		self[eventtype](self, timestamp, eventtype, sourceguid, sourcename, sourceflags, destguid, destname, destflags, ...)
	end
end

-- On form change, we loop all the modules and see if anyone wants to equip a
-- set, and if noone does, we equip the default set.
function ClosetGnomeSwitcher:SPELL_AURA_APPLIED(timestamp, eventtype, sourceguid, sourcename, sourceflags, destguid, destname, destflags, spellid, spellname, spellschool, auratype)
	if auratype ~= "BUFF" or (UnitIsDeadOrGhost("player") == 1 and not UnitIsFeignDeath("player")) then return end
	local form = self:GetCurrentShapeshiftForm()
	if lastForm == form then return end
	lastForm = form

	local handled = nil
	for _, module in self:IterateModules() do
		if type(module.FormChanged) == "function" then
			handled = module:FormChanged(form)
			if handled then
				break
			end
		end
	end
	if not handled and ClosetGnomeSwitcherDBPC.defaultSet then
		WearSet(ClosetGnomeSwitcherDBPC.defaultSet)
	elseif type(handled) == "string" then
		WearSet(handled)
	end
end

-- Detect if a buff has a duration.
function ClosetGnomeSwitcher:IsTemporaryBuff(name)
	local i = SEEA:UnitHasBuff("player", name)
	if not i then return false end
	local pi, always = GetPlayerBuff(i, "HELPFUL")
	return (pi > 0 and always == 0)
end

-- On buff lost, we loop all the modules to see if any of them cared about this
-- buff. If anyone did, we equip the default set, or the set that was returned.
function ClosetGnomeSwitcher:SpecialEvents_PlayerBuffLost(name)
	if buffRegister[name] then
		if UnitIsDeadOrGhost("player") == 1 and not UnitIsFeignDeath("player") then return end
		local handled = false
		for module in pairs(buffRegister[name]) do
			if type(module.BuffLost) == "function" then
				handled = module:BuffLost(name)
				if handled then
					break
				end
			end
		end
		if type(handled) == "boolean" then
			if ClosetGnomeSwitcherDBPC.defaultSet then
				WearSet(ClosetGnomeSwitcherDBPC.defaultSet)
			elseif lastSet then
				WearSet(lastSet)
				lastSet = nil
			end
		elseif type(handled) == "string" then
			WearSet(handled)
		end
	end
end

-- On buff gains, we just loop the modules and see if any of them want to equip
-- a set. The first one that wants to gets to, and the rest of the modules are
-- ignored.
function ClosetGnomeSwitcher:SpecialEvents_PlayerBuffGained(name)
	if buffRegister[name] then
		if UnitIsDeadOrGhost("player") == 1 and not UnitIsFeignDeath("player") then return end
		local set = nil
		for module in pairs(buffRegister[name]) do
			if type(module.BuffGained) == "function" then
				set = module:BuffGained(name)
				if set then break end
			end
		end
		if type(set) == "string" then
			-- remember the last set only for buffs with durations
			if self:IsTemporaryBuff(name) then
				if not lastSet then
					lastSet = currSet
				end
			else
				lastSet = nil
			end
			WearSet(set)
		end
	end
end

