-------------------------------------------------------------------------------
-- Locals                                                                    --
-------------------------------------------------------------------------------

local L = AceLibrary("AceLocale-2.2"):new("ClosetGnomeBanker")
local CGL = AceLibrary("AceLocale-2.2"):new("ClosetGnome")

local bankAvailable = nil
local bankSlots = { -1, 5, 6, 7, 8, 9, 10, 11 }
local slotLocks = nil

-------------------------------------------------------------------------------
-- Localization                                                              --
-------------------------------------------------------------------------------

L:RegisterTranslations("enUS", function() return {
	["Banker"] = true,
	["Store or retrieve item sets from your bank."] = true,
	["Store"] = true,
	["Store a set in the bank."] = true,
	["Retrieve"] = true,
	["Retrieve a set from the bank."] = true,
	["Store the set %s in the bank."] = true,
	["Retrieve the set %s from the bank."] = true,
	["Keep"] = true,
	["Always keep the items for these sets in your inventory when flushing."] = true,
	["Flush!"] = true,
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = true,
	["Flush&Get!"] = true,
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = true,
	["Always keep the items for %s in your inventory."] = true,

	["Container"] = true,
} end)

L:RegisterTranslations("ruRU", function() return {
	["Banker"] = "Банкир",
	["Store or retrieve item sets from your bank."] = "Положить или забрать сет из Банка.",
	["Store"] = "Положить",
	["Store a set in the bank."] = "Положить сет в Банк.",
	["Retrieve"] = "Забрать",
	["Retrieve a set from the bank."] = "Забрать сет из Банка.",
	["Store the set %s in the bank."] = "Положить сет %s в Банк.",
	["Retrieve the set %s from the bank."] = "Забрать сет %s из Банка.",
	["Keep"] = "Оставить",
	["Always keep the items for these sets in your inventory when flushing."] = "Всегда оставлять вещи этих сетов в инвентаре при закидывании.",
	["Flush!"] = "Закинуть!",
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = "Закидывает все сеты из инвентаря в Банк, кроме тех, что вы пометили как 'Оставить'.",
	["Flush&Get!"] = "Закинуть и забрать!",
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = "Забирает все вещи из списка 'Оставить' и закидывает все остальные.",
	["Always keep the items for %s in your inventory."] = "Всегда оставлять вещи из %s в инвентаре.",

	["Container"] = "Сумка",
} end)

L:RegisterTranslations("frFR", function() return {
	["Banker"] = "Banquier",
	["Store or retrieve item sets from your bank."] = "Dépose ou retire des sets d'objets de votre banque.",
	["Store"] = "Dépôt",
	["Store a set in the bank."] = "Dépose un set dans la banque.",
	["Retrieve"] = "Retrait",
	["Retrieve a set from the bank."] = "Retire un set de la banque.",
	["Store the set %s in the bank."] = "Dépose le set %s dans la banque.",
	["Retrieve the set %s from the bank."] = "Retire le set %s de la banque.",
	["Keep"] = "Garder",
	["Always keep the items for these sets in your inventory when flushing."] = "Conserve toujours les objets de ces sets dans votre inventaire lors d'une vidange.",
	["Flush!"] = "Vidange !",
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = "Dépose tous les sets de votre inventaire vers la banque, sauf ceux que vous avez coché dans la liste à garder.",
	["Flush&Get!"] = "Vidange et Récupération !",
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = "Récupère les sets de la liste à garder dans votre inventaire et dépose tous les autres objets.",
	["Always keep the items for %s in your inventory."] = "Conserve toujours les objets pour %s dans votre inventaire.",

	["Container"] = "Conteneur",
} end)

L:RegisterTranslations("zhCN", function() return {
	["Banker"] = "银行",
	["Store or retrieve item sets from your bank."] = "在你的银行中存储或取出套装物品",
	["Store"] = "存入",
	["Store a set in the bank."] = "存入一套装备到银行中",
	["Retrieve"] = "取回",
	["Retrieve a set from the bank."] = "从银行中取回一套装备",
	["Store the set %s in the bank."] = "把套装 %s 存到银行中",
	["Retrieve the set %s from the bank."] = "从银行中取出套装 %s.",

	["Container"] = "容器",
} end)

L:RegisterTranslations("koKR", function() return {
	["Banker"] = "은행",
	["Store or retrieve item sets from your bank."] = "당신의 은행으로부터 아이템 세트를 보관하거나 회수합니다.",
	["Store"] = "보관",
	["Store a set in the bank."] = "세트를 은행에 보관합니다.",
	["Retrieve"] = "회수",
	["Retrieve a set from the bank."] = "은행에서 세트를 가져옵니다.",
	["Store the set %s in the bank."] = "%s 세트를 은행에 보관합니다.",
	["Retrieve the set %s from the bank."] = "은행에서 %s 세트를 가져옵니다.",
	["Keep"] = "유지",
	["Always keep the items for these sets in your inventory when flushing."] = "아이템을 은행으로 옮길 때, 이 세트는 인벤토리에 그대로 둡니다.",
	["Flush!"] = "옮기기",
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = "유지 리스트를 제외하고는 모든 세트를 은행으로 옮깁니다.",
	["Always keep the items for %s in your inventory."] = "인벤토리에 %s 세트의 아이템은 그대로 유지합니다.",
	["Flush&Get!"] = "교환하기",
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = "유지 목록의 아이템은 인벤토리로 기타 세트는 모두 은행으로 옮깁니다.",

	["Container"] = "가방",
} end)

L:RegisterTranslations("esES", function() return {
	["Banker"] = "Banquero",
	["Store or retrieve item sets from your bank."] = "Guardar o recoger objetos de conjunto de tu banco.",
	["Store"] = "Guardar",
	["Store a set in the bank."] = "Guarda un conjunto en el banco.",
	["Retrieve"] = "Recoger",
	["Retrieve a set from the bank."] = "Recoge un conjunto del banco",
	["Store the set %s in the bank."] = "Guarda el conjunto %s en el banco.",
	["Retrieve the set %s from the bank."] = "Recoge el conjunto %s del banco",
	["Keep"] = "Mantener",
	["Always keep the items for these sets in your inventory when flushing."] = "Mantiene siempre en tu inventario los objetos de estos conjuntos cuando usas \194\161Descargar!",
	["Flush!"] = "\194\161Descargar!",
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = "Coloca todos los conjuntos de ClosetGnome de tu inventario en tu banco, excepto los que has puesto en la lista de Mantener",
	["Flush&Get!"] = "\194\161Descargar y Recoger!",
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = "Recoge el equipo que est\195\161 en la lista de Mantener y coloca en el banco el resto",
	["Always keep the items for %s in your inventory."] = "Mantiene siempre los objetos de %s en tu inventario.",

	["Container"] = "Contenedor",
} end)

L:RegisterTranslations("zhTW", function() return {
	["Banker"] = "銀行",
	["Store or retrieve item sets from your bank."] = "從銀行中存取裝備",
	["Store"] = "儲存",
	["Store a set in the bank."] = "將套裝儲存在銀行中",
	["Retrieve"] = "取出",
	["Retrieve a set from the bank."] = "從銀行中取出套裝。",
	["Store the set %s in the bank."] = "將套裝 %s 儲存在銀行中。",
	["Retrieve the set %s from the bank."] = "從銀行中取出套裝 %s。",
	["Keep"] = "保持",
	["Always keep the items for these sets in your inventory when flushing."] = "那些套裝要用到的裝備總是保持在背包中。",
	["Flush!"] = "存放!",
	["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."] = "除了保持名單中的套裝外將從你背包存放全部 ClosetGnome 套裝到銀行。",
	["Flush&Get!"] = "存放&取出",
	["Gets the sets in the keep list to your inventory, and flushes all other items."] = "取出保持名單中的套裝到背包，然後存放其它的物品。",
	["Always keep the items for %s in your inventory."] = "總是保持 %s 的裝備在你背包中。",

	["Container"] = "容器",
} end)

-------------------------------------------------------------------------------
-- Addon declaration                                                         --
-------------------------------------------------------------------------------
ClosetGnomeBanker = AceLibrary("AceAddon-2.0"):new("AceHook-2.1", "AceConsole-2.0", "AceEvent-2.0", "AceDB-2.0")

-------------------------------------------------------------------------------
-- Initialization                                                            --
-------------------------------------------------------------------------------

local options = nil

local disabledFunc = function()
	return next(options.args.store.args) == nil and not ClosetGnomeBanker:IsBankAvailable()
end
local disabledFuncWithoutBank = function()
	return next(options.args.store.args) == nil
end

options = {
	type = "group",
	name = L["Banker"],
	desc = L["Store or retrieve item sets from your bank."],
	handler = ClosetGnomeBanker,
	order = 107,
	args = {
		flush = {
			type = "execute",
			name = L["Flush!"],
			desc = L["Flushes all the ClosetGnome sets from your inventory to your bank, except the ones checked in the keep list."],
			func = "Flush",
			disabled = "~IsBankAvailable",
			order = 101,
		},
		flushandget = {
			type = "execute",
			name = L["Flush&Get!"],
			desc = L["Gets the sets in the keep list to your inventory, and flushes all other items."],
			func = "FlushAndGet",
			disabled = "~IsBankAvailable",
			order = 102,
		},
		keep = {
			type = "group",
			name = L["Keep"],
			desc = L["Always keep the items for these sets in your inventory when flushing."],
			order = 103,
			args = {},
			pass = true,
			set = function(key, value) ClosetGnomeBanker.db.char.keep[key] = value and true or nil end,
			get = function(key) return ClosetGnomeBanker.db.char.keep[key] end,
			disabled = disabledFuncWithoutBank,
		},
		separator = {
			type = "header",
			name = " ",
			order = 104
		},
		store = {
			type = "group",
			name = L["Store"],
			desc = L["Store a set in the bank."],
			order = 105,
			args = {},
			pass = true,
			func = "StoreSet",
			disabled = disabledFunc,
		},
		retrieve = {
			type = "group",
			name = L["Retrieve"],
			desc = L["Retrieve a set from the bank."],
			order = 106,
			args = {},
			pass = true,
			func = "RetrieveSet",
			disabled = disabledFunc,
		},
	},
}


function ClosetGnomeBanker:OnInitialize()
	if not ClosetGnome then return end

	self:RegisterDB("ClosetGnomeBankerPlugThatBugDB", "ClosetGnomeBankerDB")
	self:RegisterDefaults("char", {
		keep = {}
	})
end

function ClosetGnomeBanker:OnEnable()
	if not ClosetGnome then return end
	self:RegisterEvent("BANKFRAME_OPENED", function() bankAvailable = true end)
	self:RegisterEvent("BANKFRAME_CLOSED", function() bankAvailable = nil end)

	self:RegisterEvent("ClosetGnome_DeleteSet", "UpdateSets")
	self:RegisterEvent("ClosetGnome_AddSet", "UpdateSets")

	if AceLibrary("AceEvent-2.0"):IsFullyInitialized() then
		self:UpdateSets()
	end
	self:RegisterEvent("AceEvent_FullyInitialized", "UpdateSets")

	ClosetGnome.OnMenuRequest.args.banker = options
end

-------------------------------------------------------------------------------
-- Events                                                                    --
-------------------------------------------------------------------------------

local sets = {}
function ClosetGnomeBanker:UpdateSets()
	for k in pairs(ClosetGnome.db.char.set) do
		table.insert(sets, k)
	end
	table.sort(sets)

	-- update menus
	options.args.store.args = nil
	options.args.retrieve.args = nil

	options.args.store.args = {}
	options.args.retrieve.args = {}

	for i, set in ipairs(sets) do
		local menuKey = set:gsub(" ", "")
		options.args.keep.args[menuKey] = {
			type = "toggle",
			name = set,
			desc = L["Always keep the items for %s in your inventory."]:format(set),
			passValue = set,
		}
		options.args.store.args[menuKey] = {
			type = "execute",
			name = set,
			desc = L["Store the set %s in the bank."]:format(set),
			passValue = set,
			icon = ClosetGnome.db.char.icons[set],
			disabled = CursorHasItem
		}
		options.args.retrieve.args[menuKey] = {
			type = "execute",
			name = set,
			desc = L["Retrieve the set %s from the bank."]:format(set),
			passValue = set,
			icon = ClosetGnome.db.char.icons[set],
			disabled = CursorHasItem
		}
	end
	for k in pairs(sets) do sets[k] = nil end
end

-------------------------------------------------------------------------------
-- Meat                                                                      --
-------------------------------------------------------------------------------

function ClosetGnomeBanker:IsBankAvailable()
	return bankAvailable
end

function ClosetGnomeBanker:Flush()
	if CursorHasItem() or not self:IsBankAvailable() then return false end

	-- Okay, so, we have to do three things:
	-- 1. Build a list of all the items in all the sets from CG.
	-- 2. Remove the items in the Keep sets from that list.
	-- 3. Move the remaining items from the inventory to the bank.

	-- Step 1.
	local items = {}
	for k, v in pairs(ClosetGnome.db.char.set) do
		for id, item in pairs(v) do
			items[item] = true
		end
	end

	-- Step 2.
	for k in pairs(self.db.char.keep) do
		if not ClosetGnome.db.char.set[k] then
			self.db.char.keep[k] = nil
		else
			for id, item in pairs(ClosetGnome.db.char.set[k]) do
				items[item] = nil
			end
		end
	end

	-- Step 3.
	for item in pairs(items) do
		local fromBagId, fromSlotId = ClosetGnome:FindItem(item)
		if fromBagId > -1 then
			local toBagId, toSlotId = self:FindFreeBankSlot()
			if toBagId == nil then break end
			if not slotLocks then slotLocks = {} end
			if not slotLocks[toBagId] then slotLocks[toBagId] = {} end
			slotLocks[toBagId][toSlotId] = true
			PickupContainerItem(fromBagId, fromSlotId)
			PickupContainerItem(toBagId, toSlotId)
		end
	end

	slotLocks = nil
end

function ClosetGnomeBanker:FlushAndGet()
	if CursorHasItem() or not self:IsBankAvailable() then return false end

	self:Flush()

	for k in pairs(self.db.char.keep) do
		self:RetrieveSet(k)
	end
end

function ClosetGnomeBanker:IsBankBag(bagId)
	if bagId == -1 then return true end
	if bagId > 11 or (bagId < 5 and bagId ~= -1) then return false end
	local link = GetInventoryItemLink("player", ContainerIDToInventoryID(bagId))
	if not link then return false end
	local linkId = select(3, link:find("item:(%d+)"))
	if not linkId then return false end
	local bagType = ClosetGnome:GetItemSubtype(linkId)
	if bagType and bagType == CGL["Bag"] then return true end
	return false
end

function ClosetGnomeBanker:FindFreeBankSlot()
	if not self:IsBankAvailable() then return nil end
	for i, v in ipairs(bankSlots) do
		if self:IsBankBag(v) then
			if not slotLocks then slotLocks = {} end
			if not slotLocks[v] then slotLocks[v] = {} end
			local numSlots = GetContainerNumSlots(v)
			for slot = 1, numSlots do
				if not slotLocks[v][slot] then
					local tex = GetContainerItemInfo(v, slot)
					if not tex then
						return v, slot
					end
				end
			end
		end
	end
	return nil
end

function ClosetGnomeBanker:FindFreeBagSlot()
	for theBag = NUM_BAG_FRAMES, 0, -1 do
		if ClosetGnome:IsNormalBag(theBag) then
			if not slotLocks then slotLocks = {} end
			if not slotLocks[theBag] then slotLocks[theBag] = {} end
			local numSlot = GetContainerNumSlots(theBag)
			for theSlot = 1, numSlot do
				if not slotLocks[theBag][theSlot] then
					local texture = GetContainerItemInfo(theBag, theSlot)
					if not texture then
						return theBag, theSlot
					end
				end
			end
		end
	end
	return nil
end

function ClosetGnomeBanker:StoreSet(setName)
	if CursorHasItem() or not setName or type(setName) ~= "string" or not self:IsBankAvailable() then return false end
	local set = ClosetGnome.db.char.set[setName]
	if not set then return false end

	for _, item in pairs(set) do
		local fromBagId, fromSlotId = ClosetGnome:FindItem(item)
		if fromBagId > -1 then
			local toBagId, toSlotId = self:FindFreeBankSlot()
			if toBagId == nil then break end
			if not slotLocks then slotLocks = {} end
			if not slotLocks[toBagId] then slotLocks[toBagId] = {} end
			slotLocks[toBagId][toSlotId] = true
			PickupContainerItem(fromBagId, fromSlotId)
			PickupContainerItem(toBagId, toSlotId)
		end
	end

	slotLocks = nil
end

function ClosetGnomeBanker:RetrieveSet(setName)
	if CursorHasItem() or not setName or type(setName) ~= "string" or not self:IsBankAvailable() then return false end
	local set = ClosetGnome.db.char.set[setName]
	if not set then return false end

	for i, fromBagId in ipairs(bankSlots) do
		if self:IsBankBag(fromBagId) then
			local numSlots = GetContainerNumSlots(fromBagId)
			if not slotLocks then slotLocks = {} end
			if not slotLocks[fromBagId] then slotLocks[fromBagId] = {} end
			for fromSlotId = 1, numSlots do
				local itemLink = GetContainerItemLink(fromBagId, fromSlotId)
				if itemLink then
					for _, item in pairs(set) do
						if itemLink == item and not slotLocks[fromBagId][fromSlotId] then
							-- Found the item in the bank.
							local toBagId, toSlotId = self:FindFreeBagSlot()
							if toBagId ~= nil then
								slotLocks[toBagId][toSlotId] = true
								slotLocks[fromBagId][fromSlotId] = true
								PickupContainerItem(fromBagId, fromSlotId)
								PickupContainerItem(toBagId, toSlotId)
							end
						end
					end
				end
			end
		end
	end

	slotLocks = nil
end

