﻿--[[
ClosetGnome, a World of Warcraft addon to manage item sets.
Copyright (C) 2007 Rabbit.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
]]

local ClosetGnome = ClosetGnome
assert(ClosetGnome, "ClosetGnome-UI-FuBar requires ClosetGnome to run.")

-----------------------------------------------------------------------
-- Localization
local L = AceLibrary("AceLocale-2.2"):new("ClosetGnome-UI-FuBar")

L:RegisterTranslations("enUS", function() return {
	["Menu"] = true,
	["Menu options."] = true,
	["Available sets"] = true,
	["%s |cffd9d919*|r"] = true,
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = true,
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = true,
	["Use the options menu to equip, edit, add or delete item sets."] = true,
	["ClosetGnome"] = true,
	["Show the set window"] = true,
} end)

L:RegisterTranslations("deDE", function() return {
	["Available sets"] = "Verfügbare Sets:",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "|cffeda55fKlicke|r ein Set an um die Ausrüstung zu wechseln. |cffeda55fStrg-Klick|r um es anzuprobieren. |cffeda55fShift-Klicke|r ein Set um es zu editieren. Das zuletzt benutzte Set ist mit einem Stern gekennzeichnet.",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "Es wurden noch keine Ausrüstungssets erstellt. Klicke mit der rechten Maustaste auf das Icon, dann 'Hinzufügen' um ein Set zu erstellen",
} end)

L:RegisterTranslations("esES", function() return {
	["Menu"] = "Men\195\186",
	["Menu options."] = "Opciones de Men\195\186",
	["Available sets"] = "Conjuntos disponibles",
	["%s |cffd9d919*|r"] = "%s |cffd9d919*|r",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "\nHaz |cffeda55fClick|r en un conjunto de arriba para cambiar tu equipo.\n|cffeda55fCtrl+Click|r para prob\195\161rtelo.\n|cffeda55fMay\195\186s+Click|r para editarlo.\n\nEl \195\186ltimo conjunto equipado est\195\161 marcado con un asterisco.",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "Todav\195\173a no has creado ning\195\186n conjunto de equipo.\nHaz |cffeda55fDoble-Clic|r para crear uno.",
	["Use the options menu to equip, edit, add or delete item sets."] = "Usar las opciones de men\195\186 para equipar, editar, a\195\145adir o eliminar elementos del conjunto.",
	["ClosetGnome"] = "ClosetGnome",
	["Show the set window"] = "Ver la ventana de conjuntos",
} end)

L:RegisterTranslations("frFR", function() return {
	["Menu options."] = "Options du menu.",
	["Available sets"] = "Ensembles disponibles",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "|cffeda55fCliquez|r sur un ensemble ci-dessus pour changer votre équipement. |cffeda55fCtrl-clic|r sur un ensemble pour l'essayer dans la cabine d'essayage. |cffeda55fShift-Clic|r sur un ensemble pour l'éditer. L'ensemble équipé en dernier est indiqué par une étoile.",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "Vous n'avez pas encore créé un seul ensemble. Faites un |cffeda55fdouble-clic|r pour en créer un.",
	["Use the options menu to equip, edit, add or delete item sets."] = "Utilise le menu des options pour équiper, éditer, ajouter ou supprimer des ensembles.",
} end)

L:RegisterTranslations("koKR", function() return {
	["Menu"] = "메뉴",
	["Menu options."] = "메뉴 옵션을 설정합니다.",
	["Available sets"] = "가능한 세트",
	["%s |cffd9d919*|r"] = "%s |cffd9d919*|r",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "장비를 변경하려면 세트를 |cffeda55f클릭|r하시오. 착용해 보려면 세트를 |cffeda55fCtrl-클릭|r하시오. 세트를 편집하려면 |cffeda55fShift-클릭|r하시오. 마지막 착용 세트는 별 표시로 가리킵니다.",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "어떠한 장비 세트도 현재 생성되지 않았습니다. 세트를 생성하려면 |cffeda55f더블-클릭|r하시오.",
	["Use the options menu to equip, edit, add or delete item sets."] = "아이템 세트의 착용, 편집, 추가나 삭제 옵션을 사용합니다.",
	["ClosetGnome"] = "ClosetGnome",
	["Show the set window"] = "세트 창 표시",
} end)

L:RegisterTranslations("zhCN", function() return {
	["Menu"] = "菜单",
	["Menu options."] = "菜单选项。",
	["Available sets"] = "可用套装",
	["%s |cffd9d919*|r"] = "%s |cffd9d919*|r",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "|cffeda55f左击: |r更换套装。\n|cffeda55fCtrl-左击: |r试穿套装。\n|cffeda55fShift-左击: |r编辑套装。\n最后使用的套装用星号标记。",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "没有可用的套装方案，|cffeda55f右键点击|r添加新的套装方案。",
	["Use the options menu to equip, edit, add or delete item sets."] = "使用菜单选项更换，编辑，增加，移除套装。",
	["ClosetGnome"] = "一键换装",
	["Show the set window"] = "打开一键换装的套装窗口",
} end)

L:RegisterTranslations("zhTW", function() return {
	["Menu"] = "選單",
	["Menu options."] = "設定選單。",
	["Available sets"] = "可用套裝",
	["%s |cffd9d919*|r"] = "%s |cffd9d919*|r",
	["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."] = "\n|cffeda55f左擊: |r更換套裝。\n|cffeda55fCtrl-左擊: |r試穿套裝。\n|cffeda55fShift-左擊: |r編輯套裝。\n最後一次使用的套裝用星號表示。",
	["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."] = "你尚未創建任何套裝，|cffeda55f雙擊左鍵|r創建套裝設置。",
	["Use the options menu to equip, edit, add or delete item sets."] = "使用設定選單更換，編輯，增加，移除套裝。",
	["ClosetGnome"] = "一鍵換裝",
	["Show the set window"] = "打開一鍵換裝的套裝視窗",
} end)

-----------------------------------------------------------------------
-- FuBar UI for ClosetGnome

local tablet = AceLibrary:HasInstance("Tablet-2.0") and AceLibrary("Tablet-2.0") or nil
local defaultIcon = "Interface\\Icons\\INV_Chest_Cloth_17"

CGFu = AceLibrary("AceAddon-2.0"):new("AceDB-2.0", "FuBarPlugin-2.0", "AceEvent-2.0")
local CGFu = CGFu
CGFu.hasIcon = true

local lastEquippedSet = nil
local sortedSets = {}

local function updateSortedSets()
	for i in ipairs(sortedSets) do sortedSets[i] = nil end
	for k in pairs(ClosetGnome.db.char.set) do table.insert(sortedSets, k) end
	table.sort(sortedSets)
end

function CGFu:OnInitialize()
	self:RegisterDB("ClosetGnomeDB", "ClosetGnomePerCharDB")

	local args = AceLibrary("FuBarPlugin-2.0"):GetAceOptionsDataTable(self)
	local options = ClosetGnome.options
	if not options.args[L["Menu"]] then
		options.args[L["Menu"]] = {
			type = "group",
			name = L["Menu"],
			desc = L["Menu options."],
			args = args,
			order = 300,
		}
	end

	self.hideWithoutStandby = true
	self.hasIcon = defaultIcon
	self.hasNoColor = true
	self.defaultMinimapPosition = 240
	self.clickableTooltip  = true
	self.OnMenuRequest = options

	if not tablet then
		self.blizzardTooltip = true
	end

	if tablet and EarthFeature_AddButton then
		EarthFeature_AddButton({
			id = "ClosetGnome",
			name = L["ClosetGnome"],
			subtext = "ClosetGnome",
			tooltip = L["Show the set window"],
			icon = "Interface\\Icons\\INV_Chest_Cloth_17",
			callback = function()
				if self:IsTooltipDetached() then
					tablet:Attach(self.frame)
				else
					tablet:Detach(self.frame)
				end
			end,
		})
	end
end

function CGFu:OnEnable()
	local function updateLatest(set)
		lastEquippedSet = set
		updateSortedSets()
		self:UpdateDisplay()
	end
	local function removeLatest(set)
		if lastEquippedSet == set then
			lastEquippedSet = nil
		end
		updateSortedSets()
		self:UpdateDisplay()
	end
	self:RegisterEvent("ClosetGnome_UpdateSet", updateLatest)
	self:RegisterEvent("ClosetGnome_AddSet", updateLatest)
	self:RegisterEvent("ClosetGnome_WearSet", updateLatest)
	self:RegisterEvent("ClosetGnome_DeleteSet", removeLatest)
	self:RegisterEvent("ClosetGnome_PartlyWearSet", "UpdateDisplay")
	self:RegisterEvent("ClosetGnome_KeybindingsUpdated", "UpdateDisplay")

	if AceLibrary("AceEvent-2.0"):IsFullyInitialized() then
		self:AceEvent_FullyInitialized()
	end
	self:RegisterEvent("AceEvent_FullyInitialized")

	updateSortedSets()
end

function CGFu:AceEvent_FullyInitialized()
	-- Loop through the current sets and check if we have any of them equipped.
	for k,v in pairs(ClosetGnome.db.char.set) do
		local ret = ClosetGnome:IsSetFullyEquipped(k)
		if ret then
			lastEquippedSet = k
			ClosetGnome:TriggerEvent("ClosetGnome_WearSet", k)
			break
		end
	end
	self:UpdateDisplay()
end

function CGFu:OnTextUpdate()
	if lastEquippedSet then
		self:SetText(ClosetGnome:GetSetDisplayName(lastEquippedSet))
	else
		self:SetText("ClosetGnome")
	end
	if self.db.char.icons[lastEquippedSet] then
		self:SetIcon(ClosetGnome.db.char.icons[lastEquippedSet])
	else
		self:SetIcon(defaultIcon)
	end
end

function CGFu:OnDoubleClick()
	if IsShiftKeyDown() then
		ClosetGnome:CreateOrUpdateSet(lastEquippedSet)
	else
		ClosetGnome:CreateOrUpdateSet()
	end
end

local function clickfunc(name)
	if IsControlKeyDown() then
		local set = ClosetGnome.db.char.set[name]
		DressUpModel:Undress()
		for i = 1, 19 do
			local item = set[i]
			if i ~= 18 and item ~= false then
				DressUpItemLink(item)
			end
		end
	elseif IsShiftKeyDown() then
		ClosetGnome:CreateOrUpdateSet(name)
	else
		ClosetGnome:WearSet(name)
	end
end

local function blizzardTooltip()
	GameTooltip:AddLine("ClosetGnome")
	if #sortedSets > 0 then
		GameTooltip:AddLine(L["Available sets"], 1, 1, 1)
		for i, set in ipairs(sortedSets) do
			local displayName = ClosetGnome:GetSetDisplayName(set)
			if lastEquippedSet == set then
				displayName = L["%s |cffd9d919*|r"]:format(ClosetGnome:GetSetDisplayName(set))
			end
			GameTooltip:AddDoubleLine(
				displayName,
				ClosetGnome.db.char.keybindings[set] or "",
				0.7, 1, 0.2,
				0.5, 0.5, 0.5
			)
		end
		GameTooltip:AddLine(L["Use the options menu to equip, edit, add or delete item sets."], 0.2, 1, 0.2, 1)
	else
		GameTooltip:AddLine(L["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."], 0.2, 1, 0.2, 1)
	end
end

local function tabletTooltip()
	local cat = nil
	for i, set in ipairs(sortedSets) do
		if not ClosetGnome:HideSetFromUI(set) then
			if not cat then
				cat = tablet:AddCategory(
					"columns", 2,
					"text", L["Available sets"],
					"showWithoutChildren", false,
					"child_hasCheck", true,
					"child_checked", true,
					"child_text2R", 0.5,
					"child_text2G", 0.5,
					"child_text2B", 0.5
				)
			end
			local displayName = ClosetGnome:GetSetDisplayName(set)
			if lastEquippedSet == set then
				displayName = L["%s |cffd9d919*|r"]:format(ClosetGnome:GetSetDisplayName(set))
			end
			cat:AddLine(
				"text", displayName,
				"func", clickfunc,
				"arg1", set,
				"checkIcon", ClosetGnome.db.char.icons[set] or "",
				"text2", ClosetGnome.db.char.keybindings[set] or ""
			)
		end
	end

	if cat ~= nil then
		tablet:SetHint(L["|cffeda55fClick|r a set above to change your equipment. |cffeda55fCtrl-Click|r a set to try it on. |cffeda55fShift-Click|r a set to edit it. The last equipped set is indicated by a star."])
	else
		tablet:SetHint(L["You have not created any equipment sets yet. |cffeda55fDouble-Click|r to create a set."])
	end
end

function CGFu:OnTooltipUpdate()
	if self.blizzardTooltip then
		blizzardTooltip()
	else
		tabletTooltip()
	end
end

