--[[ Set the module up ]]--
ClassLoot = LibStub("AceAddon-3.0"):NewAddon("ClassLoot", "AceConsole-3.0", "AceHook-3.0")

local L, db
local CD = ClassLoot_Data
local CC = ClassLoot_Constants
local ttLine = { 
	["GameTooltip"] = {},
	["ItemRefTooltip"] = {},
	["AtlasLootTooltip"] = {}
}

ClassLoot_Data = nil
ClassLoot_Constants = nil

--[[ Run when ClassLoot is initialized ]]--
function ClassLoot:OnInitialize()
	-- Get Locale
	L = LibStub("AceLocale-3.0"):GetLocale("ClassLoot", true)
	
	-- Register Saved Variables
	local defaults = {
		profile = {
			showTooltip = true,
			showBoss = true,
			showInstance = true
		}
	}
	db = LibStub("AceDB-3.0"):New("ClassLootDB", defaults, "profile")
	
	-- Register slash command
	local opts = {
		name = "ClassLoot",
		handler = ClassLoot,
		type='group',
		args = {
			check = {
				type = 'input',
				name = L["Check an item (Local)"],
				desc = L["Display ClassLoot for an item locally"],
				usage = L["<item link>"],
				get = false,
				set = function(info, val) self:ShowItemStars(val, "LOCAL") end,
				pattern = "item:(%d+)"
			},
			gcheck = {
				type = 'input',
				name = L["Check an item (Guild)"],
				desc = L["Display ClassLoot for an item in guild chat"],
				usage = L["<item link>"],
				get = false,
				set = function(info, val) self:ShowItemStars(val, "GUILD") end,
				pattern = "item:(%d+)"
			},
			rcheck = {
				type = 'input',
				name = L["Check an item (Raid)"],
				desc = L["Display ClassLoot for an item in raid chat"],
				usage = L["<item link>"],
				get = false,
				set = function(info, val) self:ShowItemStars(val, "RAID") end,
				pattern = "item:(%d+)"
			}
		}
	}
	LibStub("AceConfig-3.0"):RegisterOptionsTable("ClassLoot", opts, CC["Slash-Commands"])
end

--[[ Run when ClassLoot is enabled ]]--
function ClassLoot:OnEnable()
	-- Hook tooltips if needed
	if db.profile.showTooltip then
		self:HookTooltips()
	end
	-- Build Interface Options window
	self:CreateInterfaceOptions()
end

--[[ Hook Tooltips ]]--
function ClassLoot:HookTooltips()
	self:HookScript(GameTooltip, "OnTooltipSetItem")
	self:HookScript(GameTooltip, "OnTooltipSetSpell")
	self:HookScript(ItemRefTooltip, "OnTooltipSetItem")
	if AtlasLootTooltip then
		self:HookScript(AtlasLootTooltip, "OnTooltipSetItem")
	end
end

--[[ Unhook Tooltips ]]--
function ClassLoot:UnhookTooltips()
	self:Unhook(GameTooltip, "OnTooltipSetItem")
	self:Unhook(GameTooltip, "OnTooltipSetSpell")
	self:Unhook(ItemRefTooltip, "OnTooltipSetItem")
	if AtlasLootTooltip then
		self:Unhook(AtlasLootTooltip, "OnTooltipSetItem")
	end
end

-- Make Star Table
--  itemInfo - Table of class/stars
function ClassLoot:MakeStarTable(itemInfo)
	local starTable = { [3] = {}; [4] = {}; [5] = {}; }
	local output = {}
	
	-- First, set the class names to the starTable sub-tables
	for class, stars in pairs(itemInfo) do
		table.insert(starTable[stars], CC[class])
	end
	
	-- Now, concat all the class names and set them as the value 
	for stars, _ in pairs(starTable) do
		starTable[stars] = table.concat(starTable[stars], ", ")
	end
	
	-- Walk the table backwards to generate the output
	for i = 5, 3, -1 do
		if starTable[i] ~= "" then
			local stars = ''
			for j = 1, 6, 1 do
				if j <= i then
					stars = stars..ICON_LIST[1]..'0|t'
				else
					stars = stars..'|T:0|t'
				end
			end
			table.insert(output, { ["stars"] = stars, ["classes"] = starTable[i] })
		end
	end
	
	-- Finally, return the output
	return output
end

-- Show Item Stars
--	target   - The chat target (LOCAL, RAID or GUILD)
--	itemLink - The link for the item
function ClassLoot:ShowItemStars(itemLink, target)
	-- Validate Target
	if target == "RAID" and GetNumRaidMembers() == 0 then
		self:Print(L["Not in a raid!"])
		return
	elseif target == "GUILD" and not IsInGuild() then
		self:Print(L["Not in a guild!"])
		return
	end
	
	-- Get the info table
	local itemInfo = CD[tonumber(itemLink:match('item:(%d+)'))]
	if not itemInfo then
		-- Not in ClassLoot_Data
		self:Print(itemLink..' '..L["could not be found"])
		return
	end
	
	local PT = LibStub("LibPeriodicTable-3.1"):ItemSearch(itemLink)
	local BB = LibStub("LibBabble-Boss-3.0"):GetUnstrictLookupTable()
	local BZ = LibStub("LibBabble-Zone-3.0"):GetUnstrictLookupTable()
	local output = {}
	
	-- Header
	table.insert(output, L["ClassLoot info for"]..' '..itemLink)
	
	-- Who drops it where
	if PT then
		-- k = index, v = table name [InstanceLoot.InstanceName.BossName]
		for _, v in pairs(PT) do
			local set, instance, boss = strsplit(".", v)
			if set == "InstanceLoot" then
				-- Special cases where the "boss" is not the Boss!
				boss = CC[boss] or boss
				-- Localise "boss" and "instance" first, as it could be in 2 different places
				boss = BB[boss] or L[boss]
				instance = BZ[instance] or L[instance]
				table.insert(output, L["Dropped in"]..' '..instance..' '..L["by"]..' '..boss)
			end
		end
	end
	
	-- Ouptput the Item Info!
	for _, v in pairs(self:MakeStarTable(itemInfo)) do
		if target == "LOCAL" then
			self:Print(v["stars"]..v["classes"])
		else
			SendChatMessage(v["stars"]..v["classes"], target)
		end
	end
end

--[[ Tooltip Handler to add ClassLoot info to items ]]--
function ClassLoot:OnTooltipSetItem(tooltip, ...)
	local ttName = tooltip:GetName()
	
	-- Reset the 'right text' anchor points on this tooltip
	for n, _ in pairs(ttLine[ttName]) do
		_G[ttName.."TextRight"..n]:ClearAllPoints()
		ttLine[ttName][n] = nil
	end
	
	local itemLink = select(2, tooltip:GetItem())
	local itemInfo = CD[tonumber(itemLink:match('item:(%d+)'))]
	if itemInfo and db.profile.showTooltip then
		local firstLine = tooltip:NumLines() + 2
		-- Header
		tooltip:AddLine("|cffff0000ClassLoot:|r")
		
		-- Build the star table
		for _, v in pairs(self:MakeStarTable(itemInfo)) do
			tooltip:AddDoubleLine(v["stars"], "|cff0099ff"..v["classes"].."|r")
		end
		
		if (db.profile.showBoss or db.profile.showInstance) and tooltip ~= AtlasLootTooltip then
			-- Search the item in PT to get instances and bosses
			local PT = LibStub("LibPeriodicTable-3.1"):ItemSearch(itemLink)
			if PT then
				local BB = LibStub("LibBabble-Boss-3.0"):GetUnstrictLookupTable()
				local BZ = LibStub("LibBabble-Zone-3.0"):GetUnstrictLookupTable()
				local bSet, iSet, bList, iList = {}, {}, {}, {}
				
				-- Loop the PT results to get the instances and bosses. k = index, v = table name [InstanceLoot.InstanceName.BossName]
				for _, v in pairs(PT) do
					local set, instance, boss = strsplit(".", v)
					if set == "InstanceLoot" then
						-- Special cases where the "boss" is not the Boss!
						boss = CC[boss] or boss
						-- Localize into the  boss and instance sets
						bSet[BB[boss] or L[boss]] = true
						iSet[BZ[instance] or L[instance]] = true
					end
				end
				
				-- Convert boss and instance sets to lists for table.concat()
				for k, _ in pairs(bSet) do
					bList[#bList+1] = k
				end
				for k, _ in pairs(iSet) do
					iList[#iList+1] = k
				end
				
				-- Finally, append to the tooltip
				if db.profile.showBoss and next(bList) then
					tooltip:AddDoubleLine("|cfff0e68c"..L["Boss"].."|r", "|cfff0e68c"..table.concat(bList, ", ").."|r")
				end
				if db.profile.showInstance and next(iList) then
					tooltip:AddDoubleLine("|cfff0e68c"..L["Instance"].."|r", "|cfff0e68c"..table.concat(iList, ", ").."|r")
				end
			end
		end
		
		local maxLeft = 0
		local lastLine = tooltip:NumLines()
		
		-- Check our new lines for the longest 'left text' (To cope with long localised strings)
		for i = firstLine, lastLine, 1 do
			maxLeft = max(_G[ttName.."TextLeft"..i]:GetWidth(), maxLeft)
		end
		
		-- Now adjust our 'right text' to sit left-aligned, making a note of what we've changed
		for i = firstLine, lastLine, 1 do
			ttLine[ttName][i] = true
			_G[ttName.."TextRight"..i]:SetPoint("LEFT", _G[ttName.."TextLeft"..i], "LEFT", maxLeft+10, 0)
		end
	end
end

--[[ Tooltip Handler to fix 'right text' anchor points on spell tooltips ]]--
function ClassLoot:OnTooltipSetSpell(tooltip, ...)
	local ttName = tooltip:GetName()
	for n, _ in pairs(ttLine[ttName]) do
		_G[ttName.."TextRight"..n]:ClearAllPoints()
		ttLine[ttName][n] = nil
	end
end

--[[ Interface Options Window ]]--
function ClassLoot:CreateInterfaceOptions()
	local cfgFrame = CreateFrame("FRAME", nil, UIParent)
	cfgFrame.name = "ClassLoot"
	
	local cfgFrameHeader = cfgFrame:CreateFontString("OVERLAY", nil, "GameFontNormalLarge")
	cfgFrameHeader:SetPoint("TOPLEFT", 15, -15)
	cfgFrameHeader:SetText("ClassLoot")
	
	local cfgShowTooltip = CreateFrame("CHECKBUTTON", "ClassLoot_cfgShowTooltip", cfgFrame, "InterfaceOptionsCheckButtonTemplate")
	ClassLoot_cfgShowTooltip:SetPoint("TOPLEFT", 20, -40)
	ClassLoot_cfgShowTooltipText:SetText(L["Enable ClassLoot Tooltips"])
	ClassLoot_cfgShowTooltip:SetChecked(db.profile.showTooltip)
	ClassLoot_cfgShowTooltip:SetScript("OnClick", function(self)
		db.profile.showTooltip = not db.profile.showTooltip
		-- Toggle the sub-options
		if db.profile.showTooltip then
			-- Enable Sub-Options
			ClassLoot_cfgShowBossTooltip:Enable()
			ClassLoot_cfgShowInstanceTooltip:Enable()
			-- Hook the tooltips
			ClassLoot:HookTooltips()
		else
			-- Disable Sub-Options
			ClassLoot_cfgShowBossTooltip:Disable()
			ClassLoot_cfgShowInstanceTooltip:Disable()
			-- Unhook the tooltips
			ClassLoot:UnhookTooltips()
		end
	end)
	
	local cfgShowBossTooltip = CreateFrame("CHECKBUTTON", "ClassLoot_cfgShowBossTooltip", cfgFrame, "InterfaceOptionsCheckButtonTemplate")
	ClassLoot_cfgShowBossTooltip:SetPoint("TOPLEFT", 40, -64)
	ClassLoot_cfgShowBossTooltipText:SetText(L["Display Boss Name"])
	ClassLoot_cfgShowBossTooltip:SetChecked(db.profile.showBoss)
	ClassLoot_cfgShowBossTooltip:SetScript("OnClick", function(self)
		db.profile.showBoss = not db.profile.showBoss
	end)
	
	local cfgShowInstanceTooltip = CreateFrame("CHECKBUTTON", "ClassLoot_cfgShowInstanceTooltip", cfgFrame, "InterfaceOptionsCheckButtonTemplate")
	ClassLoot_cfgShowInstanceTooltip:SetPoint("TOPLEFT", 40, -88)
	ClassLoot_cfgShowInstanceTooltipText:SetText(L["Display Instance Name"])
	ClassLoot_cfgShowInstanceTooltip:SetChecked(db.profile.showInstance)
	ClassLoot_cfgShowInstanceTooltip:SetScript("OnClick", function(self)
		db.profile.showInstance = not db.profile.showInstance
	end)
	
	-- Check for disabled tooltips on startup
	if not db.profile.showTooltip then
		ClassLoot_cfgShowBossTooltip:Disable()
		ClassLoot_cfgShowInstanceTooltip:Disable()
	end
	
	InterfaceOptions_AddCategory(cfgFrame)
end
