-- $Id$

local L = AceLibrary("AceLocale-2.2"):new("Sidelist")
local Tablet = AceLibrary("Tablet-2.0")

-- Table of registered Cartographer_Notes icons
local iconList = nil

-- Track session state of sidelist
local sidelistIsOpen

-- Session cache of open/closed states for categories.  Default is closed.
--      category name => bool
local categoryOpen = { }

Cartographer_Sidelist = Cartographer:NewModule("Sidelist",
            "AceEvent-2.0", "AceConsole-2.0", "AceHook-2.1", "AceDebug-2.0")

Cartographer_Sidelist:SetDebugLevel(2)


-- Return the list of keys in table t
local
function keys(t)
    local ret = { }

    for k, _ in pairs(t)
    do
        table.insert(ret, k)
    end

    return ret
end     -- keys


-- Function that returns key-value pairs for a table, sorted by value.
-- Keep in mind that if t's values are tables, you will need a comparator
-- (because you can't compare two tables).
-- iterFunc should be either pairs or ipairs.  If ipairs is used, the 1st
-- value in the returned list will be a 1-based integer offset denoting the
-- position in the array of the sorted values.
local
function sortedValueIterFunc(t, comparator, iterFunc)
    local value2key = { }

    -- Create reverse mapping of values to keys so we can sort on the values.
    -- The reverse mapping's keys are t's values, each placed in an
    -- anonymous table so that we can handle identical values from t.
    for k, v in iterFunc(t)
    do
        value2key[{ v }] = k
    end

    local sortedValues = keys(value2key)

    -- Actually sort the values.  Since each value is in an anonymous table,
    -- we have to invoke the comparator (whether default or supplied) with
    -- the values from within each anonymous table.
    table.sort(sortedValues,
        function(lhs, rhs)
            if comparator
            then
                return comparator(lhs[1], rhs[1])
            else
                return lhs[1] < rhs[1]
            end
        end)

    local i = 0
    local n = # sortedValues
    return function()
        i = i + 1
        if i <= n
        then
            return iterFunc == ipairs and i or value2key[sortedValues[i]],
                        sortedValues[i][1]
        end
    end
end     -- sortedValueIterFunc


-- Iterator that returns key-value pairs for a table, sorted by value.
local
function sortedValue_pairs(t, comparator)
    return sortedValueIterFunc(t, comparator, pairs)
end     -- sortedValue_pairs


-- Iterator that returns index-value pairs for an array, sorted by value.
local
function sortedValue_ipairs(t, comparator)
    return sortedValueIterFunc(t, comparator, ipairs)
end     -- sortedValue_ipairs


-- Return number of keys in t.
local
function numberOfKeys(t)
    return not t and 0 or # keys(t)
end     -- numberOfKeys


-- Return a list containing the stable order of unique elements in the given
-- list.
--[[
local
function uniq(l)
    local ret = { }

    for _, v in ipairs(l)
    do
        ret[v] = true
    end

    return keys(ret)
end     -- uniq
]]


-- Return list containing names of Cartographer Notes' external databases.
local
function getExternalDatabaseNames(zone)
    return keys(Cartographer_Notes.externalDBs)
end     -- getExternalDatabaseNames


-- Return list containing all creators for Cartographer Notes' internal
-- database containing notes for the given zone.
--[[
local
function getInternalDatabaseNames(zone)
    local list = { }
    local zone2notes = Cartographer_Notes.db.account.pois[zone]

    if zone2notes
    then
        for id, note in pairs(zone2notes)
        do
            table.insert(list, note.creator or "<unknown>")
        end
    end

    return list
end     -- getInternalDatabaseNames
]]


-- Return reference to the table containing notes in the given zone for the
-- given dbName.
-- If dbName is nil, the internal note table is used.
local
function getNoteTable(zone, dbName)
    return rawget(dbName and Cartographer_Notes.externalDBs[dbName] or
                    Cartographer_Notes.db.account.pois, zone)
end     -- getNoteTable


-- Return table with note's icon data, given a particular db.  Needed
-- because some note databases don't have the usual note table data (e.g.
-- Cartographer_Mailboxes).  Returning a table in case we want to return
-- extra icon info in the future.
-- ret = {
--    path = icon's path
-- }
-- XXX:  This should be factored into a generic function that returns
-- note info in a standard table.
local
function getNoteIconPath(dbName, note)
    -- Mapping of special-case dbName's to info needed to get the icon's
    -- path for the given note.  An entry in this mapping implies note does
    -- not have the usual title, icon, etc. fields.
    local dbMap = {
        Mailboxes = {
            iconPathFunc = function()
              return "Interface\\Addons\\Cartographer_Mailboxes\\Artwork\\Mail"
            end,
        },
        -- For Herbalism, Mining and Fishing, the localized text is stored
        -- for the note's title.  We need to use Babble lib to get the
        -- reverse translation, since that's how the icons are registered
        -- with Cartographer_Notes.  Also, note.title doesn't exist--the
        -- note is the note's title.
        Herbalism = {
            iconPathFunc = function()
                local key =
                    AceLibrary("Babble-Herbs-2.2"):GetReverseTranslation(note)
                return iconList[key].path
            end,
        },
        Mining = {
            iconPathFunc = function()
                local key =
                    AceLibrary("Babble-Ore-2.2"):GetReverseTranslation(note)
                return iconList[key].path
            end,
        },
        Fishing = {
            iconPathFunc = function()
                local key =
                    AceLibrary("Babble-Fish-2.2"):GetReverseTranslation(note)
                return iconList[key].path
            end,
        },
    }

    local iconPath = nil

    -- See if special handling is needed to get the icon path.  Do this
    -- check first rather than the cheaper check directly against iconList
    -- just in case there is an icon registered with a name that matches 
    -- note or note.title--which would most likely be wrong.
    if dbMap[dbName] and dbMap[dbName].iconPathFunc
    then
        iconPath = dbMap[dbName].iconPathFunc()
    end

    return {
        path = iconPath or iconList[note.icon].path
    }
end     -- getNoteIconPath


-- Return the note's title, given a particular db.  Needed because some
-- note databases don't have the usual note table data.
local
function getNoteTitle(dbName, note)
    -- XXX:  yes, we currently don't use the dbName at all--it's an
    -- assumption that the note.title is valid or the title is the note.
    -- XXX:  This should be factored into a generic function that returns
    -- note info in a standard table.
    return note.title or note
end     -- getNoteTitle


-- Return the pretty category name for the given DB.  dbName of nil refers
-- to the General category.
local
function dbNameToCategory(dbName)
    local db2pretty = {
        InstanceNotes = "Instance Notes",
        POI = "Points of Interest",
        QuestObjectives = "Quest Objectives",
        ToFuNotes = "Flightmasters"
    }

    -- If there is no key for the given DB, use dbName as-is.
    return dbName and db2pretty[dbName] or
            not db2pretty[dbName] and dbName or
            "General"
end     -- dbNameToCategory


-- Show or hide the side list tablet.
-- show - true=>show, false=>hide
local
function showTablet(self, show)
    if show
    then
        Cartographer:AcquireSideTablet(self.db.profile.side, self)
        sidelistIsOpen = true
    else
        Cartographer:ReleaseSideTablet(self.db.profile.side, self)
        sidelistIsOpen = false
    end
end     -- showTablet


function Cartographer_Sidelist:OnInitialize()
    self.author = "Sekuyo"
    self.category = "Map"
    self.email = "sekuyo13@gmail.com"
    self.license = "GPL v2"
    self.name = "Cartographer_Sidelist"
    self.notes = "Toggle list of map notes in current zone"
    self.title = "Cartographer_Sidelist"
    self.version = "(pre-alpha)"
    --self.website = "http://code.google.com/p/sekuyo-wow/"


    self.db = Cartographer:AcquireDBNamespace("Sidelist")

    Cartographer:RegisterDefaults("Sidelist", "profile",
    {
        side = "RIGHT",
        isOpen = true,
    })

    local sidelistArgs =
    {
        toggleSide = {
            name = "Right side",
            desc = "Toggle where sidelist is shown",
            type = "toggle",
            get = function() return self.db.profile.side == "RIGHT" end,
            set = function(v)
                local isOpen = sidelistIsOpen

                if isOpen
                then
                    showTablet(self, false)
                end

                self.db.profile.side =
                    self.db.profile.side == "RIGHT" and "LEFT" or "RIGHT"

                if isOpen
                then
                    showTablet(self, true)
                end
            end,
            guiNameIsMap = true,
            map = {
                [true] = "Right side",
                [false] = "Left side",
            }
        },
        showSidelist = {
            name = "Toggle sidelist display",
            desc = "Show or hide the sidelist",
            type = "execute",
            func = function() 
                showTablet(self, not sidelistIsOpen)
            end
        },
        showDBs = {
            name = "Show",
            desc = "Specify databases that should appear in the sidelist",
            type = "group",
            args = { },     -- filled in OnInitialize because that's when
                            -- the external DB names become known in
                            -- Cartographer_Notes.
        },
    }  -- sidelistArgs

    local cartoMenuOpts = {
        name = "Sidelist",
        desc = self.notes,
        handler = self,
        type = "group",
        args = sidelistArgs,
    }

    Cartographer.options.args.Sidelist = cartoMenuOpts
    AceLibrary("AceConsole-2.0"):InjectAceOptionsTable(self,
        Cartographer.options.args.Sidelist)
end     -- Cartographer_Sidelist:OnInitialize


function Cartographer_Sidelist:OnEnable()
    self:RegisterEvent("CartographerNotes_NoteSet")
    self:RegisterEvent("CartographerNotes_NoteDeleted")

    self:RegisterEvent("Cartographer_MapOpened")
    self:RegisterEvent("Cartographer_MapClosed")
    self:RegisterEvent("Cartographer_ChangeZone")

    -- Hook instance maps
    if Cartographer:GetInstanceWorldMapButton()
    then
        self:HookScript(Cartographer:GetInstanceWorldMapButton(),
            "OnClick", "SetupSidelist")
    else
        self:RegisterEvent("Cartographer_RegisterInstanceWorldMapButton",
            function(frame)
                self:HookScript(frame, "OnClick", "SetupSidelist")
            end)
    end
end     -- Cartographer_Sidelist:OnEnable


-- Handler to set up sidelist tablet
function Cartographer_Sidelist:SetupSidelist(frame, mouseButton)
    self:Print("in setupsidelist")
    if mouseButton == "LeftButton" and IsControlKeyDown() and IsShiftKeyDown()
    then
        self:Print("sidelist click")
        showTablet(self, not sidelistIsOopen)
    else
        self:Print("sidelist click passthru")
        return self.hooks[frame].OnClick(frame, mouseButton)
    end
end


-- Populate the tablet with categories and notes for the currently viewed
-- zone.
-- A curious point:  you have to draw all content within categories
-- yourself because there is no concept of a tree control within the tablet.
-- Hence, keeping track of which categories are "open".
local
function drawTablet(self)
    local zone = Cartographer:GetCurrentEnglishZoneName()

    Tablet:SetTitle(string.format("Notes for %s",
        Cartographer:GetCurrentEnglishZoneName()))

    -- First category is always the non-externalDB category.  It contains
    -- notes that are manually created or have been imported.
    local generalCategory = Tablet:AddCategory(
        "id", "general",
        "columns", 1,
        "text", dbNameToCategory(nil),
        "textR", 1, "textG", 1, "textB", 0,
        "wrap", true,
        "func", function()
                categoryOpen.general = not categoryOpen.general
                showTablet(self, true)
            end,
        "showWithoutChildren", true,    -- general category always shown
        "checked", true,
        "hasCheck", true,
        "checkIcon", "Interface\\Buttons\\UI-" ..
            (categoryOpen.general and "Minus" or "Plus") .. "Button-Up"
        )

    local iconList = Cartographer_Notes:GetIconList()

    -- Add all internal database notes to general category if it's open
    if categoryOpen.general
    then
        local noteTable = getNoteTable(zone)
        if noteTable
        then
            for id, note in sortedValue_pairs(noteTable,
                                function(lhs, rhs)
                                    return getNoteTitle(nil, lhs) <
                                            getNoteTitle(nil, rhs)
                                end)
            do
                generalCategory:AddLine(
                    "text", getNoteTitle(nil, note),
                    "textR", 1, "textG", 1, "textB", 1,
                    "wrap", true,
                    "func", function(id)
                                self:Debug("Clicked id="..id.." in general")
                            end,
                    "arg1", id,
                    "checked", true,
                    "hasCheck", true,
                    "checkIcon", getNoteIconPath(nil, note, iconList).path,
                    "indentation", 10
                )
            end
        end     -- if general notetable exists for this zone
    end     -- if general category is open

    -- Used to ensure there is vertical space between the General category
    -- and the next category containing notes.  Set to true after the first
    -- non-general category is shown.
    local hideBlankLine = false

    -- Add categories for any external db's with notes in this zone
    for index, dbName in sortedValue_ipairs(getExternalDatabaseNames(zone))
    do
        local noteTable = getNoteTable(zone, dbName)
        local numNotes = numberOfKeys(noteTable)

        -- Only add this db as a category if it has notes for this zone, and
        -- if the user says we should show it.
        if numNotes > 0 and self.db.profile["show"..dbName]
        then
            local category = Tablet:AddCategory(
                "id", dbName,
                "columns", 1,
                "text", dbNameToCategory(dbName),
                "textR", 1, "textG", 1, "textB", 0,
                "wrap", true,
                "func", function()
                        categoryOpen[dbName] = not categoryOpen[dbName]
                        showTablet(self, true)
                    end,
                "showWithoutChildren", true,
                -- 1st category needs leading blank line to separate it from
                -- the General category
                "hideBlankLine", hideBlankLine,
                "checked", true,
                "hasCheck", true,
                "checkIcon", "Interface\\Buttons\\UI-" ..
                    (categoryOpen[dbName] and "Minus" or "Plus") .. "Button-Up"
            )

            hideBlankLine = true

        -- Add this external database's notes to this category if it's open
            if categoryOpen[dbName]
            then
                for id, note in sortedValue_pairs(noteTable,
                                    function(lhs, rhs)
                                        return getNoteTitle(dbName, lhs) <
                                                getNoteTitle(dbName, rhs)
                                    end)
                do
                    category:AddLine(
                        "text", getNoteTitle(dbName, note),
                        "textR", 1, "textG", 1, "textB", 1,
                        "wrap", true,
                        "func", function(id, db)
                                    self:Debug("Clicked id="..id.." in db="..db)
                                end,
                        "arg1", id,
                        "arg2", dbName,
                        "checked", true,
                        "hasCheck", true,
                        "checkIcon",
                            getNoteIconPath(dbName, note, iconList).path,
                        "indentation", 10
                    )
                end
            end     -- if category is open
        end     -- number of notes for this category and zone > 0
    end     -- add category for each external db
end     -- drawTablet


-- Called when map is opened
function Cartographer_Sidelist:Cartographer_MapOpened()
    -- (Re)grab the list of known icons (it may have changed since we were
    -- last in here).
    iconList = Cartographer_Notes:GetIconList()

    -- Populate the Cartographer|Sidelist|"Show" submenu with each external
    -- DB name.  This is done on map open so that we can be sure any
    -- Cartographer_Notes external DBs are available.  This also allows us
    -- to notice new DBs becoming available since the last time the map was
    -- opened.
    for _, dbName in sortedValue_ipairs(getExternalDatabaseNames())
    do
        local key = "show" .. dbName
        if self.db.profile[key] == nil
        then
            self.db.profile[key] = true     -- default is to show
        end

        Cartographer.options.args.Sidelist.args.showDBs.args[dbName] =
        {
            name = dbName,
            desc = "Toggle showing " .. dbName,
            type = "toggle",
            get = function() return self.db.profile[key] end,
            set = function(v)
                    self.db.profile[key] =
                        not self.db.profile[key] 
                end,
        }
    end

    -- If the sidelist was open when the map was closed, make sure it's
    -- open.
    showTablet(self, self.db.profile.isOpen)
end     -- Cartographer_Sidelist:Cartographer_MapOpened


-- Called when map is closed.
function Cartographer_Sidelist:Cartographer_MapClosed()
    -- Save current sidelist state
    self.db.profile.isOpen = sidelistIsOpen

    showTablet(self, false)
end     -- Cartographer_Sidelist:Cartographer_MapClosed


-- Draw tablet when tablet is requested (on left side).
function Cartographer_Sidelist:OnCartographerRightTabletRequest()
    drawTablet(self)
end     -- Cartographer_Sidelist:OnCartographerRightTabletRequest


-- Draw tablet when tablet is requested (on left side).
function Cartographer_Sidelist:OnCartographerLeftTabletRequest()
    drawTablet(self)
end     -- Cartographer_Sidelist:OnCartographerLeftTabletRequest


-- On map zone change, redraw the tablet according to the new visible zone.
-- Since this is also fired when the map opens, we have to make sure we're
-- obeying whether or not the tablet was already shown and keep it that way,
-- otherwise, don't show it.
function Cartographer_Sidelist:Cartographer_ChangeZone()
    showTablet(self, sidelistIsOpen)
end     -- Cartographer_Sidelist:Cartographer_ChangeZone


-- Called when a note is added anywhere so we can refresh the sidelist, if
-- open.
function Cartographer_Sidelist:CartographerNotes_NoteSet(zone,
    x, y, icon, creator, setNoteFromComm)

    if sidelistIsOpen
    then
        showTablet(self, true)
    end
end     -- Cartographer_Sidelist:CartographerNotes_NoteSet


-- Called when a note is deleted anywhere, so we can refresh the sidelist,
-- if open.
function Cartographer_Sidelist:CartographerNotes_NoteDeleted(zone,
    x, y, icon, db)

    if sidelistIsOpen
    then
        showTablet(self, true)
    end
end     -- Cartographer_Sidelist:CartographerNotes_NoteDeleted
