﻿--[[
********************************************************************************
Cartographer_Routes
25 January 2008
(Written for live servers v2.3.3.7799)

Author: Xaros @ EU Doomhammer Alliance & Xinhuan @ US Blackrock Alliance
********************************************************************************

Description:
	Cartographer_Routes is a module for Cartographer (Rock version). It will
	allow you to draw lines on the worldmap linking nodes together into an
	efficient farming route from existing databases. The route will be shown
	(by default) on the minimap and zone map as well.

Features:
	- Select node-types to build a line upon. The following are supported
	   * Cartographer_Fishing
	   * Cartographer_Mining
	   * Cartographer_Herbalism
	   * Cartographer_ExtractGas
	   * Cartographer_Treasure
	- Optimize your route using the traveling salesmen problem (TSP) ant
	  colony optimization (ACO) algorithm
	- Background (nonblocking) and foreground (blocking) optimization
	- Select color/thickness/transparancy/visibility for each route
	- For any route created, finding a new node will try to add that as
	  optimal as possible
	- Fubar plugin available to quickly access your routes
	- Cartographer_Waypoints support for quickly following a route

Download:
	Latest version should always be available at:
	http://files.wowace.com/Cartographer_Routes/Cartographer_Routes.zip
	http://files.wowace.com/FuBar_RoutesFu/FuBar_RoutesFu.zip

Contact:
	If you find any bugs or have any suggestions, you can contact us on:

	Forum: http://www.wowace.com/forums/index.php?topic=9300.0
	Wiki : http://wiki.wowace.com/wiki/Cartographer_Routes
	IRC  : Grum or Xinhuan on irc://irc.freenode.org/wowace
	Email: Grum ( cartographer_routes AT grum DOT nl )
	       Xinhuan ( xinhuan AT gmail DOT com )
	       Paypal donations are welcome ;)

Credits:
	Trell_  - first pre-alpha tester
	vhaarr  - suggestions for configuration
	|Stan|  - deDE translation
	Gnarfoz - deDE translation

--]]

local Cartographer = Cartographer
Cartographer_Routes = Cartographer:NewModule("Routes", "LibRockHook-1.0", "LibRockEvent-1.0", "LibRockTimer-1.0", "LibRockDB-1.0", "LibRockConsole-1.0", "LibRockConfig-1.0")

local Rock = Rock
local L = Rock("LibRockLocale-1.0"):GetTranslationNamespace("Cartographer_Routes")
Cartographer_Routes.L = L
local argCheck = Rock:GetContractFunctions("Cartographer", "argCheck")
local G = {} -- was Graph-1.0, but we removed the dependency
local T = Rock("LibTourist-3.0")
local BZ = Rock("LibBabble-Zone-3.0"):GetUnstrictLookupTable()
local BZR = Rock("LibBabble-Zone-3.0"):GetReverseLookupTable()
local BH = Rock("Babble-Herbs-2.2")
local BF = Rock("Babble-Fish-2.2")
local BO = Rock("Babble-Ore-2.2")
local CT = LibStub:GetLibrary("AceLocale-2.2", 1)
if CT then CT = CT:new("Cartographer_Treasure") end
local BG = AceLibrary("Babble-Gas-2.2")

-- used to translate the datadropdown
local translate_type = {
	Herbalism = BH,
	Mining = BO,
	Fishing = BF,
	Treasure = CT,
	ExtractGas = BG,
}

-- speedup localizations
local Cartographer_Routes = Cartographer_Routes
local db
local format = string.format
local math_abs = math.abs
local math_sin = math.sin
local math_cos = math.cos
local pairs = pairs
local tinsert, tremove = tinsert, tremove
local Minimap = Minimap
local WorldMapFrame = WorldMapFrame
local WorldMapButton = WorldMapButton

-- Set up storage space
Cartographer_Routes:SetDatabase("Cartographer_RoutesDB")

-- possible default values for profile
Cartographer_Routes:SetDatabaseDefaults("account", {
	-- stored routes
	routes = {
		['*'] = { -- zone name
			['*'] = { -- route name
				route           = {},    -- point, point, point
				color           = nil,   -- defaults to db.defaults.color if nil
				width           = nil,   -- defaults to db.defaults.width if nil
				width_minimap   = nil,   -- defaults to db.defaults.width_minimap if nil
				width_battlemap = nil,   -- defaults to db.defaults.width_battlemap if nil
				hidden          = false, -- boolean
				looped          = 1,     -- looped? 1 is used (instead of true) because initial early code used 1 inside route creation code
				visible         = true,  -- visible?
				length          = 0,     -- length
				source          = {
					['**'] = {       -- Database
						['**'] = false -- Node
					},
				},
			},
		},
	},
	edit_routes = {
		['*'] = nil -- zone name
	},
	defaults = {            --    r,    g,    b,   a
		color           = {   1, 0.75, 0.75,   1 },
		hidden_color    = {   1,    1,    1, 0.5 },
		width           = 35,
		width_minimap   = 30,
		width_battlemap = 15,
		show_hidden     = false,
		update_distance = 1,
		fake_point      = -1,
		fake_data       = 'dummy',
		draw_minimap    = 1,
		draw_worldmap   = 1,
		draw_battlemap  = 1,
		tsp = {
			initial_pheromone  = 0,     -- Initial pheromone trail value
			alpha              = 1,     -- Likelihood of ants to follow pheromone trails (larger value == more likely)
			beta               = 6,     -- Likelihood of ants to choose closer nodes (larger value == more likely)
			local_decay        = 0.2,   -- Governs local trail decay rate [0, 1]
			local_update       = 0.4,   -- Amount of pheromone to reinforce local trail update by
			global_decay       = 0.2,   -- Governs global trail decay rate [0, 1]
			twoopt_passes      = 3,		-- Number of times to perform 2-opt passes
			two_point_five_opt = false, -- Perform optimized 2-opt pass
		},
		node_databases = {
			Herbalism = true,
			Mining = true,
			Fishing = true,
			Treasure = true,
			ExtractGas = true,
		},
		prof_options = {
			Herbalism = "Always",
			Mining = "Always",
			Fishing = "Always",
			Treasure = "Always",
			ExtractGas = "Always",
		},
		use_auto_showhide = false,
		waypoint_hit_distance = 50,
		line_gaps = true,
	},
})

local prof_options = {
	["Always"] = L["Always show"],
	["With Profession"] = L["Only show when you have the profession"],
	["When active"] = L["Only show when tracking skill is active"],
	["Never"] = L["Never show"],
}
local prof_options2 = { -- For Treasure, which isn't a profession
	["Always"] = L["Always show"],
	["When active"] = L["Only show when tracking skill is active"],
	["Never"] = L["Never show"],
}
local prof_options3 = { -- For Gas, which doesn't have tracking as a skill
	["Always"] = L["Always show"],
	["With Profession"] = L["Only show when you have the profession"],
	["Never"] = L["Never show"],
}
local have_prof = {
	Herbalism = false,
	Mining = false,
	Fishing = false,
	ExtractGas = false, -- Engineering
}
local texture_to_profession = {
	["Interface\\Icons\\Spell_Nature_Earthquake"] = "Mining",
	["Interface\\Icons\\INV_Misc_Flower_02"] = "Herbalism",
	["Interface\\Icons\\INV_Misc_Fish_02"] = "Fishing",
	["Interface\\Icons\\Racial_Dwarf_FindTreasure"] = "Treasure",
}
local active_tracking

local function deep_copy_table(a, b)
	for k, v in pairs(b) do
		if type(v) == "table" then
			--a[k] = {} -- no need this, AceDB defaults should handle it
			deep_copy_table(a[k], v)
		else
			a[k] = v
		end
	end
end

local function DrawWorldmapLines()
	local zone = Cartographer:GetCurrentEnglishZoneName()
	Rock("LibRockConfig-1.0"):RefreshConfigMenu(Cartographer)

	local frame = zone == Cartographer:GetCurrentInstance() and Cartographer:GetInstanceWorldMapButton() or WorldMapButton
	local fh, fw = frame:GetHeight(), frame:GetWidth() -- Mapframe height and width
	local BattlefieldMinimap = BattlefieldMinimap  -- local reference if it exists
	local bfh, bfw  -- BattlefieldMinimap height and width
	local defaults = db.defaults

	G:HideLines(frame)
	if (BattlefieldMinimap) then
		-- The Blizzard addon "Blizzard_BattlefieldMinimap" is loaded
		G:HideLines(BattlefieldMinimap)
		bfh, bfw = BattlefieldMinimap:GetHeight(), BattlefieldMinimap:GetWidth()
	end

	local flag1 = db.defaults.draw_worldmap and WorldMapFrame:IsShown() -- Draw worldmap lines?
	local flag2 = db.defaults.draw_battlemap and BattlefieldMinimap and BattlefieldMinimap:IsShown() -- Draw battlemap lines?
	if (not flag1) and (not flag2) then
		-- Nothing to draw
		return
	end
	
	for route_name, route_data in pairs( db.routes[zone] ) do
		if type(route_data) == "table" and type(route_data.route) == "table" and #route_data.route > 1 then
			local width = route_data.width or defaults.width
			local halfwidth = route_data.width_battlemap or defaults.width_battlemap
			local color = route_data.color or defaults.color

			if (not route_data.hidden and (route_data.visible or not defaults.use_auto_showhide)) or defaults.show_hidden then
				if route_data.hidden then color = defaults.hidden_color end
				local last_point
				local sx, sy
				if route_data.looped then
					last_point = route_data.route[ #route_data.route ]
					sx, sy = Cartographer_Notes.getXY( last_point )
					sy = (1 - sy)
				end
				for i = 1, #route_data.route do
					local point = route_data.route[i]
					if point == defaults.fake_point then
						point = nil
					end
					if last_point and point then
						local ex, ey = Cartographer_Notes.getXY( point )
						ey = (1 - ey)
						if (flag1) then
							G:DrawLine(frame, sx*fw, sy*fh, ex*fw, ey*fh, width, color , "OVERLAY")
						end
						if (flag2) then
							G:DrawLine(BattlefieldMinimap, sx*bfw, sy*bfh, ex*bfw, ey*bfh, halfwidth, color , "OVERLAY")
						end
						sx, sy = ex, ey
					end
					last_point = point
				end
			end
		end
	end
end
Cartographer_Routes.DrawWorldmapLines = DrawWorldmapLines

local MinimapShapes = {
	-- quadrant booleans (same order as SetTexCoord)
	-- {upper-left, lower-left, upper-right, lower-right}
	-- true = rounded, false = squared
	["ROUND"]                 = { true,  true,  true,  true},
	["SQUARE"]                = {false, false, false, false},
	["CORNER-TOPLEFT"]        = { true, false, false, false},
	["CORNER-TOPRIGHT"]       = {false, false,  true, false},
	["CORNER-BOTTOMLEFT"]     = {false,  true, false, false},
	["CORNER-BOTTOMRIGHT"]    = {false, false, false,  true},
	["SIDE-LEFT"]             = { true,  true, false, false},
	["SIDE-RIGHT"]            = {false, false,  true,  true},
	["SIDE-TOP"]              = { true, false,  true, false},
	["SIDE-BOTTOM"]           = {false,  true, false,  true},
	["TRICORNER-TOPLEFT"]     = { true,  true,  true, false},
	["TRICORNER-TOPRIGHT"]    = { true, false,  true,  true},
	["TRICORNER-BOTTOMLEFT"]  = { true,  true, false,  true},
	["TRICORNER-BOTTOMRIGHT"] = {false,  true,  true,  true},
}

local outland_zones = {
	"Blade's Edge Mountains",
	"Hellfire Peninsula",
	"Nagrand",
	"Netherstorm",
	"Shadowmoon Valley",
	"Shattrath City",
	"Terokkar Forest",
	"Zangarmarsh",
}

local minimap_radius
local minimap_rotate
local indoors = 'outdoor'

local MinimapSize = { -- radius of minimap
	indoor = {
		[0] = 150,
		[1] = 120,
		[2] = 90,
		[3] = 60,
		[4] = 40,
		[5] = 25,
	},
	outdoor = {
		[0] = 233 + 1/3,
		[1] = 200,
		[2] = 166 + 2/3,
		[3] = 133 + 1/3,
		[4] = 100,
		[5] = 66 + 2/3,
	},
}

local function clearMinimap()
	G:HideLines( Minimap )
end
Cartographer_Routes.clearMinimap = clearMinimap

local function is_round( dx, dy )
	local map_shape = GetMinimapShape and GetMinimapShape() or "ROUND"

	local q = 1
	if dx > 0 then q = q + 2 end -- right side
	-- XXX Tripple check this
	if dy > 0 then q = q + 1 end -- bottom side

	return MinimapShapes[map_shape][q]
end

local function is_inside( sx, sy, cx, cy, radius )
	local dx = sx - cx
	local dy = sy - cy

	if is_round( dx, dy ) then
		return dx*dx+dy*dy <= radius*radius
	else
		return math_abs( dx ) <= radius and math_abs( dy ) <= radius
	end
end

local last_X, last_Y, last_facing = 1/0, 1/0, 1/0

-- implementation of cache - use zone in the key for an unique identifier
-- because every zone has a different X/Y location and possible yardsizes
local X_cache = {}
local Y_cache = {}
local XY_cache_mt = {
	__index = function(t, key)
		local zone, coord = (';'):split( key )
		local yardX, yardY = T:GetZoneYardSize(BZ[zone])
		local X, Y = yardX * (coord % 10001)/10000, yardY * floor(coord / 10001)/10000;

		X_cache[key] = X
		Y_cache[key] = Y

		-- figure out which one to return
		if t == X_cache then return X else return Y end
	end
}

setmetatable( X_cache, XY_cache_mt )
setmetatable( Y_cache, XY_cache_mt )

local function DrawMinimapLines(forceUpdate)
	-- Note, if this function is called by an outside source by
	-- Cartographer_Routes:DrawMinimapLines() then forceUpdate will contain
	-- "self == Cartographer_Routes" unless they call
	-- Cartographer_Routes.DrawMinimapLines()
	if not db.defaults.draw_minimap then
		clearMinimap()
		return
	end

	local _x, _y = Cartographer:GetCurrentPlayerPosition()

	-- invalid coordinates - clear map
	if not _x or not _y or _x < 0 or _x > 1 or _y < 0 or _y > 1 then
		clearMinimap()
		return
	end

	-- instance/indoor .. no routes
	if T:IsInstance(GetRealZoneText()) or indoors == "indoor" then
		clearMinimap()
		return
	end

	local defaults = db.defaults
	local cx, cy = Cartographer:PointToYards(_x, _y)

	local facing, sin, cos
	if minimap_rotate then
		facing = -MiniMapCompassRing:GetFacing()
	end

	if (not forceUpdate) and facing == last_facing and (last_X-cx)^2 + (last_Y-cy)^2 < defaults.update_distance^2 then
		-- no update!
		return
	end

	last_X = cx
	last_Y = cy
	last_facing = facing

	if minimap_rotate then
		sin = math_sin(facing)
		cos = math_cos(facing)
	end

	minimap_radius = MinimapSize[indoors][Minimap:GetZoom()]
	local radius = minimap_radius
	local radius2 = radius * radius

	local minX = cx - radius
	local maxX = cx + radius
	local minY = cy - radius
	local maxY = cy + radius

	local div_by_zero_nudge = 0.000001

	clearMinimap()

	local zone = GetRealZoneText()
	if BZR[zone] then
		zone = BZR[zone]
	end

	local minimap_w = Minimap:GetWidth()
	local minimap_h = Minimap:GetHeight()
	local scale_x = minimap_w / (radius*2)
	local scale_y = minimap_h / (radius*2)

	for route_name, route_data in pairs( db.routes[zone] ) do
		if type(route_data) == "table" and type(route_data.route) == "table" and #route_data.route > 1 then
			-- store color/width
			local width = route_data.width_minimap or defaults.width_minimap
			local color = route_data.color or defaults.color

			-- unless we show hidden
			if (not route_data.hidden and (route_data.visible or not defaults.use_auto_showhide)) or defaults.show_hidden then
				-- use this special color
				if route_data.hidden then
					color = defaults.hidden_color
				end

				-- some state data
				local last_x = nil
				local last_y = nil
				local last_inside = nil

				-- if we loop - make sure the 'last' gets filled with the right info
				if route_data.looped and route_data.route[ #route_data.route ] ~= defaults.fake_point then
					local key = format("%s;%s", zone, route_data.route[ #route_data.route ])
					last_x, last_y = X_cache[key], Y_cache[key]
					if minimap_rotate then
						local dx = last_x - cx
						local dy = last_y - cy
						last_x = cx + dx*cos - dy*sin
						last_y = cy + dx*sin + dy*cos
					end
					last_inside = is_inside( last_x, last_y, cx, cy, radius )
				end

				-- loop over the route
				for i = 1, #route_data.route do
					local point = route_data.route[i]
					local cur_x, cur_y, cur_inside

					-- if we have a 'fake point' (gap) - clear current values
					if point == defaults.fake_point then
						cur_x = nil
						cur_y = nil
						cur_inside = false
					else
						local key = format("%s;%s", zone, point)
						cur_x, cur_y = X_cache[key], Y_cache[key]
						if minimap_rotate then
							local dx = cur_x - cx
							local dy = cur_y - cy
							cur_x = cx + dx*cos - dy*sin
							cur_y = cy + dx*sin + dy*cos
						end
						cur_inside = is_inside( cur_x, cur_y, cx, cy, radius )
					end

					-- check if we have any nil values (we cant draw) and check boundingbox
					if cur_x and cur_y and last_x and last_y and not (
						( cur_x < minX and last_x < minX ) or
						( cur_x > maxX and last_x > maxX ) or
						( cur_y < minY and last_y < minY ) or
						( cur_y > maxY and last_y > maxY )
					)
					then
						-- default all to not drawing
						local draw_sx = nil
						local draw_sy = nil
						local draw_ex = nil
						local draw_ey = nil

						-- both inside - easy! draw
						if cur_inside and last_inside then
							draw_sx = last_x
							draw_sy = last_y
							draw_ex = cur_x
							draw_ey = cur_y
						else
							-- direction of line
							local dx = last_x - cur_x
							local dy = last_y - cur_y

							-- calculate point on perpendicular line
							local zx = cx - dy
							local zy = cy + dx

							-- nudge it a bit so we dont get div by 0 problems
							if dx == 0 then dx = div_by_zero_nudge end
							if dy == 0 then dy = div_by_zero_nudge end

							-- calculate intersection point
							local nd = ((cx   -last_x)*(cy-zy) - (cx-zx)*(cy   -last_y)) /
									   ((cur_x-last_x)*(cy-zy) - (cx-zx)*(cur_y-last_y))

							-- perpendicular point (closest to center on the line given)
							local px = last_x + nd * -dx
							local py = last_y + nd * -dy

							-- check range of intersect point
							local dpc_x = cx - px
							local dpc_y = cy - py

							-- distance^2 of the perpendicular point
							local lenpc = dpc_x*dpc_x + dpc_y*dpc_y

							-- the line can only intersect if the perpendicular point is at
							-- least closer than the furthest away point (one of the corners)
							if lenpc < 2*radius2 then

								-- if inside - ready to draw
								if cur_inside then
									draw_ex = cur_x
									draw_ey = cur_y
								else
									-- if we're not inside we can still be in the square - if so dont do any intersection
									-- calculations yet
									if math_abs( cur_x - cx ) < radius and math_abs( cur_y - cy ) < radius then
										draw_ex = cur_x
										draw_ey = cur_y
									else
										-- need to intersect against the square
										-- likely x/y to intersect with
										local minimap_cur_x  = cx + radius * (dx < 0 and 1 or -1)
										local minimap_cur_y  = cy + radius * (dy < 0 and 1 or -1)

										-- which intersection is furthest?
										local delta_cur_x = (minimap_cur_x - cur_x) / -dx
										local delta_cur_y = (minimap_cur_y - cur_y) / -dy

										-- dark magic - needs to be changed to positive signs whenever i can care about it
										if delta_cur_x < delta_cur_y and delta_cur_x < 0 then
											draw_ex = minimap_cur_x
											draw_ey = cur_y + -dy*delta_cur_x
										else
											draw_ex = cur_x + -dx*delta_cur_y
											draw_ey = minimap_cur_y
										end

										-- check if we didn't calculate some wonky offset - has to be inside with
										-- some slack on accuracy
										if math_abs( draw_ex - cx ) > radius*1.01 or
										   math_abs( draw_ey - cy ) > radius*1.01
										then
											draw_ex = nil
											draw_ey = nil
										end
									end

									-- we might have a round corner here - lets see if the quarter with the intersection is round
									if draw_ex and draw_ey and is_round( draw_ex - cx, draw_ey - cy ) then
										-- if we are also within the circle-range
										if lenpc < radius2 then
											-- circle intersection
											local dcx = cx - cur_x
											local dcy = cy - cur_y
											local len_dc = dcx*dcx + dcy*dcy

											local len_d = dx*dx + dy*dy
											local len_ddc = dx*dcx + dy*dcy

											-- discriminant
											local d_sqrt = ( len_ddc*len_ddc - len_d * (len_dc - radius2) )^0.5

											-- calculate point
											draw_ex = cur_x - dx * (-len_ddc + d_sqrt) / len_d
											draw_ey = cur_y - dy * (-len_ddc + d_sqrt) / len_d

											-- have to be on the *same* side of the perpendicular point else it's fake
											if (draw_ex - px)/math_abs(draw_ex - px) ~= (cur_x- px)/math_abs(cur_x - px) or
											   (draw_ey - py)/math_abs(draw_ey - py) ~= (cur_y- py)/math_abs(cur_y - py)
											then
												draw_ex = nil
												draw_ey = nil
											end
										else
											draw_ex = nil
											draw_ey = nil
										end
									end
								end

								-- if inside - ready to draw
								if last_inside then
									draw_sx = last_x
									draw_sy = last_y
								else
									-- if we're not inside we can still be in the square - if so dont do any intersection
									-- calculations yet
									if math_abs( last_x - cx ) < radius and math_abs( last_y - cy ) < radius then
										draw_sx = last_x
										draw_sy = last_y
									else
										-- need to intersect against the square
										-- likely x/y to intersect with
										local minimap_last_x = cx + radius * (dx > 0 and 1 or -1)
										local minimap_last_y = cy + radius * (dy > 0 and 1 or -1)

										-- which intersection is furthest?
										local delta_last_x = (minimap_last_x - last_x) / dx
										local delta_last_y = (minimap_last_y - last_y) / dy

										-- dark magic - needs to be changed to positive signs whenever i can care about it
										if delta_last_x < delta_last_y and delta_last_x < 0 then
											draw_sx = minimap_last_x
											draw_sy = last_y + dy*delta_last_x
										else
											draw_sx = last_x + dx*delta_last_y
											draw_sy = minimap_last_y
										end

										-- check if we didn't calculate some wonky offset - has to be inside with
										-- some slack on accuracy
										if math_abs( draw_sx - cx ) > radius*1.01 or
										   math_abs( draw_sy - cy ) > radius*1.01
										then
											draw_sx = nil
											draw_sy = nil
										end
									end

									-- we might have a round corner here - lets see if the quarter with the intersection is round
									if draw_sx and draw_sy and is_round( draw_sx - cx, draw_sy - cy ) then
										-- if we are also within the circle-range
										if lenpc < radius2 then
											-- circle intersection
											local dcx = cx - cur_x
											local dcy = cy - cur_y
											local len_dc = dcx*dcx + dcy*dcy

											local len_d = dx*dx + dy*dy
											local len_ddc = dx*dcx + dy*dcy

											-- discriminant
											local d_sqrt = ( len_ddc*len_ddc - len_d * (len_dc - radius2) )^0.5

											-- calculate point
											draw_sx = cur_x - dx * (-len_ddc - d_sqrt) / len_d
											draw_sy = cur_y - dy * (-len_ddc - d_sqrt) / len_d

											-- have to be on the *same* side of the perpendicular point else it's fake
											if (draw_sx - px)/math_abs(draw_sx - px) ~= (last_x- px)/math_abs(last_x - px) or
											   (draw_sy - py)/math_abs(draw_sy - py) ~= (last_y- py)/math_abs(last_y - py)
											then
												draw_sx = nil
												draw_sy = nil
											end
										else
											draw_sx = nil
											draw_sy = nil
										end
									end
								end
							end
						end

						if draw_sx and draw_sy and draw_ex and draw_ey then
							-- translate to left bottom corner and apply scale
							draw_sx =			 (draw_sx - minX) * scale_x
							draw_sy = minimap_h - (draw_sy - minY) * scale_y
							draw_ex =			 (draw_ex - minX) * scale_x
							draw_ey = minimap_h - (draw_ey - minY) * scale_y

							if defaults.line_gaps then
								-- shorten the line by 5 pixels (modified by scale) on endpoints inside the Minimap
								local pix = 5 * Cartographer_Notes.db.profile.minimapIconSize
								local dx = draw_sx - draw_ex
								local dy = draw_sy - draw_ey
								local l = (dx*dx + dy*dy)^0.5
								local x = pix * dx / l
								local y = pix * dy / l
								if last_inside and cur_inside and l > pix*2 then -- draw if line is 10 or more pixels
									G:DrawLine( Minimap, draw_sx-x, draw_sy-y, draw_ex+x, draw_ey+y, width, color, "ARTWORK")
								elseif last_inside and not cur_inside and l > pix then
									G:DrawLine( Minimap, draw_sx-x, draw_sy-y, draw_ex, draw_ey, width, color, "ARTWORK")
								elseif cur_inside and not last_inside and l > pix then
									G:DrawLine( Minimap, draw_sx, draw_sy, draw_ex+x, draw_ey+y, width, color, "ARTWORK")
								elseif not last_inside and not cur_inside then
									G:DrawLine( Minimap, draw_sx, draw_sy, draw_ex, draw_ey, width, color, "ARTWORK")
								end
							else
								G:DrawLine( Minimap, draw_sx, draw_sy, draw_ex, draw_ey, width, color, "ARTWORK")
							end
						end
					end

					-- store last point
					last_x = cur_x
					last_y = cur_y
					last_inside = cur_inside
				end
			end
		end
	end
end
Cartographer_Routes.DrawMinimapLines = DrawMinimapLines


-- This line by itself is necessary as function closures inside the table references itself as an upvalue
-- The actual table is defined later
local aceopts

-- This table is referenced inside CreateAceOptZoneRouteTable() defined right below this
local two_point_five_opt_table = {
	name = L["Extra optimization"],
	desc = L["ExtraOptDesc"],
	type  = 'toggle',
	order = 1301,
	get   = function() return db.defaults.tsp.two_point_five_opt end,
	set   = function(value) db.defaults.tsp.two_point_five_opt = value end,
}

-- This function creates the ace options table for a zone/route, used in function closures in aceopts[]
local function CreateAceOptZoneRouteTable(zone, route)
	local self = Cartographer_Routes

	-- Yes, return this huge table for given zone/route
	return {
		name = route, type = 'group',
		desc = route,
		args = {
			setting_group = {
				name = L['Line settings'], type = 'group',
				desc = L['Line settings'],
				groupType = 'inline',
				order = 100,
				args = {
					width = {
						name  = L['Width (Map)'], type = 'number',
						desc  = L['Width of the line in the map'],
						min   = 10,
						max   = 100,
						step  = 1,
						get   = function() return db.routes[zone][route].width or db.defaults.width end,
						set   = function(v) db.routes[zone][route].width = v; DrawWorldmapLines() end,
						order = 200,
					},
					width_minimap = {
						name  = L["Width (Minimap)"], type = 'number',
						desc  = L["Width of the line in the Minimap"],
						min   = 10,
						max   = 100,
						step  = 1,
						get   = function() return db.routes[zone][route].width_minimap or db.defaults.width_minimap end,
						set   = function(v) db.routes[zone][route].width_minimap = v; DrawMinimapLines(true) end,
						order = 210,
					},
					width_battlemap = {
						name  = L["Width (Zone Map)"], type = 'number',
						desc  = L["Width of the line in the Zone Map"],
						min   = 10,
						max   = 100,
						step  = 1,
						get   = function() return db.routes[zone][route].width_battlemap or db.defaults.width_battlemap end,
						set   = function(v) db.routes[zone][route].width_battlemap = v; DrawWorldmapLines() end,
						order = 220,
					},
					color = {
						name  = L['Color'], type = 'color',
						desc  = L['Change a color'],
						get   = function()
							local c = db.routes[zone][route].color or db.defaults.color
							return unpack( c )
						end,
						set   = function(r,g,b,a) db.routes[zone][route].color = {r,g,b,a}; DrawWorldmapLines(); DrawMinimapLines(true) end,
						order = 300,
						hasAlpha = true,
					},
					hidden = {
						name  = L['Hidden'], type = 'boolean',
						desc  = L['Route hidden?'],
						get   = function() return db.routes[zone][route].hidden end,
						set   = function(v) db.routes[zone][route].hidden = v; DrawWorldmapLines(); DrawMinimapLines(true) end,
						order = 400,
					},
					--[[
					looped = {
						name  = L['Looped'], type = 'boolean',
						desc  = L['Connect start and end for this route?'],
						get   = function() return db.routes[zone][route].looped end,
						set   = function(v) db.routes[zone][route].looped = v; DrawWorldmapLines() end,
						order = 600,
					},
					--]]
					delete = {
						name  = L['Delete'], type = 'execute',
						desc  = L['Permanently delete a route'],
						func  = function()
							local is_running, route_table = self.TSP:IsTSPRunning()
							if is_running and route_table == db.routes[zone][route].route then
								self:Print(L["You may not delete a route that is being optimized in the background."])
								return
							end
							db.routes[zone][route] = nil
							aceopts[zone].args[route] = nil
							if db.edit_routes[zone] == route then
								db.edit_routes[zone] = nil
							end
							if next(db.routes[zone]) == nil then
								aceopts[zone] = nil
							end
							DrawWorldmapLines()
							DrawMinimapLines(true)
						end,
						confirmText = L["Are you sure?"],
						buttonText = L["Delete"],
						order = 1200,
					},
					reset_all = {
						name  = L['Reset'], type = 'execute',
						desc  = L['Reset to defaults'],
						func  = function()
							db.routes[zone][route].color = nil
							db.routes[zone][route].width = nil
							db.routes[zone][route].width_minimap = nil
							db.routes[zone][route].width_battlemap = nil
							DrawWorldmapLines()
							DrawMinimapLines(true)
						end,
						buttonText = L['Reset'],
						order = 1300,
					},
				},
			},
			optimize_group = {
				name = function ()
					if not db.routes[zone][route].length or db.routes[zone][route].length == 0 then
						db.routes[zone][route].length = Cartographer_Routes.TSP:PathLength(db.routes[zone][route].route, zone)
					end
					return L['Optimize route [%d nodes, %d yards]']:format(#db.routes[zone][route].route, db.routes[zone][route].length)
				end,
				type = 'group',
				desc = "Small test",
				groupType = 'inline',
				order = 1300,
				args = {
					two_point_five_opt = two_point_five_opt_table,
					foreground = {
						name  = L['Foreground'], type = 'execute',
						desc  = L['Foreground Disclaimer'],
						func  = function()
							local output, length, iter, timetaken = self.TSP:SolveTSP(db.routes[zone][route].route, zone, db.defaults.tsp)
							db.routes[zone][route].route = output
							db.routes[zone][route].length = length
							self:Print(L["Path with %d nodes found with length %.2f yards after %d iterations in %.2f seconds."]:format(#output, length, iter, timetaken))

							-- redraw lines
							DrawWorldmapLines()
							DrawMinimapLines(true)
						end,
						confirmText = L["Are you sure?"],
						buttonText = L["Optimize"],
						order = 1310,
					},
					background = {
						name  = L['Background'], type = 'execute',
						desc  = L['Background Disclaimer'],
						func  = function()
							local running, errormsg = self.TSP:SolveTSPBackground(db.routes[zone][route].route, zone, db.defaults.tsp)
							if (running == 1) then
								self:Print(L["Now running TSP in the background..."])
								self.TSP:SetFinishFunction(function(output, length, iter, timetaken)
									db.routes[zone][route].route = output
									db.routes[zone][route].length = length
									self:Print(L["Path with %d nodes found with length %.2f yards after %d iterations in %.2f seconds."]:format(#output, length, iter, timetaken))
									-- redraw lines
									DrawWorldmapLines()
									DrawMinimapLines(true)
								end)
							elseif (running == 2) then
								self:Print(L["There is already a TSP running in background. Wait for it to complete first."])
							elseif (running == 3) then
								-- This should never happen, but is here as a fallback
								self:Print(L["The following error occured in the background path generation coroutine, please report to Grum or Xinhuan:"]);
								self:Print(errormsg);
							end
						end,
						buttonText = L['Optimize'],
						order = 1320,
					},
				},
			},
		},
	}
end

-- Some upvalues used in the aceopts[] table for creating new routes
local create_name = ""
local create_choices = {}
local create_data = {}
local create_zones = {}
local create_zone
-- The following table is the aceoptions table for RockConfig for Cartographer_Routes
aceopts = {
	add_group = {
		name = L["Add"], type = "group",
		desc = L["Add"],
		order = 100,
		args = {
			route_name = {
				name = L["Name"], type = 'string',
				desc = L["Name of the route to add to %s"]:format((Cartographer:GetCurrentEnglishZoneName() or L["Unknown Zone!"])),
				get  = function() return create_name end,
				onChange = function(route_name) create_name = route_name end,
				set  = function(route_name) create_name = route_name end,
				usage = L["<new item>"],
				order = 100,
			},
			zone_choice = {
				name = L["Zone"], type = 'choice',
				desc = L["Zone to create route in"],
				order = 150,
				choices = function()
					-- reuse table
					for k in pairs(create_zones) do create_zones[k] = nil end

					for i = 1, #outland_zones do
						create_zones[ outland_zones[i] ] = BZ[ outland_zones[i] ]
					end

					-- add current player zone
					local player_zone = GetRealZoneText()
					if not BZ[player_zone] then
						if BZR[player_zone] then
							player_zone = BZR[player_zone]
						end
					end
					if player_zone and T:IsZone(player_zone) and not T:IsInstance(player_zone) then
						create_zones[ player_zone ] = BZ[ player_zone ]
						if not create_zone then create_zone = player_zone end
					end

					-- add current cartographer zone
					local carto_zone = Cartographer:GetCurrentEnglishZoneName()
					if carto_zone and T:IsZone(carto_zone) and not T:IsInstance(carto_zone) then
						create_zones[ carto_zone ] = BZ[ carto_zone ]
						if not create_zone then create_zone = carto_zone end
					end

					return create_zones
				end,
				get = function() return create_zone end,
				set = function(key) create_zone = key end,
			},
			data_choices = {
				name = L["Data"], type = 'multichoice',
				desc = L["Which nodes to use in the route"],
				order = 200,
				get = function(key)
					--Cartographer_Routes:Print(("Getting choice for: %s"):format(key or "nil"));
					if not create_zone then return end
					if key == db.defaults.fake_data then return end
					if not create_choices[create_zone] then create_choices[create_zone] = {} end
					return create_choices[create_zone][key]
				end,
				set = function(key,value)
					if not create_zone then return end
					if key == db.defaults.fake_data then return end
					if not create_choices[create_zone] then create_choices[create_zone] = {} end
					create_choices[create_zone][key] = value
					--Cartographer_Routes:Print(("Setting choice: %s to %s"):format(key or "nil", value and "true" or "false"));
				end,
				choices = function()
					if not create_zone then return {} end

					local CN = Cartographer:GetModule("Notes")

					-- reuse table
					for k in pairs(create_data) do create_data[k] = nil end

					-- for all types of nodes stored in the external node-database (herbalism/mining/quests/etc)
					for db_type,db_data in pairs(CN.externalDBs) do
						-- get the babble localization for this db type
						local LN = translate_type[ db_type ]

						-- if we selected this node-type as valid
						if db.defaults.node_databases[ db_type ] then
							local amount_of = {}
							-- only look for data for this currentzone
							if db_data[ create_zone ] then
								-- count the unique values (structure is: location => item)
								if db_type == "Treasure" then
									for _,node in pairs(db_data[create_zone]) do
										amount_of[node.title] = (amount_of[node.title] or 0) + 1
									end
								else
									for _,node in pairs(db_data[create_zone]) do
										amount_of[node] = (amount_of[node] or 0) + 1
									end
								end

								-- XXX Localize these strings
								-- store combinations with all information we have
								for node,count in pairs(amount_of) do
									local translatednode = LN:HasTranslation(node) and LN[node] or node
									create_data[ ("%s;%s;%s"):format(db_type, node, count) ] = ("%s - %s - %d"):format(L[db_type],translatednode,count)
								end
							end
						end
					end

					-- found no data - insert dummy message
					if not next(create_data) then
						create_data[ db.defaults.fake_data ..";;" ] = L["No Herbalism/Mining/Fishing data found"]
					end

					return create_data
				end,
			},
			add_route = {
				name = L["Add"], type = 'execute',
				desc = L["Add"],
				func = function()
					create_name = strtrim( create_name )
					if not create_name or create_name == "" then
						Cartographer_Routes:Print(L["No name given for new route"]);
						return
					end

					local CN = Cartographer:GetModule("Notes")
					local zone = create_zone

					--the real 'action', we use a temporary table in case of data corruption and only commit this to the db if successful
					local new_route = { route = {}, source = {} }

					-- if for every selected nodetype on this map
					if type(create_choices[zone]) == "table" then
						for data_string,wanted in pairs(create_choices[zone]) do
							-- if we want em
							if (wanted) then
								local db_type, node_type, amount = (';'):split( data_string );
								--Cartographer_Routes:Print(("found %s %s %s"):format( db,node_type,amount ))

								-- ignore any fake data, and check if the db_type exists (could be disabled via _Professions)
								if db_type ~= db.defaults.fake_data and CN.externalDBs[db_type] then
									-- store the sources of the data so we can auto-add them
									if type(new_route.source[db_type]) ~= "table" then
										new_route.source[db_type] = {}
									end
									new_route.source[db_type][node_type] = true

									-- Find all of the notes
									for loc, t in pairs(CN.externalDBs[db_type][zone]) do
										-- And are of a selected type - store
										if db_type == "Treasure" and t.title == node_type then
											tinsert( new_route.route, loc )
										elseif t == node_type then
											tinsert( new_route.route, loc )
										end
									end
								end
							end
						end
					end

					if #new_route.route == 0 then
						Cartographer_Routes:Print(L["No data selected for new route"]);
						return
					end

					--db.routes[zone][create_name] = new_route
					-- Perform a deep copy instead so that db defaults apply
					deep_copy_table(db.routes[zone][create_name], new_route)

					-- TODO Check if we can do a one-pass TSP run here as well if the user selected it.
					db.routes[zone][create_name].length = Cartographer_Routes.TSP:PathLength(new_route.route, zone)
					db.edit_routes[zone] = create_name

					-- Create the aceopts table entry for our new or overwritten route
					if not aceopts[zone] then
						aceopts[zone] = {
							name = BZ[zone] or zone, type = 'group',
							desc = L['Routes in %s']:format(BZ[zone] or zone),
							args = {},
							order = 1000,
						}
					end
					aceopts[zone].args[create_name] = CreateAceOptZoneRouteTable(zone, create_name)

					-- Now scroll the menu to our new aceopt entry
					if Cartographer.OpenConfigMenu then
						Cartographer:OpenConfigMenu("Routes", zone, create_name)
					end

					-- Draw it
					DrawWorldmapLines()

					-- clear stored name
					create_name = ""
					create_zone = nil
				end,
				confirmText = function()
					if not create_name or create_name == "" then return nil end
					local zone = Cartographer:GetCurrentEnglishZoneName()

					-- route exists
					if type(db.routes[zone][create_name].route) == "table" and #db.routes[zone][create_name].route > 0 then
						return L["Exists, overwrite?"]
					else return nil end
				end,
				usage = L["<new item>"],
				order = 300,
			},

		},
	},
	default_group = {
		name = L["Normal lines"], type = "group",
		desc = L["Normal lines"],
		groupType = "inline",
		order = 200,
		args = {
			width = {
				name = L["Width (Map)"], type = 'number',
				desc = L["Width of the line in the map"],
				min  = 10,
				max  = 100,
				step = 1,
				get  = function() return db.defaults.width end,
				set  = function(v) db.defaults.width = v; DrawWorldmapLines() end,
				order = 100,
			},
			width_minimap = {
				name = L["Width (Minimap)"], type = 'number',
				desc = L["Width of the line in the Minimap"],
				min  = 10,
				max  = 100,
				step = 1,
				get  = function() return db.defaults.width_minimap end,
				set  = function(v) db.defaults.width_minimap = v; DrawWorldmapLines() end,
				order = 110,
			},
			width_battlemap = {
				name = L["Width (Zone Map)"], type = 'number',
				desc = L["Width of the line in the Zone Map"],
				min  = 10,
				max  = 100,
				step = 1,
				get  = function() return db.defaults.width_battlemap end,
				set  = function(v) db.defaults.width_battlemap = v; DrawWorldmapLines() end,
				order = 120,
			},
			color = {
				name = L['Color'], type = 'color',
				desc = L['Change default route color'],
				get  = function() return unpack( db.defaults.color ) end,
				set  = function(r,g,b,a) db.defaults.color = {r,g,b,a}; DrawWorldmapLines() end,
				hasAlpha = true,
				order = 200,
			},
		},
	},
	hidden_group = {
		name = L["Hidden lines"], type = "group",
		desc = L["Hidden lines"],
		groupType = "inline",
		order = 300,
		args = {
			show_hidden = {
				name = L['Show'], type = 'boolean',
				desc = L['Show hidden routes?'],
				get  = function() return db.defaults.show_hidden end,
				set  = function(v) db.defaults.show_hidden = v; DrawWorldmapLines(); DrawMinimapLines(true) end,
				order = 300,
			},
			hidden_color = {
				name = L['Color'], type = 'color',
				desc = L['Change default hidden route color'],
				get  = function() return unpack( db.defaults.hidden_color ) end,
				set  = function(r,g,b,a) db.defaults.hidden_color = {r,g,b,a}; DrawWorldmapLines(); DrawMinimapLines(true) end,
				hasAlpha = true,
				order = 400,
			},
		},
	},
	minimap_group = {
		name = L["Minimap lines"], type = "group",
		desc = L["Minimap lines"],
		groupType = "inline",
		order = 400,
		args = {
			update_distance = {
				name = L['Update distance'], type = 'number',
				desc = L['Yards to move before triggering a minimap update'],
				min  = 0,
				max  = 10,
				step = 0.1,
				get  = function() return db.defaults.update_distance end,
				set  = function(v) db.defaults.update_distance = v end,
				order = 500,
			},
			line_gaps = {
				name = L["Line gaps"], type = 'boolean',
				desc = L["Shorten the lines drawn on the minimap slightly so that they do not overlap the icons and minimap tracking blips."],
				get  = function() return db.defaults.line_gaps end,
				set  = function(v) db.defaults.line_gaps = v; DrawMinimapLines(true) end,
				order = 600,
			},
		},
	},
	drawing = {
		name = L["Map Drawing"], type = "group",
		desc = L["Map Drawing"],
		groupType = "inline",
		order = 500,
		args = {
			worldmap_toggle = {
				name = L["Worldmap drawing"],
				desc = L["Worldmap drawing"],
				type  = 'toggle',
				order = 500,
				get  = function() return db.defaults.draw_worldmap end,
				set  = function(v) db.defaults.draw_worldmap = v; DrawWorldmapLines() end,
			},
			minimap_toggle = {
				name = L["Minimap drawing"],
				desc = L["Minimap drawing"],
				type  = 'toggle',
				order = 600,
				get  = function() return db.defaults.draw_minimap end,
				set  = function(v)
					db.defaults.draw_minimap = v
					if v then
						Cartographer_Routes:AddEventListener("MINIMAP_UPDATE_ZOOM")
						Cartographer_Routes:AddEventListener("CVAR_UPDATE")
						Cartographer_Routes:AddRepeatingTimer("Cartographer-Routes-MinimapRoutes", 0, DrawMinimapLines)
						Cartographer_Routes:AddEventListener("MINIMAP_ZONE_CHANGED", DrawMinimapLines, nil, true)
						minimap_rotate = GetCVar("rotateMinimap") == "1"
						Cartographer_Routes:MINIMAP_UPDATE_ZOOM()
					else
						Cartographer_Routes:RemoveEventListener("MINIMAP_UPDATE_ZOOM")
						Cartographer_Routes:RemoveEventListener("CVAR_UPDATE")
						Cartographer_Routes:RemoveTimer("Cartographer-Routes-MinimapRoutes")
						Cartographer_Routes:RemoveEventListener("MINIMAP_ZONE_CHANGED")
						clearMinimap()
					end
				end,
			},
			battlemap_toggle = {
				name = L["Zone map drawing"],
				desc = L["Zone map drawing"],
				type  = 'toggle',
				order = 700,
				get  = function() return db.defaults.draw_battlemap end,
				set  = function(v) db.defaults.draw_battlemap = v; DrawWorldmapLines() end,
			},
		},
	},
	auto_group = {
		name = L["Auto Show/Hide Routes"], type = "group",
		desc = L["Auto show and hide routes based on your professions"],
		groupType = "inline",
		order = 600,
		args = {
			use_auto_showhide = {
				name = L["Use Auto Show/Hide"],
				desc = L["Use Auto Show/Hide"],
				type  = 'toggle',
				order = 700,
				get  = function() return db.defaults.use_auto_showhide end,
				set  = function(v)
					db.defaults.use_auto_showhide = v
					if v then
						Cartographer_Routes:AddEventListener("SKILL_LINES_CHANGED")
						Cartographer_Routes:AddEventListener("MINIMAP_UPDATE_TRACKING")
						Cartographer_Routes:MINIMAP_UPDATE_TRACKING()
						Cartographer_Routes:SKILL_LINES_CHANGED()
						aceopts.auto_group.args.fishing.disabled = nil
						aceopts.auto_group.args.herbalism.disabled = nil
						aceopts.auto_group.args.mining.disabled = nil
						aceopts.auto_group.args.treasure.disabled = nil
						aceopts.auto_group.args.gas.disabled = nil
					else
						Cartographer_Routes:RemoveEventListener("SKILL_LINES_CHANGED")
						Cartographer_Routes:RemoveEventListener("MINIMAP_UPDATE_TRACKING")
						aceopts.auto_group.args.fishing.disabled = true
						aceopts.auto_group.args.herbalism.disabled = true
						aceopts.auto_group.args.mining.disabled = true
						aceopts.auto_group.args.treasure.disabled = true
						aceopts.auto_group.args.gas.disabled = true
					end
				end,
			},
			fishing = {
				name = L["Routes with Fish"], type = 'choice',
				desc = L["Routes with Fish"],
				order = 710,
				choices = prof_options,
				get = function() return db.defaults.prof_options.Fishing end,
				set = function(key) db.defaults.prof_options.Fishing = key; Cartographer_Routes:ApplyVisibility() end,
			},
			gas = {
				name = L["Routes with Gas"], type = 'choice',
				desc = L["Routes with Gas"],
				order = 715,
				choices = prof_options3,
				get = function() return db.defaults.prof_options.ExtractGas end,
				set = function(key) db.defaults.prof_options.ExtractGas = key; Cartographer_Routes:ApplyVisibility() end,
			},
			herbalism = {
				name = L["Routes with Herbs"], type = 'choice',
				desc = L["Routes with Herbs"],
				order = 720,
				choices = prof_options,
				get = function() return db.defaults.prof_options.Herbalism end,
				set = function(key) db.defaults.prof_options.Herbalism = key; Cartographer_Routes:ApplyVisibility() end,
			},
			mining = {
				name = L["Routes with Ore"], type = 'choice',
				desc = L["Routes with Ore"],
				order = 730,
				choices = prof_options,
				get = function() return db.defaults.prof_options.Mining end,
				set = function(key) db.defaults.prof_options.Mining = key; Cartographer_Routes:ApplyVisibility() end,
			},
			treasure = {
				name = L["Routes with Treasure"], type = 'choice',
				desc = L["Routes with Treasure"],
				order = 740,
				choices = prof_options2,
				get = function() return db.defaults.prof_options.Treasure end,
				set = function(key) db.defaults.prof_options.Treasure = key; Cartographer_Routes:ApplyVisibility() end,
			},
		},
	},
	waypoints = {
		name = L["Cartographer_Waypoints support"], type = "group",
		desc = L["Integrated support options for Cartographer_Waypoints"],
		groupType = "inline",
		order = 700,
		args = {
			hit_distance = {
				name = L["Waypoint hit distance"], type = 'number',
				desc = L["This is the distance in yards away from a waypoint to consider as having reached it so that the next node in the route can be added as the waypoint"],
				min  = 5,
				max  = 80,  -- This is the maximum range of node detection for "Find X" profession skills
				step = 1,
				get  = function() return db.defaults.waypoint_hit_distance end,
				set  = function(v) db.defaults.waypoint_hit_distance = v end,
				order = 700,
			},
			start = {
				name  = L["Start using Waypoints"], type = 'execute',
				desc  = L["Start using Cartographer_Waypoints by finding the closest visible route/node in the current zone and using that as the waypoint"],
				func  = function() Cartographer_Routes:QueueFirstNode() end,
				buttonText = L["Start using Waypoints"],
				order = 710,
			},
			direction = {
				name  = L["Change direction"], type = 'execute',
				desc  = L["Change the direction of the nodes in the route being added as the next waypoint"],
				func  = function() Cartographer_Routes:ChangeWaypointDirection() end,
				buttonText = L["Change direction"],
				order = 720,
			},
			stop = {
				name  = L["Stop using Waypoints"], type = 'execute',
				desc  = L["Stop using Cartographer_Waypoints by clearing the last queued node"],
				func  = function() Cartographer_Routes:RemoveQueuedNode() end,
				buttonText = L["Stop using Waypoints"],
				order = 730,
			},
		},
	},
	toggle = {
		name = L["Enabled"],
		desc = L["Suspend/resume this module."],
		type  = 'toggle',
		order = -1,
		get   = function() return Cartographer:IsModuleActive(Cartographer_Routes) end,
		set   = function() Cartographer:ToggleModuleActive(Cartographer_Routes) end,
	},
}

local function SetZoomHook()
	DrawMinimapLines(true)
end

function Cartographer_Routes:MINIMAP_UPDATE_ZOOM()
	self:RemoveHook(Minimap, "SetZoom")
	local zoom = Minimap:GetZoom()
	if GetCVar("minimapZoom") == GetCVar("minimapInsideZoom") then
		Minimap:SetZoom(zoom < 2 and zoom + 1 or zoom - 1)
	end
	indoors = GetCVar("minimapZoom")+0 == Minimap:GetZoom() and "outdoor" or "indoor"
	Minimap:SetZoom(zoom)
	DrawMinimapLines(true)
	self:AddSecureHook(Minimap, "SetZoom", SetZoomHook)
end

local function NoteSetEvent(namespace, event, zone, x, y, node, db_type )
	if db_type == "Treasure" then
		node = Cartographer:GetModule("Notes").externalDBs.Treasure[zone][Cartographer_Notes.getID(x,y)].title
	end
	local modified = false
	for route_name, route_data in pairs( db.routes[zone] ) do
		-- for every route check if we had this node as a source
		if route_data.source and route_data.source[db_type] and route_data.source[db_type][node] then
			-- Add the node
			route_data.route, route_data.length = Cartographer_Routes.TSP:InsertNode(route_data.route, zone, Cartographer_Notes.getID(x,y), false)
			modified = true
		end
	end
	if modified then
		-- redraw worldmap + minimap
		DrawWorldmapLines()
		DrawMinimapLines(true)
	end
end

local function NoteDeleteEvent(namespace, event, zone, x, y, node, db_type )
	local id = Cartographer_Notes.getID(x,y)
	--[[ Can't get the data if its already deleted!
	if db_type == "Treasure" then
		node = Cartographer:GetModule("Notes").externalDBs.Treasure[zone][Cartographer_Notes.getID(x,y)].title
	end]]
	local modified = false
	for route_name, route_data in pairs( db.routes[zone] ) do
		-- for every route check if we had this node as a source, or just assume its used if its the Treasure database
		if route_data.source and route_data.source[db_type] and (route_data.source[db_type][node] or db_type == "Treasure") then
			-- Delete the node
			for i = 1, #route_data.route do
				if id == route_data.route[i] then
					tremove(route_data.route, i)
					route_data.length = Cartographer_Routes.TSP:PathLength(route_data.route, zone)
					modified = true
					break
				end
			end

		end
	end
	if modified then
		-- redraw worldmap + minimap
		DrawWorldmapLines()
		DrawMinimapLines(true)
	end
end

function Cartographer_Routes:OnInitialize()
	-- Localized so i dont have to do: self.db.account all the time.
	db = self.db.account

	-- generate menus
	for zone, zone_table in pairs(db.routes) do
		-- do not show unless we have routes.
		-- This because lua cant do '#' on hash-tables
		if next(zone_table) ~= nil then
			aceopts[zone] = {
				name = BZ[zone] or zone, type = 'group',
				desc = L['Routes in %s']:format(BZ[zone] or zone),
				args = {},
				order = 1000,
			}
			for route in pairs(zone_table) do
				aceopts[zone].args[route] = CreateAceOptZoneRouteTable(zone, route)
			end
		end
	end

	-- dropdown menu configuration for Cartographer addon menu
	Cartographer.options.args.Routes = {
		name = L["Routes"],
		desc = L["Module description"],
		type = 'group',
		args = aceopts,
		handler = self,
	}
	-- make options available via /xxxxx
	--AceLibrary("AceConsole-2.0"):InjectAceOptionsTable(self, Cartographer.options.args.Routes)
end

function Cartographer_Routes:CVAR_UPDATE(caller, event, cvar, value)
	if cvar == "ROTATE_MINIMAP" then
		minimap_rotate = value == "1"
	end
end

function Cartographer_Routes:PLAYER_ENTERING_WORLD()
	-- Nag message
	self:Print("Cartographer_Routes is no longer being maintained. Please delete this addon and download Routes instead.")
	self:Print("Routes is the replacement upgrade addon with newer and better features such as node collapsing, taboo regions, faster route generation and supports GatherMate/Gatherer/Cartographer_<Profs>.")
end

function Cartographer_Routes:OnEnable()
	-- Called when the addon is enabled
	if not Cartographer:HasModule("Notes") then
		self:Print(L["Routes requires the Notes module to operate."])
		Cartographer:ToggleModuleActive(self, false)
		return
	end

	self:AddSecureHook("ToggleWorldMap", DrawWorldmapLines)
	self:AddSecureHook(Minimap, "SetZoom", SetZoomHook)
	self:AddEventListener("WORLD_MAP_UPDATE", DrawWorldmapLines)
	self:AddEventListener("Cartographer_SetCurrentInstance", DrawWorldmapLines)
	-- Minimap line drawing
	if db.defaults.draw_minimap then
		self:AddEventListener("MINIMAP_UPDATE_ZOOM")
		self:AddEventListener("CVAR_UPDATE")
		self:AddRepeatingTimer("Cartographer-Routes-MinimapRoutes", 0, DrawMinimapLines)
		self:AddEventListener("MINIMAP_ZONE_CHANGED", DrawMinimapLines, nil, true)
		minimap_rotate = GetCVar("rotateMinimap") == "1"
		indoors = "indoor"
		-- Notes: Do not call self:MINIMAP_UPDATE_ZOOM() here because the CVARs aren't applied yet.
		-- MINIMAP_UPDATE_ZOOM gets fired automatically by wow when it applies the CVARs.
	end
	-- Hooking adding of new notes
	self:AddEventListener(Cartographer_Notes.name, "NoteSet", NoteSetEvent)
	-- Hooking deleting of old notes
	self:AddEventListener(Cartographer_Notes.name, "NoteDeleted", NoteDeleteEvent)

	-- For profession stuff
	if db.defaults.use_auto_showhide then
		self:AddEventListener("SKILL_LINES_CHANGED")
		self:AddEventListener("MINIMAP_UPDATE_TRACKING")
		self:MINIMAP_UPDATE_TRACKING()
		self:SKILL_LINES_CHANGED()
	else
		aceopts.auto_group.args.fishing.disabled = true
		aceopts.auto_group.args.herbalism.disabled = true
		aceopts.auto_group.args.mining.disabled = true
		aceopts.auto_group.args.treasure.disabled = true
		aceopts.auto_group.args.gas.disabled = true
	end

	if WorldMapFrame:IsShown() then
		DrawWorldmapLines()
	end
	self:AddEventListener("PLAYER_ENTERING_WORLD")
end

function Cartographer_Routes:OnDisable()
	-- Called when the addon is disabled
	self:RemoveAllEventListeners()
	self:RemoveAllHooks()
	local frame = zone == Cartographer:GetCurrentInstance() and Cartographer:GetInstanceWorldMapButton() or WorldMapButton
	G:HideLines(frame)
	clearMinimap()
	if BattlefieldMinimap then
		G:HideLines(BattlefieldMinimap)
	end
end

function Cartographer_Routes:SKILL_LINES_CHANGED()
	local skillname, isHeader
	for i = 1, GetNumSkillLines() do
		skillname, isHeader = GetSkillLineInfo(i)
		if not isHeader and skillname then
			if strfind(skillname, GetSpellInfo(7620), 1, true) then
				have_prof.Fishing = true
			elseif strfind(skillname, GetSpellInfo(9134), 1, true) then
				have_prof.Herbalism = true
			elseif strfind(skillname, GetSpellInfo(2575), 1, true) then
				have_prof.Mining = true
			elseif strfind(skillname, GetSpellInfo(4036), 1, true) then
				have_prof.ExtractGas = true
			end
		end
	end
	self:ApplyVisibility()
end

function Cartographer_Routes:MINIMAP_UPDATE_TRACKING()
	active_tracking = texture_to_profession[GetTrackingTexture()]
	self:ApplyVisibility()
end

function Cartographer_Routes:ApplyVisibility()
	local modified = false
	for zone, zone_table in pairs(db.routes) do -- for each zone
		if next(zone_table) ~= nil then
			for route_name, route_data in pairs(zone_table) do -- for each route
				if route_data.source then
					local visible = false
					for db_type in pairs(route_data.source) do -- for each db type used
						local status = db.defaults.prof_options[db_type]
						if status == "Always" then
							visible = true
						elseif status == "With Profession" and have_prof[db_type] then
							visible = true
						elseif status == "When active" and active_tracking == db_type then
							visible = true
						--elseif status == "Never" then
						--	visible = false
						end
						if not visible == not route_data.visible then
							modified = true
						end
						route_data.visible = visible
					end
				end
			end
		end
	end
	if modified then
		-- redraw worldmap + minimap
		DrawWorldmapLines()
		DrawMinimapLines(true)
	end
end

do
	local route_table
	local route_name
	local direction = 1
	local node_num = 1
	local zone
	local stored_hit_distance
	
	function Cartographer_Routes:FindClosestVisibleRoute()
		if not (Cartographer:HasModule("Waypoints") and Cartographer:IsModuleActive("Waypoints")) then
			self:Print(L["Cartographer_Waypoints module is missing or disabled"])
			return
		end
		local zone = GetRealZoneText()
		if BZR[zone] then
			zone = BZR[zone]
		end
		local closest_zone, closest_route, closest_node
		local min_distance = 1/0
		local defaults = db.defaults
		for route_name, route_data in pairs(db.routes[zone]) do  -- for each route in current zone
			if type(route_data) == "table" and type(route_data.route) == "table" and #route_data.route > 1 then  -- if it is valid
				if (not route_data.hidden and (route_data.visible or not defaults.use_auto_showhide)) or defaults.show_hidden then  -- if it is visible
					for i = 1, #route_data.route do  -- for each node
						local x, y = Cartographer_Notes.getXY(route_data.route[i])
						local dist = Cartographer:GetDistanceToPoint(x, y, zone)
						if dist < min_distance then
							min_distance = dist
							closest_zone = zone
							closest_route = route_name
							closest_node = i
						end
					end
				end
			end
		end
		return closest_zone, closest_route, closest_node
	end

	function Cartographer_Routes:QueueFirstNode()
		if not (Cartographer:HasModule("Waypoints") and Cartographer:IsModuleActive("Waypoints")) then
			self:Print(L["Cartographer_Waypoints module is missing or disabled"])
			return
		end
		local a, b, c = self:FindClosestVisibleRoute()
		if a then
			if stored_hit_distance then
				-- We are already following a route in Waypoints
				self:RemoveQueuedNode()
			end
			zone = a
			route_name = b
			route_table = db.routes[a][b]
			node_num = c
			Cartographer_Waypoints:AddRoutesWaypoint(zone, route_table.route[node_num], L["%s - Node %d"]:format(route_name, node_num))
			self:AddEventListener(Cartographer_Waypoints.name, "WaypointHit", "WaypointHit")
			stored_hit_distance = Cartographer_Waypoints:GetWaypointHitDistance()
			Cartographer_Waypoints:SetWaypointHitDistance(db.defaults.waypoint_hit_distance)
		end
	end

	function Cartographer_Routes:WaypointHit(namespace, event, waypoint)
		if stored_hit_distance then
			-- Try to match the removed waypointID with a node in the route. This
			-- is necessary because the route could have changed dynamically from node
			-- insertion/deletion/optimization/etc causing a change to the node numbers
			local id = tonumber((gsub(waypoint.WaypointID, zone, ""))) -- Extra brackets necessary to reduce to 1 return value
			if not id then return end -- Not a waypoint from this zone
			local route = route_table.route
			for i = 1, #route do
				if id == route[i] then
					-- Match found, get the next node to waypoint
					node_num = i + direction
					if node_num > #route then
						node_num = 1
					elseif node_num < 1 then
						node_num = #route
					end
					--self:Print("Adding node "..node_num)
					Cartographer_Waypoints:AddRoutesWaypoint(zone, route[node_num], L["%s - Node %d"]:format(route_name, node_num))
					break
				end
			end
		end
	end

	function Cartographer_Routes:RemoveQueuedNode()
		if not (Cartographer:HasModule("Waypoints") and Cartographer:IsModuleActive("Waypoints")) then
			self:Print(L["Cartographer_Waypoints module is missing or disabled"])
			return
		end
		if stored_hit_distance then
			Cartographer_Waypoints:CancelWaypoint(route_table.route[node_num]..zone)
			Cartographer_Waypoints:SetWaypointHitDistance(stored_hit_distance)
			stored_hit_distance = nil
			self:RemoveEventListener("Waypoints", "WaypointHit")
		end
	end

	function Cartographer_Routes:ChangeWaypointDirection()
		direction = -direction
		self:Print(L["Direction changed"])
	end
end


-- The following function is used with permission from Daniel Stephens <iriel@vigilance-committee.org>
-- with reference to TaxiFrame.lua in Blizzard's UI and Graph-1.0 Ace2 library (by Cryect) which I now
-- maintain after porting it to LibGraph-2.0 LibStub library -- Xinhuan
local TAXIROUTE_LINEFACTOR = 128/126; -- Multiplying factor for texture coordinates
local TAXIROUTE_LINEFACTOR_2 = TAXIROUTE_LINEFACTOR / 2; -- Half of that

-- T        - Texture
-- C        - Canvas Frame (for anchoring)
-- sx,sy    - Coordinate of start of line
-- ex,ey    - Coordinate of end of line
-- w        - Width of line
-- relPoint - Relative point on canvas to interpret coords (Default BOTTOMLEFT)
function G:DrawLine(C, sx, sy, ex, ey, w, color, layer)
	local relPoint = "BOTTOMLEFT"
	
	if not C.Cartographer_Routes_Lines then
		C.Cartographer_Routes_Lines={}
		C.Cartographer_Routes_Lines_Used={}
	end

	local T = tremove(C.Cartographer_Routes_Lines) or C:CreateTexture(nil, "ARTWORK")
	T:SetTexture("Interface\\AddOns\\Cartographer_Routes\\line")
	tinsert(C.Cartographer_Routes_Lines_Used,T)

	T:SetDrawLayer(layer or "ARTWORK")

	T:SetVertexColor(color[1],color[2],color[3],color[4]);
	-- Determine dimensions and center point of line
	local dx,dy = ex - sx, ey - sy;
	local cx,cy = (sx + ex) / 2, (sy + ey) / 2;

	-- Normalize direction if necessary
	if (dx < 0) then
		dx,dy = -dx,-dy;
	end

	-- Calculate actual length of line
	local l = sqrt((dx * dx) + (dy * dy));

	-- Sin and Cosine of rotation, and combination (for later)
	local s,c = -dy / l, dx / l;
	local sc = s * c;

	-- Calculate bounding box size and texture coordinates
	local Bwid, Bhgt, BLx, BLy, TLx, TLy, TRx, TRy, BRx, BRy;
	if (dy >= 0) then
		Bwid = ((l * c) - (w * s)) * TAXIROUTE_LINEFACTOR_2;
		Bhgt = ((w * c) - (l * s)) * TAXIROUTE_LINEFACTOR_2;
		BLx, BLy, BRy = (w / l) * sc, s * s, (l / w) * sc;
		BRx, TLx, TLy, TRx = 1 - BLy, BLy, 1 - BRy, 1 - BLx; 
		TRy = BRx;
	else
		Bwid = ((l * c) + (w * s)) * TAXIROUTE_LINEFACTOR_2;
		Bhgt = ((w * c) + (l * s)) * TAXIROUTE_LINEFACTOR_2;
		BLx, BLy, BRx = s * s, -(l / w) * sc, 1 + (w / l) * sc;
		BRy, TLx, TLy, TRy = BLx, 1 - BRx, 1 - BLx, 1 - BLy;
		TRx = TLy;
	end

	-- Set texture coordinates and anchors
	T:ClearAllPoints();
	T:SetTexCoord(TLx, TLy, BLx, BLy, TRx, TRy, BRx, BRy);
	T:SetPoint("BOTTOMLEFT", C, relPoint, cx - Bwid, cy - Bhgt);
	T:SetPoint("TOPRIGHT",   C, relPoint, cx + Bwid, cy + Bhgt);
	T:Show()
	return T
end

function G:HideLines(C)
	if C.Cartographer_Routes_Lines then
		for i = #C.Cartographer_Routes_Lines_Used, 1, -1 do
			C.Cartographer_Routes_Lines_Used[i]:Hide()
			tinsert(C.Cartographer_Routes_Lines,tremove(C.Cartographer_Routes_Lines_Used))
		end
	end
end

-- vim: ts=4 noexpandtab
