
local quixote = LibStub("LibQuixote-2.0")
local LSM3 = LibStub("LibSharedMedia-3.0")

local addon = LibStub("AceAddon-3.0"):NewAddon("BetterQuest", "AceConsole-3.0", "AceHook-3.0")

function addon:OnInitialize()
	local colors = {
		notstarted = {1,0,0,1},
		underway = {1,1,0,1},
		done = {0,1,0,1},
	}
	for k,v in pairs(QuestDifficultyColor) do
		colors[k] = {v.r,v.g,v.b,v.a}
	end
	-- (Gets us: trivial, standard, difficult, verydifficult, impossible, header.)

	self.db = LibStub("AceDB-3.0"):New("BetterQuestDB", {
		profile = {
			trackerNames = {
				full = true,
				tracked = true,
			},
			trackers = {
				['**'] = {
					frame = {
						position = {point="CENTER", x=0, y=0},
						lock = false,
						scale = 1,
						strata = "MEDIUM",
						fade = 1,
						opacity = 1,
						width = 250,
						border = {0.9, 0.82, 0, 1},
						background = {0, 0, 0, 1},
						min_height = 20,
					},
					filters = {
						['*'] = -1,
						tag = {ANY = true},
						--zone = {ANY = true},
						--All 'bool' filters check for 1/0/-1
						--current_zone = bool
						--zone = string
						--tag = string
						--daily = bool
						--tracked = bool
						--complete = bool
					},
					header = {
						show = true,
						name = true,
						counts = true,
						buttons = true,
						border = {0.9, 0.82, 0, 1},
						background = {0, 0, 0, 1},
						attach = "top",
						adjust = 0,
					},
					headers = true,
					colorTitles = true,
					colorObjectives = true,
					colorPartyObjectives = true,
					wrapTitles = false,
					wrapObjectives = false,
					levels = true,
					groupBy = 'zone',
					hideCompletedObjectives = false,
					descIfNoObjective = true,
					partyCount = true,
					partyObjectives = true,
					font = "Arial Narrow",
					font_size = select(2, GameFontNormal:GetFont()),
					hideWhenEmpty = true,
				},
				full = {
					enabled = true,
					collapsed = true,
					frame = {
						scale = 0.8,
					},
				},
				tracked = {
					enabled = true,
					frame = {
						position = {
							point = "TOPLEFT",
							relpoint = "BOTTOMLEFT",
							x = 700,
							y = 550,
						},
						border = {0.9, 0.82, 0, 0},
						background = {0, 0, 0, 0},
						lock = true,
					},
					filters = {tracked = 1,},
					header = {show = false,},
					headers = false,
					groupby = 'level',
				},
			},
			colors = colors,
			hideDefault = true,
		},
		char = {
			hidden = {
				['*'] = {},
			},
			tracked = {},
		},
	})
	
	self.trackers = {}
	self.headers = {}
	self:ConfigSetup()
end

function addon:OnEnable()
	quixote.RegisterCallback(self, "Update", "UpdateTrackers")
	quixote.RegisterCallback(self, "Party_Update", "UpdateTrackers")
	quixote.RegisterCallback(self, "Quest_Lost")
	self:SecureHook("AddQuestWatch")
	self:SecureHook("RemoveQuestWatch")
	self:SecureHook("QuestLogTitleButton_OnClick")
	self:SecureHook("QuestWatch_Update")
end
function addon:OnDisable()
	quixote.UnregisterAll(self)
end

-- Tracked quests:

function addon:AddQuestWatch(id)
	local uid = quixote:GetQuestById(id)
	if uid then
		self.db.char.tracked[uid] = true
	end
	self:UpdateTrackers()
end
function addon:RemoveQuestWatch(id)
	local uid = quixote:GetQuestById(id)
	if uid then
		self.db.char.tracked[uid] = nil
	end
	self:UpdateTrackers()
end
function addon:QuestLogTitleButton_OnClick(button)
	if IsModifiedClick("QUESTWATCHTOGGLE") then
		local uid, _, _, _, _, objectives = quixote:GetQuestById(this:GetID() + FauxScrollFrame_GetOffset(QuestLogListScrollFrame))
		if (not uid) or (objectives > 0) then return end
		if self.db.char.tracked[uid] then
			self.db.char.tracked[uid] = nil
		else
			-- possibly do a UIErrorsFrame:Clear() here to remove the error spam?
			self.db.char.tracked[uid] = true
		end
		self:UpdateTrackers()
	end
end
function addon:QuestWatch_Update()
	if self.db.profile.hideDefault then
		QuestWatchFrame:Hide()
	end
end
function addon:Quest_Lost(event, title, uid, zone)
	self.db.char.tracked[uid] = nil
end

-- Frame creation:

function addon:CreateTracker(id)
	self.trackers[id] = LibStub("LibSimpleFrame-1.0"):New("BetterQuest_"..id, self.db.profile.trackers[id].frame, id)
	self.db.profile.trackerNames[id] = true
	return self.trackers[id]
end
function addon:DeleteTracker(id)
	if self.trackers[id] then self.trackers[id]:Hide() end
	if self.headers[id] then self.headers[id]:Hide() end
	self.trackers[id] = nil
	self.headers[id] = nil
	self.db.profile.trackerNames[id] = nil
	return true
end

function addon:CreateHeader(id, tracker)
	--local tracker = self.trackers[id]
	--Note: headers are not parented to a tracker, since they need to be separately visible
	local header = CreateFrame("Frame", "BetterQuest_Header_"..id, UIParent)
	header.id = id
	header:SetBackdrop(tracker:GetBackdrop())
	header:SetMovable(true)
	header:SetClampedToScreen(true)
	header:SetHeight(32)
	
	local counts = header:CreateFontString(header:GetName().."_Counts", "OVERLAY")
	counts:SetFontObject(GameFontNormal)
	counts:SetPoint("TOPLEFT", header, "TOPLEFT", 10, -10)
	counts:SetHeight(12)
	header.counts = counts
	
	local title = header:CreateFontString(header:GetName().."_Title", "OVERLAY")
	title:SetFontObject(GameFontNormal)
	title:SetPoint("CENTER", header)
	title:SetHeight(12)
	header.title = title
	
	local button = CreateFrame("Button", header:GetName().."_Button", header)
	button:SetPoint("TOPRIGHT", header, "TOPRIGHT")
	button:SetHeight(header:GetHeight())
	button:SetWidth(header:GetHeight())
	
	local texture = button:CreateTexture(nil, "ARTWORK")
	texture:SetPoint("CENTER", button)
	texture:SetWidth(16)
	texture:SetHeight(16)
	button.texture = texture
	
	button:SetScript("OnClick", function()
		self.db.profile.trackers[id].collapsed = not self.db.profile.trackers[id].collapsed
		if tracker:IsShown() then
			tracker:Hide()
		else
			tracker:Show()
		end
		self:UpdateTracker(id)
	end)
	header.button = button
	
	tracker:Associate(button)
	tracker:Associate(header)
	self.headers[id] = header
	return header
end

-- Trackers:

function addon:UpdateTrackers()
	for id, _ in pairs(self.db.profile.trackerNames) do
		local settings = self.db.profile.trackers[id]
		if settings then
			if settings.enabled then
				self:UpdateTracker(id)
			elseif settings.enabled == false and self.trackers[id] then
				self.trackers[id]:Hide()
				if self.headers[id] then self.headers[id]:Hide() end
			end
		end
	end
	if not self.loaded_tracked_quests then
		self.loaded_tracked_quests = true
		for uid in pairs(self.db.char.tracked) do
			if not quixote:AddQuestWatchByUid(uid) then
				self.db.char.tracked[uid] = nil
			end
		end
		QuestWatch_Update()
	end
end

-- texture code: |T<path>:<width>[:<height>:<xOffset>:<yOffset>]|t
local plus = [[|TInterface\Buttons\UI-PlusButton-Up:12|t]]
local minus = [[|TInterface\Buttons\UI-MinusButton-Up:12|t]]
local function zone_click(trackerid, zone)
	addon.db.char.hidden[trackerid][zone] = not addon.db.char.hidden[trackerid][zone]
	addon:UpdateTrackers()
end
local function quest_click(trackerid, uid)
	local _, qid, title, level, _, objectives, complete = quixote:GetQuestByUid(uid)
	if IsAltKeyDown() then
		addon:ToggleQuestWatch(uid)
		GameTooltip:Hide()
	elseif IsShiftKeyDown() and ChatFrameEditBox:IsVisible() then
		if IsControlKeyDown() then
			-- Add the quest objectives to the chat editbox, if it's open.
			if objectives then
				for objective, got, need, t in quixote:IterateObjectivesForQuest(uid) do
					ChatFrameEditBox:Insert(string.format("{%s %s/%s} ", objective, got, need))
				end
			end
		else
			-- Add quest title to the chat editbox if it's open.
			--ChatEdit_InsertLink(GetQuestLink(qid):gsub(title, level..quixote:GetShortTagForQuest(uid)..' '..title):gsub('|', '\124'))
			ChatEdit_InsertLink(GetQuestLink(qid))
			--ChatFrameEditBox:Insert("\124Hquest:"..uid..':'..level..'\124h'..quixote:GetTaggedQuestName(uid)..'\124h')
		end
	elseif IsControlKeyDown() then
		quixote:ShareQuest(uid)
	else
		ItemRefTooltip:ClearLines()
		ShowUIPanel(ItemRefTooltip)
		if not ItemRefTooltip:IsVisible() then
			ItemRefTooltip:SetOwner(UIParent, "ANCHOR_PRESERVE")
		end
		local link = GetQuestLink(qid)
		if link then
			ItemRefTooltip:SetHyperlink(link)
		end
	end
end
local function quest_rightclick(trackerid, uid)
	quixote:ShowQuestLog(uid)
end
local function quest_enter(trackerid, uid)
	local _, qid, title, level, _, objectives, complete = quixote:GetQuestByUid(uid)
	local link = GetQuestLink(qid)
	if link then
		GameTooltip:SetOwner(addon.trackers[trackerid], "ANCHOR_LEFT")
		GameTooltip:SetHyperlink(link)
	end
end
local function quest_leave(trackerid) GameTooltip:Hide() end

local display_grouped_quests, add_quest_to_tracker, quest_matches_filters

function addon:UpdateTracker(id)
	if not self.trackers[id] then self:CreateTracker(id) end
	local settings = self.db.profile.trackers[id]
	if not settings.enabled then settings.enabled = true end
	local tracker = self.trackers[id]:Clear()
	local num_quests, num_complete = display_grouped_quests[settings.groupBy](id, tracker, settings)
	tracker:Size()
	tracker:SetPosition()
	if (settings.hideWhenEmpty and num_quests == 0) or (settings.collapsed and settings.header.show) then
		tracker:Hide()
		tracker:SetClampedToScreen(false)
	else
		tracker:Show()
		tracker:SetClampedToScreen(true)
	end
	if settings.header.show then
		if not self.headers[id] then self.headers[id] = self:CreateHeader(id, tracker) end
		local header = self.headers[id]
		local width = 40
		header:ClearAllPoints()
		if settings.header.attach == "top" then
			header:SetPoint("BOTTOM", tracker, "TOP", 0, settings.header.adjust)
		else
			header:SetPoint("TOP", tracker, "BOTTOM", 0, settings.header.adjust * -1)
		end
		header:SetBackdropBorderColor(unpack(settings.header.border))
		header:SetBackdropColor(unpack(settings.header.background))

		local default_font, default_size, default_flags = GameFontNormal:GetFont()
		header.title:SetFont(settings.font, default_size, default_flags)
		header.counts:SetFont(settings.font, default_size, default_flags)

		header.title:SetText(settings.header.name and id or "")
		header.counts:SetText(settings.header.counts and ("%d/%d"):format(num_complete, num_quests) or "")
		width = width + header.title:GetWidth() + header.counts:GetWidth()
		if settings.header.buttons then
			if tracker:IsShown() then
				header.button.texture:SetTexture([[Interface\Buttons\UI-MinusButton-Up]])
			else
				header.button.texture:SetTexture([[Interface\Buttons\UI-PlusButton-Up]])
			end
			width = width + header.button:GetWidth()
			header.button:Show()
		else
			header.button:Hide()
		end
		header:SetWidth(width)
		header:Show()
	elseif self.headers[id] then
		self.headers[id]:Hide()
	end
end

display_grouped_quests = {
	zone = function(trackerid, tracker, settings)
		local num_quests, num_complete = 0, 0
		for _, zone, n in quixote:IterateZones() do
			local header_added = false
			local hidden = addon.db.char.hidden[trackerid][zone]
			if not (hidden and not settings.headers) then
				for _, uid, qid, title, level, tag, objectives, complete, group, daily, zone in quixote:IterateQuestsInZone(zone) do
					if quest_matches_filters(settings.filters, uid, qid, title, level, tag, objectives, complete, group, daily, zone) then
						if settings.headers and not header_added then
							-- This is *here*, beacuse otherwise the headers would still show for categories that are empty due to filters.
							tracker:AddLine((hidden and plus or minus)..zone, n)
								:Color(unpack(addon.db.profile.colors.header))
								:Handler('OnLeftClick', zone_click, zone)
								:Font(LSM3:Fetch("font", settings.font), settings.font_size, "OUTLINE")
							header_added = true
						end
						num_quests = num_quests + 1
						if complete == 1 then num_complete = num_complete + 1 end
						if not hidden then
							add_quest_to_tracker(tracker, settings, uid, qid, title, level, tag, objectives, complete, group, daily, zone)
						end
					end
				end
			end
		end
		return num_quests, num_complete
	end,
	level = function(trackerid, tracker, settings)
		local num_quests, num_complete = 0, 0
		local current_header
		for _, uid, qid, title, level, tag, objectives, complete, group, daily, zone in quixote:IterateQuestsByLevel() do
			if quest_matches_filters(settings.filters, uid, qid, title, level, tag, objectives, complete, group, daily, zone) then
				if settings.headers and level ~= current_header then
					tracker:AddLine(level)
						:Color(addon:GetColorFromLevel(level))
						:Font(LSM3:Fetch("font", settings.font), settings.font_size, "OUTLINE")
					current_header = level
				end
				add_quest_to_tracker(tracker, settings, uid, qid, title, level, tag, objectives, complete, group, daily, zone)
				num_quests = num_quests + 1
				if complete == 1 then num_complete = num_complete + 1 end
			end
		end
		return num_quests, num_complete
	end,
}

local ternary = function(bool, if_true, if_false) if bool then return if_true else return if_false end end
local simple_filter = function(nth)
	return function(show, ...)
		local bool = select(nth, ...)
		return ternary(show == 1, bool, not bool)
	end
end
local filter_checks = {
	--zone = string
	zone = function(zone, ...)
		return select(9, ...) == zone
	end,
	--current_zone = bool
	current_zone = function(show, ...)
		local is_current = select(10, ...) == GetRealZoneText()
		return ternary(show == 1, is_current, not is_current)
	end,
	--tracked = bool
	tracked = function(show, uid, qid)
		local is_tracked = addon.db.char.tracked[uid]
		return ternary(show == 1, is_tracked, not is_tracked)
	end,
	--tag = string
	tag = function(tags, ...)
		if tags.ANY then return true end
		--if tags.NONE and quest_tag == nil then return true end
		return tags[(select(5, ...) or "NONE")]
	end,
	--daily = bool
	daily = simple_filter(9),
	--complete = bool
	complete = simple_filter(7),
}
function quest_matches_filters(filters, ...)
	for id, value in pairs(filters) do
		-- If any filter returns false, immediately return false for quest
		if value ~= -1 and filter_checks[id] and not filter_checks[id](value, ...) then
			return false
		end
	end
	return true
end
function add_quest_to_tracker(tracker, settings, uid, qid, title, level, tag, objectives, complete, group, daily, zone)
	local r,g,b,a
	local count = (settings.partyCount or settings.partyObjectives) and quixote:GetNumPartyMembersWithQuest(uid)
	if settings.levels then
		title = quixote:GetTaggedQuestName(uid)
	end
	if settings.partyCount and count > 0 then
		-- Color?
		title = title .. " <" .. count .. ">"
	end
	if settings.colorTitles then
		r,g,b,a = addon:GetColorFromLevel(level)
	end
	
	if quixote:IsQuestWatchedByUid(uid) then
		
	end
	
	if complete == 1 then complete = 'done'
	elseif complete == -1 then complete = 'fail' end
	
	tracker:AddLine(title, complete, settings.wrapTitles, 6)
		:Color(r,g,b,a)
		:Font(LSM3:Fetch("font", settings.font), settings.font_size)
		:Handler('OnLeftClick', quest_click, uid)
		:Handler('OnRightClick', quest_rightclick, uid)
		:Handler('OnEnter', quest_enter, uid):Handler('OnLeave', quest_leave)
	if objectives > 0 then
		for objective, got, need, t in quixote:IterateObjectivesForQuest(uid) do
			if got ~= need or not settings.hideCompletedObjectives then
				local status = got..'/'..need
				local r,g,b,a
				if settings.colorObjectives then
					if t == 'reputation' then
						r,g,b,a = addon:GetColorFromCompletion(quixote:GetReactionLevel(got)/quixote:GetReactionLevel(need))
					else
						r,g,b,a = addon:GetColorFromCompletion(got/need)
					end
				end
				if settings.partyObjectives then
					for i=1, GetNumPartyMembers() do
						local unit = "party"..i
						local s = ""
						local pgot = quixote:GetPartyQuestObjective(i, uid, objective)
						if pgot and type(pgot) == 'number' then
							s = (UnitName(unit)):sub(1,3) .. ":" .. (pgot == need and 'd' or pgot)
							if settings.colorPartyObjectives then
								local color = RAID_CLASS_COLORS[(select(2, UnitClass(unit)))]
								s = addon:Colorize(s, color.r, color.g, color.b)
							end
						end
						status = s .. " " .. status
					end
				end
				tracker:AddLine(objective, status, settings.wrapObjectives, 12)
					:Color(r,g,b,a, r,g,b,a)
					:Font(LSM3:Fetch("font", settings.font), settings.font_size)
					:Handler('OnLeftClick', quest_click, uid)
					:Handler('OnRightClick', quest_rightclick, uid)
					:Handler('OnEnter', quest_enter, uid):Handler('OnLeave', quest_leave)
			end
		end
	elseif settings.descIfNoObjective and ((not complete or complete == 0) or not settings.hideCompletedObjectives) then
		local _, objectives = quixote:GetQuestText(uid)
		tracker:AddLine(objectives, '', true, 12)
			:Font(LSM3:Fetch("font", settings.font), settings.font_size)
			:Handler('OnLeftClick', quest_click, uid)
			:Handler('OnRightClick', quest_rightclick, uid)
			:Handler('OnEnter', quest_enter, uid):Handler('OnLeave', quest_leave)
	end
end

-- Handy utilities:

function addon:ToggleQuestWatch(uid)
	if self.db.char.tracked[uid] then
		quixote:RemoveQuestWatchByUid(uid)
	else
		quixote:AddQuestWatchByUid(uid)	
	end
	QuestWatch_Update()
end

function addon:Colorize(s, r, g, b)
	return string.format("|c00%02X%02X%02X%s|r", r * 255, g * 255, b * 255, s)
end

function addon:GetColorFromLevel(level)
	local color = GetDifficultyColor(level)
	-- Color should be the output of GetDifficultyColor; a table.
	if color == QuestDifficultyColor.trivial then
		return unpack(self.db.profile.colors.trivial)
	elseif color == QuestDifficultyColor.standard then
		return unpack(self.db.profile.colors.standard)
	elseif color == QuestDifficultyColor.difficult then
		return unpack(self.db.profile.colors.difficult)
	elseif color == QuestDifficultyColor.verydifficult then
		return unpack(self.db.profile.colors.verydifficult)
	elseif color == QuestDifficultyColor.impossible then
		return unpack(self.db.profile.colors.impossible)
	end
end

function addon:GetColorFromCompletion(percent)
	if percent <= 0 then
		return unpack(self.db.profile.colors.notstarted)
	elseif percent >= 1 then
		return unpack(self.db.profile.colors.done)
	elseif percent == 0.5 then
		return unpack(self.db.profile.colors.underway)
	elseif percent < 0.5 then
		percent = percent / 0.5
		local r1,g1,b1 = unpack(self.db.profile.colors.notstarted)
		local r2,g2,b2 = unpack(self.db.profile.colors.underway)
		return r1+(r2-r1)*percent, g1+(g2-g1)*percent, b1+(b2-b1)*percent
	elseif percent < 1 then
		percent = (percent-0.5) / 0.5
		local r1,g1,b1 = unpack(self.db.profile.colors.underway)
		local r2,g2,b2 = unpack(self.db.profile.colors.done)
		return r1+(r2-r1)*percent, g1+(g2-g1)*percent, b1+(b2-b1)*percent
	end
end
