--[[
--
--	BarTracker
--	by Dust of Turalyon
--
--]]

BarTracker_Version = GetAddOnMetadata("BarTracker", "Version");

BarTracker_PowerTranslation = {
	[0]	= "Mana";
	[1]	= "Rage";
	[2]	= "Focus";
	[3]	= "Energy";
	[4]	= "Happiness";
	[5]	= "Undefined";
	[6]	= "Rune";
};

BarTracker_LoggedPowers = {
	"Mana",
	"Rage",
	"Energy",
	"Rune"
}

BarTracker_StatAnchors = {
	["Mana"] = "TOPRIGHT";
	["Energy"] = "TOPRIGHT";
	["Rage"] = "TOPRIGHT";
	["Rune"] = "TOPRIGHT";
	["Health"] = "TOPLEFT";
	["TargetHealth"] = "TOPRIGHT";
};

BarTracker_StatScales = {
	["Mana"] = 0.5;
	["Energy"] = 0.5;
	["Rage"] = 0.5;
	["Rune"] = 0.5;
	["Health"] = 0.5;
	["TargetHealth"] = 1.0;
};

BarTracker_Status = "Paused";

BarTracker_OnLoad = function ()
	--Slash command
	SlashCmdList["BARTRACKER"] = BarTracker_SlashHandler;
	SLASH_BARTRACKER1 = "/bartracker";
	SLASH_BARTRACKER2 = "/bt";	

	--Events
	BarTracker_MainFrame:SetScript("OnEvent", BarTracker_OnEvent);
	BarTracker_MainFrame:RegisterEvent("VARIABLES_LOADED");
	BarTracker_MainFrame:RegisterEvent("PLAYER_REGEN_DISABLED");
	BarTracker_MainFrame:RegisterEvent("PLAYER_REGEN_ENABLED");
	BarTracker_MainFrame:RegisterEvent("PLAYER_LEAVING_WORLD");
	BarTracker_MainFrame:RegisterEvent("PLAYER_LOGIN");
	BarTracker_OnLoad = nil;
end;

BarTracker_VariablesLoaded = function ()
	if (BarTracker_Settings == nil) or (BarTracker_Settings["Version"] ~= BarTracker_Version) then
		BarTracker_MSG("BT: BarTracker settings reset");
		BarTracker_Settings = {
			["Show"]	= true;
			["Scale"]	= 0.75;
			["Alpha"]	= 0.75;
			["TimeScale"]	= 0;
			["BattleMode"]	= "auto"; -- "auto", "manual"
			["Timeout"]	= 10;
			["Combatlog"]	= false;
			["CombatFade"]	= false;
			["LiveUpdate"]	= true;
			["TargetHealth"]	= true;
			["Color"]	= {
				["Health"]	= {0.0, 1.0, 0.0, 0.5};
				["Mana"]	= {0.0, 0.5, 1.0, 0.5}; -- for better readability
				["Energy"]	= {1.0, 1.0, 0.0, 0.5};
				["Rage"]	= {1.0, 0.0, 0.0, 0.5};
				["Rune"]	= {0.0, 0.5, 1.0, 0.5};
				["Focus"]	= {1.0, 0.5, 0.5, 0.5};
				["TargetHealth"]	= {0.75, 0.0, 0.0, 0.75};
			};

			["Version"]	= BarTracker_Version;
		};
	end;
	if (BarTracker_StoredLogs == nil) then
		BarTracker_StoredLogs = {};
	end;

	--Layout
	if BarTracker_Settings["Show"] then
		BarTracker_OutputFrame0:Show();
	else
		BarTracker_OutputFrame0:Hide();
	end;
	BarTracker_OutputFrame0:SetScale(BarTracker_Settings["Scale"]);
	BarTracker_OutputFrame0:SetAlpha(BarTracker_Settings["Alpha"]);
	BarTracker_OutputFrame0:SetClampedToScreen("TRUE");
	BarTracker_OutputFrame0:SetMinResize(64, 64);
	BarTracker_LogViewer:SetClampedToScreen("TRUE");
	BarTracker_LogViewer:SetMinResize(320, 240);
	if BarTracker_Settings["TimeScale"] == 0 then
		BarTracker_VisualModeScale();
	else
		BarTracker_VisualModeScroll();
	end;
	BarTracker_VariablesLoaded = nil;
end;

BarTracker_PlayerLogin = function ()
	BarTracker_PlayerGUID = UnitGUID("player");
	BarTracker_InitLog();
	BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
	-- Install handlers which don't work without a log
	BarTracker_OutputFrame0:SetScript("OnSizeChanged", BarTracker_FrameResizing);
	BarTracker_OutputFrame0:SetScript("OnEnter", BarTracker_OnMouseEnter);
	BarTracker_OutputFrame0:SetScript("OnLeave", BarTracker_OnMouseLeave);
	BarTracker_LogViewer:SetScript("OnSizeChanged", BarTracker_LogViewUpdate);
	BarTracker_PlayerLogin = nil;
end;

BarTracker_RegisterEvents = function ()
	for _, e in ipairs(BarTracker_EventList) do
		BarTracker_MainFrame:RegisterEvent(e);
	end

	-- Register separately to lessen performance overhead when it's disabled
	if BarTracker_Settings["Combatlog"] then
		BarTracker_MainFrame:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED");
	end
end;

BarTracker_UnregisterEvents = function ()
	for _, e in ipairs(BarTracker_EventList) do
		BarTracker_MainFrame:UnregisterEvent(e);
	end

	-- Unregister separately
	BarTracker_MainFrame:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED");
end;

-- General events that get registered
BarTracker_EventList =
{
	"UNIT_HEALTH",
	"UNIT_MAXHEALTH",
	"UNIT_MANA",
	"UNIT_MAXMANA",
	"UNIT_RAGE",
	"UNIT_MAXRAGE",
	"UNIT_ENERGY",
	"UNIT_MAXENERGY",
	"UNIT_RUNIC_POWER",
	"UNIT_MAXRUNIC_POWER",
	"UNIT_TARGET",
	"UNIT_DISPLAYPOWER"
}

-- Event handlers
BarTracker_EventHandlers =
{
	["UNIT_HEALTH"] = function (frame, event, arg1)
		if ("player" == arg1) then
			BarTracker_HealthChanged();
		elseif (BarTracker_Settings["TargetHealth"] and "target" == arg1) then
			BarTracker_TargetHealthChanged();
		end;
	end;
	["UNIT_MANA"] = function (frame, event, arg1)
		if (UnitIsUnit("player", arg1)) then
			BarTracker_PowerChanged();
		end;
	end;
	["UNIT_MAXHEALTH"] = function (frame, event, arg1)
		if (UnitIsUnit("player", arg1)) then
			BarTracker_MaxHealthChanged();
		end;
	end;
	["UNIT_MAXMANA"] = function (frame, event, arg1)
		if (UnitIsUnit("player", arg1)) then
			BarTracker_MaxPowerChanged();
		end;
	end;
	["UNIT_TARGET"] = function (frame, event, arg1)
		if (BarTracker_Settings["TargetHealth"] and "player" == arg1) then
			BarTracker_TargetHealthChanged();
		end;
	end;
	["COMBAT_LOG_EVENT_UNFILTERED"] = function (frame, event, ...)
		BarTracker_AddChatLine(...);
	end;
	["PLAYER_REGEN_DISABLED"] = function (frame, event)
		-- Entered Combat
		if BarTracker_Settings["BattleMode"] == "auto" then
			BarTracker_EnterCombat();
		end;
	end;
	["PLAYER_REGEN_ENABLED"] = function (frame, event)
		-- Left Combat
		if BarTracker_Settings["BattleMode"] == "auto" then
			TimeLight_Schedule(BarTracker_Settings["Timeout"], "BarTracker_LeaveCombat", BarTracker_LeaveCombat);
		end;
	end;
	["UNIT_DISPLAYPOWER"] = function (frame, event)
		if (UnitIsUnit("player", arg1)) then
			local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
			BarTracker_PowerTerminate();
			BarTracker_MaxPowerChanged(powerType);
		end;
	end;
	["VARIABLES_LOADED"] = function (frame, event)
		BarTracker_VariablesLoaded();
	end;
	["PLAYER_LOGIN"] = function (frame, event)
		BarTracker_PlayerLogin();
	end;
}

-- Leaving the world also stops the combat
BarTracker_EventHandlers["PLAYER_LEAVING_WORLD"] = BarTracker_EventHandlers["PLAYER_REGEN_ENABLED"];
-- Alternate events for power types
BarTracker_EventHandlers["UNIT_RAGE"] = BarTracker_EventHandlers["UNIT_MANA"];
BarTracker_EventHandlers["UNIT_ENERGY"] = BarTracker_EventHandlers["UNIT_MANA"];
BarTracker_EventHandlers["UNIT_RUNIC_POWER"] = BarTracker_EventHandlers["UNIT_MANA"];
BarTracker_EventHandlers["UNIT_MAXRAGE"] = BarTracker_EventHandlers["UNIT_MAXMANA"];
BarTracker_EventHandlers["UNIT_MAXENERGY"] = BarTracker_EventHandlers["UNIT_MAXMANA"];
BarTracker_EventHandlers["UNIT_MAXRUNIC_POWER"] = BarTracker_EventHandlers["UNIT_MAXMANA"];

BarTracker_OnEvent = function (frame, event, ...)
	local handler = BarTracker_EventHandlers[event];
	if handler then
		handler(frame, event, ...);
	end;
end;

-- Implementation constants
BarTracker_vBorder = 8;
BarTracker_hBorder = 8; -- 8 or 32
BarTracker_MaxWidth = 160;

BarTracker_VisualModeScale = function ()
	-- Set Mode to scale
--	if BarTracker_Status == "Paused" then
		BarTracker_OutputFrame0_Slider:SetVerticalScroll(0);
		BarTracker_OutputFrame0_Slider:Hide();
		BarTracker_OutputFrame0_Slider:SetScrollChild(nil);
		BarTracker_OutputFrame0_Slider_ScrollContainer:SetParent(BarTracker_OutputFrame0);
		BarTracker_OutputFrame0_Slider_ScrollContainer:Show()
		BarTracker_hBorder = 8;
--	end;
end;

BarTracker_VisualModeScroll = function ()
	-- Set Mode to scroll
--	if BarTracker_Status == "Paused" then
		BarTracker_OutputFrame0_Slider:Show();
		BarTracker_OutputFrame0_Slider:SetScrollChild(BarTracker_OutputFrame0_Slider_ScrollContainer);
		BarTracker_OutputFrame0_Slider_ScrollContainer:Show()
		BarTracker_hBorder = 32;
--	end;
end;

BarTracker_EnterCombat = function ()
	if BarTracker_Status == "Paused" then
		BarTracker_Status = "Running";
		BarTracker_InitLog();
		BarTracker_RegisterEvents();
		if BarTracker_Settings["Show"] and BarTracker_Settings["CombatFade"] then
			UIFrameFadeIn(BarTracker_OutputFrame0, 1, BarTracker_Settings["Alpha"], 0);
			TimeLight_Schedule(1, "BarTracker_FrameHide", BarTracker_OutputFrame0.Hide, BarTracker_OutputFrame0);
		end;
	else
		TimeLight_Unschedule("BarTracker_LeaveCombat");
	end;
end;

BarTracker_LeaveCombat = function ()
	if BarTracker_Status == "Running" then
		BarTracker_Status = "Paused";
		BarTracker_UnregisterEvents();
		BarTracker_FinishLog();
		BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
		BarTracker_FilterLog(BarTracker_Log["Combatlog"], BarTracker_LogFilter);
		if BarTracker_Settings["Show"] and BarTracker_Settings["CombatFade"] then
			BarTracker_OutputFrame0:Show();
			UIFrameFadeIn(BarTracker_OutputFrame0, 1, 0, BarTracker_Settings["Alpha"]);
		end;
		BarTracker_StoreLog(1);
		TimeLight_Unschedule("BarTracker_LeaveCombat");
	end;
end;

BarTracker_InitLog = function ()
	local health = UnitHealth("player");
	local healthMax = UnitHealthMax("Player");
	local mana = UnitMana("player");
	local manaMax = UnitManaMax("player");
	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local time =  GetTime();
	BarTracker_Log = {
		["Health"]	= {{health, time}};
		["HealthMax"]	= healthMax;
		["HealthMin"]	= health;
		["TargetHealth"] = {{0, time}};
		["TargetHealthMax"] = 100;
		["StartTime"]	= date("!%Y%m%d%H%MZ");
		["Combatlog"]	= {};
		["StartTimeStamp"] = time;
		["EndTimeStamp"] = time;
	}
	for _, p in ipairs(BarTracker_LoggedPowers) do
		BarTracker_Log[p] = {{0, time}};
		BarTracker_Log[p.."Min"] = 0;
		BarTracker_Log[p.."Max"] = 0;
	end;
	BarTracker_Log[powerType][1][1] = mana;
	BarTracker_Log[powerType.."Max"] = manaMax;
	BarTracker_Log[powerType.."Min"] = mana;
end;

BarTracker_FinishLog = function ()
	BarTracker_HealthChanged();
	BarTracker_PowerTerminate();
end;

BarTracker_LoadLog = function (name)
	BarTracker_LeaveCombat();
	local idx = tonumber(name);
	if idx ~= nil then
		name = idx;
	end;
	if BarTracker_StoredLogs[name] ~= nil then
		BarTracker_Log = BarTracker_StoredLogs[name];
		BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
		BarTracker_FilterLog(BarTracker_Log["Combatlog"], BarTracker_LogFilter);
		BarTracker_LogViewUpdate();
		BarTracker_MSG("BarTracker: Loaded entry: "..name);
	else
		BarTracker_MSG("BarTracker: Loading entry "..name.." failed");
	end;
end;

BarTracker_StoreLog = function (name)
	BarTracker_LeaveCombat();
	local idx = tonumber(name);
	if idx ~= nil then
		table.insert(BarTracker_StoredLogs, idx, BarTracker_Log);
		BarTracker_StoredLogs[6] = nil; -- TODO: Recycle
	else
		BarTracker_StoredLogs[name] = BarTracker_Log;
		BarTracker_MSG("BarTracker: Stored entry "..name);
	end;
end;

BarTracker_DeleteLog = function (name)
	local idx = tonumber(name);
	if idx ~= nil then
		BarTracker_StoredLogs[idx] = nil;
	else
		BarTracker_StoredLogs[name] = nil;
	end;
	BarTracker_MSG("BarTracker: Deleted entry "..name);
end;

BarTracker_ListLog = function ()
	BarTracker_MSG("BarTracker: Stored Logs:");
	for name, log in pairs(BarTracker_StoredLogs) do
		BarTracker_MSG("> "..name.." @ "..log["StartTime"]);
	end;
end;

BarTracker_HealthChanged = function ()
	local health = UnitHealth("player");
	local time =  math.floor(GetTime()*5)/5; -- Limit Event Resolution, less data consumption
	local n = table.getn(BarTracker_Log["Health"]);
	if time > BarTracker_Log["Health"][n][2] then
		table.insert(BarTracker_Log["Health"], n+1, {health, time});
	else
		BarTracker_Log["Health"][n][1] = health;
	end;
	BarTracker_Log["HealthMin"] = math.min(BarTracker_Log["HealthMin"], health);
	BarTracker_Log["EndTimeStamp"] = time;
	BarTracker_LiveUpdate();
end;

BarTracker_TargetHealthChanged = function ()
	local health = 0;
	if UnitExists("target") then
		health = UnitHealth("target")/UnitHealthMax("target")*100;
	end
	local time =  math.floor(GetTime()*5)/5; -- Limit Event Resolution, less data consumption
	local n = table.getn(BarTracker_Log["TargetHealth"]);
	if time > BarTracker_Log["TargetHealth"][n][2] then
		table.insert(BarTracker_Log["TargetHealth"], n+1, {health, time});
	else
		BarTracker_Log["TargetHealth"][n][1] = health;
	end;
	BarTracker_Log["EndTimeStamp"] = time;
	BarTracker_LiveUpdate();
end;

BarTracker_MaxHealthChanged = function ()
	local healthMax = UnitHealthMax("Player");
	local healthMaxOld = BarTracker_Log["HealthMax"];
	BarTracker_Log["HealthMax"] = math.max(healthMax, healthMaxOld);
end;

-- For all mana-like stats
BarTracker_PowerChanged = function ()
	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local mana = UnitMana("player");
	local time =  math.floor(GetTime()*5)/5; -- Limit Event Resolution, less data consumption
	local n = table.getn(BarTracker_Log[powerType]);
	if time > BarTracker_Log[powerType][n][2] then
		table.insert(BarTracker_Log[powerType], n+1, {mana, time});
	else
		BarTracker_Log[powerType][n][1] = mana;
	end;
	BarTracker_Log[powerType.."Min"] = math.min(BarTracker_Log[powerType.."Min"], mana);
	BarTracker_Log["EndTimeStamp"] = time;
	BarTracker_LiveUpdate();
end;

-- For all mana-like stats
BarTracker_MaxPowerChanged = function ()
	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local manaMax = UnitManaMax("Player");
	local manaMaxOld = BarTracker_Log[powerType.."Max"];
	BarTracker_Log[powerType.."Max"] = math.max(manaMax, manaMaxOld);
end;

BarTracker_PowerTerminate = function ()
	local time = GetTime();
	for _, p in ipairs(BarTracker_LoggedPowers) do
		table.insert(BarTracker_Log[p], {0, time});
	end
	BarTracker_PowerChanged();
end;

BarTracker_AddChatLine = function (timestamp, event, sourceGUID, sourceName, sourceFlags, destGUID, destName, destFlags, ...)
	if (sourceGUID == BarTracker_PlayerGUID) or (destGUID == BarTracker_PlayerGUID) then
		local time =  GetTime();
		--local argNum = select("#", ...);
		table.insert(BarTracker_Log["Combatlog"], {time,
			CombatLog_OnEvent(Blizzard_CombatLog_CurrentSettings, timestamp, event,
			sourceGUID, sourceName, sourceFlags,
			destGUID, destName, destFlags, ... )});
	end
end;

BarTracker_LastLiveUpdate = 0;
BarTracker_LiveUpdate = function ()
	if BarTracker_Settings["LiveUpdate"] then
		local currentTime = GetTime();
		if currentTime - BarTracker_LastLiveUpdate > 2 then
			BarTracker_LastLiveUpdate = currentTime;
			BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");

			-- Scroll to the end
			local container = BarTracker_OutputFrame0_Slider_ScrollContainer;
			local slider = BarTracker_OutputFrame0_Slider;
			if slider then
				slider:SetVerticalScroll(container:GetHeight()-slider:GetHeight()+10);
			end;
		end;
	end;
end;

BarTracker_RebuildAllDiagrams = function (basename)
	local base = getglobal(basename);
	if base then
		BarTracker_MaxWidth = (base:GetWidth()-BarTracker_hBorder);
		local duration = (BarTracker_Log["EndTimeStamp"]-BarTracker_Log["StartTimeStamp"]+1);
		local heightScale;

		if BarTracker_Settings["TimeScale"] > 0 then
			heightScale = BarTracker_Settings["TimeScale"];
		else
			heightScale = (BarTracker_OutputFrame0:GetHeight()-BarTracker_vBorder)/duration;
		end
		local height = duration*heightScale;
		local container = getglobal(basename.."_Slider_ScrollContainer");
		if container then
			container:SetWidth(BarTracker_MaxWidth);
			container:SetHeight(height);
		end;

		BarTracker_RebuildDiagram(basename, "TargetHealth", heightScale);
		BarTracker_RebuildDiagram(basename, "Health", heightScale);
		for _, p in ipairs(BarTracker_LoggedPowers) do
			BarTracker_RebuildDiagram(basename, p, heightScale);
		end;
		BarTracker_RebuildDiagramText(basename, heightScale);
	end;
end;

BarTracker_RebuildDiagram = function (basename, Stat, heightScale)
	local i = 0;
	local container = getglobal(basename.."_Slider_ScrollContainer");
	local statMax = BarTracker_Log[Stat.."Max"];
	if statMax ~= nil then
		local tLast = BarTracker_Log["StartTimeStamp"]-1; -- BarTracker_Log[Stat][1][2]
		local vLast = BarTracker_Log[Stat][1][1];
		local pLast = 0;

		local color = BarTracker_Settings["Color"][Stat];
		local statWidth = BarTracker_MaxWidth*BarTracker_StatScales[Stat]/statMax;
		local anchor = BarTracker_StatAnchors[Stat];
		local tex = nil;
		local width; local height;
		for _, a in ipairs(BarTracker_Log[Stat]) do
			width = vLast*statWidth;
			if width == 0 then width = 0.001; end;
			height = (a[2]-tLast)*heightScale;
			if height < 0.25 and tex ~= nil then -- arbitrary treshold
				pLast = pLast - height;
				-- Reuse last texture
				local oldHeight = tex:GetHeight();
				-- Weighted average
				width = (tex:GetWidth()*oldHeight + width*height)/(oldHeight + height);
				height = oldHeight + height;
			else
				-- New Texture
				tex = getglobal(basename.."_Tex"..Stat..i);
				if tex == nil then
					tex = container:CreateTexture(
						basename.."_Tex"..Stat..i,
						basename.."_Slider_ScrollContainer_ScrollLayerBars");
				end;
				tex:ClearAllPoints();
				tex:SetTexture(color[1], color[2], color[3], color[4]);
				tex:SetPoint(anchor, 0, pLast);
				pLast = pLast - height;
				i = i + 1;
			end;
			tex:SetWidth(width);
			tex:SetHeight(height);
			tLast = a[2];
			vLast = a[1];
		end;
		-- Last line
		tex = getglobal(basename.."_Tex"..Stat..i);
		if tex == nil then
			tex = container:CreateTexture(
				basename.."_Tex"..Stat..i,
				basename.."_Slider_ScrollContainer_ScrollLayerBars");
		end;
		tex:ClearAllPoints();
		tex:SetTexture(color[1], color[2], color[3], color[4]);
		tex:SetPoint(anchor, 0, pLast);
		i = i + 1;
		width = vLast*statWidth;
		if width == 0 then width = 0.01; end;
		height = math.max((BarTracker_Log["EndTimeStamp"]-tLast)*heightScale, 0.001);
		tex:SetWidth(width);
		tex:SetHeight(height);
	end;
	while true do
		local tex = getglobal(basename.."_Tex"..Stat..i);
		if tex == nil then
			break;
		else
			tex:SetTexture(0.0, 0.0, 0.0, 0.0);
		end;
		i = i + 1;
	end;
end;

BarTracker_RebuildDiagramText = function (basename, heightScale)
	local container = getglobal(basename.."_Slider_ScrollContainer");
	local pText = 0;
	local pLast = -container:GetHeight();
	local i = 0;
	while pText > pLast do
		local text = getglobal(basename.."_Text"..i);
		if text == nil then
			text = container:CreateFontString(
				basename.."_Text"..i,
				basename.."_Slider_ScrollContainer_ScrollLayerText");
			text:SetFontObject(GameFontNormalSmall);
			text:SetJustifyH("CENTER");
			text:SetTextColor(1,1,1,0.5);
		end;
		text:SetText(string.format("%2.1f",(i*30/heightScale)).." sec");
		text:SetWidth(BarTracker_MaxWidth);
		text:SetPoint("TOPLEFT", 0, pText);
		pText = pText - 30;
		i = i + 1;
	end;
	while true do
		local text = getglobal(basename.."_Text"..i);
		if text == nil then
			break;
		else
			text:SetText("");
		end;
		i = i + 1;
	end;
	for i = 0, 1, 0.125 do
		local tex = getglobal(basename.."_Mark"..i);
		if tex == nil then
			tex = container:CreateTexture(basename.."_Mark"..i,
				basename.."_Slider_ScrollContainer_ScrollLayerBars");
			tex:SetTexture(1,1,1, 0.25);
		end;
		tex:SetWidth(1/BarTracker_Settings["Scale"]);
		tex:SetPoint("TOPLEFT", BarTracker_MaxWidth*i, 0);
		tex:SetHeight(-pLast);
	end;
end;

--[[ Fullscreen View ]]
BarTracker_ShowFullscreen = function ()
	local base = BarTracker_FullscreenFrame;
	local basename = "BarTracker_FullscreenFrame";
	local w = GetScreenWidth();
	local h = GetScreenHeight();
	base:SetWidth(w);
	base:SetHeight(h);
	BarTracker_MaxWidth = (w-32);
	BarTracker_FullscreenFrame_Slider_ScrollContainer:SetWidth(BarTracker_MaxWidth);
	base:Show();

	local heightScale = 14;
	
	BarTracker_RebuildDiagram(basename, "TargetHealth", heightScale);
	BarTracker_RebuildDiagram(basename, "Health", heightScale);
	BarTracker_RebuildDiagram(basename, "Mana", heightScale);
	BarTracker_RebuildDiagram(basename, "Rage", heightScale);
	BarTracker_RebuildDiagram(basename, "Energy", heightScale);
	BarTracker_RebuildDiagram(basename, "Rune", heightScale);
	BarTracker_RebuildDiagramText(basename, heightScale);
	BarTracker_RebuildLogText(basename, heightScale);
	BarTracker_FullscreenFrame_Slider:SetVerticalScroll(0);
end;

BarTracker_RebuildLogText = function (basename, heightScale)
	local container = getglobal(basename.."_Slider_ScrollContainer");
	local logStart = BarTracker_Log["StartTimeStamp"]
	local log = BarTracker_Log["Combatlog"];
	if not log then return; end;

	heightScale = -1*heightScale;
	local red, green, blue, str, time, text;
	local pLast, pNow, pIndent; pLast = 14;
	local i = 1;
	while log[i] do
		str = log[i][2];
		time = log[i][1]-logStart;
		red, green, blue = BarTracker_ColorForLogEntry(log[i]);

		text = getglobal(basename.."_LogText"..i);
		if text == nil then
			text = container:CreateFontString(
				basename.."_LogText"..i,
				basename.."_Slider_ScrollContainer_ScrollLayerText");
			text:SetFontObject(GameFontNormalSmall);
			text:SetJustifyH("LEFT");
		end;
		text:SetText(string.format("%2.1f: ",time)..str);
		text:SetTextColor(red, green, blue, 1);
		text:SetWidth(BarTracker_MaxWidth);
		pNow = time*heightScale;
		local diff = math.min(pNow - pLast, 0); -- limits indent to 10
		if diff > -10 then
			pNow = pLast - 10;
			pIndent = math.min(pIndent + 10 + diff, 60);
		else
			pIndent = 4;
		end;
		text:SetPoint("TOPLEFT", pIndent, pNow);
		pLast = pNow;
		
		i = i+1;
	end;
	text = getglobal(basename.."_LogText"..i);
	while text do
		text:SetText("");
		text:SetPoint("TOPLEFT", 0, 0);
		i = i+1;
		text = getglobal(basename.."_LogText"..i);
	end;
end;

--[[ Log View ]]
BarTracker_LogFilter = "";
BarTracker_LogFiltered = {};
BarTracker_LogFrameLines = 10;
BarTracker_LogFrameLineHeight = 16;
BarTracker_LogViewUpdate = function ()
	BarTracker_ResizeFrame();
	if not BarTracker_Log["Combatlog"] then return; end;
	
	local logStart = BarTracker_Log["StartTimeStamp"];
	local numEntries = table.getn(BarTracker_LogFiltered);
	local numAll = table.getn(BarTracker_Log["Combatlog"]);
	local offset = math.floor(FauxScrollFrame_GetOffset(BarTracker_LogViewer_ScrollFrame));
	if numEntries < BarTracker_LogFrameLines then
		offset = 0;
	elseif numEntries-offset < BarTracker_LogFrameLines then
		offset = numEntries-BarTracker_LogFrameLines;
	end;

	for i = 1,BarTracker_LogFrameLines,1 do
		local logEntry = getglobal("BarTracker_LogViewer".."_LogText"..i);
		if BarTracker_LogFiltered[i+offset] then
			local logEntryText = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Text");
			local logEntryTime = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Time");
			local logEntryHealth = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Health");
			local logEntryPower = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Power");
			local str = BarTracker_LogFiltered[i+offset][2];
			local t = BarTracker_LogFiltered[i+offset][1];
			local red, green, blue = BarTracker_ColorForLogEntry(BarTracker_LogFiltered[i+offset]);
			local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
			local h, m = BarTracker_StatsAtTime(t, powerType);
			logEntryText:SetText(str);
			logEntryText:SetTextColor(red, green, blue, 1);
			logEntryTime:SetText(string.format("%2.1f s",t-logStart));
			logEntryHealth:SetText(h);
			logEntryPower:SetText(m);
			logEntry.time = t-logStart;
			logEntry:Show();
		else
			logEntry:Hide();
		end;
	end;

	getglobal("BarTracker_LogViewer_Count"):SetText(numEntries.."/"..numAll);
	
	FauxScrollFrame_Update(BarTracker_LogViewer_ScrollFrame,
		numEntries,
		BarTracker_LogFrameLines,
		BarTracker_LogFrameLineHeight);
end;

BarTracker_FilterLog = function (logInput, pattern)
	local logOutput = {};
	for i, l in ipairs(logInput) do
		if string.find(l[2], pattern) then
			table.insert(logOutput, l);
		end;
	end;
	BarTracker_LogFiltered = logOutput;
	BarTracker_LogViewUpdate();
	return logOutput;
end;

BarTracker_ShowLogViewer = function ()
	BarTracker_LogViewUpdate();
	BarTracker_LogViewer:Show();
end;

BarTracker_ResizeFrame = function ()
	BarTracker_LogFrameLines = math.floor((BarTracker_LogViewer:GetHeight()-32)/BarTracker_LogFrameLineHeight);
	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local logEntry;
	local hc = BarTracker_Settings["Color"]["Health"];
	local mc = BarTracker_Settings["Color"][powerType];
	for i = 1,BarTracker_LogFrameLines,1 do
		logEntry = getglobal("BarTracker_LogViewer".."_LogText"..i);
		if logEntry == nil then
			logEntry = CreateFrame(
				"Button",
				"BarTracker_LogViewer".."_LogText"..i,
				BarTracker_LogViewer, -- BarTracker_LogViewer_ScrollFrame
				"BarTracker_LogLineFrameTemplate");
			logEntry:SetPoint("TOPLEFT", BarTracker_LogViewer_ScrollFrame,
				"TOPLEFT", 0, -(i-1)*BarTracker_LogFrameLineHeight);
			logEntry:SetPoint("BOTTOMRIGHT", BarTracker_LogViewer_ScrollFrame,
				"TOPRIGHT", 0, -(i)*BarTracker_LogFrameLineHeight);
			local logEntryH = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Health");
			local logEntryP = getglobal("BarTracker_LogViewer".."_LogText"..i.."_Power");
			logEntryH:SetTextColor(hc[1], hc[2], hc[3]);
			logEntryP:SetTextColor(mc[1], mc[2], mc[3]);
		end;
	end;
	local i = 1;
	logEntry = getglobal("BarTracker_LogViewer".."_LogText"..i);
	while logEntry do
		logEntry:Hide();
		i = i + 1;
		logEntry = getglobal("BarTracker_LogViewer".."_LogText"..i);
	end;
end;

BarTracker_ScrollToTime = function (t)
	local pointInTime = BarTracker_Log["StartTimeStamp"]+t;
	local i = 1; -- linear search! omg!
	while BarTracker_LogFiltered[i] and BarTracker_LogFiltered[i][1] < pointInTime do
		i = i + 1;
	end;
	i = math.floor(math.max(0, i-BarTracker_LogFrameLines/2));
	BarTracker_LogViewer_ScrollFrameScrollBar:SetValue(i*BarTracker_LogFrameLineHeight);
	FauxScrollFrame_SetOffset(BarTracker_LogViewer_ScrollFrame, i);
	BarTracker_LogViewUpdate();
end;

-- [[ Tooltip Functions ]]
BarTracker_OnMouseEnter = function (self)
	if (BarTracker_Tooltip:IsOwned(UIParent)) then
		BarTracker_Tooltip:ClearLines();
	else
		local anchor;
		local x, y = self:GetCenter();
		local scale = self:GetScale();
		if x*scale > GetScreenWidth()/2 then
			anchor = "ANCHOR_LEFT";
		else
			anchor = "ANCHOR_RIGHT";
		end;
		BarTracker_Tooltip:SetOwner(self, anchor);
	end;
	-- Fill the Tooltip
	BarTracker_Tooltip:AddLine("BarTracker "..BarTracker_Version, 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Start Time", BarTracker_Log["StartTime"], 1,1,1, 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Duration",
		string.format("%2.1f",BarTracker_Log["EndTimeStamp"]-BarTracker_Log["StartTimeStamp"]).."s", 1,1,1, 1,1,1);
	local color = BarTracker_Settings["Color"]["Health"];
	BarTracker_Tooltip:AddDoubleLine("Health", nil, color[1],color[2],color[3], color[1],color[2],color[3]);
	BarTracker_Tooltip:AddDoubleLine("Max", BarTracker_Log["HealthMax"], color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Min", BarTracker_Log["HealthMin"], color[1],color[2],color[3], 1,1,1);
	local gain, loss, maxGain, maxLoss = BarTracker_CalculateChanges("Health");
	BarTracker_Tooltip:AddDoubleLine("Gain", gain, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Max Gain", maxGain, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Loss", loss, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Max Loss", maxLoss, color[1],color[2],color[3], 1,1,1);

	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local color = BarTracker_Settings["Color"][powerType];
	BarTracker_Tooltip:AddDoubleLine(powerType, nil, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Max", BarTracker_Log[powerType.."Max"], color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Min", BarTracker_Log[powerType.."Min"], color[1],color[2],color[3], 1,1,1);
	local gain, loss, maxGain, maxLoss = BarTracker_CalculateChanges(powerType);
	BarTracker_Tooltip:AddDoubleLine("Gain", gain, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Max Gain", maxGain, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Loss", loss, color[1],color[2],color[3], 1,1,1);
	BarTracker_Tooltip:AddDoubleLine("Max Loss", maxLoss, color[1],color[2],color[3], 1,1,1);
	
	BarTracker_Tooltip:Show();
	
	if BarTracker_Settings["Combatlog"] then
		BarTracker_MainFrame:SetScript("OnUpdate", BarTracker_TooltipOnUpdate);
	end;
end;

BarTracker_OldX = -1;
BarTracker_OldY = -1;
BarTracker_TooltipOnUpdate = function (self, elapsed)
	local mx, my = GetCursorPosition();
	if mx == BarTracker_OldX and my == BarTracker_OldY then
		return;
	else
		BarTracker_OldX = mx;
		BarTracker_OldY = my;
	end;
	local x = BarTracker_OutputFrame0:GetLeft() + BarTracker_hBorder/2;
	local y = BarTracker_OutputFrame0:GetTop() - BarTracker_vBorder/2;
	local scaleL, scaleW;
	scaleL = BarTracker_OutputFrame0:GetScale();
	scaleW = UIParent:GetScale();
	mx = mx/scaleW;
	my = my/scaleW;
	x = x*scaleL;
	y = y*scaleL;
	x = (mx - x)/scaleL;
	y = (my - y)/scaleL;
	local heightScale;
	if BarTracker_Settings["TimeScale"] > 0 then
		heightScale = BarTracker_Settings["TimeScale"];
	else
		heightScale = (BarTracker_OutputFrame0:GetHeight()-BarTracker_vBorder)/
			(BarTracker_Log["EndTimeStamp"]-BarTracker_Log["StartTimeStamp"]+1);
	end;
	
	if (BarTracker_TooltipLog:IsOwned(UIParent)) then
		BarTracker_TooltipLog:ClearLines();
	else
		local anchor, anchorO;
		local x, y = BarTracker_OutputFrame0:GetCenter();
		local scale = BarTracker_OutputFrame0:GetScale();
		if x*scale > GetScreenWidth()/2 then
			anchor = "TOPRIGHT";
			anchorO = "TOPLEFT";
		else
			anchor = "TOPLEFT";
			anchorO = "TOPRIGHT";
		end;
		BarTracker_TooltipLog:SetOwner(BarTracker_OutputFrame0, "ANCHOR_NONE");
		BarTracker_TooltipLog:SetPoint(anchor, BarTracker_OutputFrame0, anchorO);
	end;
	
	local i = 1;
	local log = BarTracker_Log["Combatlog"];
	local logStart = BarTracker_Log["StartTimeStamp"];
	local logTime = -y/heightScale + logStart;

	local powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	local hc = BarTracker_Settings["Color"]["Health"];
	local mc = BarTracker_Settings["Color"][powerType];
	local h,m = BarTracker_StatsAtTime(logTime, powerType);
	
	BarTracker_TooltipLog:AddDoubleLine("Combatlog", string.format("%2.1f", logTime-logStart), 1,1,1, 1,1,1);
	BarTracker_TooltipLog:AddDoubleLine("Health: "..h,
		powerType..": "..m,
		hc[1],hc[2],hc[3], mc[1],mc[2],mc[3]);

	if (not log) or (not log[1]) then return; end;
	while log[i] and log[i][1] < logTime do -- linear search! omg!
		i = i+1;
	end;
	i = math.max(i - 3,1);
	local j = 0;
	while log[i] and j < 7 do
		local str = log[i][2];
		local red, green, blue = BarTracker_ColorForLogEntry(log[i]);
		BarTracker_TooltipLog:AddDoubleLine(str, string.format("%2.1f",log[i][1]-logStart),
			red, green, blue, 0.75,0.75,0.75);
		i = i+1;
		j = j+1;
	end;
	BarTracker_TooltipLog:Show();
end;

BarTracker_StatsAtTime = function (logTime, powerType)
	if not powerType then
		powerType = BarTracker_PowerTranslation[UnitPowerType("player")];
	end;
	local iH = 1;
	local iM = 1;
	local logH = BarTracker_Log["Health"];
	local logM = BarTracker_Log[powerType];
	while logH[iH] and logH[iH][2] < logTime do -- linear search! omg!
		iH = iH+1;
	end;
	while logM[iM] and logM[iM][2] < logTime do -- linear search! omg!
		iM = iM+1;
	end;
	
	return logH[math.max(1,iH-1)][1], logM[math.max(1,iM-1)][1];
end;

BarTracker_ColorForLogEntry = function (logEntry)
	if logEntry[3] and logEntry[4] and logEntry[5] then
		return logEntry[3], logEntry[4], logEntry[5];
	else	
		return 0.75, 0.75, 0.75;
	end;
end;

BarTracker_CalculateChanges = function (Stat)
	local gain = 0;
	local loss = 0;
	local maxGain = 0;
	local maxLoss = 0;
	local last = BarTracker_Log[Stat][1][1];
	for i, a in ipairs(BarTracker_Log[Stat]) do
		local change = a[1] - last;
		if change > 0 then
			gain = gain + change;
			if change > maxGain then maxGain = change; end;
		else
			loss = loss - change;
			if change < maxLoss then maxLoss = change; end;
		end;
		last = a[1];
	end;
	return gain, loss, maxGain, -maxLoss;
end;

BarTracker_OnMouseLeave = function (self)
	BarTracker_Tooltip:Hide();
	BarTracker_TooltipLog:Hide();
	BarTracker_MainFrame:SetScript("OnUpdate", nil);
end;

BarTracker_PrintLog = function (stat)
	for _, a in pairs(BarTracker_Log[stat]) do
		BarTracker_MSG(a[2] .. ": " .. a[1]);
	end;
end;

--[[ Settings ]]
BarTracker_ToggleShown = function (state)
	BarTracker_TogglePref("Show", state);
	if BarTracker_Settings["Show"] then
		BarTracker_OutputFrame0:Show();
	else
		BarTracker_OutputFrame0:Hide();
	end;
end;

BarTracker_SetTimescale = function (scale)
	BarTracker_SetPrefNum("TimeScale", scale);
	if BarTracker_Settings["TimeScale"] == 0 then
		BarTracker_VisualModeScale();
	else
		BarTracker_VisualModeScroll();
	end;
	BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
end;

BarTracker_SetBattleMode = function (mode)
	if mode == "auto" then
		BarTracker_Settings["BattleMode"] = "auto";
		BarTracker_MSG("BT:  Battle Mode set to Automatic");
	elseif mode == "manual" then
		BarTracker_Settings["BattleMode"] = "manual";
		BarTracker_MSG("BT:  Battle Mode set to Manual");
	else
		BarTracker_MSG("BT: Battle Mode Input Error");
	end;
end;

BarTracker_TogglePref = function (setting, state)
	local array = BarTracker_Settings;
	if state == "off" or state == "false" then
		array[setting] = false;
		BarTracker_MSG("BT: "..setting.." Disabled");
	elseif state == "on" or state == "true" then
		array[setting] = true;
		BarTracker_MSG("BT: "..setting.." Enabled");
	else
		array[setting] = not array[setting];
		if array[setting] then
			BarTracker_MSG("BT: "..setting.." Enabled");
		else
			BarTracker_MSG("BT: "..setting.." Disabled");
		end;
	end;
end;

BarTracker_SetPrefNum = function (setting, value)
	local valueNum = tonumber(value);
	if valueNum == nil then
		BarTracker_MSG("BT: "..setting.." Input Error, "..value.." is not a number");
	else 
		BarTracker_Settings[setting] = valueNum;
		BarTracker_MSG("BT: "..setting.." set to "..valueNum);
	end;
end;

BarTracker_SlashHandler = function (msg)
 	local Cmd, SubCmd = BarTracker_GetCmd(msg);
 	Cmd = string.lower(Cmd);
 	if (Cmd == "show") then
		BarTracker_ToggleShown(SubCmd);
	elseif (Cmd == "scale") then
		BarTracker_SetPrefNum("Scale", SubCmd);
		BarTracker_OutputFrame0:SetScale(BarTracker_Settings["Scale"]);
	elseif (Cmd == "alpha") then
		BarTracker_SetPrefNum("Alpha", SubCmd);
		BarTracker_OutputFrame0:SetAlpha(BarTracker_Settings["Alpha"]);
	elseif (Cmd == "timescale") then
		BarTracker_SetTimescale(SubCmd);
	elseif (Cmd == "timeout") then
		BarTracker_SetPrefNum("Timeout", SubCmd);
	elseif (Cmd == "combatlog") then
		BarTracker_TogglePref("Combatlog", SubCmd);
	elseif (Cmd == "targethealth") then
		BarTracker_TogglePref("TargetHealth", SubCmd);
	elseif (Cmd == "combatfade") then
		BarTracker_TogglePref("CombatFade", SubCmd);
	elseif (Cmd == "liveupdate") then
		BarTracker_TogglePref("LiveUpdate", SubCmd);
	elseif (Cmd == "battlemode") then
		BarTracker_SetBattleMode(SubCmd);
	elseif (Cmd == "print") then
		BarTracker_PrintLog(SubCmd);
	elseif (Cmd == "rebuild") then
		BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
	elseif (Cmd == "load") then
		BarTracker_LoadLog(SubCmd);
	elseif (Cmd == "store") then
		BarTracker_StoreLog(SubCmd);
	elseif (Cmd == "list") then
		BarTracker_ListLog();
	elseif (Cmd == "delete") then
		BarTracker_DeleteLog(SubCmd);
	elseif (Cmd == "start") then
		BarTracker_EnterCombat();
	elseif (Cmd == "pause") then
		BarTracker_LeaveCombat();
 	else
 		BarTracker_MSG("|cffccccccBarTracker v"..BarTracker_Version);
 		BarTracker_MSG("Shift Click to Move, Alt Click to resize, Ctrl Click for menu");
		BarTracker_MSG("/bartracker show [on|off]");
		BarTracker_MSG("/bartracker scale <num>");
		BarTracker_MSG("/bartracker battlemode [auto|manual]");
		BarTracker_MSG("/bartracker timeout <num>");
		BarTracker_MSG("/bartracker timescale <num>");
		BarTracker_MSG("/bartracker combatlog [on|off]")
		BarTracker_MSG("/bartracker combatfade [on|off]")
		BarTracker_MSG("/bartracker liveupdate [on|off]");
		BarTracker_MSG("/bartracker load <numame>");
		BarTracker_MSG("/bartracker store <numame>");
		BarTracker_MSG("/bartracker list");
		BarTracker_MSG("/bartracker delete <numame>");
		BarTracker_MSG("/bartracker start");
		BarTracker_MSG("/bartracker pause");
 	end;
end;

BarTracker_MSG = function (msg)
	DEFAULT_CHAT_FRAME:AddMessage(msg);
end;

BarTracker_Transforming = false;
BarTracker_FrameMouseDown = function (self, button)
	if (button == "LeftButton") then
		if (IsShiftKeyDown()) then
			self:StartMoving();
			BarTracker_Transforming = true;
		elseif (IsAltKeyDown()) then
			local w = self:GetWidth();
			local h = self:GetHeight();
			local x, y = self:GetCenter();
			local mx, my = GetCursorPosition();
			local scaleL, scaleW;
			scaleL = self:GetScale();
			scaleW = UIParent:GetScale();
			mx = mx/scaleW;
			my = my/scaleW;
			x = x*scaleL;
			y = y*scaleL;
			x = mx - x;
			y = my - y;
			x = x/w*h;
			if (abs(x) > abs(y)) then
				if (x > 0) then
					self:StartSizing("RIGHT");
				else
					self:StartSizing("LEFT");
				end;
			else
				if (y > 0) then
					self:StartSizing("TOP");
				else
					self:StartSizing("BOTTOM");
				end;
			end;
			BarTracker_Transforming = true;
		end;
	elseif (button == "RightButton") then
		ToggleDropDownMenu(1, nil, BarTracker_Menu, "cursor", 0, 0);
	end;
end;

--[[
List of button attributes
======================================================
info.text = [STRING]  --  The text of the button
info.value = [ANYTHING]  --  The value that UIDROPDOWNMENU_MENU_VALUE is set to when the button is clicked
info.func = [function()]  --  The function that is called when you click the button
info.checked = [nil, true, function]  --  Check the button if true or function returns true
info.isTitle = [nil, true]  --  If it's a title the button is disabled and the font color is set to yellow
info.disabled = [nil, true]  --  Disable the button and show an invisible button that still traps the mouseover event so menu doesn't time out
info.hasArrow = [nil, true]  --  Show the expand arrow for multilevel menus
info.hasColorSwatch = [nil, true]  --  Show color swatch or not, for color selection
info.r = [1 - 255]  --  Red color value of the color swatch
info.g = [1 - 255]  --  Green color value of the color swatch
info.b = [1 - 255]  --  Blue color value of the color swatch
info.colorCode = [STRING] -- "|cAARRGGBB" embedded hex value of the button text color. Only used when button is enabled
info.swatchFunc = [function()]  --  Function called by the color picker on color change
info.hasOpacity = [nil, 1]  --  Show the opacity slider on the colorpicker frame
info.opacity = [0.0 - 1.0]  --  Percentatge of the opacity, 1.0 is fully shown, 0 is transparent
info.opacityFunc = [function()]  --  Function called by the opacity slider when you change its value
info.cancelFunc = [function(previousValues)] -- Function called by the colorpicker when you click the cancel button (it takes the previous values as its argument)
info.notClickable = [nil, 1]  --  Disable the button and color the font white
info.notCheckable = [nil, 1]  --  Shrink the size of the buttons and don't display a check box
info.owner = [Frame]  --  Dropdown frame that "owns" the current dropdownlist
info.keepShownOnClick = [nil, 1]  --  Don't hide the dropdownlist after a button is clicked
info.tooltipTitle = [nil, STRING] -- Title of the tooltip shown on mouseover
info.tooltipText = [nil, STRING] -- Text of the tooltip shown on mouseover
info.justifyH = [nil, "CENTER"] -- Justify button text
info.arg1 = [ANYTHING] -- This is the first argument used by info.func
info.arg2 = [ANYTHING] -- This is the second argument used by info.func
info.fontObject = [FONT] -- font object replacement for Normal and Highlight
info.menuTable = [TABLE] -- This contains an array of info tables to be displayed as a child menu
]]

BarTracker_Menu_Initialize = function (frame, level)
	local checked;
	if level == 1 then
		UIDropDownMenu_AddButton({
			["text"] = "BarTracker v"..BarTracker_Version;
			["isTitle"] = 1;
			["notCheckable"] = 1;
			["owner"] = BarTracker_OutputFrame0;
			});
		if BarTracker_Settings["BattleMode"] == "manual" then
		UIDropDownMenu_AddButton({
			["text"] = "Start";
			["notCheckable"] = 1;
			["func"] = BarTracker_EnterCombat;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Pause";
			["notCheckable"] = 1;
			["func"] = BarTracker_LeaveCombat;
			});
		end;
		UIDropDownMenu_AddButton({
			["text"] = "Show Log View";
			["notCheckable"] = 1;
			["func"] = BarTracker_ShowLogViewer;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Show Fullscreen";
			["notCheckable"] = 1;
			["func"] = BarTracker_ShowFullscreen;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Store";
			["notCheckable"] = 1;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_StoreName.Show;
			["arg2"] = BarTracker_StoreName;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Load";
			["value"] = "Load";
			["notCheckable"] = 1;
			["hasArrow"] = 1;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Hide";
			["notCheckable"] = 1;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_ToggleShown;
			["arg2"] = "off";
			});
		UIDropDownMenu_AddButton({
			["text"] = "Options";
			["isTitle"] = 1;
			["notCheckable"] = 1;
			});
		UIDropDownMenu_AddButton({
			["text"] = "BattleMode";
			["value"] = "BattleMode";
			["notCheckable"] = 1;
			["hasArrow"] = 1;
			});
		UIDropDownMenu_AddButton({
			["text"] = "Timescale";
			["value"] = "Timescale";
			["notCheckable"] = 1;
			["hasArrow"] = 1;
			});
		UIDropDownMenu_AddButton({
			["text"] = "OOC Timeout";
			["value"] = "Timeout";
			["notCheckable"] = 1;
			["hasArrow"] = 1;
			});
		if BarTracker_Settings["Combatlog"] then
			checked = 1;
		else
			checked = nil;
		end;
		UIDropDownMenu_AddButton({
			["text"] = "Combatlog";
			["value"] = "Combatlog";
			["checked"] = checked;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_TogglePref;
			["arg2"] = "Combatlog";
			});
		if BarTracker_Settings["TargetHealth"] then
			checked = 1;
		else
			checked = nil;
		end;
		UIDropDownMenu_AddButton({
			["text"] = "Target Health";
			["value"] = "TargetHealth";
			["checked"] = checked;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_TogglePref;
			["arg2"] = "TargetHealth";
			});
		if BarTracker_Settings["CombatFade"] then
			checked = 1;
		else
			checked = nil;
		end;
		UIDropDownMenu_AddButton({
			["text"] = "CombatFade";
			["value"] = "CombatFade";
			["checked"] = checked;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_TogglePref;
			["arg2"] = "CombatFade";
			});
		if BarTracker_Settings["LiveUpdate"] then
			checked = 1;
		else
			checked = nil;
		end;
		UIDropDownMenu_AddButton({
			["text"] = "LiveUpdate";
			["value"] = "LiveUpdate";
			["checked"] = checked;
			["func"] = BarTracker_Menu_Exec;
			["arg1"] = BarTracker_TogglePref;
			["arg2"] = "LiveUpdate";
			});
	elseif level == 2 then
		if UIDROPDOWNMENU_MENU_VALUE == "Timescale" then
			UIDropDownMenu_AddButton({
				["text"] = "Autoscale";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetTimescale;
				["arg2"] = 0;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "1";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetTimescale;
				["arg2"] = 1;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "2";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetTimescale;
				["arg2"] = 2;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "5";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetTimescale;
				["arg2"] = 5;
				}, 2);
		elseif UIDROPDOWNMENU_MENU_VALUE == "Timeout" then
			UIDropDownMenu_AddButton({
				["text"] = "off";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_SetPref;
				["arg1"] = "Timeout";
				["arg2"] = 0;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "15 secs";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_SetPref;
				["arg1"] = "Timeout";
				["arg2"] = 15;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "30 secs";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_SetPref;
				["arg1"] = "Timeout";
				["arg2"] = 30;
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "60 secs";
				["notCheckable"] = 1;
				["func"] = BarTracker_Menu_SetPref;
				["arg1"] = "Timeout";
				["arg2"] = 60;
				}, 2);
		elseif UIDROPDOWNMENU_MENU_VALUE == "BattleMode" then
			checked = BarTracker_Settings["BattleMode"] == "auto";
			UIDropDownMenu_AddButton({
				["text"] = "Automatic";
				["checked"] = checked;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetBattleMode;
				["arg2"] = "auto";
				}, 2);
			UIDropDownMenu_AddButton({
				["text"] = "Manual";
				["checked"] = not checked;
				["func"] = BarTracker_Menu_Exec;
				["arg1"] = BarTracker_SetBattleMode;
				["arg2"] = "manual";
				}, 2);
		elseif UIDROPDOWNMENU_MENU_VALUE == "Load" then
			local i = 1;
			for name, log in pairs(BarTracker_StoredLogs) do
				if i > 20 then	
					UIDropDownMenu_AddButton({
						["text"] = "More...";
						["notCheckable"] = 1;
						["disabled"] = 1;
						}, 2);
					return;
				end;
				UIDropDownMenu_AddButton({
					["text"] = name.." @ "..log["StartTime"];
					["notCheckable"] = 1;
					["func"] = BarTracker_Menu_Exec;
					["arg1"] = BarTracker_LoadLog;
					["arg2"] = name;
					}, 2);
				i = i + 1;
			end;
		end;
	end
end;

BarTracker_Menu_Exec = function (self, func, arg)
	func(arg);
end;

BarTracker_Menu_SetPref = function (self, setting, value)
	BarTracker_SetPrefNum(setting, value);
end;

BarTracker_Menu_Load = function (self)
	UIDropDownMenu_Initialize(self, BarTracker_Menu_Initialize, "MENU");
end;

BarTracker_FrameResizing = function ()
	local height = BarTracker_OutputFrame0:GetHeight();
	BarTracker_MaxWidth = (BarTracker_OutputFrame0:GetWidth()-BarTracker_hBorder)/2;
	BarTracker_RebuildAllDiagrams("BarTracker_OutputFrame0");
end;

-- Some CMD parsing code, written by Tigerheart on http://www.wowwiki.com/HOWTO:_Extract_Info_from_a_Slash_Command
BarTracker_GetCmd = function (msg)
	if msg then
		local a,b,c=string.find(msg, "(%S+)"); --contiguous string of non-space characters
		if a then
			return c, string.sub(msg, b+2);
		else	
			return "";
		end
	end
end;

BarTracker_GetArgument = function (msg)
	if msg then
		local a,b=strfind(msg, "=");
		if a then
			return strsub(msg,1,a-1), strsub(msg, b+1);
		else
			return "";
		end
	end
end;
