-------------------------------------------------------------------------------
-- Localization                                                              --
-------------------------------------------------------------------------------

local al = LibStub("AceLocale-3.0")

do
	local L = al:NewLocale("BanzaiAlert", "enUS", true)
	L["Sound alert"] = true
	L["Plays a sound alert when you get aggro."] = true
	L["Only in group"] = true
	L["Only alert for aggro when you are grouped."] = true
	L["Health frame flash"] = true
	L["Flash the health frame when you get aggro."] = true
	L["Lost aggro"] = true
	L["Alerts you when you lose aggro."] = true

	L["None"] = true

	L["Aggro!"] = true
	L["Lost aggro!"] = true

	L = al:NewLocale("BanzaiAlert", "koKR")
	if L then
		L["Sound alert"] = "경고 효과음"
		L["Plays a sound alert when you get aggro."] = "어그로 획득시 경고 효과음을 재생합니다."
		L["Only in group"] = "파티만 사용"
		L["Only alert for aggro when you are grouped."] = "파티에 속해 있을 때에만 경고를 사용합니다."
		L["Health frame flash"] = "화면 번쩍임"
		L["Flash the health frame when you get aggro."] = "어그로 획득시 생명력 프레임(화면 외곽)을 번쩍입니다."
		L["Lost aggro"] = "어그로 손실"
		L["Alerts you when you lose aggro."] = "어그로 손실시 알립니다."

		L["None"] = "없음"

		L["Aggro!"] = "어그로 획득!"
		L["Lost aggro!"] = "어그로 손실!"
	end

	L = al:NewLocale("BanzaiAlert", "zhCN")
	if L then
		L["Sound alert"] = "声音警报"
		L["Plays a sound alert when you get aggro."] = "当你被攻击时发出警报。"
		L["Only in group"] = "队伍模式"
		L["Only alert for aggro when you are grouped."] = "只有在队伍中时发出被攻击警报。"
		L["Health frame flash"] = "全屏闪光警告"
		L["Flash the health frame when you get aggro."] = "在你被攻击后发出全屏闪光警报。"
		L["Lost aggro"] = "失去被攻击"
		L["Alerts you when you lose aggro."] = "当你失去被攻击时发出警报。"
	
		L["Aggro!"] = "被攻击！"
		L["Lost aggro!"] = "失去被攻击！"
	end

	L = al:NewLocale("BanzaiAlert", "zhTW")
	if L then
		L["Sound alert"] = "聲音警告"
		L["Plays a sound alert when you get aggro."] = "當你得到Aggro時發出聲音警告"
		L["Only in group"] = "隊伍模式"
		L["Only alert for aggro when you are grouped."] = "只有當你在隊伍裡時發出Aggro警告"
		L["Health frame flash"] = "全螢幕閃光警告"
		L["Flash the health frame when you get aggro."] = "在你得到Aggro後發出全螢幕閃光警告"
		L["Lost aggro"] = "失去Aggro"
		L["Alerts you when you lose aggro."] = "當你失去Aggro時發出警告"

		L["Lost aggro!"] = "失去 Aggro！"
	end

	L = al:NewLocale("BanzaiAlert", "esES")
	if L then
		L["Sound alert"] = "Alerta sonora"
		L["Plays a sound alert when you get aggro."] = "Reproduce un sonido de alerta cuando tienes aggro."
		L["Only in group"] = "Solo en grupo"
		L["Only alert for aggro when you are grouped."] = "Solo alertar de aggro cuando est\195\161s en un grupo"
		L["Health frame flash"] = "Destello del marco de salud"
		L["Flash the health frame when you get aggro."] = "El marco de salud se ilumina cuando tienes aggro."
		L["Lost aggro"] = "Aggro perdido"
		L["Alerts you when you lose aggro."] = "Te avisa cuando pierdes aggro."

		L["Lost aggro!"] = "\194\161Has perdido aggro!"
	end

	L = al:NewLocale("BanzaiAlert", "deDE")
	if L then
		L["Sound alert"] = "Sound Alarm"
		L["Plays a sound alert when you get aggro."] = "Spielt einen WarnSound wenn du Aggro bekommst."
		L["Only in group"] = "Nur in Gruppe"
		L["Only alert for aggro when you are grouped."] = "Zeigt den Aggro Alarm nur wenn du in einer Gruppe bist."
		L["Health frame flash"] = "Bildschirmrand aufblitzen"
		L["Flash the health frame when you get aggro."] = "L\195\164st den Bildschirmrahmen aufblitzen wenn du Aggro bekommst."
		L["Lost aggro"] = "Aggro verlust"
		L["Alerts you when you lose aggro."] = "Meldet dir wenn du Aggro verlierst."
		L["None"] = "Keinen"

		L["Lost aggro!"] = "Aggro verloren!"
	end

	L = al:NewLocale("BanzaiAlert", "frFR")
	if L then
		L["Sound alert"] = "Alerte sonore"
		L["Plays a sound alert when you get aggro."] = "Emet une alerte sonore quand vous prenez l'aggro."
		L["Only in group"] = "Seulement en groupe"
		L["Only alert for aggro when you are grouped."] = "N'alerte pour l'aggro que si vous êtes en groupe."
		L["Health frame flash"] = "Alerte visuelle"
		L["Flash the health frame when you get aggro."] = "Fais flasher les bords de l'écran lorsque vous prenez l'aggro."
		L["Lost aggro"] = "Alerte perte d'aggro"
		L["Alerts you when you lose aggro."] = "Alerte lorsque vous perdez l'aggro."
		L["None"] = "Aucun"

		L["Lost aggro!"] = "Aggro perdue !"
	end
end

-------------------------------------------------------------------------------
-- Locals
-------------------------------------------------------------------------------

local L = al:GetLocale("BanzaiAlert")

local sml = LibStub("LibSharedMedia-3.0", true)
local banzai = AceLibrary("LibBanzai-2.0")

local playerName = UnitName("player")
local lastAggroAlert = nil

local sounds = {
	babe = "Interface\\AddOns\\BanzaiAlert\\sounds\\babe.wav",
	zenham = "Interface\\AddOns\\BanzaiAlert\\sounds\\zenham.wav",
	BellTollAlliance = "Sound\\Doodad\\BellTollAlliance.wav",
	bell = "Sound\\interface\\AuctionWindowClose.wav",
}

-------------------------------------------------------------------------------
-- Addon declaration                                                         --
-------------------------------------------------------------------------------

BanzaiAlert = CreateFrame("Frame")
local addon = BanzaiAlert
LibStub("LibSink-2.0"):Embed(BanzaiAlert)

local options = {
	type = "group",
	handler = addon,
	get = function(info)
		return addon.db.profile[info[1]]
	end,
	set = function(info, value)
		addon.db.profile[info[1]] = value
		if info[1] == "onlyInGroup" then
			addon:ToggleOnlyInGroup()
		end
	end,
	args = {
		sound = {
			type = "select",
			name = L["Sound alert"],
			desc = L["Plays a sound alert when you get aggro."],
			width = "full",
			order = 1,
		},
		onlyInGroup = {
			type = "toggle",
			name = L["Only in group"],
			desc = L["Only alert for aggro when you are grouped."],
			width = "full",
			order = 2,
		},
		flash = {
			type = "toggle",
			name = L["Health frame flash"],
			desc = L["Flash the health frame when you get aggro."],
			width = "full",
			order = 3,
		},
		lost = {
			type = "toggle",
			name = L["Lost aggro"],
			desc = L["Alerts you when you lose aggro."],
			width = "full",
			order = 4,
		},
	},
}

-------------------------------------------------------------------------------
-- Initialization                                                            --
-------------------------------------------------------------------------------

local function init()
	if sml then
		sml:Register("sound", "BanzaiAlert: Babe", sounds.babe)
		sml:Register("sound", "BanzaiAlert: Zenham", sounds.zenham)
		sml:Register("sound", "BanzaiAlert: Ship Bell", sounds.BellTollAlliance)
		sml:Register("sound", "BanzaiAlert: Auction Close", sounds.bell)

		options.args.sound.values = sml:List("sound")
	else
		options.args.sound.values = {
			babe = "Babe",
			zenham = "Zenham",
			BellTollAlliance = "BellTollAlliance",
			bell = "Bell",
			none = L["None"],
		}
	end

	addon.db = LibStub("AceDB-3.0"):New("BanzaiAlertDB", {
		profile = {
			message = true,
			onlyInGroup = true,
			flash = true,
			lost = false,
			sound = "babe",
			sink20OutputSink = "UIErrorsFrame",
		},
	})

	addon:SetSinkStorage(addon.db.profile)

	LibStub("AceConfig-3.0"):RegisterOptionsTable("BanzaiAlert", options)
	LibStub("AceConfig-3.0"):RegisterOptionsTable("BanzaiAlert-Output", addon:GetSinkAce3OptionsDataTable())
	LibStub("AceConfigDialog-3.0"):AddToBlizOptions("BanzaiAlert", "BanzaiAlert")
	LibStub("AceConfigDialog-3.0"):AddToBlizOptions("BanzaiAlert-Output", "Output", "BanzaiAlert")

	-- Make sure we get our sound setting in order.
	if sml then
		if type(addon.db.profile.sound) ~= "string" or not sml:Fetch("sound", addon.db.profile.sound, true) then
			addon.db.profile.sound = "BanzaiAlert: Babe"
		end
	else
		if addon.db.profile.sound ~= L["None"] and not sounds[addon.db.profile.sound] then
			addon.db.profile.sound = "babe"
		end
	end

	lastAggroAlert = nil
	addon:ToggleOnlyInGroup()
end

addon:SetScript("OnEvent", function(self, event, addon)
	if event == "PLAYER_LOGIN" then
		init()
	elseif event == "PARTY_MEMBERS_CHANGED" then
		self:ToggleOnlyInGroup()
	end
end)
addon:RegisterEvent("PARTY_MEMBERS_CHANGED")
addon:RegisterEvent("PLAYER_LOGIN")

-------------------------------------------------------------------------------
-- Addon methods                                                             --
-------------------------------------------------------------------------------

local callbackRegistered = nil
local function callback(aggro, name, ...)
	if name ~= playerName then return end
	if aggro == 1 then
		BanzaiAlert:PlayerGainedAggro()
	elseif aggro == 0 then
		BanzaiAlert:PlayerLostAggro()
	end
end

function BanzaiAlert:ToggleOnlyInGroup()
	if self.db.profile.onlyInGroup and not UnitExists("party1") then
		if callbackRegistered then
			banzai:UnregisterCallback(callback)
			callbackRegistered = nil
		end
		if type(aUFBanzai) == "table" and type(aUFBanzai.Disable) == "function" then aUFBanzai:Disable() end
		if type(PitBull_Banzai) == "table" and type(PitBull_Banzai.Disable) == "function" then PitBull_Banzai:Disable() end
	elseif not callbackRegistered then
		banzai:RegisterCallback(callback)
		callbackRegistered = true
		if type(aUFBanzai) == "table" and type(aUFBanzai.Enable) == "function" then aUFBanzai:Enable() end
		if type(PitBull_Banzai) == "table" and type(PitBull_Banzai.Enable) == "function" then PitBull_Banzai:Enable() end
	end
end

function BanzaiAlert:Alert(gainedAggro)
	local text = gainedAggro and L["Aggro!"] or L["Lost aggro!"]
	self:Pour(text, 1, 0, 0)
end

-------------------------------------------------------------------------------
-- Events                                                                    --
-------------------------------------------------------------------------------

function BanzaiAlert:PlayerGainedAggro()
	if self.db.profile.onlyInGroup and GetNumPartyMembers() == 0 then
		return
	end

	if not lastAggroAlert or (GetTime() - lastAggroAlert) > 5 then
		if sml and type(self.db.profile.sound) == "string" then
			PlaySoundFile(sml:Fetch("sound", self.db.profile.sound))
		else
			if self.db.profile.sound ~= L["None"] and sounds[self.db.profile.sound] then
				local sound = sounds[self.db.profile.sound]
				if sound:find("%\\") then
					PlaySoundFile(sound)
				else
					PlaySound(sound)
				end
			end
		end
		self:Alert(true)
		if self.db.profile.flash then
			self:StartFlashing()
		end
		lastAggroAlert = GetTime()
	end
end

function BanzaiAlert:PlayerLostAggro()
	if self.db.profile.flash then
		self:StopFlashing()
	end

	if self.db.profile.onlyInGroup and GetNumPartyMembers() == 0 then
		return
	end

	if self.db.profile.lost then
		self:Alert()
	end
end

-------------------------------------------------------------------------------
-- Frame flashing                                                            --
-------------------------------------------------------------------------------

local total = nil
local function flash(self, elapsed)
	total = total + elapsed
	if total >= 1.2 then
		if banzai:GetUnitAggroByUnitName(playerName) then
			UIFrameFlash(LowHealthFrame, 0.6, 0.6, 1.2, false)
		else
			BanzaiAlert:StopFlashing()
		end
	end
end

function BanzaiAlert:StartFlashing()
	if not UIFrameIsFading(LowHealthFrame) then
		UIFrameFlash(LowHealthFrame, 0.6, 0.6, 1.2, false)
		total = 0
		self:SetScript("OnUpdate", flash)
	end
end

function BanzaiAlert:StopFlashing()
	if UIFrameIsFading(LowHealthFrame) then
		self:SetScript("OnUpdate", nil)
		UIFrameFlashRemoveFrame(LowHealthFrame)
		UIFrameFadeRemoveFrame(LowHealthFrame)
		UIFrameFadeOut(LowHealthFrame, 0.5, LowHealthFrame:GetAlpha(), 0)
	end
end

