------------------------------
--      Are you local?      --
------------------------------

local L = AceLibrary('AceLocale-2.2'):new("BagginsAnywhereBags")

local _G = _G
local type = _G.type
local pairs = _G.pairs
local ipairs = _G.ipairs
local tonumber = _G.tonumber
local time = _G.time
local date = _G.date
local floor = _G.floor

local strmatch = _G.strmatch
local format = _G.format
local strlower = _G.strlower
local strfind = _G.strfind
local GetContainerNumSlots = _G.GetContainerNumSlots
local GetContainerItemInfo = _G.GetContainerItemInfo
local GetContainerItemLink = _G.GetContainerItemLink
local GetItemInfo = _G.GetItemInfo
local GetInventoryItemTexture = _G.GetInventoryItemTexture
local GetInventoryItemLink = _G.GetInventoryItemLink
local GetInventoryItemCount = _G.GetInventoryItemCount
local GetInventoryItemQuality = _G.GetInventoryItemQuality
local GetInboxNumItems = _G.GetInboxNumItems
local GetInboxItemLink = _G.GetInboxItemLink
local GetInboxItem = _G.GetInboxItem
local SetItemRef = _G.SetItemRef
local GetCoinIcon = _G.GetCoinIcon

local argCheck = _G.AceLibrary.argCheck

local playerName = UnitName('player')
local Baggins = assert(_G.Baggins, "BagginsAnywhereBags requires Baggins!")
local r,p

----------------------------
--      Localization      --
----------------------------

L:RegisterTranslations('enUS', function() return {	
	["Anywhere Bags"] = true,
	["View any character's bank and bags from anywhere."] = true,
	["Compressed"] = true,
	["Search"] = true,
	["Search detailed"] = true,
	["Mail"] = true,
	["Displays the mail item nearest expiration."] = true,
	["Search results includes item locations."] = true,
	["Purge"] = true,
	["<itemLink, itemName, itemID>"] = true,
	["|cff7fff7fSearching all players for:|r "] = true,
	["Bags"] = true,
	["Equiped"] = true,
	["Keyring"] = true,
	["Mail"] = true,
	["Unknown id"] = true,
	["Empty Bag Slots: "] = true,
	["Empty Bank Slots: "] = true,
} end)

L:RegisterTranslations('zhCN', function() return {	
	["Anywhere Bags"] = "多角色背包",
	["View any character's bank and bags from anywhere."] = "点查看任意角色背包",
	["Compressed"] = "压缩的",
	["Search"] = "查找",
	["Search detailed"] = "查找详情",
	["Mail"] = "邮件",
	["Displays the mail item nearest expiration."] = "显示邮件物品最短的过期时间",
	["Search results includes item locations."] = "查找结果包括物品位置",
	["Purge"] = "清理",
	["<itemLink, itemName, itemID>"] = "<itemLink, itemName, itemID>",
	["|cff7fff7fSearching all players for:|r "] = "|cff7fff7f在所有角色查找:|r",
	["Bags"] = "背包",
	["Equiped"] = "可装备",
	["Keyring"] = "钥匙链",
	["Mail"] = "邮件",
	["Unknown id"] = "未知物品",
	["Empty Bag Slots: "] = "空背包位:",
	["Empty Bank Slots: "] = "空银行位:",
} end)

L:RegisterTranslations('zhTW', function() return {	
	["Anywhere Bags"] = "多角色背包",
	["View any character's bank and bags from anywhere."] = "查看任意角色背包",
	["Compressed"] = "壓縮的",
	["Search"] = "查找",
	["Search detailed"] = "查找細節",
	["Mail"] = "郵件",
	["Displays the mail item nearest expiration."] = "顯示郵件物品最短的過期時間",
	["Search results includes item locations."] = "察看結果 包括物品位置",
	["Purge"] = "清理",
	["<itemLink, itemName, itemID>"] = "<itemLink, itemName, itemID>",
	["|cff7fff7fSearching all players for:|r "] = "|cff7fff7f在所有角色查找:|r",
	["Bags"] = "背包",
	["Equiped"] = "可裝備",
	["Keyring"] = "鑰匙圈",
	["Mail"] = "研究",
	["Unknown id"] = "未知物品",
	["Empty Bag Slots: "] = "背包空位:",
	["Empty Bank Slots: "] = "銀行空位:",
} end)
---------------------------------
--      Addon Declaration      --
---------------------------------

BagginsAnywhereBags = AceLibrary('AceAddon-2.0'):new('AceEvent-2.0', 'AceDB-2.0', 'AceConsole-2.0', 'AceDebug-2.0', 'AceHook-2.1')
local BagginsAnywhereBags = BagginsAnywhereBags
BagginsAnywhereBags.revision = tonumber(string.sub('$Revision: 80990 $', 12, -3))

---------------------------------
--      AceOptions Config      --
---------------------------------

local options = {
	name = L["Anywhere Bags"], type='group',
	desc = L["View any character's bank and bags from anywhere."],
	handler = BagginsAnywhereBags,
	order = 200,
	wfHidden = true,
	cmdHidden = true,
	args = {
		search = {
			name = L["Search"], type='text',
			desc = L["Search"],
			get = false,
			set = "Search",
			usage = L["<itemLink, itemName, itemID>"],
			validate = function(v) return true end,
			order = -40,
		},
		searchd = {
			name = L["Search detailed"], type='text',
			desc = L["Search results includes item locations."],
			get = false,
			set = "Search",
			passValue = true, -- pass `true?to Search()
			usage = L["<itemLink, itemName, itemID>"],
			validate = function(v) return true end,
			order = -35,
		},
		mail = {
			name = L["Mail"], type='execute',
			desc = L["Displays the mail item nearest expiration."],
			func = "MailExpire",
			order = -32,
		},
		purge = {
			name = L["Purge"], type='group',
			desc = L["Purge"],
			pass = true,
			func = "PurgePlayer",
			args = {},
			order = -30,
		},
		spacer = {
			type='header',
			order = -20,
		},
		compressed = {
			name = L["Compressed"], type='toggle',
			desc = L["Compressed"],	
			get = function() return BagginsAnywhereBags.db.profile.compressed end,
			set = "SetCompressed",
			order = -10,
		},
	},
}
BagginsAnywhereBags.options = options

------------------------------
--      Initialization      --
------------------------------

function BagginsAnywhereBags:OnInitialize()
	self:RegisterDB("BagginsAnywhereBagsDB")
	self:RegisterDefaults('realm', {
		[playerName] = {
			[-2] = {}, [-1] = {}, [0] = {};
			-- 1 to 11:
			{},{},{},{},{},{},{},{},{},{},{};
			Equiped = {},
			Money = 0,
			Mail = {},
		},
		DBVersion = self.revision, --TODO: testing code, remove later
	})
	self:RegisterDefaults('profile', {
		compressed = true,
		DBVersion = self.revision, --TODO: testing code, remove later
	})
	self:RegisterChatCommand({"/BagginsAnywhereBags", "/BAB"}, {
		name = options.name, type='group',
		desc = options.desc,
		args = {
			search = options.args.search,
			searchd = options.args.searchd,
			mail = options.args.mail,
			purge = options.args.purge,
			compressed = options.args.compressed,
		},
	})
end

--[[----------------
items stored as:
BagginsAnywhereBags.db.realm[player][bag][slot] = {itemLink, count, texture, quality, Baggins_bagid[, Baggins_sectionid]}

itemLink regex:
itemstring, itemname = |H(.+)|h, |h(.+)|h
----------------]]--

function BagginsAnywhereBags:OnEnable(first)
	if not Baggins then return end
	
	r = self.db.realm
	p = self.db.profile
	
--~ 	self:SetDebugging(true)

	if first then
		--fix databases here
		if type(p.DBVersion) ~= 'number' then
			p.DBVersion = 3
		end
		if p.DBVersion < 4 then
			p.sortType = nil
			p.useFrames = nil
			p.DBVersion = 4
		end
		if p.DBVersion < 47189 then
			p.DBVersion = 47189
		end
		--
		if type(r.DBVersion) ~= 'number' then
			r.DBVersion = 3
		end
		if r.DBVersion < 4 then
			for player,bags in pairs(r) do
				if type(bags) == 'table' then
					for bag,slots in pairs(bags) do
						if bag == "Mail" or bag == -2 or bag == "Equiped" then
							for slot,item in pairs(slots) do
								item[5] = bag == -2 and L["Keyring"] or bag == "Mail" and L["Mail"] or bag == "Equiped" and L["Equiped"] or item[5]
								item[6] = nil
							end
						end
					end
				end
			end
			r.DBVersion = 4
		end
		if r.DBVersion < 47189 then
			r.DBVersion = 47189
		end
		--
	end
	
	--bank and mail need stored whenever they change
	--process on closed event, however it is sometimes fired once or twice
	--therefore register open event to register close event
	self:RegisterEvent('MAIL_SHOW', function() self:RegisterEvent('MAIL_CLOSED') end)
	self:RegisterEvent('BANKFRAME_OPENED', function() self:RegisterEvent('BANKFRAME_CLOSED') end)
	-- guildbank:
	self:RegisterEvent('GUILDBANKFRAME_OPENED', function() self:RegisterEvent('GUILDBANKFRAME_CLOSED') end)

	--only care about storing bags when logging out (you can always access your own bags)
	if type(_G['Logout']) == 'function' then
		self:Hook('Logout', "OnLogout", true)
	end
	
	if type(_G['Quit']) == 'function' then
		self:Hook('Quit', "OnQuit", true)
	end
	
	self:RebuildOptions()
	
	Baggins.OnMenuRequest.args.anywherebags = options
end

function BagginsAnywhereBags:OnDisable()
	if not Baggins then return end
	Baggins.OnMenuRequest.args.anywherebags = nil
end

------------------------------
--      Core Functions      --
------------------------------

local function BagDisabled(player, bag)
	return player == playerName and bag ~= L["Mail"] and not (tonumber(bag) and (bag == -1 or bag > 4 and bag < 12))
end

local function GetBagginsIDs(bag, slot)
	if bag == -2 then return L["Keyring"] end
	
	local key = bag..":"..slot
	
	local frames = Baggins.bagframes
	for bagid, bag in pairs(Baggins.db.profile.bags) do
		for sectionid, section in pairs(bag.sections) do
			local s = frames[bagid].sections[sectionid]
			if s and s.slots[key] then
				return bag.name, section.name
			end
		end
	end
	
	return L["Unknown id"]
end

--~ returns number of empty slots
local function ProcessBag(bag)
	r[playerName][bag] = {}
	local emptySlots = 0
	
	for i = 1, GetContainerNumSlots(bag) do
		local texture, count = GetContainerItemInfo(bag, i);
		if texture then
			local link = GetContainerItemLink(bag, i)
			local _,_,quality = GetItemInfo(link)
			r[playerName][bag][i] = {
				link,
				count,
				texture,
				quality,
				GetBagginsIDs(bag, i)
			}
		else
			emptySlots = emptySlots + 1
		end
	end
	
	return emptySlots
end

function BagginsAnywhereBags:ProcessEquiped()
	for i = 0, 19 do
		local texture = GetInventoryItemTexture('player',i)
		if texture then
			r[playerName].Equiped[i] = {
				GetInventoryItemLink('player', i),
				GetInventoryItemCount('player',i),
				texture,
				GetInventoryItemQuality('player',i),
				L["Equiped"],
			}
		else
			r[playerName].Equiped[i] = nil
		end
	end
end

function BagginsAnywhereBags:OnQuit(...)
	if not pcall(self.FullProcess,self) then
		self:Debug("Error in FullProcess.")
	end
	self.hooks['Quit'](...)
end

function BagginsAnywhereBags:OnLogout(...)
	if not pcall(self.FullProcess,self) then
		self:Debug("Error in FullProcess.")
	end
	self.hooks['Logout'](...)
end

function BagginsAnywhereBags:FullProcess()
	local new = Baggins.GetCategoryCache and Baggins:GetCategoryCache().New
	if Baggins.db.profile.highlightnew and new and next(new) then
		Baggins:SaveItemCounts()
		Baggins:ForceFullUpdate()
	end
	local emptySlots = 0
	ProcessBag(-2)	--don't want emptySlots of keyring
	for i = 0,4 do
		emptySlots = emptySlots + ProcessBag(i)
	end
	self:ProcessEquiped()
	r[playerName].Money = GetMoney()
	r[playerName].BagSlots = emptySlots
end

function BagginsAnywhereBags:Search(detailed, v)
	argCheck(self, detailed, 2, 'string', 'boolean')
	argCheck(self, v, 3, 'string', 'nil')
	if not v then
		v = detailed
		detailed = false
	end
	
	local find = strlower(strmatch(v,"|h(.+)|h") or v)
	DEFAULT_CHAT_FRAME:AddMessage(L["|cff7fff7fSearching all players for:|r "]..v)
	local links = {}
	for player, bags in pairs(r) do
		local found = nil
		if type(bags) == 'table' then
			for bag,slots in pairs(bags) do
				if type(slots) == 'table' then
					for slot,item in pairs(slots) do
						if type(item) == 'table' and not BagDisabled(player, bag) then
							if strfind(strlower(item[1]), find,1,true) then
								local itemName = strmatch(item[1], "|h(.+)|h")
								if type(found) ~= 'table' then
									found = {}
								end
								local pre = ": |r"
								if detailed then
									local i6 = item[6]
									if i6 then
										pre = "["..i6.."]"..pre
									end
									pre = "["..item[5].."]"..pre
									itemName = pre..itemName
								end
								if not links[itemName] then
									links[itemName] = pre..item[1]
								end
								found[itemName] = (found[itemName] or 0) + item[2]
							end
						end
					end
				end
			end
		end
		if found then
			for itemName,count in pairs(found) do
				DEFAULT_CHAT_FRAME:AddMessage("|cffffff00["..player.."]"..links[itemName].."x"..count)
			end
		end
	end
end

-- :GetItemCount(itemid)
-- Public API. Returns a total count of an itemid (not including own bags)
-- Currently used by LibPossessions (in Skillet)
function BagginsAnywhereBags:GetItemCount(itemid)	
	local count=0
	local find="|Hitem:"..itemid..":"
	for player, bags in pairs(r) do
		if type(bags) == 'table' then
			for bag,slots in pairs(bags) do
				if type(slots) == 'table' then
					for slot,item in pairs(slots) do
						if type(item) == 'table' and not BagDisabled(player, bag) then
							if strfind(item[1], find,1,true) then
								count = count + item[2]
							end
						end
					end
				end
			end
		end
	end
	
	return count
end

function BagginsAnywhereBags:MailExpire()
	local p, t
	for player, bags in pairs(r) do
		if type(bags) == 'table' and type(bags.Mail) == 'table' then
			for _,item in pairs(bags.Mail) do
				local time = item.daysLeft
				if not t or time < t then
					p = player
					t = time
				end
			end
		end
	end
	if t then
		DEFAULT_CHAT_FRAME:AddMessage("|cffffff00["..p..(date("] |cff26f420%d days", t - time()) or "] |cff26f420EXP|r"))
	end
end

function BagginsAnywhereBags:PurgePlayer(v)
	options.args.purge.args[v] = nil
	options.args[v] = nil
	if v == playerName then
		r[v] = {
			[-2] = {}, [-1] = {}, [0] = {};
			-- 1 to 11:
			{},{},{},{},{},{},{},{},{},{},{};
			Equiped = {},
			Money = 0,
			Mail = {},
		}
		self:RebuildOptions()
	else
		r[v] = nil
	end
end

function BagginsAnywhereBags:SetCompressed(v)
	p.compressed = v 
	self:RebuildOptions()
end

-------------------------------
--      Event Functions      --
-------------------------------

function BagginsAnywhereBags:BANKFRAME_CLOSED()
	self:UnregisterEvent('BANKFRAME_CLOSED')

	local emptySlots = ProcessBag(-1)
	for i = 5,11 do
		emptySlots = emptySlots + ProcessBag(i)
	end
	r[playerName].BankSlots = emptySlots
	self:RebuildBankOptions(emptySlots)
end

function BagginsAnywhereBags:MAIL_CLOSED()
	self:UnregisterEvent('MAIL_CLOSED')
	
	r[playerName][L["Mail"]] = {}

	for i = 1,GetInboxNumItems() do
		if GetInboxItemLink(i) then
			local _,_,_,_,_,_,daysLeft = GetInboxHeaderInfo(i);
			for j=1,ATTACHMENTS_MAX_RECEIVE do
				local link = GetInboxItemLink(i,j)
				if link then
					local _,texture,count = GetInboxItem(i,j)
					r[playerName][L["Mail"]][i*ATTACHMENTS_MAX_RECEIVE+j] = {
						link,
						count,
						texture,
						daysLeft,
						L["Mail"];
						daysLeft = time()+floor(daysLeft)*86400, -- seconds in a day
					}
				end
			end
		end
	end
	if next(r[playerName][L["Mail"]]) then
		self:RebuildMailOptions()
	end
end

---------------------------------
--      Display Functions      --
---------------------------------

function FormatMoney(money)
	local str = ""
--~ 	money = gggsscc
	if money > 9999 then
		str = format("%d|cffffd700g |r", money/10000) -- ggg.sscc automatically truncated with %d
	end
	if money > 99 then
		str = str..format("%d|cffc7c7cfs |r", money%10000/100) -- ss.cc automatically truncated with %d
	end
	if money > 0 then
		str = str..format("%d|cffeda55fc |r", money%100) -- cc
	end

	return str
end

--~ anywherebags = {
--~ 	player = {
--~ 		bag = {
--~ 			section = {
--~ 				item1 = Icon Name x count,
--~ 				...
--~ 			},
--~ 			...
--~ 		},
--~ 		...,
--~ 		-spacer-
--~ 		Empty Bag Slots: #
--~ 		Empty Bank Slots: #
--~ 		-spacer-
--~ 		Money
--~ 	},
--~ 	...,
--~ 	-spacer-
--~ 	Search
--~ 	SearchD
--~ 	Purge
--~ 	-spacer
--~ 	Compressed
--~ }
function BagginsAnywhereBags:AddBag(player,bag,slots)
	for slot,item in pairs(slots) do
		local itemLink, count, icon, quality, bagid, sectionid = item[1], item[2], item[3], item[4], item[5], item[6]
		if type(options.args[player].args[bagid]) ~= 'table' then
			options.args[player].args[bagid] = {
				name = bagid, type='group',
				desc = bagid,
				args = {},
				disabled = BagDisabled(player, bag) and not self:IsDebugging(),
				isBank = tonumber(bag) and (bag == -1 or bag > 4 and bag < 12),
			}
		end
		local opBag = options.args[player].args[bagid].args
		if sectionid then
			if type(opBag[sectionid]) ~= 'table' then
				opBag[sectionid] = {
					name = sectionid, type='group',
					desc = sectionid,
					args = {},
				}
			end
			opBag = opBag[sectionid].args
		end
		local itemName = p.compressed and strmatch(itemLink, "|h(.+)|h") or itemLink
		if type(opBag[itemName]) == 'table' and p.compressed then
			opBag[itemName].count = opBag[itemName].count + count
		else
			opBag[itemName] = {
				type='execute',
				func = function() SetItemRef(itemLink, itemLink, "LeftButton") end, -- thanks Elkano!
				icon = icon,
				order = 106-quality,
				count = count,
			}
		end
		opBag[itemName].name = itemLink..(opBag[itemName].count > 1 and "x"..opBag[itemName].count or "")
		if bag == "Mail" then
			local daysLeft = opBag[itemName].daysLeft or 2592000 -- seconds in 30 days
			local itemDaysLeft = item.daysLeft and item.daysLeft - time() or 2592000 -- seconds in 30 days
			
			daysLeft = daysLeft < itemDaysLeft and daysLeft or itemDaysLeft
			
			opBag[itemName].daysLeft = daysLeft
			
			if daysLeft < 0 then
				opBag[itemName].name = "|cff26f420EXP|r : "..opBag[itemName].name
			else
				opBag[itemName].name = date("|cff26f420%d|r : ", daysLeft)..opBag[itemName].name
			end
		end
		opBag[itemName].desc = opBag[itemName].name
	end
end

local dummyFunc = function() end
function BagginsAnywhereBags:RebuildOptions()
	for player,bags in pairs(r) do
		if type(bags) == 'table' then --TODO: testing code, remove check later (Checking because of DBVersion)
			--purge
			options.args.purge.args[player] = {
				name = player, type='execute',
				desc = player,
			}
			--
			local money = r[player].Money
			local bagSlots = r[player].BagSlots
			local bankSlots = r[player].BankSlots
			options.args[player] = {
				name = player, type='group',
				desc = player,
				pass = true,
				func = dummyFunc,
				args = (player ~= playerName or nil) and {
					BagSlots = bagSlots and {
						name = L["Empty Bag Slots: "]..bagSlots, type='execute',
						desc = L["Empty Bag Slots: "]..bagSlots,
						order = -30,
					},
					BankSlots = bankSlots and {
						name = L["Empty Bank Slots: "]..bankSlots, type='execute',
						desc = L["Empty Bank Slots: "]..bankSlots,
						order = -20,
					},
--~ 				spacer if money and (bankslots or bagslots) are shown
					spacer = money and (bankSlots or bagSlots) and {
						type='header',
						order = -15,
					},
					Money = money and {
						name = FormatMoney(money), type='execute',
						desc = FormatMoney(money),
						icon = GetCoinIcon(money),
						order = -10,
					},
				} or {
					BankSlots = bankSlots and {
						name = L["Empty Bank Slots: "]..bankSlots, type='execute',
						desc = L["Empty Bank Slots: "]..bankSlots,
						order = -20,
					},
				},
			}
			for bag,slots in pairs(bags) do
				if type(slots) == 'table' then
					self:AddBag(player, bag, slots)
				end
			end
		end --TODO: testing code, remove later
	end
end

function BagginsAnywhereBags:RebuildBankOptions(emptySlots)
	for bagid,bag in pairs(options.args[playerName].args) do
		if bag.isBank then
			options.args[playerName].args[bagid] = nil
		end
	end
	for bag,slots in pairs(r[playerName]) do
		if tonumber(bag) and (bag == -1 or bag > 4 and bag < 12) then
			self:AddBag(playerName, bag, slots)
		end
	end
	options.args[playerName].args.BankSlots = {
		name = L["Empty Bank Slots: "]..emptySlots, type='execute',
		desc = L["Empty Bank Slots: "]..emptySlots,
		func = dummyFunc,
		order = -20,
	}
end

function BagginsAnywhereBags:RebuildMailOptions()
	options.args[playerName].args[L["Mail"]] = nil
	self:AddBag(playerName, "Mail", r[playerName][L["Mail"]])
end
