-- version
local MAJOR_VERSION = "2.0"
local MINOR_VERSION = tonumber(("$Rev: 3 $"):match("(%d+)"))

-- namespace
bp = Rock:NewAddon("BagPress", "LibRockEvent-1.0", "LibRockComm-1.0", "LibRockConfig-1.0", "LibRockDB-1.0", "LibFuBarPlugin-3.0")
local bp = bp

bp.title = "BagPress"
bp.version = MAJOR_VERSION.."."..MINOR_VERSION
bp.date = string.gsub("$Date: 2008-08-26 08:59:49 +0000 (Tue, 26 Aug 2008) $", "^.-(%d%d%d%d%-%d%d%-%d%d).-$", "%1")

-- locals
local tinsert = tinsert

-- constants, DO NOT LOCALIZE
bp.constants = {
	FILTER = "filter",
	FORCE = "force",
	filterOPPOSITE = "force",
	forceOPPOSITE = "filter",
	COMMPREFIX = "BPr",
	COMPATVER = "2.0.1",
	INITIATE = "INI",
	ABORT = "ABO",
}

-- librockdb
bp:SetDatabase("BagPressDB", "BagPressDBPC")
bp:SetDefaultProfile("char")

bp:SetDatabaseDefaults("char", { filter = { }, force = { }, --[[defaultToggle = false, default = "" ]]})

-- libfubarplugin
bp:SetFuBarOption("iconPath", "interface\\addons\\bagpress\\media\\icon")
bp:SetFuBarOption("hasNoColor", true)
bp:SetFuBarOption("defaultPosition", "MINIMAP")
bp:SetFuBarOption("defaultMinimapPosition", "138")
bp:SetFuBarOption("independentProfile", true)

function bp:OnUpdateFuBarTooltip()
	GameTooltip:AddLine(GREEN_FONT_COLOR_CODE.."Hint: Click to BagPress with your current target"..FONT_COLOR_CODE_CLOSE)
end

function bp:OnFuBarClick()
	bp:Player1Comm(UnitName("target"))
end

-- librockconfig
bp:SetConfigTable({
	type = "group",
	name = "BagPress",
	desc = "Inventory efficiency for leveling duos",
	icon = function() return bp:GetFuBarIcon() end,
	args = {
		keybinding = {
			type = "keybinding",
			name = "Keybinding",
			desc = "Set keybinding for BagPress",
			order = 100,
			get = "GetKeybinding",
			set = "SetKeybinding",
		},
		filter = {
			type = "multichoice",
			name = "Filter",
			desc = "Items you don't want to trade",
			order = 200,
			get = "GetFilter",
			set = "SetFilter",
			choices = "GetChoices",
			choiceOrder = "GetChoiceOrder",
			passValue = bp.constants.FILTER,
		},
		force = {
			type = "multichoice",
			name = "Force",
			desc = "Items you always want to trade (ignores stack size)",
			order = 300,
			get = "GetFilter",
			set = "SetFilter",
			choices = "GetChoices",
			choiceOrder = "GetChoiceOrder",
			passValue = bp.constants.FORCE,
		},
		--[[ NYI
		defaultToggle = {
			type = "boolean",
			name = "Use a Default Trade Target",
			desc = "Allows you to specify a character to BagPress with by default. You can bypass the default by shift-clicking the BagPress button.",
			order = 400,
			get = function() return bp.db.char.defaultToggle end,
			set = function(value) bp.db.char.defaultToggle = value end,
		},
		default = {
			type = "string",
			name = "Default BagPress Target",
			desc = "Name of the character you wish to BagPress with by default.",
			order = 500,
			disabled = "DefaultDisabled",
			get = "GetDefaultTarget",
			set = "SetDefaultTarget",
		},
		defaultTarget = {
			type = "execute",
			name = "Set target as default",
			desc = "Sets your current target as the default BagPress target"
			order = 600,
			disabled = "DefaultDisabled",
			func = function()
				if (UnitExists("target")) and
				(UnitIsPlayer("target")) and
				(UnitFactionGroup("target") == UnitFactionGroup("player")) and
				(not UnitIsUnit("target", "player") then
					bp.db.char.default = UnitName("target")
				end
			end,
		--]]
	},
})

bp:SetConfigSlashCommand("/bagpress", "/bp")

function bp:DefaultDisabled()
	return (not bp.db.char.defaultToggle)
end

BINDING_HEADER_BAGPRESS = "BagPress"
BINDING_NAME_BAGPRESS = "Initiate BagPress"

function bp:GetKeybinding()
	return GetBindingKey("BAGPRESS")
end

function bp:SetKeybinding(value)
	if (type(co) == "thread") and (coroutine.status(co) == "suspended") then
		bp:Print("You can't do that yet, please wait until you have left combat")
		return
	end
	-- since we can't set keybindings in combat...
	local co
	bp:AddEventListener("PLAYER_LEAVE_COMBAT", function()
		if (type(co) == "thread") and (coroutine.status(co) == "suspended") then
			local errorfree, error = coroutine.resume(co)
			if not errorfree then
				geterrorhandler()(error)
			end
		end
	end)

	co = coroutine.create(function()
		while true do
			if UnitAffectingCombat("player") then
				bp:Print("Keybinding will be set once you leave combat")
				coroutine.yield()
			else
				SetBinding(value, "BAGPRESS")
				SaveBindings(GetCurrentBindingSet())
				bp:Print("Keybinding set to "..(_G[GetBindingText("KEY_"..value)] or value))
				bp:RemoveEventListener("PLAYER_LEAVE_COMBAT")
				return
			end
		end
	end)

	local errorfree, error = coroutine.resume(co)
	if not errorfree then
		geterrorhandler()(error)
	end
end

function bp:GetFilter(type, key)
	return bp.db.char[type][key] or false -- not sure if it accepts nil and too lazy to find out
end

function bp:SetFilter(type, key, value)
	bp.db.char[type][key] = value or nil

	if (value == true) then -- unset the opposite (can't filter and force the same item)
		local opposite = bp.constants[type.."OPPOSITE"]
		bp.db.char[opposite][key] = nil
	end
end

do -- choice list functions
	local choices

	function bp:GetChoices(type)
		local choiceItemList = bp:GenerateTradableItems(1) -- returns simple indexed table of itemIDs

		choices = { }

		for _, itemID in ipairs(choiceItemList) do
			name, _, rarity = GetItemInfo(itemID)
			choices[itemID] = (name and ITEM_QUALITY_COLORS[rarity].hex..name.."|r") or itemID -- will show itemID if item not cached
		end

		for _, itemID in ipairs(bp.db.char[type]) do
			name, _, rarity = GetItemInfo(itemID)
			choices[itemID] = (name and ITEM_QUALITY_COLORS[rarity].hex..name.."|r") or itemID -- will show itemID if item not cached
		end

		return choices
	end

	function bp:GetChoiceOrder(type)
		local choiceOrder = { }

		for itemID in pairs(choices) do -- dictionary of itemIDs/names retained from GetChoices
			local name = GetItemInfo(itemID)
			for index, itemID2 in ipairs(choiceOrder) do -- in order to sort, we need to iterate through
				local name2 = GetItemInfo(itemID2)       --  the table we're going to return (choiceOrder)
				name2 = name2 or tostring(itemID2)       --  for every value in the table we want to sort (choiceItemList)

				if (name < name2) then
					tinsert(choiceOrder, index, itemID)
					break
				end

				if (index == #choiceOrder) then -- reached end of table
					tinsert(choiceOrder, itemID)
					break
				end
			end

			if (#choiceOrder == 0) then -- first entry in table
				tinsert(choiceOrder, itemID)
			end
		end

		return choiceOrder
	end
end

--[[ NYI
function bp:GetDefaultTarget()
	return self.db.char.default
end

function bp:SetDefaultTarget(value)
	self.db.char.default = strupper(strsub(value, 1, 1))..strlower(strsub(value, 2)) -- proper capitalization because i'm anal
end
--]]

-- librockcomm
bp:SetCommPrefix(bp.constants.COMMPREFIX)
bp:SetDefaultCommPriority("BULK") -- since we're transmitting tables that have the potential to be quite large
-- this setting really shouldn't hinder the speed under normal circumstances

-- init
function bp:OnEnable()
	bp:AddCommListener(bp.constants.COMMPREFIX, "WHISPER")
end

do -- the bulk of it
	local commcoro
	local tradecoro
	local commMessage
	local tradeTarget
	local tradeItems

	-- push coroutine errors to errorhandler
	local function coresume(coro, ...)
		local errorfree, error = coroutine.resume(coro, ...)
		if not errorfree then
			geterrorhandler()(error)
		end
	end

	-- addon version receptor
	Rock("LibRockComm-1.0"):AddAddonVersionReceptor(function(player, addon, version)
		if (addon ~= "BagPress") then
			return
		end

		if (type(version) == "string") and (version >= bp.constants.COMPATVER) then
			bptimer:Hide()
			coresume(commcoro)
			return

		end
		-- "Joeblow has an incompatible version of BagPress (1.0.0 < 2.0.1)"
		bptimer:Hide()
		bp:Abort(("%s has an incompatible version of %s (%s < %s)"):format(player, addon, tostring(version), bp.constants.COMPATVER))

	end)

	-- comm listener
	function bp:OnCommReceive(prefix, dist, sender, message)
		if (message == bp.constants.ABORT) then -- silent abort
			if (sender == tradeTarget) then
				bp:Debug("Silent abort requested")
				bp:Abort()
			end

		elseif (message == bp.constants.INITIATE) then
			if (not commcoro) or (coroutine.status(commcoro) == "dead") then
				bp:Debug("Initiation requested, allowing")
				bp:Player2Comm(sender)
			else
				bp:SendCommMessage("WHISPER", sender, bp.constants.BUSY)
			end

		elseif (message == bp.constants.BUSY) then
			if (sender == tradeTarget) then
				bp:Abort("Failed: "..sender.." is busy.")
			end


		elseif (type(commcoro) == "thread") and (coroutine.status(commcoro) == "suspended") then
			if (sender == tradeTarget) then
				bp:Debug("Comm Message received!")
				commMessage = message
				coresume(commcoro)
			else
				bp:Debug(("Found coroutine but conditional failed: %s ~= %s"):format(sender, tradeTarget))
			end
		else
			bp:Debug(("Received CommMessage but failed conditional. prefix == %s, dist == %s, sender == %s, message == %s"):format(prefix, dist, sender, tostring(message)))
			bp:Debug(("type(commcoro) == %s, coroutine.status(commcoro) == %s"):format(type(commcoro), (type(commcoro) == "thread" and coroutine.status(commcoro)) or "nil"))

		end

	end

	-- player 1 comm handlers
	function bp:Player1Comm(target)
		if (type(commcoro) == "thread") and (coroutine.status(commcoro) ~= "dead") then
			bp:Print("You can't do that yet")
			return
		end

		tradeTarget = target -- tradeTarget should only be used for comm messages

		if target == UnitName("target") then -- can't use the player name if they're not in your group/raid
			target = "target"
		end

		local invalid =
			((not UnitExists(target)) and "Target does not exist") or -- this should only happen when using a default (NYI)
			((not UnitIsPlayer(target)) and "Target is not a player") or
			((UnitFactionGroup(target) ~= UnitFactionGroup("player")) and "You can only trade with members of your faction") or
			((UnitIsUnit(target, "player")) and "You can't trade with yourself")

		if invalid then
			bp:Print(("Invalid target (%s)"):format(invalid))
			return
		end

		commcoro = coroutine.create( function(target)

			local start = GetTime()
			local current = GetTime()

			-- the addon version receptor doesn't time out on its own if target doesn't have LibRockComm so we make our own timer
			if (not bptimer) then
				CreateFrame("FRAME", "bptimer", UIParent)
			end

			bptimer:SetScript("OnUpdate", function(self, elapsed)
				current = current + elapsed
				if ((current - start) >= 2) then
					self:Hide()
					bp:Abort("Version check timed out.")
					return
				end
			end)

			bp:Debug(("Beginning BagPress with %s (%s), sending version request"):format(tradeTarget, target))

			Rock("LibRockComm-1.0"):QueryAddonVersion("BagPress", "WHISPER", tradeTarget)
			bptimer:Show()

			-- wait for response
			coroutine.yield()
			-- incompatible versions handled above

			bp:Debug("Version request succeeded, sending intiation request")

			bp:SendCommMessage("WHISPER", tradeTarget, bp.constants.INITIATE)
			-- wait for response
			coroutine.yield()

			bp:Debug("Got player 2's items")

			-- commMessage = list of player 2's tradable items

			local player2items = commMessage
			local myItems = bp:GenerateTradableItems()

			tradeItems = { } -- these are the items i'm going to put into the trade window
			local player2trade = { } -- these are the items player 2 is going to put into the trade window
			-- both have the same format, a nested table of bagslots


			local emptyDiff = myItems.empty - player2items.empty
			-- emptyDiff is an attempt at "evening out" the inventories
			-- if there's a lot of items to trade or a lot of filters, it doesn't have much of an effect
			-- emptyDiff > 0 then item will be traded by player 2
			-- emptyDiff <= 0 then item will be traded by player 1

			-- extra: p1full, p2full
			-- boolean, indicates when the player can't trade any more items
			-- ex: p1full will be true if PLAYER 2's inventory is full, or PLAYER 1's tradeItems is full
			-- it can be confusing, because "full" refers to the trade window, NOT the inventory

			-- step 1: add all force items
			for _, slot in ipairs(myItems.force) do -- player 1
				local p1full = (#tradeItems == MAX_TRADABLE_ITEMS) or (#tradeItems == player2items.empty)
				if p1full then
					break
				end

				tinsert(tradeItems, slot)
			end

			for _, slot in ipairs(player2items.force) do -- player 2
				local p2full = (#player2trade == MAX_TRADABLE_ITEMS) or (#player2trade == myItems.empty)

				if p2full then
					break
				end

				tinsert(player2trade, slot)
			end

			-- step 2: iterate through item table to find common items, we only need to use one of the tables
			-- i chose player 1 to make the code somewhat easier to parse
			for itemID, info in pairs(myItems.items) do

				if (player2items.items[itemID]) then -- common item
					local info2 = player2items.items[itemID] -- make parsing easier
					if ((info.count + info2.count) <= info.max) then -- we can fit these items into a single stack
						-- now to add them to the tradeItems tables based on filters and empty bag slots... fun stuff!
						local p1full = (#tradeItems == MAX_TRADABLE_ITEMS)
						local p2full = (#player2trade == MAX_TRADABLE_ITEMS)
						-- note that unlike force items, here we don't prevent an item from being traded if inventory is full
						-- that's because we're only trading items that will fit into stacks in the other's person inventory
						-- (this wasn't true in earlier versions of bagpress because it just hit me as i'm rewriting it :P)

						if (p1full and p2full) then
							break -- so we don't keep iterating when it's impossible for more items to be added
						end

						if (emptyDiff > 0) and (not info2.filter) and (not p2full) then -- player 2 will trade this item
							tinsert(player2trade, info2.slot)
							emptyDiff = emptyDiff - 1

						elseif (not info.filter) and (not p1full) then -- player 1 will trade this item
							tinsert(tradeItems, info.slot)
							emptyDiff = emptyDiff + 1

						elseif (not info2.filter) and (not p2full) then -- player 2 will trade this item
							-- in order to trade as many items as possible, we ignore emptyDiff
							-- if player 1 can't trade any more items or has this item filtered
							tinsert(player2trade, info2.slot)
							emptyDiff = emptyDiff - 1

						end
					end
				end
			end

			if (#player2trade == 0) and (#tradeItems == 0) then -- failed to find any trade items
				bp:Abort("No tradable items", tradeTarget)
				return
			end

			bp:SendCommMessage("WHISPER", tradeTarget, player2trade)
			-- wait for response
			coroutine.yield()

			bp:Debug("Player 2 is ready to trade!")

			-- we don't depend on player 2 to start the trade, because then i'd have to add an extra conditional to OnCommReceive

			CancelTrade()
			bp:RegisterTradeEvents()

			if (UnitExists(target)) then
				InitiateTrade(target)
			else
				bp:Abort("Can't find "..tradeTarget..", please try again.", tradeTarget)
			end

		end) -- that's it for the coro, the rest is handled by the trade events!

		coresume(commcoro, target)
	end

	-- player 2 coroutine
	function bp:Player2Comm(sender)
		bp:Debug("Player2Comm called")
		commcoro = coroutine.create(function(sender)
			tradeTarget = sender

			local t = bp:GenerateTradableItems()
			bp:SendCommMessage("WHISPER", sender, t)

			-- wait for response
			bp:Debug(("Sent item table to %s"):format(sender))
			coroutine.yield()
			-- returns list of items i'm going to trade

			tradeItems = commMessage
			CancelTrade()
			bp:RegisterTradeEvents()

			bp:SendCommMessage("WHISPER", sender, 1) -- indicate that we're ready for trade
			bp:Debug(("Sent message to %s to indicate we're ready"):format(sender))

		end) -- end of coro, pretty short huh? rest is handled by trade events

		coresume(commcoro, sender)
	end

	-- abort function
	function bp:Abort(message, remote) -- remote is a player name (optional), will send a silent abort request if provided
		bp:RemoveAllEventListeners()
		commcoro = nil
		tradecoro = nil
		if message then
			bp:Print(message)
		end
		if remote then
			bp:SendCommMessage("WHISPER", remote, bp.constants.ABORT)
		end
	end

	function bp:RegisterTradeEvents()
		bp:AddEventListener("TRADE_SHOW", "EventTradeShow")
		bp:AddEventListener({ "UI_ERROR_MESSAGE", "UI_INFO_MESSAGE" }, "EventUIMessage")
	end

	function bp:EventTradeShow(self, namespace, event, ...)
		bp:AddEventListener("BAG_UPDATE", function()
			if (type(tradecoro) == "thread") and (coroutine.status(tradecoro) == "suspended") then
				coresume(tradecoro)
			end
		end)

		tradecoro = coroutine.create(function()
			for tradeslot, bagslot in ipairs(tradeItems) do
				ClearCursor()
				local bag, slot = bagslot[1], bagslot[2]
				while true do -- loop to prevent us from trying to pick up a locked item
					local _, _, locked = GetContainerItemInfo(bag, slot)
					if locked then
						coroutine.yield()
					else
						PickupContainerItem(bag, slot)
						ClickTradeButton(tradeslot) -- using table index as trade slot, aren't i clever?
						break
					end
				end
			end
			bp:RemoveEventListener("BAG_UPDATE")
		end)

		coresume(tradecoro)
	end

	function bp:EventUIMessage(self, namespace, event, message)
		-- clean up when trade ends or fails
		-- parsing system messages is necessary because TRADE_CLOSED isn't always triggered
		local t = "ERR_TRADE_"
		if
		message:find(ERR_PLAYER_BUSY_S) or -- TODO: change this to actually work
		message:find(_G[t.."BAG_FULL"]) or
		message:find(_G[t.."BLOCKED_S"]) or -- TODO: change this to actually work
		message:find(_G[t.."CANCELLED"]) or
		message:find(_G[t.."COMPLETE"]) or
		message:find(_G[t.."MAX_COUNT_EXCEEDED"]) or
		message:find(_G[t.."TARGET_BAG_FULL"]) or
		message:find(_G[t.."TARGET_DEAD"]) or
		message:find(_G[t.."TARGET_MAX_COUNT_EXCEEDED"]) or
		message:find(_G[t.."TOO_FAR"]) or
		message:find(_G[t.."WRONG_REALM"]) then -- hey, it could happen...
			bp:Abort(nil, tradeTarget) -- use a silent abort to clean up
		end

	end

	function bp:OnDisable()
		bp:Abort("BagPress disabled", tradeTarget)
	end

end

--[[

item table generation

if indexedOnly, then a simple table of itemIDs is returned:
{34062, 22832}

otherwise, more detailed information is provided in the form of a dictionary:
{
	empty = 8,
	force = {
		{1, 12}, {2, 2}, {0, 5}
	},
	items = {
		34062 = {
			filter = true,
			max = 5,
			count = 3,
			slot = {0, 1},
		},
		22832 = {
			max = 20,
			count = 11,
			slot = {3, 17},
		},
	},
}

fun to look at, huh?

--]]

function bp:GenerateTradableItems(indexedOnly)
	local t = { }
	local db = bp.db.char

	if not indexedOnly then
		t.empty = 0
		t.items = { }
		t.force = { }
	end

	for bag = 0, 4 do -- iterate through bags

		-- find free slots
		if not indexedOnly then
			local free, family = GetContainerNumFreeSlots(bag)
			if (family == 0) then -- only count free slots in normal bags
				t.empty = t.empty + free
			end
		end

		for slot = 1, GetContainerNumSlots(bag) do -- iterate through bag slots
			local link = GetContainerItemLink(bag, slot)
			if link then
				local tradable = true

				-- scan tooltip
				if not bptooltip then
					CreateFrame("GameTooltip", "bptooltip", UIParent, "GameTooltipTemplate")
					bptooltip:SetWidth(1)
					bptooltip:SetHeight(1)
					bptooltip:Hide()
				end

				bptooltip:SetOwner(UIParent, "ANCHOR_NONE")
				bptooltip:SetBagItem(bag, slot)

				if bptooltip:NumLines() > 1 and bptooltipTextLeft2:GetText() then
					local bop = bptooltipTextLeft2:GetText()
					tradable = not (
						bop == ITEM_SOULBOUND or
						bop == ITEM_BIND_ON_PICKUP or
						bop == ITEM_BIND_QUEST
					)
				end

				bptooltip:Hide()

				local max = select(8, GetItemInfo(link))
				tradable = ((max > 1) and tradable) -- we're not interested in trading non-stackable items

				if tradable then
					local itemID = select(2, strsplit(":", link))
					if indexedOnly then -- indexedOnly denotes a request for a simple indexed table of item IDs
						tinsert(t, itemID)

					else -- now the fun stuff...
						if db.force[itemID] then -- force items are put into a special table, when we build the tradeItems table these get added first
							tinsert(t.force, { bag, slot })

						else
							local _, stack = GetContainerItemInfo(bag, slot)
							if not t.items[itemID] then
								t.items[itemID] = {
									filter = db.filter[itemID], -- we include the filter so players will still trade items the other has filtered
									max = max,
									count = stack,
									slot = { bag, slot },
								}
							end
							if stack < t.items[itemID].count then -- eventually i might modify this to support multiple stacks
								t.items[itemID].count = stack     -- but for the time being, l2use a restacking mod!
								t.items[itemID].slot = { bag, slot }
							end
						end
					end
				end
			end
		end
	end

	return t

end

function bp:Print(message)
	DEFAULT_CHAT_FRAME:AddMessage("|cffffff78BagPress:|r "..tostring(message))
end

function bp:Debug(message)
	--[===[@debug@
	bp:Print(message)
	--@end-debug@]===]
end

function bp:OnKeybinding()
	bp:Player1Comm(UnitName("target"))
end
