-- BGTracker: track battlefield wins/losses. Inspired by Titan BattleTracker.
--
-- BGTracker keeps track of your battlefield statistics on a per-character basis.
-- For each battlefield (Alterac Valley, Arathi Basin, Eye of the Storm, and
-- Warsong Gulch), it tracks the total number of games played, the number of wins
-- and losses, and the average honor earned per game.
--
-- Only completed games are tracked. That is, you have to be in the battlefield
-- at the end when your team has either won or lost. Going AFK, disconnecting,
-- leaving one battlefield for another, etc., are all ignored.
--
-- The only locale currently supported is English.

-- Things still to do:
--
-- Options GUI panel.
--
-- Long tooltip/short tooltip option.
--
-- Verbose messages option.

BGTracker = LibStub("AceAddon-3.0"):NewAddon("BGTracker", "AceConsole-3.0", "AceEvent-3.0", "AceHook-3.0")
local L = LibStub("AceLocale-3.0"):GetLocale("BGTracker")

BGTracker.title = "BGTracker"

local Defaults = {
	char = {
		AB = {
			Wins = 0, Losses = 0, Honor = 0,
		},
		AV = {
			Wins = 0, Losses = 0, Honor = 0,
		},
		ES = {
			Wins = 0, Losses = 0, Honor = 0,
		},
		WG = {
			Wins = 0, Losses = 0, Honor = 0,
		},
	},
}

local Alliance = L["Alliance"]
local Horde = L["Horde"]

local AlteracValley = L["Alterac Valley"]
local ArathiBasin = L["Arathi Basin"]
local EyeoftheStorm = L["Eye of the Storm"]
local WarsongGulch = L["Warsong Gulch"]

local BattlefieldPoints
local BattlefieldWinner
local Faction
local ScoreHooked

function BGTracker:OnInitialize()
  self.db = LibStub("AceDB-3.0"):New("BGTrackerDB", Defaults, "char")

  BattlefieldPoints = 0
	BattlefieldWinner = nil
  Faction = UnitFactionGroup("player")
  ScoreHooked = false

  -- Need a frame to hang our tooltip on.
  self.frame = CreateFrame("Frame")
end

function BGTracker:OnEnable()
  -- Create the tooltip.
	self.tooltip = CreateFrame("GameTooltip", "BGTrackerTooltip", self.frame, "GameTooltipTemplate")
	self.tooltip:SetOwner(self.frame, "ANCHOR_NONE")

  -- Hook the paper doll stuff.
	PLAYERSTAT_BG_TRACKER = L["Battlegrounds"]
	table.insert(PLAYERSTAT_DROPDOWN_OPTIONS, "PLAYERSTAT_BG_TRACKER")

	self:SecureHook("UpdatePaperdollStats")
	self:HookScript(PaperDollFrame, "OnShow", PaperDollFrame_OnShow)

  -- Events we want to see for tracking zone changes.
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA")
end

function BGTracker:OnDisable()
  self:UnhookScore()
  BattlefieldPoints = 0
  BattlefieldWinner = nil
end

function BGTracker:ZONE_CHANGED_NEW_AREA()
  local zone = GetRealZoneText()

  if ((zone == AlteracValley) or (zone == ArathiBasin) or (zone == EyeoftheStorm) or (zone == WarsongGulch)) then
  	-- Entering a battlefield. Set up for it.
  	BattlefieldPoints = GetHonorCurrency()
  	BattlefieldWinner = nil
  	self:HookScore()
  else
  	-- Not entering a battlefield. We're probably just crossing a normal zone
  	-- boundary. But, we could be leaving a battlefield before it successfully
  	-- completed, maybe because we went AFK or something like that. Either way,
  	-- make sure the battlefield variables are all reset.
		self:UnhookScore()
		BattlefieldPoints = 0
		BattlefieldWinner = nil
  end
end

function BGTracker:UPDATE_BATTLEFIELD_SCORE()
	BattlefieldWinner = GetBattlefieldWinner()
	if (BattlefieldWinner == nil) then
	  -- Just a score update. No winner.
	else
	  -- Got a winner. Process it.
	  self:UnhookScore()
	  self:ExitBattlefield(BattlefieldWinner)
	  BattlefieldPoints = 0
	  BattlefieldWinner = nil
	end
end

function BGTracker:DebugPrint(msg)
	self:Print("[Debug] " .. msg)
end

function BGTracker:ExitBattlefield(winner)
  local zone = GetRealZoneText()

  if (((winner == 0) and (Faction == Horde)) or ((winner == 1) and (Faction == Alliance))) then
  	-- We won.
    self:Print("[" .. zone .. "] " .. L["Game over, win tracked"])

    if (zone == AlteracValley) then
      self.db.char.AV.Wins = self.db.char.AV.Wins + 1
    elseif (zone == ArathiBasin) then
      self.db.char.AB.Wins = self.db.char.AB.Wins + 1
    elseif (zone == EyeoftheStorm) then
      self.db.char.ES.Wins = self.db.char.ES.Wins + 1
    elseif (zone == WarsongGulch) then
      self.db.char.WG.Wins = self.db.char.WG.Wins + 1
    end
  else
  	-- We lost.
    self:Print("[" .. zone .. "] " .. L["Game over, loss tracked"])

    if (zone == AlteracValley) then
      self.db.char.AV.Losses = self.db.char.AV.Losses + 1
    elseif (zone == ArathiBasin) then
      self.db.char.AB.Losses = self.db.char.AB.Losses + 1
    elseif (zone == EyeoftheStorm) then
      self.db.char.ES.Losses = self.db.char.ES.Losses + 1
    elseif (zone == WarsongGulch) then
      self.db.char.WG.Losses = self.db.char.WG.Losses + 1
    end
  end

  -- Total honor obtained in this battlefield.
  BattlefieldPoints = GetHonorCurrency() - BattlefieldPoints
  if (zone == AlteracValley) then
    self.db.char.AV.Honor = self.db.char.AV.Honor + BattlefieldPoints
  elseif (zone == ArathiBasin) then
    self.db.char.AB.Honor = self.db.char.AB.Honor + BattlefieldPoints
  elseif (zone == EyeoftheStorm) then
    self.db.char.ES.Honor = self.db.char.ES.Honor + BattlefieldPoints
  elseif (zone == WarsongGulch) then
    self.db.char.WG.Honor = self.db.char.WG.Honor + BattlefieldPoints
  end
end

function BGTracker:HookScore()
	if (ScoreHooked == false) then
		self:RegisterEvent("UPDATE_BATTLEFIELD_SCORE")
		ScoreHooked = true
	end
end

function BGTracker:PaperDollFrame_OnShow(prefix)
	getglobal(prefix):Show()
end

function BGTracker:UnhookScore()
	if (ScoreHooked == true) then
		self:UnregisterEvent("UPDATE_BATTLEFIELD_SCORE")
		ScoreHooked = false
	end
end

function BGTracker:UpdatePaperdollBG(prefix)
	local id, total, wins, losses, name

	for id = 1, 4, 1 do
	  if (id == 1) then
	    name = AlteracValley
	    wins = self.db.char.AV.Wins
	    losses = self.db.char.AV.Losses
	    total = wins + losses
	    honor = math.floor(self.db.char.AV.Honor / total)
	  elseif (id == 2) then
	    name = ArathiBasin
	    wins = self.db.char.AB.Wins
	    losses = self.db.char.AB.Losses
	    total = wins + losses
	    honor = math.floor(self.db.char.AB.Honor / total)
	  elseif (id == 3) then
	    name = EyeoftheStorm
	    wins = self.db.char.ES.Wins
	    losses = self.db.char.ES.Losses
	    total = wins + losses
	    honor = math.floor(self.db.char.ES.Honor / total)
	  else
	    name = WarsongGulch
	    wins = self.db.char.WG.Wins
	    losses = self.db.char.WG.Losses
	    total = wins + losses
	    honor = math.floor(self.db.char.WG.Honor / total)
	  end

	  local row = getglobal(prefix .. id)
	  local label = getglobal(prefix .. id .. "Label")
	  local stat = getglobal(prefix .. id .. "StatText")

	  label:SetText(name)
	  stat:SetText("")  -- Was total instead of empty string, but collided with label when number hit three digits.

	  row.tooltip = HIGHLIGHT_FONT_COLOR_CODE .. name .. FONT_COLOR_CODE_CLOSE
	  row:Show()

	  if (total == 0) then
	    row.tooltip2 = L["No games played"]
	  else
	    row.tooltip2 = string.format(L["%d Won, %d Lost, %d Average"], wins, losses, honor)
	  end
	end

  -- We only use the first four of the six possible lines, so hide the other two.
	getglobal(prefix .. 5):Hide()
	getglobal(prefix .. 6):Hide()
end

function BGTracker:UpdatePaperdollStats(prefix, index)
	if (index == "PLAYERSTAT_BG_TRACKER") then
		BGTracker:UpdatePaperdollBG(prefix)
	end
end
