-- General
local BASE_ATTACK = "^.+ claims the (.+)!  If left unchallenged, the (.+) will control it"; -- Happens when base was uncontrolled, Neutral to Horde or Alliance

-- Warsong Gulch
local WSG_GAME_START = "^The battle for (.+) begins in (%d+) (.+)%.";
local WSG_FLAG_CAPTURE ="^.+ captured the (.+) flag!";
local WSG_FLAG_PICK = "^The (.+) [Ff]lag was picked up by (.+)!$"; -- No idea why the F can be both lower and upper case
local WSG_FLAG_RESPAWN = 22;
local WSG_FLAG_DROP = "^The (.+) [Ff]lag was dropped by (.+)!"
local WSG_ZONE_NAME = "Warsong Gulch";

-- Alterac Valley
local AV_GAME_START = "^(%d+) (.+) until the battle for (.+) begins%.";
local AV_CAPTURE_TIME = (60 * 4);
local AV_CAPTURE_TIME_SNOWFALL = (60 * 5);
local AV_GY_ATTACK =  "^The (.+) is under attack!  If left unchecked, the (.+) will .+ it!"; -- Last .+ is either "destroy" or "control"
local AV_TOWER_ATTACK = "^(.+) is under attack!  If left unchecked, the (.+) will destroy it!"; -- Sometimes starts with "The" at the start, but then it will be cought by the AV_GY_ATTACK
local AV_DEFEND = "^The (.+) was taken by the (.+)!";
local AV_DEFEND2 = "^(.+) was taken by the (.+)!";

-- Arathi Basin
local AB_GAME_START ="^The Battle for (.+) will begin in (%d+) (.+)";
local AB_CAPTURE_TIME = 63;
local AB_BASE_ATTACK = "^.+ has assaulted the (.+)!";
local AB_BASE_DEFEND = "^.+ has defended the (.+)!";

-- Eye of the Storm
local EOTS_GAME_START = "^The battle begins in (%d+) (.+)!";

-- Icons
local BG_ICONS = {
	["Horde"] = "Interface\\Icons\\INV_BannerPVP_01",
	["Alliance"] = "Interface\\Icons\\INV_BannerPVP_02",
};

-- Vars
local b = AzCastBar_MakeBar();
local timers = {};
local frames = {};

BGTimers = {};

--------------------------------------------------------------------------------------------------------
--                                           Event Handling                                           --
--------------------------------------------------------------------------------------------------------
local function OnEvent(self,event)
	-- End if disabled
	if (b.cfg) and (not b.cfg.enabled) then
		return;
	-- Hide all bars upon zoning.
	elseif (event == "PLAYER_ENTERING_WORLD" or event == "ZONE_CHANGED_NEW_AREA") then
		if (event == "PLAYER_ENTERING_WORLD") then
			for index, table in ipairs(timers) do
				table.fadeTime = b.cfg.fadeTime;
			end
		end
		if (GetZoneText() == WSG_ZONE_NAME) then
			BGTimers:MakeFrames();
		elseif (not InCombatLockdown()) then
			for _, frame in pairs(frames) do
				frame:Hide();
			end
		-- Az: Debug
		else
			AzMsg("Could not hide WSG Flag Frames, was in combat on zone change!");
		end
	-- BG Status Update
	elseif (event == "UPDATE_BATTLEFIELD_STATUS") then
		local status, mapName, instanceID, levelRangeMin, levelRangeMax, teamSize, registeredMatch;
		for i = 1, MAX_BATTLEFIELD_QUEUES do
			status, mapName, instanceID, levelRangeMin, levelRangeMax, teamSize, registeredMatch = GetBattlefieldStatus(i);
			if (status == "confirm") and (not BGTimers:TimerExist(mapName.." "..instanceID)) then
				BGTimers:AddTimer(mapName.." "..instanceID,GetBattlefieldPortExpiration(i) / 1000);
			end
		end
	------------------------
	-- End if not in a BG --
	------------------------
	elseif (select(2,IsInInstance()) ~= "pvp") then
		return;
	-- Yells
	elseif (event == "CHAT_MSG_MONSTER_YELL") and (arg2 == "Herald") then
		-- attack (gy + tower)
		local place, faction = arg1:match(AV_GY_ATTACK);
		if (not place) then
			place, faction = arg1:match(AV_TOWER_ATTACK);
		end
		if (place) then
			BGTimers:AddTimer(place,BGTimers:GetDuration(place),faction);
			return;
		end
		-- defend
		place, faction = arg1:match(AV_DEFEND);
		if (not place) then
			place, faction = arg1:match(AV_DEFEND2);
		end
		if (place) then
			BGTimers:RemoveTimer(place);
			return;
		end
	-- Alliance / Horde
	elseif (event == "CHAT_MSG_BG_SYSTEM_HORDE" or event == "CHAT_MSG_BG_SYSTEM_ALLIANCE") then
		-- attack
		local name, faction = arg1:match(BASE_ATTACK);
		if (not name) then
			name = arg1:match(AB_BASE_ATTACK);
		end
		if (name) then
			faction = (event == "CHAT_MSG_BG_SYSTEM_HORDE" and "Horde" or "Alliance");
			BGTimers:AddTimer(name,BGTimers:GetDuration(name),faction);
			return;
		end
		-- defend
		name = arg1:match(AB_BASE_DEFEND);
		if (name) then
			BGTimers:RemoveTimer(name);
			return;
		end
		-- wsg: flag pick
		faction, name = arg1:match(WSG_FLAG_PICK);
		if (faction) then
			AzMsg(faction.." flag picked up by |1"..name.."|r.");
			BGTimers:SetupFrame(faction,name);
			return;
		end
		-- wsg: flag capture
		faction = arg1:match(WSG_FLAG_CAPTURE);
		if (faction) then
			BGTimers:AddTimer("Flag Respawn",WSG_FLAG_RESPAWN,faction);
			if (frames[faction]) then
				BGTimers:SetupFrame(faction,"");
			end
			return;
		end
		-- wsg: flag drop
		faction, name = arg1:match(WSG_FLAG_DROP);
		if (name) then
			if (frames[faction]) then
				BGTimers:SetupFrame(faction,"");
			end
		end
	-- Neutral
	elseif (event == "CHAT_MSG_BG_SYSTEM_NEUTRAL") then
		local bg, time, units = arg1:match(WSG_GAME_START);
		if (not time) then
			time, units, bg = arg1:match(AV_GAME_START);
			if (not time) then
				bg, time, units = arg1:match(AB_GAME_START);
				if (not time) then
					time, units = arg1:match(EOTS_GAME_START);
				end
			end
		end
		if (time) then
			BGTimers:AddTimer((bg or "The Game").." Begins",units:match("minutes?") and time * 60 or time);
			return;
		end
	-- Combat End
	elseif (event == "PLAYER_REGEN_ENABLED") then
		for faction, frame in pairs(frames) do
			if (frame.fixAfter) then
				BGTimers:SetupFrame(faction);
				frame.fixAfter = nil;
			end
		end
	end
end

--------------------------------------------------------------------------------------------------------
--                                   WSG Flag Carrier Secure Frames                                   --
--------------------------------------------------------------------------------------------------------

-- Create Button
local function CreateNameButton(index)
	local button = CreateFrame("Button",nil,UIParent,"SecureActionButtonTemplate");
	button.text = button:CreateFontString(nil,"ARTWORK","GameFontNormal");
	button.text:SetPoint("LEFT",4,0);
	button:SetWidth(85);
	button:SetHeight(20);
	button:SetPoint("LEFT","AlwaysUpFrame"..index.."DynamicIconButton","RIGHT",8,0);
	button:EnableMouse(1);
	button:SetAttribute("type","macro");

--	button:SetBackdrop(TipTacAnchor:GetBackdrop());
--	button:SetBackdropColor(0.1,0.1,0.2,1.0);
--	button:SetBackdropBorderColor(0.3,0.3,0.4,1.0);

	return button;
end

-- Make Both Frames
function BGTimers:MakeFrames()
	if (InCombatLockdown() or frames.Alliance or frames.Horde) then
 		return;
	end
	AzMsg("Making Frames");
	-- Req BG Score Info
	SetBattlefieldScoreFaction(nil);
	RequestBattlefieldScoreData();
	-- Create Frame if it doesn't exist
	frames.Alliance = CreateNameButton(2);
	frames.Horde = CreateNameButton(1);
end

-- Setup the Frame
function BGTimers:SetupFrame(faction,name)
	-- Create Frame if it doesn't exist
	if (not frames[faction]) then
		AzMsg("|2THIS SHOULD NEVER HAPPEN|r - Frames are not set at this point!!!!");
		if (InCombatLockdown()) then
			return;
		else
			BGTimers:MakeFrames();
		end
	end
	-- Req BG Score Info
	SetBattlefieldScoreFaction(nil);
	RequestBattlefieldScoreData();
	-- Set Text & Find BG Names
	if (name) then
		frames[faction].name = name;
		frames[faction].text:SetText(name);
		frames[faction].text:SetTextColor(0,1,0);
		if (faction ~= UnitFactionGroup("player")) then
			for i = 1, GetNumRaidMembers() do
				if (name == UnitName("raid"..i)) then
					local color = RAID_CLASS_COLORS[select(2,UnitClass("raid"..i))];
					frames[faction].text:SetTextColor(color.r,color.g,color.b);
				end
			end
		else
			for i = 1, GetNumBattlefieldScores() do
				if (name == GetBattlefieldScore(i):match("^(.+)%-")) then
					local color = RAID_CLASS_COLORS[select(10,GetBattlefieldScore(i))];
					frames[faction].text:SetTextColor(color.r,color.g,color.b);
					break;
				end
			end
		end
	end
	-- Set Macro
	if (InCombatLockdown()) then
		frames[faction].fixAfter = 1;
	else
		frames[faction]:SetAttribute("macrotext","/target "..frames[faction].name);
		frames[faction]:Show();
	end
end

--------------------------------------------------------------------------------------------------------
--                                            Bar OnUpdate                                            --
--------------------------------------------------------------------------------------------------------

local function OnUpdate(self,elapsed)
	local timer = self.timer;
	if (timer.fadeTime == -1) then
		self.timeProgress = (GetTime() - timer.startTime);
		self.timeLeft = max(0,timer.seconds - self.timeProgress);
		self.status:SetValue(self.timeProgress);
		self.right:SetText(AzCastBar_FormatTime(self.timeLeft));
		if (self.timeLeft == 0) then
			timer.fadeTime = b.cfg.fadeTime;
		end
	elseif (timer.fadeElapsed < timer.fadeTime) then
		timer.fadeElapsed = (timer.fadeElapsed + elapsed);
		self:SetAlpha(1 - timer.fadeElapsed / timer.fadeTime);
	else
		self:Hide();
		BGTimers:RemoveTimer(timer.label);
		BGTimers:UpdateTimers();
	end
end

--------------------------------------------------------------------------------------------------------
--                                                Code                                                --
--------------------------------------------------------------------------------------------------------

-- BGTimers:AddTimer("bg timer test #1",12)

-- Add Timer
function BGTimers:AddTimer(label,seconds,faction)
	for index, table in ipairs(timers) do
		if (table.label == label) then
			tremove(timers,index);
			break;
		end
	end
	tinsert(timers,{ label = label, seconds = seconds, faction = faction, startTime = GetTime(), fadeTime = -1, fadeElapsed = 0 });
	table.sort(timers,function(a,b) return (a.startTime + a.seconds) < (b.startTime + b.seconds) end);
	BGTimers:UpdateTimers();
end

-- Remove Timer with given label
function BGTimers:RemoveTimer(label)
	for index, table in ipairs(timers) do
		if (table.label == label) then
			tremove(timers,index);
			BGTimers:UpdateTimers();
			break;
		end
	end
end

-- Does Timer Exist?
function BGTimers:TimerExist(label)
	for index, table in ipairs(timers) do
		if (table.label == label) then
			return 1;
		end
	end
	return;
end

-- Update Timers
function BGTimers:UpdateTimers()
	-- Loop Timers and update bars
	local bar;
	for index, table in ipairs(timers) do
		bar = b.bars[index];
		if (not bar) then
			bar = AzCastBar_MakeBar();
			bar.id = index;
			bar:SetPoint("TOP",b.bars[index - 1],"BOTTOM",0,b.cfg.backdropIndent*2-1);
			bar:SetScript("OnUpdate",OnUpdate);
			tinsert(b.bars,bar);
			AzCastBar_ApplyBarSettingsSpecific(b,index);
		end

		bar.timer = table;

		bar.status:SetMinMaxValues(0,table.seconds);
		if (table.faction == "Horde") then
			bar.status:SetStatusBarColor(1.0,0.5,0.5,1.0);
		else
			bar.status:SetStatusBarColor(unpack(b.cfg.colNormal));
		end

		bar.left:SetText(BGTimers:TitleCase(table.label));
		bar.icon:SetTexture(table.faction and BG_ICONS[table.faction] or "Interface\\Icons\\Spell_Nature_UnrelentingStorm");

		if (table.fadeTime == -1) then
			bar:SetAlpha(b.cfg.alpha);
		end
		bar:Show();
	end
	-- Hide all other frames
	for i = #timers + 1, #b.bars do
		b.bars[i]:Hide();
	end
end

-- Get time duration
function BGTimers:GetDuration(base)
	if (base == "Snowfall graveyard") then
		return AV_CAPTURE_TIME_SNOWFALL;
	elseif (GetRealZoneText() == "Alterac Valley") then
		return AV_CAPTURE_TIME;
	else
		return AB_CAPTURE_TIME;
	end
end

-- Title Case
function BGTimers:TitleCase(string)
	return string:gsub("^(.)",strupper):gsub("( .)",strupper);
end


-- Config Change
local function OnConfigChanged()
	local bar;
	for i = 2, #b.bars do
		bar = b.bars[i];
		bar:ClearAllPoints();
		bar:SetPoint("TOP",b.bars[i - 1],"BOTTOM",0,b.cfg.backdropIndent*2-1);
	end
	--BGTimers:UpdateTimers();
end

--------------------------------------------------------------------------------------------------------
--                                              Finalize                                              --
--------------------------------------------------------------------------------------------------------

-- Init AzCastBar Entry
b.token = "BGTimers";
b.bars = { b };
b.id = 1;
b.OnConfigChanged = OnConfigChanged;
b.OnEditMode = BGTimers.UpdateTimers;
b:ClearAllPoints();
b:SetPoint("CENTER",0,100);
tinsert(AzCastBar_Frames,b);
-- Events
b:RegisterEvent("PLAYER_ENTERING_WORLD");
b:RegisterEvent("CHAT_MSG_MONSTER_YELL");
b:RegisterEvent("CHAT_MSG_BG_SYSTEM_HORDE");
b:RegisterEvent("CHAT_MSG_BG_SYSTEM_ALLIANCE");
b:RegisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL");
b:RegisterEvent("UPDATE_BATTLEFIELD_STATUS");
b:RegisterEvent("PLAYER_REGEN_ENABLED");

b:RegisterEvent("ZONE_CHANGED_NEW_AREA");

b:SetScript("OnUpdate",OnUpdate);
b:SetScript("OnEvent",OnEvent);