﻿Auditor = DongleStub("Dongle-1.1"):New("Auditor")

--[[ DECLARE LOCALS ]]--

-- Locale
local L = AUDITOR2_LOCALS

-- Auditor locals
local auditorCurrentMoney = nil
local auditorLastMoney = nil
local auditorMode = ""

-- Cash format locals
local gsub = string.gsub

local COPPER_ABBR
local SILVER_ABBR
local GOLD_ABBR
local GOLD, SILVER, COPPER = GOLD, SILVER, COPPER

if not COPPER and COPPER_AMOUNT then
	GOLD = GOLD_AMOUNT:gsub("%s*%%d%s*", "")
	SILVER = SILVER_AMOUNT:gsub("%s*%%d%s*", "")
	COPPER = COPPER_AMOUNT:gsub("%s*%%d%s*", "")
end

if (COPPER:byte(1) or 128) > 127 then
	-- non-western
	COPPER_ABBR = COPPER
	SILVER_ABBR = SILVER
	GOLD_ABBR = GOLD
else
	COPPER_ABBR = COPPER:sub(1, 1):lower()
	SILVER_ABBR = SILVER:sub(1, 1):lower()
	GOLD_ABBR = GOLD:sub(1, 1):lower()
end

local COLOR_WHITE = "ffffff"
local COLOR_GREEN = "00ff00"
local COLOR_RED = "ff0000"
local COLOR_COPPER = "eda55f"
local COLOR_SILVER = "c7c7cf"
local COLOR_GOLD = "ffd700"
local inf = 1/0

-- Cashnotify locals
local auditorCN_cash = 0
local auditorCN_oldCash = 0
local auditorCN_gain = 0
local auditorCN_spend = 0
local auditorCN_open = false
local auditorCN_mode = nil

-- Database defaults
local databasedefaults = {
	profile = {
		autoRepair = false,
		guildRepair = false,
	},
	realm = {
		timeSystemUpgraded = false,
		sellGreys = false,
		notifyVendor = true,
        notifyMailbox = true,
        notifyAH = true,
		notifyGuildbank = true,
		notifyTrainer = true,
		timeOffset = 0,
		useOffset = false
	},
}

--[[ STARTUP ]]--

function Auditor:Initialize()
	self.db = self:InitializeDB("Auditor2DB", databasedefaults)
	auditorPlayer = UnitName("player")
	auditorSessionTime = time()
	auditorFuToonToShow = auditorPlayer
	auditorHDay = { "6days", "5days", "4days", "3days", "2days", "1days", "0days" }
	auditorLogMode = { "loot", "merch", "quest", "trade", "mail", "ah", "train", "taxi", "repairs", "other", "reconciliation", "guildbank" }
	auditorTimeframe = { "session", "day", "week", "total" }
	-- Create DB for toon if it doesn't exist
	self:NewToonDB()
	-- Clear out last session's data	
	self:DataReset("session")
end

function Auditor:Enable()
	-- Event registration
	self:RegisterEvent("MERCHANT_SHOW")
	self:RegisterEvent("MERCHANT_CLOSED")
	self:RegisterEvent("BANKFRAME_OPENED")
	self:RegisterEvent("BANKFRAME_CLOSED")
	self:RegisterEvent("GUILDBANKFRAME_OPENED")
	self:RegisterEvent("GUILDBANKFRAME_CLOSED")
	self:RegisterEvent("QUEST_COMPLETE")
	self:RegisterEvent("LOOT_OPENED")
	self:RegisterEvent("TAXIMAP_OPENED")
	self:RegisterEvent("TRADE_SHOW")
	self:RegisterEvent("TRADE_CLOSE")
	self:RegisterEvent("MAIL_SHOW")
	self:RegisterEvent("MAIL_CLOSED")
	self:RegisterEvent("TRAINER_SHOW")
	self:RegisterEvent("TRAINER_CLOSED")
	self:RegisterEvent("AUCTION_HOUSE_SHOW")
	self:RegisterEvent("AUCTION_HOUSE_CLOSED")
	self:RegisterEvent("CHAT_MSG_MONEY")
	self:RegisterEvent("PLAYER_MONEY")
	self:RegisterEvent("CONFIRM_TALENT_WIPE")
	-- Hooks
	hooksecurefunc("RepairAllItems", function() self:RepairAllItems() end)
	hooksecurefunc("CursorHasItem", function() self:CursorHasItem() end)
	hooksecurefunc("TakeInboxMoney", function(...) self:TakeInboxMoney(...) end)
	-- Data housekeeping functions
	self:UpdateTimeFrame()
	self:ReconcileData()
end

--[[ DATA MANIPULATION FUNCTIONS]]--

function Auditor:DataReset(timeFrame, toon)
	local toonToReset = ""
	if toon ~= nil then
		toonToReset = toon
	else
		toonToReset = auditorPlayer
	end
	
	for _,logmode in pairs(auditorLogMode) do
			self.db.realm[toonToReset].data[logmode][timeFrame] = {incomings = 0, outgoings = 0}
	end
	-- Reset time for gold per hour
	if timeFrame == "session" then
		auditorSessionTime = time()
	end
end

function Auditor:ReconcileData()
	-- Set current cash
	auditorCurrentMoney = GetMoney()
	auditorLastMoney = auditorCurrentMoney
	auditorCN_cash = auditorCurrentMoney
	self.db.realm.chars[auditorPlayer] = auditorCurrentMoney
	self.db.realm[auditorPlayer].data.totals.incomings = 0
	self.db.realm[auditorPlayer].data.totals.outgoings = 0
	
	-- Add up total incomings and outgoings according to Auditor
	for _, logmode in pairs(auditorLogMode) do
		self.db.realm[auditorPlayer].data.totals.incomings = self.db.realm[auditorPlayer].data.totals.incomings + self.db.realm[auditorPlayer].data[logmode]["total"].incomings
		self.db.realm[auditorPlayer].data.totals.outgoings = self.db.realm[auditorPlayer].data.totals.outgoings + self.db.realm[auditorPlayer].data[logmode]["total"].outgoings
	end
	self.db.realm[auditorPlayer].data.totals.actual = self.db.realm[auditorPlayer].data.totals.incomings - self.db.realm[auditorPlayer].data.totals.outgoings
	
	-- Does Auditor's data sync with reality? If not, make it so
	if self.db.realm[auditorPlayer].data.totals.actual ~= auditorCurrentMoney then
		local diff = auditorCurrentMoney - self.db.realm[auditorPlayer].data.totals.actual
		if diff > 0 then
			for _,logmode in pairs(auditorTimeframe) do
				self.db.realm[auditorPlayer].data["reconciliation"][logmode].incomings = self.db.realm[auditorPlayer].data["reconciliation"][logmode].incomings + diff
			end
			self.db.realm[auditorPlayer].historical[auditorHDay[7]]["reconciliation"].incomings = self.db.realm[auditorPlayer].data["reconciliation"]["day"].incomings
			self.db.realm[auditorPlayer].data.totals.incomings = self.db.realm[auditorPlayer].data.totals.incomings + diff
		elseif diff < 0 then
			diff = diff * -1;
			for _,logmode in pairs(auditorTimeframe) do
				self.db.realm[auditorPlayer].data["reconciliation"][logmode].outgoings = self.db.realm[auditorPlayer].data["reconciliation"][logmode].outgoings + diff
			end
			self.db.realm[auditorPlayer].historical[auditorHDay[7]]["reconciliation"].outgoings = self.db.realm[auditorPlayer].data["reconciliation"]["day"].outgoings
			self.db.realm[auditorPlayer].data.totals.outgoings = self.db.realm[auditorPlayer].data.totals.outgoings + diff
		end
		self.db.realm[auditorPlayer].data.totals.actual = auditorCurrentMoney
		self:Print(L.Reconciled..auditorPlayer)
	end	
end

--[[ EVENT FUNCTIONS ]]--

function Auditor:MERCHANT_SHOW()
	if CanMerchantRepair() and self.db.profile.autoRepair == true then
		self:AutoRepairPlease()
	end
	
	if auditorMode ~= "repairs" then
		auditorMode = "merch"
	end
		
    if self.db.realm.notifyVendor == true then
	    auditorCN_open = true
	    auditorCN_gain = 0
	    auditorCN_spend = 0
	    auditorCN_mode = L.Vendor
    end
	
	if self.db.realm.sellGreys == true then
		self:ScheduleTimer("AUDITOR_SELLGREYS_DELAY", function() self:SellGreys() end, 2)
	end
end
function Auditor:MERCHANT_CLOSED()
	auditorMode = ""
	self:CashNotify()
end
function Auditor:CONFIRM_TALENT_WIPE()
	auditorMode = "train"
end
function Auditor:GUILDBANKFRAME_OPENED()
	auditorMode = "guildbank"
	
	if self.db.realm.notifyTrainer == true then
        auditorCN_open = true
        auditorCN_gain = 0
        auditorCN_spend = 0
        auditorCN_mode = L.Guildbank
    end
end
function Auditor:GUILDBANKFRAME_CLOSED()
	auditorMode = ""
	self:CashNotify()
end
function Auditor:QUEST_COMPLETE()
	auditorMode = "quest"
end
function Auditor:LOOT_OPENED()
	auditorMode = "loot"
end
function Auditor:TAXIMAP_OPENED()
	auditorMode = "taxi"
end
function Auditor:TRADE_SHOW()
	auditorMode = "trade"
end
function Auditor:TRADE_CLOSE()
	auditorMode = ""
end
function Auditor:MAIL_SHOW()
	auditorMode = "mail"
	
    if self.db.realm.notifyMailbox == true then
        auditorCN_open = true
        auditorCN_gain = 0
        auditorCN_spend = 0
        auditorCN_mode = L.Mailbox
    end
end
function Auditor:MAIL_CLOSED()
	auditorMode = ""
	self:CashNotify()
end
function Auditor:BANKFRAME_OPENED()
	auditorMode = "merch"
end
function Auditor:BANKFRAME_CLOSED()
	auditorMode = ""
end
function Auditor:TRAINER_SHOW()
	auditorMode = "train"
	
    if self.db.realm.notifyTrainer == true then
        auditorCN_open = true
        auditorCN_gain = 0
        auditorCN_spend = 0
        auditorCN_mode = L.Trainer
    end
end
function Auditor:TRAINER_CLOSED()
	auditorMode = ""
	self:CashNotify()
end
function Auditor:AUCTION_HOUSE_SHOW()
	auditorMode = "ah"
	
    if self.db.realm.notifyAH == true then
        auditorCN_open = true
        auditorCN_gain = 0
        auditorCN_spend = 0
        auditorCN_mode = L.AH
    end
end
function Auditor:AUCTION_HOUSE_CLOSED()
	auditorMode = ""
	self:CashNotify()
end
function Auditor:CHAT_MSG_MONEY()
	self:OnShareMoney(event, arg1)
end
function Auditor:PLAYER_MONEY()
	self:UpdateTimeFrame()
	self:UpdateFigures()
	self:CashNotifyUpdate()
	if (AuditorFu) then AuditorFu:UpdateDisplay() end
end

function Auditor:OnShareMoney(event, arg1) 
	local gold, silver, copper, money, oldMode 

	-- Parse the message for money gained. 
	_, _, gold = string.find(arg1, "(%d+) " .. GOLD) 
	_, _, silver = string.find(arg1, "(%d+) " .. SILVER)
	_, _, copper = string.find(arg1, "(%d+) " .. COPPER) 
	if gold then gold = tonumber(gold) else gold = 0 end 
	if silver then silver = tonumber(silver) else silver = 0 end 
	if copper then copper = tonumber(copper) else copper = 0 end 
	money = copper + silver * 100 + gold * 10000

	oldMode = auditorMode 
	if not auditorLastMoney then auditorLastMoney = 0 end 

	-- This will force a money update with calculated amount. 
	auditorLastMoney = auditorLastMoney - money 
	auditorMode = "loot" 
	Auditor:UpdateFigures() 
	auditorMode = oldMode 

	-- This will suppress the incoming PLAYER_MONEY event. 
	auditorLastMoney = auditorLastMoney + money
end


--[[ MAIN MONEY DATA ALLOCATION FUNCTION ]]--

function Auditor:UpdateFigures()
	auditorCurrentMoney = GetMoney()
	self.db.realm[auditorPlayer].data.totals.actual = auditorCurrentMoney
	self.db.realm.chars[auditorPlayer] = auditorCurrentMoney
	local diff = auditorCurrentMoney - auditorLastMoney
	auditorLastMoney = auditorCurrentMoney
	
	if (diff == 0 or diff == nil) then
		return
	end		

	-- Deals with random cash from an obfuscated source. Good word, eh?
	if auditorMode == "" then
		auditorMode = "other"
	end
	
	if diff > 0 then
		-- Prevention of incomings to categories that can't have incomings - not perfect, but better than nothing
		if (auditorMode == "taxi" or auditorMode == "train" or auditorMode == "repairs") then
			auditorMode = "other"
		end
		for _,logmode in pairs(auditorTimeframe) do
			self.db.realm[auditorPlayer].data[auditorMode][logmode].incomings = self.db.realm[auditorPlayer].data[auditorMode][logmode].incomings + diff
		end
		self.db.realm[auditorPlayer].data.totals.incomings = self.db.realm[auditorPlayer].data.totals.incomings + diff
		self.db.realm[auditorPlayer].historical[auditorHDay[7]][auditorMode].incomings = self.db.realm[auditorPlayer].data[auditorMode]["day"].incomings
	elseif diff < 0 then
		-- Prevention of outgoings to categories that can't have outgoings - not perfect, but better than nothing
		if (auditorMode == "loot") then
			auditorMode = "other"
		end
		diff = diff * -1;
		for _,logmode in pairs(auditorTimeframe) do
			self.db.realm[auditorPlayer].data[auditorMode][logmode].outgoings = self.db.realm[auditorPlayer].data[auditorMode][logmode].outgoings + diff
		end
		self.db.realm[auditorPlayer].data.totals.outgoings = self.db.realm[auditorPlayer].data.totals.outgoings + diff
		self.db.realm[auditorPlayer].historical[auditorHDay[7]][auditorMode].outgoings = self.db.realm[auditorPlayer].data[auditorMode]["day"].outgoings
	end
			
	-- Reset in case we just repaired and are still at a merchant, like
	if auditorMode == "repairs" then
		self:Print(L.RepairSuccessful..Auditor:FormatMoneyFull(diff, true, coloured))
		auditorMode = "merch"
	end
end

function Auditor:UpdateTimeFrame()
	-- Check if Auditor should reset
	local dateTable = date("*t")
	local timeDiff = time{year = dateTable.year, day = dateTable.day, month = dateTable.month, hour = dateTable.hour, min = dateTable.min, sec = dateTable.sec} - self.db.realm[auditorPlayer].timeDate.timeDate
	local daysPassedSinceLogin = math.floor(timeDiff / 86400)
	-- Keep date current
	self.db.realm[auditorPlayer].timeDate.niceDate = date("%d %b '%y")
			
	if daysPassedSinceLogin ~= 0 then
		-- If we haven't logged in for a week, then we need to clear the historical days data and the week data
		if daysPassedSinceLogin > 6 then
			self.db.realm[auditorPlayer].historical = nil
			self:NewToonDB()
			self:DataReset("day")
			self:DataReset("week")
		-- If the last time we logged in was less than a week ago, then we need to start shifting the data from historicalData to historicalData, and make sure the week data is kept up to date too.
		-- Begin the Truffle Shuffle!
		else
			self:DataReset("week")
			while daysPassedSinceLogin > 0 do
				for i = 1, 6 do
					for _,logmode in pairs(auditorLogMode) do
						self.db.realm[auditorPlayer].historical[auditorHDay[i]][logmode].incomings = self.db.realm[auditorPlayer].historical[auditorHDay[i+1]][logmode].incomings
						self.db.realm[auditorPlayer].historical[auditorHDay[i]][logmode].outgoings = self.db.realm[auditorPlayer].historical[auditorHDay[i+1]][logmode].outgoings
					end
				end
				for _,logmode in pairs(auditorLogMode) do
					self.db.realm[auditorPlayer].historical[auditorHDay[7]][logmode] = {incomings = 0, outgoings = 0}
				end
				daysPassedSinceLogin = daysPassedSinceLogin - 1
			end
			for _,hday in pairs(auditorHDay) do
				for _,logmode in pairs(auditorLogMode) do
					self.db.realm[auditorPlayer].data[logmode]["week"].incomings = self.db.realm[auditorPlayer].data[logmode]["week"].incomings + self.db.realm[auditorPlayer].historical[hday][logmode].incomings
					self.db.realm[auditorPlayer].data[logmode]["week"].outgoings = self.db.realm[auditorPlayer].data[logmode]["week"].outgoings + self.db.realm[auditorPlayer].historical[hday][logmode].outgoings
				end
			end
		end
		-- Lastly, if the date has changed, obviously we need to clear out the day's data.
		self:DataReset("day")
		
		-- Time offset crap. Amfg, I hate this stuff. It's retarded.
		self.db.realm[auditorPlayer].timeDate.timeDate = time{year = dateTable.year, day = dateTable.day, month = dateTable.month, hour = 0} + (self.db.realm.timeOffset * 3600)
	end
end

function Auditor:PurgeData(name)
	auditorFuToonToShow = auditorPlayer
	self.db.realm[name] = nil
	self.db.realm.chars[name] = nil
	if auditorPlayer == name then
		-- Create DB for Toon
		self:NewToonDB()
		-- Reconcile DB with reality
		self:ReconcileData()
	end
	AuditorFu.cached_options = nil
	AuditorFu:OpenMenu()
end

function Auditor:NewToonDB()
	-- Normal
	if not self.db.realm[auditorPlayer] then
		self.db.realm[auditorPlayer] = { data = { totals = {incomings = 0, outgoings = 0, actual = 0}}, timeDate = {niceDate = date("%d %b '%y"), timeDate = (math.floor((time() + (self.db.realm.timeOffset * 3600)) / 86400)) * 86400}}
		for _,logmode in pairs(auditorLogMode) do
			self.db.realm[auditorPlayer].data[logmode] = {}
			for _,timeframe in pairs(auditorTimeframe) do
				self.db.realm[auditorPlayer].data[logmode][timeframe] = {incomings = 0, outgoings = 0}
			end
		end
	end	
	-- Historical
	if not self.db.realm[auditorPlayer].historical then
		self.db.realm[auditorPlayer].historical = {}
		for _,hday in pairs(auditorHDay) do
			self.db.realm[auditorPlayer].historical[hday] = {}
			for _,logmode in pairs(auditorLogMode) do
				self.db.realm[auditorPlayer].historical[hday][logmode] = {incomings = 0, outgoings = 0}
			end
		end
	end
	-- Character totals
	if not self.db.realm.chars then
		self.db.realm.chars = {}
	end
	if not self.db.realm.chars[auditorPlayer] then
		self.db.realm.chars[auditorPlayer] = 0
	end
	-- Guild repairs tally
	if not self.db.realm[auditorPlayer].guildRepairsTally then
		self.db.realm[auditorPlayer].guildRepairsTally = 0
	end
	-- Reconciliation category for people who had Auditor before this category was added
	for name in pairs(self.db.realm.chars) do
		if not self.db.realm[name].data["reconciliation"] then
			self.db.realm[name].data["reconciliation"] = {}
			for _,timeframe in pairs(auditorTimeframe) do
				self.db.realm[name].data["reconciliation"][timeframe] = {incomings = 0, outgoings = 0}
			end
			for _,hday in pairs(auditorHDay) do
				self.db.realm[name].historical[hday]["reconciliation"] = {incomings = 0, outgoings = 0}
			end
		end
	end
	-- Guildbank category for people who had Auditor before this category was added
	for name in pairs(self.db.realm.chars) do
		if not self.db.realm[name].data["guildbank"] then
			self.db.realm[name].data["guildbank"] = {}
			for _,timeframe in pairs(auditorTimeframe) do
				self.db.realm[name].data["guildbank"][timeframe] = {incomings = 0, outgoings = 0}
			end
			for _,hday in pairs(auditorHDay) do
				self.db.realm[name].historical[hday]["guildbank"] = {incomings = 0, outgoings = 0}
			end
		end
	end
	-- Timesystem Upgrade
	if self.db.realm.timeSystemUpgraded == false then
		for name in pairs(self.db.realm.chars) do
			self.db.realm[name].timeDate.niceDate = date("%d %b '%y")
		end
		
		local dateTable = date("*t")
		for name in pairs(Auditor.db.realm.chars) do
			Auditor.db.realm[name].historical = nil
			Auditor:DataReset("session", name)
			Auditor:DataReset("day", name)
			Auditor:DataReset("week", name)
			Auditor.db.realm[name].timeDate.timeDate = time{year = dateTable.year, day = dateTable.day, month = dateTable.month, hour = 0} + (Auditor.db.realm.timeOffset * 3600)
		end
		self.db.realm.timeSystemUpgraded = true -- Only run this function once per realm
		Auditor:NewToonDB()
	end
end

--[[ HOOKING FUNCTIONS ]]--

function Auditor:RepairAllItems()
	auditorMode = "repairs"
end

function Auditor:CursorHasItem()
	if InRepairMode() then
		auditorMode = "repairs"
	end
end

function Auditor:TakeInboxMoney(mailIndex)
	local _, _, sender, _, _, _, _, _, _, _, _, _= GetInboxHeaderInfo(mailIndex)
	if sender ~= nil then
		if strfind(sender, L.AuctionHouse) then
			auditorMode = "ah"
		else
			auditorMode = "mail"
		end
	else
		auditorMode = "mail"
	end
end

--[[ UTILITY FUNCTIONS ]]--

function Auditor:PrintResetTime()
	self:Print(date("Auditor will reset at %H:%M (24hr clock), on %A, %d %b '%y (local time).", self.db.realm[auditorPlayer].timeDate.timeDate + 86400))
end

function Auditor:CashNotifyUpdate()
	auditorCN_oldCash, auditorCN_cash = auditorCN_cash, GetMoney()
	local diff = auditorCN_cash - auditorCN_oldCash
	
	if auditorCN_open == true then
		if diff > 0 then
			auditorCN_gain = auditorCN_gain + diff
		elseif diff < 0 then
			auditorCN_spend = auditorCN_spend - diff
		end
	end
end

function Auditor:CashNotify()
	auditorCN_open = false
	
	if auditorCN_gain > 0 then
		self:Print(L.Gained..Auditor:FormatMoneyFull(auditorCN_gain, true, coloured)..auditorCN_mode)
	end
	
	if auditorCN_spend > 0 then
		self:Print(L.Spent..Auditor:FormatMoneyFull(auditorCN_spend, true, coloured)..auditorCN_mode)
	end
	
	local diff = auditorCN_gain - auditorCN_spend
	if diff ~= 0 and auditorCN_gain > 0 and auditorCN_spend > 0 then
		self:Print(L.Change..Auditor:FormatMoneyFull(auditorCN_gain - auditorCN_spend, true, coloured)..auditorCN_mode)
	end
	
	-- Spank variables
	auditorCN_gain = 0
	auditorCN_spend = 0
end

function Auditor:AutoRepairPlease()
	local repairCost = GetRepairAllCost()
	local currentMoney = GetMoney()
	
	if repairCost > 0 then
		if self.db.profile.guildRepair and IsInGuild() then
			local guildMoney = GetGuildBankWithdrawMoney()
			if guildMoney == -1 then
				guildMoney = GetGuildBankMoney()
			end
			if guildMoney > repairCost then
				RepairAllItems(1)
				self.db.realm[auditorPlayer].guildRepairsTally = self.db.realm[auditorPlayer].guildRepairsTally + repairCost
				self:Print(L.GuildRepairSuccessful..Auditor:FormatMoneyFull(repairCost, true, coloured))
				self:Print(L.GuildTally..Auditor:FormatMoneyFull(self.db.realm[auditorPlayer].guildRepairsTally, true, coloured))
				return
			else
				self:Print(L.GuildRepairFailed)
			end
		end
	
		if (currentMoney > repairCost) then
			auditorMode = "repairs"
			RepairAllItems()
		else
			self:Print(L.InsufficientFunds)
		end
	end
end

function Auditor:SellGreys()
	auditorMode = "merch"
	for bag=0,4 do
		for slot=0,GetContainerNumSlots(bag) do
			local link = GetContainerItemLink(bag, slot)
			if link and select(3, GetItemInfo(link)) == 0 then
				ShowMerchantSellCursor(1)
				UseContainerItem(bag, slot)
			end
		end
	end
end

function Auditor:FormatMoneyFull(value, colorize, textColor)
	local gold = abs(value / 10000)
	local silver = abs(mod(value / 100, 100))
	local copper = abs(mod(value, 100))
	
	local negl = ""
	local color = COLOR_WHITE
	if value > 0 then
		if textColor then
			color = COLOR_GREEN
		end
	elseif value < 0 then
		negl = "-"
		if textColor then
			color = COLOR_RED
		end
	end
	if colorize then
		if value == inf or value == -inf then
			return format("|cff%s%s|r", color, value)
		elseif value ~= value then
			return format("|cff%s0|r|cff%s%s|r", COLOR_WHITE, COLOR_COPPER, COPPER_ABBR)
		elseif value >= 10000 or value <= -10000 then
			return format("|cff%s%s%d|r|cff%s%s|r |cff%s%d|r|cff%s%s|r |cff%s%d|r|cff%s%s|r", color, negl, gold, COLOR_GOLD, GOLD_ABBR, color, silver, COLOR_SILVER, SILVER_ABBR, color, copper, COLOR_COPPER, COPPER_ABBR)
		elseif value >= 100 or value <= -100 then
			return format("|cff%s%s%d|r|cff%s%s|r |cff%s%d|r|cff%s%s|r", color, negl, silver, COLOR_SILVER, SILVER_ABBR, color, copper, COLOR_COPPER, COPPER_ABBR)
		else
			return format("|cff%s%s%d|r|cff%s%s|r", color, negl, copper, COLOR_COPPER, COPPER_ABBR)
		end
	else
		if value == inf or value == -inf then
			return format("%s", value)
		elseif value ~= value then
			return format("0%s", COPPER_ABBR)
		elseif value >= 10000 or value <= -10000 then
			return format("%s%d%s %d%s %d%s", negl, gold, GOLD_ABBR, silver, SILVER_ABBR, copper, COPPER_ABBR)
		elseif value >= 100 or value <= -100 then
			return format("%s%d%s %d%s", negl, silver, SILVER_ABBR, copper, COPPER_ABBR)
		else
			return format("%s%d%s", negl, copper, COPPER_ABBR)
		end
	end
end

function Auditor:FormatMoneyShort(copper, colorize, textColor)
	local color = COLOR_WHITE
	if textColor then
		if copper > 0 then
			color = COLOR_GREEN
		elseif copper < 0 then
			color = COLOR_RED
		end
	end
	if colorize then
		if copper == inf or copper == -inf then
			return format("|cff%s%s|r", color, copper)
		elseif copper ~= copper then
			return format("|cff%s0|r|cff%s%s|r", COLOR_WHITE, COLOR_COPPER, COPPER_ABBR)
		elseif copper >= 10000 or copper <= -10000 then
			return format("|cff%s%.1f|r|cff%s%s|r", color, copper / 10000, COLOR_GOLD, GOLD_ABBR)
		elseif copper >= 100 or copper <= -100 then
			return format("|cff%s%.1f|r|cff%s%s|r", color, copper / 100, COLOR_SILVER, SILVER_ABBR)
		else
			return format("|cff%s%d|r|cff%s%s|r", color, copper, COLOR_COPPER, COPPER_ABBR)
		end
	else
		if value == copper or value == -copper then
			return format("%s", copper)
		elseif copper ~= copper then
			return format("0%s", COPPER_ABBR)
		elseif copper >= 10000 or copper <= -10000 then
			return format("%.1f%s", copper / 10000, GOLD_ABBR)
		elseif copper >= 100 or copper <= -100 then
			return format("%.1f%s", copper / 100, SILVER_ABBR)
		else
			return format("%.0f%s", copper, COPPER_ABBR)
		end
	end
end

function Auditor:FormatMoneyCondensed(value, colorize, textColor)
	local negl = ""
	local negr = ""
	if value < 0 then
		if colorize and textColor then
			negl = "|cffff0000-(|r"
			negr = "|cffff0000)|r"
		else
			negl = "-("
			negr = ")"
		end
	end
	local gold = floor(math.abs(value) / 10000)
	local silver = mod(floor(math.abs(value) / 100), 100)
	local copper = mod(floor(math.abs(value)), 100)
	if colorize then
		if value == inf or value == -inf then
			return format("%s|cff%s%s|r%s", negl, COLOR_COPPER, math.abs(value), negr)
		elseif value ~= value then
			return format("|cff%s0|r", COLOR_COPPER)
		elseif gold ~= 0 then
			return format("%s|cff%s%d|r.|cff%s%02d|r.|cff%s%02d|r%s", negl, COLOR_GOLD, gold, COLOR_SILVER, silver, COLOR_COPPER, copper, negr)
		elseif silver ~= 0 then
			return format("%s|cff%s%d|r.|cff%s%02d|r%s", negl, COLOR_SILVER, silver, COLOR_COPPER, copper, negr)
		else
			return format("%s|cff%s%d|r%s", negl, COLOR_COPPER, copper, negr)
		end
	else
		if value == inf or value == -inf then
			return tostring(value)
		elseif value ~= value then
			return "0"
		elseif gold ~= 0 then
			return format("%s%d.%02d.%02d%s", negl, gold, silver, copper, negr)
		elseif silver ~= 0 then
			return format("%s%d.%02d%s", negl, silver, copper, negr)
		else
			return format("%s%d%s", negl, copper, negr)
		end
	end
end

function Auditor:MoneyStyle(formattedCash, coloured)
	if AuditorFu.db.profile.cashFormat.toolTip == "Full" then
		return Auditor:FormatMoneyFull(formattedCash, true, coloured)
	elseif AuditorFu.db.profile.cashFormat.toolTip == "Short" then
		return Auditor:FormatMoneyShort(formattedCash, true, coloured)
	else
		return Auditor:FormatMoneyCondensed(formattedCash, true, coloured)
	end
end