--[[
    Armory Addon for World of Warcraft(tm).
    Revision: $Id: ArmoryAddonMessageFrame.lua,v 1.2, 2008-06-17 08:09:32Z, Maxim Baars$
    URL: http://www.wow-neighbours.com

    License:
        This program is free software; you can redistribute it and/or
        modify it under the terms of the GNU General Public License
        as published by the Free Software Foundation; either version 2
        of the License, or (at your option) any later version.

        This program is distributed in the hope that it will be useful,
        but WITHOUT ANY WARRANTY; without even the implied warranty of
        MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
        GNU General Public License for more details.

        You should have received a copy of the GNU General Public License
        along with this program(see GPL.txt); if not, write to the Free Software
        Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

    Note:
        This AddOn's source code is specifically designed to work with
        World of Warcraft's interpreted AddOn system.
        You have an implicit licence to use this AddOn with these facilities
        since that is it's designated purpose as per:
        http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat
--]] 

ARMORY_GUILDBANK_LINES_DISPLAYED = 20;

ARMORY_GUILDBANK_MESSAGE_TYPE = { BANK_SYNC_PUSH = "BSP", BANK_SYNC_REQUEST = "BSR" };

ARMORY_GUILDBANK_SYNC_TABLE = {};
ARMORY_GUILDBANK_SYNC_DELAY_MIN = 60;
ARMORY_GUILDBANK_SYNC_DELAY_MAX = 180;

ARMORY_GUILDBANK_QUEUE = {};
ARMORY_GUILDBANK_QUEUE_WAIT_TIME = 10;

ARMORY_GUILDBANK_PUSH_DELAY = 10;

if ( not AGB ) then
    AGB = {};
end

if ( not AgbDB ) then
    AgbDB = {};
end

StaticPopupDialogs["ARMORY_DELETE_GUILDBANK"] = {
    text = ARMORY_DELETE_UNIT,
    button1 = YES,
    button2 = NO,
    OnAccept = function()
        ArmoryGuildBankFrame_Delete();
    end,
    timeout = 0,
    whileDead = 1,
    exclusive = 1,
    showAlert = 1,
    hideOnEscape = 1
};

function ArmoryGuildBankFrame_Toggle()
    if ( ArmoryGuildBankFrame:IsShown() ) then
        HideUIPanel(ArmoryGuildBankFrame);
    else
        ShowUIPanel(ArmoryGuildBankFrame);
    end
end

function ArmoryGuildBankFrame_OnLoad()
    this:RegisterEvent("PLAYER_ENTERING_WORLD");
    this:RegisterEvent("PLAYER_GUILD_UPDATE");
    this:RegisterEvent("GUILDTABARD_UPDATE");
    this:RegisterEvent("GUILDBANKFRAME_OPENED");
    this:RegisterEvent("GUILDBANKFRAME_CLOSED");
    this:RegisterEvent("GUILDBANKBAGSLOTS_CHANGED");

    this:SetAttribute("UIPanelLayout-defined", true);
    this:SetAttribute("UIPanelLayout-enabled", true);
    this:SetAttribute("UIPanelLayout-area", "left");
    this:SetAttribute("UIPanelLayout-pushable", 5);
    this:SetAttribute("UIPanelLayout-whileDead", true);

    table.insert(UISpecialFrames, "ArmoryGuildBankFrame");

    SetPortraitToTexture("ArmoryGuildBankFramePortrait", "Interface\\LFGFrame\\BattlenetWorking4");

    ArmoryGuildBankFrame_ResetScrollBar();
    ArmoryGuildBankFrame_Register();
    
    ArmoryGuildBankFrameEditBox:SetText(SEARCH);

    AGB:SelectDb();
end

function ArmoryGuildBankFrame_Register()
    ArmoryAddonMessageFrame_RegisterHandlers(ArmoryGuildBankFrame_CheckResponse, ArmoryGuildBankFrame_ProcessRequest);

    Armory.options["ARMORY_CMD_GUILDBANK"] = {
        type = "execute",
        run = function() ArmoryGuildBankFrame_Toggle() end,
        disabled = function () return not Armory:HasDataSharing() end
    };
    Armory.options["ARMORY_CMD_SET_AGBITEMCOUNT"] = {
        type = "toggle",
        set = function(value) AGB:SetConfigShowItemCount(value and value ~= "0"); end,
        get = function() return AGB:GetConfigShowItemCount(); end,
        default = true
    };
    Armory.options["ARMORY_CMD_SET_AGBFIND"] = {
        type = "toggle",
        set = function(value) AGB:SetConfigIncludeInFind(value and value ~= "0"); end,
        get = function() return AGB:GetConfigIncludeInFind(); end,
        default = true
    };

    Armory:SetCommand("ARMORY_CMD_GUILDBANK", function(...) ArmoryGuildBankFrame_Toggle(...) end);

    if ( ArmoryFu ) then
        ArmoryFu:SetOption(ArmoryFu.OnMenuRequest["args"], 11, "ARMORY_CMD_GUILDBANK")
    end
end

function ArmoryGuildBankFrame_OnEvent(event)
    if ( event == "PLAYER_ENTERING_WORLD" ) then
        this:UnregisterEvent("PLAYER_ENTERING_WORLD");
        AGB.realm = GetRealmName();
        if ( IsInGuild() ) then
            Armory:ExecuteConditional(function() AGB.guild = GetGuildInfo("player"); return AGB.guild; end, ArmoryGuildBankFrame_Initialize);
        end
    elseif ( event == "PLAYER_GUILD_UPDATE" ) then
        if ( not IsInGuild() and AGB.realm and AGB.guild and AgbDB[AGB.realm][AGB.guild] ) then
            AgbDB[AGB.realm][AGB.guild] = nil;
        end
    elseif ( event == "GUILDTABARD_UPDATE" or event == "GUILDBANKFRAME_OPENED" ) then
        AGB:UpdateGuildInfo();
    elseif ( event == "GUILDBANKFRAME_CLOSED" ) then
        AGB.tabs = nil;
    elseif ( event == "GUILDBANKBAGSLOTS_CHANGED" ) then
        if ( not AGB.tabs ) then
            AGB.tabs = {};
        end
        AGB.tabs[GetCurrentGuildBankTab()] = 1;
    end
end

function ArmoryGuildBankFrame_Initialize()
    ArmoryGuildBankFrame_SelectGuild();
    ArmoryGuildBankNameDropDown_Initialize();
    ArmoryGuildBankFrame_PushInfo(); 
end

function ArmoryGuildBankFrame_OnShow()
    if ( ArmoryGuildBankFrame_SelectGuild() ) then
        if ( (AGB.filter or "") == "" ) then
            ArmoryGuildBankFrameEditBox:SetText(SEARCH);
        else
            ArmoryGuildBankFrameEditBox:SetText(AGB.filter);
        end
    else
        ArmoryGuildBankFrame:Hide();
        Armory:PrintTitle(ARMORY_GUILDBANK_NO_DATA.." "..ARMORY_GUILDBANK_ABORTING);
    end    
end

function ArmoryGuildBankFrame_OnTextChanged()
    local text = this:GetText();
    local refresh;

    if ( text == SEARCH ) then
        refresh = AGB:SetFilter("");
    elseif ( text ~= "=" ) then
        refresh = AGB:SetFilter(text);
    end
    if ( refresh ) then
        AGB:UpdateItemLines();
        ArmoryGuildBankFrame_ResetScrollBar();
        ArmoryGuildBankFrame_Update();
    end
end

function ArmoryGuildBankFilterDropDown_OnLoad()
    UIDropDownMenu_SetWidth(116, this);
    UIDropDownMenu_JustifyText("LEFT", this);
    ArmoryItemFilter_InitializeDropDown(this);
end

function ArmoryGuildBankFilterDropDown_OnShow()
    ArmoryItemFilter_SelectDropDown(this, ArmoryGuildBankFrame_Refresh);
end

function ArmoryGuildBankNameDropDown_OnLoad()
    UIDropDownMenu_Initialize(this, ArmoryGuildBankNameDropDown_Initialize);
    UIDropDownMenu_SetWidth(112, this);
    UIDropDownMenu_JustifyText("LEFT", this);
end

function ArmoryGuildBankNameDropDown_Initialize()
    -- Setup buttons
    local currentRealm = AGB.selectedRealm or AGB.realm;
    local currentGuild = AGB.selectedGuild or AGB.guild;
    local info, checked;
    for _, realm in ipairs(AGB:RealmList()) do
        info = UIDropDownMenu_CreateInfo();
        info.text = realm;
        info.notClickable = 1;
        info.notCheckable = 1;
        info.isTitle = 1;
        UIDropDownMenu_AddButton(info);
        for _, guild in ipairs(AGB:GuildList(realm)) do
            local profile = {realm=realm, guild=guild};
            if ( realm == currentRealm and guild == currentGuild ) then
                checked = 1;
                UIDropDownMenu_SetSelectedValue(ArmoryGuildBankNameDropDown, profile);
            else
                checked = nil;
            end
            info = UIDropDownMenu_CreateInfo();
            info.text = guild;
            info.func = ArmoryGuildBankNameDropDown_OnClick;
            info.value = profile;
            info.checked = checked;
            UIDropDownMenu_AddButton(info);
        end
    end
end

function ArmoryGuildBankNameDropDown_OnClick()
    local profile = this.value;
    UIDropDownMenu_SetSelectedValue(ArmoryGuildBankNameDropDown, profile);
    ArmoryGuildBankFrame_SelectGuild(profile.realm, profile.guild);
end

function ArmoryGuildBankFrameButton_OnEnter(button)
    if ( button.link ) then
        GameTooltip:SetHyperlink(button.link);
    end
end

function ArmoryGuildBankFrame_SelectGuild(realm, guild)
    local dbEntry, refresh = AGB:SelectDb(realm, guild);

    UIDropDownMenu_SetText(AGB.selectedGuild, ArmoryGuildBankNameDropDown);

    if ( refresh ) then
        AGB:UpdateItemLines();
        ArmoryGuildBankFrame_ResetScrollBar();

        MoneyFrame_Update("ArmoryGuildBankMoneyFrame", AGB:GetMoney(dbEntry) or 0);
        ArmoryGuildBankFrame_UpdateGuildInfo(dbEntry);
        ArmoryGuildBankFrame_Update();
    end
    
    return dbEntry;
end

function ArmoryGuildBankFrame_UpdateGuildInfo(dbEntry)
    if ( dbEntry ) then
        local tabardBackgroundUpper, tabardBackgroundLower, tabardEmblemUpper, tabardEmblemLower = AGB:GetTabardFiles(dbEntry);

        if ( tabardBackgroundUpper ) then
            ArmoryGuildBankEmblemBackgroundUL:SetTexture(tabardBackgroundUpper);
            ArmoryGuildBankEmblemBackgroundUR:SetTexture(tabardBackgroundUpper);
            ArmoryGuildBankEmblemBackgroundBL:SetTexture(tabardBackgroundLower);
            ArmoryGuildBankEmblemBackgroundBR:SetTexture(tabardBackgroundLower);

            ArmoryGuildBankEmblemUL:SetTexture(tabardEmblemUpper);
            ArmoryGuildBankEmblemUR:SetTexture(tabardEmblemUpper);
            ArmoryGuildBankEmblemBL:SetTexture(tabardEmblemLower);
            ArmoryGuildBankEmblemBR:SetTexture(tabardEmblemLower);
        else
            ArmoryGuildBankEmblemBackgroundUL:SetTexture("");
            ArmoryGuildBankEmblemBackgroundUR:SetTexture("");
            ArmoryGuildBankEmblemBackgroundBL:SetTexture("");
            ArmoryGuildBankEmblemBackgroundBR:SetTexture("");

            ArmoryGuildBankEmblemUL:SetTexture("");
            ArmoryGuildBankEmblemUR:SetTexture("");
            ArmoryGuildBankEmblemBL:SetTexture("");
            ArmoryGuildBankEmblemBR:SetTexture("");
        end

        local factionGroup, factionName = AGB:GetFaction(dbEntry);
        if ( factionGroup ) then
            ArmoryGuildBankFrameFactionIcon:SetTexture("Interface\\TargetingFrame\\UI-PVP-"..factionGroup);
        else
            ArmoryGuildBankFrameFactionIcon:SetTexture("");
        end
    else
        AGB:UpdateGuildInfo();
    end
end

function ArmoryGuildBankFrame_Update()
    if ( not AGB.itemLines ) then
        AGB:UpdateItemLines();
    end

    local numLines = #AGB.itemLines;
    local offset = FauxScrollFrame_GetOffset(ArmoryGuildBankScrollFrame);

    if ( numLines == 0 ) then
        local msg = ARMORY_GUILDBANK_NO_DATA;
        if ( AGB.selectedRealm == AGB.realm and AGB.selectedGuild == AGB.guild ) then
            msg = msg.."\n\n"..ARMORY_GUILDBANK_NO_TABS;
        end
        ArmoryGuildBankFrameMessage:SetText(msg);
        ArmoryGuildBankFrameMessage:Show();
    else
        ArmoryGuildBankFrameMessage:Hide();
    end

    if ( offset > numLines ) then
        offset = 0;
        FauxScrollFrame_SetOffset(ArmoryGuildBankScrollFrame, offset);
    end

    -- ScrollFrame update
    FauxScrollFrame_Update(ArmoryGuildBankScrollFrame, numLines, ARMORY_GUILDBANK_LINES_DISPLAYED, ARMORY_LOOKUP_HEIGHT);

    for i = 1, ARMORY_GUILDBANK_LINES_DISPLAYED do
        local lineIndex = i + offset;
        local lineButton = getglobal("ArmoryGuildBankLine"..i);
        local lineButtonText = getglobal("ArmoryGuildBankLine"..i.."Text");
        local lineButtonDisabled = getglobal("ArmoryGuildBankLine"..i.."Disabled");

        if ( lineIndex <= numLines ) then
            -- Set button widths if scrollbar is shown or hidden
            if ( ArmoryGuildBankScrollFrame:IsShown() ) then
                lineButtonText:SetWidth(265);
                lineButtonDisabled:SetWidth(295);
            else
                lineButtonText:SetWidth(285);
                lineButtonDisabled:SetWidth(315);
            end

            local name, isHeader, count, texture, link = unpack(AGB.itemLines[lineIndex]);
            local color;

            lineButton.link = link;
            if ( texture ) then
                lineButton:SetNormalTexture(texture);
            else
                lineButton:SetNormalTexture("");
            end

            if ( isHeader ) then
                lineButton:Disable();
            else
                if ( link ) then
                    color = link:match("^(|c%x+)|H");
                end
                name = (color or HIGHLIGHT_FONT_COLOR_CODE)..name..FONT_COLOR_CODE_CLOSE.." x "..count;
                lineButton:Enable();
            end
            lineButton:SetText(name);
            lineButton:Show();
        else
            lineButton:Hide();
        end
    end
    
    if ( table.getn(AGB:RealmList()) > 1 or table.getn(AGB:GuildList(AGB.selectedRealm or AGB.realm)) > 1 ) then
        ArmoryGuildBankNameDropDownButton:Enable();
    else
        ArmoryGuildBankNameDropDownButton:Disable();
    end
end

function ArmoryGuildBankFrame_Refresh()
    if ( AGB.selectedRealm == AGB.realm and AGB.selectedGuild == AGB.guild ) then
        AGB.itemLines = nil;
        if ( ArmoryGuildBankFrame:IsShown() ) then
            ArmoryGuildBankFrame_Update();
        end
    end
end

function ArmoryGuildBankFrame_ResetScrollBar()
    FauxScrollFrame_SetOffset(ArmoryGuildBankScrollFrame, 0);
    ArmoryGuildBankScrollFrameScrollBar:SetMinMaxValues(0, 0); 
    ArmoryGuildBankScrollFrameScrollBar:SetValue(0);
end

function ArmoryGuildBankFrame_Delete()
    if ( AGB.selectedRealm and AGB.selectedGuild and AgbDB[AGB.selectedRealm][AGB.selectedGuild] ) then
        AgbDB[AGB.selectedRealm][AGB.selectedGuild] = nil;
        AGB.selectedRealm = nil;
        AGB.selectedGuild = nil;
        AGB.selectedDbEntry = nil;

        ArmoryGuildBankFrame_SelectGuild();
        ArmoryGuildBankNameDropDown_Initialize();
    end
end


----------------------------------------------------------
-- Hooks
----------------------------------------------------------

local Orig_CloseGuildBankFrame = CloseGuildBankFrame;
function CloseGuildBankFrame(...)

    if ( AGB.tabs ) then
        for tab in pairs(AGB.tabs) do
            ArmoryGuildBankFrame_RemoveFromQueue(tab);
        
            local name = GetGuildBankTabInfo(tab);
            local items = {};
            local itemId, link, count;
            for i = 1, MAX_GUILDBANK_SLOTS_PER_TAB do
                link = GetGuildBankItemLink(tab, i);
                if ( link ) then
                    _, count = GetGuildBankItemInfo(tab, i);
                    _, itemId = Armory:GetItemLinkInfo(link);
                    itemId = AGB:NormalizeItemString(itemId);
                    items[itemId] = (items[itemId] or 0) + count;
                end
            end

            local ids = {};
            for itemId, count in pairs(items) do
                table.insert(ids, itemId..count);
            end
            table.sort(ids);

            AGB:UpdateTabName(tab, name);
            AGB:UpdateTabItems(tab, items, AGB:Checksum(table.concat(ids)));
        end
    end

    for tab = 1, GetNumGuildBankTabs() do
        if ( not IsTabViewable(tab) ) then
            AGB:DeleteTab(tab);
        end
    end

    AGB:UpdateMoney();
    AGB:UpdateTimestamp();

    ArmoryGuildBankFrame_Refresh();

    Armory:ExecuteDelayed(ARMORY_GUILDBANK_PUSH_DELAY, ArmoryGuildBankFrame_PushInfo);

    return Orig_CloseGuildBankFrame(...);
end

local Orig_ArmoryChatCommand = Armory.ChatCommand;
function Armory:ChatCommand(msg)
    local args = self:String2Table(msg);
    local command;
    if ( args and args[1] ) then
        command = strlower(args[1]);
    end
    if ( command == "gb" or command == "guildbank" ) then
        ArmoryGuildBankFrame_Toggle();
    else
        Orig_ArmoryChatCommand(self, msg);
    end
end

local Orig_ArmoryMenu_Initialize = ArmoryMenu_Initialize;
function ArmoryMenu_Initialize()
    Orig_ArmoryMenu_Initialize();
    ArmoryMenu_AddButton("ARMORY_CMD_GUILDBANK");
end

local Orig_ArmoryGetItemCount = Armory.GetItemCount;
function Armory:GetItemCount(link)
    local list = Orig_ArmoryGetItemCount(self, link);

    if ( AGB:GetConfigShowItemCount() ) then
        local currentRealm, currentGuild = AGB.selectedRealm, AGB.selectedGuild;
        local _, itemId = Armory:GetItemLinkInfo(link);
        itemId = AGB:NormalizeItemString(itemId);
        for realm, guilds in pairs(AgbDB) do
            if ( realm == Armory:GetPaperDollLastViewed() ) then
                for guild in pairs(guilds) do
                    local dbEntry = AGB:SelectDb(realm, guild);
                    local count = 0;
                    local itemCount;
                    for i = 1, MAX_GUILDBANK_TABS do
                        itemCount = AGB:GetItemCount(dbEntry, i, itemId);
                        if ( itemCount ) then
                            count = count + itemCount;
                        end
                    end
                    if ( count > 0 ) then
                        table.insert(list, {name=GUILD_BANK.." "..guild, count=count, details=""});
                    end
                end
            end
        end
        AGB:SelectDb(currentRealm, currentGuild);
    end
    
    return list;
end

local Orig_ContainerFrameItemButton_OnModifiedClick = ContainerFrameItemButton_OnModifiedClick;
function ContainerFrameItemButton_OnModifiedClick(button, ...)
    local bag = this:GetParent():GetID();
    local slot = this:GetID();

    ArmoryGuildBankFramePasteItem(button, GetContainerItemLink(bag, slot));

    return Orig_ContainerFrameItemButton_OnModifiedClick(button, ...);
end

local Orig_ChatFrame_OnHyperlinkShow = ChatFrame_OnHyperlinkShow;
function ChatFrame_OnHyperlinkShow(link, text, button, ...)
    ArmoryGuildBankFramePasteItem(button, link);

    return Orig_ChatFrame_OnHyperlinkShow(link, text, button, ...);
end


----------------------------------------------------------
-- Messaging
----------------------------------------------------------

function ArmoryGuildBankFrame_PushInfo()
    local dbEntry = AGB:Db();

    if ( not Armory:HasDataSharing() ) then
        AGB:PrintDebug("Sharing disabled");
        return;

    elseif ( dbEntry and ArmoryAddonMessageFrame_CanSend(true) ) then
        local money = AGB:GetMoney(dbEntry);
        local tabs = {};
        local name, timestamp, checksum;

        for i = 1, MAX_GUILDBANK_TABS do
            name = AGB:GetTabName(dbEntry, i);
            if ( name ) then
                name = AGB:Checksum(name);
                timestamp = AGB:GetTabTimestamp(dbEntry, i);
                checksum  = AGB:GetTabChecksum(dbEntry, i);
                table.insert(tabs, strjoin(ARMORY_LOOKUP_CONTENT_SEPARATOR, i, name, timestamp, checksum));
            end
        end

        timestamp = AGB:GetTimestamp(dbEntry);

        if ( timestamp ) then
            local id = ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_PUSH;
            local version = "3";
            local message = strjoin(ARMORY_LOOKUP_SEPARATOR, money, table.concat(tabs, ARMORY_LOOKUP_FIELD_SEPARATOR), timestamp, time());

            AGB:PrintDebug("Pushing", #tabs, "tab(s)");
            ArmoryAddonMessageFrame_Send(id, version, message, "GUILD", -1);

            AGB.pushed = ArmoryAddonMessageFrame_GetModule(id).msgno;
        end
    end
end

function ArmoryGuildBankFrame_PushReceived(version, message, msgNumber, sender)
    local dbEntry = AGB:Db();

    if ( sender == UnitName("player") ) then
        AGB:PrintDebug("Push echo received of message #", msgNumber, "(waiting for #", AGB.pushed..")");
        AGB.serverLag = (msgNumber < AGB.pushed);

    elseif ( AGB.serverLag ) then
        AGB:PrintDebug("Ignore push from", sender, "because I'm not sync with server message queue");

    elseif ( version ~= "3" ) then
        AGB:PrintDebug("Ignore push from", sender, "wrong protocol version", version);

    elseif ( dbEntry ) then
        local money, tabs, timestamp, remoteTime = strsplit(ARMORY_LOOKUP_SEPARATOR, message);
        local offset = AGB:CreateTimestamp() - AGB:CreateTimestamp(remoteTime);

        AGB:PrintDebug("Push received from", sender, "time offset", offset);

        -- if general info is newer, just save it
        timestamp = tonumber(timestamp) + offset;
        if ( timestamp > (AGB:GetTimestamp(dbEntry) or 0) ) then
            AGB:PrintDebug("Updating money", money, "timestamp", date("%x %H:%M", timestamp) );
            AGB:UpdateMoney(tonumber(money));
            AGB:UpdateTimestamp(timestamp);
        end

        -- get a list of outdated tabs
        local outdatedTabs = {};
        local tab, hash, checksum, name;
        local dbTime, dbHash;
        local push;
        for _, tabInfo in ipairs(Armory:StringSplit(ARMORY_LOOKUP_FIELD_SEPARATOR, tabs)) do
            tab, hash, timestamp, checksum = strsplit(ARMORY_LOOKUP_CONTENT_SEPARATOR, tabInfo);
            if ( timestamp ) then
                timestamp = tonumber(timestamp) + offset;

                if ( ARMORY_GUILDBANK_QUEUE[tab] ) then
                    dbTime, dbHash = unpack(ARMORY_GUILDBANK_QUEUE[tab]);
                    AGB:PrintDebug("Using timestamp and hash from queue for tab", tab);
                else
                    dbTime = AGB:GetTabTimestamp(dbEntry, tab);
                    dbHash = AGB:GetTabChecksum(dbEntry, tab);
                end

                if ( dbTime ) then
                    if ( dbTime > time() ) then
                        AGB:PrintDebug("Wrong db time, clearing time for tab", tab);
                        AGB:UpdateTimestamp(0, tab);

                    elseif ( timestamp > dbTime ) then
                        name = AGB:GetTabName(dbEntry, tab);
                        if ( AGB:Checksum(name) ~= hash ) then
                            AGB:PrintDebug("Wrong hash", hash, ", deleting tab", tab, name);
                            AGB:DeleteTab(tab);
                            ArmoryGuildBankFrame_Refresh();

                        elseif ( dbHash ~= checksum ) then
                            AGB:PrintDebug("Items of tab", tab, "are outdated");
                            AGB.awaitingUpdate = true;
                            table.insert(outdatedTabs, {tab, name, timestamp, checksum});

                        else
                            AGB:PrintDebug("Adjust time of tab", tab);
                            AGB:UpdateTimestamp(timestamp, tab);

                        end

                    elseif ( dbTime > timestamp ) then
                        AGB:PrintDebug("My tab", tab, "is newer than", date("%x %H:%M", timestamp), "(invoke push if no outdated)" );
                        push = true;
                        
                    end 
                end
            end
        end

        if ( #outdatedTabs > 0 ) then
            if ( not AGB.syncLock ) then

                -- keep the sync info for some random period of time before asking for an update
                local sync = ARMORY_GUILDBANK_SYNC_TABLE;
                for i = 1, #outdatedTabs do
                    tab, name, timestamp, checksum = unpack(outdatedTabs[i]);
                    if ( not sync[tab] ) then
                        sync[tab] = {};
                    end
                    if ( (not sync[tab].timestamp) or timestamp > sync[tab].timestamp ) then
                        AGB:PrintDebug("Adding tab", tab, "to sync table, timestamp", date("%x %H:%M", tonumber(timestamp)), "member to query", sender );
                        sync[tab].name = name;
                        sync[tab].timestamp = timestamp;
                        sync[tab].checksum = checksum;
                        sync[tab].members = {{sender, version}};

                    elseif ( sync[tab].timestamp == timestamp ) then
                        local found;
                        for _, memberInfo in ipairs(sync[tab].members) do
                            if ( memberInfo[1] == sender ) then
                                found = true;
                                break;
                            end
                        end

                        if ( not found ) then
                            AGB:PrintDebug("Adding", sender, "to members to query for tab", tab);
                            table.insert(sync[tab].members, {sender, version});
                        else
                            AGB:PrintDebug(sender, "has already been added to be queried for tab", tab);
                        end
                    end
                end

                -- note: will not overwrite if one is scheduled already
                local delay = random(ARMORY_GUILDBANK_SYNC_DELAY_MIN, ARMORY_GUILDBANK_SYNC_DELAY_MAX);
                AGB:PrintDebug("Processing sync table in", delay, "seconds (only first scheduled call will be invoked)");
                Armory:ExecuteDelayed(delay, ArmoryGuildBankFrame_ProcessSyncTable);
            else
                AGB:PrintDebug("Ignore, sync table locked");
            end

        elseif ( AGB.awaitingUpdate ) then
            AGB:PrintDebug("Ignore, waiting to get updated myself");

        elseif ( push ) then
            -- join the broadcast to spread the load if we are current
            Armory:ExecuteDelayed(ARMORY_GUILDBANK_PUSH_DELAY, ArmoryGuildBankFrame_PushInfo);

        else
            AGB:PrintDebug("Ignore, I have neither outdated nor newer tabs or no tabs at all");

        end
    end
end

function ArmoryGuildBankFrame_ProcessSyncTable()
    if ( not ArmoryAddonMessageFrame_CanSend(true) ) then
        return;
    end

    AGB:PrintDebug("Processing sync table..." );

    local members = {};
    local target, index, tabInfo, version;

    AGB.syncLock = true;

    ARMORY_GUILDBANK_QUEUE = {};

    -- find random targets to whisper for an update
    for tab, info in pairs(ARMORY_GUILDBANK_SYNC_TABLE) do
        tabInfo = strjoin(ARMORY_LOOKUP_FIELD_SEPARATOR, tab, info.name);
        repeat
            index = random(1, #info.members);
            target, version = unpack(info.members[index]);
            table.remove(info.members, index);
            AGB:PrintDebug("Selected", target, "for tab", tab);

            if ( not AGB:IsOnline(target) ) then
                AGB:PrintDebug(target, " is offline");
                target = nil;
            end
        until ( target or #info.members == 0 )

        if ( target ) then
            if ( members[target] ) then
                table.insert(members[target].tabs, tabInfo);
            else
                members[target] = {version=version, tabs={tabInfo}};
            end

            ARMORY_GUILDBANK_QUEUE[tab] = {info.timestamp, info.checksum};
        end

        ARMORY_GUILDBANK_SYNC_TABLE[tab] = nil;
    end

    AGB.syncLock = false;

    local sent;
    for target, info in pairs(members) do
        local id = ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_REQUEST;
        local version = info.version;
        local message = table.concat(info.tabs, ARMORY_LOOKUP_SEPARATOR);
        
        AGB:PrintDebug("Sending request to", target, "for tab(s)", table.concat(info.tabs, " "):gsub("%c", " "));
        ArmoryAddonMessageFrame_CreateRequest(id, version, message, "TARGET:"..target);
        sent = true;
    end

    AGB.awaitingUpdate = sent;
end

function ArmoryGuildBankFrame_ProcessSyncRequest(version, message, msgNumber, sender)
    local dbEntry = AGB:Db();

    if ( version ~= "3" ) then
        Armory:PrintCommunication(string.format(ARMORY_LOOKUP_IGNORED, ARMORY_IGNORE_REASON_VERSION));

    elseif ( dbEntry ) then
        local tabs = Armory:StringSplit(ARMORY_LOOKUP_SEPARATOR, message);
        local list = {};

        AGB:PrintDebug("Request received from", sender);

        for i = 1, #tabs do
            local tab, name = strsplit(ARMORY_LOOKUP_FIELD_SEPARATOR, tabs[i]);

            if ( AGB:GetTabName(dbEntry, tab) == name ) then
                local timestamp = AGB:GetTabTimestamp(dbEntry, tab);
                local checksum = AGB:GetTabChecksum(dbEntry, tab);

                AGB:PrintDebug("Getting items for tab", tab, "timestamp", date("%x %H:%M", timestamp), "checksum", checksum);

                local values = {};
                for itemId, count in pairs(AGB:GetTabItems(dbEntry, tab)) do
                    table.insert(values, strjoin("=", itemId, count));
                end

                table.insert(list, strjoin(ARMORY_LOOKUP_FIELD_SEPARATOR, tab, timestamp, checksum, table.concat(values, ARMORY_LOOKUP_CONTENT_SEPARATOR))); 

                Armory:PrintCommunication(string.format(ARMORY_LOOKUP_REQUEST_DETAIL, GUILDBANK_TAB_COLON.." "..tab));
            else
                AGB:PrintDebug("Ignore, requested tab", name, "not found");
            end
        end

        if ( #list > 0 ) then
            local id = ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_REQUEST;
            local message = time()..ARMORY_LOOKUP_SEPARATOR..table.concat(list, ARMORY_LOOKUP_SEPARATOR);

            ArmoryAddonMessageFrame_Send(id, version, message, "TARGET:"..sender, 0);
            Armory:PrintCommunication(string.format(ARMORY_LOOKUP_RESPONSE_SENT, sender));
        end
    end
end

function ArmoryGuildBankFrame_CheckResponse()
    -- requests can be sent to multiple members, so wait for more replies
    if ( ArmoryAddonMessageFrame_GetModule(ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_REQUEST).numReplies > 0 ) then
        -- note: does nothing if already scheduled for execution
        Armory:ExecuteDelayed(ARMORY_GUILDBANK_QUEUE_WAIT_TIME, ArmoryGuildBankFrame_ProcessResponse);
    end
end

function ArmoryGuildBankFrame_ProcessResponse()
    local module = ArmoryAddonMessageFrame_GetModule(ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_REQUEST);
    local dbEntry = AGB:Db();
    local tabs, timestamp, checksum, values, itemId, count;
    local remoteTime, offset;
    local updated;

    AGB:PrintDebug("Processing responses");

    for sender, reply in pairs(module.replies) do
        if ( reply.version == "3" ) then
            tabs = Armory:StringSplit(ARMORY_LOOKUP_SEPARATOR, reply.message);

            remoteTime = tonumber(tabs[1]);
            table.remove(tabs, 1);

            offset = AGB:CreateTimestamp(reply.timestamp or remoteTime) - AGB:CreateTimestamp(remoteTime);
         
            AGB:PrintDebug("Processing", #tabs, "tab(s) from", sender);

            for i = 1, #tabs do
                tab, timestamp, checksum, values = strsplit(ARMORY_LOOKUP_FIELD_SEPARATOR, tabs[i]);
                timestamp = tonumber(timestamp) + offset;

                if ( AGB:GetTabName(dbEntry, tab) ) then
                    values = Armory:StringSplit(ARMORY_LOOKUP_CONTENT_SEPARATOR, values);

                    AGB:PrintDebug("Processing values for tab", tab);

                    local items = {};
                    for _, value in ipairs(values) do
                        itemId, count = strsplit("=", value);
                        items[itemId] = tonumber(count);
                    end

                    AGB:UpdateTabItems(tab, items, checksum, timestamp);
                    updated = true;
                else
                    AGB:PrintDebug("Ignore, tab", tab, "has been deleted");
                end
            end
        else
            AGB:PrintDebug("Wrong protocol version", reply.version);
        end
        ArmoryAddonMessageFrame_RemoveReply(module, sender);
    end

    ARMORY_GUILDBANK_QUEUE = {};

    AGB.awaitingUpdate = false;

    if ( updated ) then
        ArmoryGuildBankFrame_Refresh();
        Armory:ExecuteDelayed(ARMORY_GUILDBANK_PUSH_DELAY, ArmoryGuildBankFrame_PushInfo);
    end
end

function ArmoryGuildBankFrame_ProcessRequest(id, version, message, msgNumber, sender, channel)
    if ( id == ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_PUSH ) then
        ArmoryGuildBankFrame_PushReceived(version, message, msgNumber, sender);

    elseif ( id == ARMORY_GUILDBANK_MESSAGE_TYPE.BANK_SYNC_REQUEST ) then
        ArmoryGuildBankFrame_ProcessSyncRequest(version, message, msgNumber, sender);

    end
end

function ArmoryGuildBankFrame_RemoveFromQueue(tab)
    if ( not AGB.syncLock ) then
        ARMORY_GUILDBANK_QUEUE[tab] = nil;
        ARMORY_GUILDBANK_SYNC_TABLE[tab] = nil;
    end
end

function ArmoryGuildBankFramePasteItem(button, link)
    if ( not ArmoryGuildBankFrameEditBox:IsVisible() ) then
        return;
    elseif ( button == "LeftButton" and IsAltKeyDown() ) then
        local itemName = GetItemInfo(link);
        if ( itemName ) then
            ArmoryGuildBankFrameEditBox:SetText(itemName);
        end
    end
end

----------------------------------------------------------
-- AGB namespace
----------------------------------------------------------

function AGB:Db()
    local dbEntry;

    self.realm = GetRealmName();
    self.guild = GetGuildInfo("player");

    if ( self.realm and self.guild ) then
        if ( not AgbDB[self.realm] ) then
            AgbDB[self.realm] = {};
        end
        if ( not AgbDB[self.realm][self.guild] ) then
            AgbDB[self.realm][self.guild] = {};
        end
        dbEntry = ArmoryDbEntry:new(AgbDB[self.realm][self.guild]);
    end

    return dbEntry;
end

function AGB:SelectDb(realm, guild)
    local dbEntry;
    local refresh;

    if ( not realm ) then
        realm = self.selectedRealm or GetRealmName();
    end

    if ( not guild ) then
        guild = self.selectedGuild or GetGuildInfo("player");
        if ( not guild ) then
            local guilds = self:GuildList(realm);
            if ( #guilds > 0 ) then
                guild = guilds[1];
            else
                realm, guild = self:FirstGuild();
            end
        end
    end

    if ( realm and guild and AgbDB[realm] and AgbDB[realm][guild] ) then
        dbEntry = ArmoryDbEntry:new(AgbDB[realm][guild]);
        refresh = (self.selectedRealm ~= realm or self.selectedGuild ~= guild);

        self.selectedRealm = realm;
        self.selectedGuild = guild;
        self.selectedDbEntry = dbEntry;
    end

    return dbEntry, refresh;
end

function AGB:RealmList()
    local list = {};

    if ( AgbDB ) then
        for realm in pairs(AgbDB) do
            table.insert(list, realm);
        end
        table.sort(list);    
    end

    return list;
end

function AGB:GuildList(realm)
    local list = {};

    if ( realm and AgbDB and AgbDB[realm] ) then 
        for guild in pairs(AgbDB[realm]) do
            table.insert(list, guild);
        end
        table.sort(list);
    end

    return list;
end

function AGB:FirstGuild()
    local realm = GetRealmName();

    for _, guild in ipairs(self:GuildList(realm)) do
        if ( AgbDB[realm] and AgbDB[realm][guild] ) then
            return realm, guild;
        end
    end

    for _, realm in ipairs(self:RealmList()) do
        for _, guild in ipairs(self:GuildList(realm)) do
            if ( AgbDB[realm] and AgbDB[realm][guild] ) then
                return realm, guild;
            end
        end
    end
end

function AGB:UpdateTabName(tab, name)
    local dbEntry = self:Db();

    dbEntry:SetSubValue("Tab"..tab, "Name", name);
end

function AGB:UpdateTabItems(tab, items, checksum, timestamp)
    local dbEntry = self:Db();

    dbEntry:SetSubValue("Tab"..tab, "Items", items);
    dbEntry:SetSubValue("Tab"..tab, "Checksum", checksum);

    self:UpdateTimestamp(timestamp, tab);
end

function AGB:DeleteTab(tab)
    local dbEntry = self:Db();

    dbEntry:SetValue("Tab"..tab, nil);
end

function AGB:UpdateGuildInfo()
    local dbEntry = self:Db();

    dbEntry:SetValue("Tabard", GetGuildTabardFileNames());
    dbEntry:SetValue("Faction", UnitFactionGroup("player"));
end

function AGB:UpdateMoney(money)
    local dbEntry = self:Db();

    dbEntry:SetValue("Money", money or GetGuildBankMoney());
end

function AGB:UpdateTimestamp(timestamp, tab)
    local dbEntry = self:Db();

    timestamp = self:CreateTimestamp(timestamp);

    if ( tab == nil ) then
        dbEntry:SetValue("Time", timestamp);
    else
        dbEntry:SetSubValue("Tab"..tab, "Time", timestamp);
    end
end

function AGB:CreateTimestamp(timestamp)
    if ( type(timestamp) == "string" ) then
        timestamp = tonumber(timestamp);
    end

    -- precision in minutes
    return floor((timestamp or time()) / 60) * 60;
end

function AGB:GetValue(dbEntry, key)
    return dbEntry:GetValue(key);
end

function AGB:GetTabValue(dbEntry, tab, key)
    if ( dbEntry:Contains("Tab"..tab) ) then
        return dbEntry:GetSubValue("Tab"..tab, key);
    end
end

function AGB:TabExists(dbEntry, tab)
    return dbEntry:Contains("Tab"..tab);
end

function AGB:GetTimestamp(dbEntry)
    return self:GetValue(dbEntry, "Time");
end

function AGB:GetMoney(dbEntry)
    return self:GetValue(dbEntry, "Money");
end

function AGB:GetTabardFiles(dbEntry)
    return self:GetValue(dbEntry, "Tabard");
end

function AGB:GetFaction(dbEntry)
    return self:GetValue(dbEntry, "Faction");
end

function AGB:GetTabName(dbEntry, tab)
    return self:GetTabValue(dbEntry, tab, "Name");
end

function AGB:GetTabItems(dbEntry, tab)
    return self:GetTabValue(dbEntry, tab, "Items");
end

function AGB:GetTabTimestamp(dbEntry, tab)
    return self:GetTabValue(dbEntry, tab, "Time");
end

function AGB:GetTabChecksum(dbEntry, tab)
    return self:GetTabValue(dbEntry, tab, "Checksum");
end

function AGB:GetItemCount(dbEntry, tab, itemId)
    local items = self:GetTabItems(dbEntry, tab);
    if ( items ) then
        return items[itemId];
    end        
end

function AGB:GetItemInfo(ref)
    local name, link, texture;

    -- phase 1: try to get the info from the game tooltip
    Armory:PrepareTooltip();
    Armory.tooltip:SetHyperlink("item:"..ref);
    name, link = Armory.tooltip:GetItem();
    Armory:ReleaseTooltip();

    -- phase 2: try to get the name from game tooltip link 
    if ( link and (name or "") == "" ) then
        name = Armory:GetNameFromLink(link);
    end

    -- phase 3: try GetItemInfo 
    if ( not link and (name or "") == "" ) then
        name, link = GetItemInfo(ref:match("^([-%d]+)"));
    end

    -- almost the same for the icon
    texture = GetItemIcon(link);
    if ( not texture ) then
        _, _, _, _, _, _, _, _, _, texture = GetItemInfo(link);
    end

    return name, link, texture;
end

function AGB:SetFilter(text)
    local refresh = (self.filter ~= text);

    self.filter = text;

    return refresh;
end

function AGB:UpdateItemLines()
    local dbEntry = self.selectedDbEntry;
    local include;
    
    self.itemLines = {};

    if ( dbEntry ) then
        local name, count, link, texture;
        for i = 1, MAX_GUILDBANK_TABS do
            if ( self:TabExists(dbEntry, i) ) then
                name = string.format(GUILDBANK_TAB_NUMBER, i).." "..self:GetTabName(dbEntry, i)..", "..date("%x %H:%M", self:GetTabTimestamp(dbEntry, i));
                table.insert(self.itemLines, {name, 1});

                local items = {};
                for itemId, count in pairs(self:GetTabItems(dbEntry, i)) do
                    name, link, texture = self:GetItemInfo(itemId);
                    if ( (self.filter or "") == "" ) then
                        include = true;
                    else
                        include = string.find(strlower(name), strlower(self.filter), 1, true);
                    end
                    if ( include and ArmoryItemFilter(link) ) then
                        table.insert(items, {name, nil, count, texture, link});
                    end
                end
                table.sort(items, function(a, b) return a[1] < b[1]; end);

                for _, v in ipairs(items) do
                    table.insert(self.itemLines, v);
                end
            end
        end
    end
end

function AGB:IsOnline(member)
    local name;

    GuildRoster();

    for i = 1, GetNumGuildMembers() do
        name = GetGuildRosterInfo(i);
        if ( member == name ) then
            return true;
        end
    end
end

function AGB:Checksum(text)
    local a = 1;
    for i = 1, text:len(), 3 do
        a = (a * 8257 % 16777259) + text:byte(i) + ((text:byte(i + 1) or 1) * 127) + ((text:byte(i + 2) or 2) * 16383);
    end
    return string.format("%x", a % 16777213);
end

function AGB:NormalizeItemString(itemString)
    -- itemId:enchantId:jewelId1:jewelId2:jewelId3:jewelId4:suffixId:uniqueId
    if ( itemString ) then
        ids = Armory:StringSplit(":", itemString);
        if ( #ids >= 8 ) then
            ids[8] = "0";
        end
        itemString = table.concat(ids, ":");
    end
    return itemString
end

function AGB:PrintDebug(...)
    Armory:PrintDebug("AGB"..FONT_COLOR_CODE_CLOSE, ...);
end

function AGB:Find(firstArg, ...)
    local currentRealm, currentGuild = self.selectedRealm, self.selectedGuild;
    local list = {};

    if ( AGB:GetConfigIncludeInFind() ) then
        for realm, guilds in pairs(AgbDB) do
            for guild in pairs(guilds) do
                local dbEntry = self:SelectDb(realm, guild);
                for i = 1, MAX_GUILDBANK_TABS do
                    if ( self:TabExists(dbEntry, i) ) then
                        container = self:GetTabName(dbEntry, i);
                        for id, itemCount in pairs(self:GetTabItems(dbEntry, i)) do
                            name, link = self:GetItemInfo(id);
                            if ( Armory:FindNameParts(name, firstArg, ...) ) then
                                if ( itemCount > 1 ) then
                                    table.insert(list, {realm=realm, guild=guild, label=container, value=(link or name).."x"..itemCount});
                                else
                                    table.insert(list, {realm=realm, guild=guild, label=container, value=(link or name)});
                                end
                            end
                        end
                    end
                end
            end
        end

        self:SelectDb(currentRealm, currentGuild);
    end
    
    return list;
end

function AGB:SetConfigShowItemCount(on)
    Armory:Setting("General", "HideAgbItemCount", not on);
end

function AGB:GetConfigShowItemCount()
    return not Armory:Setting("General", "HideAgbItemCount");
end

function AGB:SetConfigIncludeInFind(on)
    Armory:Setting("General", "ExcludeAgbFind", not on);
end

function AGB:GetConfigIncludeInFind()
    return not Armory:Setting("General", "ExcludeAgbFind");
end
