--[[
Finds enemies in Arena as soon as possible to prevent Gladdy from greying out and making it keybinds available at all times

TO DO:
- Find enemies even if never seen before, only important for Druids, Rogues + Mages
	- Try to find specifics from Combatlog
	- e.g. if casting Regrowth = druid
- let the user call RequestBattlefieldScoreData() by hand instead of when the doors open
- install menu that lets the user enter enemies (no GUID?) by hand
	- if no GUID, gladdy will add enemies twice, but "name" clicks work
- function that inserts your target in a certain bracket (2v2, 3v3)	

]]

ArenaIdentify = LibStub("AceAddon-3.0"):NewAddon("ArenaIdentify", "AceEvent-3.0")

local instanceType, bracket
local scanTable = {}
local alreadyLoaded = {}
local alreadyFound = {}
local alreadySaved = {}
local guidByName = {}

function ArenaIdentify:GuidInParty(guid)
	local inParty = 0
	for i=0, GetNumPartyMembers() do
		if UnitGUID("party"..i) == guid then
			inParty = 1
		end
	end
	return inParty
end


function ArenaIdentify:OnInitialize()
	-- Defaults
	self.defaults = {
		profile = {
			cutOff = 0,
		}
	}

	self.db = LibStub:GetLibrary("AceDB-3.0"):New("ArenaIdentifyDB", self.defaults)
	
	-- Upgrade data
	if( ArenaIdentifyData and not ArenaIdentifyData[2] and not ArenaIdentifyData[3] and not ArenaIdentifyData[5] ) then
		ArenaIdentifyData = nil
	end

	if( not ArenaIdentifyData ) then
		ArenaIdentifyData = {[2] = {}, [3] = {}, [5] = {}}
	end
end

function ArenaIdentify:OnEnable()
	self:RegisterEvent("ZONE_CHANGED_NEW_AREA")
	self:RegisterEvent("PLAYER_ENTERING_WORLD", "ZONE_CHANGED_NEW_AREA")
	
	self:ZONE_CHANGED_NEW_AREA()
end

function ArenaIdentify:OnDisable()
	self:UnregisterAllEvents()
end

function ArenaIdentify:Print(msg)
	DEFAULT_CHAT_FRAME:AddMessage("|cff33ff99ArenaIdentify|r: " .. msg)
end

function ArenaIdentify:ScanUnits()
	if( not self.tooltip ) then
		self.tooltip = CreateFrame("GameTooltip", "ArenaIdentifyTooltip", UIParent, "GameTooltipTemplate")
		self.tooltip:SetOwner(UIParent, "ANCHOR_NONE")
	end
		
	for _, data in pairs(scanTable) do
		if( not alreadyFound[data.guid] and instanceType == "arena" ) then
			self.tooltip:SetOwner(UIParent, "ANCHOR_PRESERVE")
			self.tooltip:SetHyperlink(string.format("unit:%s", data.guid))
			
			
			local name, unitid = self.tooltip:GetUnit()
			self.tooltip:Hide()
			
			if( name and not UnitInParty(name) and self:GuidInParty(data.guid)==0) then
				alreadyFound[data.guid] = true
				
				-- Send data to Gladdy
					--v.name, v.guid, v.class, v.classLoc, v.raceLoc, v.spec, v.health, v.healthMax, v.power, v.powerMax, v.powerType
					local msg = string.format("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s", data.name, data.guid, data.class, data.classLoc, data.raceLoc, "", data.health, data.healthMax, data.power, data.powerMax, data.powerType)
					SendAddonMessage("Gladdy", msg, "PARTY")
					SendAddonMessage("Gladdy", msg, "WHISPER", UnitName("player"))
			end
		end
	end
end

-- Get enemy/team mate races
function ArenaIdentify:PLAYER_TARGET_CHANGED()
	self:ScanUnit("target")
end

function ArenaIdentify:UPDATE_MOUSEOVER_UNIT()
	self:ScanUnit("mouseover")
end

function ArenaIdentify:PLAYER_FOCUS_CHANGED()
	self:ScanUnit("focus")
end

function ArenaIdentify:ScanUnit(unit)
	local guid = UnitGUID(unit)
	if( not alreadySaved[guid] and UnitIsPlayer(unit) and UnitIsEnemy("player", unit) and not UnitIsCharmed(unit) and not UnitIsCharmed("player")) then
		local name, _ = UnitName(unit)
		local classLoc, class = UnitClass(unit)
		local powerType = UnitPowerType(unit) or 0
		local raceLoc, _ = UnitRace(unit)
		--v.name, v.guid, v.class, v.classLoc, v.raceLoc, v.spec, v.health, v.healthMax, v.power, v.powerMax, v.powerType
		ArenaIdentifyData[bracket][guid] = string.format("%d,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s", time(), name, guid, class, classLoc, raceLoc, "", "100", "100", "100", "100", powerType)
		alreadySaved[guid] = true
	end
end

--[[ Check if match has started -- this is blizzlike, however some servers let you target players and read the combatlog before the arena starts]]
function ArenaIdentify:CHAT_MSG_BG_SYSTEM_NEUTRAL(event, msg)
	if( msg == "The Arena battle has begun!" ) then
		-- new request to catch late joiners
		-- cannot request on every update for some reason
		RequestBattlefieldScoreData()
	end
end

local function sortSeen(a, b)
	return a.time > b.time
end

-- Are we inside an arena?
local timeElapsed = 0
function ArenaIdentify:ZONE_CHANGED_NEW_AREA()
	local type = select(2, IsInInstance())
	-- Inside an arena, but wasn't already
	if( type == "arena" and type ~= instanceType --[[and select(2, IsActiveBattlefieldArena())]] ) then
		--self:RegisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL")
		self:RegisterEvent("PLAYER_FOCUS_CHANGED")
		self:RegisterEvent("PLAYER_TARGET_CHANGED")
		self:RegisterEvent("UPDATE_MOUSEOVER_UNIT")
		self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
		self:RegisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL")
		
		-- Set our prune threshold
		local pruneTime
		if( self.db.profile.cutOff > 0 ) then
			pruneTime = time() - ((60 * 60 * 24) * self.db.profile.cutOff)
		end
		
		-- Figure out the arena bracket
		bracket = 2
		for i=1, MAX_BATTLEFIELD_QUEUES do
			local status, _, _, _, _, teamSize = GetBattlefieldStatus(i)
			if( status == "active" and teamSize > 0 ) then
				bracket = teamSize
				break
			end
		end
		
		-- Load it into a table so we can sort it
		for guid, data in pairs(ArenaIdentifyData[bracket]) do
			if( not alreadyLoaded[guid] ) then
				alreadyLoaded[guid] = true

				local time, name, guid, class, classLoc, raceLoc, spec, health, healthMax, power, powerMax, powerType = string.split(",", data)
				time = tonumber(time) or 99999999999999
				
				if( not pruneTime or pruneTime <= time ) then
					table.insert(scanTable, {guid = guid, time = time, name = name, class = class, classLoc = classLoc, raceLoc = raceLoc, health = health, healthMax = healthMax, power = power, powerMax = powerMax,powerType = powerType or 0})
					guidByName[name] = guid
				else
					ArenaIdentifyData[bracket][guid] = nil
				end
			end
		end
		
		table.sort(scanTable, sortSeen)
		

	-- Was in an arena, but left it
	elseif( type ~= "arena" and instanceType == "arena" ) then
		self:UnregisterEvent("PLAYER_FOCUS_CHANGED")
		self:UnregisterEvent("PLAYER_TARGET_CHANGED")
		self:UnregisterEvent("UPDATE_MOUSEOVER_UNIT")
		self:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
		self:UnregisterEvent("CHAT_MSG_BG_SYSTEM_NEUTRAL")
		
		for k in pairs(alreadyFound) do alreadyFound[k] = nil end
		for k in pairs(alreadySaved) do alreadySaved[k] = nil end
		
		if( self.frame ) then
			self.frame:Hide()
		end
	end
	
	instanceType = type
	if( not self.frame ) then
			self.frame = CreateFrame("Frame")
			self.frame:Hide()
			self.frame:SetScript("OnUpdate", function(self, elapsed)
				timeElapsed = timeElapsed + elapsed
				if( timeElapsed >= 1 ) then
					if( InCombatLockdown() ) then
						self:UnregisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
						self:Hide()
						return
					end
					
					--ArenaIdentify:ScanUnits()
					ArenaIdentify:Scoreboard()
					timeElapsed = 0
				end
			end)
		end
		
		timeElapsed = 1
		self.frame:Show()
end

function ArenaIdentify:COMBAT_LOG_EVENT_UNFILTERED(event, timestamp, eventType, sourceGUID,sourceName,sourceFlags,destGUID,destName,destFlags,spellID,spellName, ...)
		if( not alreadyFound[sourceGUID] and instanceType == "arena" and self:GuidInParty(sourceGUID)==0) then
				local data = ArenaIdentifyData[bracket][sourceGUID]
				if data ~= nil then
					local time, name, guid, class, classLoc, raceLoc, spec, health, healthMax, power, powerMax, powerType = string.split(",", data)
					alreadyFound[sourceGUID] = true 
					--v.name, v.guid, v.class, v.classLoc, v.raceLoc, v.spec, v.health, v.healthMax, v.power, v.powerMax, v.powerType
					local msg = string.format("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s", name, guid, class, classLoc, raceLoc, "", health, healthMax, power, powerMax, powerType)
					SendAddonMessage("Gladdy", msg, "PARTY")
					SendAddonMessage("Gladdy", msg, "WHISPER", UnitName("player"))
				end		
		elseif( not alreadyFound[destGUID] and instanceType == "arena" and self:GuidInParty(destGUID)==0) then
				local data = ArenaIdentifyData[bracket][destGUID]
				if data ~= nil then
					local time, name, guid, class, classLoc, raceLoc, spec, health, healthMax, power, powerMax, powerType = string.split(",", data)
					alreadyFound[destGUID] = true 
					--v.name, v.guid, v.class, v.classLoc, v.raceLoc, v.spec, v.health, v.healthMax, v.power, v.powerMax, v.powerType
					local msg = string.format("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s", name, guid, class, classLoc, raceLoc, "", health, healthMax, power, powerMax, powerType)
					SendAddonMessage("Gladdy", msg, "PARTY")
					SendAddonMessage("Gladdy", msg, "WHISPER", UnitName("player"))
				end	
		end
end

-- faction 1 = Alliance
-- faction 0 = Horde
function ArenaIdentify:Scoreboard()
	-- find player faction
	if bracket ~=nil then
		--self:Print(playerFaction)
		for i=1, bracket*2+1 do
			local name, killingBlows, honorKills, deaths, honorGained, faction, rank, raceLoc, classLoc, class, damageDone, healingDone = GetBattlefieldScore(i);
			if not name then return end
			local guid = guidByName[name]
			if not alreadyFound[guid] and instanceType == "arena" and self:GuidInParty(guid)==0 and UnitGUID("player") ~= guid then
				alreadyFound[guid] = true
				local msg = string.format("%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s", name, guid, class, classLoc, raceLoc, "", "100", "100", "100", "100", "0")
				SendAddonMessage("Gladdy", msg, "PARTY")
				SendAddonMessage("Gladdy", msg, "WHISPER", UnitName("player"))
			end
		end
	end	
end

function ArenaIdentify:FindPlayerFaction()
	for i=1, bracket*2+1 do
		local name, _, _, _, _, faction =  GetBattlefieldScore(i)
		if name == UnitName("player") then
			return faction
		end
	end
end


SLASH_ARENAIDENTIFY1 = "/arenaidentify"
SLASH_ARENAIDENTIFY2 = "/ai"
SlashCmdList["ARENAIDENTIFY"] = function(msg)
	cmd, arg = string.split(" ", string.trim(msg or ""))
	if( not cmd and not arg ) then
		cmd = msg
	end
	
	cmd = string.lower(cmd)
	arg = arg or ""
	
	if( cmd == "prune" ) then
		arg = tonumber(arg)
		
		ArenaIdentify.db.profile.cutOff = arg
		ArenaIdentify:Print(string.format("Only saving people who you have seen within the last %d days now.", arg))
	else
		DEFAULT_CHAT_FRAME:AddMessage("/ai prune <days> - Set how many days players should be saved before being removed, use 0 to disable.")
	end
end