﻿--[[

AddonMan.lua
Version 1.1
Author Todd Miller

Changlog
08/10/2008 - Version 1.1 Changes
   - Addon Man now automatically detects addons and allows enabling and 
      disabling of them without extra work from the addon's developer, 
      and also gives a message if the addon isn't loadable for some reason
      (i.e. the interface version is out of date).
   - Addon developers can still register options with Addon Man, but this
      does require a simple function call for each option.
   - Fixed frame level bugs ugh.
   - Created GameFontBlueSmall. I mention this only because it isn't
      specific to Addon Man - I just created a new global GameFontXSmall
   - Changed a lot of XML names to be more Addon Man specific.

08/07/2008 - initial release

]]--

local DEBUG = false

local ADDONMAN_TITLE_MAX = 12
local ADDONMAN_MAINTEXT_Y = 19
local ADDONMAN_MAIN_X = 90
local ADDONMAN_LOCALIZED_X = 90
local ADDONMAN_OPTIONSTEXT_Y = 15

local ADDONMAN_DISABLED_FONT = "GameFontDisableSmall"
local ADDONMAN_ENABLED_FONT = "GameFontHighlightSmall"
local ADDONMAN_MAINTITLE_FONT = "GameFontBlueSmall"
local ADDONMAN_TITLE_FONT = "GameFontNormalSmall"
local ADDONMAN_MOUSEOVER_FONT = "GameFontGreenSmall"
local ADDONMAN_NOTLOADED_FONT = "GameFontRedSmall"

-- used for handling multiple main menu listings
local NUM_MAINFRAMES = 0
local MAINMENU_MAXADDONS = 15

local ActiveAddons = {}
local ActiveOptions = {}
local MainMenuFrames = {}

-- store the frame manager function for hookin'
local origUIParent_ManageFramePositions = UIParent_ManageFramePositions

-----------
-- initalization function
-----------
function AddonMan_Initialize(self)

   for i=1, GetNumAddOns() do
      local name, title, _, enabled, loadable, reason = GetAddOnInfo(i)
      
      if (DEBUG==true) then
         if (loadable == 1) then
            ChatFrame1:AddMessage(name.." - "..title.." - "..loadable)
         else
            ChatFrame1:AddMessage(name.." - "..title.." - "..reason)
         end
      end
      
      -- don't list myself
      if not (name == "AddonMan") then
         -- AddonMan_RegisterWithManager(title, hasOptions, funcEnable, funcDisable)
         AddonMan_RegisterWithManager(name, enabled, loadable, reason)
       end
   end

   -- change the frame width for different localizations since words are longer in other languages
   if (GetLocale() == "frFR") then
      AddonManFrame:SetWidth(100)
      AddonManFrameButton:SetWidth(100)
      ADDONMAN_LOCALIZED_X = 100
   end
end

-----------
-- external addons call this function to register an option with Addon Man
--
-- addon - (string) name of the addon this option applies to
-- text - (string) text of the option
-- isTitle - (boolean) whether or not this option is a title, and has no checkbox
-- isEnabled - (boolean) whether or not this option should start out enabled
-- toggleFunc - (function) the function to call when this option is switched
-----------
function AddonMan_RegisterOption(addon, text, isTitle, isEnabled, toggleFunc)
   local NUM_OPTIONS = #ActiveOptions + 1

   -- pretty straight forward, just store all the variables into a new table entry
   ActiveOptions[NUM_OPTIONS] = {}
   ActiveOptions[NUM_OPTIONS].addon = addon
   ActiveOptions[NUM_OPTIONS].text = text
   ActiveOptions[NUM_OPTIONS].isTitle = isTitle
   ActiveOptions[NUM_OPTIONS].isEnabled = isEnabled
   ActiveOptions[NUM_OPTIONS].toggleFunc = toggleFunc
   
   -- find this addon in the table
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.title == addon) then
      
         if (DEBUG==true) then
            ChatFrame1:AddMessage("Registering option for "..curAddon.title)
         end

         curAddon.totalOptions = curAddon.totalOptions + 1
      
         -- if this is the first option for this addon
         if (curAddon.hasOptions == false) then
            curAddon.hasOptions = true
            
            -- note i don't test which font to use because i know the addon is enabled, otherwise we wouldn't be registering options
            curAddon.optionsArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
            getglobal(curAddon.button:GetName().."OptionsArrow"):Show()
            
            -- build the frame
            local newFrameString = curAddon.optionsFrame:CreateFontString(nil, "OVERLAY", ADDONMAN_MAINTITLE_FONT)
            newFrameString:SetPoint("LEFT", curAddon.optionsFrameName.."Buttons", "LEFT", 0, 0)
            newFrameString:SetText(curAddon.display.." "..ADDONMAN_LOCALIZATION_OPTIONS)
            
         end
         
         -- create a string without a checkbox
         if (isTitle == true) then
            
            local newStringName = addon.."OptionTitle"..curAddon.totalOptions
            local newFrameString = curAddon.optionsFrame:CreateFontString(newStringName, "OVERLAY", ADDONMAN_TITLE_FONT)
            newFrameString:SetPoint("LEFT", curAddon.optionsFrameName.."Buttons", "LEFT", 4, -(curAddon.totalOptions * ADDONMAN_OPTIONSTEXT_Y))
            newFrameString:SetText(text)
            
            -- store the new information in the table entry
            ActiveOptions[NUM_OPTIONS].titleString = newFrameString
            ActiveOptions[NUM_OPTIONS].button = nil
            ActiveOptions[NUM_OPTIONS].check = nil
            
         -- create a button with a checkbox
         else
         
            local newFrameName = addon.."OptionButton"..curAddon.totalOptions
            local newFrame = CreateFrame("Frame", newFrameName, curAddon.optionsFrame, "AddonManOptionsButton")
            local newFrameButton = getglobal(newFrame:GetName().."Button")
            local newFrameCheck = getglobal(newFrame:GetName().."Check")
            newFrame:SetPoint("LEFT", curAddon.optionsFrameName.."Buttons", "LEFT", 0, -(curAddon.totalOptions * ADDONMAN_OPTIONSTEXT_Y))
            newFrameButton:SetText(text)
            newFrameButton:SetScript("OnClick", AddonMan_OptionsOnClick)
            
            -- set the check if the option starts out enabled
            if (isEnabled == true) then
               newFrameCheck:Show()
            else
               newFrameCheck:Hide()
            end
            
            -- store the new information in the table etnry
            ActiveOptions[NUM_OPTIONS].titleString = nil
            ActiveOptions[NUM_OPTIONS].button = newFrameButton
            ActiveOptions[NUM_OPTIONS].check = newFrameCheck
            
         end
         
         -- extend the height of this addon's options frame
         local newY = (ADDONMAN_OPTIONSTEXT_Y * (curAddon.totalOptions+1)) + curAddon.totalOptions + 4
         curAddon.optionsFrame:SetHeight(newY)
         
      end
   end
end

-----------
-- basically this function builds and/or adds to the main menu. it is called for each addon detected
--
-- name - (string) name of the addon
-- enabled - (boolean) whether or not the addon is enabled
-- loadable - (1/nil) - whether or not the addon is load on demand
-- reason - (string/nil) - the reason why the addon is not loadable
-----------
function AddonMan_RegisterWithManager(name, enabled, loadable, reason)

   if (DEBUG==true) then
      ChatFrame1:AddMessage(#ActiveAddons.." "..name)
   end
   
   local NUM_ADDONS = #ActiveAddons + 1
   
   -- shorten the title if it is above the max chars
   local display
   if (string.len(name) > ADDONMAN_TITLE_MAX) then
      display = string.sub(name, 1, ADDONMAN_TITLE_MAX)
   else
      display = name
   end
   
   -- builds a new main menu frame for every MAXADDONS addons
   -- basically keeps the list of addons to a certain length and creates a new frame to the left of the previous main menu
   -- to handle more addons
   if ((#ActiveAddons % MAINMENU_MAXADDONS) == 0) then
      local NUM_MAINFRAMES = #MainMenuFrames + 1
      
      -- create the frame
      local newMainFrameName = "AddonManMainMenuFrame"..NUM_MAINFRAMES
      local newMainFrame = CreateFrame("Frame", newMainFrameName, UIParent, "AddonManMainMenuFrame")
      
      -- position the frame based on whether its the first one or not
      if (NUM_MAINFRAMES == 1) then
         newMainFrame:SetPoint("TOPRIGHT", AddonManFrame, "TOPLEFT", 10, -5)
      else
         newMainFrame:SetPoint("TOPRIGHT", MainMenuFrames[#MainMenuFrames], "TOPLEFT", 0, 0)
      end
      
      -- insert the frame into UISpecialFrames so that when the user pushes ESC, it will close
      table.insert(UISpecialFrames, newMainFrameName)
      
      -- add info to the table
      MainMenuFrames[NUM_MAINFRAMES] = {}
      MainMenuFrames[NUM_MAINFRAMES] = newMainFrame
   end
   
   -- extend the height of the main menu frame
   local newY = MainMenuFrames[#MainMenuFrames]:GetHeight() + ADDONMAN_MAINTEXT_Y
   MainMenuFrames[#MainMenuFrames]:SetHeight(newY)
   
   -- create a new frame to hold the new addon's button
   local newFrame = CreateFrame("Frame", nil, MainMenuFrames[#MainMenuFrames])
   newFrame:SetPoint("TOP", MainMenuFrames[#MainMenuFrames]:GetName().."Buttons", "BOTTOM")
   newFrame:SetWidth(ADDONMAN_MAIN_X)
   newFrame:SetHeight(ADDONMAN_MAINTEXT_Y)

   -- create the new button
   local newButtonName = "ManagerMainMenuButton"..NUM_ADDONS
   local newFrameButton = CreateFrame("Button", newButtonName, MainMenuFrames[#MainMenuFrames], "AddonManMainMenuButton")
   -- position the new button so its one line lower than the previous button
   newFrameButton:SetPoint("CENTER", MainMenuFrames[#MainMenuFrames]:GetName().."Buttons", "CENTER", 0, -((#ActiveAddons%15) * ADDONMAN_MAINTEXT_Y))
   
   -- set up the necessary button properties
   newFrameButton:SetText(display)
   
   if (enabled == 1) then
      newFrameButton:SetTextFontObject(ADDONMAN_ENABLED_FONT)
   else
      newFrameButton:SetTextFontObject(ADDONMAN_DISABLED_FONT)
   end
   
   -- store the two arrows for the button
   local newOptionsArrow = getglobal(newFrameButton:GetName().."OptionsArrowText")
   local newSubMenuArrow = getglobal(newFrameButton:GetName().."SubMenuArrowText")
   if (enabled == 1) then
      newSubMenuArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
   end
   
   local newSubMenuName, newSubMenu, newOptionsFrameName, newOptionsFrame = nil
   
   -- so if the addon is not loadable (i.e. not load on demand) then we want to have different functionality depending on why its not loadable
   -- basically this is looking for addons that won't run. NOT_DEMAND_LOADED, DEP_NOT_DEMAND_LOADED, and DISABLED are all normal situations.
   if (loadable == nil) and (not (reason == "NOT_DEMAND_LOADED" or reason == "DEP_NOT_DEMAND_LOADED" or reason == "DISABLED")) then
      -- reset some things on the new button
      newFrameButton:SetTextFontObject(ADDONMAN_NOTLOADED_FONT)
      newFrameButton:SetHighlightFontObject(ADDONMAN_NOTLOADED_FONT)
      getglobal(newFrameButton:GetName().."SubMenuArrow"):Hide()
      
      -- create the new sub menu, which is just a frame for displaying the reason and not a menu at all
      newSubMenuName = "ManagerSubMenu"..NUM_ADDONS
      newSubMenu = CreateFrame("Frame", newSubMenuName, UIParent, "AddonManNotLoadableFrame")
      newSubMenu:SetPoint("TOPRIGHT", newButtonName, "TOPLEFT", 10, 0)
      
      -- create the string and put it inside the new sub menu
      local newFrameString = newSubMenu:CreateFontString(nil, "OVERLAY", ADDONMAN_TITLE_FONT)
      newFrameString:SetPoint("LEFT", newSubMenuName.."Buttons", "LEFT", 0, 0)
      
      -- get the localized text for the reason and change the width of the sub menu frame based on the string length
      local newText = AddonMan_GetLocalizedReason(reason)
      local newX = string.len(newText) * 7
      newFrameString:SetText(newText)
      newSubMenu:SetWidth(newX)

      -- note we do not have an OnClick script
      newFrameButton:SetScript("OnEnter", AddonMan_NotLoadable)
      newFrameButton:SetScript("OnLeave", AddonMan_NotLoadable)
      
      -- insert the frame into UISpecialFrames so that when the user pushes ESC, it will close
      table.insert(UISpecialFrames, newSubMenuName)
   
   -- handle the normal situations
   else
   
      -- set up a new frame for the sub menu
      newSubMenuName = "ManagerSubMenu"..NUM_ADDONS
      newSubMenu = CreateFrame("Frame", newSubMenuName, UIParent, "AddonManSubMenuFrame")
      newSubMenu:SetPoint("TOPLEFT", newButtonName, "TOPRIGHT", -10, 0)
   
      -- set up a new frame for the options menu
      newOptionsFrameName = "AddonManOptionsFrame"..NUM_ADDONS
      newOptionsFrame = CreateFrame("Frame", newOptionsFrameName, UIParent, "AddonManOptionsFrame")
      newOptionsFrame:SetPoint("TOPRIGHT", newButtonName, "TOPLEFT", 10, 0)
      
      newFrameButton:SetScript("OnClick", AddonMan_MainMenuOnClick)
      newFrameButton:SetScript("OnEnter", AddonMan_ToggleMainMenuArrows)
      newFrameButton:SetScript("OnLeave", AddonMan_ToggleMainMenuArrows)
   
      -- insert the frame into UISpecialFrames so that when the user pushes ESC, it will close
      table.insert(UISpecialFrames, newSubMenuName)
      table.insert(UISpecialFrames, newOptionsFrameName)
   end

   -- drop all the pertinent information into a table entry
   ActiveAddons[NUM_ADDONS] = {}
   ActiveAddons[NUM_ADDONS].title = name
   ActiveAddons[NUM_ADDONS].display = display
   ActiveAddons[NUM_ADDONS].button = newFrameButton
   ActiveAddons[NUM_ADDONS].buttonName = newButtonName
   ActiveAddons[NUM_ADDONS].subMenu = newSubMenu
   ActiveAddons[NUM_ADDONS].subMenuName = newSubMenuName
   ActiveAddons[NUM_ADDONS].subArrow = newSubMenuArrow
   ActiveAddons[NUM_ADDONS].hasOptions = false
   ActiveAddons[NUM_ADDONS].totalOptions = 0
   ActiveAddons[NUM_ADDONS].optionsFrame = newOptionsFrame
   ActiveAddons[NUM_ADDONS].optionsFrameName = newOptionsFrameName
   ActiveAddons[NUM_ADDONS].optionsArrow = newOptionsArrow
   ActiveAddons[NUM_ADDONS].isEnabled = enabled
end

-----------
-- called whenever the addon manager is opened or closed by clicking on the top-level button
-----------
function AddonMan_OpenOrCloseMainMenu()

   if MainMenuFrames[#MainMenuFrames]:IsVisible() then
      -- loop through all the main menu frames and hide em
      for i, curMenuFrame in ipairs(MainMenuFrames) do
         curMenuFrame:Hide()
      end
      
      -- hide everyone else
      AddonMan_CloseAllMenus()
      
   else
      -- loop through all the main menu frames and show em
      for i, curMenuFrame in ipairs(MainMenuFrames) do
         curMenuFrame:Show()
      end
   end
   
   -- click!
   PlaySound("UChatScrollButton");
end

-----------
-- display the sub menu containing the reason when the user mouseovers
--
-- self - (object) who called this function
-----------
function AddonMan_NotLoadable(self)
   local index, title, subMenu, subMenuName
   
   -- find out who we're dealing with and pull out the necessary information
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.button == self) then
         index = i
         title = curAddon.title
         subMenu = curAddon.subMenu
         subMenuName = curAddon.subMenuName
      end
   end
   
   if subMenu:IsVisible() then
      subMenu:Hide()
   else
      subMenu:Show()
   end
end

-----------
-- called whenever an addon title is clicked in the main menu
--
-- self - (object) who called this function
-----------
function AddonMan_MainMenuOnClick(self)
   local index, title, subMenu, subMenuName, optionsFrame, optionsFrameName, isEnabled, hasOptions
   
   -- find out who we're dealing with and pull out the necessary information
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.button == self) then
         index = i
         title = curAddon.title
         subMenu = curAddon.subMenu
         subMenuName = curAddon.subMenuName
         optionsFrame = curAddon.optionsFrame
         optionsFrameName = curAddon.optionsFrameName
         isEnabled = curAddon.isEnabled
         hasOptions = curAddon.hasOptions
      end
   end

   -- we're either showing this addon's option and submenu or hiding them
   if subMenu:IsVisible() then
      
      subMenu:Hide()
      
      -- test because optionsFrame could be nil
      if (hasOptions == true) then
         optionsFrame:Hide()
      end
      
   else
      -- close anyone else who may be open
      AddonMan_CloseAllMenus()
      
      -- set up the check marks
      if (isEnabled == 1) then
         getglobal(subMenuName.."CheckDisable"):Hide()
         getglobal(subMenuName.."CheckEnable"):Show()
      else
         getglobal(subMenuName.."CheckEnable"):Hide()
         getglobal(subMenuName.."CheckDisable"):Show()
      end
      
      subMenu:Show()
      
      if (hasOptions == true) then
         optionsFrame:Show()
      end
   end
   
   -- click!
   PlaySound("UChatScrollButton");
end

-----------
-- called whenever an option is clicked
--
-- self - (object) who called this function
-- button - (string) button that did the clicking (not used)
--
-- the toggle function will be sent the following arguments
-- text - (string) the text of the option that was clicked
-- isEnabled - (boolean) whether or not the option is now enabled
-----------
function AddonMan_OptionsOnClick(self, button)

   -- find the option in the table so we know which option we're dealing with
   for i, curOption in ipairs(ActiveOptions) do
      if (curOption.button == self) then
      
         -- handle the toggle and call the addon's function
         if (curOption.isEnabled == true) then
            curOption.isEnabled = false
            curOption.check:Hide()
            
            curOption.toggleFunc(curOption.text, curOption.isEnabled)
            
         else
            curOption.isEnabled = true
            curOption.check:Show()
            
            curOption.toggleFunc(curOption.text, curOption.isEnabled)
            
         end
         
         -- click!
         PlaySound("UChatScrollButton");
      end
   end
end

-----------
-- called whenever an enable button is clicked
--
-- self - (object) who called this function
-- button - (string) button that did the clicking (not used)
-----------
function AddonMan_SubMenuOnClickEnable(self, button)

   -- find out which addon we're dealing with
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.subMenu == self:GetParent()) then
         
         -- change some information
         curAddon.isEnabled = 1
         curAddon.button:SetTextFontObject(ADDONMAN_ENABLED_FONT)
         curAddon.subArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
         curAddon.optionsArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
         
         -- hide everybody, including the main menus
         AddonMan_CloseAllMenus()
         for i, curMenuFrame in ipairs(MainMenuFrames) do
            curMenuFrame:Hide()
         end
         
         -- click!
         PlaySound("UChatScrollButton");
         
         -- enable the addon and reload the UI so the change takes effect
         EnableAddOn(curAddon.title)
         ConsoleExec("reloadui")
      end
   end
end

-----------
-- called whenever a disable button is clicked
--
-- self - (object) who called this function
-- button - (string) button that did the clicking (not used)
-----------
function AddonMan_SubMenuOnClickDisable(self, button)

   -- find out which addon we're dealing with
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.subMenu == self:GetParent()) then
         
         -- change some information
         curAddon.isEnabled = nil
         curAddon.button:SetTextFontObject(ADDONMAN_DISABLED_FONT)
         curAddon.subArrow:SetFontObject(ADDONMAN_DISABLED_FONT)
         curAddon.optionsArrow:SetFontObject(ADDONMAN_DISABLED_FONT)
         
         -- hide everybody, including the main menus
         AddonMan_CloseAllMenus()
         for i, curMenuFrame in ipairs(MainMenuFrames) do
            curMenuFrame:Hide()
         end
         
         -- click!
         PlaySound("UChatScrollButton");
         
         -- disable the addon and reload the UI so the change takes effect
         DisableAddOn(curAddon.title)
         ConsoleExec("reloadui")
      end
   end
end

-----------
-- closes all open sub and option menus
-- unlikely that any more than one addon's menus will be open at a time but safety first
-----------
function AddonMan_CloseAllMenus()

   for i, curAddon in ipairs(ActiveAddons) do
      
      curAddon.subMenu:Hide()
      
      -- test since optionsFrame could be nil
      if (curAddon.hasOptions == true) then
         curAddon.optionsFrame:Hide()
      end
   end
end

-----------
-- this is the hook for the frame manager
-- really all that is happening is we're going through the frames on the right side of the screen
-- and pushing them down so they don't overlap with the addon manager
-----------
function UIParent_ManageFramePositions(...)

   -- execute the original frame manager function
   origUIParent_ManageFramePositions(...)
   
   local myYoffset = 20
   -- this is so we know if frames previous to QuestWatchFrame were offset
   local didOffset = false
   
   -- note i still need to handle the CaptureBar frame(s) during battleground play
   
   if (QuestTimerFrame:IsShown()) then
   
      -- note we are not checking the number of anchor points on the frame
      local point, relativeTo, relativePoint, offsetX, offsetY = QuestTimerFrame:GetPoint(1)
      local newY = offsetY - myYoffset
      
      -- clear points just to play it safe
      QuestTimerFrame:ClearAllPoints()
      QuestTimerFrame:SetPoint(point, relativeTo, relativePoint, offsetX, newY)
      
      didOffset = true
   end
   
   if (DurabilityFrame:IsShown()) then
   
      -- note we are not checking the number of anchor points on the frame
      local point, relativeTo, relativePoint, offsetX, offsetY = DurabilityFrame:GetPoint(1)
      local newY = offsetY - myYoffset
      
      -- clear points just to play it safe
      DurabilityFrame:ClearAllPoints()
      DurabilityFrame:SetPoint(point, relativeTo, relativePoint, offsetX, newY)
      
      didOffset = true
   end
   
   -- this is a unique situation where i still want to offset the QuestWatchFrame if other frames are visible
   -- but if they aren't, i like the positioning of the default offset
   if (didOffset == true) then
   
      -- note we are not checking the number of anchor points on the frame
      local point, relativeTo, relativePoint, offsetX, offsetY = QuestWatchFrame:GetPoint(1)
      local newY = offsetY - myYoffset
      
      -- clear points just to play it safe
      QuestWatchFrame:ClearAllPoints()
      QuestWatchFrame:SetPoint(point, relativeTo, relativePoint, offsetX, newY)
   end
end

-----------
-- simply return the localized string depending on the given reason
--
-- reason - (string) reason why the addon is not loadable
-----------
function AddonMan_GetLocalizedReason(reason)
   if (reason == "BANNED") then
      return ADDON_BANNED
   elseif (reason == "CORRUPT") then
      return ADDON_CORRUPT
   elseif (reason == "DEP_BANNED") then
      return ADDON_DEP_BANNED
   elseif (reason == "DEP_CORRUPT") then
      return ADDON_DEP_CORRUPT
   elseif (reason == "DEP_DISABLED") then
      return ADDON_DEP_DISABLED
   elseif (reason == "DEP_INCOMPATIBLE") then
      return ADDON_DEP_INCOMPATIBLE
   elseif (reason == "DEP_INSECURE") then
      return ADDON_DEP_INSECURE
   elseif (reason == "DEP_INTERFACE_VERSION") then
      return ADDON_DEP_INTERFACE_VERSION
   elseif (reason == "DEP_MISSING") then
      return ADDON_DEP_MISSING
   elseif (reason == "INCOMPATIBLE") then
      return ADDON_INCOMPATIBLE
   elseif (reason == "INSECURE") then
      return ADDON_INSECURE
   elseif (reason == "INTERFACE_VERSION") then
      return ADDON_INTERFACE_VERSION
   elseif (reason == "LOAD_FAILED") then
      return ADDON_LOAD_FAILED
   elseif (reason == "MISSING") then
      return ADDON_MISSING
   elseif (reason == "NOT_DEMAND_LOADED") then
      return ADDON_NOT_DEMAND_LOADED
   elseif (reason == "UNKNOWN_ERROR") then
      return ADDON_UNKNOWN_ERROR
   -- if all else fails, at least return something
   else
      return ADDON_UNKNOWN_ERROR
   end
end

-----------
-- flips the font color of the addon's arrows whenever it is mouse-overed
--
-- self - (object) who called this function
-----------
function AddonMan_ToggleMainMenuArrows(self)

   -- find out who we're dealing with
   for i, curAddon in ipairs(ActiveAddons) do
      if (curAddon.button == self) then
      
         -- basically we're flipping the font color of the button if its been mouse overed or unmouse overed
         if (curAddon.subArrow:GetFontObject():GetName() == ADDONMAN_MOUSEOVER_FONT) then
            -- display the right font if whether the addon is currently enabled or disabled
            if (curAddon.isEnabled == 1) then
               curAddon.subArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
               curAddon.optionsArrow:SetFontObject(ADDONMAN_ENABLED_FONT)
            else
               curAddon.subArrow:SetFontObject(ADDONMAN_DISABLED_FONT)
               curAddon.optionsArrow:SetFontObject(ADDONMAN_DISABLED_FONT)
            end
            
         else
            curAddon.subArrow:SetFontObject(ADDONMAN_MOUSEOVER_FONT)
            curAddon.optionsArrow:SetFontObject(ADDONMAN_MOUSEOVER_FONT)
         end
      end
   end
end