---------------------------------------------------------------------------
-- AS_Debug()
--  Prints a debug message to the default chat frame.
---------------------------------------------------------------------------
function AS_Debug(msg)
  if DEFAULT_CHAT_FRAME and AS_DEBUG_MODE then
    if not msg then
      DEFAULT_CHAT_FRAME:AddMessage("nil");
    else
      DEFAULT_CHAT_FRAME:AddMessage(msg);
    end
  end
end

---------------------------------------------------------------------------
-- AS_Print()
--  Prints a message to the default chat frame.
---------------------------------------------------------------------------
function AS_Print(msg)
  if DEFAULT_CHAT_FRAME then
    if not msg then
      DEFAULT_CHAT_FRAME:AddMessage("nil");
    else
      DEFAULT_CHAT_FRAME:AddMessage(msg);
    end
  end
end

---------------------------------------------------------------------------
-- AS_Error()
--  Prints an error message to the console.
---------------------------------------------------------------------------
function AS_Error(msg)
  if DEFAULT_CHAT_FRAME then
    if not msg then
      DEFAULT_CHAT_FRAME:AddMessage("nil", 1, 0, 0);
    else
      DEFAULT_CHAT_FRAME:AddMessage(msg, 1, 0, 0);
    end
  end
end

---------------------------------------------------------------------------
-- AS_Warn()
--  Prints an error message to the console.
---------------------------------------------------------------------------
function AS_Warn(msg)
  if DEFAULT_CHAT_FRAME then
    if not msg then
      DEFAULT_CHAT_FRAME:AddMessage("nil", 0, 1, 1);
    else
      DEFAULT_CHAT_FRAME:AddMessage(msg, 0, 1, 1);
    end
  end
end

---------------------------------------------------------------------------
-- spellID = AS_LocateSpell(name, rank, guessID)
--  Locates the spell with the given name and rank in a user's spellbook.
--  If the specified rank is not found, or the rank is not specified, the
--  highest rank with a matching name is returned.
-- 
-- name - the name of the spell.
-- rank - the rank of the spell, or nil if any rank is acceptable.
-- guessID - test the spell at this ID first, then scan if not correct.
-- exactMatch - true if an exact rank match is required, false if not.
-- spellID - the spell index, or nil if it's not found
---------------------------------------------------------------------------
function AS_LocateSpell(name, rank, guessID, exactMatch)
  if not name then
    return nil;
  end
  
  local spellName, spellRank, highestMatchID;
  
  -- First, check if the guess ID is correct.
  if guessID then
    spellName, spellRank = GetSpellName(guessID, BOOKTYPE_SPELL);
    if spellName == name and (not rank or spellRank == rank) then
      return guessID;
    end
  end
  
  -- Search the spellbook for the spell.
  local id = 1;
  while true do
    spellName, spellRank = GetSpellName(id, BOOKTYPE_SPELL);
    
    -- Out of spells?
    if not spellName then
      if exactMatch then
        return nil; -- No exact match was found.
      else
        return highestMatchID;
      end
    end
    
    -- Is this the spell?
    if spellName == name then
      highestMatchID = id;  -- Highest rank match yet...
      if spellRank == rank then
        return id;
      end
    end
    
    id = id + 1;
  end
end

---------------------------------------------------------------------------
-- invID, bagID, slot = AS_LocateItem(itemID)
--  Locates the item with the given item ID and returns its inventory or
--  bag slot.
--
-- itemID - ID of the item we're looking for.
-- invID - ID of the item in our equipped inventory, or nil if it's not equipped.
-- bagID - ID of the bag that contains the item, or nil if it's not in a bag, or equipped.
-- slot - slot in the bag if bagID is not nil.
---------------------------------------------------------------------------
function AS_LocateItem(id)
  if not id then
    return nil;
  end
  
  -- First, search the player's equipped inventory.
  for invID = 0,23 do    -- regular inventory items only
    if AS_GetItemIDFromLink(GetInventoryItemLink("player", invID)) == id then
      return invID, nil, nil;
    end
  end
  
  -- Now, search the player's bags.
  for bagID = 0,4 do
    for slot = 1,GetContainerNumSlots(bagID) do
      if AS_GetItemIDFromLink(GetContainerItemLink(bagID, slot)) == id then
        return nil, bagID, slot;
      end
    end
  end
  
  -- The specified item was not found.
  return nil, nil, nil;
end

---------------------------------------------------------------------------
-- macroID = AS_LocateMacro(name, texture)
--   Locates the macro with the given name and texture.
--
-- name - the name of the macro, as displayed to the user.
-- texture - the macro's texture.
-- macroID - ID of the first macro matching the specifications.
---------------------------------------------------------------------------
function AS_LocateMacro(name, texture)
  if not name or not texture then
    return nil;
  end
  
  local numGlobalMacros, numUserMacros = GetNumMacros();
  for id = 1,numGlobalMacros do
    local macroName, macroTexture = GetMacroInfo(id);
    if name == macroName and texture == macroTexture then
      return id;
    end
  end
  
  for id = 19,18+numUserMacros do
    local macroName, macroTexture = GetMacroInfo(id);
    if name == macroName and texture == macroTexture then
      return id;
    end
  end
  
  return nil;
end

---------------------------------------------------------------------------
-- itemID = AS_GetItemIDFromLink(link)
--  Returns the item ID contained within an item link.
--
-- link - the item's link.
-- itemID - the item's ID, or nil if the link is invalid.
---------------------------------------------------------------------------
function AS_GetItemIDFromLink(link)
  if link then
    return tonumber(string.match(link, "[^:]+:([0-9]+)"));
  else
    return nil;
  end
end

---------------------------------------------------------------------------
-- AS_RaiseAllSpells()
--  Raises all spells on the action bar to their highest rank.
---------------------------------------------------------------------------
function AS_RaiseAllSpells()
  local changed = false;
  
  -- We explot the fact that spells are arranged by name and increasing
  -- rank.
  for slot = 1,120 do
    local action = AS_GetAction(slot);
    if action and action._type == "spell" then
      local spellID = AS_LocateSpell(action.name, action.rank, action.id);
      if spellID then
      
        -- Look for the highest ranked spell with this name.
        local newSpellID = spellID;
        while GetSpellName(newSpellID + 1, BOOKTYPE_SPELL) == action.name do
          newSpellID = newSpellID + 1;
        end
        
        if newSpellID ~= spellID then
          local spellName, spellRank = GetSpellName(newSpellID, BOOKTYPE_SPELL);
          
          AS_Print("Upranked " .. AS_COL_GR .. action.name .. " (" ..
            action.rank .. ")" .. AS_NRM .. " to " .. AS_COL_GR .. spellName ..
            " (" .. spellRank .. ")");
          
          -- Update the action to match the higher spell.
          local newAction = {_type="spell", id=newSpellID, name=spellName, rank=spellRank};
          AS_UpdateAction(slot, newAction);
          
          changed = true;
        end
      end
    end
  end
  
  if not changed then
    AS_Print(AS_MSG_ALREADY_RAISED);
  end
end
