---------------------------------------------------------------------------
-- AS_ListProfiles()
--   Displays the list of available profiles to the user.
---------------------------------------------------------------------------
function AS_ListProfiles()
  local profiles = AS_COL_GR .. "custom" .. AS_NRM;
  for k,v in pairs(AS_ProfileList) do
    profiles = AS_COL_GR .. k .. AS_NRM .. ", " .. profiles;
  end
  AS_Print(AS_LISTPROFILES_LINE1 .. profiles);
  AS_Print(AS_LISTPROFILES_LINE2 .. AS_COL_GR .. AS_CurrentProfile .. AS_NRM);
end

---------------------------------------------------------------------------
-- result = AS_CreateProfile(name)
--  Creates a new profile based on the current action bar
--  and key binding setup.
--  
-- name - the name of the new profile.
-- result - true if creation succeeded, false otherwise.
---------------------------------------------------------------------------
function AS_CreateProfile(name)
  assert(name);
  name = string.lower(name);
  
  -- Can't create a profile named 'custom' because this profile has
  -- special functionality.
  if name == AS_PROFILE_CUSTOM then
    AS_Error(AS_ERROR_PROFILE_ALREADY_EXISTS);
    return false;
  end
  
  -- Profile already exists?
  if AS_ProfileList[name] then
    AS_Error(AS_ERROR_PROFILE_ALREADY_EXISTS);
    return false;
  end
  
  -- Save the current action set.
  local profile = {};
  profile.actionSet = AS_GetCurrentActionSet();
  profile.bindingSet = AS_GetCurrentBindingSet();
  
  -- Create the profile.
  AS_ProfileList[name] = profile;
  AS_CurrentProfile = name;
  
  AS_Print(AS_MSG_CREATED_PROFILE_1 .. name .. AS_MSG_CREATED_PROFILE_2);
  return true;
end

---------------------------------------------------------------------------
-- result = AS_DeleteProfile(name)
--   Deletes the profile with the specified name.  The 'custom' profile
--   cannot be deleted.
--
-- name - the name of the profile to delete.
-- result - true if the profile was deleted, false if an error occurred.
---------------------------------------------------------------------------
function AS_DeleteProfile(name)
  assert(name);
  name = string.lower(name);
  
  -- Can't delete the 'custom' profile which has special functionality.
  if name == AS_PROFILE_CUSTOM then
    AS_Error(AS_ERROR_CANT_DELETE_CUSTOM_PROFILE);
    return false;
  end
  
  -- Can't delete a profile which doesn't exist.
  if not AS_ProfileList[name] then
    AS_Error(AS_ERROR_PROFILE_DOESNT_EXIST);
    return false;
  end
  
  -- Deleting current profile?  Switch to the custom profile.
  if name == AS_CurrentProfile then
    AS_SwitchProfile(AS_PROFILE_CUSTOM);
  end
  
  -- Delete the profile.
  AS_ProfileList[name] = nil;
  AS_Print(AS_MSG_DELETED_PROFILE_1 .. name .. AS_MSG_DELETED_PROFILE_2);
  return true;
end

---------------------------------------------------------------------------
-- AS_DeleteAllProfiles()
--  Deletes all profiles, switching to the custom profile.
---------------------------------------------------------------------------
function AS_DeleteAllProfiles()
  if AS_CurrentProfile ~= AS_PROFILE_CUSTOM then
    AS_SwitchProfile(AS_PROFILE_CUSTOM);
  end
  AS_ProfileList = {};
  AS_Print(AS_MSG_DELETED_ALL);
end

---------------------------------------------------------------------------
-- result = AS_SwitchProfile(name)
--  Switches to the given profile, replacing all actions and key bindings
--  to match it.  Switching to the 'custom' profile replaces nothing.
--
-- name - the name of the profile to switch to.
-- result - true if successful, false if the profile doesn't exist.
---------------------------------------------------------------------------
function AS_SwitchProfile(name)
  assert(name);
  name = string.lower(name);
  
  -- We don't have to do much when switching to the custom profile, just
  -- change the current profile and return.
  if name == AS_PROFILE_CUSTOM then
  
    AS_CurrentProfile = AS_PROFILE_CUSTOM;
    
  else
    local profile = AS_ProfileList[name];
    
    -- Profile doesn't exist?
    if not profile then
      AS_Error(AS_ERROR_PROFILE_DOESNT_EXIST);
      return false;
    end
    
    -- Before switching to the new profile, clear any bindings from the
    -- profile which conflict with an action bound by another mod.
    AS_RemoveContradictingBindings(name);
    
    -- Swap out the action bar and key bindings.
    AS_CurrentProfile = name;
    AS_SetCurrentActionSet(profile.actionSet);
    AS_SetCurrentBindingSet(profile.bindingSet);
  end
  
  AS_Print(AS_MSG_PROFILE_SET_TO_1 .. name .. AS_MSG_PROFILE_SET_TO_2);
end

---------------------------------------------------------------------------
-- result = AS_ReplaceProfile(name)
--  Replaces the actions and bindings of the given profile with the current
--  ones.
--  
-- name - the name of the profile to replace.
-- result - true if replace succeeded, false otherwise.
---------------------------------------------------------------------------
function AS_ReplaceProfile(name)
  assert(name);
  name = string.lower(name);
  
  -- Can't replace the 'custom' profile.
  if name == AS_PROFILE_CUSTOM then
    AS_Error(AS_ERROR_CANT_REPLACE_CUSTOM);
    return false;
  end
  
  -- Profile doesn't exist?
  local profile = AS_ProfileList[name]
  if not profile then
    AS_Error(AS_ERROR_PROFILE_DOESNT_EXIST);
    return false;
  end
  
  -- Save the current action set.
  profile.actionSet = AS_GetCurrentActionSet();
  profile.bindingSet = AS_GetCurrentBindingSet();
  AS_Print(AS_MSG_REPLACED_PROFILE_1 .. name .. AS_MSG_REPLACED_PROFILE_2);
  
  -- Switch to the profile.
  AS_SwitchProfile(name);
  
  return true;
end

---------------------------------------------------------------------------
-- result = AS_RemoveContradictingBindings(profileName)
--  Clears any key bindings in the given profile which contradict bindings
--  set to an action not available in the bindings UI.  This is used to
--  make sure we don't overwrite mod bindings.
---------------------------------------------------------------------------
function AS_RemoveContradictingBindings(profileName)
  -- Can't remove bindings from the 'custom' profile.
  if name == AS_PROFILE_CUSTOM then
    return true;
  end
  
  local profile = AS_ProfileList[profileName];
  if not profile then
    AS_Error(AS_ERROR_PROFILE_DOESNT_EXIST);
    return false;
  end

  -- Make a list of bindings present in the binding UI.
  if not AS_BindingNames then
    local numBindings = GetNumBindings();
    AS_BindingNames = {};
    for slot = 1,numBindings do
      AS_BindingNames[GetBinding(slot)] = true;
    end
  end
  
  -- Go through bindings in this profile, if any coincide with another
  -- action, remove them from the profile.
  for k,v in pairs(profile.bindingSet) do
    local action = GetBindingAction(k);
    if action ~= "" and not AS_BindingNames[action] then
      AS_Debug("Removing conflicting binding in profile " .. profileName .. ", key " .. k);
      profile.bindingSet[k] = nil;
    end
  end
end
